﻿function Add-Cert($pfx, $credentialXml)
{
	Write-Output "Start processing certificate '$pfx.Name'." 
    $certInstance = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2

	# Get the password for certificate.
	$pathNode = $credentialXml.Credentials.Certificates.Certificate | where { $_.Path -eq "Deployment\Onebox\"+$pfx.Name}
	if($pathNode.Count -gt 0)
	{
		$password = $pathNode[0].Password
	}
	else
	{
		$password = $pathNode.Password
	}
 
	if($password -ne $null -and [string]::IsNullOrEmpty($password) -eq $false )
	{
		$certInstance.import($pfx.FullName, $password, "Exportable,PersistKeySet,MachineKeySet"); 
		$certTP = $certInstance.Thumbprint;
        Add-CertToStore -certTP:$certTP -certInstance:$certInstance -pfx:$pfx
	}
}

function Add-CertToStore([string]$certTP,$certInstance,$pfx)
{
    $store = Get-Item Cert:\LocalMachine\My
    $store.Open("ReadWrite")

    $trustedStore = Get-Item Cert:\LocalMachine\Root
    $trustedStore.Open("ReadWrite")
    
    # add certificate to trusted root store if not present
	$trustedRootCert = Get-ChildItem "Cert:\LocalMachine\Root" | where {$_.Thumbprint -eq "$certTP"}
	if($trustedRootCert -eq $null -or [string]::IsNullOrEmpty($trustedRootCert) -eq $true)
	{
		Write-output "Adding certificate $pfx.Name to the machine trusted root store."
		$trustedStore.Add($certInstance)
	}

    # add certificate to local machine store if not present
	$localStoreCert = Get-ChildItem "Cert:\LocalMachine\My" |where {$_.Thumbprint -eq "$certTP"}
	if ($localStoreCert -eq $null -or $localStoreCert -eq "") 
	{
		Write-output "Adding certificate $pfx.Name to the local machine store."
		$store.Add($certInstance)

        # Grant permissin to Network Service to the private key.
		$sslCertPrivKey = $certInstance.PrivateKey.CspKeyContainerInfo.UniqueKeyContainerName
		$privateKeyCertFile = Get-Item -path "$ENV:ProgramData\Microsoft\Crypto\RSA\MachineKeys\*"  | where {$_.Name -eq $sslCertPrivKey} 
		$privateKeyAcl = (Get-Item -Path $privateKeyCertFile.FullName).GetAccessControl("Access") 
		$permission = "NT AUTHORITY\NETWORK SERVICE","Read","Allow" 
		$accessRule = new-object System.Security.AccessControl.FileSystemAccessRule $permission 
		$privateKeyAcl.AddAccessRule($accessRule) 
		Set-Acl $privateKeyCertFile.FullName $privateKeyAcl
		Write-output "Network Service account is granted access to the private key of certificate $pfx.Name"

		# Configure the SSL certificate "star1.cloud.onebox.dynamics.com.pfx" to the port 443
		try
		{
			if($pfx.Name -eq "star1.cloud.onebox.dynamics.com.pfx")
			{
				get-item cert:\LocalMachine\My\$certTP | new-item IIS:\SslBindings\0.0.0.0!443 -ErrorAction SilentlyContinue
			}
		}
		catch 
		{
			Write-output "SSL is configured for the AOS website."
			$_.Exception.Message    
		}
    }

    $store.Close()
    $trustedStore.Close()
}

function Remove-CertFromStore($certInstance)
{
    $store = Get-Item Cert:\LocalMachine\My
    $store.Open("ReadWrite")

    $trustedStore = Get-Item Cert:\LocalMachine\Root
    $trustedStore.Open("ReadWrite")

    # remove certificate from the trusted root store if present
	$trustedRootCert = Get-ChildItem "Cert:\LocalMachine\Root" | where {$_.Thumbprint -eq "$certTP"}
	if($trustedRootCert -ne $null -or [string]::IsNullOrEmpty($trustedRootCert) -eq $false)
	{
		Write-output "Removing certificate $pfx.Name from the machine trusted root store."
		$trustedStore.Remove($certInstance)
	}

    # remove certificate from the local machine store if present
	$localStoreCert = Get-ChildItem "Cert:\LocalMachine\My" |where {$_.Thumbprint -eq "$certTP"}
	if ($localStoreCert -ne $null -or [string]::IsNullOrEmpty($localStoreCert) -eq $false) 
	{
		Write-output "Removing certificate $pfx.Name from the local machine store."
		$store.Remove($certInstance)
    }

    $store.Close()
    $trustedStore.Close()
}

function Remove-Cert($pfx, $credentialXml)
{
    $certInstance = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2

    # Get the password for certificate.
    $pathNode = $credentialXml.Credentials.Certificates.Certificate | where { $_.Path -eq "Deployment\Onebox\"+$pfx.Name}
    
    if($pathNode.Count -gt 0)
    {
        $password = $pathNode[0].Password
    }
    else
    {
        $password = $pathNode.Password
    }
 
    if($password -ne $null -and [string]::IsNullOrEmpty($password) -eq $false )
    {
       $certInstance.import($pfx.FullName, $password, "Exportable,PersistKeySet,MachineKeySet"); 
       $certTP = $certInstance.Thumbprint;
       
       Remove-CertFromStore -certInstance:$certInstance
    }
}

function AddOrRemoveWebsiteCerts([string] $rootPath, [bool] $remove)
{
	# Add all certificates to store.
	$credentialXml = [xml](Get-Content "$rootPath\DObind\Deployment\Onebox\Credentials.xml")

	# Get all the available certificate files.
	$allPFX = Get-ChildItem "$rootPath\DObind\Deployment\Onebox\*.pfx"

	# Loop through all each certificate and install it if not installed.
	foreach($pfx in $allPFX)
	{
		if($remove)
		{
			Remove-Cert $pfx $credentialXml
		}
		else
		{
			Add-Cert $pfx $credentialXml
		}
	}
}

function Add-WebsiteCerts([string] $rootPath)
{
	AddOrRemoveWebsiteCerts $rootPath $false
}

function Remove-WebsiteCerts([string] $rootPath)
{
	AddOrRemoveWebsiteCerts $rootPath $true
}

function Remove-ExistingAppPools([string] $appPoolName)
{
	$machineName = $env:COMPUTERNAME;
    $serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)
    $existingAppPool = $serverManager.ApplicationPools | Where-Object{$_.Name -eq "$appPoolName"}
    if($existingAppPool -ne $null)
    {
        Write-Output "Removing app pool '$appPoolName'."
        $serverManager.ApplicationPools.Remove($existingAppPool)
		$serverManager.CommitChanges()
    }
}

function Remove-ExistingWebsites([string] $siteName, $port, $hostHeaderStrs)
{
	$machineName = $env:COMPUTERNAME;
	$serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)
    $existingSite = $serverManager.Sites | Where-Object{$_.Name -eq "$siteName"}
    if($existingSite -ne $null)
    {
        Write-Output "Removing website '$siteName'."
        $serverManager.Sites.Remove($existingSite)
		$serverManager.CommitChanges()
    }
 
    #remove anything already using that port
    [Microsoft.Web.Administration.Site]$site
    foreach($site in $serverManager.Sites) 
    {
        foreach($hostHeader in $hostHeaderStrs)
		{
            foreach($binding in $site.Bindings)
            {
                if($binding.EndPoint.Port -eq $port -and $binding.EndPoint.Address -eq $hostHeader)
                {
                    Write-Warning "Warning: Found an existing site '$($site.Name)' already using port $port; removing it."
				    $serverManager.Sites.Remove($site)
					$serverManager.CommitChanges()
				    Write-Output "Website $($site.Name) removed."
                }
            }
        }
    }
}

function New-AppPool([string] $appPoolName, [string] $user, [string] $password, [int] $pingTimeoutSeconds)
{
	$machineName = $env:COMPUTERNAME;
	$serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)

    Write-output "Creating an app pool named '$appPoolName' under v4.0 runtime, default (Integrated) pipeline."
    $pool = $serverManager.ApplicationPools.Add($appPoolName);
    $pool.managedRuntimeVersion = "v4.0"
    $pool.processModel.identityType = [Microsoft.Web.Administration.ProcessModelIdentityType]::NetworkService
	$pool.processModel.PingResponseTime = [System.TimeSpan]::FromSeconds($pingTimeoutSeconds)
	$pool.Failure.RapidFailProtection =$false
	
	if ($user -ne $null -AND $password -ne $null -AND $user -ne "" -AND $password -ne "") {
	    Write-Output "Setting AppPool to run as user '$user'."
		$pool.processmodel.identityType = [Microsoft.Web.Administration.ProcessModelIdentityType]::SpecificUser
		$pool.processmodel.username = $user
		$pool.processmodel.password = $password
	} 
	
    $serverManager.CommitChanges()

	for($tryCount= 30; $tryCount -gt 0; --$tryCount)
	{
		if ($serverManager.ApplicationPools[$appPoolName].State -eq [Microsoft.Web.Administration.ObjectState]::Started)
		{
			Write-Output "The AppPool was created and started successfully."
			return
		}

		Start-Sleep -Milliseconds 100
	}

	throw "App pool '$appPoolName' was created but did not start automatically."
}

function New-Website([string] $siteName, [string] $aosSitePath, [string] $port, $hostHeaderStrs,$cert, [string] $appPoolName)
{
	$machineName = $env:COMPUTERNAME
	$serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)
	$bindingArguments = "*:443:"+$hostHeaderStrs[0]
	Write-Output "Creating website. Site name: '$siteName'; Bindings: '$bindingArguments'; Website root: '$aosSitePath'."
	$website = $serverManager.Sites.Add($siteName, "https", $bindingArguments, $aosSitePath)
	for($i = 1; $i -lt $hostHeaderStrs.Length ; $i++)
	{
		$bindingArguments = "*:443:"+$hostHeaderStrs[$i]
		Write-Output "Adding binding '$bindingArguments' to the AOS website."
        $website.Bindings.Add($bindingArguments, $cert.GetCertHash(), "My")
	}

	Write-Output "Setting the website app pool to '$appPoolName'."
	# Set the app pool for the website
	$website.Applications[0].ApplicationPoolName = $appPoolName
    $serverManager.CommitChanges()

    
    Set-WebConfigurationProperty -pspath "MACHINE/WEBROOT/APPHOST/$siteName/Apps"  -filter "system.webServer/security/requestFiltering/fileExtensions/add[@fileExtension='.config']" -name "allowed" -value "True"


	for($tryCount= 100; $tryCount -gt 0; --$tryCount)
	{
		if ($serverManager.Sites[$siteName].State -eq [Microsoft.Web.Administration.ObjectState]::Started)
		{
			Write-Output "AOS website was created and started successfully."
			return
		}

		Start-Sleep -Milliseconds 100
	}
	
	throw "The AOS website '$siteName' was created but did not start after waiting for 10 seconds. Check the event log for further details."
}

function New-AosWebsite([string] $rootPath, [string] $siteName, [string] $aosSitePath, [string] $appPoolName, [string] $port, $hostHeaderStrs, [string] $user, [string] $userPassword, [string] $certTP, [int] $pingTimeoutSeconds = 600)
{
	# start IIS.
	Invoke-Command -script {iisreset.exe /start}

    # Cleanup old websites and app pools
	Remove-ExistingWebsites $siteName $appPoolName $port
	Remove-ExistingAppPools $appPoolName

	# Add certs
	Add-WebsiteCerts $rootPath

	# Create new app Pool
	New-AppPool $appPoolName $user $userPassword $pingTimeoutSeconds
	
	# Create the new site
	$cert = Get-ChildItem "Cert:\LocalMachine\My" |where {$_.Thumbprint -eq "$certTP"}
	New-Website $siteName $aosSitePath $port $hostHeaderStrs $cert $appPoolName
}

function Remove-Website([string] $rootPath, [string] $websiteName, [string] $appPoolName)
{
	# Cleanup certs
	Write-Output "Removing certificates."
	Remove-WebsiteCerts $rootPath

	# Cleanup old websites and app pools
	Write-Output "Remvoing website '$websiteName'."
	Remove-ExistingWebsites $websiteName $appPoolName 0

	Write-Output "Remvoing App pool '$appPoolName'."
	Remove-ExistingAppPools $appPoolName
}

function Delete-Service([string]$serviceName)
{
    try
    {
        $service = Get-WmiObject -Class Win32_Service -Filter "Name='$serviceName'" -ErrorAction SilentlyContinue
	    if($service -ne $null)
	    {
		    # stop and delete the batch Service
            Write-output "Stopping service '$serviceName'."
		    Stop-Service "$serviceName" -ErrorAction SilentlyContinue -Force
            $service = Get-Service $serviceName
            while($service -ne $null -and $service.Status -ne "Stopped")
            {
                $service = Get-Service $serviceName
                Write-output "Sleeping for 5 minutes while waiting for the service '$serviceName' to stop."
                Start-Sleep 5
            }

            Write-output "Deleting service '$serviceName'."
		    sc.exe delete "$serviceName" 
	    }
    }
    catch
    {
    }
}

function Terminate-process
{
    # kill process
	$processes = @("batch","batch.exe","mmc")
	foreach($process in $processes)
	{
		$handle = get-process "$process" -ErrorAction SilentlyContinue
		if($handle -ne $null)
		{
			try
			{
                # special case for Event viewer
                if($process -eq "mmc" -and $handle.MainWindowTitle -eq "Event Viewer")
                {
                    $procName = $handle.MainWindowTitle
                    Write-output "Terminating process '$procName'."
				    $handle.kill()
                }
                else
                {
                    Write-output "Terminating process '$handle'."
                    $handle.Kill()
                }
			}
			catch
			{
			}
		}
	}

    # stop and delete Dynamics Rainier services
    Delete-Service -serviceName:"DynamicsAxBatch"
}

function Register-RestartTask([string]$deploymentDir,[string]$logDir,[bool]$deployChocolateyPackage,[string]$packageDir)
{
    Write-Output "Registering a scheduled task to auto start Azure storage emulator upon machine restart."
    $localMachineName = $env:COMPUTERNAME
    $batchFilename = [IO.Path]::Combine("$deploymentDir","AOSDeploy.cmd")
    Write-Output "Creating scheduler task batch file '$batchFilename'."

    "$deploymentDir\DoBind\aosdeploy.exe /action:Restart /rd:`"$deploymentDir`" /ld:`"$logDir`" /td:`"$packageDir`" /chocopkg:$deployChocolateyPackage"|Set-Content $batchFilename

    $taskName = "DynamicsRainierRestartTask"
    & schtasks /create /ru SYSTEM /sc ONSTART /tn $taskName /tr "cmd /c $batchFilename" /F /RL HIGHEST
    if (!$?)
    {
        throw "There was an error creating the task $taskName."
    }
}

function Delete-RestartTask
{
    $taskName = "DynamicsRainierRestartTask"
    Write-Output "Deleting scheduled task '$taskName'."
    $SchTsk = Get-ScheduledTask | Select Name, State | ? {$_.Name -eq $taskName}
    if($SchTsk -ne $null)
    {
        & schtasks /delete /tn $taskName /F
    }
}

function Configure-ProductConfiguration([string]$sitename)
{
    Write-output "Starting the configuration of the production configuration application."
    $aossite=Get-Website -Name $sitename
    if($aossite -eq $null){
        $message="The website '$sitename' does not exist."
        Write-Output $message
        throw $message
    }

    $webroot=$aossite.physicalPath
    $productconfigurationdir=join-path $webRoot "productconfiguration"

    if(!(Test-Path "$productconfigurationdir")){
        Write-Output "Skipping product configuration as the product configuration directory does not exist under the aos web root $webroot."
        return
    }

    $pcapppool="productconfiguration"
    Write-output "Removing web app pool $pcapppool."
    Remove-WebAppPool -Name:$pcapppool -ErrorAction SilentlyContinue
    write-output "Creating web app pool $pcapppool."
    $newapppool=New-WebAppPool -Name:$pcapppool -ErrorAction SilentlyContinue
    write-output "Setting the identity for the web app pool '$pcapppool' to NetworkService."
    $newapppool.processModel.identityType="NetworkService"
    $newapppool|Set-Item
    write-output "Starting web app pool $pcapppool."
    Start-WebAppPool -Name:$pcapppool

    Write-Output "Deleting web application 'ProductConfiguration' under web site '$sitename'."
    Remove-WebApplication -Site:$aossite.Name -Name:"ProductConfiguration" -ErrorAction SilentlyContinue
    Write-Output "Creating new web application 'ProductConfiguration' under web site '$sitename' with application pool `"$pcapppool`"."
    New-WebApplication -Site:$aossite.Name -Name:"ProductConfiguration" -PhysicalPath:$productconfigurationdir -ApplicationPool:$pcapppool -Force 
    Write-Output "Product configuration application successfully configured."
}

Export-ModuleMember -Function New-AosWebsite,Remove-Website,Terminate-process,Register-RestartTask,Delete-RestartTask,Delete-Service,Configure-ProductConfiguration,Add-WebsiteCerts,Remove-WebsiteCerts
# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCD4vYul+/4MT2Y/
# shNR1m8EcMqg8PK/RZCg1SpB/rUJ2qCCDYEwggX/MIID56ADAgECAhMzAAAB32vw
# LpKnSrTQAAAAAAHfMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAxMjE1MjEzMTQ1WhcNMjExMjAyMjEzMTQ1WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQC2uxlZEACjqfHkuFyoCwfL25ofI9DZWKt4wEj3JBQ48GPt1UsDv834CcoUUPMn
# s/6CtPoaQ4Thy/kbOOg/zJAnrJeiMQqRe2Lsdb/NSI2gXXX9lad1/yPUDOXo4GNw
# PjXq1JZi+HZV91bUr6ZjzePj1g+bepsqd/HC1XScj0fT3aAxLRykJSzExEBmU9eS
# yuOwUuq+CriudQtWGMdJU650v/KmzfM46Y6lo/MCnnpvz3zEL7PMdUdwqj/nYhGG
# 3UVILxX7tAdMbz7LN+6WOIpT1A41rwaoOVnv+8Ua94HwhjZmu1S73yeV7RZZNxoh
# EegJi9YYssXa7UZUUkCCA+KnAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUOPbML8IdkNGtCfMmVPtvI6VZ8+Mw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDYzMDA5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAnnqH
# tDyYUFaVAkvAK0eqq6nhoL95SZQu3RnpZ7tdQ89QR3++7A+4hrr7V4xxmkB5BObS
# 0YK+MALE02atjwWgPdpYQ68WdLGroJZHkbZdgERG+7tETFl3aKF4KpoSaGOskZXp
# TPnCaMo2PXoAMVMGpsQEQswimZq3IQ3nRQfBlJ0PoMMcN/+Pks8ZTL1BoPYsJpok
# t6cql59q6CypZYIwgyJ892HpttybHKg1ZtQLUlSXccRMlugPgEcNZJagPEgPYni4
# b11snjRAgf0dyQ0zI9aLXqTxWUU5pCIFiPT0b2wsxzRqCtyGqpkGM8P9GazO8eao
# mVItCYBcJSByBx/pS0cSYwBBHAZxJODUqxSXoSGDvmTfqUJXntnWkL4okok1FiCD
# Z4jpyXOQunb6egIXvkgQ7jb2uO26Ow0m8RwleDvhOMrnHsupiOPbozKroSa6paFt
# VSh89abUSooR8QdZciemmoFhcWkEwFg4spzvYNP4nIs193261WyTaRMZoceGun7G
# CT2Rl653uUj+F+g94c63AhzSq4khdL4HlFIP2ePv29smfUnHtGq6yYFDLnT0q/Y+
# Di3jwloF8EWkkHRtSuXlFUbTmwr/lDDgbpZiKhLS7CBTDj32I0L5i532+uHczw82
# oZDmYmYmIUSMbZOgS65h797rj5JJ6OkeEUJoAVwwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAd9r8C6Sp0q00AAAAAAB3zAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQg20HWb08c
# aASlXYRTyuQlB+EYvrqhgLAuc/2C1QpXH2EwQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQBe5g5rfzRmFOEfrCSxBD+erbhlKS6N1+hahPicLBA2
# dpIkppi8ytXy4ZQb82FyT+DwGayEA1iVOOpRHzMGX9FSyV1iYWqo9a8WRqqOLKtY
# HUpR1AO3v5R638CM5PrqtO1LQmUDs18FS+OU7OtgkbXFzSEoz1pyB/FtLf7om7/Y
# BQDzj+pxe4rxSZZIDNKI3DboLh+ZtplkSZFhUnIMv+JTGSZTagD6PEQofqZCQmH/
# A12unetWlGu+j2FAFVpNuudWBXdCs/yHZxmyCtrWeI78tQUPKGDL6/84DTTLouHF
# RfgPNmRcIH9SWHZQe6iP4xGvX3oOg2Qwn0tCeN3jar+ZoYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIJMV21F/10a14q/kD6pmf3adRyUMEXb2NvXep5UA
# iebkAgZhHrHUj/wYEzIwMjEwODIzMDcyMTAwLjcwOFowBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjpEOURFLUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABYfWiM16gKiRpAAAA
# AAFhMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTIxMDExNDE5MDIyMVoXDTIyMDQxMTE5MDIyMVowgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpEOURF
# LUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJeInahBrU//GzTqhxUy
# AC8UXct6UJCkb2xEZKV3gjggmLAheBrxJk7tH+Pw2tTcyarLRfmV2xo5oBk5pW/O
# cDc/n/TcTeQU6JIN5PlTcn0C9RlKQ6t9OuU/WAyAxGTjKE4ENnUjXtxiNlD/K2ZG
# MLvjpROBKh7TtkUJK6ZGWw/uTRabNBxRg13TvjkGHXEUEDJ8imacw9BCeR9L6und
# r32tj4duOFIHD8m1es3SNN98Zq4IDBP3Ccb+HQgxpbeHIUlK0y6zmzIkvfN73Zxw
# fGvFv0/Max79WJY0cD8poCnZFijciWrf0eD1T2/+7HgewzrdxPdSFockUQ8QovID
# IYkCAwEAAaOCARswggEXMB0GA1UdDgQWBBRWHpqd1hv71SVj5LAdPfNE7PhLLzAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQAQTA9bqVBmx5TTMhzj+Q8zWkPQXgCc
# SQiqy2YYWF0hWr5GEiN2LtA+EWdu1y8oysZau4CP7SzM8VTSq31CLJiOy39Z4RvE
# q2mr0EftFvmX2CxQ7ZyzrkhWMZaZQLkYbH5oabIFwndW34nh980BOY395tfnNS/Y
# 6N0f+jXdoFn7fI2c43TFYsUqIPWjOHJloMektlD6/uS6Zn4xse/lItFm+fWOcB2A
# xyXEB3ZREeSg9j7+GoEl1xT/iJuV/So7TlWdwyacQu4lv3MBsvxzRIbKhZwrDYog
# moyJ+rwgQB8mKS4/M1SDRtIptamoTFJ56Tk6DuUXx1JudToelgjEZPa5MIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpE
# OURFLUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUAFW5ShAw5ekTEXvL/4V1s0rbDz3mggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AOTNJLYwIhgPMjAyMTA4MjIyMzMyMDZaGA8yMDIxMDgyMzIzMzIwNlowdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA5M0ktgIBADAKAgEAAgIUtgIB/zAHAgEAAgIRHjAK
# AgUA5M52NgIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBAKKmsqxZtAZnR0OA
# w7NGBFN/Js6YhRew8zhb6/DpO3vUAqrKjQ1Ly7UoKm48GMWNC59qUUBaZcH0NvAu
# GK6/flzwccLYgICDOaJ7QUY81DkIqpWExWH6pMRj5iYBw/EhI6WGUjoXqPw7kl+8
# 5lXQ7cHGwgZeCPu75LvfwDslY7rcMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAFh9aIzXqAqJGkAAAAAAWEwDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQgYRUIkBDABjtRyzfR5XVLcf0rE8w5hbHepLhVH6TulCgwgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCBhz4un6mkSLd/zA+0N5YLDGp4vW/VBtNW/lpmh
# tAk4bzCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# YfWiM16gKiRpAAAAAAFhMCIEIEbIt6Ygk+/l43GhkFC+e8Aa0iWxxDdy3EcJIAcP
# Z/ThMA0GCSqGSIb3DQEBCwUABIIBAEFKma1+Xqky6PSEMDbMwUhh5saY2MPgTTby
# KpT8EIZK+0ABmJBCHe3PC1ZcXEgBpUyl76tIywBQ4mFy4YtgAgVpbCuxiCYj2qLe
# skH5SZRYimg7DF6NBGwkPHJmPNfNdpwlahp/Q0eipk3064R6h+ZlQBEMzu5sxHjd
# 4eYoiXuoW7AfIRWIa4b/qBgD9RVPPwSJCync8j80hLgYfUlGc+vGB5MzpTC2cXly
# eDsg2q7HpHoE5fNX7LfgUV8CAC/U3tqeQ0rRTRn6Nbsnngzua4+cCprVqEbtnV07
# qW6Rjvym8kHb1z+2a91tNNy0RvBFWNW8ZCxNQW764lYo3aKDyZA=
# SIG # End signature block
