﻿[CmdletBinding()]
param
(
    [Parameter(Mandatory=$true)]
    [string]$ServiceModelXml,

    [Parameter(Mandatory=$true)]
    [string]$log,

    [Parameter(Mandatory=$false)]
    [string]$config
)

Import-Module WebAdministration

#region configuration generators
function Write-Storage-Emulator-Configuration([string]$logDir)
{
    $registrykey="HKLM:\SOFTWARE\Wow6432Node\Microsoft\Windows Azure Storage Emulator\"
    $registrykeyvaluename="InstallPath"
    $emulatorInstallPath=Get-RegistryKeyValue -registryKey:$registrykey -registryKeyValueName:$registrykeyvaluename
    if([string]::IsNullOrEmpty($emulatorInstallPath)){
        Write-Log "Skipping Azure sdk configuration as the Azure emulator is not installed."
        break
    }

    $azurestorageemulator= join-path $emulatorInstallPath "azurestorageemulator.exe"

    $standardout=join-path $global:logdir "azurestorageemulator.stopprocess.output"
    $standarderr=join-path $global:logdir "azurestorageemulator.stopprocess.error"
    $script="`
    WindowsProcess StopEmulatorProcesses `
    { `
        Ensure=`"Absent`" `
        Arguments=`"start /inprocess`" `
        Path=`"$azurestorageemulator`" `
        DependsOn=@() `
    }"

    Write-Log "Creating the configuration to stop any storage emulator process(es)"
    Write-Content $script

    $script = @"
    DynamicsScheduledTask StartAzureStorageEmulator_ScheduledTask
    {
        TaskName = "DynamicsStartAzureStorageEmulator"
        Ensure = "Absent"
    }
"@
    Write-Content $script

}

function Write-Miscellaneous-Processes-Configuration([string]$webroot,[string]$logDir)
{
    $binarypathname=join-path $webroot "bin\Microsoft.Dynamics.AX.Deployment.Setup.exe"
    $standardoutdir=join-path $logDir "Microsoft.Dynamics.AX.Deployment.Setup.exe.output"
    $standarderrdir=join-path $logDir "Microsoft.Dynamics.AX.Deployment.Setup.exe.error"

    $script="`
    DynamicsProcess StopDeploymentSetupProcess `
    { `
        Ensure=`"Absent`" `
        Arguments=`"`" `
        File=`"$binarypathname`" `
        StandardErrorPath=`"$standardoutdir`" `
        StandardOutputPath=`"$standarderrdir`" `
        DependsOn=@(`"[WindowsProcess]StopEmulatorProcesses`") `
        MonitoringAssembly=`"$global:telemetrydll`" `
    } "

    Write-Log "Creating the configuration to stop the deployment setup process"
    Write-Content $script
}

function Write-BatchService-Configuration([string]$webroot,[string]$logDir)
{
    $webconfig= join-path $webroot "web.config"
    $binarypathname=join-path $webroot "bin\Batch.exe"
    $standardout=join-path $logDir "batch.exe.output"
    $standarderr=join-path $logDir "batch.exe.error"
    $deleteBatchServiceDependency=""
    if(Test-Path $binarypathname){
        $script="`
        WindowsProcess StopBatchProcess `
        { `
            Ensure=`"Absent`" `
            Arguments='-service $webconfig' `
            Path=`"$binarypathname`" `
        } "

        Write-Log "Creating the configuration to stop the Dynamics AX Batch process"
        Write-Content $script
        $deleteBatchServiceDependency="`"[WindowsProcess]StopBatchProcess`""
    }
    
    $servicepath=join-path $webroot "bin\Batch.exe -service $webconfig"
    $script = "`
    Service DeleteBatchService `
    { `
        Ensure=`"Absent`" `
        Name=`"DynamicsAxBatch`" `
        State=`"Stopped`" `
        DisplayName=`"Dynamics AX Batch Management`" `
        Path='$servicepath' `
        DependsOn=@($deleteBatchServiceDependency) `
    }"

    Write-Log "Creating the configuration to delete the Dynamics AX Batch service"
    Write-Content $script

    $script = @"
    DynamicsScheduledTask StartBatch_ScheduledTask
    {
        TaskName = "DynamicsStartBatch"
        Ensure = "Absent"
    }
"@
    Write-Content $script
}

function Write-WebSite-Configuration([string]$websitename,[string]$webroot,[string]$apppoolname)
{
    $source=$env:TEMP
    $script="`
    xWebSite DeleteAosWebSite `
    { `
        Ensure=`"Absent`" `
        Name=`"$websiteName`" `
        PhysicalPath=`"$webRoot`" `
        State=`"Stopped`" `
    }"

    Write-Log "Creating the configuration to delete the AOS web site"
    Write-Content $script

    if(![string]::IsNullOrEmpty($apppoolname)){
         # delete the aos apppool if it exists
        $script="`
        xWebAppPool DeleteAosAppPool `
        { `
            Ensure=`"Absent`" `
            State=`"Stopped`" `
            Name=`"$apppoolname`" `
            DependsOn=@(`"[xWebSite]DeleteAosWebSite`") `
        }"

        Write-Log "Creating the configuration to delete the AOS web app pool"
        Write-Content $script
    }

    $script= @"
    xWebAppPool DeleteProductConfigurationAppPool
    {
        Ensure="Absent"
        State="Stopped"
        Name="ProductConfiguration"
        DependsOn=@("[xWebSite]DeleteAosWebSite")
    }
"@
    Write-Log "Creating the configuration to delete the product configuration web app pool"
    Write-Content $script

    $script = "
    File DeleteWebSitePhysicalPath`
    { `
        Ensure=`"Absent`" `
        DestinationPath=`"$webRoot`" `
        Recurse=`$true `
        SourcePath=`"`" `
        Type=`"Directory`" `
        DependsOn=@(`"[xWebSite]DeleteAosWebSite`", `"[Service]DeleteBatchService`") `
        Force=`$true `
    }"

     Write-Log "Creating the configuration to delete the AOS web site files"
     Write-Content $script
    
}

function Write-Packages-Configuration([string]$packagedir)
{
    $script = "
    File DeletePackages`
    { `
        Ensure=`"Absent`" `
        DestinationPath=`"$packageDir`" `
        Recurse=`$true `
        SourcePath=`"`" `
        Type=`"Directory`" `
        Force=`$true `
        DependsOn=@(`"[xWebSite]DeleteAosWebSite`") `
    }"

    Write-Log "Creating the configuration to delete the Dynamics packages"
    Write-Content $script
}

function Write-Database-Configuration([string]$dbName,[string]$dbServer,[string]$dbUser,[string]$dbPwd,[string]$logDir)
{
    # for dropping the db, use the temp folder name as the package dir as it is not used
    $packageDir=$env:TEMP

    $script="`
    DynamicsDatabase DropDatabase `
    { `
        Ensure=`"Absent`" `
        PackageDirectory=`"$packageDir`" `
        DatabaseName=`"$dbName`" `
        DatabaseServer=`"$dbServer`" `
        UserName=`"$dbUser`" `
        Password=`"$dbPwd`" `
        DependsOn=@(`"[File]DeletePackages`") `
        MonitoringAssembly=`"$global:telemetrydll`" `
    }"

    Write-Log "Creating the configuration to drop the AX database"
    Write-Content $script
}

function Write-LCM-Configuration($thumbprint)
{
    Write-Verbose "Creating the LCM configuration to set the encryption certificate thumbprint..."
    $script="`
    LocalConfigurationManager 
    {
        CertificateId=`"$thumbprint`"
    }"
    
    write-content $script
}
#endregion

#region helper functions
function Initialize-Log([string]$logdir,[string]$logfile)
{
    if(-not (Test-Path $logdir)){
        New-Item -ItemType Directory -Path $logdir
    }
    
    if(Test-Path $logfile){
        Remove-Item $logfile -Force
    }

    New-item $logfile -ItemType File -Force
}

function Write-Log([string]$message)
{
    Add-Content $global:logfile $message
    Write-Host $message
}

function Write-Header
{
    $datetime=Get-Date
    $header="`
###
# ==++==
#
# Copyright (c) Microsoft Corporation. All rights reserved.
#
# Generated date: $datetime
#
# This file is auto-generated by a tool. Any changes made to this file will be lost.
#
# This file will be compiled to generate a Windows DSC configuration that will remove 
# the Dynamics AOS service on this machine.
###

#region service configuration
Configuration RemoveAosServiceConfiguration { `
`
Import-DscResource -Module xWebAdministration `
Import-DscResource -Module xDynamics `
Import-DscResource -Module PSDesiredStateConfiguration `

Node `"localhost`"{"

    Write-Content $header
}

function Write-Footer([string]$outputpath)
{
    $footer="}`
    } `

`$configData=@{
    AllNodes=@(
        @{NodeName=`"localhost`";CertificateFile=`"$global:encryptioncertpublickeyfile`";Thumbprint=`"$global:encryptioncertthumbprint`";PSDscAllowDomainUser=`$true; }
    )
}`

#endregion

# generate the MOF file `    
RemoveAosServiceConfiguration -OutputPath:$outputpath -ConfigurationData:`$configData"

    Write-Content $footer
}

function Write-Content([string]$content)
{
   Add-Content $global:dscconfigfile -Value $content
}

function Get-RegistryKeyValue([string]$registryKey,[string]$registryKeyValueName,[string]$defaultValue=[string]::Empty)
{
    $value=(Get-ItemProperty "$registryKey").$registryKeyValueName
    if([string]::IsNullOrEmpty($value)){
        return $defaultValue
    }

    return $value
}

function Copy-CustomDSCResources([string]$grandparentDir)
{
    if(Test-Path "$grandparentDir\Scripts\xDynamics"){
        Write-Log "Copying the custom DSC resources"
        Copy-Item -Path "$grandparentDir\Scripts\xDynamics" -Destination "$env:ProgramFiles\WindowsPowerShell\Modules" -Force -Recurse -Verbose
        Get-ChildItem -Path "$env:ProgramFiles\WindowsPowerShell\Modules" -Recurse | Unblock-File -Verbose  
    }else{
        Write-Log "No custom DSC resources to copy"
    }
}

function Save-EncryptionCertificate-PublicKey($certificate)
{
    Write-Log "Saving the encryption cert public key to file..."
    $global:encryptioncertpublickeyfile=join-path $global:logdir "EncryptCert.cer"
    Export-Certificate -cert:$certificate -FilePath $global:encryptioncertpublickeyfile -Force -Type CERT | Out-Null
}

function Get-EncryptionCertificate-Thumbprint
{
    $subject="MicrosoftDynamicsAXDSCEncryptionCert"

    #get or create a new self-signed encryption certificate to secure sensitive info in the MOF files
    $cert=Get-Make-Encryption-Cert -subject:$subject
    Save-EncryptionCertificate-PublicKey -certificate:$cert
    $cert.Thumbprint
}

function Get-Make-Encryption-Cert([string]$subject)
{
    Write-Log "Checking if a self-signed encryption cert with subject '$subject' exists..."
    $formattersubject="CN=$subject"
    $cert=Get-ChildItem Cert:\LocalMachine\My|where {$_.Subject -eq $formattersubject}
    if($cert -ne $null) # if cert existed make sure it is valid
    {
        if(!(Is-ValidCert -certificate:$cert))
        {
            Write-Log "Dynamics DSC self-signed encryption cert is expired. Generating a new self-signed encryption certificate..."
            Write-Log "Deleting the invalid self-signed encryption certificate with subject '$cert.subject'... "
            $thumbprint=$cert.Thumbprint
            Remove-Item -Path Cert:\LocalMachine\My\$thumbprint -Force -DeleteKey |out-null
            $cert=Make-Certificate -subject:$subject
        }
    }
    else
    {
        $cert=Make-Certificate -subject:$subject 
    }

    $cert
}

function Is-ValidCert($certificate)
{
    $subject=$certificate.Subject
    Write-Log "Checking if the certificate '$subject' is valid..."
    $thumbprint=$certificate.Thumbprint
    $cert=Get-ChildItem -Path Cert:\LocalMachine\My\$thumbprint
    if($cert -ne $null)
    {
        if($cert.NotAfter -lt (Get-Date)) #expired
        {
            return $false
        }
        else
        {
            return $true
        }
    }

    #if cert is not found, return false
    return $false
}

function Make-Certificate([string]$subject)
{
     Write-Log "Creating a new DSC self-signed encryption certificate with subject '$subject'..."
     return New-SelfSignedCertificate -DnsName $subject -CertStoreLocation cert:\LocalMachine\My
}

function Output-CurrentEnvironmentState([string]$websitePath, [string]$packagePath)
{
    $handleExePath=join-path $env:SystemDrive "SysInternals\Handle.exe"
    $openHandlesLog=join-path $global:logDir "aosservice-uninstallation-openhandles.log"
    if(-not (Test-Path $handleExePath))
    {
        return
    }
    
    #dump any handles to files in the website directory
    if(Test-Path $websitePath)
    {
        Write-Log "AOS WebRoot still exists at $websitePath. Dumping open handles to $openHandlesLog..."
        & "$handleExePath" -AcceptEula $websitePath | Out-File $openHandlesLog -Append
    }
    
    #dump any handles to files in the package directory
    if(Test-Path $packagePath)
    {
        Write-Log "AOS packages directory still exists at $packagePath. Dumping open handles to $openHandlesLog..."
        & "$handleExePath" -AcceptEula $packagePath | Out-File $openHandlesLog -Append
    }
}

#endregion

#region Main
$parentdir=Split-Path -parent $PSCommandPath
$grandparentdir=Split-Path -parent $parentdir
$global:logfile=$log
$global:logdir=[System.IO.Path]::GetDirectoryName($log)

Initialize-Log -logdir:$global:logdir -logfile:$log
Copy-CustomDSCResources -grandparentDir:$grandparentdir

$global:decodedservicemodelxml=[System.Text.Encoding]::UTF8.GetString([System.Convert]::FromBase64String($servicemodelxml))

$global:dscconfigfile=join-path $global:logdir "Uninstall.ps1"

if(Test-Path $global:dscconfigfile){
    Remove-Item $global:dscconfigfile -Force
}

$outputpath=join-path $global:logdir "Uninstall"
$etwpath=join-path $grandparentdir "ETWManifest"
$global:telemetrydll = join-path $etwpath "Microsoft.Dynamics.AX7Deployment.Instrumentation.dll"

if(-not (Test-Path $global:telemetrydll)){
    throw "The deployment telemetry assembly does not exist"
}

[System.Xml.XmlDocument] $xd = new-object System.Xml.XmlDocument
$xd.LoadXml($global:decodedservicemodelxml)
$ns=New-Object System.Xml.XmlNamespaceManager($xd.NameTable)
$ns.AddNamespace("ns",$xd.DocumentElement.NamespaceURI)

if($env:USERDOMAIN -eq $env:COMPUTERNAME){
    $global:domain="builtin"
}else{
    $global:domain=$env:USERDOMAIN
}

$global:username=$env:USERNAME

$global:encryptioncertthumbprint=Get-EncryptionCertificate-Thumbprint

[string]$websiteName=$xd.SelectSingleNode("//ns:ServiceModel",$ns).getAttribute("Name")
[string]$webRoot=$xd.SelectSingleNode("//ns:Setting[@Name='Infrastructure.WebRoot']",$ns).getAttribute("Value")
Write-Log "The web root is $webRoot"
[string]$packageDir=$xd.SelectSingleNode("//ns:WorkingFolder[@Name='PackagesLocalDirectory']",$ns).getAttribute("Directory")
Write-Log "The package directory is $packageDir"

$website=Get-Website $websiteName
if($website -ne $null)
{
    $apppool=$website.applicationPool
}

$uninstallAttempt=1
while($uninstallAttempt -le 3 -and !$success)
{
    if($uninstallAttempt -gt 1)
    {
        $retryAttempt=$uninstallAttempt-1
        Write-Log "Retry attempt $retryAttempt`: Retrying AOS service uninstallation..."
    }

    # create the configuration file
    Write-Header 
    Write-LCM-Configuration -thumbprint:$global:encryptioncertthumbprint
    Write-Storage-Emulator-Configuration -logDir:$log
    Write-Miscellaneous-Processes-Configuration -webroot:$webroot -logDir:$log
    Write-BatchService-Configuration -webroot:$webroot -logDir:$log
    Write-WebSite-Configuration -websitename:$websiteName -webroot:$webroot -apppoolname:$apppool
    Write-Packages-Configuration -packagedir:$packageDir

    Write-Footer -outputpath:$outputpath
    #endregion

    #region execute the configuration
    Write-Log "Executing the configuration.."
    & $global:dscconfigfile
    [bool]$success=$false
    $dscConfigApplied = $false
    try
    {
        $dscConfigApplied = $false
        Set-Location $outputpath
        Write-Log ("PSModulePath is currently: "+$env:PSModulePath)

        Write-Log "Setting up LCM to decrypt credentials..."
        Set-DscLocalConfigurationManager -path "$outputpath" -Verbose *>&1 | Tee-Object $log
        
        try
        {
            Write-Log("Dumping available DSC resources before applying the configuration...")
            $availableDSCResourceLog=join-path $global:logdir "aosservice-uninstallation-availabledscresources.log"
            Get-DSCResource -Name * | Format-List | Out-File -FilePath $availableDSCResourceLog
        }
        catch
        {
            Write-Log "Failed to get DSC resources, Error: $_"
        }
        
        Write-Log "Applying the configuration..."
        Start-DscConfiguration -wait -Verbose -path "$outputpath" -Force *>&1 | Tee-Object $log
        $dscConfigApplied = $true
    }
    catch
    {
        Write-Log "Uninstall attempt $uninstallAttempt`: Error: $_"
    }

    $configstatuslog=join-path $global:logdir "aosservice-uninstallation-status.log"
    $ConfigStatus = Get-DscConfigurationStatus
    $ConfigStatus | Format-List -Property * | Out-File -FilePath $configstatuslog -Force
    if($ConfigStatus.Status -ieq 'Success' -and $dscConfigApplied -eq $true)
    {
        $success=$true
    }
    else
    {
        Output-CurrentEnvironmentState $webRoot $packageDir
        Move-Item $global:dscconfigfile (join-path $global:logdir "Uninstall_Attempt_$uninstallAttempt.ps1")
        $uninstallAttempt++
    }
}

if($success)
{
    Write-Log "Configuration applied."
    return 0
}
else
{
    throw "AOS uninstallation did not complete after 3 retries, Message: $($ConfigJob.StatusMessage), see log for details."    
}

#endregion

# SIG # Begin signature block
# MIIjnwYJKoZIhvcNAQcCoIIjkDCCI4wCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAfYjrp8MucsvnP
# mhosShlB1xRhnlNlyhBUqLPqcZEgqKCCDYEwggX/MIID56ADAgECAhMzAAAB32vw
# LpKnSrTQAAAAAAHfMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAxMjE1MjEzMTQ1WhcNMjExMjAyMjEzMTQ1WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQC2uxlZEACjqfHkuFyoCwfL25ofI9DZWKt4wEj3JBQ48GPt1UsDv834CcoUUPMn
# s/6CtPoaQ4Thy/kbOOg/zJAnrJeiMQqRe2Lsdb/NSI2gXXX9lad1/yPUDOXo4GNw
# PjXq1JZi+HZV91bUr6ZjzePj1g+bepsqd/HC1XScj0fT3aAxLRykJSzExEBmU9eS
# yuOwUuq+CriudQtWGMdJU650v/KmzfM46Y6lo/MCnnpvz3zEL7PMdUdwqj/nYhGG
# 3UVILxX7tAdMbz7LN+6WOIpT1A41rwaoOVnv+8Ua94HwhjZmu1S73yeV7RZZNxoh
# EegJi9YYssXa7UZUUkCCA+KnAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUOPbML8IdkNGtCfMmVPtvI6VZ8+Mw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDYzMDA5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAnnqH
# tDyYUFaVAkvAK0eqq6nhoL95SZQu3RnpZ7tdQ89QR3++7A+4hrr7V4xxmkB5BObS
# 0YK+MALE02atjwWgPdpYQ68WdLGroJZHkbZdgERG+7tETFl3aKF4KpoSaGOskZXp
# TPnCaMo2PXoAMVMGpsQEQswimZq3IQ3nRQfBlJ0PoMMcN/+Pks8ZTL1BoPYsJpok
# t6cql59q6CypZYIwgyJ892HpttybHKg1ZtQLUlSXccRMlugPgEcNZJagPEgPYni4
# b11snjRAgf0dyQ0zI9aLXqTxWUU5pCIFiPT0b2wsxzRqCtyGqpkGM8P9GazO8eao
# mVItCYBcJSByBx/pS0cSYwBBHAZxJODUqxSXoSGDvmTfqUJXntnWkL4okok1FiCD
# Z4jpyXOQunb6egIXvkgQ7jb2uO26Ow0m8RwleDvhOMrnHsupiOPbozKroSa6paFt
# VSh89abUSooR8QdZciemmoFhcWkEwFg4spzvYNP4nIs193261WyTaRMZoceGun7G
# CT2Rl653uUj+F+g94c63AhzSq4khdL4HlFIP2ePv29smfUnHtGq6yYFDLnT0q/Y+
# Di3jwloF8EWkkHRtSuXlFUbTmwr/lDDgbpZiKhLS7CBTDj32I0L5i532+uHczw82
# oZDmYmYmIUSMbZOgS65h797rj5JJ6OkeEUJoAVwwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVdDCCFXACAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAd9r8C6Sp0q00AAAAAAB3zAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgg7Owaixk
# M5EDzafPQCiaXwrV6Wgf9p2NIOssY20/Mr8wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQCFCy1XxgGpqH8ve9Waq4bBnMZAQbBErDp/5kpXsmIU
# oZ7jqxN9Ot6y49VGYWygrTx5FhIzgkogG8t0ipD8LmDA1VqUF1ZyjKRaMUuu0Mxk
# h1Cd3sdGh3IEORzFl1Vjh3prmMwXi5tDOnhhI0Y1auNEUTdlyK7e4foasQ7iMfav
# V8aBLYB8KYkrwrzky9M1I+tpe8F7C5vAaeYKOS71e0MX6595RXbYKdETXVf05wnv
# +UQGMzCJ98SArH8/s1TJza/qwmeyb5Je/KrXDSUGhRCLIMLJhqLpmBREu5HGRsns
# pSHFjVGcTjTv+dy+v9LfgQ2yaRvHng4GwGOJsu52IoZHoYIS/jCCEvoGCisGAQQB
# gjcDAwExghLqMIIS5gYJKoZIhvcNAQcCoIIS1zCCEtMCAQMxDzANBglghkgBZQME
# AgEFADCCAVkGCyqGSIb3DQEJEAEEoIIBSASCAUQwggFAAgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIKHdDNAmlfYu6x3+sZp10JlrYiyBZ0WKBCOSoWEG
# YKObAgZg1KH4npwYEzIwMjEwODIzMDcyMDEwLjkzMlowBIACAfSggdikgdUwgdIx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1p
# Y3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlvbnMgTGltaXRlZDEmMCQGA1UECxMdVGhh
# bGVzIFRTUyBFU046OEQ0MS00QkY3LUIzQjcxJTAjBgNVBAMTHE1pY3Jvc29mdCBU
# aW1lLVN0YW1wIFNlcnZpY2Wggg5NMIIE+TCCA+GgAwIBAgITMwAAATqNjTH3d0lJ
# wgAAAAABOjANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Eg
# MjAxMDAeFw0yMDEwMTUxNzI4MjJaFw0yMjAxMTIxNzI4MjJaMIHSMQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQg
# SXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxlcyBUU1Mg
# RVNOOjhENDEtNEJGNy1CM0I3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFt
# cCBTZXJ2aWNlMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAzl8k518P
# lz8JTIXYn/O9OakqcWqdJ8ZXJhAks9hyLB8+ANW7Zngb1t7iw7TmgeooOwMnbhCQ
# QH14UwWd8hQFWexKqVpcIFnY3b15+PYmgVeQ4XKfWJ3PPMjTiXu73epXHj9XX7mh
# S2IVqwEvDOudOI3yQL8D8OOG24b+10zDDEyN5wvZ5A1Wcvl2eQhCG61GeHNaXvXO
# loTQblVFbMWOmGviHvgRlRhRjgNmuv1J2y6fQFtiEw0pdXKCQG68xQlBhcu4Ln+b
# YL4HoeT2mrtkpHEyDZ+frr+Ka/zUDP3BscHkKdkNGOODfvJdWHaV0Wzr1wnPuUgt
# ObfnBO0oSjIpBQIDAQABo4IBGzCCARcwHQYDVR0OBBYEFBRWoJ8WXxJrpslvHHWs
# rQmFRfPLMB8GA1UdIwQYMBaAFNVjOlyKMZDzQ3t8RhvFM2hahW1VMFYGA1UdHwRP
# ME0wS6BJoEeGRWh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1
# Y3RzL01pY1RpbVN0YVBDQV8yMDEwLTA3LTAxLmNybDBaBggrBgEFBQcBAQROMEww
# SgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMv
# TWljVGltU3RhUENBXzIwMTAtMDctMDEuY3J0MAwGA1UdEwEB/wQCMAAwEwYDVR0l
# BAwwCgYIKwYBBQUHAwgwDQYJKoZIhvcNAQELBQADggEBAF435D6kAS2jeAJ8BG1K
# Tm5Az0jpbdjpqSvMLt7fOVraAEHldgk04BKcTmhzjbTXsjwgCMMCS+jX4Toqi0cn
# zcSoD2LphZA98DXeH6lRH7qQdXbHgx0/vbq0YyVkltSTMv1jzzI75Z5dhpvc4Uwn
# 4Fb6CCaF2/+r7Rr0j+2DGCwl8aWqvQqzhCJ/o7cNoYUfJ4WSCHs1OsjgMmWTmglu
# PIxt3kV8iLZl2IZgyr5cNOiNiTraFDq7hxI16oDsoW0EQKCV84nV1wWSWe1SiAKI
# wr5BtqYwJ+hlocPw5qehWbBiTLntcLrwKdAbwthFr1DHf3RYwFoDzyNtKSB/TJsB
# 2bMwggZxMIIEWaADAgECAgphCYEqAAAAAAACMA0GCSqGSIb3DQEBCwUAMIGIMQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNy
# b3NvZnQgUm9vdCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0xMDA3MDEy
# MTM2NTVaFw0yNTA3MDEyMTQ2NTVaMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
# YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
# Q29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAy
# MDEwMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAqR0NvHcRijog7PwT
# l/X6f2mUa3RUENWlCgCChfvtfGhLLF/Fw+Vhwna3PmYrW/AVUycEMR9BGxqVHc4J
# E458YTBZsTBED/FgiIRUQwzXTbg4CLNC3ZOs1nMwVyaCo0UN0Or1R4HNvyRgMlhg
# RvJYR4YyhB50YWeRX4FUsc+TTJLBxKZd0WETbijGGvmGgLvfYfxGwScdJGcSchoh
# iq9LZIlQYrFd/XcfPfBXday9ikJNQFHRD5wGPmd/9WbAA5ZEfu/QS/1u5ZrKsajy
# eioKMfDaTgaRtogINeh4HLDpmc085y9Euqf03GS9pAHBIAmTeM38vMDJRF1eFpwB
# BU8iTQIDAQABo4IB5jCCAeIwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFNVj
# OlyKMZDzQ3t8RhvFM2hahW1VMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsG
# A1UdDwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFNX2VsuP6KJc
# YmjRPZSQW9fOmhjEMFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6Ly9jcmwubWljcm9z
# b2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIz
# LmNybDBaBggrBgEFBQcBAQROMEwwSgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cubWlj
# cm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3J0
# MIGgBgNVHSABAf8EgZUwgZIwgY8GCSsGAQQBgjcuAzCBgTA9BggrBgEFBQcCARYx
# aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL1BLSS9kb2NzL0NQUy9kZWZhdWx0Lmh0
# bTBABggrBgEFBQcCAjA0HjIgHQBMAGUAZwBhAGwAXwBQAG8AbABpAGMAeQBfAFMA
# dABhAHQAZQBtAGUAbgB0AC4gHTANBgkqhkiG9w0BAQsFAAOCAgEAB+aIUQ3ixuCY
# P4FxAz2do6Ehb7Prpsz1Mb7PBeKp/vpXbRkws8LFZslq3/Xn8Hi9x6ieJeP5vO1r
# VFcIK1GCRBL7uVOMzPRgEop2zEBAQZvcXBf/XPleFzWYJFZLdO9CEMivv3/Gf/I3
# fVo/HPKZeUqRUgCvOA8X9S95gWXZqbVr5MfO9sp6AG9LMEQkIjzP7QOllo9ZKby2
# /QThcJ8ySif9Va8v/rbljjO7Yl+a21dA6fHOmWaQjP9qYn/dxUoLkSbiOewZSnFj
# nXshbcOco6I8+n99lmqQeKZt0uGc+R38ONiU9MalCpaGpL2eGq4EQoO4tYCbIjgg
# tSXlZOz39L9+Y1klD3ouOVd2onGqBooPiRa6YacRy5rYDkeagMXQzafQ732D8OE7
# cQnfXXSYIghh2rBQHm+98eEA3+cxB6STOvdlR3jo+KhIq/fecn5ha293qYHLpwms
# ObvsxsvYgrRyzR30uIUBHoD7G4kqVDmyW9rIDVWZeodzOwjmmC3qjeAzLhIp9cAv
# VCch98isTtoouLGp25ayp0Kiyc8ZQU3ghvkqmqMRZjDTu3QyS99je/WZii8bxyGv
# WbWu3EQ8l1Bx16HSxVXjad5XwdHeMMD9zOZN+w2/XU/pnR4ZOC+8z1gFLu8NoFA1
# 2u8JJxzVs341Hgi62jbb01+P3nSISRKhggLXMIICQAIBATCCAQChgdikgdUwgdIx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1p
# Y3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlvbnMgTGltaXRlZDEmMCQGA1UECxMdVGhh
# bGVzIFRTUyBFU046OEQ0MS00QkY3LUIzQjcxJTAjBgNVBAMTHE1pY3Jvc29mdCBU
# aW1lLVN0YW1wIFNlcnZpY2WiIwoBATAHBgUrDgMCGgMVAAclkdn1j1gXgdyvYj41
# B8rkNZ4IoIGDMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
# b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwDQYJ
# KoZIhvcNAQEFBQACBQDkzY+JMCIYDzIwMjEwODIzMTEwNzUzWhgPMjAyMTA4MjQx
# MTA3NTNaMHcwPQYKKwYBBAGEWQoEATEvMC0wCgIFAOTNj4kCAQAwCgIBAAICGjoC
# Af8wBwIBAAICETwwCgIFAOTO4QkCAQAwNgYKKwYBBAGEWQoEAjEoMCYwDAYKKwYB
# BAGEWQoDAqAKMAgCAQACAwehIKEKMAgCAQACAwGGoDANBgkqhkiG9w0BAQUFAAOB
# gQASbFSKDanvHW36vQQ/DqyydO7oKIy3rjAmawDRI1gPSRUZihHzxeZ08QJgMPrM
# oYehqHQHw1TdS6KrhJSozyPyuIJt2R3oW0IRRfgqSIs+zRuJIpzjCak16oSKK36r
# HXq8ORKWxFfGHxwE9VApummo2Z+uueKr3qYklZfzMRCthDGCAw0wggMJAgEBMIGT
# MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMT
# HU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAABOo2NMfd3SUnCAAAA
# AAE6MA0GCWCGSAFlAwQCAQUAoIIBSjAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQ
# AQQwLwYJKoZIhvcNAQkEMSIEIFkaHUvskI3rxQL2euE+aWHPzUlX6LKTjXV6aEL4
# ewfmMIH6BgsqhkiG9w0BCRACLzGB6jCB5zCB5DCBvQQgn6/QhAepLF/7Bdsvfu8G
# OT+ihL9c4cgo5Nf1aUN8tG0wgZgwgYCkfjB8MQswCQYDVQQGEwJVUzETMBEGA1UE
# CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
# b2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQ
# Q0EgMjAxMAITMwAAATqNjTH3d0lJwgAAAAABOjAiBCAdD83x4W7akByAJfmcNZgD
# bBEj2yJ7G0yTbpL64Y/xwjANBgkqhkiG9w0BAQsFAASCAQBAPJbyAz9x1wWvq870
# wXUG9NVx2k0YI/bMcqB5Z9dr5piszKAFasiS9faU8nWxzYNh6P12j0UlfJbOHwAw
# acUt9JfLiA4Sg//L4zLpseq1LWtrZo6FvvqEWMSNP3EOzgWKQjK4FWi8gWcUpVBt
# GmulJOp2m/mnaxgTJ/ueivcLMKsZ640XOcwpJZ4dsexuew7j30cnKF9CdkYEb/4r
# Tl9FbAeXF7hbBa68ItMJayCikl/iaoyCV0p0A1ETBiSbi7YiV7ytZycGMWioxMwK
# kz9ak5Rppi+RcdEFqSPtffxR1l1Y+W1kqiGnQ3rD+Fv+Ls+bX/BoE4Uu94m9Nb6D
# qCIz
# SIG # End signature block
