﻿param
(
    [Parameter(Mandatory=$true)]
    $config,
    [Parameter(Mandatory=$true)]
    $log
)

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking

function Set-RegistryValues($settings)
{
    New-Item -Path HKLM:\SOFTWARE\Microsoft\Dynamics -Name Deployment -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name DeploymentDirectory -PropertyType String -Value $($settings.'Infrastructure.WebRoot') -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name BinDir -PropertyType String -Value $($settings.'Common.BinDir') -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name metadatadirectory -PropertyType String -Value $($settings.'Aos.MetadataDirectory') -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name frameworkdirectory -PropertyType String -Value $($settings.'Aos.PackageDirectory') -Force -Confirm:$false
}

function Configure-ClaimIssuers($webConfig, $settings)
{
    Write-Log "Start processing claim issuer restrictions."
    $ClaimIssuers = $settings.'Infrastructure.ClaimIssuers'
    if (![System.String]::IsNullOrWhitespace($ClaimIssuers))
    {
        $Issuers = $ClaimIssuers.Split(';')
        Write-Log "Processing claim issuer restrictions: $Issuers"

        #add missing elements to the web.config
        if ($Issuers.Count -gt 0)
        {
            $ClaimIssuerRestrictions = $webConfig.SelectNodes("configuration/claimIssuerRestrictions")
            if ($ClaimIssuerRestrictions -eq $null)
            {
                Write-Log "Web config does not contain element configuration/claimIssuerRestrictions. Creating."
                $ClaimIssuerRestrictions = $webConfig.CreateElement("claimIssuerRestrictions")
                $web.configuration.AppendChild($ClaimIssuerRestrictions)
            }

            $IssuerRestrictions = $ClaimIssuerRestrictions.issuerRestrictions
            if ($IssuerRestrictions -eq $null)
            {
                Write-Log "Web config does not contain element configuration/claimIssuerRestrictions/issuerRestrictions. Creating."
                $IssuerRestrictions = $webConfig.CreateElement("issuerRestrictions")
                $ClaimIssuerRestrictions.AppendChild($IssuerRestrictions)
            }
        }

        #add claim issuer restrictions
        foreach($Issuer in $Issuers)
        {
            $Tokens = $Issuer.Split(':')
            if ($Tokens.Count -lt 2 -or $Tokens.Count -gt 3)
            {
                throw "Claim issuer restriction is not in a valid format (expected format: issuerName:userId[,userId,userId,...[:defaultUserId]]): $Issuer"
            }

            $IssuerName = $Tokens[0]
            $AllowedUserIds = $Tokens[1]

            Write-Log "Processing issuer $IssuerName."

            $IssuerRestriction = $IssuerRestrictions.add | where { ($_.name -eq $IssuerName) }
            if ($IssuerRestriction -eq $null)
            {
                Write-Log "Creating new claim issuer restriction for issuer $IssuerName."

                $IssuerRestriction = $webConfig.CreateElement("add")
                $IssuerRestriction.SetAttribute("name", $IssuerName)
                $IssuerRestrictions.AppendChild($IssuerRestriction) | Out-Null
            }
            else
            {
                Write-Log "Claim issuer restriction already exists for issuer $IssuerName. Overriding with new restrictions."
            }

            $IssuerRestriction.SetAttribute("allowedUserIds", $AllowedUserIds)
            Write-Log "User ids $AllowedUserIds were set for $IssuerName."

            if ($Tokens.Count -eq 3)
            {
                $DefaultUserId = $Tokens[2]
                Write-Log "Setting default user id $DefaultUserId for $IssuerName."
                $IssuerRestriction.SetAttribute("defaultUserId", $DefaultUserId)
            }
        }
    }

    Write-Log "Finished processing claim issuer restrictions."
}

function Configure-MachineKeys($webConfig, $settings)
{
    Write-Log "Start processing machine keys."
    $DecryptionKey = $settings.'Infrastructure.MachineDecryptionKey'
    $ValidationKey = $settings.'Infrastructure.MachineValidationKey'

    if ($DecryptionKey -eq $null)
    {
        Write-Log "DecryptionKey not provided, skipping configuration."
        return
    }

    if ($ValidationKey -eq $null)
    {
        Write-Log "ValidationKey not provided, skipping configuration."
        return
    }

    $Web = $webConfig.SelectNodes("configuration/location[@path='.']/system.web")

    $MachineKey = $Web.machineKey

    if ($MachineKey -eq $null)
    {
        Write-Log "Creating machineKey element."

        $MachineKey = $webConfig.CreateElement("machineKey")
        $MachineKey.SetAttribute("decryption", "AES")
        $MachineKey.SetAttribute("decryptionKey", $DecryptionKey)
        $MachineKey.SetAttribute("validation", "HMACSHA256")
        $MachineKey.SetAttribute("validationKey", $ValidationKey)
        $Web.AppendChild($MachineKey) | Out-Null
    }
    else
    {
        Write-Log "Updating machineKey element."
        $MachineKey.decryption = "AES"
        $MachineKey.decryptionKey = $DecryptionKey
        $MachineKey.validation = "HMACSHA256"
        $MachineKey.validationKey = $ValidationKey
    }

    Write-Log "Finished processing machine keys."
}

function CreateFlightingCacheFolder($settings)
{
    Write-Log "Creating the Flighting Cache Folder for Application Platform Flighting"
    $webroot= $settings.'Infrastructure.WebRoot'
    $flightingcachefolder = $settings.'DataAccess.FlightingServiceCacheFolder'
    $flightingcachepath = Join-Path $webroot $flightingcachefolder

    if(!(Test-Path $flightingcachepath))
    {
        Write-Log "Creating $flightingcachepath."
        New-Item -ItemType Directory -Path $flightingcachepath -Force
    }

    #Flighting is enabled for SyncEngine.exe, it may be running by Admin,
    #so "Network Service" must be added at beginning in case W3Wp can not access it later.

    $User = "NETWORK SERVICE"
    $FullControlPermission = "FullControl"
    $Permission = (Get-Acl $flightingcachepath).Access | ?{$_.IdentityReference -match $User -and $_.FileSystemRights -match $FullControlPermission} | Select IdentityReference,FileSystemRights
    if ($Permission)
    {
        $Permission | % {Write-Log "User $($_.IdentityReference) has '$($_.FileSystemRights)' rights on folder $flightingcachepath"}
    }
    else
    {
        Write-Log "$User doesn't have full control on $flightingcachepath, adding $User"
        $Acl = Get-Acl $flightingcachepath

        $NetWorkService = New-Object System.Security.AccessControl.FileSystemAccessRule($User, $FullControlPermission, "ContainerInherit,ObjectInherit", "None", "Allow")

        $Acl.SetAccessRule($NetWorkService)
        Set-Acl $flightingcachepath $Acl

        Write-Log "added $User on $flightingcachepath"
    }
}

function Configure-FlightingCachePath($webConfig, $settings)
{
    Write-Log "Start processing Flighting Cache Path."

    #add missing elements to the web.config
    $add = $webConfig.configuration.appSettings.add | where key -eq 'DataAccess.FlightingCachePath' | select -First 1
    if ($add -eq $null)
    {
        $webRootDirectory = ($webConfig.configuration.appSettings.add | Where {$_.Key -eq 'Infrastructure.WebRoot'}).Value
        if ([System.String]::IsNullOrWhitespace($webRootDirectory))
        {
            Write-Log "Could not find web root!!!!"
        }
        else
        {
            $flightingcachefolder = $settings.'DataAccess.FlightingServiceCacheFolder'
            $flightingcachepath = Join-Path $webRootDirectory $flightingcachefolder
            $add = $webConfig.CreateElement("add")
            $add.SetAttribute("key", "DataAccess.FlightingCachePath")
            $add.SetAttribute("value", $flightingcachepath)
            $webConfig.configuration.appSettings.AppendChild($add) | Out-Null
        }
    }
}

function Update-WifServicesConfig($settings)
{
    $siteAdmin = $settings.'Provisioning.AdminPrincipalName'

    Write-Log "Reading wif.services.config at $($settings.'Infrastructure.WebRoot')"

    [xml]$wifServicesConfig = Get-Content "$($settings.'Infrastructure.WebRoot')\wif.services.config"
    [Uri]$uri = New-Object Uri $($settings.'Aad.AADLoginWsfedEndpointFormat' -f $($siteAdmin.Split('@')[1]))

    $wifServicesConfig.'system.identityModel.services'.federationConfiguration.wsFederation.issuer = "$($uri.AbsoluteUri)"

    $HostUri = New-Object Uri $settings.'Infrastructure.HostUrl'
    $wifServicesConfig.'system.identityModel.services'.federationConfiguration.cookieHandler.domain = $HostUri.DnsSafeHost

    $wifServicesConfig.Save("$($settings.'Infrastructure.WebRoot')\wif.services.config")
}

function Update-AADTenantInfo($settings, $web)
{
    Write-Log "Reading AAD metadata"

    $aadMetadataLocation = $($settings.'Aad.AADMetadataLocationFormat' -f $($settings.'Provisioning.AdminPrincipalName'.Split('@')[1]))
    Write-Log "AAD metadata location [$aadMetadataLocation]"

    $tempAadMetadataFile = [System.IO.Path]::GetTempFileName()
    Write-Log "Metadata will be stored in temporary file [$tempAadMetadataFile]"

    Write-Log "Invoking request"
    Invoke-WebRequest $aadMetadataLocation -UseBasicParsing -OutFile $tempAadMetadataFile

    Write-Log "Reading metadata from file at [$tempAadMetadataFile]"
    [xml]$aadMetadata = Get-Content $tempAadMetadataFile

    Write-Log "Removing temporary file at [$tempAadMetadataFile]"
    Remove-Item $tempAadMetadataFile -Force -Confirm:$false

    Write-Log "Extracting tenant guid from metadata"
    $url = New-Object System.Uri($aadMetadata.EntityDescriptor.entityId)

    $add = $web.configuration.appSettings.add | where key -eq 'Aad.TenantDomainGUID' | select -First 1
    $add.value = $url.AbsolutePath.Trim('/')

    Write-Log "Tenant guid updated"
}

function Update-AADTrustedIssuers($web)
{
    Write-Log "Start updating TrustedIssuers settings"

    $aadIssuerNameFormatNode = $web.configuration.appSettings.add | where key -eq 'Aad.AADIssuerNameFormat' | select -First 1
    $ppe = $aadIssuerNameFormatNode.value -like 'https://sts.windows-ppe.net/{0}/'

    $aadName = if($ppe) { 'PPE' } else { 'PROD' }
    Write-Log "TrustedIssuers settings will be applied for $aadName AAD"

    $trustedIssuersNode = $web.configuration.appSettings.add | where key -eq 'Aad.TrustedIssuers' | select -First 1
    Write-Log "Current TrustedIssuers settings = '$($trustedIssuersNode.value)'"

    $trustedIssuers = @($trustedIssuersNode.value -split ';' | ? { $_ })

    $requiredTrustedIssuers = 'https://sts.windows.net/72f988bf-86f1-41af-91ab-2d7cd011db47/', # MSFT tenant
                              'https://sts.windows.net/975f013f-7f24-47e8-a7d3-abc4752bf346/'  # PME tenant

    if($ppe) {
        # For dev/test/tie environments Retail test tenant should be a trusted issuer
        $requiredTrustedIssuers += 'https://sts.windows.net/34b54b07-9b04-4369-839c-380f28f3e0fe/' # Dynamics 365 - Retail - Test (dynamics365RetailTest.onmicrosoft.com)
    } else {
        $requiredTrustedIssuers += 'https://sts.windows.net/57206206-ec82-4579-9724-0a098ed1b99f/' # Dynamics 365 - Retail (dynamics365Retail.onmicrosoft.com)
    }

    foreach($issuer in $requiredTrustedIssuers) {
        if($issuer -notin $trustedIssuers) {
            $trustedIssuers += $issuer
        }
    }

    $updatedTrustedIssuers = $trustedIssuers -join ';'
    Write-Log "Storing updated TrustedIssuers = $updatedTrustedIssuers"

    if(-not $trustedIssuersNode) {
        $trustedIssuersNode = $web.CreateElement("add")
        $trustedIssuersNode.SetAttribute('key', 'Aad.TrustedIssuers')
        $trustedIssuersNode.SetAttribute('value', $updatedTrustedIssuers)
        $web.configuration.appSettings.AppendChild($trustedIssuersNode) | Out-Null
    }
    else {
        $trustedIssuersNode.value = $updatedTrustedIssuers
    }

    $trustedIssuersMetadataAddress = 'https://login.windows.net/common/FederationMetadata/2007-06/FederationMetadata.xml'
    Write-Log "Storing $trustedIssuersMetadataAddress address"

    $trustedIssuersMetadataAddressNode = $web.configuration.appSettings.add | where key -eq 'Aad.TrustedIssuersMetadataAddress' | select -First 1
    if(-not $trustedIssuersMetadataAddressNode) {
        $trustedIssuersMetadataAddressNode = $web.CreateElement("add")
        $trustedIssuersMetadataAddressNode.SetAttribute('key', 'Aad.TrustedIssuersMetadataAddress')
        $trustedIssuersMetadataAddressNode.SetAttribute('value', $trustedIssuersMetadataAddress)
        $web.configuration.appSettings.AppendChild($trustedIssuersMetadataAddressNode) | Out-Null
    }
    else {
        $trustedIssuersMetadataAddressNode.value = $trustedIssuersMetadataAddress
    }
}

function Update-DataAccessFlightingInfo($settings, $web)
{
    Write-Log "Reading DataAccess Flighting metadata"

    $flightingEnvironment = $settings.'DataAccess.FlightingEnvironment'
    Write-Log "flighting Environment [$flightingEnvironment]"

    $flightingCertificateThumbprint = $settings.'DataAccess.FlightingCertificateThumbprint'
    Write-Log "flighting Certificate Thumbprint [$flightingCertificateThumbprint]"

    $flightingServiceCatalogID = $settings.'DataAccess.FlightingServiceCatalogID'
    Write-Log "flighting ServiceCatalogID [$flightingServiceCatalogID]"

    $add = $web.configuration.appSettings.add | where key -eq 'DataAccess.FlightingEnvironment' | select -First 1
    $add.value = $flightingEnvironment

    $add = $web.configuration.appSettings.add | where key -eq 'DataAccess.FlightingCertificateThumbprint' | select -First 1
    $add.value = $flightingCertificateThumbprint

    $add = $web.configuration.appSettings.add | where key -eq 'DataAccess.FlightingServiceCatalogID' | select -First 1
    $add.value = $flightingServiceCatalogID

    Write-Log "DataAccess Flighting Infomation updated"
}

function Update-WifConfigWithPerfSdkCustomTokenHandler($settings, $wif)
{
    Write-Log "Checking if wif.config needs to be updated with PerfSDK custom token handler"

    $updateWifConfigForPerfSdkFlagString = $settings.'WIF.UpdateWifConfigForPerfSdk'
    $updateWifConfigForPerfSdkFlagBool = $null

    if(![string]::IsNullOrEmpty($updateWifConfigForPerfSdkFlagString))
    {
        if([bool]::TryParse($updateWifConfigForPerfSdkFlagString, [ref]$updateWifConfigForPerfSdkFlagBool))
        {
            if($updateWifConfigForPerfSdkFlagBool)
            {
                Write-Log "Adding custom token handler"

                $securityTokenHandlerConfiguration = $wif.SelectSingleNode("system.identityModel/identityConfiguration/securityTokenHandlers")

                $removeNode = $wif.CreateElement("remove")
                $removeNode.SetAttribute("type","System.IdentityModel.Tokens.Saml2SecurityTokenHandler, System.IdentityModel, Version=4.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089")

                $addNode = $wif.CreateElement("add")
                $addNode.SetAttribute("type","MS.Dynamics.TestTools.CloudCommonTestUtilities.Authentication.PerfSdkSaml2TokenHandler, MS.Dynamics.TestTools.CloudCommonTestUtilities")

                $securityTokenHandlerConfiguration.AppendChild($removeNode)
                $securityTokenHandlerConfiguration.AppendChild($addNode)
            }
            else
            {
                Write-Log "WIF.UpdateWifConfigForPerfSdk is false. Custom token handler is not added"
            }
        }
        else
        {
            Write-Log "WIF.UpdateWifConfigForPerfSdk cannot be parsed as a boolean value. Custom token handler is not added"
        }
    }
    else
    {
        Write-Log "WIF.UpdateWifConfigForPerfSdk is null or empty. Custom token handler is not added"
    }
}

function Update-WebConfig($settings)
{
    Write-Log "Reading web.config at $($settings.'Infrastructure.WebRoot')"

    [xml]$web = Get-Content "$($settings.'Infrastructure.WebRoot')\web.config"

    $uri = New-Object Uri $settings.'Infrastructure.HostUrl'

    $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.FullyQualifiedDomainName' | select -First 1
    $add.value = $uri.DnsSafeHost

    $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.HostedServiceName' | select -First 1
    $add.value = $uri.DnsSafeHost.Split('.')[0]

    $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.HostName' | select -First 1
    $add.value = $uri.DnsSafeHost

    $siteAdmin = $settings.'Provisioning.AdminPrincipalName'
    $add = $web.configuration.appSettings.add | where key -eq 'Aad.AADTenantId' | select -First 1
    $add.value = $siteAdmin.Split('@')[1]

    Update-AADTenantInfo $settings $web

    Update-DataAccessFlightingInfo $settings $web

    Update-AADTrustedIssuers $web

    $ClickOnceUnpackDir = $settings.'Infrastructure.ClickonceAppsDirectory'

    Write-Log "Click-once package dir $ClickOnceUnpackDir"

    if ((Test-Path $ClickOnceUnpackDir))
    {
        $ClickOnceTargetDir = Join-Path $settings.'Infrastructure.WebRoot' "apps"
        Write-Log "Click-once package dir $ClickOnceUnpackDir exists, copying click-one packages to webroot $ClickOnceTargetDir"

        if (!(Test-Path $ClickOnceTargetDir))
        {
            New-Item $ClickOnceTargetDir -ItemType Directory -Force
        }

        Copy-Item "$ClickOnceUnpackDir\*" $ClickOnceTargetDir -Recurse -Confirm:$false -Force

        $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.ClickonceAppsDirectory' | select -First 1
        $add.value = "$(split-path $ClickOnceTargetDir -Leaf)"

        Write-Log "Providing read access to the configuration files in APPS folder under web root: required for clickonce packages to work"
        Set-WebConfigurationProperty -pspath "MACHINE/WEBROOT/APPHOST/$($settings.'Infrastructure.ApplicationName')/Apps" -filter "system.webServer/security/requestFiltering/fileExtensions/add[@fileExtension='.config']" -name "allowed" -value "True"
    }

    Configure-ClaimIssuers $web $settings

    Configure-MachineKeys $web $settings

    CreateFlightingCacheFolder $settings
    Configure-FlightingCachePath $web $settings

    $web.Save("$($settings.'Infrastructure.WebRoot')\web.config")
}

function Update-WifConfig($settings)
{
    $siteAdmin = $settings.'Provisioning.AdminPrincipalName'

    Write-Log "Reading wif.config at $($settings.'Infrastructure.WebRoot')"

    $wifConfig = (Get-Content "$($settings.'Infrastructure.WebRoot')\wif.config").Replace('TENANT_ID_PLACEHOLDER',"$($siteAdmin.Split('@')[1])")
    $wifConfig = $wifConfig.Replace('https://sts.windows.net/',$settings.'Provisioning.AdminIdentityProvider')

    Add-Type -AssemblyName "System.Xml"

    Write-Log "Adding dynamic trusted certificates to wif.config"
    [xml]$wif = Get-Content "$($settings.'Infrastructure.WebRoot')\wif.config"

    $issuerNameRegistry = $wif.SelectSingleNode("system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry")
    $authorities = $issuerNameRegistry.SelectNodes("authority")
    $TrustedCerts = $($settings.'Infrastructure.TrustedCertificates').Split(';')

    $TrustedIssuer = $null
    $TokenIssuer = $authorities | where { $_.name -eq $($settings.'Aad.AcsTokenIssuer') }
    $TokenIssuerKeys = $null

    if ($TokenIssuer -ne $null)
    {
        $TokenIssuerKeys = $TokenIssuer.SelectSingleNode("keys")
    }

    foreach($TrustedCert in $TrustedCerts)
    {
        foreach ($auth in $authorities)
        {
            $TrustedIssuer = $auth.keys.add | where thumbprint -eq $TrustedCert

            if ($TrustedIssuer -ne $null)
            {
                break
            }
        }

        if ($TrustedIssuer -eq $null)
        {
            if ($TokenIssuer -eq $null)
            {
                $TokenIssuer = $wif.CreateElement("authority")
                $TokenIssuer.SetAttribute("name",$($settings.'Aad.AcsTokenIssuer'))

                $TokenIssuerKeys = $wif.CreateElement("keys")
                $TokenIssuer.AppendChild($TokenIssuerKeys)

                $ValidIssuers = $wif.CreateElement("validIssuers")

                $newAdd = $wif.CreateElement("add")
                $newAdd.SetAttribute("name", $($settings.'Aad.AcsTokenIssuer'))
                $ValidIssuers.AppendChild($newAdd)

                $TokenIssuer.AppendChild($ValidIssuers)

                $issuerNameRegistry.AppendChild($TokenIssuer) | Out-Null
            }

            $newAdd = $wif.CreateElement("add")
            $newAdd.SetAttribute("thumbprint", $TrustedCert)
            $TokenIssuerKeys.AppendChild($newAdd)
        }
    }

    Write-Log "Removing duplicate authorities in wif.config"
    #we only dedup based on the authority name since a static authority only has one issuer/thumbprint

    $authorities = $issuerNameRegistry.SelectNodes("authority")
    $uniqueAuthorities  = New-Object 'System.Collections.Generic.HashSet[System.Xml.XmlElement]'
    foreach($auth in $authorities)
    {
        $existingAuth = $uniqueAuthorities | Where-Object {$_.name -eq $auth.name}
        if ($existingAuth -eq $null)
        {
            $newAuth = $wif.CreateElement("authority")
            $newAuth.SetAttribute("name",$auth.name)
            $newAuth.InnerXml = $auth.InnerXml
            $uniqueAuthorities.Add($newAuth) | Out-Null
        }
    }

    $type = $issuerNameRegistry.type
    $issuerNameRegistry.RemoveAll()
    $issuerNameRegistry.SetAttribute("type", $type)

    foreach($auth in $uniqueAuthorities)
    {
        $issuerNameRegistry.AppendChild($auth) | Out-Null
    }

    Update-WifConfigWithPerfSdkCustomTokenHandler $settings $wif

    $wif.Save("$($settings.'Infrastructure.WebRoot')\wif.config")
}

Initialize-Log $log
Write-Log "Decoding settings"
$settings = Decode-Settings $config

Write-Log "Updating AOS web.config"
Update-WebConfig $settings
Write-Log "AOS web.config update complete"

Write-Log "Updating AOS wif.config"
Update-WifConfig $settings
Write-Log "AOS wif.config update complete"

Write-Log "Updating AOS web.services.config"
Update-WifServicesConfig $settings
Write-Log "AOS web.services.config update complete"

#Temporary removing this:
#if these are present DBSync assumes we are in local onebox mode and behaves differently
#Set-RegistryValues $settings

Write-Log "Restarting IIS"
iisreset

Write-Log "Finished, exiting"

# SIG # Begin signature block
# MIIjnwYJKoZIhvcNAQcCoIIjkDCCI4wCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBIJatmlmZTMd2x
# Q7AXCOdB1XmDTyoqcdmjSiKF5bvBOKCCDYEwggX/MIID56ADAgECAhMzAAAB32vw
# LpKnSrTQAAAAAAHfMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAxMjE1MjEzMTQ1WhcNMjExMjAyMjEzMTQ1WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQC2uxlZEACjqfHkuFyoCwfL25ofI9DZWKt4wEj3JBQ48GPt1UsDv834CcoUUPMn
# s/6CtPoaQ4Thy/kbOOg/zJAnrJeiMQqRe2Lsdb/NSI2gXXX9lad1/yPUDOXo4GNw
# PjXq1JZi+HZV91bUr6ZjzePj1g+bepsqd/HC1XScj0fT3aAxLRykJSzExEBmU9eS
# yuOwUuq+CriudQtWGMdJU650v/KmzfM46Y6lo/MCnnpvz3zEL7PMdUdwqj/nYhGG
# 3UVILxX7tAdMbz7LN+6WOIpT1A41rwaoOVnv+8Ua94HwhjZmu1S73yeV7RZZNxoh
# EegJi9YYssXa7UZUUkCCA+KnAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUOPbML8IdkNGtCfMmVPtvI6VZ8+Mw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDYzMDA5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAnnqH
# tDyYUFaVAkvAK0eqq6nhoL95SZQu3RnpZ7tdQ89QR3++7A+4hrr7V4xxmkB5BObS
# 0YK+MALE02atjwWgPdpYQ68WdLGroJZHkbZdgERG+7tETFl3aKF4KpoSaGOskZXp
# TPnCaMo2PXoAMVMGpsQEQswimZq3IQ3nRQfBlJ0PoMMcN/+Pks8ZTL1BoPYsJpok
# t6cql59q6CypZYIwgyJ892HpttybHKg1ZtQLUlSXccRMlugPgEcNZJagPEgPYni4
# b11snjRAgf0dyQ0zI9aLXqTxWUU5pCIFiPT0b2wsxzRqCtyGqpkGM8P9GazO8eao
# mVItCYBcJSByBx/pS0cSYwBBHAZxJODUqxSXoSGDvmTfqUJXntnWkL4okok1FiCD
# Z4jpyXOQunb6egIXvkgQ7jb2uO26Ow0m8RwleDvhOMrnHsupiOPbozKroSa6paFt
# VSh89abUSooR8QdZciemmoFhcWkEwFg4spzvYNP4nIs193261WyTaRMZoceGun7G
# CT2Rl653uUj+F+g94c63AhzSq4khdL4HlFIP2ePv29smfUnHtGq6yYFDLnT0q/Y+
# Di3jwloF8EWkkHRtSuXlFUbTmwr/lDDgbpZiKhLS7CBTDj32I0L5i532+uHczw82
# oZDmYmYmIUSMbZOgS65h797rj5JJ6OkeEUJoAVwwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVdDCCFXACAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAd9r8C6Sp0q00AAAAAAB3zAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQg4+XnYZJs
# hApHfYlO7rsoHzK0hIEPLX0NfSajLObWECYwQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQA0kAwEDpCu0hiOTjJH6sZYIB2FkwSagDH7Z3BJMFfn
# o/UzWGs9SAY/HA4llGvTYWoC2VYqphDp15wB5+VnV4UTfmDNbNb5lK4gCzJHbxsO
# uNnU47XR2KsRZLVz+exftV64A2IrUGGs2YJ/lOL1QQIvxpH2JFsHzy5bztnJ39c4
# 5iR0qbWAvhmTvX86bYfCRC6OmJJfF2BQuHV5url5GJlF6mBze/ET8a44zteiEul2
# XlY3Jwo4tzNrWu6yAq2SC6rv2Ok3w5jK9V/kfaOc/YHJTrVCqTxFysLib1ESNiNR
# J9MuDa+YbO6Fs07tskLOw2CwD/1f+kG6n0XjcMGV7vwFoYIS/jCCEvoGCisGAQQB
# gjcDAwExghLqMIIS5gYJKoZIhvcNAQcCoIIS1zCCEtMCAQMxDzANBglghkgBZQME
# AgEFADCCAVkGCyqGSIb3DQEJEAEEoIIBSASCAUQwggFAAgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIKngh796/sUZ1rqJ6Zoa0BobQIQmNNWYCxzMEV1e
# FUaDAgZg1KH4oQYYEzIwMjEwODIzMDcyMDMwLjA5MVowBIACAfSggdikgdUwgdIx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1p
# Y3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlvbnMgTGltaXRlZDEmMCQGA1UECxMdVGhh
# bGVzIFRTUyBFU046OEQ0MS00QkY3LUIzQjcxJTAjBgNVBAMTHE1pY3Jvc29mdCBU
# aW1lLVN0YW1wIFNlcnZpY2Wggg5NMIIE+TCCA+GgAwIBAgITMwAAATqNjTH3d0lJ
# wgAAAAABOjANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Eg
# MjAxMDAeFw0yMDEwMTUxNzI4MjJaFw0yMjAxMTIxNzI4MjJaMIHSMQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQg
# SXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxlcyBUU1Mg
# RVNOOjhENDEtNEJGNy1CM0I3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFt
# cCBTZXJ2aWNlMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAzl8k518P
# lz8JTIXYn/O9OakqcWqdJ8ZXJhAks9hyLB8+ANW7Zngb1t7iw7TmgeooOwMnbhCQ
# QH14UwWd8hQFWexKqVpcIFnY3b15+PYmgVeQ4XKfWJ3PPMjTiXu73epXHj9XX7mh
# S2IVqwEvDOudOI3yQL8D8OOG24b+10zDDEyN5wvZ5A1Wcvl2eQhCG61GeHNaXvXO
# loTQblVFbMWOmGviHvgRlRhRjgNmuv1J2y6fQFtiEw0pdXKCQG68xQlBhcu4Ln+b
# YL4HoeT2mrtkpHEyDZ+frr+Ka/zUDP3BscHkKdkNGOODfvJdWHaV0Wzr1wnPuUgt
# ObfnBO0oSjIpBQIDAQABo4IBGzCCARcwHQYDVR0OBBYEFBRWoJ8WXxJrpslvHHWs
# rQmFRfPLMB8GA1UdIwQYMBaAFNVjOlyKMZDzQ3t8RhvFM2hahW1VMFYGA1UdHwRP
# ME0wS6BJoEeGRWh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1
# Y3RzL01pY1RpbVN0YVBDQV8yMDEwLTA3LTAxLmNybDBaBggrBgEFBQcBAQROMEww
# SgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMv
# TWljVGltU3RhUENBXzIwMTAtMDctMDEuY3J0MAwGA1UdEwEB/wQCMAAwEwYDVR0l
# BAwwCgYIKwYBBQUHAwgwDQYJKoZIhvcNAQELBQADggEBAF435D6kAS2jeAJ8BG1K
# Tm5Az0jpbdjpqSvMLt7fOVraAEHldgk04BKcTmhzjbTXsjwgCMMCS+jX4Toqi0cn
# zcSoD2LphZA98DXeH6lRH7qQdXbHgx0/vbq0YyVkltSTMv1jzzI75Z5dhpvc4Uwn
# 4Fb6CCaF2/+r7Rr0j+2DGCwl8aWqvQqzhCJ/o7cNoYUfJ4WSCHs1OsjgMmWTmglu
# PIxt3kV8iLZl2IZgyr5cNOiNiTraFDq7hxI16oDsoW0EQKCV84nV1wWSWe1SiAKI
# wr5BtqYwJ+hlocPw5qehWbBiTLntcLrwKdAbwthFr1DHf3RYwFoDzyNtKSB/TJsB
# 2bMwggZxMIIEWaADAgECAgphCYEqAAAAAAACMA0GCSqGSIb3DQEBCwUAMIGIMQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNy
# b3NvZnQgUm9vdCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0xMDA3MDEy
# MTM2NTVaFw0yNTA3MDEyMTQ2NTVaMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
# YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
# Q29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAy
# MDEwMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAqR0NvHcRijog7PwT
# l/X6f2mUa3RUENWlCgCChfvtfGhLLF/Fw+Vhwna3PmYrW/AVUycEMR9BGxqVHc4J
# E458YTBZsTBED/FgiIRUQwzXTbg4CLNC3ZOs1nMwVyaCo0UN0Or1R4HNvyRgMlhg
# RvJYR4YyhB50YWeRX4FUsc+TTJLBxKZd0WETbijGGvmGgLvfYfxGwScdJGcSchoh
# iq9LZIlQYrFd/XcfPfBXday9ikJNQFHRD5wGPmd/9WbAA5ZEfu/QS/1u5ZrKsajy
# eioKMfDaTgaRtogINeh4HLDpmc085y9Euqf03GS9pAHBIAmTeM38vMDJRF1eFpwB
# BU8iTQIDAQABo4IB5jCCAeIwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFNVj
# OlyKMZDzQ3t8RhvFM2hahW1VMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsG
# A1UdDwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFNX2VsuP6KJc
# YmjRPZSQW9fOmhjEMFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6Ly9jcmwubWljcm9z
# b2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIz
# LmNybDBaBggrBgEFBQcBAQROMEwwSgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cubWlj
# cm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3J0
# MIGgBgNVHSABAf8EgZUwgZIwgY8GCSsGAQQBgjcuAzCBgTA9BggrBgEFBQcCARYx
# aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL1BLSS9kb2NzL0NQUy9kZWZhdWx0Lmh0
# bTBABggrBgEFBQcCAjA0HjIgHQBMAGUAZwBhAGwAXwBQAG8AbABpAGMAeQBfAFMA
# dABhAHQAZQBtAGUAbgB0AC4gHTANBgkqhkiG9w0BAQsFAAOCAgEAB+aIUQ3ixuCY
# P4FxAz2do6Ehb7Prpsz1Mb7PBeKp/vpXbRkws8LFZslq3/Xn8Hi9x6ieJeP5vO1r
# VFcIK1GCRBL7uVOMzPRgEop2zEBAQZvcXBf/XPleFzWYJFZLdO9CEMivv3/Gf/I3
# fVo/HPKZeUqRUgCvOA8X9S95gWXZqbVr5MfO9sp6AG9LMEQkIjzP7QOllo9ZKby2
# /QThcJ8ySif9Va8v/rbljjO7Yl+a21dA6fHOmWaQjP9qYn/dxUoLkSbiOewZSnFj
# nXshbcOco6I8+n99lmqQeKZt0uGc+R38ONiU9MalCpaGpL2eGq4EQoO4tYCbIjgg
# tSXlZOz39L9+Y1klD3ouOVd2onGqBooPiRa6YacRy5rYDkeagMXQzafQ732D8OE7
# cQnfXXSYIghh2rBQHm+98eEA3+cxB6STOvdlR3jo+KhIq/fecn5ha293qYHLpwms
# ObvsxsvYgrRyzR30uIUBHoD7G4kqVDmyW9rIDVWZeodzOwjmmC3qjeAzLhIp9cAv
# VCch98isTtoouLGp25ayp0Kiyc8ZQU3ghvkqmqMRZjDTu3QyS99je/WZii8bxyGv
# WbWu3EQ8l1Bx16HSxVXjad5XwdHeMMD9zOZN+w2/XU/pnR4ZOC+8z1gFLu8NoFA1
# 2u8JJxzVs341Hgi62jbb01+P3nSISRKhggLXMIICQAIBATCCAQChgdikgdUwgdIx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1p
# Y3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlvbnMgTGltaXRlZDEmMCQGA1UECxMdVGhh
# bGVzIFRTUyBFU046OEQ0MS00QkY3LUIzQjcxJTAjBgNVBAMTHE1pY3Jvc29mdCBU
# aW1lLVN0YW1wIFNlcnZpY2WiIwoBATAHBgUrDgMCGgMVAAclkdn1j1gXgdyvYj41
# B8rkNZ4IoIGDMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
# b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwDQYJ
# KoZIhvcNAQEFBQACBQDkzY+JMCIYDzIwMjEwODIzMTEwNzUzWhgPMjAyMTA4MjQx
# MTA3NTNaMHcwPQYKKwYBBAGEWQoEATEvMC0wCgIFAOTNj4kCAQAwCgIBAAICGjoC
# Af8wBwIBAAICETwwCgIFAOTO4QkCAQAwNgYKKwYBBAGEWQoEAjEoMCYwDAYKKwYB
# BAGEWQoDAqAKMAgCAQACAwehIKEKMAgCAQACAwGGoDANBgkqhkiG9w0BAQUFAAOB
# gQASbFSKDanvHW36vQQ/DqyydO7oKIy3rjAmawDRI1gPSRUZihHzxeZ08QJgMPrM
# oYehqHQHw1TdS6KrhJSozyPyuIJt2R3oW0IRRfgqSIs+zRuJIpzjCak16oSKK36r
# HXq8ORKWxFfGHxwE9VApummo2Z+uueKr3qYklZfzMRCthDGCAw0wggMJAgEBMIGT
# MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMT
# HU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAABOo2NMfd3SUnCAAAA
# AAE6MA0GCWCGSAFlAwQCAQUAoIIBSjAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQ
# AQQwLwYJKoZIhvcNAQkEMSIEIGOl9TSvTOeihnhKEjb++oZzrJg+yoSY/lq0uMPz
# ryVfMIH6BgsqhkiG9w0BCRACLzGB6jCB5zCB5DCBvQQgn6/QhAepLF/7Bdsvfu8G
# OT+ihL9c4cgo5Nf1aUN8tG0wgZgwgYCkfjB8MQswCQYDVQQGEwJVUzETMBEGA1UE
# CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
# b2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQ
# Q0EgMjAxMAITMwAAATqNjTH3d0lJwgAAAAABOjAiBCAdD83x4W7akByAJfmcNZgD
# bBEj2yJ7G0yTbpL64Y/xwjANBgkqhkiG9w0BAQsFAASCAQALWQFvzIaTKa3Cf3B4
# 6kfX1OKEtTLUBSwRVQKO0TRwxqVncO+b7CmheOJ5o/iONjqBsw419Fi27LWkpETh
# RTs2tGPnRiyKOVTWWEk33DVH1MlaJmxOhrVMpspXac6bsXHU+ZMZT/W74yxDzXwb
# t8+8M4Ul0uBx567vkG9+RB7xl8vRZiPGVwoixXBprrYeImkK45lZYB5chj5oof7a
# g7fDbb+J8GuPE39PbyoG+H3ibthn3DulZExSRVBI/nwuSwbK2qq2XjXLgQFe/HH9
# DH6/GaMCwwdFfZLe8nuQ91RAaf6HXbUZtQJ2tcSd+IiXs1n1z+Ue6I5o+FMJpXLh
# 7q5I
# SIG # End signature block
