﻿<#
.SYNOPSIS
    Stops the AOS service.

.DESCRIPTION

    Copyright © 2019 Microsoft. All rights reserved.
#>
[CmdletBinding()]
param
(
    [Parameter(Mandatory = $false, HelpMessage = "The path to the directory in which to write log files.")]
    [string]$LogDir
)

$ErrorActionPreference = "Stop"
Import-Module WebAdministration
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\CommonFaultHandling.psm1" -Force

function checkAosKernelFileLocked
{
    Param
    ( 
    [parameter(mandatory=$true,position=0)]
    [string]
    $AosKernelPathParam,
    [parameter(mandatory=$true, position=1)]
    [Int]
    $MaxWaitSecParam 
    )

    $AosKernelLockedFlag = $true

    $StopWatch = [System.Diagnostics.StopWatch]::StartNew()

    while ($AosKernelLockedFlag -and $StopWatch.Elapsed.TotalSeconds -lt $MaxWaitSecParam)
    {
        try
        {
            [IO.File]::OpenWrite($AosKernelPathParam).close()
            $AosKernelLockedFlag = $false
        }
        catch
        {
            Write-ServicingLog "Failed to open $($AosKernelPathParam) for write: $($_.Exception.Message)" -Vrb
            Start-Sleep -Seconds 1
        }
    }

    $StopWatch.Stop()
    return $AosKernelLockedFlag
}

# Initialize exit code.
[int]$ExitCode = 0

# Initialize the log file to use with Write-ServicingLog.
Set-ServicingLog -LogDir $LogDir

try
{
    Write-ServicingLog "Stopping AOS..."

    # For non-admin developer machine scenarios, import module to replace functions that
    # will not work within running as administrator.
    if (Test-Path "$($PSScriptRoot)\NonAdminDevToolsInterject.ps1")
    {
        & "$($PSScriptRoot)\NonAdminDevToolsInterject.ps1"
    }

    # Sometimes Get-Process will throw an exception even when called with a
    # SilentlyContinue error action.
    try
    {
        $devenv = Get-Process -Name "devenv" -ErrorAction SilentlyContinue
    }
    catch
    {
        Write-ServicingLog "Warning: Unable to get devenv processes: $($_)" -Vrb
    }

    # If any devenv processes are found, throw an exception.
    if ($devenv)
    {
        throw "Please close all instance of Visual Studio to continue with the installation."
    }

    # Determine if running in admin or non-admin mode.
    $IsAdmin = Test-IsRunningAsAdministrator

    # Get IIS service and start it if it is not already running.
    $IisService = Get-Service -Name "W3SVC"
    if ($IisService -and $IisService.Status -ine "Running")
    {
        Write-ServicingLog "IIS service is not running, starting IIS Service..."
        Start-Service -Name "W3SVC"
        Write-ServicingLog "IIS service started."
    }

    # Get the AOS web site and app pool and make sure they are both started.
    $websiteName = Get-AosWebsiteName
    $appPoolName = Get-AosAppPoolName
    if ((![string]::IsNullOrWhitespace($websiteName)) -and (![string]::IsNullOrWhitespace($appPoolName)))
    {
        Write-ServicingLog "Stopping IIS AOS web site."
        Stop-Website -Name $websiteName

        # Check if in stopped state before stopping to avoid error.
        $AosWebAppPoolState = Get-WebAppPoolState -Name $appPoolName
        if ($AosWebAppPoolState.Value -ine "Stopped")
        {
            Write-ServicingLog "Stopping IIS AOS application pool."
            Stop-WebAppPool -Name $appPoolName
        }

        # Set properties on web site and app pool, so they do not restart during reboots / IISRESETs.
        # Note: Stopping / starting via IIS mananger GUI will set these same properties.
        # Note: This is not available for non-admin users.
        if ($IsAdmin)
        {
            Write-ServicingLog "Disabling IIS auto start properties for the AOS website and application pool." -Vrb
            Set-ItemProperty -Path "IIS:\Sites\$webSiteName" -Name serverAutoStart -Value $false
            Set-ItemProperty -Path "IIS:\AppPools\$appPoolName" -Name autoStart -Value $false
        }

        $productConfigurationPoolName = Get-ProductConfigurationAppPoolName
        if (![string]::IsNullOrWhitespace($productConfigurationPoolName))
        {
            # Check if in stopped state before stopping to avoid error.
            $ProductConfigurationAppPoolState = Get-WebAppPoolState $productConfigurationPoolName
            if ($ProductConfigurationAppPoolState.Value -ine "Stopped")
            {
                Write-ServicingLog "Stopping IIS product configuration application pool."
                Stop-WebAppPool -Name $productConfigurationPoolName
            }

            # Set property on app pool, so it does not restart during reboots / IISRESETs.
            # Note: Stopping / starting via IIS mananger GUI will set the same property.
            # Note: This is not available for non-admin users.
            if ($IsAdmin)
            {
                Write-ServicingLog "Disabling IIS auto start property for the product configuration application pool." -Vrb
                Set-ItemProperty -Path "IIS:\AppPools\$productConfigurationPoolName" -Name autoStart -Value $false
            }
        }
    }

    # Try to get a list of all xppc* processes and attempt to stop them.
    try
    {
        $XppcProcesses = @(Get-Process -Name "xppc*" -ErrorAction SilentlyContinue)
        if ($XppcProcesses.Count -gt 0)
        {
            Write-ServicingLog "Stopping $($XppcProcesses.Count) 'xppc*' processes..." -Vrb
            foreach ($XppcProcess in $XppcProcesses)
            {
                Write-ServicingLog "Stopping $($XppcProcess.Name) process ID $($XppcProcess.Id)..." -Vrb
                try
                {
                    $XppcProcess | Stop-Process -Force
                }
                catch
                {
                    Write-ServicingLog "Warning: Failed to stop $($XppcProcess.Name) process ID $($XppcProcess.Id): $($_)" -Vrb
                }
            }
        }
    }
    catch
    {
        Write-ServicingLog "Warning: Unable to get xppc processes to stop: $($_)" -Vrb
    }

    # Stop the batch service and set the startup type to disabled, so it does not get started on reboot.
    Write-ServicingLog "Stopping and disabling the batch service."
    
    # Logger to pass to the retry function
    $logger = {param($message) Write-ServicingLog $message}

    # Block to kill batch in the last iteration of the retry function
    $killBatchProcess = {
        #Kill the batch process if still running at the end of the retries.
        $batchSvcName = "DynamicsAxBatch"
        $batchProcess = Get-WmiObject Win32_Service | where {$_.name -eq $batchSvcName}
        if($batchProcess)
        {
          $batchProcessInstance = $batchProcess | Format-List | Out-String
          Write-Output "Found Batch service: $batchProcessInstance"

          $batchProcessId = $batchProcess.ProcessId

          if($batchProcessId -gt 0)
          {
            Write-Output "Stopping Batch process ID: $batchProcessId"
            Stop-Process -Id $batchProcessId -Force
            Start-Sleep 5
          }

          #Final attempt to stop/disable batch
          Write-Output "Final attempt to stop and disable batch after killing the batch process"
          Stop-ServiceAndDisableStartup -ServiceName $batchSvcName
        }
        else
        {
          throw "Unable to find batch service"
        }
      }

    # Attempt to stop batch for up to 7 minutes with incrimental backoff of 4n seconds between attempts
    $output=""
    Invoke-RetryWithBackoff -Command { Stop-ServiceAndDisableStartup -ServiceName "DynamicsAxBatch" } `
                            -Logger $logger `
                            -TimeoutSeconds 420 `
                            -RetryIntervalSec 4 `
                            -TimeoutAction $killBatchProcess `
                            -CommandOutput ([ref]$output) `
                            -TreatTimeoutActionAsSuccess

    if($output -ne "")
    {
       Write-ServicingLog $output
    }

    # Get the AOS web root and find the path to the AOS kernel DLL and check that there are no file
    # locks on it. Throw an exception if there is a lock and it does not go away within the timeout.
    $webroot = Get-AosWebSitePhysicalPath
    $AosKernelPath = Join-Path -Path $webroot -ChildPath "bin\AOSKernel.dll"
    if (Test-Path -Path $AosKernelPath -ErrorAction SilentlyContinue)
    {
        Write-ServicingLog "Validating that $($AosKernelPath) is not locked..." -Vrb

        $AosKernelLocked = $true
        $MaxWaitSec = 300

        $AosKernelLocked = checkAosKernelFileLocked -AosKernelPathParam $AosKernelPath -MaxWaitSecParam $MaxWaitSec

        if ($AosKernelLocked)
        {            
            $w3wpName = "w3wp"
            $axhostName = "axhost"
            $batchname = "Batch"
            $w3wpOraxHostTerminated  = $false

            # Get the locking processes
            Get-Process | foreach {
                $processVar = $_;
                $_.Modules | foreach {
                    if($_.FileName -eq $AosKernelPath)
                    {                        
                        # Terminate w3wp or axhost or batch process
                        if ($processVar.Name -eq $w3wpName -or $processVar.Name -eq $axhostName -or $processVar.Name -eq $batchname)
                        {
                            Write-ServicingLog "Terminate the locking process: $($processVar.Name) PID:$($processVar.id) Locked file:$($AosKernelPath), and try to continue." -Vrb
                            Stop-Process -Id $processVar.Id -Force                            
                            $w3wpOraxHostTerminated = $true
                        }
                    }
                }
            }

            if ($w3wpOraxHostTerminated)
            {
                # Check again for file locking.
                Write-ServicingLog "Revalidating that $($AosKernelPath) is not locked..." -Vrb
                $AosKernelLocked = checkAosKernelFileLocked -AosKernelPathParam $AosKernelPath -MaxWaitSecParam $MaxWaitSec
            }

            # still locked
            if ($AosKernelLocked)
            {
                $LockingProcessFound = $false
                # Get the locking processes
                Get-Process | foreach {
                    $processVar = $_;
                    $_.Modules | foreach {
                        if($_.FileName -eq $AosKernelPath)
                        {
                            $LockingProcessFound = $true
                            Write-ServicingLog "Locking process: $($processVar.Name) PID:$($processVar.id) Locked file:$($AosKernelPath)" -Vrb
                        }
                    }
                }

                $ErrorMessage = "File locks found on $($AosKernelPath) for more than $($MaxWaitSec) seconds."
                if ($LockingProcessFound)
                {
                    $ErrorMessage += " Check previous logs for identified locking processes."
                }
                else
                {
                    $ErrorMessage += " No locking process was able to be automatically identified. Please manually check which processes are keeping a lock on the file."
                }

                throw $ErrorMessage
            }
            else
            {
                Write-ServicingLog "No file locks found on $($AosKernelPath) after terminating processes." -Vrb
            }
        }
        else
        {
            Write-ServicingLog "No file locks found on $($AosKernelPath)." -Vrb
        }
    }
    else
    {
        Write-ServicingLog "Warning: No AOS kernel DLL found at: $($AosKernelPath)." -Vrb
    }
}
catch
{
    # Ensure non-zero exit code if an exception is caught and no exit code set.
    if ($ExitCode -eq 0)
    {
        $ExitCode = 1024
    }

    $ErrorMessage = "Error during AOS stop: $($_)"

    Write-ServicingLog $ErrorMessage
    Write-ServicingLog $($_) -Vrb
    Write-ServicingLog "AOS stop script failed with exit code: $($ExitCode)."

    # Use throw to indicate error to AXUpdateInstaller.
    # In case of exceptions, the output is not captured, so only the error message and
    # log file contents will be available for diagnostics.
    throw "$($ErrorMessage) [Log: $(Get-ServicingLog)]"
}

Write-ServicingLog "AOS stop script completed with exit code: $($ExitCode)."
exit $ExitCode
# SIG # Begin signature block
# MIInygYJKoZIhvcNAQcCoIInuzCCJ7cCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCA25bYQ0dQc0VH6
# EKu4W9dTbOuaZqZwvKZRVHlVmKgNb6CCDYEwggX/MIID56ADAgECAhMzAAACzI61
# lqa90clOAAAAAALMMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjIwNTEyMjA0NjAxWhcNMjMwNTExMjA0NjAxWjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQCiTbHs68bADvNud97NzcdP0zh0mRr4VpDv68KobjQFybVAuVgiINf9aG2zQtWK
# No6+2X2Ix65KGcBXuZyEi0oBUAAGnIe5O5q/Y0Ij0WwDyMWaVad2Te4r1Eic3HWH
# UfiiNjF0ETHKg3qa7DCyUqwsR9q5SaXuHlYCwM+m59Nl3jKnYnKLLfzhl13wImV9
# DF8N76ANkRyK6BYoc9I6hHF2MCTQYWbQ4fXgzKhgzj4zeabWgfu+ZJCiFLkogvc0
# RVb0x3DtyxMbl/3e45Eu+sn/x6EVwbJZVvtQYcmdGF1yAYht+JnNmWwAxL8MgHMz
# xEcoY1Q1JtstiY3+u3ulGMvhAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUiLhHjTKWzIqVIp+sM2rOHH11rfQw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDcwNTI5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAeA8D
# sOAHS53MTIHYu8bbXrO6yQtRD6JfyMWeXaLu3Nc8PDnFc1efYq/F3MGx/aiwNbcs
# J2MU7BKNWTP5JQVBA2GNIeR3mScXqnOsv1XqXPvZeISDVWLaBQzceItdIwgo6B13
# vxlkkSYMvB0Dr3Yw7/W9U4Wk5K/RDOnIGvmKqKi3AwyxlV1mpefy729FKaWT7edB
# d3I4+hldMY8sdfDPjWRtJzjMjXZs41OUOwtHccPazjjC7KndzvZHx/0VWL8n0NT/
# 404vftnXKifMZkS4p2sB3oK+6kCcsyWsgS/3eYGw1Fe4MOnin1RhgrW1rHPODJTG
# AUOmW4wc3Q6KKr2zve7sMDZe9tfylonPwhk971rX8qGw6LkrGFv31IJeJSe/aUbG
# dUDPkbrABbVvPElgoj5eP3REqx5jdfkQw7tOdWkhn0jDUh2uQen9Atj3RkJyHuR0
# GUsJVMWFJdkIO/gFwzoOGlHNsmxvpANV86/1qgb1oZXdrURpzJp53MsDaBY/pxOc
# J0Cvg6uWs3kQWgKk5aBzvsX95BzdItHTpVMtVPW4q41XEvbFmUP1n6oL5rdNdrTM
# j/HXMRk1KCksax1Vxo3qv+13cCsZAaQNaIAvt5LvkshZkDZIP//0Hnq7NnWeYR3z
# 4oFiw9N2n3bb9baQWuWPswG0Dq9YT9kb+Cs4qIIwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIZnzCCGZsCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAsyOtZamvdHJTgAAAAACzDAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgZ4bxuDkH
# KXOPNZ8537UhGsnZlncfvPMVrQtkGcU5rxIwQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQCWbVPEgO3GXbZ15wMZTqY4OjJLJpRrwcHqZ550ykuf
# GBlSTmbU46J5e+9C8PVhjj6NW2Y4OaTzN0safpa3vnzhJ7k9AU6Meu9/2Kj8GqTz
# h0+q1Qy4kQJnxLUrS9Ut/6gyrDTVzz3qCzURcwlzXsmJcsnCD5tWjaxm6sJgEnrx
# 2graIlH5jLU5KSEOHo9OeaRbY34DhwiTRU6B2UgTfI6C2hd1NURXPqHSBcZzTj9C
# Q/LczgKlQk900qvWrvo303huSi8qBb03NFZSKX4eK4U08+DgaSOjupHuBN7tCMt3
# yJyr7gc7QCasxcYlt3tFTovu1Q3qHEZiVmIUwnQKFUuHoYIXKTCCFyUGCisGAQQB
# gjcDAwExghcVMIIXEQYJKoZIhvcNAQcCoIIXAjCCFv4CAQMxDzANBglghkgBZQME
# AgEFADCCAVkGCyqGSIb3DQEJEAEEoIIBSASCAUQwggFAAgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIBaLhZeLb0nY698QCsRXh6OiCtYUIEq+uiLK3387
# 2yrFAgZjovLqrUMYEzIwMjMwMTA2MjE0NTE0Ljg2NFowBIACAfSggdikgdUwgdIx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1p
# Y3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlvbnMgTGltaXRlZDEmMCQGA1UECxMdVGhh
# bGVzIFRTUyBFU046M0JENC00QjgwLTY5QzMxJTAjBgNVBAMTHE1pY3Jvc29mdCBU
# aW1lLVN0YW1wIFNlcnZpY2WgghF4MIIHJzCCBQ+gAwIBAgITMwAAAbT7gAhEBdIt
# +gABAAABtDANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Eg
# MjAxMDAeFw0yMjA5MjAyMDIyMDlaFw0yMzEyMTQyMDIyMDlaMIHSMQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQg
# SXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxlcyBUU1Mg
# RVNOOjNCRDQtNEI4MC02OUMzMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFt
# cCBTZXJ2aWNlMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAtEemnmUH
# MkIfvOiu27K86ZbwWhksGwV72Dl1uGdqr2pKm+mfzoT+Yngkq9aLEf+XDtADyA+2
# KIZU0iO8WG79eJjzz29flZpBKbKg8xl2P3O9drleuQw3TnNfNN4+QIgjMXpE3txP
# F7M7IRLKZMiOt3FfkFWVmiXJAA7E3OIwJgphg09th3Tvzp8MT8+HOtG3bdrRd/y2
# u8VrQsQTLZiVwTZ6qDYKNT8PQZl7xFrSSO3QzXa91LipZnYOl3siGJDCee1Ba7X1
# i13dQFHxKl5Ff4JzDduOBZ85e2VrpyFy1a3ayGUzBrIw59jhMbjIw9YVcQt9kUWn
# tyCmNk15WybCS+hXpEDDLVj1X5W9snmoW1qu03+unprQjWQaVuO7BfcvQdNVdyKS
# qAeKy1eT2Hcc5n1aAVeXFm6sbVJmZzPQEQR3Jr7W8YcTjkqC5hT2qrYuIcYGOf3P
# j4OqdXm1Qqhuwtskxviv7yy3Z+PxJpxKx+2e6zGRaoQmIlLfg/a42XNVHTf6Wzr5
# k7Q1w7v0uA/sFsgyKmI7HzKHX08xDDSmJooXA5btD6B0lx/Lqs6Qb4KthnA7N2IE
# dJ5sjMIhyHZwBr7fzDskU/+Sgp2UnfqrN1Vda/gb+pmlbJwi8MphvElYzjT7PZK2
# Dm4eorcjx7T2QVe3EIelLuGbxzybblZoRTkCAwEAAaOCAUkwggFFMB0GA1UdDgQW
# BBTLRIXl8ZS4Opy7Eii3Tt44zDLZfjAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJl
# pxtTNRnpcjBfBgNVHR8EWDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NybC9NaWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAx
# MCgxKS5jcmwwbAYIKwYBBQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3Rh
# bXAlMjBQQ0ElMjAyMDEwKDEpLmNydDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQM
# MAoGCCsGAQUFBwMIMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQsFAAOCAgEA
# EtEPBYwpt4JioSq0joGzwqYX6SoNH7YbqpgArdlnrdt6u3ukKREluKEVqS2XajXx
# x0UkXGc4Xi9dp2bSxpuyQnTkq+IQwkg7p1dKrwAa2vdmaNzz3mrSaeUEu40yCThH
# wquQkweoG4eqRRZe19OtVSmDDNC3ZQ6Ig0qz79vivXgy5dFWk4npxA5LxSGR4wBa
# XaIuVhoEa06vd/9/2YsQ99bCiR7SxJRt1XrQ5kJGHUi0Fhgz158qvXgfmq7qNqfq
# fTSmsQRrtbe4Zv/X+qPo/l6ae+SrLkcjRfr0ONV0vFVuNKx6Cb90D5LgNpc9x8V/
# qIHEr+JXbWXW6mARVVqNQCmXlVHjTBjhcXwSmadR1OotcN/sKp2EOM9JPYr86O9Y
# /JAZC9zug9qljKTroZTfYA7LIdcmPr69u1FSD/6ivL6HRHZd/k2EL7FtZwzNcRRd
# FF/VgpkOxHIfqvjXambwoMoT+vtGTtqgoruhhSk0bM1F/pBpi/nPZtVNLGTNaK8W
# t6kscbC9G6f09gz/wBBJOBmvTLPOOT/3taCGSoJoDABWnK+De5pie4KX8BxxKQbJ
# vxz7vRsVJ5R6mGx+Bvav5AjsxvZZw6eQmkI0vPRckxL9TCVCfWS0uyIKmyo6Tdos
# nbBO/osre7r0jS9AH8spEqVlhFcpQNfOg/CvdS2xNVMwggdxMIIFWaADAgECAhMz
# AAAAFcXna54Cm0mZAAAAAAAVMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9v
# dCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0yMTA5MzAxODIyMjVaFw0z
# MDA5MzAxODMyMjVaMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIICIjAN
# BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA5OGmTOe0ciELeaLL1yR5vQ7VgtP9
# 7pwHB9KpbE51yMo1V/YBf2xK4OK9uT4XYDP/XE/HZveVU3Fa4n5KWv64NmeFRiMM
# tY0Tz3cywBAY6GB9alKDRLemjkZrBxTzxXb1hlDcwUTIcVxRMTegCjhuje3XD9gm
# U3w5YQJ6xKr9cmmvHaus9ja+NSZk2pg7uhp7M62AW36MEBydUv626GIl3GoPz130
# /o5Tz9bshVZN7928jaTjkY+yOSxRnOlwaQ3KNi1wjjHINSi947SHJMPgyY9+tVSP
# 3PoFVZhtaDuaRr3tpK56KTesy+uDRedGbsoy1cCGMFxPLOJiss254o2I5JasAUq7
# vnGpF1tnYN74kpEeHT39IM9zfUGaRnXNxF803RKJ1v2lIH1+/NmeRd+2ci/bfV+A
# utuqfjbsNkz2K26oElHovwUDo9Fzpk03dJQcNIIP8BDyt0cY7afomXw/TNuvXsLz
# 1dhzPUNOwTM5TI4CvEJoLhDqhFFG4tG9ahhaYQFzymeiXtcodgLiMxhy16cg8ML6
# EgrXY28MyTZki1ugpoMhXV8wdJGUlNi5UPkLiWHzNgY1GIRH29wb0f2y1BzFa/Zc
# UlFdEtsluq9QBXpsxREdcu+N+VLEhReTwDwV2xo3xwgVGD94q0W29R6HXtqPnhZy
# acaue7e3PmriLq0CAwEAAaOCAd0wggHZMBIGCSsGAQQBgjcVAQQFAgMBAAEwIwYJ
# KwYBBAGCNxUCBBYEFCqnUv5kxJq+gpE8RjUpzxD/LwTuMB0GA1UdDgQWBBSfpxVd
# AF5iXYP05dJlpxtTNRnpcjBcBgNVHSAEVTBTMFEGDCsGAQQBgjdMg30BATBBMD8G
# CCsGAQUFBwIBFjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3Mv
# UmVwb3NpdG9yeS5odG0wEwYDVR0lBAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQC
# BAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYD
# VR0jBBgwFoAU1fZWy4/oolxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZF
# aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9v
# Q2VyQXV0XzIwMTAtMDYtMjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcw
# AoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJB
# dXRfMjAxMC0wNi0yMy5jcnQwDQYJKoZIhvcNAQELBQADggIBAJ1VffwqreEsH2cB
# MSRb4Z5yS/ypb+pcFLY+TkdkeLEGk5c9MTO1OdfCcTY/2mRsfNB1OW27DzHkwo/7
# bNGhlBgi7ulmZzpTTd2YurYeeNg2LpypglYAA7AFvonoaeC6Ce5732pvvinLbtg/
# SHUB2RjebYIM9W0jVOR4U3UkV7ndn/OOPcbzaN9l9qRWqveVtihVJ9AkvUCgvxm2
# EhIRXT0n4ECWOKz3+SmJw7wXsFSFQrP8DJ6LGYnn8AtqgcKBGUIZUnWKNsIdw2Fz
# Lixre24/LAl4FOmRsqlb30mjdAy87JGA0j3mSj5mO0+7hvoyGtmW9I/2kQH2zsZ0
# /fZMcm8Qq3UwxTSwethQ/gpY3UA8x1RtnWN0SCyxTkctwRQEcb9k+SS+c23Kjgm9
# swFXSVRk2XPXfx5bRAGOWhmRaw2fpCjcZxkoJLo4S5pu+yFUa2pFEUep8beuyOiJ
# Xk+d0tBMdrVXVAmxaQFEfnyhYWxz/gq77EFmPWn9y8FBSX5+k77L+DvktxW/tM4+
# pTFRhLy/AsGConsXHRWJjXD+57XQKBqJC4822rpM+Zv/Cuk0+CQ1ZyvgDbjmjJnW
# 4SLq8CdCPSWU5nR0W2rRnj7tfqAxM328y+l7vzhwRNGQ8cirOoo6CGJ/2XBjU02N
# 7oJtpQUQwXEGahC0HVUzWLOhcGbyoYIC1DCCAj0CAQEwggEAoYHYpIHVMIHSMQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNy
# b3NvZnQgSXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxl
# cyBUU1MgRVNOOjNCRDQtNEI4MC02OUMzMSUwIwYDVQQDExxNaWNyb3NvZnQgVGlt
# ZS1TdGFtcCBTZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQBlnNiQ85uX9nN4KRJt/gHk
# Jx4JCKCBgzCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMA0GCSqG
# SIb3DQEBBQUAAgUA52KJGzAiGA8yMDIzMDEwNjE5NDg0M1oYDzIwMjMwMTA3MTk0
# ODQzWjB0MDoGCisGAQQBhFkKBAExLDAqMAoCBQDnYokbAgEAMAcCAQACAgy1MAcC
# AQACAhEnMAoCBQDnY9qbAgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkK
# AwKgCjAIAgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZIhvcNAQEFBQADgYEAHUpd
# b95ne0GFt6y1/HCcJ+MeAKaCcGLla70FSDDtMHFdhIFRe0gNZ2+puOoR2KjL3UN/
# 3aQEPNkpX0YJQzdZn8yXJmSHfeGzUaZqFPSuy04gQQCX7scnMrrMUn1MBtEKZMKi
# aLpiYXncuAKxMaU0rVzlI2zzivyi/KjxPGdABWAxggQNMIIECQIBATCBkzB8MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNy
# b3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAbT7gAhEBdIt+gABAAABtDAN
# BglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMC8G
# CSqGSIb3DQEJBDEiBCBbyoRmnhVbAc65ISGYIe+z6vG2TYHhcVzuqel7FkWdjDCB
# +gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EINPI93vmozBwBlFxvfr/rElreFPR
# 4ux7vXKx2ni3AfcGMIGYMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIw
# MTACEzMAAAG0+4AIRAXSLfoAAQAAAbQwIgQgVzouNU9EvwqpWfnCWT5ztgfJn1Fy
# Zc/1ArY8qXIx43gwDQYJKoZIhvcNAQELBQAEggIAlhdv/IHTGHBLV2mWGY0VWdfi
# gEJEqFgTQPsbSWkm5kXR7uQvGCc3NdzDpWLe3UREVPfBtzRQov6qnqCMYcDwKgqu
# N+pfPoxfc/Y9Odvwc06jR0T42p2akJjdU9YzVrS+YOU+S/9PmDIB1oEri+TS2vKE
# Y5xdLUYvuu7gy0JUm2tMGTPL1BkOWI1H5l3xHqvskXSwGB21OotH2ZsmMrwVkbRX
# pkTOumEh8e4WHLcwtSgpnYyfhWQZ6J+WfpRz4OBVs3bQ6JsvzG4GH5CXVuC/z5Q1
# ekqZtUdutDq5r1lE2A2EbZcNoo/4LUuFEOP1Gd7lS5fgOsS9GJB4+m3UHG6cLwKx
# TKCA4Rwr82d5KpsMHB4OAXAipzHB4oCIm3+9arCpKf9wyktc21KrMgwJIHTZFucE
# 00skn3Mgq/ytfGCD3lNOWAHLyduwd/aNrZ/w+DGjy7ZeGqiFva0ie1DcorPRL/TC
# z+19HxH601r1P1GekUJoNwQFTRUnub56Zti0nSRL9aVl1p9sFL1R+5SCKRDaJwQg
# IjH40mQda0QnbUskGX8rbI7Y0RvYTJo178v0n0XnKVOAEUKBDPG0Ltz4UtJGpu5i
# weQuCa+2WhKm30EMn+hJpWNkEvqxKesZsowqTRwjecrsC0eKj8aIGnCTQERHV+ZH
# TjqQ9zQVDIdJyCTrE4Q=
# SIG # End signature block
