﻿[CmdletBinding()]
param
(
    [Parameter(Mandatory=$false)]
    [string]$LogDir,

    [parameter(Mandatory=$false)]
    [boolean] $useServiceFabric=$false,

    [Parameter(Mandatory = $false)]
    [string]$ActivityId,

    [Parameter(Mandatory = $false)]
    [string]$RunbookId,

    [Parameter(Mandatory=$false)]
	[hashtable]$AxDRConnectionString,

    [Parameter(Mandatory=$false)]
	[string[]]$EnabledServicingFeatures,

    [Parameter(Mandatory=$false)]
	[Switch]$PreDowntime,

    [Parameter(Mandatory=$false)]
    [hashtable]$Credentials
)

# Pre-defined list of error messages for known customer induced issues
$knownErrorHash = @{
    "The CREATE UNIQUE INDEX statement terminated because a duplicate key" = 131079
    "Please drop the original field, sync the table" = 131089
    "extension cannot be applied to the base element" = 131090
    "is missing the following dependencies" = 131091
    "Method not found" = 131092
}

# Check pre check status
function IsPreCheckSucceeded ([string]$ActivityId)
{
  if(!$ActivityId)
  {
    return $false
  }

  $axDBUser = Get-DataAccessSqlUsr

  if($Credentials -and $Credentials[$($axDBUser)])  {
     $axDBPwd = $Credentials[$($axDBUser)]
     EventWrite-RunbookScriptTrace -Message "Using axdbadmin password from DSU parameters." -Component "servicing" -RunbookId $RunbookId| Out-Null
  }
  else{
     $axDBPwd = Get-DataAccessSqlPwd
     EventWrite-RunbookScriptTrace -Message "Using axdbadmin password from Web config." -Component "servicing" -RunbookId $RunbookId| Out-Null
  }
  
  $axDBDatabaseName = Get-DataAccessDatabase
  $axDBServer = Get-DataAccessDbServer

    $target_sqlParams = @{
		'Database' = $axDBDatabaseName
		'UserName' = $axDBUser
		'Password' = $axDBPwd
		'ServerInstance' = $axDBServer
		'EncryptConnection' = !(IsLocalServer($axDBServer))
		'QueryTimeout' = 0
	    }

    $target_sqlParams.Query = "Select top 1 Success from DBSyncExecStats where SyncStep = 'PreCheck' and RelatedActivityId = '$ActivityId' order by StartDateTime desc"

    try{
        # during first precheck on this environment table will not be exists so catch exception and consider running precheck.
        $result = Invoke-SqlCmd @target_sqlParams
    }
    catch
    {
        $warningMessage = "[$(Get-Date)]:Get pre check status failed, activity id: $ActivityId, Exception: [$($_.Exception)]"
        Write-Host $warningMessage
        EventWrite-RunbookScriptTrace -Message $warningMessage -Component "servicing" -RunbookId $RunbookId| Out-Null
    }

    if ($result -ne $null)
    {
        return $result.Success
    }

    return $false
}

# Run db sync precheck to evaluate DB Sync would fail or succeed.
function RunDBSyncPreCheck([string]$ActivityId, [hashtable]$AxDRConnectionString, [string[]]$EnabledServicingFeatures)
{
    if (!$EnabledServicingFeatures -or $EnabledServicingFeatures -notcontains "DBSyncPreCheck")
    { 
       EventWrite-RunbookScriptTrace -Message "DBSyncPreCheck feature is disabled. Skipping db sync pre check, activity id: $ActivityId" -Component "servicing"  -RunbookId $RunbookId| Out-Null
       return $false
    }

    # Check if previous precheck status for activity id
    if($ActivityId)
    {
        $PreCheckResult = IsPreCheckSucceeded -ActivityId $ActivityId
        if($PreCheckResult -eq $true) 
        {
           # For prod scenario precheck will run during preparation, so we do not need to run it again.
           EventWrite-RunbookScriptTrace -Message "[$(Get-Date)]: Pre service check already succeeded for activity id: $ActivityId, Skipping precheck..." -Component "servicing" -RunbookId $RunbookId| Out-Null
           return $false
        }
     }

    # Run Precheck
    $currentDateTime = [DateTime]::UtcNow.ToString("yyyyMMddHHmmss")
    $logfile = Join-Path $LogDir "dbsyncprecheck_$currentDateTime.log"
    $errorfile = Join-Path $LogDir "dbsyncprecheck.error_$currentDateTime.log"
    Write-Host "Precheck Started..."
   
    if($PreDowntime -eq $true)
    {
      $aosWebServiceStagingPath = Get-AosServiceStagingPath
      $webroot = Join-Path -Path $aosWebServiceStagingPath -ChildPath "WebRoot"
      $PUVersion = Get-ProductPlatformBuildVersion -webroot:$webroot

      $metadatadir = Join-Path -Path $aosWebServiceStagingPath -ChildPath "PackagesLocalDirectory"
    }
    else
    {
      $PUVersion = (Get-ProductPlatformBuildVersion)
      $metadatadir = $(Get-AOSPackageDirectory)
    }

    Write-Host "AOS platform version is: $PUVersion"

    # If PU version >= PU34 then run precheck on secondary DB if exists.
    $runPrecheckOnSecondaryDB = $false
    if($PUVersion -ge 5629 -and $AxDRConnectionString -and $AxDRConnectionString['AxDRServerName'] -and $AxDRConnectionString['AxDRDatabaseName'])
    {
        $runPrecheckOnSecondaryDB =  $true
        $sqlServer =  $AxDRConnectionString['AxDRServerName']
        $sqlDB =  $AxDRConnectionString['AxDRDatabaseName']
        EventWrite-RunbookScriptTrace -Message "[$(Get-Date)]: Running precheck on secondary DB" -Component "servicing" -RunbookId $RunbookId| Out-Null
    }

    # If platform version > PU35 then run precheck with deployment setup exe
    if ($PUVersion -gt 5644)
    {
        EventWrite-RunbookScriptTrace -Message "Running precheck with deployment setup exe, activity id $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null
        $deploymentSetupParameter = "-setupmode preservicecheck -syncmode precheck"
        
        if($ActivityId) 
        {
            $deploymentSetupParameter += " -activityid $ActivityId"
        }

        $deploySetupPs1 = Resolve-Path -Path $deploySetupPs1
        & $deploySetupPs1 -deploymentSetupParameter $deploymentSetupParameter -logfile $logfile -errorfile $errorfile -axDRServerName $sqlServer -axDRDatabaseName $sqlDB -isPreDowntime $PreDowntime | Out-Null
        
        EventWrite-RunbookScriptTrace -Message "[$(Get-Date)]: Pre service check completed using deployment setup for activity id $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null
        return $true
    }
    
    EventWrite-RunbookScriptTrace -Message "Running precheck on Metadata dir: $metadatadir" -Component "servicing" -RunbookId $RunbookId| Out-Null

    # If platform version <= PU35 Run precheck with sync engine exe
    EventWrite-RunbookScriptTrace -Message "Running precheck with sync engine exe, activity id $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null

    $sqlUser = Get-DataAccessSqlUsr
    if($Credentials -and $Credentials[$($sqlUser)]) {
      $sqlPwd = $Credentials[$($sqlUser)]
    }
    else {
      $sqlPwd = Get-DataAccessSqlPwd
    }

    if($runPrecheckOnSecondaryDB -eq $false){
        $sqlServer = Get-DataAccessDbServer
        $sqlDB = Get-DataAccessDatabase
    }

    $connectionString = "Data Source=$($sqlServer); " +
        "Integrated Security=False; " +
        "User Id=$($sqlUser); " +
        "Password='$($sqlPwd)'; " +
        "Initial Catalog=$($sqlDB)"
    
    $command = "$metadatadir\bin\syncengine.exe"
    $commandParameter = "-syncmode=precheck"
    $commandParameter += " -metadatabinaries=$metadatadir"
    $commandParameter +=  " -connect=`"$connectionString`""
    $commandParameter +=  " -enableParallelSync"
                
    if($ActivityId)
    {
        $commandParameter += " -relatedactivityid=$ActivityId"
    }

    $process = Start-Process $command $commandParameter -PassThru -Wait -RedirectStandardOutput "$logfile" -RedirectStandardError "$errorfile"
    Write-Host "Log file location: $logfile"
        
    if ($process.ExitCode -ne 0) {
        $deploymentSetupError = Get-Content "$errorfile"
        throw $deploymentSetupError
    }

    EventWrite-RunbookScriptTrace -Message "[$(Get-Date)]: Pre service check completed using sync engine for activity id $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null
    return $true
}

$ErrorActionPreference = "stop"
Import-Module "$PSScriptRoot\AosEnvironmentUtilities.psm1" -Force -DisableNameChecking
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -ArgumentList $useServiceFabric -DisableNameChecking

if(!$LogDir)
{
    $LogDir = $PSScriptRoot
}

$deploySetupPs1 = "$PSScriptRoot\TriggerDeploymentSetupEngine.ps1"

# PreCheck
try
{
    # This returns false if pre check skipped.
    $result = RunDBSyncPreCheck -ActivityId $ActivityId -AxDRConnectionString $AxDRConnectionString -EnabledServicingFeatures $EnabledServicingFeatures
   
    if($result -eq $true)
    {   
        $message = "[$(Get-Date)]: Pre service check succeeded for activity id: $ActivityId."
    }
    else
    {
      $message = "[$(Get-Date)]: Pre service check did not run, activity id: $ActivityId."
    }

    Write-Host $message
    EventWrite-RunbookScriptTrace -Message $message -Component "servicing" -RunbookId $RunbookId| Out-Null
}
catch
{
    $errorMessage = "[$(Get-Date)]: Pre service check failed for activity id: $ActivityId, Exception: [$($_.Exception)]"    
    EventWrite-RunbookScriptTrace -Message $errorMessage -Component "servicing" -RunbookId $RunbookId | Out-Null

    if ($EnabledServicingFeatures -and $EnabledServicingFeatures -contains "FailDBSyncOnPrecheckFailure")
    { 
       EventWrite-RunbookScriptTrace -Message "FailDBSyncOnPrecheckFailure feature is enabled and db sync pre-check is failed, activity id: $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null

       # Only fail in dbsync pre-check if the error message contains one of the known error strings
       foreach ($knownError in $knownErrorHash.GetEnumerator())
       {
          if($errorMessage.Contains($knownError.Name))
          {
            # In case precheck failure due to a known customer-induced error scenario we do not require PITR during rollback. In precheck success scenario full sync will run and that will calculate PITR required or not.
            Write-IsPITRRequiredDuringRollback -PitrRequired 'false'

            EventWrite-RunbookScriptTrace -Message "DBSync pre-check failed due to a known customer induced error, error code: $($knownError.Value), activity id: $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null
            throw $errorMessage
          }
       }
    }
}
# SIG # Begin signature block
# MIInygYJKoZIhvcNAQcCoIInuzCCJ7cCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDf8pNtqZY7VDtd
# JiquVayV4+X8rN+4YqEWHTNh3X0wYqCCDYEwggX/MIID56ADAgECAhMzAAACzI61
# lqa90clOAAAAAALMMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjIwNTEyMjA0NjAxWhcNMjMwNTExMjA0NjAxWjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQCiTbHs68bADvNud97NzcdP0zh0mRr4VpDv68KobjQFybVAuVgiINf9aG2zQtWK
# No6+2X2Ix65KGcBXuZyEi0oBUAAGnIe5O5q/Y0Ij0WwDyMWaVad2Te4r1Eic3HWH
# UfiiNjF0ETHKg3qa7DCyUqwsR9q5SaXuHlYCwM+m59Nl3jKnYnKLLfzhl13wImV9
# DF8N76ANkRyK6BYoc9I6hHF2MCTQYWbQ4fXgzKhgzj4zeabWgfu+ZJCiFLkogvc0
# RVb0x3DtyxMbl/3e45Eu+sn/x6EVwbJZVvtQYcmdGF1yAYht+JnNmWwAxL8MgHMz
# xEcoY1Q1JtstiY3+u3ulGMvhAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUiLhHjTKWzIqVIp+sM2rOHH11rfQw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDcwNTI5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAeA8D
# sOAHS53MTIHYu8bbXrO6yQtRD6JfyMWeXaLu3Nc8PDnFc1efYq/F3MGx/aiwNbcs
# J2MU7BKNWTP5JQVBA2GNIeR3mScXqnOsv1XqXPvZeISDVWLaBQzceItdIwgo6B13
# vxlkkSYMvB0Dr3Yw7/W9U4Wk5K/RDOnIGvmKqKi3AwyxlV1mpefy729FKaWT7edB
# d3I4+hldMY8sdfDPjWRtJzjMjXZs41OUOwtHccPazjjC7KndzvZHx/0VWL8n0NT/
# 404vftnXKifMZkS4p2sB3oK+6kCcsyWsgS/3eYGw1Fe4MOnin1RhgrW1rHPODJTG
# AUOmW4wc3Q6KKr2zve7sMDZe9tfylonPwhk971rX8qGw6LkrGFv31IJeJSe/aUbG
# dUDPkbrABbVvPElgoj5eP3REqx5jdfkQw7tOdWkhn0jDUh2uQen9Atj3RkJyHuR0
# GUsJVMWFJdkIO/gFwzoOGlHNsmxvpANV86/1qgb1oZXdrURpzJp53MsDaBY/pxOc
# J0Cvg6uWs3kQWgKk5aBzvsX95BzdItHTpVMtVPW4q41XEvbFmUP1n6oL5rdNdrTM
# j/HXMRk1KCksax1Vxo3qv+13cCsZAaQNaIAvt5LvkshZkDZIP//0Hnq7NnWeYR3z
# 4oFiw9N2n3bb9baQWuWPswG0Dq9YT9kb+Cs4qIIwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIZnzCCGZsCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAsyOtZamvdHJTgAAAAACzDAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgLPWykckg
# eyzPZUVRJPlSm+4ktAGi5YWFWPIyVder8t4wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQANfEuEnZH1s98gWekzYPfyCrxuqXYH+qJezpSRbAwg
# WpRcUt6N3+hRY1QvFvmYhwgF1MGJSeKKkhl1jSL8MX80jKtdXv3ToEUX81j5TYLt
# 5HaKa3dKHJky3vtQ2BqAJcb6p5Rg7SA+b1tKP7Z80uW9B1OUv/xOlU8tHJl1F05m
# 9tiU8/gX7bPLx94Nr97qxkfOfzuLcNeMKqAiheredUSNdPklk5/SgSuA5t135jne
# /hJFvQ5C1qJ3LSXPurMYtQ0fnB+Q4VIsTGtlZYDDZKL58D2ngeb+uo1lustHFMv0
# Jbv1wPsThjOobp2GKpWY6BqUu2vCRMwAvcNHP3OYSPsXoYIXKTCCFyUGCisGAQQB
# gjcDAwExghcVMIIXEQYJKoZIhvcNAQcCoIIXAjCCFv4CAQMxDzANBglghkgBZQME
# AgEFADCCAVkGCyqGSIb3DQEJEAEEoIIBSASCAUQwggFAAgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIBRq2FTFYeR6tXtVPAvYaSWzLnq/LDc3qbVjJAi8
# 7iEsAgZjovLqrUsYEzIwMjMwMTA2MjE0NTE1LjEwM1owBIACAfSggdikgdUwgdIx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1p
# Y3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlvbnMgTGltaXRlZDEmMCQGA1UECxMdVGhh
# bGVzIFRTUyBFU046M0JENC00QjgwLTY5QzMxJTAjBgNVBAMTHE1pY3Jvc29mdCBU
# aW1lLVN0YW1wIFNlcnZpY2WgghF4MIIHJzCCBQ+gAwIBAgITMwAAAbT7gAhEBdIt
# +gABAAABtDANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Eg
# MjAxMDAeFw0yMjA5MjAyMDIyMDlaFw0yMzEyMTQyMDIyMDlaMIHSMQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQg
# SXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxlcyBUU1Mg
# RVNOOjNCRDQtNEI4MC02OUMzMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFt
# cCBTZXJ2aWNlMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAtEemnmUH
# MkIfvOiu27K86ZbwWhksGwV72Dl1uGdqr2pKm+mfzoT+Yngkq9aLEf+XDtADyA+2
# KIZU0iO8WG79eJjzz29flZpBKbKg8xl2P3O9drleuQw3TnNfNN4+QIgjMXpE3txP
# F7M7IRLKZMiOt3FfkFWVmiXJAA7E3OIwJgphg09th3Tvzp8MT8+HOtG3bdrRd/y2
# u8VrQsQTLZiVwTZ6qDYKNT8PQZl7xFrSSO3QzXa91LipZnYOl3siGJDCee1Ba7X1
# i13dQFHxKl5Ff4JzDduOBZ85e2VrpyFy1a3ayGUzBrIw59jhMbjIw9YVcQt9kUWn
# tyCmNk15WybCS+hXpEDDLVj1X5W9snmoW1qu03+unprQjWQaVuO7BfcvQdNVdyKS
# qAeKy1eT2Hcc5n1aAVeXFm6sbVJmZzPQEQR3Jr7W8YcTjkqC5hT2qrYuIcYGOf3P
# j4OqdXm1Qqhuwtskxviv7yy3Z+PxJpxKx+2e6zGRaoQmIlLfg/a42XNVHTf6Wzr5
# k7Q1w7v0uA/sFsgyKmI7HzKHX08xDDSmJooXA5btD6B0lx/Lqs6Qb4KthnA7N2IE
# dJ5sjMIhyHZwBr7fzDskU/+Sgp2UnfqrN1Vda/gb+pmlbJwi8MphvElYzjT7PZK2
# Dm4eorcjx7T2QVe3EIelLuGbxzybblZoRTkCAwEAAaOCAUkwggFFMB0GA1UdDgQW
# BBTLRIXl8ZS4Opy7Eii3Tt44zDLZfjAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJl
# pxtTNRnpcjBfBgNVHR8EWDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NybC9NaWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAx
# MCgxKS5jcmwwbAYIKwYBBQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3Rh
# bXAlMjBQQ0ElMjAyMDEwKDEpLmNydDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQM
# MAoGCCsGAQUFBwMIMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQsFAAOCAgEA
# EtEPBYwpt4JioSq0joGzwqYX6SoNH7YbqpgArdlnrdt6u3ukKREluKEVqS2XajXx
# x0UkXGc4Xi9dp2bSxpuyQnTkq+IQwkg7p1dKrwAa2vdmaNzz3mrSaeUEu40yCThH
# wquQkweoG4eqRRZe19OtVSmDDNC3ZQ6Ig0qz79vivXgy5dFWk4npxA5LxSGR4wBa
# XaIuVhoEa06vd/9/2YsQ99bCiR7SxJRt1XrQ5kJGHUi0Fhgz158qvXgfmq7qNqfq
# fTSmsQRrtbe4Zv/X+qPo/l6ae+SrLkcjRfr0ONV0vFVuNKx6Cb90D5LgNpc9x8V/
# qIHEr+JXbWXW6mARVVqNQCmXlVHjTBjhcXwSmadR1OotcN/sKp2EOM9JPYr86O9Y
# /JAZC9zug9qljKTroZTfYA7LIdcmPr69u1FSD/6ivL6HRHZd/k2EL7FtZwzNcRRd
# FF/VgpkOxHIfqvjXambwoMoT+vtGTtqgoruhhSk0bM1F/pBpi/nPZtVNLGTNaK8W
# t6kscbC9G6f09gz/wBBJOBmvTLPOOT/3taCGSoJoDABWnK+De5pie4KX8BxxKQbJ
# vxz7vRsVJ5R6mGx+Bvav5AjsxvZZw6eQmkI0vPRckxL9TCVCfWS0uyIKmyo6Tdos
# nbBO/osre7r0jS9AH8spEqVlhFcpQNfOg/CvdS2xNVMwggdxMIIFWaADAgECAhMz
# AAAAFcXna54Cm0mZAAAAAAAVMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9v
# dCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0yMTA5MzAxODIyMjVaFw0z
# MDA5MzAxODMyMjVaMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIICIjAN
# BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA5OGmTOe0ciELeaLL1yR5vQ7VgtP9
# 7pwHB9KpbE51yMo1V/YBf2xK4OK9uT4XYDP/XE/HZveVU3Fa4n5KWv64NmeFRiMM
# tY0Tz3cywBAY6GB9alKDRLemjkZrBxTzxXb1hlDcwUTIcVxRMTegCjhuje3XD9gm
# U3w5YQJ6xKr9cmmvHaus9ja+NSZk2pg7uhp7M62AW36MEBydUv626GIl3GoPz130
# /o5Tz9bshVZN7928jaTjkY+yOSxRnOlwaQ3KNi1wjjHINSi947SHJMPgyY9+tVSP
# 3PoFVZhtaDuaRr3tpK56KTesy+uDRedGbsoy1cCGMFxPLOJiss254o2I5JasAUq7
# vnGpF1tnYN74kpEeHT39IM9zfUGaRnXNxF803RKJ1v2lIH1+/NmeRd+2ci/bfV+A
# utuqfjbsNkz2K26oElHovwUDo9Fzpk03dJQcNIIP8BDyt0cY7afomXw/TNuvXsLz
# 1dhzPUNOwTM5TI4CvEJoLhDqhFFG4tG9ahhaYQFzymeiXtcodgLiMxhy16cg8ML6
# EgrXY28MyTZki1ugpoMhXV8wdJGUlNi5UPkLiWHzNgY1GIRH29wb0f2y1BzFa/Zc
# UlFdEtsluq9QBXpsxREdcu+N+VLEhReTwDwV2xo3xwgVGD94q0W29R6HXtqPnhZy
# acaue7e3PmriLq0CAwEAAaOCAd0wggHZMBIGCSsGAQQBgjcVAQQFAgMBAAEwIwYJ
# KwYBBAGCNxUCBBYEFCqnUv5kxJq+gpE8RjUpzxD/LwTuMB0GA1UdDgQWBBSfpxVd
# AF5iXYP05dJlpxtTNRnpcjBcBgNVHSAEVTBTMFEGDCsGAQQBgjdMg30BATBBMD8G
# CCsGAQUFBwIBFjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3Mv
# UmVwb3NpdG9yeS5odG0wEwYDVR0lBAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQC
# BAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYD
# VR0jBBgwFoAU1fZWy4/oolxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZF
# aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9v
# Q2VyQXV0XzIwMTAtMDYtMjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcw
# AoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJB
# dXRfMjAxMC0wNi0yMy5jcnQwDQYJKoZIhvcNAQELBQADggIBAJ1VffwqreEsH2cB
# MSRb4Z5yS/ypb+pcFLY+TkdkeLEGk5c9MTO1OdfCcTY/2mRsfNB1OW27DzHkwo/7
# bNGhlBgi7ulmZzpTTd2YurYeeNg2LpypglYAA7AFvonoaeC6Ce5732pvvinLbtg/
# SHUB2RjebYIM9W0jVOR4U3UkV7ndn/OOPcbzaN9l9qRWqveVtihVJ9AkvUCgvxm2
# EhIRXT0n4ECWOKz3+SmJw7wXsFSFQrP8DJ6LGYnn8AtqgcKBGUIZUnWKNsIdw2Fz
# Lixre24/LAl4FOmRsqlb30mjdAy87JGA0j3mSj5mO0+7hvoyGtmW9I/2kQH2zsZ0
# /fZMcm8Qq3UwxTSwethQ/gpY3UA8x1RtnWN0SCyxTkctwRQEcb9k+SS+c23Kjgm9
# swFXSVRk2XPXfx5bRAGOWhmRaw2fpCjcZxkoJLo4S5pu+yFUa2pFEUep8beuyOiJ
# Xk+d0tBMdrVXVAmxaQFEfnyhYWxz/gq77EFmPWn9y8FBSX5+k77L+DvktxW/tM4+
# pTFRhLy/AsGConsXHRWJjXD+57XQKBqJC4822rpM+Zv/Cuk0+CQ1ZyvgDbjmjJnW
# 4SLq8CdCPSWU5nR0W2rRnj7tfqAxM328y+l7vzhwRNGQ8cirOoo6CGJ/2XBjU02N
# 7oJtpQUQwXEGahC0HVUzWLOhcGbyoYIC1DCCAj0CAQEwggEAoYHYpIHVMIHSMQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNy
# b3NvZnQgSXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxl
# cyBUU1MgRVNOOjNCRDQtNEI4MC02OUMzMSUwIwYDVQQDExxNaWNyb3NvZnQgVGlt
# ZS1TdGFtcCBTZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQBlnNiQ85uX9nN4KRJt/gHk
# Jx4JCKCBgzCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMA0GCSqG
# SIb3DQEBBQUAAgUA52KJGzAiGA8yMDIzMDEwNjE5NDg0M1oYDzIwMjMwMTA3MTk0
# ODQzWjB0MDoGCisGAQQBhFkKBAExLDAqMAoCBQDnYokbAgEAMAcCAQACAgy1MAcC
# AQACAhEnMAoCBQDnY9qbAgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkK
# AwKgCjAIAgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZIhvcNAQEFBQADgYEAHUpd
# b95ne0GFt6y1/HCcJ+MeAKaCcGLla70FSDDtMHFdhIFRe0gNZ2+puOoR2KjL3UN/
# 3aQEPNkpX0YJQzdZn8yXJmSHfeGzUaZqFPSuy04gQQCX7scnMrrMUn1MBtEKZMKi
# aLpiYXncuAKxMaU0rVzlI2zzivyi/KjxPGdABWAxggQNMIIECQIBATCBkzB8MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNy
# b3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAbT7gAhEBdIt+gABAAABtDAN
# BglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMC8G
# CSqGSIb3DQEJBDEiBCCpHNhhwWcnwUDBMjo6hQzyVcpWGhewa/I13esVhQy5azCB
# +gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EINPI93vmozBwBlFxvfr/rElreFPR
# 4ux7vXKx2ni3AfcGMIGYMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIw
# MTACEzMAAAG0+4AIRAXSLfoAAQAAAbQwIgQgVzouNU9EvwqpWfnCWT5ztgfJn1Fy
# Zc/1ArY8qXIx43gwDQYJKoZIhvcNAQELBQAEggIAfIJTraxmI6hW2p77kBF6cZKb
# 7aPCXmFX/n1slqh7yTCOnU0K7LWy5sbzUwjyfjoCBVhcOvT/0QcGEpOG+zg+gVci
# 1vkl6PW91sM+ZEoHpnXHS01KHfX1iucrOo234kBXc15v5rLVg1Hgu/PsIGmVw+mc
# j0dn8vlRvvmb1Se9bmiSnNm+/MiccrATEhEch7n7FLVMM8dOksuqTz7oslVdA2FJ
# r2GRjApLkJH3R+3evWE6lQMYN/q8OxC0jH3LBsPPH08WvsrOVSkQ4rsMxkezxTo4
# 0lwDA+UrMdVHAhdyegZ5mdqKU/WKs3W1mKXn9njWeKg1QhG0LuW6drihLRod3OpT
# bKhPIiXOpqOd545gWNWXWzJm5MSfx05aQ+boOrmI1AimK/uVlG7REy0u91HyISk6
# nwx2uJEYx6V4AsowQYP+77cfmq4shd2TJenxI+QTWbdQ5GZOlx5NUMheinGe1tV7
# fUaE8iJs9/RfbUeozOORFaUWrB7v3H1Y9yaofLKSiaNBLGGEdeTiP0Xem/Uwb78Z
# dTpDNYXozUBpdD0MMZaSl9kUQ46r7CCbBtJmRLRRDnrLh6cgzeh8cpDoZAEe2Chv
# djpshNVYstJa7LQbFmstVPItyObzClnRRxtz8dH2+QlNB1f4U5qaWLtX2eevAgA7
# Y8nviJp5LOeWGH2LZpY=
# SIG # End signature block
