﻿param
(
    [Parameter(Mandatory=$true)]
    $config,
    [Parameter(Mandatory=$true)]
    $log
)

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking

function Set-RegistryValues($settings)
{
    New-Item -Path HKLM:\SOFTWARE\Microsoft\Dynamics -Name Deployment -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name DeploymentDirectory -PropertyType String -Value $($settings.'Infrastructure.WebRoot') -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name BinDir -PropertyType String -Value $($settings.'Common.BinDir') -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name metadatadirectory -PropertyType String -Value $($settings.'Aos.MetadataDirectory') -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name frameworkdirectory -PropertyType String -Value $($settings.'Aos.PackageDirectory') -Force -Confirm:$false
}

function Configure-ClaimIssuers($webConfig, $settings)
{
    Write-Log "Start processing claim issuer restrictions."
    $ClaimIssuers = $settings.'Infrastructure.ClaimIssuers'
    if (![System.String]::IsNullOrWhitespace($ClaimIssuers))
    {
        $Issuers = $ClaimIssuers.Split(';')
        Write-Log "Processing claim issuer restrictions: $Issuers"

        #add missing elements to the web.config
        if ($Issuers.Count -gt 0)
        {
            $ClaimIssuerRestrictions = $webConfig.SelectNodes("configuration/claimIssuerRestrictions")
            if ($ClaimIssuerRestrictions -eq $null)
            {
                Write-Log "Web config does not contain element configuration/claimIssuerRestrictions. Creating."
                $ClaimIssuerRestrictions = $webConfig.CreateElement("claimIssuerRestrictions")
                $web.configuration.AppendChild($ClaimIssuerRestrictions)
            }

            $IssuerRestrictions = $ClaimIssuerRestrictions.issuerRestrictions
            if ($IssuerRestrictions -eq $null)
            {
                Write-Log "Web config does not contain element configuration/claimIssuerRestrictions/issuerRestrictions. Creating."
                $IssuerRestrictions = $webConfig.CreateElement("issuerRestrictions")
                $ClaimIssuerRestrictions.AppendChild($IssuerRestrictions)
            }
        }

        #add claim issuer restrictions
        foreach($Issuer in $Issuers)
        {
            $Tokens = $Issuer.Split(':')
            if ($Tokens.Count -lt 2 -or $Tokens.Count -gt 3)
            {
                throw "Claim issuer restriction is not in a valid format (expected format: issuerName:userId[,userId,userId,...[:defaultUserId]]): $Issuer"
            }

            $IssuerName = $Tokens[0]
            $AllowedUserIds = $Tokens[1]

            Write-Log "Processing issuer $IssuerName."

            $IssuerRestriction = $IssuerRestrictions.add | where { ($_.name -eq $IssuerName) }
            if ($IssuerRestriction -eq $null)
            {
                Write-Log "Creating new claim issuer restriction for issuer $IssuerName."

                $IssuerRestriction = $webConfig.CreateElement("add")
                $IssuerRestriction.SetAttribute("name", $IssuerName)
                $IssuerRestrictions.AppendChild($IssuerRestriction) | Out-Null
            }
            else
            {
                Write-Log "Claim issuer restriction already exists for issuer $IssuerName. Overriding with new restrictions."
            }

            $IssuerRestriction.SetAttribute("allowedUserIds", $AllowedUserIds)
            Write-Log "User ids $AllowedUserIds were set for $IssuerName."

            if ($Tokens.Count -eq 3)
            {
                $DefaultUserId = $Tokens[2]
                Write-Log "Setting default user id $DefaultUserId for $IssuerName."
                $IssuerRestriction.SetAttribute("defaultUserId", $DefaultUserId)
            }
        }
    }

    Write-Log "Finished processing claim issuer restrictions."
}

function Configure-MachineKeys($webConfig, $settings)
{
    Write-Log "Start processing machine keys."
    $DecryptionKey = $settings.'Infrastructure.MachineDecryptionKey'
    $ValidationKey = $settings.'Infrastructure.MachineValidationKey'

    if ($DecryptionKey -eq $null)
    {
        Write-Log "DecryptionKey not provided, skipping configuration."
        return
    }

    if ($ValidationKey -eq $null)
    {
        Write-Log "ValidationKey not provided, skipping configuration."
        return
    }

    $Web = $webConfig.SelectNodes("configuration/location[@path='.']/system.web")

    $MachineKey = $Web.machineKey

    if ($MachineKey -eq $null)
    {
        Write-Log "Creating machineKey element."

        $MachineKey = $webConfig.CreateElement("machineKey")
        $MachineKey.SetAttribute("decryption", "AES")
        $MachineKey.SetAttribute("decryptionKey", $DecryptionKey)
        $MachineKey.SetAttribute("validation", "HMACSHA256")
        $MachineKey.SetAttribute("validationKey", $ValidationKey)
        $Web.AppendChild($MachineKey) | Out-Null
    }
    else
    {
        Write-Log "Updating machineKey element."
        $MachineKey.decryption = "AES"
        $MachineKey.decryptionKey = $DecryptionKey
        $MachineKey.validation = "HMACSHA256"
        $MachineKey.validationKey = $ValidationKey
    }

    Write-Log "Finished processing machine keys."
}

function CreateFlightingCacheFolder($settings)
{
    Write-Log "Creating the Flighting Cache Folder for Application Platform Flighting"
    $webroot= $settings.'Infrastructure.WebRoot'
    $flightingcachefolder = $settings.'DataAccess.FlightingServiceCacheFolder'
    $flightingcachepath = Join-Path $webroot $flightingcachefolder

    if(!(Test-Path $flightingcachepath))
    {
        Write-Log "Creating $flightingcachepath."
        New-Item -ItemType Directory -Path $flightingcachepath -Force
    }

    #Flighting is enabled for SyncEngine.exe, it may be running by Admin,
    #so "Network Service" must be added at beginning in case W3Wp can not access it later.

    $User = "NETWORK SERVICE"
    $FullControlPermission = "FullControl"
    $Permission = (Get-Acl $flightingcachepath).Access | ?{$_.IdentityReference -match $User -and $_.FileSystemRights -match $FullControlPermission} | Select IdentityReference,FileSystemRights
    if ($Permission)
    {
        $Permission | % {Write-Log "User $($_.IdentityReference) has '$($_.FileSystemRights)' rights on folder $flightingcachepath"}
    }
    else
    {
        Write-Log "$User doesn't have full control on $flightingcachepath, adding $User"
        $Acl = Get-Acl $flightingcachepath

        $NetWorkService = New-Object System.Security.AccessControl.FileSystemAccessRule($User, $FullControlPermission, "ContainerInherit,ObjectInherit", "None", "Allow")

        $Acl.SetAccessRule($NetWorkService)
        Set-Acl $flightingcachepath $Acl

        Write-Log "added $User on $flightingcachepath"
    }
}

function Configure-FlightingCachePath($webConfig, $settings)
{
    Write-Log "Start processing Flighting Cache Path."

    #add missing elements to the web.config
    $add = $webConfig.configuration.appSettings.add | where key -eq 'DataAccess.FlightingCachePath' | select -First 1
    if ($add -eq $null)
    {
        $webRootDirectory = ($webConfig.configuration.appSettings.add | Where {$_.Key -eq 'Infrastructure.WebRoot'}).Value
        if ([System.String]::IsNullOrWhitespace($webRootDirectory))
        {
            Write-Log "Could not find web root!!!!"
        }
        else
        {
            $flightingcachefolder = $settings.'DataAccess.FlightingServiceCacheFolder'
            $flightingcachepath = Join-Path $webRootDirectory $flightingcachefolder
            $add = $webConfig.CreateElement("add")
            $add.SetAttribute("key", "DataAccess.FlightingCachePath")
            $add.SetAttribute("value", $flightingcachepath)
            $webConfig.configuration.appSettings.AppendChild($add) | Out-Null
        }
    }
}

function Update-WifServicesConfig($settings)
{
    $siteAdmin = $settings.'Provisioning.AdminPrincipalName'

    Write-Log "Reading wif.services.config at $($settings.'Infrastructure.WebRoot')"

    [xml]$wifServicesConfig = Get-Content "$($settings.'Infrastructure.WebRoot')\wif.services.config"
    [Uri]$uri = New-Object Uri $($settings.'Aad.AADLoginWsfedEndpointFormat' -f $($siteAdmin.Split('@')[1]))

    $wifServicesConfig.'system.identityModel.services'.federationConfiguration.wsFederation.issuer = "$($uri.AbsoluteUri)"

    $HostUri = New-Object Uri $settings.'Infrastructure.HostUrl'
    $wifServicesConfig.'system.identityModel.services'.federationConfiguration.cookieHandler.domain = $HostUri.DnsSafeHost

    $wifServicesConfig.Save("$($settings.'Infrastructure.WebRoot')\wif.services.config")
}

function Update-AADTenantInfo($settings, $web)
{
    Write-Log "Reading AAD metadata"

    $aadMetadataLocation = $($settings.'Aad.AADMetadataLocationFormat' -f $($settings.'Provisioning.AdminPrincipalName'.Split('@')[1]))
    Write-Log "AAD metadata location [$aadMetadataLocation]"

    $tempAadMetadataFile = [System.IO.Path]::GetTempFileName()
    Write-Log "Metadata will be stored in temporary file [$tempAadMetadataFile]"

    Write-Log "Invoking request"
    Invoke-WebRequest $aadMetadataLocation -UseBasicParsing -OutFile $tempAadMetadataFile

    Write-Log "Reading metadata from file at [$tempAadMetadataFile]"
    [xml]$aadMetadata = Get-Content $tempAadMetadataFile

    Write-Log "Removing temporary file at [$tempAadMetadataFile]"
    Remove-Item $tempAadMetadataFile -Force -Confirm:$false

    Write-Log "Extracting tenant guid from metadata"
    $url = New-Object System.Uri($aadMetadata.EntityDescriptor.entityId)

    $add = $web.configuration.appSettings.add | where key -eq 'Aad.TenantDomainGUID' | select -First 1
    $add.value = $url.AbsolutePath.Trim('/')

    Write-Log "Tenant guid updated"
}

function Update-AADTrustedIssuers($web)
{
    Write-Log "Start updating TrustedIssuers settings"

    $aadIssuerNameFormatNode = $web.configuration.appSettings.add | where key -eq 'Aad.AADIssuerNameFormat' | select -First 1
    $ppe = $aadIssuerNameFormatNode.value -like 'https://sts.windows-ppe.net/{0}/'

    $aadName = if($ppe) { 'PPE' } else { 'PROD' }
    Write-Log "TrustedIssuers settings will be applied for $aadName AAD"

    $trustedIssuersNode = $web.configuration.appSettings.add | where key -eq 'Aad.TrustedIssuers' | select -First 1
    Write-Log "Current TrustedIssuers settings = '$($trustedIssuersNode.value)'"

    $trustedIssuers = @($trustedIssuersNode.value -split ';' | ? { $_ })

    $requiredTrustedIssuers = 'https://sts.windows.net/72f988bf-86f1-41af-91ab-2d7cd011db47/', # MSFT tenant
                              'https://sts.windows.net/975f013f-7f24-47e8-a7d3-abc4752bf346/'  # PME tenant

    if($ppe) {
        # For dev/test/tie environments Retail test tenant should be a trusted issuer
        $requiredTrustedIssuers += 'https://sts.windows.net/34b54b07-9b04-4369-839c-380f28f3e0fe/' # Dynamics 365 - Retail - Test (dynamics365RetailTest.onmicrosoft.com)
    } else {
        $requiredTrustedIssuers += 'https://sts.windows.net/57206206-ec82-4579-9724-0a098ed1b99f/' # Dynamics 365 - Retail (dynamics365Retail.onmicrosoft.com)
    }

    foreach($issuer in $requiredTrustedIssuers) {
        if($issuer -notin $trustedIssuers) {
            $trustedIssuers += $issuer
        }
    }

    $updatedTrustedIssuers = $trustedIssuers -join ';'
    Write-Log "Storing updated TrustedIssuers = $updatedTrustedIssuers"

    if(-not $trustedIssuersNode) {
        $trustedIssuersNode = $web.CreateElement("add")
        $trustedIssuersNode.SetAttribute('key', 'Aad.TrustedIssuers')
        $trustedIssuersNode.SetAttribute('value', $updatedTrustedIssuers)
        $web.configuration.appSettings.AppendChild($trustedIssuersNode) | Out-Null
    }
    else {
        $trustedIssuersNode.value = $updatedTrustedIssuers
    }

    $trustedIssuersMetadataAddress = 'https://login.windows.net/common/FederationMetadata/2007-06/FederationMetadata.xml'
    Write-Log "Storing $trustedIssuersMetadataAddress address"

    $trustedIssuersMetadataAddressNode = $web.configuration.appSettings.add | where key -eq 'Aad.TrustedIssuersMetadataAddress' | select -First 1
    if(-not $trustedIssuersMetadataAddressNode) {
        $trustedIssuersMetadataAddressNode = $web.CreateElement("add")
        $trustedIssuersMetadataAddressNode.SetAttribute('key', 'Aad.TrustedIssuersMetadataAddress')
        $trustedIssuersMetadataAddressNode.SetAttribute('value', $trustedIssuersMetadataAddress)
        $web.configuration.appSettings.AppendChild($trustedIssuersMetadataAddressNode) | Out-Null
    }
    else {
        $trustedIssuersMetadataAddressNode.value = $trustedIssuersMetadataAddress
    }
}

function Update-DataAccessFlightingInfo($settings, $web)
{
    Write-Log "Reading DataAccess Flighting metadata"

    $flightingEnvironment = $settings.'DataAccess.FlightingEnvironment'
    Write-Log "flighting Environment [$flightingEnvironment]"

    $flightingCertificateThumbprint = $settings.'DataAccess.FlightingCertificateThumbprint'
    Write-Log "flighting Certificate Thumbprint [$flightingCertificateThumbprint]"

    $flightingServiceCatalogID = $settings.'DataAccess.FlightingServiceCatalogID'
    Write-Log "flighting ServiceCatalogID [$flightingServiceCatalogID]"

    $add = $web.configuration.appSettings.add | where key -eq 'DataAccess.FlightingEnvironment' | select -First 1
    $add.value = $flightingEnvironment

    $add = $web.configuration.appSettings.add | where key -eq 'DataAccess.FlightingCertificateThumbprint' | select -First 1
    $add.value = $flightingCertificateThumbprint

    $add = $web.configuration.appSettings.add | where key -eq 'DataAccess.FlightingServiceCatalogID' | select -First 1
    $add.value = $flightingServiceCatalogID

    Write-Log "DataAccess Flighting Infomation updated"
}

function Update-WifConfigWithPerfSdkCustomTokenHandler($settings, $wif)
{
    Write-Log "Checking if wif.config needs to be updated with PerfSDK custom token handler"

    $updateWifConfigForPerfSdkFlagString = $settings.'WIF.UpdateWifConfigForPerfSdk'
    $updateWifConfigForPerfSdkFlagBool = $null

    if(![string]::IsNullOrEmpty($updateWifConfigForPerfSdkFlagString))
    {
        if([bool]::TryParse($updateWifConfigForPerfSdkFlagString, [ref]$updateWifConfigForPerfSdkFlagBool))
        {
            if($updateWifConfigForPerfSdkFlagBool)
            {
                Write-Log "Adding custom token handler"

                $securityTokenHandlerConfiguration = $wif.SelectSingleNode("system.identityModel/identityConfiguration/securityTokenHandlers")

                $removeNode = $wif.CreateElement("remove")
                $removeNode.SetAttribute("type","System.IdentityModel.Tokens.Saml2SecurityTokenHandler, System.IdentityModel, Version=4.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089")

                $addNode = $wif.CreateElement("add")
                $addNode.SetAttribute("type","MS.Dynamics.TestTools.CloudCommonTestUtilities.Authentication.PerfSdkSaml2TokenHandler, MS.Dynamics.TestTools.CloudCommonTestUtilities")

                $securityTokenHandlerConfiguration.AppendChild($removeNode)
                $securityTokenHandlerConfiguration.AppendChild($addNode)
            }
            else
            {
                Write-Log "WIF.UpdateWifConfigForPerfSdk is false. Custom token handler is not added"
            }
        }
        else
        {
            Write-Log "WIF.UpdateWifConfigForPerfSdk cannot be parsed as a boolean value. Custom token handler is not added"
        }
    }
    else
    {
        Write-Log "WIF.UpdateWifConfigForPerfSdk is null or empty. Custom token handler is not added"
    }
}

function Update-WebConfig($settings)
{
    Write-Log "Reading web.config at $($settings.'Infrastructure.WebRoot')"

    [xml]$web = Get-Content "$($settings.'Infrastructure.WebRoot')\web.config"

    $uri = New-Object Uri $settings.'Infrastructure.HostUrl'

    $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.FullyQualifiedDomainName' | select -First 1
    $add.value = $uri.DnsSafeHost

    $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.HostedServiceName' | select -First 1
    $add.value = $uri.DnsSafeHost.Split('.')[0]

    $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.HostName' | select -First 1
    $add.value = $uri.DnsSafeHost

    $siteAdmin = $settings.'Provisioning.AdminPrincipalName'
    $add = $web.configuration.appSettings.add | where key -eq 'Aad.AADTenantId' | select -First 1
    $add.value = $siteAdmin.Split('@')[1]

    Update-AADTenantInfo $settings $web

    Update-DataAccessFlightingInfo $settings $web

    Update-AADTrustedIssuers $web

    $ClickOnceUnpackDir = $settings.'Infrastructure.ClickonceAppsDirectory'

    Write-Log "Click-once package dir $ClickOnceUnpackDir"

    if ((Test-Path $ClickOnceUnpackDir))
    {
        $ClickOnceTargetDir = Join-Path $settings.'Infrastructure.WebRoot' "apps"
        Write-Log "Click-once package dir $ClickOnceUnpackDir exists, copying click-one packages to webroot $ClickOnceTargetDir"

        if (!(Test-Path $ClickOnceTargetDir))
        {
            New-Item $ClickOnceTargetDir -ItemType Directory -Force
        }

        Copy-Item "$ClickOnceUnpackDir\*" $ClickOnceTargetDir -Recurse -Confirm:$false -Force

        $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.ClickonceAppsDirectory' | select -First 1
        $add.value = "$(split-path $ClickOnceTargetDir -Leaf)"

        Write-Log "Providing read access to the configuration files in APPS folder under web root: required for clickonce packages to work"
        Set-WebConfigurationProperty -pspath "MACHINE/WEBROOT/APPHOST/$($settings.'Infrastructure.ApplicationName')/Apps" -filter "system.webServer/security/requestFiltering/fileExtensions/add[@fileExtension='.config']" -name "allowed" -value "True"
    }

    Configure-ClaimIssuers $web $settings

    Configure-MachineKeys $web $settings

    CreateFlightingCacheFolder $settings
    Configure-FlightingCachePath $web $settings

    $web.Save("$($settings.'Infrastructure.WebRoot')\web.config")
}

function Update-WifConfig($settings)
{
    $siteAdmin = $settings.'Provisioning.AdminPrincipalName'

    Write-Log "Reading wif.config at $($settings.'Infrastructure.WebRoot')"

    $wifConfig = (Get-Content "$($settings.'Infrastructure.WebRoot')\wif.config").Replace('TENANT_ID_PLACEHOLDER',"$($siteAdmin.Split('@')[1])")
    $wifConfig = $wifConfig.Replace('https://sts.windows.net/',$settings.'Provisioning.AdminIdentityProvider')

    Add-Type -AssemblyName "System.Xml"

    Write-Log "Adding dynamic trusted certificates to wif.config"
    [xml]$wif = Get-Content "$($settings.'Infrastructure.WebRoot')\wif.config"

    $issuerNameRegistry = $wif.SelectSingleNode("system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry")
    $authorities = $issuerNameRegistry.SelectNodes("authority")
    $TrustedCerts = $($settings.'Infrastructure.TrustedCertificates').Split(';')

    $TrustedIssuer = $null
    $TokenIssuer = $authorities | where { $_.name -eq $($settings.'Aad.AcsTokenIssuer') }
    $TokenIssuerKeys = $null

    if ($TokenIssuer -ne $null)
    {
        $TokenIssuerKeys = $TokenIssuer.SelectSingleNode("keys")
    }

    foreach($TrustedCert in $TrustedCerts)
    {
        foreach ($auth in $authorities)
        {
            $TrustedIssuer = $auth.keys.add | where thumbprint -eq $TrustedCert

            if ($TrustedIssuer -ne $null)
            {
                break
            }
        }

        if ($TrustedIssuer -eq $null)
        {
            if ($TokenIssuer -eq $null)
            {
                $TokenIssuer = $wif.CreateElement("authority")
                $TokenIssuer.SetAttribute("name",$($settings.'Aad.AcsTokenIssuer'))

                $TokenIssuerKeys = $wif.CreateElement("keys")
                $TokenIssuer.AppendChild($TokenIssuerKeys)

                $ValidIssuers = $wif.CreateElement("validIssuers")

                $newAdd = $wif.CreateElement("add")
                $newAdd.SetAttribute("name", $($settings.'Aad.AcsTokenIssuer'))
                $ValidIssuers.AppendChild($newAdd)

                $TokenIssuer.AppendChild($ValidIssuers)

                $issuerNameRegistry.AppendChild($TokenIssuer) | Out-Null
            }

            $newAdd = $wif.CreateElement("add")
            $newAdd.SetAttribute("thumbprint", $TrustedCert)
            $TokenIssuerKeys.AppendChild($newAdd)
        }
    }

    Write-Log "Removing duplicate authorities in wif.config"
    #we only dedup based on the authority name since a static authority only has one issuer/thumbprint

    $authorities = $issuerNameRegistry.SelectNodes("authority")
    $uniqueAuthorities  = New-Object 'System.Collections.Generic.HashSet[System.Xml.XmlElement]'
    foreach($auth in $authorities)
    {
        $existingAuth = $uniqueAuthorities | Where-Object {$_.name -eq $auth.name}
        if ($existingAuth -eq $null)
        {
            $newAuth = $wif.CreateElement("authority")
            $newAuth.SetAttribute("name",$auth.name)
            $newAuth.InnerXml = $auth.InnerXml
            $uniqueAuthorities.Add($newAuth) | Out-Null
        }
    }

    $type = $issuerNameRegistry.type
    $issuerNameRegistry.RemoveAll()
    $issuerNameRegistry.SetAttribute("type", $type)

    foreach($auth in $uniqueAuthorities)
    {
        $issuerNameRegistry.AppendChild($auth) | Out-Null
    }

    Update-WifConfigWithPerfSdkCustomTokenHandler $settings $wif

    $wif.Save("$($settings.'Infrastructure.WebRoot')\wif.config")
}

Initialize-Log $log
Write-Log "Decoding settings"
$settings = Decode-Settings $config

Write-Log "Updating AOS web.config"
Update-WebConfig $settings
Write-Log "AOS web.config update complete"

Write-Log "Updating AOS wif.config"
Update-WifConfig $settings
Write-Log "AOS wif.config update complete"

Write-Log "Updating AOS web.services.config"
Update-WifServicesConfig $settings
Write-Log "AOS web.services.config update complete"

#Temporary removing this:
#if these are present DBSync assumes we are in local onebox mode and behaves differently
#Set-RegistryValues $settings

Write-Log "Restarting IIS"
iisreset

Write-Log "Finished, exiting"

# SIG # Begin signature block
# MIInpQYJKoZIhvcNAQcCoIInljCCJ5ICAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBIJatmlmZTMd2x
# Q7AXCOdB1XmDTyoqcdmjSiKF5bvBOKCCDYUwggYDMIID66ADAgECAhMzAAACzfNk
# v/jUTF1RAAAAAALNMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjIwNTEyMjA0NjAyWhcNMjMwNTExMjA0NjAyWjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDrIzsY62MmKrzergm7Ucnu+DuSHdgzRZVCIGi9CalFrhwtiK+3FIDzlOYbs/zz
# HwuLC3hir55wVgHoaC4liQwQ60wVyR17EZPa4BQ28C5ARlxqftdp3H8RrXWbVyvQ
# aUnBQVZM73XDyGV1oUPZGHGWtgdqtBUd60VjnFPICSf8pnFiit6hvSxH5IVWI0iO
# nfqdXYoPWUtVUMmVqW1yBX0NtbQlSHIU6hlPvo9/uqKvkjFUFA2LbC9AWQbJmH+1
# uM0l4nDSKfCqccvdI5l3zjEk9yUSUmh1IQhDFn+5SL2JmnCF0jZEZ4f5HE7ykDP+
# oiA3Q+fhKCseg+0aEHi+DRPZAgMBAAGjggGCMIIBfjAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQU0WymH4CP7s1+yQktEwbcLQuR9Zww
# VAYDVR0RBE0wS6RJMEcxLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9wZXJh
# dGlvbnMgTGltaXRlZDEWMBQGA1UEBRMNMjMwMDEyKzQ3MDUzMDAfBgNVHSMEGDAW
# gBRIbmTlUAXTgqoXNzcitW2oynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8v
# d3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9NaWNDb2RTaWdQQ0EyMDExXzIw
# MTEtMDctMDguY3JsMGEGCCsGAQUFBwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDov
# L3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNDb2RTaWdQQ0EyMDEx
# XzIwMTEtMDctMDguY3J0MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQELBQADggIB
# AE7LSuuNObCBWYuttxJAgilXJ92GpyV/fTiyXHZ/9LbzXs/MfKnPwRydlmA2ak0r
# GWLDFh89zAWHFI8t9JLwpd/VRoVE3+WyzTIskdbBnHbf1yjo/+0tpHlnroFJdcDS
# MIsH+T7z3ClY+6WnjSTetpg1Y/pLOLXZpZjYeXQiFwo9G5lzUcSd8YVQNPQAGICl
# 2JRSaCNlzAdIFCF5PNKoXbJtEqDcPZ8oDrM9KdO7TqUE5VqeBe6DggY1sZYnQD+/
# LWlz5D0wCriNgGQ/TWWexMwwnEqlIwfkIcNFxo0QND/6Ya9DTAUykk2SKGSPt0kL
# tHxNEn2GJvcNtfohVY/b0tuyF05eXE3cdtYZbeGoU1xQixPZAlTdtLmeFNly82uB
# VbybAZ4Ut18F//UrugVQ9UUdK1uYmc+2SdRQQCccKwXGOuYgZ1ULW2u5PyfWxzo4
# BR++53OB/tZXQpz4OkgBZeqs9YaYLFfKRlQHVtmQghFHzB5v/WFonxDVlvPxy2go
# a0u9Z+ZlIpvooZRvm6OtXxdAjMBcWBAsnBRr/Oj5s356EDdf2l/sLwLFYE61t+ME
# iNYdy0pXL6gN3DxTVf2qjJxXFkFfjjTisndudHsguEMk8mEtnvwo9fOSKT6oRHhM
# 9sZ4HTg/TTMjUljmN3mBYWAWI5ExdC1inuog0xrKmOWVMIIHejCCBWKgAwIBAgIK
# YQ6Q0gAAAAAAAzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNV
# BAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
# c29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlm
# aWNhdGUgQXV0aG9yaXR5IDIwMTEwHhcNMTEwNzA4MjA1OTA5WhcNMjYwNzA4MjEw
# OTA5WjB+MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
# BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYD
# VQQDEx9NaWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG
# 9w0BAQEFAAOCAg8AMIICCgKCAgEAq/D6chAcLq3YbqqCEE00uvK2WCGfQhsqa+la
# UKq4BjgaBEm6f8MMHt03a8YS2AvwOMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc
# 6Whe0t+bU7IKLMOv2akrrnoJr9eWWcpgGgXpZnboMlImEi/nqwhQz7NEt13YxC4D
# dato88tt8zpcoRb0RrrgOGSsbmQ1eKagYw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+
# lD3v++MrWhAfTVYoonpy4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+EGvKhL1nk
# kDstrjNYxbc+/jLTswM9sbKvkjh+0p2ALPVOVpEhNSXDOW5kf1O6nA+tGSOEy/S6
# A4aN91/w0FK/jJSHvMAhdCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rhKEmd
# X4jiJV3TIUs+UsS1Vz8kA/DRelsv1SPjcF0PUUZ3s/gA4bysAoJf28AVs70b1FVL
# 5zmhD+kjSbwYuER8ReTBw3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zd
# sGbiwZeBe+3W7UvnSSmnEyimp31ngOaKYnhfsi+E11ecXL93KCjx7W3DKI8sj0A3
# T8HhhUSJxAlMxdSlQy90lfdu+HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS
# 4NaIjAsCAwEAAaOCAe0wggHpMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBRI
# bmTlUAXTgqoXNzcitW2oynUClTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTAL
# BgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBRyLToCMZBD
# uRQFTuHqp8cx0SOJNDBaBgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFf
# MDNfMjIuY3JsMF4GCCsGAQUFBwEBBFIwUDBOBggrBgEFBQcwAoZCaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFf
# MDNfMjIuY3J0MIGfBgNVHSAEgZcwgZQwgZEGCSsGAQQBgjcuAzCBgzA/BggrBgEF
# BQcCARYzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1h
# cnljcHMuaHRtMEAGCCsGAQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAHAAbwBsAGkA
# YwB5AF8AcwB0AGEAdABlAG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn
# 8oalmOBUeRou09h0ZyKbC5YR4WOSmUKWfdJ5DJDBZV8uLD74w3LRbYP+vj/oCso7
# v0epo/Np22O/IjWll11lhJB9i0ZQVdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0b
# pdS1HXeUOeLpZMlEPXh6I/MTfaaQdION9MsmAkYqwooQu6SpBQyb7Wj6aC6VoCo/
# KmtYSWMfCWluWpiW5IP0wI/zRive/DvQvTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvy
# CInWH8MyGOLwxS3OW560STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn+N4sOiBp
# mLJZiWhub6e3dMNABQamASooPoI/E01mC8CzTfXhj38cbxV9Rad25UAqZaPDXVJi
# hsMdYzaXht/a8/jyFqGaJ+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw3MYb
# BL7fQccOKO7eZS/sl/ahXJbYANahRr1Z85elCUtIEJmAH9AAKcWxm6U/RXceNcbS
# oqKfenoi+kiVH6v7RyOA9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sL
# gOppO6/8MO0ETI7f33VtY5E90Z1WTk+/gFcioXgRMiF670EKsT/7qMykXcGhiJtX
# cVZOSEXAQsmbdlsKgEhr/Xmfwb1tbWrJUnMTDXpQzTGCGXYwghlyAgEBMIGVMH4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01p
# Y3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTECEzMAAALN82S/+NRMXVEAAAAA
# As0wDQYJYIZIAWUDBAIBBQCgga4wGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQw
# HAYKKwYBBAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIOPl
# 52GSbIQKR32JTu67KB8ytISBDy19DX0moyzm1hAmMEIGCisGAQQBgjcCAQwxNDAy
# oBSAEgBNAGkAYwByAG8AcwBvAGYAdKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20wDQYJKoZIhvcNAQEBBQAEggEA1WenE/pSwosLJvoYSSuAPvk0vNG5TpqIuBFM
# Gu/xSh52dH3bb1HJ3lNyTE8Hlwb4R4ZZ8KsJzi5A3+bGikwUass8ydCupD8ZXYC3
# tL4fNb0jYr7YzQIotKKsZ7APYcjN0OSHzt+KWI/4RtLrCQPFZIPG6vQCm1vTwcWy
# s81IPcYKnY2vVHR+snlYO+rKOvkk36wzjVaa9mp8u7NuNiSR+PzuN5gkjzXbbXBn
# DZqcIyuDcTVilUZrhMpOSWG8lVQgyNp/C9MXNuFBuTk6VIBoAA4fAom89oIFxO4f
# xl27wYwu39GbGSF7j7JfYXK6WV2mWQvwTMRa4FBx8LHAi431R6GCFwAwghb8Bgor
# BgEEAYI3AwMBMYIW7DCCFugGCSqGSIb3DQEHAqCCFtkwghbVAgEDMQ8wDQYJYIZI
# AWUDBAIBBQAwggFRBgsqhkiG9w0BCRABBKCCAUAEggE8MIIBOAIBAQYKKwYBBAGE
# WQoDATAxMA0GCWCGSAFlAwQCAQUABCBTBiuOqbHKiQUNrReQ5WRkldpqrPMSldBs
# roaaO4ha8QIGZBMMvtHJGBMyMDIzMDMyNTA2Mzk1NS40NzdaMASAAgH0oIHQpIHN
# MIHKMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQL
# ExxNaWNyb3NvZnQgQW1lcmljYSBPcGVyYXRpb25zMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjozQkJELUUzMzgtRTlBMTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCEVcwggcMMIIE9KADAgECAhMzAAABxjDNLtbTocD0AAEA
# AAHGMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTIyMTEwNDE5MDEzNFoXDTI0MDIwMjE5MDEzNFowgcoxCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJTAjBgNVBAsTHE1pY3Jvc29mdCBBbWVy
# aWNhIE9wZXJhdGlvbnMxJjAkBgNVBAsTHVRoYWxlcyBUU1MgRVNOOjNCQkQtRTMz
# OC1FOUExMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNlMIIC
# IjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA770iOr6v4Hk1m3SZj+1BR/iu
# ofv+f6eVb7Hc21YxzAzro4G6kKXF47YAsEgrWWT1ogvp0IroFm8CdRZTf/DlQ0fb
# NNO9pCA01KJ03zH82Clmk9ze9r1jPJ1ZJaMnsZmAy7VpY9mNqX9dhPvnW1/Zxbbi
# Hv7qwwgw9U2ST5mfcpPutsI/Qr/gLC6aTI3UCYziVPZ/Qfag8NQhKkpHZO3Kr5r8
# 3cy7jz4OWPy5M2WitWv5bJJ5rBTW518QPEzFwzq8e8P722CWKZJFjN8etBgsK05g
# HeHaN9kmlpYJJL84v9JiaX7NFJkORhApEFZiUIaZoLxJt4pcBDzf+WD9UAjRKCrA
# seJ/ckzQvOn95X4Ot4asnOuNhcCdcQWcrZoykFmEaoYkrsD7n/4nFFHwJDKUaBYZ
# ZLwPj7ux48S1Ye+cMccMxdRSjuoG4rqJqpEd6gzfz239v36L+LtOlQhfL5cnclhN
# SWdmKw1THyekH96RNtOpIE7c+9Tnsv1aE9hphejLOJxfsXTkyw0FIdilc0CP7zzU
# sqaCGF2mFXRwL4jfX1RyV7QGKEeOpvGZqQKLHVZbLD32ztW8Lfv99xQ24d/TIxO9
# LReuHsnwotn+8BsCrzu+/24vOcA9Xcpd7kIeW4zmUxhFsv4KCfiqMptZhhMAA0Sq
# Duj23cj10smXROLUnhUCAwEAAaOCATYwggEyMB0GA1UdDgQWBBRxX/lHiShECp1n
# 2lMa6G1uLvNglDAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJlpxtTNRnpcjBfBgNV
# HR8EWDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2Ny
# bC9NaWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcmwwbAYI
# KwYBBQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0ElMjAy
# MDEwKDEpLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
# CSqGSIb3DQEBCwUAA4ICAQALzF724jXugDU486PMBYR7Cc6aHr2nJDnGsnkqwZYm
# jRi28qv6S1Ly772zwm5DI189zgAQ99xhEjW6uYkrX5UrtVu7JUQP6bBBBJ98c28F
# AIPIK3fkQNCv2rxRjfQMJSdcwsJeTK7Ld09OuA5hY4PWCBgJpfY71LcaXz3FR8AN
# PFX6zcKYbgYOZregtpDub34N0QhR7wc/FcmV+g4I3IdTAoMD2/WI5ZsfKTzBUn/U
# 3ApUhNwbOl5YSC+f9S1LStbZLwPzMS+fLPXJUSe7SSvspfSsr/VEe0oQhmaR+5vc
# q+7MLw861WBVhYpJ7TB5YBS5ORO9XdIbcpbBFwcHPmb8iZqSIqW9JpgG76+5NQUL
# PVzZ75z5W2R5ZiyQktiHpMwjX2OO29Z8+nTw2tOsVCcwzH9LoELedv3PjcpbwOyL
# jtm1T4XHYd3qbd9DXoBjNYkSjdi37pNp58u+rITltLKOjjQCJwj1FpnuBY825B5C
# 0uC/NYESEKsTicEjhS/4ujBXLcNGDhVBl2vHE6qY/YW4ky1vcypvUrsG81gpv2+8
# /ihOwg4wTLO7XqikeIiU3ZWAUAoOpTl14tedQqxbHTDveJYR3OU0yKB2xwf87EWC
# Ab0CJimhDmyQaKEvSV0fLW9iVyI0wYcG4V2aVN6TrZ4mr+ffaqDQD9F+HpPhP0pl
# AzCCB3EwggVZoAMCAQICEzMAAAAVxedrngKbSZkAAAAAABUwDQYJKoZIhvcNAQEL
# BQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
# EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNV
# BAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eSAyMDEwMB4X
# DTIxMDkzMDE4MjIyNVoXDTMwMDkzMDE4MzIyNVowfDELMAkGA1UEBhMCVVMxEzAR
# BgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
# Y3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3Rh
# bXAgUENBIDIwMTAwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDk4aZM
# 57RyIQt5osvXJHm9DtWC0/3unAcH0qlsTnXIyjVX9gF/bErg4r25PhdgM/9cT8dm
# 95VTcVrifkpa/rg2Z4VGIwy1jRPPdzLAEBjoYH1qUoNEt6aORmsHFPPFdvWGUNzB
# RMhxXFExN6AKOG6N7dcP2CZTfDlhAnrEqv1yaa8dq6z2Nr41JmTamDu6GnszrYBb
# fowQHJ1S/rboYiXcag/PXfT+jlPP1uyFVk3v3byNpOORj7I5LFGc6XBpDco2LXCO
# Mcg1KL3jtIckw+DJj361VI/c+gVVmG1oO5pGve2krnopN6zL64NF50ZuyjLVwIYw
# XE8s4mKyzbnijYjklqwBSru+cakXW2dg3viSkR4dPf0gz3N9QZpGdc3EXzTdEonW
# /aUgfX782Z5F37ZyL9t9X4C626p+Nuw2TPYrbqgSUei/BQOj0XOmTTd0lBw0gg/w
# EPK3Rxjtp+iZfD9M269ewvPV2HM9Q07BMzlMjgK8QmguEOqEUUbi0b1qGFphAXPK
# Z6Je1yh2AuIzGHLXpyDwwvoSCtdjbwzJNmSLW6CmgyFdXzB0kZSU2LlQ+QuJYfM2
# BjUYhEfb3BvR/bLUHMVr9lxSUV0S2yW6r1AFemzFER1y7435UsSFF5PAPBXbGjfH
# CBUYP3irRbb1Hode2o+eFnJpxq57t7c+auIurQIDAQABo4IB3TCCAdkwEgYJKwYB
# BAGCNxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQUKqdS/mTEmr6CkTxGNSnPEP8v
# BO4wHQYDVR0OBBYEFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMFwGA1UdIARVMFMwUQYM
# KwYBBAGCN0yDfQEBMEEwPwYIKwYBBQUHAgEWM2h0dHA6Ly93d3cubWljcm9zb2Z0
# LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5Lmh0bTATBgNVHSUEDDAKBggrBgEF
# BQcDCDAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYD
# VR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvXzpoYxDBW
# BgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
# bC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYIKwYBBQUH
# AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtp
# L2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDANBgkqhkiG9w0BAQsF
# AAOCAgEAnVV9/Cqt4SwfZwExJFvhnnJL/Klv6lwUtj5OR2R4sQaTlz0xM7U518Jx
# Nj/aZGx80HU5bbsPMeTCj/ts0aGUGCLu6WZnOlNN3Zi6th542DYunKmCVgADsAW+
# iehp4LoJ7nvfam++Kctu2D9IdQHZGN5tggz1bSNU5HhTdSRXud2f8449xvNo32X2
# pFaq95W2KFUn0CS9QKC/GbYSEhFdPSfgQJY4rPf5KYnDvBewVIVCs/wMnosZiefw
# C2qBwoEZQhlSdYo2wh3DYXMuLGt7bj8sCXgU6ZGyqVvfSaN0DLzskYDSPeZKPmY7
# T7uG+jIa2Zb0j/aRAfbOxnT99kxybxCrdTDFNLB62FD+CljdQDzHVG2dY3RILLFO
# Ry3BFARxv2T5JL5zbcqOCb2zAVdJVGTZc9d/HltEAY5aGZFrDZ+kKNxnGSgkujhL
# mm77IVRrakURR6nxt67I6IleT53S0Ex2tVdUCbFpAUR+fKFhbHP+CrvsQWY9af3L
# wUFJfn6Tvsv4O+S3Fb+0zj6lMVGEvL8CwYKiexcdFYmNcP7ntdAoGokLjzbaukz5
# m/8K6TT4JDVnK+ANuOaMmdbhIurwJ0I9JZTmdHRbatGePu1+oDEzfbzL6Xu/OHBE
# 0ZDxyKs6ijoIYn/ZcGNTTY3ugm2lBRDBcQZqELQdVTNYs6FwZvKhggLOMIICNwIB
# ATCB+KGB0KSBzTCByjELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
# EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjElMCMGA1UECxMcTWljcm9zb2Z0IEFtZXJpY2EgT3BlcmF0aW9uczEmMCQGA1UE
# CxMdVGhhbGVzIFRTUyBFU046M0JCRC1FMzM4LUU5QTExJTAjBgNVBAMTHE1pY3Jv
# c29mdCBUaW1lLVN0YW1wIFNlcnZpY2WiIwoBATAHBgUrDgMCGgMVAC01yuYmIVvs
# okSacJmWe8Mu2QFtoIGDMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIw
# MTAwDQYJKoZIhvcNAQEFBQACBQDnyL5kMCIYDzIwMjMwMzI1MDgyNzE2WhgPMjAy
# MzAzMjYwODI3MTZaMHcwPQYKKwYBBAGEWQoEATEvMC0wCgIFAOfIvmQCAQAwCgIB
# AAICBV8CAf8wBwIBAAICEiswCgIFAOfKD+QCAQAwNgYKKwYBBAGEWQoEAjEoMCYw
# DAYKKwYBBAGEWQoDAqAKMAgCAQACAwehIKEKMAgCAQACAwGGoDANBgkqhkiG9w0B
# AQUFAAOBgQCVokgXZfEY89vN00VZn83Avpf3Vba0wWvG0f1bF4RTcjMZqlK6v6ju
# r1WdhJuYKxMxVZqSXymhyiP7nsMME+ESOi+Iki10g5lp+4UZflJXemw+2bvEJH1G
# LusuRgpvGqSOvUUilnsIwm60590wIFhayvQkAcVgo4B3CL12okHgHTGCBA0wggQJ
# AgEBMIGTMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
# VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAk
# BgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAABxjDNLtbT
# ocD0AAEAAAHGMA0GCWCGSAFlAwQCAQUAoIIBSjAaBgkqhkiG9w0BCQMxDQYLKoZI
# hvcNAQkQAQQwLwYJKoZIhvcNAQkEMSIEIK4eolBEJMvX42ndpdSBxV5mx7tJyGQD
# aoRRx4sCMYvNMIH6BgsqhkiG9w0BCRACLzGB6jCB5zCB5DCBvQQgVjETXFXJvZgp
# xiZqk/BwDWA5Pxw6hehULpkFr9rjry0wgZgwgYCkfjB8MQswCQYDVQQGEwJVUzET
# MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
# TWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1T
# dGFtcCBQQ0EgMjAxMAITMwAAAcYwzS7W06HA9AABAAABxjAiBCCkDwRTH/0SiSpF
# 0hP2L/DpoK+xHosxUSupSToH8JsFEzANBgkqhkiG9w0BAQsFAASCAgDrBTucUT/2
# dTkVIAjMsq6T4jE9wEjGz4uKGgEIGbT5orXgORduhgCvgq4nUB4lW9c3kTymEM/a
# WZmVzKyf5EquOfLK+rIyrOkH2SrG1LQFU3i5PvgcKviEkgau4YmdgzKG8RVAN101
# SK9AJ2l11s6xLG1VNCd7UwFgFHvOZyLR0YzLTomHQwtYZ+dEFpy8YmMS+VKwheRe
# NsW6hdTNcjR/8VDGMxGFK5DTHd/giCcbrayWssPaZEzc9kePsm3Uja7BcFcB2Y/W
# Lxh4t7038i7dxtWDhb3doKcLXahjJrZYrZlRBshhhlLD153ZzZG3hiBk2KdQb0/5
# 93nx69Y2iqNLLX750OgqYWAhFR6M6wfOaHhFRDe/TpW4SJibAHx6YfSwZI0p6lsi
# Mi3tLgiIphInzrm4aafGpTvkOASNROVXSfISHoui4Tk35ABU9OYW8v1EUkcIrltX
# XmHi67IkzuraEVl7ZYsX+CAx+0yW9QnhQmDsSFs96UP0sKRn6OJERo8a5qu5OwtN
# etxPg+MPeOOdECmoAVFLZpBwVycezxMltAmuSQR12au8m/VvAB63yOA+Z7OwjPYT
# fpbRxmcnMalzEz6QuE9T9tJdREz6YvN6NfX9WTU3sW5swX/Xtu/phvG2rFICVqE/
# F7YW/K0i9HZUBsARWHpxH2Qm2CueBv5CcA==
# SIG # End signature block
