﻿$global:logfile=""
$NEWSETTING = "###CTP8###"
$BatchService = "DynamicsAxBatch"
$PrintService = "DynamicsAXPrintService"
$DataSigningCertThumbprintLegacyKey="DataAccess.DataSigningCertificateThumbprintLegacy"
$DataEncryptionCertThumbprintLegacyKey="DataAccess.DataEncryptionCertificateThumbprintLegacy"
$DataSigningCertThumbprintKey="DataAccess.DataSigningCertificateThumbprint"
$DataEncryptionCertThumbprintKey="DataAccess.DataEncryptionCertificateThumbprint"
$configEncryptor="Microsoft.Dynamics.AX.Framework.ConfigEncryptor.exe"
$InternalServiceCertThumbprints="Infrastructure.InternalServiceCertificateThumbprints"
$DataAccessAxAdminSqlPwd="DataAccess.AxAdminSqlPwd"
$DataAccessAxAdminSqlUser="DataAccess.AxAdminSqlUser"

$ErrorActionPreference="Stop"

function Initialize-Log([string]$log)
{
    if(Test-Path -Path $log)
    {
        Write-Output "Removing the existing log file '$log'."
        Remove-Item -Path $log -Force|Out-Null
    }

    Write-Output "Creating the log file '$log'."
    New-Item -Path $log -ItemType File -Force|out-null
    $global:logfile=$log
}

function Write-Log([string]$message)
{
    $datetime=Get-Date -Format "MM-dd-yyyy:HH:mm:ss"
    Add-Content -Path $global:logfile -Value "$datetime`: $message"|out-null
    Write-Output "$datetime`: $message"
}

function Log-Error([string]$error,[switch]$throw)
{
    Write-Error $error
    if($throw)
    {
        throw $error
    }
}

function Create-Backup([string]$webroot,[string]$backupdir)
{
    $orig_webconfig= Join-Path -Path $webroot -ChildPath "web.config"
    $orig_wifconfig= Join-Path -Path $webroot -ChildPath "wif.config"
    $orig_wifservicesconfig=Join-Path -Path $webroot -ChildPath "wif.services.config"

    $backup_webconfig= Join-Path -Path $backupdir -ChildPath "web.config.backup"
    $backup_wifconfig= Join-Path -Path $backupdir -ChildPath "wif.config.backup"
    $backup_wifservicesconfig=Join-Path -Path $backupdir -ChildPath "wif.services.config.backup"
    
    Copy-Item -Path $orig_webconfig -Destination $backup_webconfig -Force|out-null
    Write-Log "Copied '$orig_webconfig' to '$backup_webconfig."

    Copy-item -Path $orig_wifconfig -Destination $backup_wifconfig -Force|out-null
    Write-Log "Copied '$orig_wifconfig' to '$backup_wifconfig'."

    Copy-item -Path $orig_wifservicesconfig -Destination $backup_wifservicesconfig -Force|out-null
    Write-Log "Copied '$orig_wifservicesconfig' to '$backup_wifservicesconfig'."
}

function Upgrade-Web-Config([string]$webroot,[string]$ctp8configdir)
{
    Decrypt-Config -webroot:$webroot
    Upgrade-App-Settings -webroot:$webroot -parentdir:$ctp8configdir
    Upgrade-Bindings -webroot:$webroot -parentdir:$ctp8configdir
    Upgrade-Service-Behaviors -webroot:$webroot -parentdir:$ctp8configdir
    Update-Machine-Config-Keys -webroot:$webroot -parenrdir:$ctp8configdir

    $ctp7webconfig=Join-Path -Path $webroot -ChildPath "web.config"
    $ctp7webconfigbackup=Join-Path -Path $webroot -ChildPath "web.config.ctp7"
    $ctp8webconfig=Join-Path -Path $webroot -ChildPath "web.config.ctp8"
    $webconfig=Join-Path -Path $webroot -ChildPath "web.config"
    
    Rename-File -from:$ctp8webconfig -to:$webconfig
    Encrypt-Config -webroot:$webroot
}

function Upgrade-Wif-Config([string]$webroot,[string]$ctp8configdir)
{
    $ctp7wifconfig=Join-Path -Path $webroot -ChildPath "wif.config"
    $ctp8_wifconfig_template=Join-Path -Path $ctp8configdir -ChildPath "wif.config"
    $ctp8wifconfig=Join-Path -Path $webroot -ChildPath "wif.config.ctp8"
    Copy-Item -Path $ctp8_wifconfig_template -Destination $ctp8wifconfig -Force |out-null
    Write-Log "Copied the CTP8 wif.config template from '$ctp8_wifconfig_template' to '$ctp8wifconfig'."

    [System.Xml.XmlDocument] $ctp7xd = new-object System.Xml.XmlDocument
    $ctp7xd.Load($ctp7wifconfig)

    [System.Xml.XmlDocument] $ctp8xd = new-object System.Xml.XmlDocument
    $ctp8xd.Load($ctp8wifconfig)

    #region audienceUris
    $ctp7audienceUri=$ctp7xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/audienceUris/add")
    if($ctp7audienceUri -ne $null)
    {
        $ctp7audienceUriValue=$ctp7audienceUri.Attributes.GetNamedItem("value").Value
        $ctp8audienceUri=$ctp8xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/audienceUris/add")
        $ctp8audienceUri.Attributes.GetNamedItem("value").Value=$ctp7audienceUriValue
        Write-Log "setting the audienceUri value to '$ctp7audienceUriValue'."
    }

    $ctp8xd.Save($ctp8wifconfig)
    #endregion

    #region authority
    $ctp8IssuerNameRegistry=$ctp8xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry")
    $ctp7AuthorityNameNodes=$ctp7xd.SelectNodes("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority/@name")
    $ctp8AuthorityNameNodes=$ctp8xd.SelectNodes("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority/@name")

    $ctp7AuthorityNames=@()
    $ctp8AuthorityNames=@()

    foreach($authority in $ctp7AuthorityNameNodes)
    {
        $ctp7AuthorityNames += $authority.'#text'
    }

    foreach($authority in $ctp8AuthorityNameNodes)
    {
        $ctp8AuthorityNames += $authority.'#text'
    }

    $ctp7UniqueAuthorities=$ctp7AuthorityNames|?{-not ($ctp8AuthorityNames -contains $_)}
    $intersection=$ctp7AuthorityNames|?{($ctp8AuthorityNames -contains $_)}
    
    # process the intersections
    foreach($name in $intersection)
    {
        $ctp7authority=$ctp7xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority[@name='$name']")
        $ctp8authoritykeys=$ctp8xd.SelectNodes("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority[@name='$name']/keys")

        # remove the CTP8 authority keys
        $ctp8authoritykeys.RemoveAll()|out-null

        # get the thumbprint value(s) from the ctp7 wif.config and update it in the ctp8 wif.config
        $ctp7authoritykeys=$ctp7xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority[@name='$name']/keys/add")

        # add the authority keys from the CTP7 wif.config
        foreach($authoritykey in $ctp7authoritykeys)
        {
            $thumbprint=$authoritykey.Attributes.GetNamedItem("thumbprint").Value
            $newauthority = $ctp8xd.CreateElement("add")
            $newauthority.SetAttribute("thumbprint",$thumbprint)|out-null
            $ctp8authoritykeys.AppendChild($newauthority)|out-null
            Write-Log "Added an authority key to authority '$name' with thumbprint '$thumbprint'."
        }
    }

    # add the ctp7 only authorities to ctp8
    foreach($name in $ctp7UniqueAuthorities)
    {
        $ctp7Authority=$ctp7xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority[@name='$name']")
        $newauthority=$ctp8xd.CreateElement("authority")
        $newauthority.SetAttribute("name",$ctp7Authority.Attributes.GetNamedItem("name").Value)|out-null
        $newkeys=$ctp8xd.CreateElement("keys")
        $newValidIssuers=$ctp8xd.CreateElement("validIssuers")

        # add thumbprints
        foreach($thumbprint in $ctp7Authority.keys.add)
        {
            $newthumbprint = $ctp8xd.CreateElement("add")
            $newthumbprint.SetAttribute("thumbprint",$thumbprint.Attributes.GetNamedItem("thumbprint").value)|out-null
            $newkeys.AppendChild($newthumbprint)|out-null
        }

        # add valid issuers
        foreach($validIssuer in $ctp7Authority.validIssuers.add)
        {
            $newValidIssuer = $ctp8xd.CreateElement("add")
            $newValidIssuer.SetAttribute("name",$validIssuer.Attributes.GetNamedItem("name").value)|out-null
            $newValidIssuers.AppendChild($newValidIssuer)|out-null
        }

        $newauthority.AppendChild($newkeys)|out-null
        $newauthority.AppendChild($newValidIssuers)|out-null
        $ctp8IssuerNameRegistry.AppendChild($newAuthority)|out-null
    }

    $ctp8xd.Save($ctp8wifconfig)
    
    # set the thumbprints for the AxTokenIssuer authority
    $wc = Join-Path -Path $webroot -ChildPath "web.config"
    [System.Xml.XmlDocument] $wcxd = new-object System.Xml.XmlDocument
    $wcxd.Load($wc)

    $ctp8xd.Load($ctp8wifconfig)
    $trustedThumbprintsNode = $wcxd.SelectSingleNode("/configuration/appSettings/add[@key='Infrastructure.TrustedCertificates']")
    if($trustedThumbprintsNode -ne $null)
    {
        $trustedThumbprints =  $trustedThumbprintsNode.Attributes.GetNamedItem("value").Value
        [string[]]$thumbprints = $trustedThumbprints.Split(";")
        $axTokenIssuerKeys = $ctp8xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority[@name='AxTokenIssuer']/keys")
        $axTokenIssuers = $ctp8xd.SelectNodes("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority[@name='AxTokenIssuer']/keys/add")

        if($axTokenIssuers -ne $null)
        {
            $axTokenIssuers.RemoveAll()|out-null
        }

        foreach($thumbprint in $thumbprints)
        {
            $newthumbprint = $ctp8xd.CreateElement("add")
            $newthumbprint.SetAttribute("thumbprint",$thumbprint)|out-null
            $axTokenIssuerKeys.AppendChild($newthumbprint)|out-null
        }

        $ctp8xd.Save($ctp8wifconfig)
    }
    #endregion

     Write-Log "Saved the CTP8 wif.config with updated values from CTP7 wif.config."

     # replace the wif.config file
     $wifconfig=Join-Path -Path $webroot -ChildPath "wif.config"
     Rename-File -from:$ctp8wifconfig -to:$wifconfig
}

function Upgrade-Wif-Services-Config([string]$webroot,[string]$ctp8configdir)
{
    $ctp7wifservicesconfig=Join-Path -Path $webroot -ChildPath "wif.services.config"
    $ctp8_wifservicesconfig_template=Join-Path -Path $ctp8configdir -ChildPath "wif.services.config"
    $ctp8wifservicesconfig=Join-Path -Path $webroot -ChildPath "wif.services.config.ctp8"
    Copy-Item -Path $ctp8_wifservicesconfig_template -Destination $ctp8wifservicesconfig -Force |out-null
    Write-Log "Copied the CTP8 wif.services.config template from '$ctp8_wifservicesconfig_template' to '$ctp8wifservicesconfig'."

    [System.Xml.XmlDocument] $ctp7xd = new-object System.Xml.XmlDocument
    $ctp7xd.Load($ctp7wifservicesconfig)

    [System.Xml.XmlDocument] $ctp8xd = new-object System.Xml.XmlDocument
    $ctp8xd.Load($ctp8wifservicesconfig)

    #region wsFederation
    $ctp8wsfederation=$ctp8xd.SelectSingleNode("/system.identityModel.services/federationConfiguration/wsFederation")
    $ctp7wsfederation=$ctp7xd.SelectSingleNode("/system.identityModel.services/federationConfiguration/wsFederation")

    # replace reply url
    $ctp7reply=$ctp7wsfederation.Attributes.GetNamedItem("reply").Value
    $ctp8wsfederation.Attributes.GetNamedItem("reply").Value=$ctp7reply
    Write-Log "Setting wsFederation reply='$ctp7reply'."

    # replace realm
    $ctp7realm=$ctp7wsfederation.Attributes.GetNamedItem("realm").Value
    $ctp8wsfederation.Attributes.GetNamedItem("realm").Value=$ctp7realm
    Write-Log "Setting wsFederation relam='$ctp7realm'."

    # replace issuer
    $ctp7issuer=$ctp7wsfederation.Attributes.GetNamedItem("issuer").Value
    $ctp8wsfederation.Attributes.GetNamedItem("issuer").Value=$ctp7issuer
    Write-Log "Setting wsFederation issuer='$ctp7issuer'."

    #endregion

    #region cookieHandler
    $ctp8cookieHandler=$ctp8xd.SelectSingleNode("/system.identityModel.services/federationConfiguration/cookieHandler")
    $ctp7cookieHandler=$ctp7xd.SelectSingleNode("/system.identityModel.services/federationConfiguration/cookieHandler")

    #replace path
    $ctp7path=$ctp7cookieHandler.Attributes.GetNamedItem("path").Value
    $ctp8cookieHandler.Attributes.GetNamedItem("path").Value=$ctp7path
    Write-Log "Setting cookie handler path='$path'."

    # replace domain
    $ctp7domain=$ctp7cookieHandler.Attributes.GetNamedItem("domain").Value
    $ctp8cookieHandler.Attributes.GetNamedItem("domain").Value=$ctp7domain
    Write-Log "Setting cookie handler domain='$ctp7domain'."
    #endregion

    #region certificateReference
    $ctp8certReference=$ctp8xd.SelectSingleNode("/system.identityModel.services/federationConfiguration/serviceCertificate/certificateReference")
    $ctp7certReference=$ctp7xd.SelectSingleNode("/system.identityModel.services/federationConfiguration/serviceCertificate/certificateReference")

    # replace findValue
    $ctp7findValue=$ctp7certReference.Attributes.GetNamedItem("findValue").Value
    $ctp8certReference.Attributes.GetNamedItem("findValue").Value=$ctp7findValue
    Write-Log "Setting certificateReference findValue='$ctp7findValue'."
    #endregion
    
    $ctp8xd.Save($ctp8wifservicesconfig)
    Write-Log "Saved the CTP8 wif.services.config with updated values from CTP7 wif.services.config."
    
    # replace the wif.services.config file
    $wifservicesconfig=Join-Path -Path $webroot -ChildPath "wif.services.config"
    Rename-File -from:$ctp8wifservicesconfig -to:$wifservicesconfig
}

function Upgrade-App-Settings([string]$webroot,[string]$parentdir)
{
    $ctp7webconfig=Join-Path -Path $webroot -ChildPath "web.config"
    $ctp8_webconfig_template=Join-Path -Path $ctp8configdir -ChildPath "web.config"
    $ctp8webconfig=Join-Path -Path $webroot -ChildPath "web.config.ctp8"
    Copy-Item -Path $ctp8_webconfig_template -Destination $ctp8webconfig -Force |out-null
    Write-Log "Copied the CTP8 web.config template from '$ctp8_webconfig_template' to '$ctp8webconfig'."

    # add/update appsettings
    [System.Xml.XmlDocument] $ctp7xd = new-object System.Xml.XmlDocument
    $ctp7xd.Load($ctp7webconfig)

    [System.Xml.XmlDocument] $ctp8xd = new-object System.Xml.XmlDocument
    $ctp8xd.Load($ctp8webconfig)

    $ctp8appsettings=$ctp8xd.SelectNodes("/configuration/appSettings/add")

    foreach($setting in $ctp8appsettings)
    {
        $ctp8keyname=$setting.Attributes.GetNamedItem("key").Value

        # special handling for the 'DataAccess.DataEncryptionCertificateThumbprintLegacy' and 'DataAccess.DataSigningCertificateThumbprintLegacy' keys
        if($ctp8keyname -eq $DataSigningCertThumbprintLegacyKey)
        {
            # get the legacy signing cert thumbprint from the CTP7 web.config
            $signingcertthumbprintnode=$ctp7xd.SelectSingleNode("/configuration/appSettings/add[@key='$DataSigningCertThumbprintKey']")
            $keyvalue=$signingcertthumbprintnode.Attributes.GetNamedItem("value").Value
        }
        elseif($ctp8keyname -eq $DataEncryptionCertThumbprintLegacyKey)
        {
            # get the legacy encryption cert thumbprint from the CTP7 web.config
            $encryptioncertthumbprintnode=$ctp7xd.SelectSingleNode("/configuration/appSettings/add[@key='$DataEncryptionCertThumbprintKey']")
            $keyvalue=$encryptioncertthumbprintnode.Attributes.GetNamedItem("value").Value
        }
        elseif($ctp8keyname -eq $DataEncryptionCertThumbprintKey -or
                $ctp8keyname -eq $DataSigningCertThumbprintKey
               )
        {
            # if the key is either a encryption or signing cert thumbprint key, continue
            continue
        }
        elseif($ctp8keyname -eq $InternalServiceCertThumbprints)
        {
            # combine the existing thumbrints from the CTP7 web.config and the thumbprint in the ctp8 web.config
            # the ctp8 cert thumbprint should be the third value after the values are combined
            $internalservicecertthumbprintsnode=$ctp7xd.SelectSingleNode("/configuration/appSettings/add[@key='$InternalServiceCertThumbprints']")
            $ctp7value=$internalservicecertthumbprintsnode.Attributes.GetNamedItem("value").Value
            $ctp8value=$setting.Attributes.GetNamedItem("value").Value
            $keyvalue="$ctp7value`;$ctp8value"
        }
        elseif($ctp8keyname -eq $DataAccessAxAdminSqlUser)
        {
            $keyvalue=$ctp7xd.SelectSingleNode("/configuration/appSettings/add[@key='DataAccess.SqlUser']").Attributes.GetNamedItem("value").Value
        }
        elseif($ctp8keyname -eq $DataAccessAxAdminSqlPwd)
        {
            $keyvalue=$ctp7xd.SelectSingleNode("/configuration/appSettings/add[@key='DataAccess.SqlPwd']").Attributes.GetNamedItem("value").Value
        }
        else
        {
            # if the key exists in the ctp7 web.config use its value
            $ctp7keyname=$ctp7xd.SelectSingleNode("/configuration/appSettings/add[@key='$ctp8keyname']")
            if($ctp7keyname -eq $null)
            {
                # we found a new CTP8 appsetting. Update the key value to indicate it
                # $keyvalue=$NEWSETTING
            }
            else
            {
                # use the value from the ctp7 appsetting in the ctp8 appsetting
                $keyvalue=$ctp7keyname.Attributes.GetNamedItem("value").Value
            }
        }

        $setting.Attributes.GetNamedItem("value").Value=$keyvalue
        Write-Log "Setting '$ctp8keyname=$keyvalue'."
    }
    
    $ctp8xd.Save($ctp8webconfig)
    Write-Log "Saved the CTP8 web.config with the updated appsettings."
}

function Upgrade-Service-Behaviors([string]$webroot,[string]$parentdir)
{
    $ctp7webconfig=Join-Path -Path $webroot -ChildPath "web.config"
    $ctp8webconfig=Join-Path -Path $webroot -ChildPath "web.config.ctp8"

    [System.Xml.XmlDocument] $ctp7xd = new-object System.Xml.XmlDocument
    $ctp7xd.Load($ctp7webconfig)

    [System.Xml.XmlDocument] $ctp8xd = new-object System.Xml.XmlDocument
    $ctp8xd.Load($ctp8webconfig)

    # retrieve the service certificate thumbprint
    $servicecertthumbprint=Get-ServiceBehavior-Thumbprint -xmldoc:$ctp7xd

    if([system.string]::IsNullOrEmpty($servicecertthumbprint))
    {
        Log-Error "Unable to find the certificate thumbprint that is required to configure the service behaviors." -throw
    }

    # set the service certificate thumbprint to the ctp7 value
    $ctp8servicecertificates=$ctp8xd.SelectNodes("/configuration/location/system.serviceModel/behaviors/serviceBehaviors/behavior/serviceCredentials/serviceCertificate")
    foreach($servicecertificate in $ctp8servicecertificates)
    {
        $servicecertificate.Attributes.GetNamedItem("findValue").Value=$servicecertthumbprint
    }

    Write-Log "Setting the service behavior certificate thumbprint to '$servicecertthumbprint'."
    $ctp8xd.Save($ctp8webconfig)
}

function Upgrade-Bindings([string]$webroot,[string]$parentdir)
{
    # open both files
    $ctp7webconfig=Join-Path -Path $webroot -ChildPath "web.config"
    $ctp8webconfig=Join-Path -Path $webroot -ChildPath "web.config.ctp8"

    [System.Xml.XmlDocument] $ctp7xd = new-object System.Xml.XmlDocument
    $ctp7xd.Load($ctp7webconfig)

    [System.Xml.XmlDocument] $ctp8xd = new-object System.Xml.XmlDocument
    $ctp8xd.Load($ctp8webconfig)

    #region customBinding
    $ctp8customBindings = $ctp8xd.SelectNodes("/configuration/location/system.serviceModel/bindings/customBinding")
    foreach($binding in $ctp8customBindings)
    {
        $name=$binding.name
        $ctp7CustomBinding = $ctp7xd.SelectNodes("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']")
        if($ctp7CustomBinding -ne $null)
        {
            $ctp7CustomBinding.sendTimeout = $binding.sendTimeout
            $ctp7CustomBinding.receiveTimeout = $binding.receiveTimeout
        


        # update security
        $ctp8Security = $ctp8xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']/security")
        $ctp7Security = $ctp7xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']/security")
     
        $ctp8Security.authenticationMode = $ctp7Security.authenticationMode
      
     

        # update issuedTokenParameters
        $ctp8IssuedTokenParameters = $ctp8xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']/security/issuedTokenParameters")
        $ctp7IssuedTokenParameters = $ctp7xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']/security/issuedTokenParameters")
        $ctp8IssuedTokenParameters.keyType = $ctp7IssuedTokenParameters.keyType
        $ctp8IssuedTokenParameters.tokenType = $ctp7IssuedTokenParameters.tokenType

        # update textMessageEncoding
        $ctp8textMessageEncodingReaderQuotas = $ctp8xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']/textMessageEncoding/readerQuotas")
        $ctp7textMessageEncodingReaderQuotas = $ctp7xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']/textMessageEncoding/readerQuotas")
        $ctp8textMessageEncodingReaderQuotas.maxDepth = $ctp7textMessageEncodingReaderQuotas.maxDepth 
        $ctp8textMessageEncodingReaderQuotas.maxStringContentLength = $ctp7textMessageEncodingReaderQuotas.maxStringContentLength
        $ctp8textMessageEncodingReaderQuotas.maxArrayLength = $ctp7textMessageEncodingReaderQuotas.maxArrayLength
        $ctp8textMessageEncodingReaderQuotas.maxBytesPerRead = $ctp7textMessageEncodingReaderQuotas.maxBytesPerRead
        $ctp8textMessageEncodingReaderQuotas.maxNameTableCharCount = $ctp7textMessageEncodingReaderQuotas.maxNameTableCharCount
        }
    }
    #endregion

    #region webHttpBinding
    $ctp8webHttpBinding = $ctp8xd.SelectNodes("/configuration/location/system.serviceModel/bindings/webHttpBinding")
    foreach($binding in $ctp8webHttpBinding)
    {
        $name=$binding.name
        $ctp7webHttpBinding = $ctp7xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/webHttpBinding/binding[@name='$name']")
        if($ctp7webHttpBinding -ne $null)
        {
            $binding.allowCookies = $ctp7webHttpBinding.allowCookies
            $binding.maxReceivedMessageSize = $ctp7webHttpBinding.maxReceivedMessageSize
            if($ctp7webHttpBinding.Attributes["contentTypeMapper"] -ne $null)
            {
                $binding.contentTypeMapper = $ctp7webHttpBinding.contentTypeMapper
            }
        

        # update security
        $ctp8webHttpBindingSecurity = $ctp8xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/webHttpBinding/binding[@name='$name']/security")
        $ctp7webHttpBindingSecurity = $ctp7xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/webHttpBinding/binding[@name='$name']/security")
        $ctp8webHttpBindingSecurity.mode = $ctp7webHttpBindingSecurity.mode

        # update readerQuotas
        $ctp8webHttpBindingReaderQuotas = $ctp8xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/webHttpBinding/binding[@name='$name']/readerQuotas")
        $ctp7webHttpBindingReaderQuotas = $ctp7xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/webHttpBinding/binding[@name='$name']/readerQuotas")
        $ctp8webHttpBindingReaderQuotas.maxStringContentLength = $ctp7webHttpBindingReaderQuotas.maxStringContentLength
        }
    }
    #endregion

    $ctp8xd.Save($ctp8webconfig)
    Write-Log "Upgraded the bindings."
}

function Update-Machine-Config-Keys([string]$webroot,[string]$parenrdir)
{
    $ctp7webconfig=Join-Path -Path $webroot -ChildPath "web.config"
    $ctp8webconfig=Join-Path -Path $webroot -ChildPath "web.config.ctp8"

    # load the web.config files
    [System.Xml.XmlDocument] $ctp7xd = new-object System.Xml.XmlDocument
    $ctp7xd.Load($ctp7webconfig)

    [System.Xml.XmlDocument] $ctp8xd = new-object System.Xml.XmlDocument
    $ctp8xd.Load($ctp8webconfig)

    # fetch the ctp7 machineKey values
    $ctp7machineKey = $ctp7xd.SelectSingleNode("/configuration/location/system.web/machineKey")
    $decryption=$ctp7machineKey.Attributes.GetNamedItem("decryption").Value
    $decryptionKey = $ctp7machineKey.Attributes.GetNamedItem("decryptionKey").Value
    $validationKey = $ctp7machineKey.Attributes.GetNamedItem("validationKey").Value
    $validation = $ctp7machineKey.Attributes.GetNamedItem("validation").Value

    # update the ctp8 machineKey values
    $ctp8machineKey=$ctp8xd.SelectSingleNode("/configuration/location/system.web/machineKey")
    $ctp8machineKey.Attributes.GetNamedItem("decryption").Value=$decryption
    $ctp8machineKey.Attributes.GetNamedItem("decryptionKey").Value=$decryptionKey
    $ctp8machineKey.Attributes.GetNamedItem("validation").Value=$validation
    $ctp8machineKey.Attributes.GetNamedItem("validationKey").Value=$validationKey

    # log
    Write-Log "Setting the machine decryption value to '$decryption'."
    Write-Log "Setting the machine decryption key value to '$decryptionKey'."
    Write-Log "Setting the machine validation value to '$validation'."
    Write-Log "Setting the machine validation key value to '$validationKey'."

    # save the ctp8 web.config with the updated values
    $ctp8xd.Save($ctp8webconfig)
}

function Get-ServiceBehavior-Thumbprint($xmldoc)
{
    # well-known service behavior names
    $behaviornames=@("AxILSessionServiceBehaviour","AxInteractiveILSessionServiceBehaviour","InteractionServiceBehavior","ODataQueryServiceBehavior")

    # try to get the service behavior thumbprint using one of the known behavior names
    foreach($name in $behaviornames)
    {
        $thumbprint=$xmldoc.SelectSingleNode("/configuration/location/system.serviceModel/behaviors/serviceBehaviors/behavior[@name='$name']/serviceCredentials/serviceCertificate/@findValue").value
        if(![system.string]::IsNullOrEmpty($thumbprint))
        {
            return $thumbprint
        }
    }

    return [string]::Empty
}

function Rename-File([string]$from,[string]$to)
{
    Move-Item -Path $from -Destination $to -Force|out-null
    Write-Log "Renamed file '$from' to '$to'."
}

function Validate
{
    Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
    
    $aosWebsiteName = Get-AosWebSiteName
    if(!(Is-Website-Stopped -name:$aosWebsiteName))
    {
        Log-Error "Cannot perform the upgrade as the AOS website '$aosWebsiteName' is started. Stop the website and retry." -throw
    }

    if(!(Is-Service-Stopped -name:$BatchService))
    {
        Log-Error "Cannot perform the upgrade as the NT service '$BatchService' is running. Stop the service and retry." -throw
    }

    if(!(Is-Service-Stopped -name:$PrintService))
    {
        Log-Error "Cannot perform the upgrade as the NT service '$PrintService' is running. Stop the service and retry." -throw
    }
}

function Is-Website-Stopped([string]$name)
{
    Import-Module WebAdministration
    Write-Log "Checking the state of the web site '$name'."|Out-Null
    $website=Get-Website -Name:$name -ErrorAction SilentlyContinue
    if($website -eq $null)
    {
        return $true
    }

    if($website.State -eq "Started")
    {
        return $false
    }

    return $true
}

function Is-Service-Stopped([string]$name)
{
    Write-Log "Checking the status of the service '$name'."|out-null
    $service=Get-Service -Name:$name -ErrorAction SilentlyContinue
    if($service -eq $null)
    {
        return $true
    }

    if($service.Status -eq "Running")
    {
        return $false
    }

    return $true
}

function Decrypt-Config([string]$webroot)
{
    $command = Join-Path -Path "$webroot\bin" -ChildPath $configEncryptor
    if(!(Test-Path -Path $command))
    {
        Log-Error "Cannot find the Microsoft.Dynamics.AX.Framework.ConfigEncryptor.exe at '$webroot\bin\'." -throw
    }

    $webconfig=Join-Path -Path $webroot -ChildPath "Web.config"
    $commandParameter = " -decrypt `"$webconfig`""
    $logdir=[System.IO.Path]::GetDirectoryName($global:logfile)
    $stdOut=Join-Path -Path $logdir -ChildPath "config_decrypt.log"
    $stdErr= Join-Path -Path $logdir -ChildPath "config_decrypt.error.log"
    Start-Process $command $commandParameter -PassThru -Wait -RedirectStandardOutput $stdOut -RedirectStandardError $stdErr

    $decryptError = Get-Content $stdErr
    if ($decryptError -ne $null) {
        Log-Error $decryptError -throw
    }

    Write-Log "Finished decrypting the CTP7 web.config."
}

function Encrypt-Config([string]$webroot)
{
    $command = Join-Path -Path "$PSScriptRoot\EncryptionTool" -ChildPath $configEncryptor
    if(!(Test-Path -Path $command))
    {
        Log-Error "Cannot find the CTP8 Microsoft.Dynamics.AX.Framework.ConfigEncryptor.exe at '$PSScriptRoot\EncryptionTool\'." -throw
    }

    $webconfig=Join-Path -Path $webroot -ChildPath "Web.config"
    $commandParameter = " -encrypt `"$webconfig`""
    $logdir=[System.IO.Path]::GetDirectoryName($global:logfile)
    $stdOut=Join-Path -Path $logdir -ChildPath "config_encrypt.log"
    $stdErr= Join-Path -Path $logdir -ChildPath "config_encrypt.error.log"
    Start-Process $command $commandParameter -PassThru -Wait -RedirectStandardOutput $stdOut -RedirectStandardError $stdErr

    $encryptError = Get-Content $stdErr
    if ($encryptError -ne $null) {
        Log-Error $encryptError -throw
    }

    Write-Log "Finished encrypting the CTP8 web.config."
}

Export-ModuleMember -Function Initialize-Log,Write-Log,Write-Error,Validate,Create-Backup,Upgrade-Web-Config,Upgrade-Wif-Config,Upgrade-Wif-Services-Config -Variable $logfile

# SIG # Begin signature block
# MIInrQYJKoZIhvcNAQcCoIInnjCCJ5oCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAFtoHbPK5XPz6U
# yKSkeOH9OrU39SSz6puyJiuuoTo186CCDYEwggX/MIID56ADAgECAhMzAAACzI61
# lqa90clOAAAAAALMMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjIwNTEyMjA0NjAxWhcNMjMwNTExMjA0NjAxWjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQCiTbHs68bADvNud97NzcdP0zh0mRr4VpDv68KobjQFybVAuVgiINf9aG2zQtWK
# No6+2X2Ix65KGcBXuZyEi0oBUAAGnIe5O5q/Y0Ij0WwDyMWaVad2Te4r1Eic3HWH
# UfiiNjF0ETHKg3qa7DCyUqwsR9q5SaXuHlYCwM+m59Nl3jKnYnKLLfzhl13wImV9
# DF8N76ANkRyK6BYoc9I6hHF2MCTQYWbQ4fXgzKhgzj4zeabWgfu+ZJCiFLkogvc0
# RVb0x3DtyxMbl/3e45Eu+sn/x6EVwbJZVvtQYcmdGF1yAYht+JnNmWwAxL8MgHMz
# xEcoY1Q1JtstiY3+u3ulGMvhAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUiLhHjTKWzIqVIp+sM2rOHH11rfQw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDcwNTI5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAeA8D
# sOAHS53MTIHYu8bbXrO6yQtRD6JfyMWeXaLu3Nc8PDnFc1efYq/F3MGx/aiwNbcs
# J2MU7BKNWTP5JQVBA2GNIeR3mScXqnOsv1XqXPvZeISDVWLaBQzceItdIwgo6B13
# vxlkkSYMvB0Dr3Yw7/W9U4Wk5K/RDOnIGvmKqKi3AwyxlV1mpefy729FKaWT7edB
# d3I4+hldMY8sdfDPjWRtJzjMjXZs41OUOwtHccPazjjC7KndzvZHx/0VWL8n0NT/
# 404vftnXKifMZkS4p2sB3oK+6kCcsyWsgS/3eYGw1Fe4MOnin1RhgrW1rHPODJTG
# AUOmW4wc3Q6KKr2zve7sMDZe9tfylonPwhk971rX8qGw6LkrGFv31IJeJSe/aUbG
# dUDPkbrABbVvPElgoj5eP3REqx5jdfkQw7tOdWkhn0jDUh2uQen9Atj3RkJyHuR0
# GUsJVMWFJdkIO/gFwzoOGlHNsmxvpANV86/1qgb1oZXdrURpzJp53MsDaBY/pxOc
# J0Cvg6uWs3kQWgKk5aBzvsX95BzdItHTpVMtVPW4q41XEvbFmUP1n6oL5rdNdrTM
# j/HXMRk1KCksax1Vxo3qv+13cCsZAaQNaIAvt5LvkshZkDZIP//0Hnq7NnWeYR3z
# 4oFiw9N2n3bb9baQWuWPswG0Dq9YT9kb+Cs4qIIwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIZgjCCGX4CAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAsyOtZamvdHJTgAAAAACzDAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgDTUB352b
# 6s2FsIO9SiWa9UjBGMe/7uVFIozK+QzeUfMwQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQCFy8x0S8ja+cRpMOmWHDIumpr4DgU3ODO+JuxkQv0z
# HvyZ2gvvLzNaUgALtm1Ro7Wo9OQZySTjadrO3+UTnv7VkgZdR/m/b4tdxYBuPTCJ
# WbTGwIrPiX5boEnSwQbjezE4TkrbCGwjTFo77ogOigMgZMnVdr4o+e6y1N45I62t
# kRoau9ikh/cm8hm3+QmczA0SZeHVzuyJPk1w2NRYyWiIXJXchXlBzwtt7jsvYAJ4
# VpSZwUzpFFX2pabsTgCypk5I5UFpZ182VvntR+CPSGIKiw37iQXtOOBAxRQ9tdlJ
# zpdRa2kPldU2ML215yBjbiped3lIny/Ao8FaIzl9l/KaoYIXDDCCFwgGCisGAQQB
# gjcDAwExghb4MIIW9AYJKoZIhvcNAQcCoIIW5TCCFuECAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIA1xNkB6Zb5BtTas/M3SICKwioaZXmnioXOGahCj
# 1H6bAgZjoaGqNfcYEzIwMjMwMTA2MjE0NTEzLjY0MVowBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjo3ODgwLUUzOTAtODAxNDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCEV8wggcQMIIE+KADAgECAhMzAAABqFXwYanMMBhcAAEA
# AAGoMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTIyMDMwMjE4NTEyM1oXDTIzMDUxMTE4NTEyM1owgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjo3ODgw
# LUUzOTAtODAxNDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAKPabcrALiXX8pjyXpcM
# N89KTvcmlAiDw4pU+HejZhibUeo/HUy+P9VxWhCX7ogeeKPJ677+LeVdPdG5hTvG
# DgSuo3w+AcmzcXZ2QCGUgUReLUKbmrr06bB0xhvtZwoelhxtPkjJFsbTGtSt+V7E
# 4VCjPdYqQZ/iN0ArXXmgbEfVyCwS+h2uooBhM5UcbPogtr5VpgdzbUM4/rWupmFV
# jPB1asn3+wv7aBCK8j9QUJroY4y1pmZSf0SuGMWY7cm2cvrbdm7XldljqRdHW+CQ
# AB4EqiOqgumfR+aSpo5T75KG0+nsBkjlGSsU1Bi15p4rP88pZnSop73Gem9GWO2G
# RLwP15YEnKsczxhGY+Z8NEa0QwMMiVlksdPU7J5qK9gxAQjOJzqISJzhIwQWtELq
# gJoHwkqTxem3grY7B7DOzQTnQpKWoL0HWR9KqIvaC7i9XlPv+ue89j9e7fmB4nh1
# hulzEJzX6RMU9THJMlbO6OrP3NNEKJW8jipCny8H1fuvSuFfuB7t++KK9g2c2NKu
# 5EzSs1nKNqtl4KO3UzyXLWvTRDO4D5PVQOda0tqjS/AWoUrxKC5ZPlkLE+YPsS5G
# +E/VCgCaghPyBZsHNK7wHlSf/26uhLnKp6XRAIroiEYl/5yW0mShjvnARPr0GIlS
# m0KrqSwCjR5ckWT1sKaEb8w3AgMBAAGjggE2MIIBMjAdBgNVHQ4EFgQUNsfb4+L4
# UutlNh/MxjGkj0kLItUwHwYDVR0jBBgwFoAUn6cVXQBeYl2D9OXSZacbUzUZ6XIw
# XwYDVR0fBFgwVjBUoFKgUIZOaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9w
# cy9jcmwvTWljcm9zb2Z0JTIwVGltZS1TdGFtcCUyMFBDQSUyMDIwMTAoMSkuY3Js
# MGwGCCsGAQUFBwEBBGAwXjBcBggrBgEFBQcwAoZQaHR0cDovL3d3dy5taWNyb3Nv
# ZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENB
# JTIwMjAxMCgxKS5jcnQwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcD
# CDANBgkqhkiG9w0BAQsFAAOCAgEAcTuCS2Rqqmf2mPr6OUydhmUx+m6vpEPszWio
# JXbnsRbny62nF9YXTKuSNWH1QFfyc/2N3YTEp4hE8YthYKgDM/HUhUREX3WTwGse
# YuuDeSxWRJWCorAHF1kwQzIKgrUc3G+uVwAmG/EI1ELRExA4ftx0Ehrf59aJm7On
# gn0lTSSiKUeuGA+My6oCi/V8ETxz+eblvQANaltJgGfppuWXYT4jisQKETvoJjBv
# 5x+BA0oEFu7gGaeMDkZjnO5vdf6HeKneILs9ZvwIWkgYQi2ZeozbxglG5YwExoix
# ekxrRTDZwMokIYxXmccscQ0xXmh+I3vo7hV9ZMKTa9Paz5ne4cc8Odw1T+624mB0
# WaW9HAE1hojB6CbfundtV/jwxmdKh15plJXnN1yM7OL924HqAiJisHanpOEJ4Um9
# b3hFUXE2uEJL9aYuIgksVYIq1P29rR4X7lz3uEJH6COkoE6+UcauN6JYFghN9I8J
# RBWAhHX4GQHlngsdftWLLiDZMynlgRCZzkYI24N9cx+D367YwclqNY6CZuAgzwy1
# 2uRYFQasYHYK1hpzyTtuI/A2B8cG+HM6X1jf2d9uARwH6+hLkPtt3/5NBlLXpOl5
# iZyRlBi7iDXkWNa3juGfLAJ3ISDyNh7yu+H4yQYyRs/MVrCkWUJs9EivLKsNJ2B/
# IjNrStYwggdxMIIFWaADAgECAhMzAAAAFcXna54Cm0mZAAAAAAAVMA0GCSqGSIb3
# DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
# A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIw
# MAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAx
# MDAeFw0yMTA5MzAxODIyMjVaFw0zMDA5MzAxODMyMjVaMHwxCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1l
# LVN0YW1wIFBDQSAyMDEwMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA
# 5OGmTOe0ciELeaLL1yR5vQ7VgtP97pwHB9KpbE51yMo1V/YBf2xK4OK9uT4XYDP/
# XE/HZveVU3Fa4n5KWv64NmeFRiMMtY0Tz3cywBAY6GB9alKDRLemjkZrBxTzxXb1
# hlDcwUTIcVxRMTegCjhuje3XD9gmU3w5YQJ6xKr9cmmvHaus9ja+NSZk2pg7uhp7
# M62AW36MEBydUv626GIl3GoPz130/o5Tz9bshVZN7928jaTjkY+yOSxRnOlwaQ3K
# Ni1wjjHINSi947SHJMPgyY9+tVSP3PoFVZhtaDuaRr3tpK56KTesy+uDRedGbsoy
# 1cCGMFxPLOJiss254o2I5JasAUq7vnGpF1tnYN74kpEeHT39IM9zfUGaRnXNxF80
# 3RKJ1v2lIH1+/NmeRd+2ci/bfV+AutuqfjbsNkz2K26oElHovwUDo9Fzpk03dJQc
# NIIP8BDyt0cY7afomXw/TNuvXsLz1dhzPUNOwTM5TI4CvEJoLhDqhFFG4tG9ahha
# YQFzymeiXtcodgLiMxhy16cg8ML6EgrXY28MyTZki1ugpoMhXV8wdJGUlNi5UPkL
# iWHzNgY1GIRH29wb0f2y1BzFa/ZcUlFdEtsluq9QBXpsxREdcu+N+VLEhReTwDwV
# 2xo3xwgVGD94q0W29R6HXtqPnhZyacaue7e3PmriLq0CAwEAAaOCAd0wggHZMBIG
# CSsGAQQBgjcVAQQFAgMBAAEwIwYJKwYBBAGCNxUCBBYEFCqnUv5kxJq+gpE8RjUp
# zxD/LwTuMB0GA1UdDgQWBBSfpxVdAF5iXYP05dJlpxtTNRnpcjBcBgNVHSAEVTBT
# MFEGDCsGAQQBgjdMg30BATBBMD8GCCsGAQUFBwIBFjNodHRwOi8vd3d3Lm1pY3Jv
# c29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9yeS5odG0wEwYDVR0lBAwwCgYI
# KwYBBQUHAwgwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGG
# MA8GA1UdEwEB/wQFMAMBAf8wHwYDVR0jBBgwFoAU1fZWy4/oolxiaNE9lJBb186a
# GMQwVgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDovL2NybC5taWNyb3NvZnQuY29tL3Br
# aS9jcmwvcHJvZHVjdHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3JsMFoGCCsG
# AQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29t
# L3BraS9jZXJ0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcnQwDQYJKoZIhvcN
# AQELBQADggIBAJ1VffwqreEsH2cBMSRb4Z5yS/ypb+pcFLY+TkdkeLEGk5c9MTO1
# OdfCcTY/2mRsfNB1OW27DzHkwo/7bNGhlBgi7ulmZzpTTd2YurYeeNg2LpypglYA
# A7AFvonoaeC6Ce5732pvvinLbtg/SHUB2RjebYIM9W0jVOR4U3UkV7ndn/OOPcbz
# aN9l9qRWqveVtihVJ9AkvUCgvxm2EhIRXT0n4ECWOKz3+SmJw7wXsFSFQrP8DJ6L
# GYnn8AtqgcKBGUIZUnWKNsIdw2FzLixre24/LAl4FOmRsqlb30mjdAy87JGA0j3m
# Sj5mO0+7hvoyGtmW9I/2kQH2zsZ0/fZMcm8Qq3UwxTSwethQ/gpY3UA8x1RtnWN0
# SCyxTkctwRQEcb9k+SS+c23Kjgm9swFXSVRk2XPXfx5bRAGOWhmRaw2fpCjcZxko
# JLo4S5pu+yFUa2pFEUep8beuyOiJXk+d0tBMdrVXVAmxaQFEfnyhYWxz/gq77EFm
# PWn9y8FBSX5+k77L+DvktxW/tM4+pTFRhLy/AsGConsXHRWJjXD+57XQKBqJC482
# 2rpM+Zv/Cuk0+CQ1ZyvgDbjmjJnW4SLq8CdCPSWU5nR0W2rRnj7tfqAxM328y+l7
# vzhwRNGQ8cirOoo6CGJ/2XBjU02N7oJtpQUQwXEGahC0HVUzWLOhcGbyoYIC0jCC
# AjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5n
# dG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9y
# YXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNv
# MSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjo3ODgwLUUzOTAtODAxNDElMCMGA1UE
# AxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZaIjCgEBMAcGBSsOAwIaAxUA
# bLr8xJ9BB4rL4Yg58X1LZ5iQdyyggYMwgYCkfjB8MQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFt
# cCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIFAOdiiWswIhgPMjAyMzAxMDYxNTUw
# MDNaGA8yMDIzMDEwNzE1NTAwM1owdzA9BgorBgEEAYRZCgQBMS8wLTAKAgUA52KJ
# awIBADAKAgEAAgIPVgIB/zAHAgEAAgITnjAKAgUA52Pa6wIBADA2BgorBgEEAYRZ
# CgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIBAAIDB6EgoQowCAIBAAIDAYagMA0G
# CSqGSIb3DQEBBQUAA4GBAKlTsp48T/9YE7prLe40L2mnUPThp3mydQ7sXe744CYA
# dYGNUehL2Rya7rK4cPzOwAV/DOR5ogcf8AJT+P0U5Y84418V5ZVimdx70ZbPiya5
# 7va97xuLmDjNUDtVdMyEu80el9KDaUgqDNT1913butHhKNinoVnn5KA7yq0eTwi+
# MYIEDTCCBAkCAQEwgZMwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
# b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTACEzMA
# AAGoVfBhqcwwGFwAAQAAAagwDQYJYIZIAWUDBAIBBQCgggFKMBoGCSqGSIb3DQEJ
# AzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0BCQQxIgQgbRQws6TSqQRTHQBr2Cdn
# NBaMi79UQ1GttLltKFgRaiYwgfoGCyqGSIb3DQEJEAIvMYHqMIHnMIHkMIG9BCB0
# /ssdAMsHwnNwhfFBXPlFnRvWhHqSX9YLUxBDl1xlpjCBmDCBgKR+MHwxCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4w
# HAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29m
# dCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAABqFXwYanMMBhcAAEAAAGoMCIEIElo
# 2bhRlp35iVZ9DgDGGrHwKh0qXTwaWTbuiS45D/wOMA0GCSqGSIb3DQEBCwUABIIC
# ABbxNZi6G2i5OHXuOLdmbIK+lDR2wh9f16XWy8ANcXBaUDhgM3tH3fENJDy6HsBu
# ME6Q3OcIXKfZ90z3WYkLvtebRzFGKuOzTrIgsRybf9RjLc9whEC/tcDqzaCx9qKp
# 61CKzhbBZu6WT6is7n/pDGIvKvGdDhuLZ0aHpsblJjoaINYAZhNFTo3iFDIQZjrY
# 5hIM4tEfdCLeF0zkj8ojpCAXIi/4s5X9bbwFNQnhEfs/f6H60/4VqjLtY5xrKQZ9
# WXL22kU9lLOAOAZPdkGWjHOpKgO0/H41UuhHPdi8GtdnVFW93N9s90l+rLCVp++0
# TD5ZKoKjH6jtoz/l49rW0M0B7FEOlwpEWBtT1/6MoPcTauSp54utrrA28gdXJ58e
# 1kKeua0A4l6EQKnjaTLNt+FMqYNY+MWJYptJsqmA59B9VeZ1Mh68FHc18IpI10B+
# GUNNBoiPCbXYpG7seajczX4bSpPmRzX16etBQpbwuEFcZZr8z7FV/VYXX89k0hRO
# vDHOYQAy85GgyHpsQCP5/BKAcPMruRTsQdYj6ueL98WV+qw/RkK4GsQqsl+x/RxW
# WVsNVajgbwLOe60wnXVawlWTJhLIiBbSij/1M5xyCnnhjmcLI7foiC4Lg8nfAr0G
# /dYlSvgf4V9Qr6duSPU9tOl0f5JvoYuJAzNU4yTrBKIR
# SIG # End signature block
