﻿<#
.SYNOPSIS
    Create the AOS staging environment.

.DESCRIPTION

    Copyright © 2019 Microsoft. All rights reserved.
#>
[CmdletBinding()]
param
(
    [Parameter(Mandatory = $false, HelpMessage = "The path to the directory in which to write log files.")]
    [string]$LogDir,
    
    [Parameter(Mandatory = $false, HelpMessage = "The ID of the runbook this is executing as part of.")]
    [string]$RunbookId,

    [Parameter(Mandatory = $false, HelpMessage = "The activity id for current operation.")]        
    [string]$ActivityId,

    [Parameter(Mandatory=$false, HelpMessage = "The secondary database connection info if exists.")]
	[hashtable]$AxDRConnectionString,

    [Parameter(Mandatory=$false, HelpMessage = "The servicing features enabled for this package.")]
	[string[]]$EnabledServicingFeatures,

    [Parameter(Mandatory=$false, HelpMessage = "The global update step VMs for each service model.")]
	[hashtable]$GlobalUpdateVMs,

    [Parameter(Mandatory=$false)]
    [hashtable]$Credentials
)

function PrepareDeltaDBsyncReportDeployment([string]$metadataPackagePath)
{
    $MetadataParentPath = Split-Path -Path $metadataPackagePath -Parent
    $deltaSyncFolder = Join-Path -Path $MetadataParentPath -ChildPath "DeltaSync"

    if (!(Test-Path -Path $deltaSyncFolder))
    {
        Write-ServicingLog "Creating delta sync folder." -Vrb
        New-Item -Path $deltaSyncFolder -ItemType Directory -Force | Out-Null
    }
    else
    {
        Write-ServicingLog "Deleting delta sync folder contents." -Vrb
        Get-ChildItem -Path $deltaSyncFolder | Remove-Item -Force -Recurse
    }

    Write-ServicingLog "Copying MD files from metadata package folder to delta sync folder." -Vrb
    Robocopy.exe $metadataPackagePath $deltaSyncFolder /s *AXSecurity*.md *AXTable*.md *AXView*.md *AXEdt*.md *.rdl  >$null
    Write-ServicingLog "Copying MD files completed with RoboCopy exit code $($LASTEXITCODE)." -Vrb
}

$ErrorActionPreference = "Stop"
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\AosEnvironmentUtilities.psm1" -Force -DisableNameChecking

# Initialize the log file to use with Write-ServicingLog.
Set-ServicingLog -LogDir $LogDir

# Initialize exit code.
[int]$ExitCode = 0

# Get the runbook ID
if ([string]::IsNullOrWhiteSpace($RunbookId))
{
    $RunbookId = Get-RunbookId

    #If the runbook ID still wasn't found, set it to a unique value
    if ([string]::IsNullOrWhiteSpace($RunbookId))
    {
        $runbookIdPlaceholder = [DateTime]::UtcNow.Ticks.ToString()
        Write-ServicingLog "Warning: Runbook ID was not specified and could not be determined. Using [$runbookIdPlaceholder] as a temporary Runbook ID."
        $RunbookId = $runbookIdPlaceholder
    }
}

# Progress breadcrumb names
$copyStagingStarted = "CopyToStagingStarted"
$copyStagingCompleted = "CopyToStagingCompleted"
$prepareDeltaSyncCompleted = "PrepareDeltaSyncReportCompleted"
$upgradeConfigCompleted = "UpgradeConfigAosServiceCompleted"
$updateAosCompleted = "UpdateAosServiceCompleted"
$overallCompleted = "CreateAOSStagingEnvCompleted"

try
{
    Write-ServicingLog "Starting AOS create staging environment script..."

    Disable-CopyStagingTask
    Assert-CopyStagingTaskNotRunning

    $aosWebServicePath = Get-AosServicePath
    $aosWebServiceStagingPath = Get-AosServiceStagingPath
    $packagePath = Join-Path -Path $aosWebServiceStagingPath -ChildPath "PackagesLocalDirectory"
    
    $progressFileNameFormat = "AOSStagingProgress_{0}.json"
    $script:progressFilePath = Join-Path -Path $aosWebServiceStagingPath -ChildPath ($progressFileNameFormat -f $RunbookId)

    if (!(Test-Path -Path $aosWebServiceStagingPath))
    {
        Write-ServicingLog "Creating AOS staging folder." -Vrb
        New-Item -Path $aosWebServiceStagingPath -ItemType Directory -Force | Out-Null
    }
    else
    {
        Write-ServicingLog "Detected AOS Staging folder '$aosWebServiceStagingPath'."
    }

    Initialize-ScriptProgress -progressFile $script:progressFilePath -Scope "Global" -ErrorAction Continue
    Add-ScriptProgress $copyStagingStarted -ErrorAction Continue
    
    if (Test-ScriptProgress $copyStagingCompleted -ErrorAction SilentlyContinue)
    {
        Write-ServicingLog "Detected the previous copy to staging completed at [$(Get-ScriptProgress $copyStagingCompleted)] for runbook [$RunbookId]. Skipping copy."
    }
    else
    {
        Write-ServicingLog "Copying AOS folder to AOS staging folder."
        $robocopyLogFile = Join-Path $LogDir ("Robocopy_{0}.txt" -f (Get-Date).ToString("yyyyMMddHHMMss"))

        $roboCopyExitCode = Copy-FullFolder -SourcePath $aosWebServicePath -DestinationPath $aosWebServiceStagingPath -LogFile $robocopyLogFile 
        Write-ServicingLog "Copying AOS folder to AOS staging folder completed with exit code: $roboCopyExitCode"
        Add-ScriptProgress $copyStagingCompleted -ErrorAction Continue
    }

    # Attempt to clean up any progress files copied over from previous runs
    Get-ChildItem $aosWebServiceStagingPath -Filter ($progressFileNameFormat -f "*") -File | ForEach-Object {
        if ($_.Name -ne ($progressFileNameFormat -f $RunbookId))
        {
            Write-Host "Removing progress file from a previous run [$($_.Name)]"
            Remove-Item -Path $_.FullName -Force -ErrorAction Continue
        }
    }
    
    # Prepare delta dbsyc / report deployment
    if (Test-ScriptProgress $prepareDeltaSyncCompleted)
    {
        Write-ServicingLog "Detected PrepareDeltaDBSyncReportDeployment previously completed at [$(Get-ScriptProgress $prepareDeltaSyncCompleted)] for runbook [$RunbookId]. Skipping."
    }
    else 
    {
        Write-ServicingLog "Invoking PrepareDeltaDBSyncReportDeployment function."
        PrepareDeltaDBsyncReportDeployment -metadataPackagePath:$packagePath
        Add-ScriptProgress $prepareDeltaSyncCompleted -ErrorAction Continue
    }
    
    # Update the web.config
    if (Test-ScriptProgress $upgradeConfigCompleted)
    {
        Write-ServicingLog "Detected updating AOSService configuration previously completed at [$(Get-ScriptProgress $upgradeConfigCompleted)] for runbook [$RunbookId]. Skipping."
    }
    else 
    {
        Write-ServicingLog "Invoking AutoUpgradeConfigAosServicePlatformUpdate3.ps1 script."
        Invoke-Expression "$PSScriptRoot\AutoUpgradeConfigAosServicePlatformUpdate3.ps1 -useStaging"
        Write-ServicingLog "Completed AutoUpgradeConfigAosServicePlatformUpdate3.ps1 script with exit code: $($LASTEXITCODE)." -Vrb
        Add-ScriptProgress $upgradeConfigCompleted -ErrorAction Continue
    }

    # Update AOSService
    if (Test-ScriptProgress $updateAosCompleted)
    {
        Write-ServicingLog "Detected updating AOSService previously completed at [$(Get-ScriptProgress $updateAosCompleted)] for runbook [$RunbookId]. Skipping."
    }
    else
    {
        # Do not pass LogDir to AutoUpdateAosService to keep output going to the current log file.
        Write-ServicingLog "Invoking AutoUpdateAosService.ps1 script."
        Invoke-Expression "$PSScriptRoot\AutoUpdateAosService.ps1 -useStaging"
        Write-ServicingLog "Completed AutoUpdateAosService.ps1 script with exit code: $($LASTEXITCODE)." -Vrb
        Add-ScriptProgress $updateAosCompleted -ErrorAction Continue
    }

    # Indicates the script logic has completed - ensure this stays the last line of the execution block.
    Add-ScriptProgress $overallCompleted -ErrorAction Continue
}
catch
{
    # Ensure non-zero exit code if an exception is caught and no exit code set.
    if ($ExitCode -eq 0)
    {
        $ExitCode = 1024
    }

    $ErrorMessage = "Error during AOS create staging: $($_)"

    Write-ServicingLog $ErrorMessage
    Write-ServicingLog $($_) -Vrb
    Write-ServicingLog "AOS create staging environment script failed with exit code: $($ExitCode)."

    # Use throw to indicate error to AXUpdateInstaller.
    # In case of exceptions, the output is not captured, so only the error message and
    # log file contents will be available for diagnostics.
    throw "$($ErrorMessage) [Log: $(Get-ServicingLog)]"
}
finally
{
    # Back up the progress file for this script's execution.
    $backupDir = if ([string]::IsNullOrEmpty($LogDir)) { $PSScriptRoot } else { $LogDir }
    $logTimeSuffix = [DateTime]::UtcNow.ToString('yyyyMMdd-HHmmss')
    $scriptProgressFileBackup = Join-Path -Path $backupDir -ChildPath "CreateAOSStagingEnv-ProgressBackup-$logTimeSuffix.json"
    Copy-ScriptProgressFile -Destination $scriptProgressFileBackup -ErrorAction Continue
}

Write-ServicingLog "AOS create staging environment script completed with exit code: $($ExitCode)."

$machineName = $env:COMPUTERNAME
if ($GlobalUpdateVMs -and $GlobalUpdateVMs['AOSService'] -eq $machineName)
{
    Write-ServicingLog "Running DB sync precheck..."

    # Run DB Sync PreCheck
    $precheckScriptPath = "$PSScriptRoot\DBSyncPrecheck.ps1"
    $precheckScriptPath = Resolve-Path -Path $precheckScriptPath
    & $precheckScriptPath -LogDir $LogDir -ActivityId $ActivityId -RunbookId $RunbookId -AxDRConnectionString $AxDRConnectionString -EnabledServicingFeatures $EnabledServicingFeatures -Credentials $Credentials -PreDowntime
    Write-ServicingLog "DB sync precheck completed."
}

exit $ExitCode
# SIG # Begin signature block
# MIInygYJKoZIhvcNAQcCoIInuzCCJ7cCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCANzfGR69dTmu4N
# agG3gv4EVIU7zNnpQ3QVhosX9+z71KCCDYEwggX/MIID56ADAgECAhMzAAACzI61
# lqa90clOAAAAAALMMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjIwNTEyMjA0NjAxWhcNMjMwNTExMjA0NjAxWjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQCiTbHs68bADvNud97NzcdP0zh0mRr4VpDv68KobjQFybVAuVgiINf9aG2zQtWK
# No6+2X2Ix65KGcBXuZyEi0oBUAAGnIe5O5q/Y0Ij0WwDyMWaVad2Te4r1Eic3HWH
# UfiiNjF0ETHKg3qa7DCyUqwsR9q5SaXuHlYCwM+m59Nl3jKnYnKLLfzhl13wImV9
# DF8N76ANkRyK6BYoc9I6hHF2MCTQYWbQ4fXgzKhgzj4zeabWgfu+ZJCiFLkogvc0
# RVb0x3DtyxMbl/3e45Eu+sn/x6EVwbJZVvtQYcmdGF1yAYht+JnNmWwAxL8MgHMz
# xEcoY1Q1JtstiY3+u3ulGMvhAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUiLhHjTKWzIqVIp+sM2rOHH11rfQw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDcwNTI5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAeA8D
# sOAHS53MTIHYu8bbXrO6yQtRD6JfyMWeXaLu3Nc8PDnFc1efYq/F3MGx/aiwNbcs
# J2MU7BKNWTP5JQVBA2GNIeR3mScXqnOsv1XqXPvZeISDVWLaBQzceItdIwgo6B13
# vxlkkSYMvB0Dr3Yw7/W9U4Wk5K/RDOnIGvmKqKi3AwyxlV1mpefy729FKaWT7edB
# d3I4+hldMY8sdfDPjWRtJzjMjXZs41OUOwtHccPazjjC7KndzvZHx/0VWL8n0NT/
# 404vftnXKifMZkS4p2sB3oK+6kCcsyWsgS/3eYGw1Fe4MOnin1RhgrW1rHPODJTG
# AUOmW4wc3Q6KKr2zve7sMDZe9tfylonPwhk971rX8qGw6LkrGFv31IJeJSe/aUbG
# dUDPkbrABbVvPElgoj5eP3REqx5jdfkQw7tOdWkhn0jDUh2uQen9Atj3RkJyHuR0
# GUsJVMWFJdkIO/gFwzoOGlHNsmxvpANV86/1qgb1oZXdrURpzJp53MsDaBY/pxOc
# J0Cvg6uWs3kQWgKk5aBzvsX95BzdItHTpVMtVPW4q41XEvbFmUP1n6oL5rdNdrTM
# j/HXMRk1KCksax1Vxo3qv+13cCsZAaQNaIAvt5LvkshZkDZIP//0Hnq7NnWeYR3z
# 4oFiw9N2n3bb9baQWuWPswG0Dq9YT9kb+Cs4qIIwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIZnzCCGZsCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAsyOtZamvdHJTgAAAAACzDAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQghJ1MNuax
# ywvU5MHyzsR7zUO6otzKgItvFaoCqeQD2q4wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQAhHnebyyWCJdSXgCbLCJyaCy2ORpSfeAOKXKQxeufq
# hlcY16x1jw/RXRCk1Ze9GNpwEOH5uIFHzURx805Ob3yEy4XNnNRt7m20l8730fhw
# Cq/yT56LyY6bPnWJXmk/Ff6ARobYlWAker4EKAzlyoeJUmf8Los2s8ELIvCuNyqL
# Tk4Gy4rmYAGAPfYe2KSUOTy4riKHi90Ciu1xuJgL0HH1G+HvIIEkT7tr5X5sfLSG
# lYFCAKKF1eg9yun3MxA7z6kNtzYpO2RCcVMtFnehAqLXkrxJPf2upSQogCkjMUaY
# MonNny0ouDot2uVls4FT3VkAVvBA7EEulCq/R5K4YRMaoYIXKTCCFyUGCisGAQQB
# gjcDAwExghcVMIIXEQYJKoZIhvcNAQcCoIIXAjCCFv4CAQMxDzANBglghkgBZQME
# AgEFADCCAVkGCyqGSIb3DQEJEAEEoIIBSASCAUQwggFAAgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIKbRDObxm26kOMhozajD1kaF6hiNz9mazPFhyGjt
# L3tzAgZjovMWCXMYEzIwMjMwMTA2MjE0NTE1LjE2OFowBIACAfSggdikgdUwgdIx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1p
# Y3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlvbnMgTGltaXRlZDEmMCQGA1UECxMdVGhh
# bGVzIFRTUyBFU046RkM0MS00QkQ0LUQyMjAxJTAjBgNVBAMTHE1pY3Jvc29mdCBU
# aW1lLVN0YW1wIFNlcnZpY2WgghF4MIIHJzCCBQ+gAwIBAgITMwAAAbn2AA1lVE+8
# AwABAAABuTANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Eg
# MjAxMDAeFw0yMjA5MjAyMDIyMTdaFw0yMzEyMTQyMDIyMTdaMIHSMQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQg
# SXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxlcyBUU1Mg
# RVNOOkZDNDEtNEJENC1EMjIwMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFt
# cCBTZXJ2aWNlMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA40k+yWH1
# FsfJAQJtQgg3EwXm5CTI3TtUhKEhNe5sulacA2AEIu8JwmXuj/Ycc5GexFyZIg0n
# +pyUCYsis6OdietuhwCeLGIwRcL5rWxnzirFha0RVjtVjDQsJzNj7zpT/yyGDGqx
# p7MqlauI85ylXVKHxKw7F/fTI7uO+V38gEDdPqUczalP8dGNaT+v27LHRDhq3HSa
# QtVhL3Lnn+hOUosTTSHv3ZL6Zpp0B3LdWBPB6LCgQ5cPvznC/eH5/Af/BNC0L2WE
# DGEw7in44/3zzxbGRuXoGpFZe53nhFPOqnZWv7J6fVDUDq6bIwHterSychgbkHUB
# xzhSAmU9D9mIySqDFA0UJZC/PQb2guBI8PwrLQCRfbY9wM5ug+41PhFx5Y9fRRVl
# Sxf0hSCztAXjUeJBLAR444cbKt9B2ZKyUBOtuYf/XwzlCuxMzkkg2Ny30bjbGo3x
# UX1nxY6IYyM1u+WlwSabKxiXlDKGsQOgWdBNTtsWsPclfR8h+7WxstZ4GpfBunhn
# zIAJO2mErZVvM6+Li9zREKZE3O9hBDY+Nns1pNcTga7e+CAAn6u3NRMB8mi285Kp
# wyA3AtlrVj4RP+VvRXKOtjAW4e2DRBbJCM/nfnQtOm/TzqnJVSHgDfD86zmFMYVm
# AV7lsLIyeljT0zTI90dpD/nqhhSxIhzIrJUCAwEAAaOCAUkwggFFMB0GA1UdDgQW
# BBS3sDhx21hDmgmMTVmqtKienjVEUjAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJl
# pxtTNRnpcjBfBgNVHR8EWDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NybC9NaWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAx
# MCgxKS5jcmwwbAYIKwYBBQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3Rh
# bXAlMjBQQ0ElMjAyMDEwKDEpLmNydDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQM
# MAoGCCsGAQUFBwMIMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQsFAAOCAgEA
# zdxns0VQdEywsrOOXusk8iS/ugn6z2SS63SFmJ/1ZK3rRLNgZQunXOZ0+pz7Dx4d
# OSGpfQYoKnZNOpLMFcGHAc6bz6nqFTE2UN7AYxlSiz3nZpNduUBPc4oGd9UEtDJR
# q+tKO4kZkBbfRw1jeuNUNSUYP5XKBAfJJoNq+IlBsrr/p9C9RQWioiTeV0Z+OcC2
# d5uxWWqHpZZqZVzkBl2lZHWNLM3+jEpipzUEbhLHGU+1x+sB0HP9xThvFVeoAB/T
# Y1mxy8k2lGc4At/mRWjYe6klcKyT1PM/k81baxNLdObCEhCY/GvQTRSo6iNSsElQ
# 6FshMDFydJr8gyW4vUddG0tBkj7GzZ5G2485SwpRbvX/Vh6qxgIscu+7zZx4NVBC
# 8/sYcQSSnaQSOKh9uNgSsGjaIIRrHF5fhn0e8CADgyxCRufp7gQVB/Xew/4qfdeA
# wi8luosl4VxCNr5JR45e7lx+TF7QbNM2iN3IjDNoeWE5+VVFk2vF57cH7JnB3ckc
# Mi+/vW5Ij9IjPO31xTYbIdBWrEFKtG0pbpbxXDvOlW+hWwi/eWPGD7s2IZKVdfWz
# vNsE0MxSP06fM6Ucr/eas5TxgS5F/pHBqRblQJ4ZqbLkyIq7Zi7IqIYEK/g4aE+y
# 017sAuQQ6HwFfXa3ie25i76DD0vrII9jSNZhpC3MA/0wggdxMIIFWaADAgECAhMz
# AAAAFcXna54Cm0mZAAAAAAAVMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9v
# dCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0yMTA5MzAxODIyMjVaFw0z
# MDA5MzAxODMyMjVaMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIICIjAN
# BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA5OGmTOe0ciELeaLL1yR5vQ7VgtP9
# 7pwHB9KpbE51yMo1V/YBf2xK4OK9uT4XYDP/XE/HZveVU3Fa4n5KWv64NmeFRiMM
# tY0Tz3cywBAY6GB9alKDRLemjkZrBxTzxXb1hlDcwUTIcVxRMTegCjhuje3XD9gm
# U3w5YQJ6xKr9cmmvHaus9ja+NSZk2pg7uhp7M62AW36MEBydUv626GIl3GoPz130
# /o5Tz9bshVZN7928jaTjkY+yOSxRnOlwaQ3KNi1wjjHINSi947SHJMPgyY9+tVSP
# 3PoFVZhtaDuaRr3tpK56KTesy+uDRedGbsoy1cCGMFxPLOJiss254o2I5JasAUq7
# vnGpF1tnYN74kpEeHT39IM9zfUGaRnXNxF803RKJ1v2lIH1+/NmeRd+2ci/bfV+A
# utuqfjbsNkz2K26oElHovwUDo9Fzpk03dJQcNIIP8BDyt0cY7afomXw/TNuvXsLz
# 1dhzPUNOwTM5TI4CvEJoLhDqhFFG4tG9ahhaYQFzymeiXtcodgLiMxhy16cg8ML6
# EgrXY28MyTZki1ugpoMhXV8wdJGUlNi5UPkLiWHzNgY1GIRH29wb0f2y1BzFa/Zc
# UlFdEtsluq9QBXpsxREdcu+N+VLEhReTwDwV2xo3xwgVGD94q0W29R6HXtqPnhZy
# acaue7e3PmriLq0CAwEAAaOCAd0wggHZMBIGCSsGAQQBgjcVAQQFAgMBAAEwIwYJ
# KwYBBAGCNxUCBBYEFCqnUv5kxJq+gpE8RjUpzxD/LwTuMB0GA1UdDgQWBBSfpxVd
# AF5iXYP05dJlpxtTNRnpcjBcBgNVHSAEVTBTMFEGDCsGAQQBgjdMg30BATBBMD8G
# CCsGAQUFBwIBFjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3Mv
# UmVwb3NpdG9yeS5odG0wEwYDVR0lBAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQC
# BAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYD
# VR0jBBgwFoAU1fZWy4/oolxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZF
# aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9v
# Q2VyQXV0XzIwMTAtMDYtMjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcw
# AoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJB
# dXRfMjAxMC0wNi0yMy5jcnQwDQYJKoZIhvcNAQELBQADggIBAJ1VffwqreEsH2cB
# MSRb4Z5yS/ypb+pcFLY+TkdkeLEGk5c9MTO1OdfCcTY/2mRsfNB1OW27DzHkwo/7
# bNGhlBgi7ulmZzpTTd2YurYeeNg2LpypglYAA7AFvonoaeC6Ce5732pvvinLbtg/
# SHUB2RjebYIM9W0jVOR4U3UkV7ndn/OOPcbzaN9l9qRWqveVtihVJ9AkvUCgvxm2
# EhIRXT0n4ECWOKz3+SmJw7wXsFSFQrP8DJ6LGYnn8AtqgcKBGUIZUnWKNsIdw2Fz
# Lixre24/LAl4FOmRsqlb30mjdAy87JGA0j3mSj5mO0+7hvoyGtmW9I/2kQH2zsZ0
# /fZMcm8Qq3UwxTSwethQ/gpY3UA8x1RtnWN0SCyxTkctwRQEcb9k+SS+c23Kjgm9
# swFXSVRk2XPXfx5bRAGOWhmRaw2fpCjcZxkoJLo4S5pu+yFUa2pFEUep8beuyOiJ
# Xk+d0tBMdrVXVAmxaQFEfnyhYWxz/gq77EFmPWn9y8FBSX5+k77L+DvktxW/tM4+
# pTFRhLy/AsGConsXHRWJjXD+57XQKBqJC4822rpM+Zv/Cuk0+CQ1ZyvgDbjmjJnW
# 4SLq8CdCPSWU5nR0W2rRnj7tfqAxM328y+l7vzhwRNGQ8cirOoo6CGJ/2XBjU02N
# 7oJtpQUQwXEGahC0HVUzWLOhcGbyoYIC1DCCAj0CAQEwggEAoYHYpIHVMIHSMQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNy
# b3NvZnQgSXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxl
# cyBUU1MgRVNOOkZDNDEtNEJENC1EMjIwMSUwIwYDVQQDExxNaWNyb3NvZnQgVGlt
# ZS1TdGFtcCBTZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQDHYh4YeGTnwxCTPNJaScZw
# uN+BOqCBgzCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMA0GCSqG
# SIb3DQEBBQUAAgUA52KJRzAiGA8yMDIzMDEwNjE5NDkyN1oYDzIwMjMwMTA3MTk0
# OTI3WjB0MDoGCisGAQQBhFkKBAExLDAqMAoCBQDnYolHAgEAMAcCAQACAgEEMAcC
# AQACAhFOMAoCBQDnY9rHAgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkK
# AwKgCjAIAgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZIhvcNAQEFBQADgYEAE5PE
# EH1bb0s1sl6NArO8OgFd9hR84X00EkFSNw+6HuTM8uPYxoJDUqU/yH1uHcof7hxa
# RsoHKbvzEI7lx7MnIEgd9+PX2jzvLftp6wzCxhqzSPYDHA/Y00RAU/MFrp1YJI3Z
# AVd7qiJo1qh3gZZ49UzJ/449nXkMm9KSwvOactkxggQNMIIECQIBATCBkzB8MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNy
# b3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAbn2AA1lVE+8AwABAAABuTAN
# BglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMC8G
# CSqGSIb3DQEJBDEiBCDRg8l1iCNBbdm4O3Ry7GYWN7q7ZSsvE05VwDP7dhw9MzCB
# +gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EIGTrRs7xbzm5MB8lUQ7e9fZotpAV
# yBwal3Cw6iL5+g/0MIGYMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIw
# MTACEzMAAAG59gANZVRPvAMAAQAAAbkwIgQgmf46C1eV9GX+tWUz3KI2nzewnoUR
# PvdzGo3aitzf4kUwDQYJKoZIhvcNAQELBQAEggIActdCWuGSk1GuMsCKt0YbJDZA
# VZXBh2/ynRi4ybMJ8vkaFSL2z6K132W5GhDhStKUK6gwZIvMDzhOan7hLYe7mD60
# Nsi2cU0jei5U8NxvfwZSBaoqebDQ5+gSoqHmdtq8RzCLLmHYPPptnAB+18OjGABv
# KeXyqftaF505bcQEGETPuFfKP6A2ZyyvqoJbV9M95Df+UQ46yg2uyuCqhNzkFfxq
# BD4nX/MxfQhRbzMLBPbaFT1FVmc+TnP73UPKhkOYriB5GEAOsqPll5uvRuYZjSz+
# 32krWjVktzqN+np0mfrksXMe3AgrF6n2g9q8WwqNFQkYDKu+zARpSs+ekQAljp+M
# phTyjuXERMBQg0ZX3e7Q4uM5ziFfyBYMIV0T5yooE9s3SMVDzDSVBTp+hQIiS7al
# cSzhX4R4JsEsDWrEaRZE2qlZST+zCLKUErV8pK3DUs7141qoasN1LkPxYNs3edma
# PKJn7P6ImPpWxZmy8hleALJxZNmcSuHW3IPqRFABJqBXHBAVZqkk2Wd+cl5HMTgm
# rL/StyHdULPVT8ezMKhamXQxSTxYsRYrUQdjenSibPERBimYVrwDX0Wai1YV3dgb
# cAbkMCQrR4cyHibafqw1nJyAHaAkOXdBk2N1xwXdhaZHGJr2rPh4MiVSh0DmrgKo
# 4i5Cmw997nY1L/F9Ql0=
# SIG # End signature block
