﻿<#
.SYNOPSIS
    Starts the AOS service.

.DESCRIPTION

    Copyright © 2019 Microsoft. All rights reserved.
#>
[CmdletBinding()]
param
(
    [Parameter(Mandatory = $false, HelpMessage = "The path to the directory in which to write log files.")]
    [string]$LogDir
)

$ErrorActionPreference = "Stop"
Import-Module WebAdministration
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking

# Initialize exit code.
[int]$ExitCode = 0

# Initialize the log file to use with Write-ServicingLog.
Set-ServicingLog -LogDir $LogDir

try
{
    Write-ServicingLog "Starting AOS..."

    # For non-admin developer machine scenarios, import module to replace functions that
    # will not work within running as administrator.
    if (Test-Path "$($PSScriptRoot)\NonAdminDevToolsInterject.ps1")
    {
        & "$($PSScriptRoot)\NonAdminDevToolsInterject.ps1"
    }

    # Create copying staging task if it doesn't exist
    try 
    {
        Write-ServicingLog "Creating CopyStagingTask..."
        Set-CopyStagingTask
    }
    catch
    {
        Write-Warning "Failed to create CopyStagingTask: $_"
        Write-ServicingLog "Failed to create CopyStagingTask: $_"
    }

    # Determine if running in admin or non-admin mode.
    $IsAdmin = Test-IsRunningAsAdministrator

    # Get IIS service and start it if it is not already running.
    $IisService = Get-Service -Name "W3SVC"
    if ($IisService -and $IisService.Status -ine "Running")
    {
        Write-ServicingLog "IIS service is not running, starting IIS Service..."
        Start-Service -Name "W3SVC"
        Write-ServicingLog "IIS service started."
    }

    # Get the AOS web site and app pool and make sure they are both started.
    $websiteName = Get-AosWebsiteName
    $appPoolName = Get-AosAppPoolName
    if ((![string]::IsNullOrWhitespace($websiteName)) -and (![string]::IsNullOrWhitespace($appPoolName)))
    {
        # Set properties on web site and app pool, so they restart during reboots / IISRESETs.
        # Note: Stopping / starting via IIS mananger GUI will set these same properties.
        # Note: This is not available for non-admin users.
        if ($IsAdmin)
        {
            Write-ServicingLog "Enabling IIS auto start properties for the AOS website and application pool." -Vrb
            Set-ItemProperty -Path "IIS:\Sites\$webSiteName" -Name serverAutoStart -Value $true
            Set-ItemProperty -Path "IIS:\AppPools\$appPoolName" -Name autoStart -Value $true
        }

        Write-ServicingLog "Starting IIS AOS web site."
        Start-Website -Name $websiteName

        Write-ServicingLog "Starting IIS AOS application pool."
        Start-WebAppPool -Name $appPoolName

        # Get the product configuration app pool and make sure it is started.
        $productConfigurationPoolName = Get-ProductConfigurationAppPoolName
        if (![string]::IsNullOrWhitespace($productConfigurationPoolName))
        {
            # Set property on app pool, so it restarts during reboots / IISRESETs.
            # Note: Stopping / starting via IIS mananger GUI will set these same properties.
            # Note: This is not available for non-admin users.
            if ($IsAdmin)
            {
                Write-ServicingLog "Enabling IIS auto start property for the product configuration application pool." -Vrb
                Set-ItemProperty -Path "IIS:\AppPools\$productConfigurationPoolName" -Name autoStart -Value $true
            }

            Write-ServicingLog "Starting IIS product configuration application pool."
            Start-WebAppPool -Name $productConfigurationPoolName
        }
    }

    # Get Dynamics Batch service and start it if it is not already running.
    $BatchService = Get-Service -Name "DynamicsAxBatch" -ErrorAction SilentlyContinue
    if ($BatchService -and $BatchService.Status -ine "Running")
    {
        Write-ServicingLog "Configuring batch service startup type to delayed-auto using SC command..." -Vrb
        $ScOutput = & sc.exe \\localhost config DynamicsAxBatch start= delayed-auto
        Write-ServicingLog $ScOutput -Vrb
        Write-ServicingLog "Batch service startup type configured." -Vrb

        Write-ServicingLog "Starting batch service..."
        Start-Service -Name "DynamicsAxBatch"
        Write-ServicingLog "Batch service started."
    }
}
catch
{
    # Ensure non-zero exit code if an exception is caught and no exit code set.
    if ($ExitCode -eq 0)
    {
        $ExitCode = 1024
    }

    $ErrorMessage = "Error during AOS start: $($_)"

    Write-ServicingLog $ErrorMessage
    Write-ServicingLog $($_) -Vrb
    Write-ServicingLog "AOS start script failed with exit code: $($ExitCode)."

    # Use throw to indicate error to AXUpdateInstaller.
    # In case of exceptions, the output is not captured, so only the error message and
    # log file contents will be available for diagnostics.
    throw "$($ErrorMessage) [Log: $(Get-ServicingLog)]"
}

Write-ServicingLog "AOS start script completed with exit code: $($ExitCode)."
exit $ExitCode
# SIG # Begin signature block
# MIIr/gYJKoZIhvcNAQcCoIIr7zCCK+sCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDXVesG7TJhSj+W
# ln8ovp7nCKybzV/pIXDcgl0Wn6a8AqCCEW4wggh+MIIHZqADAgECAhM2AAACAO38
# jbec3qFIAAIAAAIAMA0GCSqGSIb3DQEBCwUAMEExEzARBgoJkiaJk/IsZAEZFgNH
# QkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxFTATBgNVBAMTDEFNRSBDUyBDQSAwMTAe
# Fw0yNDExMDgxMjQzMjhaFw0yNTExMDgxMjQzMjhaMCQxIjAgBgNVBAMTGU1pY3Jv
# c29mdCBBenVyZSBDb2RlIFNpZ24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK
# AoIBAQC5L/UPrOpwYjxcoZC0TqqvMF1WUELvwXN+k27SrA5rohJknn7Cgbxg4hGT
# XKqpcdbtsVTN3ZY896SJ20uQ+INL5OVLzpW408nCNTPYg2LtGJbqHUjpNm0hLCJ+
# gO5Jn2T8DDzIJoUijGXj1m+hRLKb2nOIicCED2GuYBmuWXnaY7INmVEaU3peryty
# ZjDuxdyGDuiPURz8lW1SUiDzoszNp1oswVr+WjDvLDUx4HlxPsG8zUjIst0NnJ6o
# z4tNFKaUBDCetcMjQxpCETn29a1CuRddxZLjPHZHfcotr5sh1S6bNQdzVaMNsxV8
# L3wjHb7XJ6ZVm662mHEiPgpyNcLhAgMBAAGjggWKMIIFhjApBgkrBgEEAYI3FQoE
# HDAaMAwGCisGAQQBgjdbAQEwCgYIKwYBBQUHAwMwPQYJKwYBBAGCNxUHBDAwLgYm
# KwYBBAGCNxUIhpDjDYTVtHiE8Ys+hZvdFs6dEoFgg93NZoaUjDICAWQCAQ4wggJ2
# BggrBgEFBQcBAQSCAmgwggJkMGIGCCsGAQUFBzAChlZodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpaW5mcmEvQ2VydHMvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDEu
# YW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUy
# MDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDIuYW1lLmdibC9haWEv
# QlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBS
# BggrBgEFBQcwAoZGaHR0cDovL2NybDMuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAx
# LkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZG
# aHR0cDovL2NybDQuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDCBrQYIKwYBBQUHMAKGgaBsZGFwOi8vL0NO
# PUFNRSUyMENTJTIwQ0ElMjAwMSxDTj1BSUEsQ049UHVibGljJTIwS2V5JTIwU2Vy
# dmljZXMsQ049U2VydmljZXMsQ049Q29uZmlndXJhdGlvbixEQz1BTUUsREM9R0JM
# P2NBQ2VydGlmaWNhdGU/YmFzZT9vYmplY3RDbGFzcz1jZXJ0aWZpY2F0aW9uQXV0
# aG9yaXR5MB0GA1UdDgQWBBST/HE52ZUlmsYqZcZBdrXZ5u4ZnzAOBgNVHQ8BAf8E
# BAMCB4AwRQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEWMBQGA1UEBRMNMjM2MTY3KzUwMzE1NTCCAeYGA1UdHwSCAd0wggHZMIIB
# 1aCCAdGgggHNhj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpaW5mcmEvQ1JM
# L0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwxLmFtZS5nYmwv
# Y3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwyLmFtZS5n
# YmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwzLmFt
# ZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmw0
# LmFtZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGgb1sZGFwOi8v
# L0NOPUFNRSUyMENTJTIwQ0ElMjAwMSgyKSxDTj1CWTJQS0lDU0NBMDEsQ049Q0RQ
# LENOPVB1YmxpYyUyMEtleSUyMFNlcnZpY2VzLENOPVNlcnZpY2VzLENOPUNvbmZp
# Z3VyYXRpb24sREM9QU1FLERDPUdCTD9jZXJ0aWZpY2F0ZVJldm9jYXRpb25MaXN0
# P2Jhc2U/b2JqZWN0Q2xhc3M9Y1JMRGlzdHJpYnV0aW9uUG9pbnQwHwYDVR0jBBgw
# FoAUllGE4Gtve/7YBqvD8oXmKa5q+dQwHwYDVR0lBBgwFgYKKwYBBAGCN1sBAQYI
# KwYBBQUHAwMwDQYJKoZIhvcNAQELBQADggEBAEDd8Wf5RkHsB64vgn2slxDtHzSo
# It9xN/Dm3RdFjNZ0diTUPMgSPYQlSk8nIAfudnB9FLavGlvZLlyUpfrPSuikepj3
# i3pqNEFn6fNdNFv/wHMxv7hQTIDCmuoR1v1rX+w3oeleBPMnN3QmH4ff1NsynyV4
# dZdYgN9Cw9sC/S3pWZpJrbOs7YOM3vqyU6DciHhC4D9i2zByHCF2pu9nYfiQf5A2
# iUZenRvyo1E5rC+UP2VZXa4k7g66W20+zAajIKKIqEmRtWahekMkCcOIHFBY4RDA
# ybgPRSGur4VDAiZPjTXS90wQXrX9CwU20cfiCC6e76F4H95KtQjKYpzuNVAwggjo
# MIIG0KADAgECAhMfAAAAUeqP9pxzDKg7AAAAAABRMA0GCSqGSIb3DQEBCwUAMDwx
# EzARBgoJkiaJk/IsZAEZFgNHQkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxEDAOBgNV
# BAMTB2FtZXJvb3QwHhcNMjEwNTIxMTg0NDE0WhcNMjYwNTIxMTg1NDE0WjBBMRMw
# EQYKCZImiZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQD
# EwxBTUUgQ1MgQ0EgMDEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDJ
# mlIJfQGejVbXKpcyFPoFSUllalrinfEV6JMc7i+bZDoL9rNHnHDGfJgeuRIYO1LY
# /1f4oMTrhXbSaYRCS5vGc8145WcTZG908bGDCWr4GFLc411WxA+Pv2rteAcz0eHM
# H36qTQ8L0o3XOb2n+x7KJFLokXV1s6pF/WlSXsUBXGaCIIWBXyEchv+sM9eKDsUO
# LdLTITHYJQNWkiryMSEbxqdQUTVZjEz6eLRLkofDAo8pXirIYOgM770CYOiZrcKH
# K7lYOVblx22pdNawY8Te6a2dfoCaWV1QUuazg5VHiC4p/6fksgEILptOKhx9c+ia
# piNhMrHsAYx9pUtppeaFAgMBAAGjggTcMIIE2DASBgkrBgEEAYI3FQEEBQIDAgAC
# MCMGCSsGAQQBgjcVAgQWBBQSaCRCIUfL1Gu+Mc8gpMALI38/RzAdBgNVHQ4EFgQU
# llGE4Gtve/7YBqvD8oXmKa5q+dQwggEEBgNVHSUEgfwwgfkGBysGAQUCAwUGCCsG
# AQUFBwMBBggrBgEFBQcDAgYKKwYBBAGCNxQCAQYJKwYBBAGCNxUGBgorBgEEAYI3
# CgMMBgkrBgEEAYI3FQYGCCsGAQUFBwMJBggrBgEFBQgCAgYKKwYBBAGCN0ABAQYL
# KwYBBAGCNwoDBAEGCisGAQQBgjcKAwQGCSsGAQQBgjcVBQYKKwYBBAGCNxQCAgYK
# KwYBBAGCNxQCAwYIKwYBBQUHAwMGCisGAQQBgjdbAQEGCisGAQQBgjdbAgEGCisG
# AQQBgjdbAwEGCisGAQQBgjdbBQEGCisGAQQBgjdbBAEGCisGAQQBgjdbBAIwGQYJ
# KwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMBIGA1UdEwEB/wQI
# MAYBAf8CAQAwHwYDVR0jBBgwFoAUKV5RXmSuNLnrrJwNp4x1AdEJCygwggFoBgNV
# HR8EggFfMIIBWzCCAVegggFToIIBT4YxaHR0cDovL2NybC5taWNyb3NvZnQuY29t
# L3BraWluZnJhL2NybC9hbWVyb290LmNybIYjaHR0cDovL2NybDIuYW1lLmdibC9j
# cmwvYW1lcm9vdC5jcmyGI2h0dHA6Ly9jcmwzLmFtZS5nYmwvY3JsL2FtZXJvb3Qu
# Y3JshiNodHRwOi8vY3JsMS5hbWUuZ2JsL2NybC9hbWVyb290LmNybIaBqmxkYXA6
# Ly8vQ049YW1lcm9vdCxDTj1BTUVSb290LENOPUNEUCxDTj1QdWJsaWMlMjBLZXkl
# MjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9uLERDPUFNRSxE
# Qz1HQkw/Y2VydGlmaWNhdGVSZXZvY2F0aW9uTGlzdD9iYXNlP29iamVjdENsYXNz
# PWNSTERpc3RyaWJ1dGlvblBvaW50MIIBqwYIKwYBBQUHAQEEggGdMIIBmTBHBggr
# BgEFBQcwAoY7aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraWluZnJhL2NlcnRz
# L0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6Ly9jcmwyLmFt
# ZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6
# Ly9jcmwzLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUH
# MAKGK2h0dHA6Ly9jcmwxLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQw
# gaIGCCsGAQUFBzAChoGVbGRhcDovLy9DTj1hbWVyb290LENOPUFJQSxDTj1QdWJs
# aWMlMjBLZXklMjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9u
# LERDPUFNRSxEQz1HQkw/Y0FDZXJ0aWZpY2F0ZT9iYXNlP29iamVjdENsYXNzPWNl
# cnRpZmljYXRpb25BdXRob3JpdHkwDQYJKoZIhvcNAQELBQADggIBAFAQI7dPD+jf
# XtGt3vJp2pyzA/HUu8hjKaRpM3opya5G3ocprRd7vdTHb8BDfRN+AD0YEmeDB5HK
# QoG6xHPI5TXuIi5sm/LeADbV3C2q0HQOygS/VT+m1W7a/752hMIn+L4ZuyxVeSBp
# fwf7oQ4YSZPh6+ngZvBHgfBaVz4O9/wcfw91QDZnTgK9zAh9yRKKls2bziPEnxeO
# ZMVNaxyV0v152PY2xjqIafIkUjK6vY9LtVFjJXenVUAmn3WCPWNFC1YTIIHw/mD2
# cTfPy7QA1pT+GPARAKt0bKtq9aCd/Ym0b5tPbpgCiRtzyb7fbNS1dE740re0COE6
# 7YV2wbeo2sXixzvLftH8L7s9xv9wV+G22qyKt6lmKLjFK1yMw4Ni5fMabcgmzRvS
# jAcbqgp3tk4a8emaaH0rz8MuuIP+yrxtREPXSqL/C5bzMzsikuDW9xH10graZzSm
# PjilzpRfRdu20/9UQmC7eVPZ4j1WNa1oqPHfzET3ChIzJ6Q9G3NPCB+7KwX0OQmK
# yv7IDimj8U/GlsHD1z+EF/fYMf8YXG15LamaOAohsw/ywO6SYSreVW+5Y0mzJutn
# BC9Cm9ozj1+/4kqksrlhZgR/CSxhFH3BTweH8gP2FEISRtShDZbuYymynY1un+Ry
# fiK9+iVTLdD1h/SxyxDpZMtimb4CgJQlMYIZ5jCCGeICAQEwWDBBMRMwEQYKCZIm
# iZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQDEwxBTUUg
# Q1MgQ0EgMDECEzYAAAIA7fyNt5zeoUgAAgAAAgAwDQYJYIZIAWUDBAIBBQCgga4w
# GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisG
# AQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEICE1hn8KKTLOnErjbKMUmwkBVhbGqKfw
# a0mx/QHc+KT5MEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8AcwBvAGYA
# dKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEA
# Z9ivUo7Rfxvwaq46ssmoNF0pMDAG7mcFSISAND6Jbi8UdDM/BJsuqmJvwyp1qfMI
# mKtpxfREevpmtM9awxLKLxTxQblbH6vFxEzQ6EfdUNdKH1sEE0lFfly0LreJbpmm
# aRq1mbo87RNODKsua8zcqb+zp8p/ifEJrlvC7hwgp0vKD1Tp4X3qiHF0FJkwQteQ
# SI1m2tuKDEJ2eTmTzsRz1NbIUebqdThmq+17s7TYGZ5B/7eDUHHCbg3vxww+Svxl
# 9DmjdHo7oN734fcgtQ7QX183Lu17uvmkp17/+EjQHEngS2A5MjGERQxBUTBX9gJu
# 7BT46+R/UnUASjw7F+lk7qGCF64wgheqBgorBgEEAYI3AwMBMYIXmjCCF5YGCSqG
# SIb3DQEHAqCCF4cwgheDAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFZBgsqhkiG9w0B
# CRABBKCCAUgEggFEMIIBQAIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFlAwQCAQUA
# BCCW1peHhAxZahJ1D1bl8D+bco78jn8I2S4ij9FiUp2VlAIGZ5DaPfGFGBIyMDI1
# MDEyMjIwNDIzMS44N1owBIACAfSggdmkgdYwgdMxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9w
# ZXJhdGlvbnMgTGltaXRlZDEnMCUGA1UECxMeblNoaWVsZCBUU1MgRVNOOjU1MUEt
# MDVFMC1EOTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNl
# oIIR/TCCBygwggUQoAMCAQICEzMAAAIB0UVZmBDMQk8AAQAAAgEwDQYJKoZIhvcN
# AQELBQAwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
# BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQG
# A1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwHhcNMjQwNzI1MTgz
# MTIyWhcNMjUxMDIyMTgzMTIyWjCB0zELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEtMCsGA1UECxMkTWljcm9zb2Z0IElyZWxhbmQgT3BlcmF0aW9u
# cyBMaW1pdGVkMScwJQYDVQQLEx5uU2hpZWxkIFRTUyBFU046NTUxQS0wNUUwLUQ5
# NDcxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2UwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC1at/4fMO7uyTnTLlgeF4IgkbS7FFI
# UVwc16T5N31mbImPbQQSNpTwkMm7mlZzik8CwEjfw0QAnVv4oGeVK2pTy69cXiqc
# RKEN2Lyc+xllCxNkMpYCBQzaJlM6JYi5lwWzlkLz/4aWtUszmHVj2d8yHkHgOdRA
# 5cyt6YBP0yS9SGDe5piCaouWZjI4OZtriVdkG7XThIsAWxc5+X9MuGlOhPjrLuUj
# 2xsj26rf8B6uHdo+LaSce8QRrOKVd6ihc0sLB274izqjyRAui5SfcrBRCbRvtpS2
# y/Vf86A+aw4mLrI3cthbIchK+s24isniJg2Ad0EG6ZBgrwuNmZBpMoVpzGGZcnSr
# aDNoh/EXbIjAz5X2xCqQeSD9D6JIM2kyvqav87CSc4QiMjSDpkw7KaK+kKHMM2g/
# P2GQreBUdkpbs1Xz5QFc3gbRoFfr18pRvEEEvKTZwL4+E6hSOSXpQLz9zSG6qPnF
# fyb5hUiTzV7u3oj5X8TjJdF55mCvQWFio2m9OMZxo7ZauQ/leaxhLsi8w8h/gMLI
# glRlqqgExOgAkkcZF74M+oIeDpuYY+b3sys5a/Xr8KjpL1xAORen28xJJFBZfLgq
# 0mFl+a4PPa+vWPDg16LHC4gMbDWa1X9N1Ij6+ksl9SIuX9v3D+0kH3YEAtBPx7Vg
# fw2mF06jXELCRwIDAQABo4IBSTCCAUUwHQYDVR0OBBYEFLByr1uWoug8+JWvKb2Y
# WYVZvLJSMB8GA1UdIwQYMBaAFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMF8GA1UdHwRY
# MFYwVKBSoFCGTmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01p
# Y3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0ElMjAyMDEwKDEpLmNybDBsBggrBgEF
# BQcBAQRgMF4wXAYIKwYBBQUHMAKGUGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9w
# a2lvcHMvY2VydHMvTWljcm9zb2Z0JTIwVGltZS1TdGFtcCUyMFBDQSUyMDIwMTAo
# MSkuY3J0MAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwDgYD
# VR0PAQH/BAQDAgeAMA0GCSqGSIb3DQEBCwUAA4ICAQA6NADLPRxXO1MUapdfkktH
# EUr87+gx7nm4OoQLxV3WBtwzbwoFQ+C9Qg9eb+90M3YhUGRYebAKngAhzLh1m2S5
# SZ3R+e7ppP0y+jWd2wunZglwygUsS3dO2uIto76Lgau/RlQu1ZdQ8Bb8yflJyOCf
# TFl24Y8EP9ezcnv6B6337xm8GKmyD83umiKZg5WwfEtx6btXld0w2zK1Ob+4KiaE
# z/CBHkqUNhNU0BcHFxIox4lqIPdXX4eE2RWWIyXlU4/6fDnFYXnm72Hp4XYbt4E+
# pP6pIVD6tAJB0is3TIbVA308muiC4r4UlAl1DN18PdFZWxyIHKBthpmVPVwYkjUj
# JDvgNDRQF1Ol94azKsRD08jxDKpUupvarsu0joMkw2mFi76Ov//SymvVRW/IM+25
# GdsZBE2LUI7AlyP05iaWQWAo14J9sNPtTe4Q69aiZ6RfrRj+bm61FxQ9V4A92GQH
# 4PENp6/cnXLAM13K73XWcBU+BGTIqAwrdRIsbfsR2Vq0OTwXK4KvHi2IfKoc7fAT
# rE/DfZDx7++a5A+gFm5fepR6gUizJkR6cerZJwy6eFypbfZJRUCLmhnhned/t0CA
# 1q7bU0Q/CBb7bCSs2oODsenzIfKg4puAQG7pERBu9J9nkqHg9X5LaDF/a6roahgO
# eWoAE4xjDPfT0hKLRs8yHzCCB3EwggVZoAMCAQICEzMAAAAVxedrngKbSZkAAAAA
# ABUwDQYJKoZIhvcNAQELBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmljYXRlIEF1
# dGhvcml0eSAyMDEwMB4XDTIxMDkzMDE4MjIyNVoXDTMwMDkzMDE4MzIyNVowfDEL
# MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1v
# bmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWlj
# cm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwggIiMA0GCSqGSIb3DQEBAQUAA4IC
# DwAwggIKAoICAQDk4aZM57RyIQt5osvXJHm9DtWC0/3unAcH0qlsTnXIyjVX9gF/
# bErg4r25PhdgM/9cT8dm95VTcVrifkpa/rg2Z4VGIwy1jRPPdzLAEBjoYH1qUoNE
# t6aORmsHFPPFdvWGUNzBRMhxXFExN6AKOG6N7dcP2CZTfDlhAnrEqv1yaa8dq6z2
# Nr41JmTamDu6GnszrYBbfowQHJ1S/rboYiXcag/PXfT+jlPP1uyFVk3v3byNpOOR
# j7I5LFGc6XBpDco2LXCOMcg1KL3jtIckw+DJj361VI/c+gVVmG1oO5pGve2krnop
# N6zL64NF50ZuyjLVwIYwXE8s4mKyzbnijYjklqwBSru+cakXW2dg3viSkR4dPf0g
# z3N9QZpGdc3EXzTdEonW/aUgfX782Z5F37ZyL9t9X4C626p+Nuw2TPYrbqgSUei/
# BQOj0XOmTTd0lBw0gg/wEPK3Rxjtp+iZfD9M269ewvPV2HM9Q07BMzlMjgK8Qmgu
# EOqEUUbi0b1qGFphAXPKZ6Je1yh2AuIzGHLXpyDwwvoSCtdjbwzJNmSLW6CmgyFd
# XzB0kZSU2LlQ+QuJYfM2BjUYhEfb3BvR/bLUHMVr9lxSUV0S2yW6r1AFemzFER1y
# 7435UsSFF5PAPBXbGjfHCBUYP3irRbb1Hode2o+eFnJpxq57t7c+auIurQIDAQAB
# o4IB3TCCAdkwEgYJKwYBBAGCNxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQUKqdS
# /mTEmr6CkTxGNSnPEP8vBO4wHQYDVR0OBBYEFJ+nFV0AXmJdg/Tl0mWnG1M1Gely
# MFwGA1UdIARVMFMwUQYMKwYBBAGCN0yDfQEBMEEwPwYIKwYBBQUHAgEWM2h0dHA6
# Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5Lmh0bTAT
# BgNVHSUEDDAKBggrBgEFBQcDCDAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTAL
# BgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+ii
# XGJo0T2UkFvXzpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0y
# My5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNy
# dDANBgkqhkiG9w0BAQsFAAOCAgEAnVV9/Cqt4SwfZwExJFvhnnJL/Klv6lwUtj5O
# R2R4sQaTlz0xM7U518JxNj/aZGx80HU5bbsPMeTCj/ts0aGUGCLu6WZnOlNN3Zi6
# th542DYunKmCVgADsAW+iehp4LoJ7nvfam++Kctu2D9IdQHZGN5tggz1bSNU5HhT
# dSRXud2f8449xvNo32X2pFaq95W2KFUn0CS9QKC/GbYSEhFdPSfgQJY4rPf5KYnD
# vBewVIVCs/wMnosZiefwC2qBwoEZQhlSdYo2wh3DYXMuLGt7bj8sCXgU6ZGyqVvf
# SaN0DLzskYDSPeZKPmY7T7uG+jIa2Zb0j/aRAfbOxnT99kxybxCrdTDFNLB62FD+
# CljdQDzHVG2dY3RILLFORy3BFARxv2T5JL5zbcqOCb2zAVdJVGTZc9d/HltEAY5a
# GZFrDZ+kKNxnGSgkujhLmm77IVRrakURR6nxt67I6IleT53S0Ex2tVdUCbFpAUR+
# fKFhbHP+CrvsQWY9af3LwUFJfn6Tvsv4O+S3Fb+0zj6lMVGEvL8CwYKiexcdFYmN
# cP7ntdAoGokLjzbaukz5m/8K6TT4JDVnK+ANuOaMmdbhIurwJ0I9JZTmdHRbatGe
# Pu1+oDEzfbzL6Xu/OHBE0ZDxyKs6ijoIYn/ZcGNTTY3ugm2lBRDBcQZqELQdVTNY
# s6FwZvKhggNYMIICQAIBATCCAQGhgdmkgdYwgdMxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9w
# ZXJhdGlvbnMgTGltaXRlZDEnMCUGA1UECxMeblNoaWVsZCBUU1MgRVNOOjU1MUEt
# MDVFMC1EOTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNl
# oiMKAQEwBwYFKw4DAhoDFQDX7bpxH/IfXTQOI0UZaG4C/atgGqCBgzCBgKR+MHwx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1p
# Y3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMA0GCSqGSIb3DQEBCwUAAgUA6ztY
# vTAiGA8yMDI1MDEyMjExNDUwMVoYDzIwMjUwMTIzMTE0NTAxWjB2MDwGCisGAQQB
# hFkKBAExLjAsMAoCBQDrO1i9AgEAMAkCAQACAUcCAf8wBwIBAAICElQwCgIFAOs8
# qj0CAQAwNgYKKwYBBAGEWQoEAjEoMCYwDAYKKwYBBAGEWQoDAqAKMAgCAQACAweh
# IKEKMAgCAQACAwGGoDANBgkqhkiG9w0BAQsFAAOCAQEAADEGrCD6N6NrOcGBCXow
# ebv+wggk98eS0vT1rzZuvX1ZwLvC6YX2D3PABqmt/hC8h1Jpw3CEIeShzjZd41KX
# rJ8cTYzDRa0v4LruXhVd4vOEkqo3BLhuri8+fkj90oxPaFoxIURp+KuMB634F1l7
# GHODrqTYJiN7DkarfVTrUE2wfg+6P+bAg+LHKD6SrCipmWqkCH6MFzMeqJ/klwm4
# 1K82iO3yDF2jx+EUsvm88z20hTWwuo42IuZKdOCGG5uN0hHh1kCPOdd8vlEFdF3p
# aU0/weKejjdATZ5/ph/4upn7m9NRJQa4ec/j4uTf9WK8CtueVHrfDAmgHYT9ZLr2
# OzGCBA0wggQJAgEBMIGTMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5n
# dG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9y
# YXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMz
# AAACAdFFWZgQzEJPAAEAAAIBMA0GCWCGSAFlAwQCAQUAoIIBSjAaBgkqhkiG9w0B
# CQMxDQYLKoZIhvcNAQkQAQQwLwYJKoZIhvcNAQkEMSIEIJo+jhgiJd3sU7Vk/PjH
# ceCAnzMZ32SBL5vGE8SJ5ObBMIH6BgsqhkiG9w0BCRACLzGB6jCB5zCB5DCBvQQg
# WGuyOkyUEXJsdB2kk1mXSYKMHa5ffma69KSHah7XOCMwgZgwgYCkfjB8MQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
# MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3Nv
# ZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAgHRRVmYEMxCTwABAAACATAiBCCz
# 50aWX7TLqoJSy0kT3bwvKKAoNwfavDa9YjXZWopnrTANBgkqhkiG9w0BAQsFAASC
# AgCpSQa78fonDsu/3B62rwBNfn2arAEyNFtEoi5h+44hB99zsLCDRoqGUXEO1XZ7
# WOOJBRXf6fD2CaDIqvwVAeKVUHh5P3Co0AaEJMdo6yb7bBlPG3ZUbLlYb7My/wZH
# smhiRByA5l/2np+t/s3luL57zGQ/6yp7qeHWT/lCTn2MzDyeZgcm0hS7P+zWh5XA
# NrEbMP9NXAVHJ9ZqAhyAxDlWByk8pYQeUFhx56lHKGQQdYaTnRORicLnGmbahM7W
# RgfIRn6P9gFrG0ZAm4oF9Zwsma0W5mlIOOXBSIA0kG3zO2Ici3p00Pw7lyD66tfg
# dGqlsLWzZF0udDBNQYOdBePbhUXRToLzLgVYdBafMKw0vPp5gSV0YHOA1bgyX2YT
# jyq/NrEIvGOwFj+qlzFHE4lEn035LuTPJJZih3LGKyzTRmCiBzjptcXDsJMUQigC
# dEdOWB4CE6OdpX5dozs4tjKM5RTHxuioOtpusOSTPAGd4LPv+Cl0jDWszNA9MlEH
# YXFHuTmWdv+ywku2pE3+GPjwVcKCZ7QrHudmqeOc8guadb5oNd3hBmmw1PIugygn
# wRy2ZQTT95UvZEtHvJN6gJXEnRKl3wJocsBlNC763+lwKdYqQ4yly19g/TaZEKol
# Uye+/5vXBS0v76cu2okF9cSD9QGY99Ep7LJzvTVQJ8Amnw==
# SIG # End signature block
