﻿[CmdletBinding()]
param
(
	# Directory to place logs. Passed in when upgrade is triggered through LCS.
    [string]
	$LogDir
)

Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\AOSEnvironmentUtilities.psm1" -DisableNameChecking

function Get-ApplicationEnvironment
{
    $ErrorActionPreference = 'Stop'
	       
    $webroot = Get-AosWebSitePhysicalPath
    $InstalledPlatformBuildVersion = Get-ProductPlatformBuildFullVersion
	$isKeyVault = Get-IsMSIKeyVault

    if (($isKeyVault -eq $true) -and ($InstalledPlatformBuildVersion -ne $null) -and ($InstalledPlatformBuildVersion.Build -ge 5644))
	{        
        # Get-PackageRoot comes from CommonRollbackUtilities.psm1. it resolves the root of the deployable package
        $packageRoot = Get-PackageRoot
        $packageTemp = ""
        if(![string]::IsNullOrEmpty($packageRoot) -and (Test-Path $packageRoot))
        {
            $packageTemp = Join-Path $packageRoot "RunbookWorkingFolder\KeyVaultAssemblyTemp"
        }

        if ([string]::IsNullOrEmpty($packageTemp))
        {   
            $suffix = $(Get-Date).ToString("yyyyMMdd_HHmmss")
            $dirName = "AssemblyTemp_$suffix"
            $packageTemp = Join-Path $env:Temp "dirName"
        }

        if(!(Test-Path $packageTemp -PathType Container))
        {
            New-Item $packageTemp -ItemType Container | Out-Null
        }
        
        #need to load the Microsoft.Dynamics.ApplicationPlatform.Environment.dll and all the dll it referenced
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Configuration.Base.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.Instrumentation.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Framework.EncryptionEngine.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.BusinessPlatform.SharedTypes.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.KeyVaultHelper.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Carbon.Flighting.Runtime.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Carbon.CertificateCommon.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Commerce.Flighting.Common.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Flighting.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.XppServices.Instrumentation.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.CE.VaultSDK.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.CE.VaultSDK.ADALv2.AuthHelper.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.KeyVault.Core.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.KeyVault.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.Common.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.Common.NetFramework.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Hyak.Common.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Authentication.Instrumentation.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.Extensions.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.Extensions.Desktop.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.IdentityModel.Clients.ActiveDirectory.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Practices.EnterpriseLibrary.TransientFaultHandling.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Newtonsoft.Json.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.SharedUtility.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp    
                
        try
        {
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
            $fileTempPath =  Join-Path $packageTemp $(Get-Item $dllPath).Name
	        if (Test-Path $fileTempPath)
            {
                # Load file from temp folder
	            [System.Reflection.Assembly]::LoadFrom($fileTempPath)
            }
        }
        catch
        {    
            #write-log "Failed to load from temp folder with Exception: [$($_.Exception)]"
            throw $_
        }
	}              
    else
    {
        #need to load the Microsoft.Dynamics.ApplicationPlatform.Environment.dll and all the dll it referenced
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Configuration.Base.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.Instrumentation.dll'
        if (Test-Path $dllPath)
        {
            Load-DllinMemory -dllPath $dllPath
        }
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Framework.EncryptionEngine.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.BusinessPlatform.SharedTypes.dll'
        Load-DllinMemory -dllPath $dllPath  
	}

    $config = [Microsoft.Dynamics.ApplicationPlatform.Environment.EnvironmentFactory]::GetApplicationEnvironment()
    
    return $config
}

function Load-DllinMemory([string] $dllPath)
{
    #try catch as not all dll exist in RTM version, some dependency/dll are introduced at update 1 or later
    #powershell cannot unload dll once it's loaded, the trick is to create an in-memory copy of the dll than load it
    #after the loading of in-memory dll, the physical dll stay unlocked

    try
    {
	    if (Test-Path $dllPath)
        {
            $bytes = [System.IO.File]::ReadAllBytes($dllPath)
            [System.Reflection.Assembly]::Load($bytes) | Out-Null
        }
    }
    catch
    {}
}

function CopyDllToTempFolder([string] $dllPath, [string] $packageTemp)
{
    #try catch as not all dll exist in RTM version, some dependency/dll are introduced at update 1 or later    
    $fileTempPath =  Join-Path $packageTemp $(Get-Item $dllPath).Name
    try
    {
	    if ((Test-Path $dllPath) -And !$(Test-Path $fileTempPath))
        {
            $Buffer = New-Object 'byte[]' 1024
            # Open file in read mode
            $SourceFile = [System.IO.File]::Open($dllPath, [System.IO.FileMode]::Open, [System.IO.FileAccess]::Read, [System.IO.FileShare]::Read)
            # Create the new file
            $DestinationFile = [System.IO.File]::Open($fileTempPath, [System.IO.FileMode]::CreateNew)
            try{
                # Copy the contents of the source file to the destination
                while(($readLength = $SourceFile.Read($Buffer, 0, $Buffer.Length)) -gt 0)
                {
                    $DestinationFile.Write($Buffer, 0, $readLength)
                }
            }
            catch{
                throw $_
            }
            finally{
                $SourceFile.Close()
                $DestinationFile.Close()
            }
        }
    }
    catch
    {
        Write-Log "Failed to copy file to temp folder with Exception: [$($_.Exception)]"
        throw $_
    }    
}

function Get-IsMSIKeyVault
{
    $webroot = Get-AosWebSitePhysicalPath
    $webconfig=join-path $webroot "web.config"
    $isMSIKeyVault=$false
    
    [System.Xml.XmlDocument] $xd=new-object System.Xml.XmlDocument
    $xd.Load($webconfig)
    $ns=New-Object System.Xml.XmlNamespaceManager($xd.NameTable)
    $ns.AddNamespace("ns",$xd.DocumentElement.NamespaceURI)

    $key="CertificateHandler.IsMSIKeyVault"

    $isKeyVaultValue = $xd.SelectSingleNode("//ns:add[@key='$key']",$ns)

    if($isKeyVaultValue -ne $null)
    {        
        $isMSIKeyVault = $isKeyVaultValue.GetAttribute("value")
    }

    return $isMSIKeyVault 
}

Start-Sleep -s 20

Import-Module "$PSScriptRoot\AOSCommon.psm1" -DisableNameChecking

$datetime=get-date -Format "MMddyyyyhhmmss"

if(!$LogDir)
{
    $LogDir = $PSScriptRoot
}

$log=join-path $LogDir "AutoRunDVT_AOS_$datetime.log"
  
Initialize-Log $log

$TestList = Get-ChildItem "$PSScriptRoot\DVT" -Filter 'Validate*.ps1'

$config = Get-ApplicationEnvironment

$ServiceName = $config.Infrastructure.ApplicationName
$endpointValue = $config.Infrastructure.HostUrl
$batchService = 'DynamicsAxBatch'
$serviceState = 'Running'
$appPoolState = 'Started'
$installPath = $config.Infrastructure.AOSWebRoot
$manifestPath = Join-Path $installPath 'web.config'
$appPoolName = $config.Infrastructure.ApplicationName
#Need to update the environmentassembly to add this value.
[xml]$WebConfig = Get-Content "$(join-path $installPath "web.config")"
$IsPrivateAOSInstance = $($WebConfig.configuration.appSettings.add | where { $_.key -eq 'Infrastructure.IsPrivateAOSInstance' }).value
$ValidateBatch = ([System.String]::IsNullOrWhiteSpace($IsPrivateAOSInstance) -or ![System.Convert]::ToBoolean($IsPrivateAOSInstance))

if (!$env:SERVICEDRIVE)
{
	Write-Log '%SERVICEDRIVE% was not defined. Exiting.' -Warning
	exit
}

Write-Log "Running DVT Process for $ServiceName"

$DiagnosticsPath = Join-Path -Path $env:SERVICEDRIVE -ChildPath "DynamicsDiagnostics\$ServiceName"

foreach ($test in $testList)
{
    Write-Log "Processing $test"    
    
    $DVTLocalRoot = Join-Path -Path $DiagnosticsPath -ChildPath ([System.IO.Path]::GetFileNameWithoutExtension($test))

    $DVTLocalBin = Join-Path $DVTLocalRoot "input"
    if(-not (Test-Path -Path $DVTLocalBin))
    {
        Write-Log "Creating DVT local bin at: '$DVTLocalBin'"
        New-Item -Path $DVTLocalBin -Type Directory -Force | Out-Null
    }   
        
    $DVTScript = Join-Path -Path "$PSScriptRoot\DVT" -ChildPath $test
    Write-Log "Copy DVT Script '$DVTScript' to $DVTLocalBin"
    Copy-Item -Path $DVTScript -Destination $DVTLocalBin -Recurse -Force | Out-Null

    Write-Log "Copy AosCommon.psm1 to DVTLocalBin"
    $AosCommon = Join-Path -Path $PSScriptRoot -ChildPath "AosCommon.psm1"
    Copy-Item -Path $AosCommon -Destination $DVTLocalBin -Recurse -Force | Out-Null

        
    [string]$DVTOutputBin = Join-Path -Path $DVTLocalRoot -ChildPath "Output"

    ####################################################################################
    ## DVT input XML Template
    ####################################################################################
    [xml]$xmlTemplate = "<?xml version=`"1.0`"?>
    <DVTParameters xmlns:xsi=`"http://www.w3.org/2001/XMLSchema-instance`" xmlns:xsd=`"http://www.w3.org/2001/XMLSchema`">
        <AosWebRootPath>$installPath</AosWebRootPath>
        <ServiceName>$ServiceName</ServiceName>        
        <OutputPath>$DVTOutputBin</OutputPath>
        <EndPoint>$endpointValue</EndPoint>
        <InstallPath>$installPath</InstallPath>
        <ManifestPath>$manifestPath</ManifestPath>
        <BatchService>$batchService</BatchService>
        <ServiceState>$serviceState</ServiceState>
        <AppPoolName>$appPoolName</AppPoolName>
        <AppPoolState>$appPoolState</AppPoolState>
        <ValidateBatch>$ValidateBatch</ValidateBatch>
    </DVTParameters>"

    $XMLInputPath = Join-Path -Path $DVTLocalBin -ChildPath "$([System.IO.Path]::GetFileNameWithoutExtension($test)).xml"
    Write-Log "Executing DVT XML at: $XMLInputPath"
    $xmlTemplate.InnerXml | Out-File -FilePath $XMLInputPath -Force -Encoding utf8
    
    ####################################################################################
    ## Execute DVT Script
    ####################################################################################
    try
    {
        $DVTLocalScript = Join-Path -Path $DVTLocalBin -ChildPath $test
        if(Test-Path -Path $DVTLocalScript)
        {
            Write-Log "Executing DVT Script: $DVTLocalScript"
            $commandArgs = @{
                "InputXML" = $XMLInputPath;                
                "Log" = $Log
            }
    
            & $DVTLocalScript @commandArgs
            
        }
        else
        {
            throw "$DVTLocalScript was not found."
        }
    }
    catch
    {     
        Write-Exception $_       
        throw "DVT Script Failed, see $log for details."
    }
}    

# SIG # Begin signature block
# MIIr/AYJKoZIhvcNAQcCoIIr7TCCK+kCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCA86YOltOSTzBet
# okCiOQ7oUgszHKt0kwcQFZAZfzQiC6CCEW4wggh+MIIHZqADAgECAhM2AAACAO38
# jbec3qFIAAIAAAIAMA0GCSqGSIb3DQEBCwUAMEExEzARBgoJkiaJk/IsZAEZFgNH
# QkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxFTATBgNVBAMTDEFNRSBDUyBDQSAwMTAe
# Fw0yNDExMDgxMjQzMjhaFw0yNTExMDgxMjQzMjhaMCQxIjAgBgNVBAMTGU1pY3Jv
# c29mdCBBenVyZSBDb2RlIFNpZ24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK
# AoIBAQC5L/UPrOpwYjxcoZC0TqqvMF1WUELvwXN+k27SrA5rohJknn7Cgbxg4hGT
# XKqpcdbtsVTN3ZY896SJ20uQ+INL5OVLzpW408nCNTPYg2LtGJbqHUjpNm0hLCJ+
# gO5Jn2T8DDzIJoUijGXj1m+hRLKb2nOIicCED2GuYBmuWXnaY7INmVEaU3peryty
# ZjDuxdyGDuiPURz8lW1SUiDzoszNp1oswVr+WjDvLDUx4HlxPsG8zUjIst0NnJ6o
# z4tNFKaUBDCetcMjQxpCETn29a1CuRddxZLjPHZHfcotr5sh1S6bNQdzVaMNsxV8
# L3wjHb7XJ6ZVm662mHEiPgpyNcLhAgMBAAGjggWKMIIFhjApBgkrBgEEAYI3FQoE
# HDAaMAwGCisGAQQBgjdbAQEwCgYIKwYBBQUHAwMwPQYJKwYBBAGCNxUHBDAwLgYm
# KwYBBAGCNxUIhpDjDYTVtHiE8Ys+hZvdFs6dEoFgg93NZoaUjDICAWQCAQ4wggJ2
# BggrBgEFBQcBAQSCAmgwggJkMGIGCCsGAQUFBzAChlZodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpaW5mcmEvQ2VydHMvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDEu
# YW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUy
# MDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDIuYW1lLmdibC9haWEv
# QlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBS
# BggrBgEFBQcwAoZGaHR0cDovL2NybDMuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAx
# LkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZG
# aHR0cDovL2NybDQuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDCBrQYIKwYBBQUHMAKGgaBsZGFwOi8vL0NO
# PUFNRSUyMENTJTIwQ0ElMjAwMSxDTj1BSUEsQ049UHVibGljJTIwS2V5JTIwU2Vy
# dmljZXMsQ049U2VydmljZXMsQ049Q29uZmlndXJhdGlvbixEQz1BTUUsREM9R0JM
# P2NBQ2VydGlmaWNhdGU/YmFzZT9vYmplY3RDbGFzcz1jZXJ0aWZpY2F0aW9uQXV0
# aG9yaXR5MB0GA1UdDgQWBBST/HE52ZUlmsYqZcZBdrXZ5u4ZnzAOBgNVHQ8BAf8E
# BAMCB4AwRQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEWMBQGA1UEBRMNMjM2MTY3KzUwMzE1NTCCAeYGA1UdHwSCAd0wggHZMIIB
# 1aCCAdGgggHNhj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpaW5mcmEvQ1JM
# L0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwxLmFtZS5nYmwv
# Y3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwyLmFtZS5n
# YmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwzLmFt
# ZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmw0
# LmFtZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGgb1sZGFwOi8v
# L0NOPUFNRSUyMENTJTIwQ0ElMjAwMSgyKSxDTj1CWTJQS0lDU0NBMDEsQ049Q0RQ
# LENOPVB1YmxpYyUyMEtleSUyMFNlcnZpY2VzLENOPVNlcnZpY2VzLENOPUNvbmZp
# Z3VyYXRpb24sREM9QU1FLERDPUdCTD9jZXJ0aWZpY2F0ZVJldm9jYXRpb25MaXN0
# P2Jhc2U/b2JqZWN0Q2xhc3M9Y1JMRGlzdHJpYnV0aW9uUG9pbnQwHwYDVR0jBBgw
# FoAUllGE4Gtve/7YBqvD8oXmKa5q+dQwHwYDVR0lBBgwFgYKKwYBBAGCN1sBAQYI
# KwYBBQUHAwMwDQYJKoZIhvcNAQELBQADggEBAEDd8Wf5RkHsB64vgn2slxDtHzSo
# It9xN/Dm3RdFjNZ0diTUPMgSPYQlSk8nIAfudnB9FLavGlvZLlyUpfrPSuikepj3
# i3pqNEFn6fNdNFv/wHMxv7hQTIDCmuoR1v1rX+w3oeleBPMnN3QmH4ff1NsynyV4
# dZdYgN9Cw9sC/S3pWZpJrbOs7YOM3vqyU6DciHhC4D9i2zByHCF2pu9nYfiQf5A2
# iUZenRvyo1E5rC+UP2VZXa4k7g66W20+zAajIKKIqEmRtWahekMkCcOIHFBY4RDA
# ybgPRSGur4VDAiZPjTXS90wQXrX9CwU20cfiCC6e76F4H95KtQjKYpzuNVAwggjo
# MIIG0KADAgECAhMfAAAAUeqP9pxzDKg7AAAAAABRMA0GCSqGSIb3DQEBCwUAMDwx
# EzARBgoJkiaJk/IsZAEZFgNHQkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxEDAOBgNV
# BAMTB2FtZXJvb3QwHhcNMjEwNTIxMTg0NDE0WhcNMjYwNTIxMTg1NDE0WjBBMRMw
# EQYKCZImiZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQD
# EwxBTUUgQ1MgQ0EgMDEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDJ
# mlIJfQGejVbXKpcyFPoFSUllalrinfEV6JMc7i+bZDoL9rNHnHDGfJgeuRIYO1LY
# /1f4oMTrhXbSaYRCS5vGc8145WcTZG908bGDCWr4GFLc411WxA+Pv2rteAcz0eHM
# H36qTQ8L0o3XOb2n+x7KJFLokXV1s6pF/WlSXsUBXGaCIIWBXyEchv+sM9eKDsUO
# LdLTITHYJQNWkiryMSEbxqdQUTVZjEz6eLRLkofDAo8pXirIYOgM770CYOiZrcKH
# K7lYOVblx22pdNawY8Te6a2dfoCaWV1QUuazg5VHiC4p/6fksgEILptOKhx9c+ia
# piNhMrHsAYx9pUtppeaFAgMBAAGjggTcMIIE2DASBgkrBgEEAYI3FQEEBQIDAgAC
# MCMGCSsGAQQBgjcVAgQWBBQSaCRCIUfL1Gu+Mc8gpMALI38/RzAdBgNVHQ4EFgQU
# llGE4Gtve/7YBqvD8oXmKa5q+dQwggEEBgNVHSUEgfwwgfkGBysGAQUCAwUGCCsG
# AQUFBwMBBggrBgEFBQcDAgYKKwYBBAGCNxQCAQYJKwYBBAGCNxUGBgorBgEEAYI3
# CgMMBgkrBgEEAYI3FQYGCCsGAQUFBwMJBggrBgEFBQgCAgYKKwYBBAGCN0ABAQYL
# KwYBBAGCNwoDBAEGCisGAQQBgjcKAwQGCSsGAQQBgjcVBQYKKwYBBAGCNxQCAgYK
# KwYBBAGCNxQCAwYIKwYBBQUHAwMGCisGAQQBgjdbAQEGCisGAQQBgjdbAgEGCisG
# AQQBgjdbAwEGCisGAQQBgjdbBQEGCisGAQQBgjdbBAEGCisGAQQBgjdbBAIwGQYJ
# KwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMBIGA1UdEwEB/wQI
# MAYBAf8CAQAwHwYDVR0jBBgwFoAUKV5RXmSuNLnrrJwNp4x1AdEJCygwggFoBgNV
# HR8EggFfMIIBWzCCAVegggFToIIBT4YxaHR0cDovL2NybC5taWNyb3NvZnQuY29t
# L3BraWluZnJhL2NybC9hbWVyb290LmNybIYjaHR0cDovL2NybDIuYW1lLmdibC9j
# cmwvYW1lcm9vdC5jcmyGI2h0dHA6Ly9jcmwzLmFtZS5nYmwvY3JsL2FtZXJvb3Qu
# Y3JshiNodHRwOi8vY3JsMS5hbWUuZ2JsL2NybC9hbWVyb290LmNybIaBqmxkYXA6
# Ly8vQ049YW1lcm9vdCxDTj1BTUVSb290LENOPUNEUCxDTj1QdWJsaWMlMjBLZXkl
# MjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9uLERDPUFNRSxE
# Qz1HQkw/Y2VydGlmaWNhdGVSZXZvY2F0aW9uTGlzdD9iYXNlP29iamVjdENsYXNz
# PWNSTERpc3RyaWJ1dGlvblBvaW50MIIBqwYIKwYBBQUHAQEEggGdMIIBmTBHBggr
# BgEFBQcwAoY7aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraWluZnJhL2NlcnRz
# L0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6Ly9jcmwyLmFt
# ZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6
# Ly9jcmwzLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUH
# MAKGK2h0dHA6Ly9jcmwxLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQw
# gaIGCCsGAQUFBzAChoGVbGRhcDovLy9DTj1hbWVyb290LENOPUFJQSxDTj1QdWJs
# aWMlMjBLZXklMjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9u
# LERDPUFNRSxEQz1HQkw/Y0FDZXJ0aWZpY2F0ZT9iYXNlP29iamVjdENsYXNzPWNl
# cnRpZmljYXRpb25BdXRob3JpdHkwDQYJKoZIhvcNAQELBQADggIBAFAQI7dPD+jf
# XtGt3vJp2pyzA/HUu8hjKaRpM3opya5G3ocprRd7vdTHb8BDfRN+AD0YEmeDB5HK
# QoG6xHPI5TXuIi5sm/LeADbV3C2q0HQOygS/VT+m1W7a/752hMIn+L4ZuyxVeSBp
# fwf7oQ4YSZPh6+ngZvBHgfBaVz4O9/wcfw91QDZnTgK9zAh9yRKKls2bziPEnxeO
# ZMVNaxyV0v152PY2xjqIafIkUjK6vY9LtVFjJXenVUAmn3WCPWNFC1YTIIHw/mD2
# cTfPy7QA1pT+GPARAKt0bKtq9aCd/Ym0b5tPbpgCiRtzyb7fbNS1dE740re0COE6
# 7YV2wbeo2sXixzvLftH8L7s9xv9wV+G22qyKt6lmKLjFK1yMw4Ni5fMabcgmzRvS
# jAcbqgp3tk4a8emaaH0rz8MuuIP+yrxtREPXSqL/C5bzMzsikuDW9xH10graZzSm
# PjilzpRfRdu20/9UQmC7eVPZ4j1WNa1oqPHfzET3ChIzJ6Q9G3NPCB+7KwX0OQmK
# yv7IDimj8U/GlsHD1z+EF/fYMf8YXG15LamaOAohsw/ywO6SYSreVW+5Y0mzJutn
# BC9Cm9ozj1+/4kqksrlhZgR/CSxhFH3BTweH8gP2FEISRtShDZbuYymynY1un+Ry
# fiK9+iVTLdD1h/SxyxDpZMtimb4CgJQlMYIZ5DCCGeACAQEwWDBBMRMwEQYKCZIm
# iZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQDEwxBTUUg
# Q1MgQ0EgMDECEzYAAAIA7fyNt5zeoUgAAgAAAgAwDQYJYIZIAWUDBAIBBQCgga4w
# GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisG
# AQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIE8s487YIla5Mud8WPXNDPtKsVAIfw81
# KYDvAyHeLGpNMEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8AcwBvAGYA
# dKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEA
# t6LyazmlPkI4Sxq5EAQ995mlDVAGgQT/dXhIzh6xLcXEv3SCd49VfHTX6un8rICD
# sdaMFyIheEGhMv5G35lxSQMiAbpQbgUzk1tlZjCNGMfgFVsWJEl5ZoXXIHnjgnHq
# r+kmxeexby42dnIFAFt3JNr6Dplz58uXxA4oAJt1n+8aeuteAA7N2Vx3DM/enJtd
# 8Lm0rt9zOEQJsQELKvWWGznkN/q6zIDDBnPWahvteeg2OtBE/1/znS8LKdY3Jzdx
# yZDvrTpIdnYNkyT0k7XPiXT+1Hs6ZWFYkBr0viwRCfjfmMAshWkjbGfdRjr5fWD/
# Rsu697w0jUOCh1s2eUedhKGCF6wwgheoBgorBgEEAYI3AwMBMYIXmDCCF5QGCSqG
# SIb3DQEHAqCCF4UwgheBAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFaBgsqhkiG9w0B
# CRABBKCCAUkEggFFMIIBQQIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFlAwQCAQUA
# BCDcQHsSxOVcwtDS+S8ShH5L6aXVi4jf2RljqmyBokOEAgIGZ5Eh2NinGBMyMDI1
# MDEyMjIwNDIyNS44OTRaMASAAgH0oIHZpIHWMIHTMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJlbGFuZCBP
# cGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjo1MjFB
# LTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZaCCEfowggcoMIIFEKADAgECAhMzAAACAAvXqn8bKhdWAAEAAAIAMA0GCSqGSIb3
# DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
# VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAk
# BgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMB4XDTI0MDcyNTE4
# MzEyMVoXDTI1MTAyMjE4MzEyMVowgdMxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
# YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
# Q29ycG9yYXRpb24xLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlv
# bnMgTGltaXRlZDEnMCUGA1UECxMeblNoaWVsZCBUU1MgRVNOOjUyMUEtMDVFMC1E
# OTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNlMIICIjAN
# BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAr1XaadKkP2TkunoTF573/tF7KJM9
# Doiv3ccv26mqnUhmv2DM59ikET4WnRfo5biFIHc6LqrIeqCgT9fT/Gks5VKO90ZQ
# W2avh/PMHnl0kZfX/I5zdVooXHbdUUkPiZfNXszWswmL9UlWo8mzyv9Lp9TAtw/o
# XOYTAxdYSqOB5Uzz1Q3A8uCpNlumQNDJGDY6cSn0MlYukXklArChq6l+KYrl6r/W
# nOqXSknABpggSsJ33oL3onmDiN9YUApZwjnNh9M6kDaneSz78/YtD/2pGpx9/LXE
# LoazEUFxhyg4KdmoWGNYwdR7/id81geOER69l5dJv71S/mH+Lxb6L692n8uEmAVw
# 6fVvE+c8wjgYZblZCNPAynCnDduRLdk1jswCqjqNc3X/WIzA7GGs4HUS4YIrAUx8
# H2A94vDNiA8AWa7Z/HSwTCyIgeVbldXYM2BtxMKq3kneRoT27NQ7Y7n8ZTaAje7B
# lfju83spGP/QWYNZ1wYzYVGRyOpdA8Wmxq5V8f5r4HaG9zPcykOyJpRZy+V3RGig
# hFmsCJXAcMziO76HinwCIjImnCFKGJ/IbLjH6J7fJXqRPbg+H6rYLZ8XBpmXBFH4
# PTakZVYxB/P+EQbL5LNw0ZIM+eufxCljV4O+nHkM+zgSx8+07BVZPBKslooebsmh
# IcBO0779kehciYMCAwEAAaOCAUkwggFFMB0GA1UdDgQWBBSAJSTavgkjKqge5xQO
# Xn35fXd3OjAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJlpxtTNRnpcjBfBgNVHR8E
# WDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9N
# aWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcmwwbAYIKwYB
# BQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20v
# cGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0ElMjAyMDEw
# KDEpLmNydDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMA4G
# A1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQsFAAOCAgEAKPCG9njRtIqQ+fuECgxz
# WMsQOI3HvW7sV9PmEWCCOWlTuGCIzNi3ibdLZS0b2IDHg0yLrtdVuBi3FxVdesIX
# uzYyofIe/alTBdV4DhijLTXtB7NgOno7G12iO3t6jy1hPSquzGLry/2mEZBwIsSo
# S2D+H+3HCJxPDyhzMFqP+plltPACB/QNwZ7q+HGyZv3v8et+rQYg8sF3PTuWeDg3
# dR/zk1NawJ/dfFCDYlWNeCBCLvNPQBceMYXFRFKhcSUws7mFdIDDhZpxqyIKD2WD
# wFyNIGEezn+nd4kXRupeNEx+eSpJXylRD+1d45hb6PzOIF7BkcPtRtFW2wXgkjLq
# tTWWlBkvzl2uNfYJ3CPZVaDyMDaaXgO+H6DirsJ4IG9ikId941+mWDejkj5aYn9Q
# N6ROfo/HNHg1timwpFoUivqAFu6irWZFw5V+yLr8FLc7nbMa2lFSixzu96zdnDsP
# Imz0c6StbYyhKSlM3uDRi9UWydSKqnEbtJ6Mk+YuxvzprkuWQJYWfpPvug+wTnio
# ykVwc0yRVcsd4xMznnnRtZDGMSUEl9tMVnebYRshwZIyJTsBgLZmHM7q2TFK/X99
# 44SkIqyY22AcuLe0GqoNfASCIcZtzbZ/zP4lT2/N0pDbn2ffAzjZkhI+Qrqr983m
# QZWwZdr3Tk1MYElDThz2D0MwggdxMIIFWaADAgECAhMzAAAAFcXna54Cm0mZAAAA
# AAAVMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2Fz
# aGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
# cnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0ZSBB
# dXRob3JpdHkgMjAxMDAeFw0yMTA5MzAxODIyMjVaFw0zMDA5MzAxODMyMjVaMHwx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1p
# Y3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIICIjANBgkqhkiG9w0BAQEFAAOC
# Ag8AMIICCgKCAgEA5OGmTOe0ciELeaLL1yR5vQ7VgtP97pwHB9KpbE51yMo1V/YB
# f2xK4OK9uT4XYDP/XE/HZveVU3Fa4n5KWv64NmeFRiMMtY0Tz3cywBAY6GB9alKD
# RLemjkZrBxTzxXb1hlDcwUTIcVxRMTegCjhuje3XD9gmU3w5YQJ6xKr9cmmvHaus
# 9ja+NSZk2pg7uhp7M62AW36MEBydUv626GIl3GoPz130/o5Tz9bshVZN7928jaTj
# kY+yOSxRnOlwaQ3KNi1wjjHINSi947SHJMPgyY9+tVSP3PoFVZhtaDuaRr3tpK56
# KTesy+uDRedGbsoy1cCGMFxPLOJiss254o2I5JasAUq7vnGpF1tnYN74kpEeHT39
# IM9zfUGaRnXNxF803RKJ1v2lIH1+/NmeRd+2ci/bfV+AutuqfjbsNkz2K26oElHo
# vwUDo9Fzpk03dJQcNIIP8BDyt0cY7afomXw/TNuvXsLz1dhzPUNOwTM5TI4CvEJo
# LhDqhFFG4tG9ahhaYQFzymeiXtcodgLiMxhy16cg8ML6EgrXY28MyTZki1ugpoMh
# XV8wdJGUlNi5UPkLiWHzNgY1GIRH29wb0f2y1BzFa/ZcUlFdEtsluq9QBXpsxREd
# cu+N+VLEhReTwDwV2xo3xwgVGD94q0W29R6HXtqPnhZyacaue7e3PmriLq0CAwEA
# AaOCAd0wggHZMBIGCSsGAQQBgjcVAQQFAgMBAAEwIwYJKwYBBAGCNxUCBBYEFCqn
# Uv5kxJq+gpE8RjUpzxD/LwTuMB0GA1UdDgQWBBSfpxVdAF5iXYP05dJlpxtTNRnp
# cjBcBgNVHSAEVTBTMFEGDCsGAQQBgjdMg30BATBBMD8GCCsGAQUFBwIBFjNodHRw
# Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9yeS5odG0w
# EwYDVR0lBAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
# CwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYDVR0jBBgwFoAU1fZWy4/o
# olxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDovL2NybC5taWNy
# b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYt
# MjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+aHR0cDovL3d3dy5t
# aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5j
# cnQwDQYJKoZIhvcNAQELBQADggIBAJ1VffwqreEsH2cBMSRb4Z5yS/ypb+pcFLY+
# TkdkeLEGk5c9MTO1OdfCcTY/2mRsfNB1OW27DzHkwo/7bNGhlBgi7ulmZzpTTd2Y
# urYeeNg2LpypglYAA7AFvonoaeC6Ce5732pvvinLbtg/SHUB2RjebYIM9W0jVOR4
# U3UkV7ndn/OOPcbzaN9l9qRWqveVtihVJ9AkvUCgvxm2EhIRXT0n4ECWOKz3+SmJ
# w7wXsFSFQrP8DJ6LGYnn8AtqgcKBGUIZUnWKNsIdw2FzLixre24/LAl4FOmRsqlb
# 30mjdAy87JGA0j3mSj5mO0+7hvoyGtmW9I/2kQH2zsZ0/fZMcm8Qq3UwxTSwethQ
# /gpY3UA8x1RtnWN0SCyxTkctwRQEcb9k+SS+c23Kjgm9swFXSVRk2XPXfx5bRAGO
# WhmRaw2fpCjcZxkoJLo4S5pu+yFUa2pFEUep8beuyOiJXk+d0tBMdrVXVAmxaQFE
# fnyhYWxz/gq77EFmPWn9y8FBSX5+k77L+DvktxW/tM4+pTFRhLy/AsGConsXHRWJ
# jXD+57XQKBqJC4822rpM+Zv/Cuk0+CQ1ZyvgDbjmjJnW4SLq8CdCPSWU5nR0W2rR
# nj7tfqAxM328y+l7vzhwRNGQ8cirOoo6CGJ/2XBjU02N7oJtpQUQwXEGahC0HVUz
# WLOhcGbyoYIDVTCCAj0CAQEwggEBoYHZpIHWMIHTMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJlbGFuZCBP
# cGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjo1MjFB
# LTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZaIjCgEBMAcGBSsOAwIaAxUAjJOfLZb3ivipL3sSLlWFbLrWjmSggYMwgYCkfjB8
# MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
# bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1N
# aWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQsFAAIFAOs7
# oFYwIhgPMjAyNTAxMjIxNjUwMzBaGA8yMDI1MDEyMzE2NTAzMFowczA5BgorBgEE
# AYRZCgQBMSswKTAKAgUA6zugVgIBADAGAgEAAgFQMAcCAQACAhNyMAoCBQDrPPHW
# AgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkKAwKgCjAIAgEAAgMHoSCh
# CjAIAgEAAgMBhqAwDQYJKoZIhvcNAQELBQADggEBAFBFwmiwpIHTHSSIhHT3VNak
# hSrUPY2flh8OK6YH4KpAlctDDMepPrLfXrD7tGG2YKhjBpFle/uS0jnPWCuq4bH7
# le6sMHBmuUR3nrn+GnmSrQaM8CpASWvfogG/Pug/+qVg9olce2B6OWGjMJaT24GO
# T2ynDsqBEmfe38g2NQbxM9cpJszVqnTkpU8eoPppdYgO664XtRl1WzjZP71zzk5H
# ovykuJ4kWgrtUIfDSRxBDFN++nDbpLPmEavru4KoLQqwRHQlIby8rHjiuOvH40/M
# flPD6tt/YS8gL5HPowVJzLLNKqFSU3+IY+qeAF5iO/8aHjLgttZsU+6oF/uJ4asx
# ggQNMIIECQIBATCBkzB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAA
# AgAL16p/GyoXVgABAAACADANBglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkD
# MQ0GCyqGSIb3DQEJEAEEMC8GCSqGSIb3DQEJBDEiBCAqP0IPO3nN0vApV2tQ8qy4
# t2KtGe0Bf7DcXM8p1CzLFDCB+gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EINTI
# 7ew1ndu6sE0MZQJXg18zaAfhpa5G50iT/0oCT9knMIGYMIGApH4wfDELMAkGA1UE
# BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAc
# BgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0
# IFRpbWUtU3RhbXAgUENBIDIwMTACEzMAAAIAC9eqfxsqF1YAAQAAAgAwIgQgqbiz
# IxX/Ds5MkbPzU0fGq5UDvetlxCGg6gfqWH51XUIwDQYJKoZIhvcNAQELBQAEggIA
# DroCYO9dGnrel2Dheta32uzJ4nTvIFMsph8tyLY4gMI/RwNzK1zOOObNsS4xacGp
# 1Q31/HAAAqJoBgiLUgFe+U9LiFjVvc6SkMu4lNu8wJr8+sAnwMoEYQ9Ucrre/KuM
# Qm1zAilbGaozB0lsH2Hz6RYnXXphZSbBxbscvohTd7iiNgyiJJxGM6oC49lPLxBR
# Mha32pW+C8StsMWB0fAFmJnB1kW8Qsmh3pEQ7t2lSek7XOew9qHef89U+Yju1WVW
# E7HyehNnebbuXfWtHi/Zo96P6WQoqbE2m1ksBf58zejGqSQ/p/2HwrE0kBZSV6SF
# mHdCDh9UbQJxik2rl+rP5f81E+zx9A1+roWwpl4QTsiEvsn2sd2epSWhYXaogEEf
# a3kPVVAYDu0HwsziNNkPbIDrn+cnlPnRJkSrRlNBD+6MtgkBhuYZqCgm1kDptLAq
# SatpSYCn4kdroZEGpk0nNVWzZWgLEIqIeQqS1jdTnDrn8RWae45Ax3cbNrwXabPU
# uFjV0rMcY3JENNDEyo0JarOjh6/Yhw0LTjIwboT7sxEVFZeNhhcZFeRMNc6Hr/a6
# ybs3HP2gHdcM34c8LiibkAQE7u8x4Jzq+42Su9JsFIWtux9MVOMHLjDEx02eBr8J
# XN8WqYl7+wO0moeo5bRGZEU1zsITSvF+L7oSqao6qpM=
# SIG # End signature block
