﻿[CmdletBinding()]
param
(
    [Parameter(Mandatory=$false)]
    [string]$LogDir,

    [parameter(Mandatory=$false)]
    [boolean] $useServiceFabric=$false,

    [Parameter(Mandatory = $false)]
    [string]$ActivityId,

    [Parameter(Mandatory = $false)]
    [string]$RunbookId,

    [Parameter(Mandatory=$false)]
	[hashtable]$AxDRConnectionString,

    [Parameter(Mandatory=$false)]
	[string[]]$EnabledServicingFeatures,

    [Parameter(Mandatory=$false)]
	[Switch]$PreDowntime,

    [Parameter(Mandatory=$false)]
    [hashtable]$Credentials
)

# Pre-defined list of error messages for known customer induced issues
$knownErrorHash = @{
    "The CREATE UNIQUE INDEX statement terminated because a duplicate key" = 131079
    "Please drop the original field, sync the table" = 131089
    "extension cannot be applied to the base element" = 131090
    "is missing the following dependencies" = 131091
    "Method not found" = 131092
}

# Check pre check status
function IsPreCheckSucceeded ([string]$ActivityId)
{
  if(!$ActivityId)
  {
    return $false
  }

  $axDBUser = Get-DataAccessSqlUsr

  if($Credentials -and $Credentials[$($axDBUser)])  {
     $axDBPwd = $Credentials[$($axDBUser)]
     EventWrite-RunbookScriptTrace -Message "Using axdbadmin password from DSU parameters." -Component "servicing" -RunbookId $RunbookId| Out-Null
  }
  else{
     $axDBPwd = Get-DataAccessSqlPwd
     EventWrite-RunbookScriptTrace -Message "Using axdbadmin password from Web config." -Component "servicing" -RunbookId $RunbookId| Out-Null
  }
  
  $axDBDatabaseName = Get-DataAccessDatabase
  $axDBServer = Get-DataAccessDbServer

    $target_sqlParams = @{
		'Database' = $axDBDatabaseName
		'UserName' = $axDBUser
		'Password' = $axDBPwd
		'ServerInstance' = $axDBServer
		'EncryptConnection' = !(IsLocalServer($axDBServer))
		'QueryTimeout' = 0
	    }

    $target_sqlParams.Query = "Select top 1 Success from DBSyncExecStats where SyncStep = 'PreCheck' and RelatedActivityId = '$ActivityId' order by StartDateTime desc"

    try{
        # during first precheck on this environment table will not be exists so catch exception and consider running precheck.
        $result = Invoke-SqlCmd @target_sqlParams
    }
    catch
    {
        $warningMessage = "[$(Get-Date)]:Get pre check status failed, activity id: $ActivityId, Exception: [$($_.Exception)]"
        Write-Host $warningMessage
        EventWrite-RunbookScriptTrace -Message $warningMessage -Component "servicing" -RunbookId $RunbookId| Out-Null
    }

    if ($result -ne $null)
    {
        return $result.Success
    }

    return $false
}

# Run db sync precheck to evaluate DB Sync would fail or succeed.
function RunDBSyncPreCheck([string]$ActivityId, [hashtable]$AxDRConnectionString, [string[]]$EnabledServicingFeatures)
{
    if (!$EnabledServicingFeatures -or $EnabledServicingFeatures -notcontains "DBSyncPreCheck")
    { 
       EventWrite-RunbookScriptTrace -Message "DBSyncPreCheck feature is disabled. Skipping db sync pre check, activity id: $ActivityId" -Component "servicing"  -RunbookId $RunbookId| Out-Null
       return $false
    }

    # Check if previous precheck status for activity id
    if($ActivityId)
    {
        $PreCheckResult = IsPreCheckSucceeded -ActivityId $ActivityId
        if($PreCheckResult -eq $true) 
        {
           # For prod scenario precheck will run during preparation, so we do not need to run it again.
           EventWrite-RunbookScriptTrace -Message "[$(Get-Date)]: Pre service check already succeeded for activity id: $ActivityId, Skipping precheck..." -Component "servicing" -RunbookId $RunbookId| Out-Null
           return $false
        }
     }

    # Run Precheck
    $currentDateTime = [DateTime]::UtcNow.ToString("yyyyMMddHHmmss")
    $logfile = Join-Path $LogDir "dbsyncprecheck_$currentDateTime.log"
    $errorfile = Join-Path $LogDir "dbsyncprecheck.error_$currentDateTime.log"
    Write-Host "Precheck Started..."
   
    if($PreDowntime -eq $true)
    {
      $aosWebServiceStagingPath = Get-AosServiceStagingPath
      $webroot = Join-Path -Path $aosWebServiceStagingPath -ChildPath "WebRoot"
      $PUVersion = Get-ProductPlatformBuildVersion -webroot:$webroot

      $metadatadir = Join-Path -Path $aosWebServiceStagingPath -ChildPath "PackagesLocalDirectory"
    }
    else
    {
      $PUVersion = (Get-ProductPlatformBuildVersion)
      $metadatadir = $(Get-AOSPackageDirectory)
    }

    Write-Host "AOS platform version is: $PUVersion"

    # If PU version >= PU34 then run precheck on secondary DB if exists.
    $runPrecheckOnSecondaryDB = $false
    if($PUVersion -ge 5629 -and $AxDRConnectionString -and $AxDRConnectionString['AxDRServerName'] -and $AxDRConnectionString['AxDRDatabaseName'])
    {
        $runPrecheckOnSecondaryDB =  $true
        $sqlServer =  $AxDRConnectionString['AxDRServerName']
        $sqlDB =  $AxDRConnectionString['AxDRDatabaseName']
        EventWrite-RunbookScriptTrace -Message "[$(Get-Date)]: Running precheck on secondary DB" -Component "servicing" -RunbookId $RunbookId| Out-Null
    }

    # If platform version > PU35 then run precheck with deployment setup exe
    if ($PUVersion -gt 5644)
    {
        EventWrite-RunbookScriptTrace -Message "Running precheck with deployment setup exe, activity id $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null
        $deploymentSetupParameter = "-setupmode preservicecheck -syncmode precheck"
        
        if($ActivityId) 
        {
            $deploymentSetupParameter += " -activityid $ActivityId"
        }

        $deploySetupPs1 = Resolve-Path -Path $deploySetupPs1
        & $deploySetupPs1 -deploymentSetupParameter $deploymentSetupParameter -logfile $logfile -errorfile $errorfile -axDRServerName $sqlServer -axDRDatabaseName $sqlDB -isPreDowntime $PreDowntime | Out-Null
        
        EventWrite-RunbookScriptTrace -Message "[$(Get-Date)]: Pre service check completed using deployment setup for activity id $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null
        return $true
    }
    
    EventWrite-RunbookScriptTrace -Message "Running precheck on Metadata dir: $metadatadir" -Component "servicing" -RunbookId $RunbookId| Out-Null

    # If platform version <= PU35 Run precheck with sync engine exe
    EventWrite-RunbookScriptTrace -Message "Running precheck with sync engine exe, activity id $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null

    $sqlUser = Get-DataAccessSqlUsr
    if($Credentials -and $Credentials[$($sqlUser)]) {
      $sqlPwd = $Credentials[$($sqlUser)]
    }
    else {
      $sqlPwd = Get-DataAccessSqlPwd
    }

    if($runPrecheckOnSecondaryDB -eq $false){
        $sqlServer = Get-DataAccessDbServer
        $sqlDB = Get-DataAccessDatabase
    }

    $connectionString = "Data Source=$($sqlServer); " +
        "Integrated Security=False; " +
        "User Id=$($sqlUser); " +
        "Password='$($sqlPwd)'; " +
        "Initial Catalog=$($sqlDB)"
    
    $command = "$metadatadir\bin\syncengine.exe"
    $commandParameter = "-syncmode=precheck"
    $commandParameter += " -metadatabinaries=$metadatadir"
    $commandParameter +=  " -connect=`"$connectionString`""
    $commandParameter +=  " -enableParallelSync"
                
    if($ActivityId)
    {
        $commandParameter += " -relatedactivityid=$ActivityId"
    }

    $process = Start-Process $command $commandParameter -PassThru -Wait -RedirectStandardOutput "$logfile" -RedirectStandardError "$errorfile"
    Write-Host "Log file location: $logfile"
        
    if ($process.ExitCode -ne 0) {
        $deploymentSetupError = Get-Content "$errorfile"
        throw $deploymentSetupError
    }

    EventWrite-RunbookScriptTrace -Message "[$(Get-Date)]: Pre service check completed using sync engine for activity id $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null
    return $true
}

$ErrorActionPreference = "stop"
Import-Module "$PSScriptRoot\AosEnvironmentUtilities.psm1" -Force -DisableNameChecking
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -ArgumentList $useServiceFabric -DisableNameChecking

if(!$LogDir)
{
    $LogDir = $PSScriptRoot
}

$deploySetupPs1 = "$PSScriptRoot\TriggerDeploymentSetupEngine.ps1"

# PreCheck
try
{
    # This returns false if pre check skipped.
    $result = RunDBSyncPreCheck -ActivityId $ActivityId -AxDRConnectionString $AxDRConnectionString -EnabledServicingFeatures $EnabledServicingFeatures
   
    if($result -eq $true)
    {   
        $message = "[$(Get-Date)]: Pre service check succeeded for activity id: $ActivityId."
    }
    else
    {
      $message = "[$(Get-Date)]: Pre service check did not run, activity id: $ActivityId."
    }

    Write-Host $message
    EventWrite-RunbookScriptTrace -Message $message -Component "servicing" -RunbookId $RunbookId| Out-Null
}
catch
{
    $errorMessage = "[$(Get-Date)]: Pre service check failed for activity id: $ActivityId, Exception: [$($_.Exception)]"    
    EventWrite-RunbookScriptTrace -Message $errorMessage -Component "servicing" -RunbookId $RunbookId | Out-Null

    if ($EnabledServicingFeatures -and $EnabledServicingFeatures -contains "FailDBSyncOnPrecheckFailure")
    { 
       EventWrite-RunbookScriptTrace -Message "FailDBSyncOnPrecheckFailure feature is enabled and db sync pre-check is failed, activity id: $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null

       # Only fail in dbsync pre-check if the error message contains one of the known error strings
       foreach ($knownError in $knownErrorHash.GetEnumerator())
       {
          if($errorMessage.Contains($knownError.Name))
          {
            # In case precheck failure due to a known customer-induced error scenario we do not require PITR during rollback. In precheck success scenario full sync will run and that will calculate PITR required or not.
            Write-IsPITRRequiredDuringRollback -PitrRequired 'false'

            EventWrite-RunbookScriptTrace -Message "DBSync pre-check failed due to a known customer induced error, error code: $($knownError.Value), activity id: $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null
            throw $errorMessage
          }
       }
    }
}
# SIG # Begin signature block
# MIIr/QYJKoZIhvcNAQcCoIIr7jCCK+oCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDf8pNtqZY7VDtd
# JiquVayV4+X8rN+4YqEWHTNh3X0wYqCCEW4wggh+MIIHZqADAgECAhM2AAACAO38
# jbec3qFIAAIAAAIAMA0GCSqGSIb3DQEBCwUAMEExEzARBgoJkiaJk/IsZAEZFgNH
# QkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxFTATBgNVBAMTDEFNRSBDUyBDQSAwMTAe
# Fw0yNDExMDgxMjQzMjhaFw0yNTExMDgxMjQzMjhaMCQxIjAgBgNVBAMTGU1pY3Jv
# c29mdCBBenVyZSBDb2RlIFNpZ24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK
# AoIBAQC5L/UPrOpwYjxcoZC0TqqvMF1WUELvwXN+k27SrA5rohJknn7Cgbxg4hGT
# XKqpcdbtsVTN3ZY896SJ20uQ+INL5OVLzpW408nCNTPYg2LtGJbqHUjpNm0hLCJ+
# gO5Jn2T8DDzIJoUijGXj1m+hRLKb2nOIicCED2GuYBmuWXnaY7INmVEaU3peryty
# ZjDuxdyGDuiPURz8lW1SUiDzoszNp1oswVr+WjDvLDUx4HlxPsG8zUjIst0NnJ6o
# z4tNFKaUBDCetcMjQxpCETn29a1CuRddxZLjPHZHfcotr5sh1S6bNQdzVaMNsxV8
# L3wjHb7XJ6ZVm662mHEiPgpyNcLhAgMBAAGjggWKMIIFhjApBgkrBgEEAYI3FQoE
# HDAaMAwGCisGAQQBgjdbAQEwCgYIKwYBBQUHAwMwPQYJKwYBBAGCNxUHBDAwLgYm
# KwYBBAGCNxUIhpDjDYTVtHiE8Ys+hZvdFs6dEoFgg93NZoaUjDICAWQCAQ4wggJ2
# BggrBgEFBQcBAQSCAmgwggJkMGIGCCsGAQUFBzAChlZodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpaW5mcmEvQ2VydHMvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDEu
# YW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUy
# MDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDIuYW1lLmdibC9haWEv
# QlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBS
# BggrBgEFBQcwAoZGaHR0cDovL2NybDMuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAx
# LkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZG
# aHR0cDovL2NybDQuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDCBrQYIKwYBBQUHMAKGgaBsZGFwOi8vL0NO
# PUFNRSUyMENTJTIwQ0ElMjAwMSxDTj1BSUEsQ049UHVibGljJTIwS2V5JTIwU2Vy
# dmljZXMsQ049U2VydmljZXMsQ049Q29uZmlndXJhdGlvbixEQz1BTUUsREM9R0JM
# P2NBQ2VydGlmaWNhdGU/YmFzZT9vYmplY3RDbGFzcz1jZXJ0aWZpY2F0aW9uQXV0
# aG9yaXR5MB0GA1UdDgQWBBST/HE52ZUlmsYqZcZBdrXZ5u4ZnzAOBgNVHQ8BAf8E
# BAMCB4AwRQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEWMBQGA1UEBRMNMjM2MTY3KzUwMzE1NTCCAeYGA1UdHwSCAd0wggHZMIIB
# 1aCCAdGgggHNhj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpaW5mcmEvQ1JM
# L0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwxLmFtZS5nYmwv
# Y3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwyLmFtZS5n
# YmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwzLmFt
# ZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmw0
# LmFtZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGgb1sZGFwOi8v
# L0NOPUFNRSUyMENTJTIwQ0ElMjAwMSgyKSxDTj1CWTJQS0lDU0NBMDEsQ049Q0RQ
# LENOPVB1YmxpYyUyMEtleSUyMFNlcnZpY2VzLENOPVNlcnZpY2VzLENOPUNvbmZp
# Z3VyYXRpb24sREM9QU1FLERDPUdCTD9jZXJ0aWZpY2F0ZVJldm9jYXRpb25MaXN0
# P2Jhc2U/b2JqZWN0Q2xhc3M9Y1JMRGlzdHJpYnV0aW9uUG9pbnQwHwYDVR0jBBgw
# FoAUllGE4Gtve/7YBqvD8oXmKa5q+dQwHwYDVR0lBBgwFgYKKwYBBAGCN1sBAQYI
# KwYBBQUHAwMwDQYJKoZIhvcNAQELBQADggEBAEDd8Wf5RkHsB64vgn2slxDtHzSo
# It9xN/Dm3RdFjNZ0diTUPMgSPYQlSk8nIAfudnB9FLavGlvZLlyUpfrPSuikepj3
# i3pqNEFn6fNdNFv/wHMxv7hQTIDCmuoR1v1rX+w3oeleBPMnN3QmH4ff1NsynyV4
# dZdYgN9Cw9sC/S3pWZpJrbOs7YOM3vqyU6DciHhC4D9i2zByHCF2pu9nYfiQf5A2
# iUZenRvyo1E5rC+UP2VZXa4k7g66W20+zAajIKKIqEmRtWahekMkCcOIHFBY4RDA
# ybgPRSGur4VDAiZPjTXS90wQXrX9CwU20cfiCC6e76F4H95KtQjKYpzuNVAwggjo
# MIIG0KADAgECAhMfAAAAUeqP9pxzDKg7AAAAAABRMA0GCSqGSIb3DQEBCwUAMDwx
# EzARBgoJkiaJk/IsZAEZFgNHQkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxEDAOBgNV
# BAMTB2FtZXJvb3QwHhcNMjEwNTIxMTg0NDE0WhcNMjYwNTIxMTg1NDE0WjBBMRMw
# EQYKCZImiZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQD
# EwxBTUUgQ1MgQ0EgMDEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDJ
# mlIJfQGejVbXKpcyFPoFSUllalrinfEV6JMc7i+bZDoL9rNHnHDGfJgeuRIYO1LY
# /1f4oMTrhXbSaYRCS5vGc8145WcTZG908bGDCWr4GFLc411WxA+Pv2rteAcz0eHM
# H36qTQ8L0o3XOb2n+x7KJFLokXV1s6pF/WlSXsUBXGaCIIWBXyEchv+sM9eKDsUO
# LdLTITHYJQNWkiryMSEbxqdQUTVZjEz6eLRLkofDAo8pXirIYOgM770CYOiZrcKH
# K7lYOVblx22pdNawY8Te6a2dfoCaWV1QUuazg5VHiC4p/6fksgEILptOKhx9c+ia
# piNhMrHsAYx9pUtppeaFAgMBAAGjggTcMIIE2DASBgkrBgEEAYI3FQEEBQIDAgAC
# MCMGCSsGAQQBgjcVAgQWBBQSaCRCIUfL1Gu+Mc8gpMALI38/RzAdBgNVHQ4EFgQU
# llGE4Gtve/7YBqvD8oXmKa5q+dQwggEEBgNVHSUEgfwwgfkGBysGAQUCAwUGCCsG
# AQUFBwMBBggrBgEFBQcDAgYKKwYBBAGCNxQCAQYJKwYBBAGCNxUGBgorBgEEAYI3
# CgMMBgkrBgEEAYI3FQYGCCsGAQUFBwMJBggrBgEFBQgCAgYKKwYBBAGCN0ABAQYL
# KwYBBAGCNwoDBAEGCisGAQQBgjcKAwQGCSsGAQQBgjcVBQYKKwYBBAGCNxQCAgYK
# KwYBBAGCNxQCAwYIKwYBBQUHAwMGCisGAQQBgjdbAQEGCisGAQQBgjdbAgEGCisG
# AQQBgjdbAwEGCisGAQQBgjdbBQEGCisGAQQBgjdbBAEGCisGAQQBgjdbBAIwGQYJ
# KwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMBIGA1UdEwEB/wQI
# MAYBAf8CAQAwHwYDVR0jBBgwFoAUKV5RXmSuNLnrrJwNp4x1AdEJCygwggFoBgNV
# HR8EggFfMIIBWzCCAVegggFToIIBT4YxaHR0cDovL2NybC5taWNyb3NvZnQuY29t
# L3BraWluZnJhL2NybC9hbWVyb290LmNybIYjaHR0cDovL2NybDIuYW1lLmdibC9j
# cmwvYW1lcm9vdC5jcmyGI2h0dHA6Ly9jcmwzLmFtZS5nYmwvY3JsL2FtZXJvb3Qu
# Y3JshiNodHRwOi8vY3JsMS5hbWUuZ2JsL2NybC9hbWVyb290LmNybIaBqmxkYXA6
# Ly8vQ049YW1lcm9vdCxDTj1BTUVSb290LENOPUNEUCxDTj1QdWJsaWMlMjBLZXkl
# MjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9uLERDPUFNRSxE
# Qz1HQkw/Y2VydGlmaWNhdGVSZXZvY2F0aW9uTGlzdD9iYXNlP29iamVjdENsYXNz
# PWNSTERpc3RyaWJ1dGlvblBvaW50MIIBqwYIKwYBBQUHAQEEggGdMIIBmTBHBggr
# BgEFBQcwAoY7aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraWluZnJhL2NlcnRz
# L0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6Ly9jcmwyLmFt
# ZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6
# Ly9jcmwzLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUH
# MAKGK2h0dHA6Ly9jcmwxLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQw
# gaIGCCsGAQUFBzAChoGVbGRhcDovLy9DTj1hbWVyb290LENOPUFJQSxDTj1QdWJs
# aWMlMjBLZXklMjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9u
# LERDPUFNRSxEQz1HQkw/Y0FDZXJ0aWZpY2F0ZT9iYXNlP29iamVjdENsYXNzPWNl
# cnRpZmljYXRpb25BdXRob3JpdHkwDQYJKoZIhvcNAQELBQADggIBAFAQI7dPD+jf
# XtGt3vJp2pyzA/HUu8hjKaRpM3opya5G3ocprRd7vdTHb8BDfRN+AD0YEmeDB5HK
# QoG6xHPI5TXuIi5sm/LeADbV3C2q0HQOygS/VT+m1W7a/752hMIn+L4ZuyxVeSBp
# fwf7oQ4YSZPh6+ngZvBHgfBaVz4O9/wcfw91QDZnTgK9zAh9yRKKls2bziPEnxeO
# ZMVNaxyV0v152PY2xjqIafIkUjK6vY9LtVFjJXenVUAmn3WCPWNFC1YTIIHw/mD2
# cTfPy7QA1pT+GPARAKt0bKtq9aCd/Ym0b5tPbpgCiRtzyb7fbNS1dE740re0COE6
# 7YV2wbeo2sXixzvLftH8L7s9xv9wV+G22qyKt6lmKLjFK1yMw4Ni5fMabcgmzRvS
# jAcbqgp3tk4a8emaaH0rz8MuuIP+yrxtREPXSqL/C5bzMzsikuDW9xH10graZzSm
# PjilzpRfRdu20/9UQmC7eVPZ4j1WNa1oqPHfzET3ChIzJ6Q9G3NPCB+7KwX0OQmK
# yv7IDimj8U/GlsHD1z+EF/fYMf8YXG15LamaOAohsw/ywO6SYSreVW+5Y0mzJutn
# BC9Cm9ozj1+/4kqksrlhZgR/CSxhFH3BTweH8gP2FEISRtShDZbuYymynY1un+Ry
# fiK9+iVTLdD1h/SxyxDpZMtimb4CgJQlMYIZ5TCCGeECAQEwWDBBMRMwEQYKCZIm
# iZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQDEwxBTUUg
# Q1MgQ0EgMDECEzYAAAIA7fyNt5zeoUgAAgAAAgAwDQYJYIZIAWUDBAIBBQCgga4w
# GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisG
# AQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEICz1spHJIHssz2VFUST5UpvuJLQBouWF
# hVjyMlXXq/LeMEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8AcwBvAGYA
# dKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEA
# kXYKvj3HGbvPqItBro4xzMKthJRKhu3kw+GKDPvfEt9CDYHhNLeudhsL0F4q2qz5
# D/Cv5EtFHsil03r6szvku3zMQKEAZIyfonc/C4sjvyKNvNmetktArrY4mWExpGNq
# NO4J06R0kZCR+0fj6klHpRehKy+AdtQqKQ/J8luMrdEGBQeP4q8Q2lpzV/38MrvU
# VFAFojn1B3J2eZN1eYS32zYxHjiQ480H3NsMKYfjkjzQc0vlcIu6Yq+343vZaF3p
# kzfsRwGwT8m96LUnTI3sk/TcRJKciUstt74cIy9a5OdezNZJaOSc2gvlonq2/c0+
# e5ldeCUR6FZwOie28rbVPqGCF60wghepBgorBgEEAYI3AwMBMYIXmTCCF5UGCSqG
# SIb3DQEHAqCCF4YwgheCAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFaBgsqhkiG9w0B
# CRABBKCCAUkEggFFMIIBQQIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFlAwQCAQUA
# BCBZtjshRebLAzZTQHMUqdEhf8YiFuYBWgIFHd08ItKW+gIGZ5E0wRV2GBMyMDI1
# MDEyMjIwNDEzNy4yNzhaMASAAgH0oIHZpIHWMIHTMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJlbGFuZCBP
# cGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjo1NzFB
# LTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZaCCEfswggcoMIIFEKADAgECAhMzAAAB+8vLbDdn5TCVAAEAAAH7MA0GCSqGSIb3
# DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
# VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAk
# BgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMB4XDTI0MDcyNTE4
# MzExM1oXDTI1MTAyMjE4MzExM1owgdMxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
# YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
# Q29ycG9yYXRpb24xLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlv
# bnMgTGltaXRlZDEnMCUGA1UECxMeblNoaWVsZCBUU1MgRVNOOjU3MUEtMDVFMC1E
# OTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNlMIICIjAN
# BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAqMJWQeWAq4LwvSjYsjP0Uvhvm0j0
# aAOJiMLg0sLfxKoTXAdKD6oMuq5rF5oEiOxV+9ox0H95Q8fhoZq3x9lxguZyTOK4
# l2xtcgtJCtjXRllM2bTpjOg35RUrBy0cAloBU9GJBs7LBNrcbH6rBiOvqDQNicPR
# Zwq16xyjMidU1J1AJuat9yLn7taifoD58blYEcBvkj5dH1la9zU846QDeOoRO6Nc
# qHLsDx8/zVKZxP30mW6Y7RMsqtB8cGCgGwVVurOnaNLXs31qTRTyVHX8ppOdoSih
# CXeqebgJCRzG8zG/e/k0oaBjFFGl+8uFELwCyh4wK9Z5+azTzfa2GD4p6ihtskXs
# 3lnW05UKfDJhAADt6viOc0Rk/c8zOiqzh0lKpf/eWUY2o/hvcDPZNgLaHvyfDqb8
# AWaKvO36iRZSXqhSw8SxJo0TCpsbCjmtx0LpHnqbb1UF7cq09kCcfWTDPcN12pbY
# Lqck0bIIfPKbc7HnrkNQks/mSbVZTnDyT3O8zF9q4DCfWesSr1akycDduGxCdKBv
# gtJh1YxDq1skTweYx5iAWXnB7KMyls3WQZbTubTCLLt8Xn8t+slcKm5DkvobubmH
# SriuTA3wTyIy4FxamTKm0VDu9mWds8MtjUSJVwNVVlBXaQ3ZMcVjijyVoUNVuBY9
# McwYcIQK62wQ20ECAwEAAaOCAUkwggFFMB0GA1UdDgQWBBRHVSGYUNQ3RwOl71zI
# AuUjIKg1KjAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJlpxtTNRnpcjBfBgNVHR8E
# WDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9N
# aWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcmwwbAYIKwYB
# BQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20v
# cGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0ElMjAyMDEw
# KDEpLmNydDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMA4G
# A1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQsFAAOCAgEAwzoIKOY2dnUjfWuMiGoz
# /ovoc1e86VwWaZNFdgRmOoQuRe4nLdtZONtTHNk3Sj3nkyBszzxSbZEQ0DduyKHH
# I5P8V87jFttGnlR0wPP22FAebbvAbutkMMVQMFzhVBWiWD0VAnu9x0fjifLKDAVX
# Lwoun5rCFqwbasXFc7H/0DPiC+DBn3tUxefvcxUCys4+DC3s8CYp7WWXpZ8Wb/vd
# BhDliHmB7pWcmsB83uc4/P2GmAI3HMkOEu7fCaSYoQhouWOr07l/KM4TndylIirm
# 8f2WwXQcFEzmUvISM6ludUwGlVNfTTJUq2bTDEd3tlDKtV9AUY3rrnFwHTwJryLt
# T4IFhvgBfND3mL1eeSakKf7xTII4Jyt15SXhHd5oI/XGjSgykgJrWA57rGnAC7ru
# 3/ZbFNCMK/Jj6X8X4L6mBOYa2NGKwH4A37YGDrecJ/qXXWUYvfLYqHGf8ThYl12Y
# g1rwSKpWLolA/B1eqBw4TRcvVY0IvNNi5sm+//HJ9Aw6NJuR/uDR7X7vDXicpXMl
# RNgFMyADb8AFIvQPdHqcRpRorY+YUGlvzeJx/2gNYyezAokbrFhACsJ2BfyeLyCE
# o6AuwEHn511PKE8dK4JvlmLSoHj7VFR3NHDk3zRkx0ExkmF8aOdpvoKhuwBCxoZ/
# JhbzSzrvZ74GVjKKIyt5FA0wggdxMIIFWaADAgECAhMzAAAAFcXna54Cm0mZAAAA
# AAAVMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2Fz
# aGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
# cnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0ZSBB
# dXRob3JpdHkgMjAxMDAeFw0yMTA5MzAxODIyMjVaFw0zMDA5MzAxODMyMjVaMHwx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1p
# Y3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIICIjANBgkqhkiG9w0BAQEFAAOC
# Ag8AMIICCgKCAgEA5OGmTOe0ciELeaLL1yR5vQ7VgtP97pwHB9KpbE51yMo1V/YB
# f2xK4OK9uT4XYDP/XE/HZveVU3Fa4n5KWv64NmeFRiMMtY0Tz3cywBAY6GB9alKD
# RLemjkZrBxTzxXb1hlDcwUTIcVxRMTegCjhuje3XD9gmU3w5YQJ6xKr9cmmvHaus
# 9ja+NSZk2pg7uhp7M62AW36MEBydUv626GIl3GoPz130/o5Tz9bshVZN7928jaTj
# kY+yOSxRnOlwaQ3KNi1wjjHINSi947SHJMPgyY9+tVSP3PoFVZhtaDuaRr3tpK56
# KTesy+uDRedGbsoy1cCGMFxPLOJiss254o2I5JasAUq7vnGpF1tnYN74kpEeHT39
# IM9zfUGaRnXNxF803RKJ1v2lIH1+/NmeRd+2ci/bfV+AutuqfjbsNkz2K26oElHo
# vwUDo9Fzpk03dJQcNIIP8BDyt0cY7afomXw/TNuvXsLz1dhzPUNOwTM5TI4CvEJo
# LhDqhFFG4tG9ahhaYQFzymeiXtcodgLiMxhy16cg8ML6EgrXY28MyTZki1ugpoMh
# XV8wdJGUlNi5UPkLiWHzNgY1GIRH29wb0f2y1BzFa/ZcUlFdEtsluq9QBXpsxREd
# cu+N+VLEhReTwDwV2xo3xwgVGD94q0W29R6HXtqPnhZyacaue7e3PmriLq0CAwEA
# AaOCAd0wggHZMBIGCSsGAQQBgjcVAQQFAgMBAAEwIwYJKwYBBAGCNxUCBBYEFCqn
# Uv5kxJq+gpE8RjUpzxD/LwTuMB0GA1UdDgQWBBSfpxVdAF5iXYP05dJlpxtTNRnp
# cjBcBgNVHSAEVTBTMFEGDCsGAQQBgjdMg30BATBBMD8GCCsGAQUFBwIBFjNodHRw
# Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9yeS5odG0w
# EwYDVR0lBAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
# CwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYDVR0jBBgwFoAU1fZWy4/o
# olxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDovL2NybC5taWNy
# b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYt
# MjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+aHR0cDovL3d3dy5t
# aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5j
# cnQwDQYJKoZIhvcNAQELBQADggIBAJ1VffwqreEsH2cBMSRb4Z5yS/ypb+pcFLY+
# TkdkeLEGk5c9MTO1OdfCcTY/2mRsfNB1OW27DzHkwo/7bNGhlBgi7ulmZzpTTd2Y
# urYeeNg2LpypglYAA7AFvonoaeC6Ce5732pvvinLbtg/SHUB2RjebYIM9W0jVOR4
# U3UkV7ndn/OOPcbzaN9l9qRWqveVtihVJ9AkvUCgvxm2EhIRXT0n4ECWOKz3+SmJ
# w7wXsFSFQrP8DJ6LGYnn8AtqgcKBGUIZUnWKNsIdw2FzLixre24/LAl4FOmRsqlb
# 30mjdAy87JGA0j3mSj5mO0+7hvoyGtmW9I/2kQH2zsZ0/fZMcm8Qq3UwxTSwethQ
# /gpY3UA8x1RtnWN0SCyxTkctwRQEcb9k+SS+c23Kjgm9swFXSVRk2XPXfx5bRAGO
# WhmRaw2fpCjcZxkoJLo4S5pu+yFUa2pFEUep8beuyOiJXk+d0tBMdrVXVAmxaQFE
# fnyhYWxz/gq77EFmPWn9y8FBSX5+k77L+DvktxW/tM4+pTFRhLy/AsGConsXHRWJ
# jXD+57XQKBqJC4822rpM+Zv/Cuk0+CQ1ZyvgDbjmjJnW4SLq8CdCPSWU5nR0W2rR
# nj7tfqAxM328y+l7vzhwRNGQ8cirOoo6CGJ/2XBjU02N7oJtpQUQwXEGahC0HVUz
# WLOhcGbyoYIDVjCCAj4CAQEwggEBoYHZpIHWMIHTMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJlbGFuZCBP
# cGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjo1NzFB
# LTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZaIjCgEBMAcGBSsOAwIaAxUABHHn7NCGusZz2RfVbyuwYwPykBWggYMwgYCkfjB8
# MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
# bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1N
# aWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQsFAAIFAOs7
# s0AwIhgPMjAyNTAxMjIxODExMTJaGA8yMDI1MDEyMzE4MTExMlowdDA6BgorBgEE
# AYRZCgQBMSwwKjAKAgUA6zuzQAIBADAHAgEAAgIAkDAHAgEAAgIRwzAKAgUA6z0E
# wAIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIBAAIDB6Eg
# oQowCAIBAAIDAYagMA0GCSqGSIb3DQEBCwUAA4IBAQDk0LtTxfA93eVgOqUQNBTj
# VbTwSRvXD2RFqLw3diveqBcB0O4LcpX1aBorKGgKoTU94/5O/MwgUAjZFEzhXdM+
# SZKih+YLYNMOsHIpAy0RFezwSBwkRjVreGrXXb8v82IMRTLLoe0hlmKqLCeHakyc
# tbIFnEcHU1CEt+jUXJ0WEFXICNH8Dutuad++zjciEepCNGlGbN/5Y4LBj7W4Y1TB
# bpNIq0oiXE/aZpFWW2FoIHeWmKzNWW7C+p5KUJ+Gz24ZnwRpuPEaMSCGZqfxBr2E
# jyqjnvIKZPWy9jCZdtqEFeD95mS1zfLSvCaM8Refpz1yu9ND5b39pLIeBfvbBDZ1
# MYIEDTCCBAkCAQEwgZMwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
# b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTACEzMA
# AAH7y8tsN2flMJUAAQAAAfswDQYJYIZIAWUDBAIBBQCgggFKMBoGCSqGSIb3DQEJ
# AzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0BCQQxIgQgrVGQIgfnxqUCympaWNR4
# Xej6k3/4WTQnh/SH21O7XiMwgfoGCyqGSIb3DQEJEAIvMYHqMIHnMIHkMIG9BCA5
# 2wKr/KCFlVNYiWsCLsB4qhjEYEP3xHqYqDu1SSTlGDCBmDCBgKR+MHwxCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4w
# HAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29m
# dCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB+8vLbDdn5TCVAAEAAAH7MCIEIKfE
# TpE9zR0/CqV4HovZTYnPTxfROIIul9d2kvnvMGPvMA0GCSqGSIb3DQEBCwUABIIC
# AILerF5thf7sz5mkaoNEWXAqOd8xVWcdXKVB0BeLkZBN6ggb1ioea4Zgwxboye/r
# XzmQLhaD8pIHRsPXgJz/UrAPMzv4Z9lqAqnjKXIspkf0rsXhTNnZp5TlgUYP9S/e
# jT0Z1gesEikA+Bb5StSwGhuriYAU5GlHdgZ/9j/03X+kMutSWB4lK4D1NMh0yuEY
# ve6bbQhpzHJab91CMvakNij0ZQz5ryesbGFZegFZW9ZbKpTeUDANGWRwHG7CxxCd
# YB83YJK1xXdberJDuPFzd2Qi8WAswoCp3IwDvJyhksxP1q4mhYH4pnQxLYbaX6YC
# 9seDNIWjjf+Z5bBhsO5EI1eshzNDL/5f1lPtxxNQAv00vvi4pa/P8q+T7bykSQcn
# D9s5qfP/MhE4eZp8yn8JOik6OsnhEO/RKOb+G1kjY4lvdPmO1pdww1ShlVIyqvbz
# pjxcuD4Zhb5FTy4zRqXIbkMquKYtaJeXnecj8jPwIhIbEFNYFPaBSG/tfh2qEz3P
# +jPWcwbUneDRWqsZCM/T89nchNhOclVSKPmmwuTDE8IeB84GUV3EQS/iEmvrxgVw
# vuFNIQm+ncoKp16CEjBOnxN0EjtEbylwTA1ylBadaaKfwpRHhEpG7BctJEPuQhw5
# /T00KXChg7reWHyp9LaNPfEFboAb1Kr6lXwDnYsGoOZU
# SIG # End signature block
