﻿################################################################################
# Write message to log file and output stream
################################################################################
Function Write-LogOutput($message) {
    Write-Log $message
    Write-Output $message
}

################################################################################
# Install AX SSRS extension components with HTTP endpoints
################################################################################
Function Install-AxSsrsExtension {
    param
    (
        [Parameter(Mandatory = $true)]
        [string]$config,
        [Parameter(Mandatory = $true)]
        [string]$log
    )

    Import-Module "$PSScriptRoot\AosCommon.psm1" -DisableNameChecking

    # Initialize log file
    Initialize-Log $log
    $settings = Decode-Settings($config)
    Write-LogOutput $("")
    Write-LogOutput $settings | Format-List *

    # Get EnableSecurity value
    if ($null -eq $settings.EnableSecurity) {
        throw "Missing EnableSecurity value."
    }

    [Switch]$enableSecurity = [System.Convert]::ToBoolean($settings.EnableSecurity)

    [Switch]$isReportingClusterDeployment = $false
    [System.Boolean]::TryParse($settings.IsReportingClusterDeployment, [ref]$isReportingClusterDeployment)    

    # Check if use https
    if ((![System.String]::IsNullOrWhiteSpace($settings.SsrsSslCertificateThumbprint)) -and (![System.String]::IsNullOrWhiteSpace($settings.SsrsServerFqdn))) {
        Write-LogOutput "Install AX SSRS extension with HTTPS endpoints."
        Install-AxSsrsExtensionForHttps -config $config -log $log
        return
    }

    # Gets account list to acess reports
    [string[]] $accounts = $null
    if ($null -ne $settings.AccountListToAccessReports) {
        $accounts = $settings.AccountListToAccessReports.split(',')
    }

    # Use AxReportVmRoleStartupTask.exe to install reporting extensions
    Write-LogOutput $("Start installing AX SSRS extensions ...")

    [string] $reportServerUri = "http://localhost:80/ReportServer"
    [string] $ManagementAssemblyPath = Join-Path -Path $PSScriptRoot -ChildPath "Microsoft.Dynamics.AX.Framework.Management.dll"
    [string] $SystemReportTimeoutConfigName = "SystemReportTimeout"
    [string] $SessionTimeoutConfigName = "SessionTimeout"
    Import-Module $ManagementAssemblyPath
    
    # Need to install SSRS extension first with windows auth to install data sources for reporting cluster mode
    if ($isReportingClusterDeployment) {
        $settings.IsReportingClusterDeployment = "False"
        $settings.EnableSecurity = "True"
        $settingsJson = ConvertTo-Json $settings
        $windowsAuthConfig = [System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($settingsJson))

        Setup-SSRSExtension -config $windowsAuthConfig -log $log -ErrorAction Stop

        # Set SSRS system report timeout to 1 hour
        Write-LogOutput $("Set the SSRS system report timeout to 60 minutes...")
        Set-SsrsConfigurationInfoTimeout -timeoutInSeconds 3600 -reportServerUri $reportServerUri -configName $SystemReportTimeoutConfigName

        # Set SSRS session timeout to 1 hour
        Write-LogOutput $("Set the SSRS session timeout to 60 minutes...")
        Set-SsrsConfigurationInfoTimeout -timeoutInSeconds 3600 -reportServerUri $reportServerUri -configName $SessionTimeoutConfigName

        # Set execution log days kept to 1 day
        # The records of execution log would be uploaded to Kusto every 5 minutes
        $dayskeptValue = 1
        Write-LogOutput $("Set SSRS execution log days kept value to $dayskeptValue days...")
        Set-SsrsExecutionLogDaysKept -daysKept $dayskeptValue -reportServerUri $reportServerUri

        $settings.IsReportingClusterDeployment = "True"
        $settings.EnableSecurity = $enableSecurity.ToString();

        # Install shared data sources
        Install-DataSource -reportServerUri $reportServerUri -log $log

        # Create the root report folder /Dynamics
        Install-AxReportRootFolder -ReportServerAddress "localhost" -SkipReportServerAdminCheck  -ErrorAction Stop | Tee-Object -Variable InstallLog
        Add-Content -Path $log -Value $InstallLog
    }

    # Deploy SSRS extension
    Setup-SSRSExtension -config $config -log $log -ErrorAction Stop
    Write-LogOutput $("Completed installing AX SSRS extension.")
		
    #Added this step to warmup SSRS to avoid the timeout error
    $serviceName = Get-SsrsServiceName
    Write-LogOutput ("The reporting service name is: $serviceName.")
    Restart-WindowsService -log $log -computerName $env:COMPUTERNAME -serviceName $serviceName -ErrorAction stop
    Test-ReportServer -reportServerUri $reportServerUri -log $log -ErrorAction Stop

    # Install shared data sources
    if (-not $isReportingClusterDeployment) {
        Install-DataSource -reportServerUri $reportServerUri -log $log
    }

    if ($enableSecurity) {
        # Start Creating and configuring AX reports root folder
        Write-LogOutput $("")
        Write-LogOutput $("Start creating and configuring AX reports root folder ...")			
        Install-AxReportRootFolder -Accounts $accounts  -ErrorAction Stop | Tee-Object -Variable InstallLog          
        Add-Content -Path $log -Value $InstallLog
        Write-LogOutput $("Completed creating and configuring AX reports root folder.")
        Write-LogOutput $("")
    }

    # Test the SSRS instance prior to completing this step. If this is not successful then we should fail.
    Restart-WindowsService -log $log -computerName $env:COMPUTERNAME -serviceName $serviceName -ErrorAction stop
    Test-ReportServer -reportServerUri $reportServerUri -log $log -ErrorAction Stop
    Write-LogOutput $("")
}

################################################################################
# Install AX SSRS extension components with HTTP endpoints
################################################################################
Function Install-AxSsrsExtensionForHttps {
    param
    (
        [Parameter(Mandatory = $true)]
        [string]$config,
        [Parameter(Mandatory = $true)]
        [string]$log
    )

    Import-Module "$PSScriptRoot\AosCommon.psm1" -DisableNameChecking

    # Initialize log file
    Initialize-Log $log
    $settings = Decode-Settings($config)
    Write-LogOutput $("")
    Write-LogOutput $settings | Format-List *    

    # Enable security
    [Switch]$enableSecurity = [System.Convert]::ToBoolean($settings.EnableSecurity)

    # IsServiceFabricDeployment (Set to true by LBD deployment)
    [Switch]$isServiceFabricDeployment = $false
    [System.Boolean]::TryParse($settings.IsServiceFabricDeployment, [ref]$isServiceFabricDeployment)

    # Gets account list to acess reports
    [string[]] $accounts = $null
    if ($null -ne $settings.AccountListToAccessReports) {
        $accounts = $settings.AccountListToAccessReports.split(',')
    }

    # Make sure HTTPS required parameters are set properly
    $sslCertThumbprint = $settings.SsrsSslCertificateThumbprint
    $serverFqdn = $settings.SsrsServerFqdn
    $httpsPort = $settings.SsrsHttpsPort
    $useHttps = (![System.String]::IsNullOrWhiteSpace($sslCertThumbprint)) -and (![System.String]::IsNullOrWhiteSpace($serverFqdn))
    if (-not $useHttps) {
        throw "Install-AxSsrsExtensionForHttps should not be called because SSLCertThumbprint: $sslCertThumbprint, Server FQDN: $serverFqdn."
    }

    $httpsPortValue = 443
    if ($httpsPort) {
        if (-not [Int32]::TryParse($settings.SsrsHttpsPort, [ref]$httpsPortValue)) {
            throw "Invalid vlaue for SsrsHttpsPort. The value is: $httpsPort"
        }
    }

    # Import management module
    [string] $ManagementAssemblyPath = Join-Path -Path $PSScriptRoot -ChildPath "Microsoft.Dynamics.AX.Framework.Management.dll"
    Import-Module $ManagementAssemblyPath
    $lcid = [System.Globalization.CultureInfo]::GetCultureInfo('en-US').LCID

    try {
        # Use AxReportVmRoleStartupTask.exe to install reporting extensions. This will create an http endpoint for data source and ax report folder configuration
        Write-LogOutput $("Start installing AX SSRS extensions ...")
        Setup-SSRSExtension -config $config -log $log -ErrorAction Stop
        Write-LogOutput $("Completed installing AX SSRS extension.")
    
        # Install shared data sources and their parent folder if they do not exist
        Install-DataSource -reportServerUri "http://localhost/reportserver" -log $log
            
        # Start creating and configuring AX reports root folder security
        if ($enableSecurity) {
            Write-LogOutput $("")
            Write-LogOutput $("Start creating and configuring AX reports root folder ...")			
            Install-AxReportRootFolder -Accounts $accounts -DeleteExisting:$false -GrantPublishReportPermission:$isServiceFabricDeployment -ErrorAction Stop | Tee-Object -Variable InstallLog
            Add-Content -Path $log -Value $InstallLog
            Write-LogOutput $("Completed creating and configuring AX reports root folder.")
            Write-LogOutput $("")
        }
    }
    finally {
        # Make sure that http endpoint is deleted.
        $rsConfig = Get-RsConfig -ComputerName $env:COMPUTERNAME
        Remove-HttpUrl -rsConfig $rsConfig -lcid $lcid
    }

    # Ensure SSRS endpoints to Use HTTPS
    Write-LogOutput $("Start configuring SSRS HTTPS endpoint with port: $httpsPort, cert: $sslCertThumbprint, FQDN: $serverFqdn")

    $httpsReportServerUrl = "https://" + $serverFqdn + ":" + $httpsPortValue.ToString() + "/ReportServer"
    Write-LogOutput $("HTTPS Report Server URL is: $httpsReportServerUrl")

    Confirm-SsrsHttpsEndpoints -Port $httpsPortValue -Log $log -SslCertThumbprint $sslCertThumbprint -ErrorAction Stop
    Write-LogOutput $("SSRS HTTPS endpoints were configured.")
    Write-LogOutput $("")
        
    # Added this step to warmup SSRS to avoid the timeout error
    $serviceName = Get-SsrsServiceName
    Write-LogOutput ("The reporting service name is: $serviceName.")
    Restart-WindowsService -log $log -computerName $env:COMPUTERNAME -serviceName $serviceName -ErrorAction stop
    $localHttpsReportServerUrl = $("https://localhost:" + $httpsPortValue.ToString() + "/ReportServer")
    Test-ReportServer -reportServerUri $localHttpsReportServerUrl -forceTls12 $true -ignoreServerCertificate $true -log $log -ErrorAction Stop
}


################################################################################
# Install Data source (for reporting that does not need deployment)
################################################################################
Function Install-DataSource {
    param
    (
        [Parameter(Mandatory = $true)]
        [string]$reportServerUri,
        [Parameter(Mandatory = $true)]
        [string]$log
    )

    Test-ReportServer -reportServerUri $reportServerUri -log $log -ErrorAction Stop
    Write-LogOutput $("")
    Write-LogOutput $("Start creating report shared data sources...")

    Install-SharedDataSource -ErrorAction Stop | Tee-Object -Variable InstallLog
    Add-Content -Path $log -Value $InstallLog
    Write-LogOutput $("Completed creating report shared data sources.")
}

################################################################################
# Setup SSRS Extension by calling exe
################################################################################
Function Setup-SSRSExtension {
    param
    (
        [Parameter(Mandatory = $true)]
        [string]$config,
        [Parameter(Mandatory = $true)]
        [string]$log
    )

    $deployExtensionExe = Join-Path -Path $PSScriptRoot -ChildPath "AxReportVmRoleStartupTask.exe"
    Write-LogOutput $deployExtensionExe
    $allArgs = @("-config", $config)
    & $deployExtensionExe $allArgs *>&1
    $exitCode = $LASTEXITCODE
    if ($exitCode -ne 0) {
        throw "Error occurred when running AxReportVmRoleStartupTask.exe to install extension. Please see details in events for AX-ReportPVMSetup."
    }
}


################################################################################
# Uninstall AX SSRS extension components
# $config parameter must have a property EnableSecurity with value "true" or "false"
################################################################################
Function Remove-AxSsrsExtension {
    param
    (
        [Parameter(Mandatory = $true)]
        [string]$config,
        [Parameter(Mandatory = $true)]
        [string]$log
    )

    Import-Module "$PSScriptRoot\AosCommon.psm1" -DisableNameChecking

    # Initialize log file
    Initialize-Log $log

    # Verify deployment type
    $settings = Decode-Settings($config)
    Write-LogOutput $("")
    Write-LogOutput $settings | Format-List *
    if ($settings.EnableSecurity -eq $null) {
        throw "Missing EnableSecurity value."
    }

    [Switch]$enableSecurity = [System.Convert]::ToBoolean($settings.EnableSecurity)

    # Remove Https
    $removeHttps = $false
    if ($settings.RemoveHttps) {
        if (-not [System.Boolean]::TryParse($settings.RemoveHttps, [ref]$removeHttps)) {
            throw "RemoveHttps must be 'true' or 'false'."
        }
    }

    # Verify/Add IsUninstallOnly value (true) to the $config string
    Write-LogOutput $("Verify/Add IsUninstallOnly parameter to the config string...")
    [string]$uninstallConfig = $config
    if (Get-Member -InputObject $settings -Name "IsUninstallOnly" -MemberType Properties) {
        if (![string]::Equals($settings.IsUninstallOnly, "true", [System.StringComparison]::OrdinalIgnoreCase)) {
            throw "IsUninstallOnly must be true for uninstallation."
        }

        Write-LogOutput $("Completed verifying IsUninstallOnly value in the config string.")
    }
    else {
        $settings | Add-Member -type NoteProperty -Name "IsUninstallOnly" -Value "true"
        $uninstallConfig = ConvertTo-Json $settings
        $uninstallConfig = [System.Convert]::ToBase64String([System.Text.Encoding]::UTF8.GetBytes($uninstallConfig))
        Write-LogOutput $("Completed Adding IsUninstallOnly value in the config string.")
    }

    # Use AxReportVmRoleStartupTask.exe to uninstall reporting extensions
    Write-LogOutput("")
    Write-LogOutput $("Start uninstalling AX SSRS extensions ...")

    $deployExtensionExe = Join-Path -Path $PSScriptRoot -ChildPath "AxReportVmRoleStartupTask.exe"
    $allArgs = @("-config", $uninstallConfig)
    & $deployExtensionExe $allArgs *>&1
    $exitCode = $LASTEXITCODE
    if ($exitCode -eq 0) {
        $serviceName = Get-SsrsServiceName
        Write-LogOutput ("The reporting service name is: $serviceName.")
        Restart-WindowsService -log $log -computerName $env:COMPUTERNAME -serviceName $serviceName -ErrorAction stop
        Write-LogOutput $("Completed uninstalling AX SSRS extension.")
    }
    else {
        throw "Error occurred when running AxReportVmRoleStartupTask.exe to uninstall extension. Please see details in events for AX-ReportPVMSetup."
    }

    if ($removeHttps) {
        Write-LogOutput "Start removing all https endpoints and SSL cert bindings"
        Config-SsrsEndpoint -Port 80 -Log $log -ErrorAction Stop
        Write-LogOutput "Completed removing all https endpoints and SSL cert bindings"
    }
}

################################################################################
# Format report publishing results
################################################################################
Function Format-PublishOutput {
    param
    (
        [Parameter(ValueFromPipeline = $true, Position = 0)] 
        $result
    )

    process {
        if ($result -is [Microsoft.Dynamics.AX.Framework.Management.Reports.ReportDeploymentResult]) {
            $result | Select-Object ReportName, ReportDeploymentStatus
        }
        else {
            $result
        }
    }
}


################################################################################
# Deploy AX reports To SSRS server
################################################################################
Function Deploy-AxReport {
    param
    (
        [Parameter(Mandatory = $true)]
        [string]$config,
        [Parameter(Mandatory = $true)]
        [string]$log
    )

    Import-Module "$PSScriptRoot\AosCommon.psm1" -DisableNameChecking

    # Initialize the log file
    Initialize-Log $log

    # Getting values from the JSON configuration string
    $settings = Decode-Settings($config)

    [string]$reportingServers = $settings."BiReporting.ReportingServers"
    [string]$packageInstallLocation = $settings."Microsoft.Dynamics.AX.AosConfig.AzureConfig.bindir"
    [string]$useHttps = $settings."BiReporting.SsrsUseHttps"
    [string]$httpsPort = $settings."BiReporting.SsrsHttpsPort"
    [string[]]$module = $settings.Module 
    [string[]]$reportName = $settings.ReportName

    # Popuate default values 
    Write-LogOutput $('Microsoft.Dynamics.AX.AosConfig.AzureConfig.bindir: ' + $packageInstallLocation)
    Write-LogOutput $('Module: ' + $module)
    Write-LogOutput $('ReportName: ' + $reportName)

    if ($module -eq $null -or $module.Length -eq 0) {
        $module = "*"
    }

    if ($reportName -eq $null -or $reportName.Length -eq 0) {
        $reportName = "*"
    }

    Write-LogOutput ""

    # Getting Protocol and Port for ReportServer service
    if ($useHttps -eq $true) {
        $rsUri = Get-ReportServerUri -UseHttps -HttpsPort $HttpsPort
    }
    else {
        $rsUri = Get-ReportServerUri -HttpsPort $HttpsPort
    }
    $protocol = $rsUri[0]
    $port = $rsUri[1]
    Write-LogOutput $("Protocol: $protocol; Port: $port")
    Write-LogOutput ""

    # Get report servers IP address
    $rsServers = Get-ReportServers -reportingServers $reportingServers
    
    # Test each SSRS instance prior to deploying any reports. If this is not successful
    # then there is no need to deploy reports.
    $reportServerUrls = @{}
    foreach ($rsServer in $rsServers) {
        [string]$rsUrl = $($protocol + $rsServer + ":" + $port.ToString() + "/ReportServer")
        Write-LogOutput $("Testing: $rsUrl ...")
        Test-ReportServer -reportServerUri $rsUrl -log $log -ErrorAction Stop
        $reportServerUrls.Add($rsServer, $rsUrl)
    }

    Write-LogOutput $("Testing reporting services completed.")
    Write-LogOutput ""
    
    # Start deploying reports
    Write-LogOutput $("Start deploying reports ...")
    $startTime = $(Get-Date)

    if ($rsServers.Count -eq 1) {
        [string]$ManagementAssemblyPath = Join-Path -Path $PSScriptRoot -ChildPath "Microsoft.Dynamics.AX.Framework.Management.dll"
        Import-Module $ManagementAssemblyPath

        $err = @()
        Publish-AXReport -MaxDegreeOfParallelism 1 -ErrorAction Continue -ErrorVariable err -ReportName $reportName -SkipReportServerAdminCheck -ReportServerAddress $reportServerUrls.Values[0] -BinDirectory $PackageInstallLocation -DeleteExisting -Module $module *>&1 | Format-PublishOutput | Tee-Object -Variable DeployReportsLog
        Add-Content -Path $log -Value $($DeployReportsLog | Out-String)
        Write-LogOutput $(($(Get-Date) - $startTime).TotalSeconds.ToString() + " Seconds.")

        if ($err.Count -gt 0) {
            throw "Errors occured during report deployment."
        }
    }
    else {
        foreach ($rsServer in $rsServers) {
            Start-Job -Name $("Job" + $rsServer) -ScriptBlock {
                Param
                (
                    [Parameter(Mandatory = $true)]
                    [string]$scriptRoot,
                    [Parameter(Mandatory = $true)]
                    [string]$rsUrlArg,
                    [Parameter(Mandatory = $true)]
                    [string[]]$reportNameArg,
                    [Parameter(Mandatory = $true)]
                    [string]$packageInstallLocationArg,
                    [Parameter(Mandatory = $true)]
                    [string[]]$moduleArg
                )

                Import-Module "$scriptRoot\AosCommon.psm1" -Force -DisableNameChecking -ErrorAction Stop
                Import-Module "$scriptRoot\Reporting.psm1" -Force -DisableNameChecking -ErrorAction Stop

                [string]$ManagementAssemblyPath = Join-Path -Path $scriptRoot -ChildPath "Microsoft.Dynamics.AX.Framework.Management.dll"
                Import-Module $ManagementAssemblyPath -ErrorAction Stop

                $err = @()
                Publish-AXReport -MaxDegreeOfParallelism 1 -ErrorAction Continue -ErrorVariable err -ReportName $reportNameArg -SkipReportServerAdminCheck -ReportServerAddress $rsUrlArg -BinDirectory $packageInstallLocationArg -DeleteExisting -Module $moduleArg *>&1 | Format-PublishOutput | Out-String -OutVariable DeployReportsLog
                return @{"Error" = $err; "Log" = $DeployReportsLog }
            } -ArgumentList $PSScriptRoot, $reportServerUrls[$rsServer], $reportName, $packageInstallLocation, $module
        }

        $allErrors = @() 
        foreach ($rsServer in $rsServers) {
            $job = Get-Job $("Job" + $rsServer)
            $jobResult = Receive-Job -Job $job -Wait -AutoRemoveJob

            if ($jobResult."Error") {
                $allErrors += $jobResult."Error"
            }

            if ($jobResult."Log") {
                Write-LogOutput $($jobResult."Log" | Out-String)
            }
        }

        Write-LogOutput $(($(Get-Date) - $startTime).TotalSeconds.ToString() + " Seconds.")

        foreach ($error in $allErrors) {
            if ($error.Count -gt 0) {
                throw @("Errors occured during report deployment for server " + $rsServer)
            }
        }
    }
}


################################################################################
# Remove all AX reports from SSRS server
################################################################################
Function Remove-AllAxReports {
    param
    (
        [Parameter(Mandatory = $true)]
        [string]$config,
        [Parameter(Mandatory = $true)]
        [string]$log
    )

    Import-Module "$PSScriptRoot\AosCommon.psm1" -DisableNameChecking

    # Initialize the log file
    Initialize-Log $log

    # Getting values from the JSON configuration string
    $settings = Decode-Settings($config)

    [string]$reportingServers = $settings."BiReporting.ReportingServers"
    [string[]]$reportName = $settings.ReportName
    [string]$useHttps = $settings."BiReporting.SsrsUseHttps"
    [string]$httpsPort = $settings."BiReporting.SsrsHttpsPort"    

    # populate default values
    Write-LogOutput $('ReportName: ' + $reportName)
    if ($reportName -eq $null -or $reportName.Length -eq 0) {
        $reportName = "*"
    }

    # Getting Protocol and Port for ReportServer service
    $protocol = "http://"
    $port = 80

    Write-LogOutput $("Getting the protocol and port ...")
    if (-not [String]::IsNullOrWhiteSpace($useHttps)) {
        $useHttpsValue = $false
        if (-not [Boolean]::TryParse($useHttps, [ref]$useHttpsValue)) {
            throw "Invalid value for BiReporting.SsrsUseHttps. The value is : $($useHttps)"
        }
    
        if ($useHttpsValue) {
            $protocol = "https://"
            $port = 443
            if (-not [String]::IsNullOrWhiteSpace($httpsPort)) {
                if (-not [Int32]::TryParse($httpsPort, [ref]$port)) {
                    throw "Invalid vlaue for BiReporting.SsrsHttpsPort. The value is: $httpsPort"
                }
            }
        }
    }
    
    Write-LogOutput $("Protocol: $protocol; Port: $port")
    Write-LogOutput ""    

    # split report servers string
    Write-LogOutput $('Reporting Servers: ' + $reportingServers)
    $rsServers = $reportingServers.Split(",", [System.StringSplitOptions]::RemoveEmptyEntries) | % { $_.trim() }
    $rsServers = $rsServers.Where({ -not [System.String]::IsNullOrWhiteSpace($_) })
    
    # Test each SSRS instance prior to deploying any reports. If this is not successful
    # then there is no need to deploy reports.
    $reportServerUrls = @{}
    foreach ($rsServer in $rsServers) {
        [string]$rsUrl = $($protocol + $rsServer + ":" + $port.ToString() + "/ReportServer")
        Write-LogOutput $("Testing: $rsUrl ...")
        Test-ReportServer -reportServerUri $rsUrl -log $log -ErrorAction Stop
        $reportServerUrls.Add($rsServer, $rsUrl)
    }
    Write-LogOutput $("Testing reporting services completed.")
    Write-LogOutput ""

    # Start clearing reports
    Write-LogOutput $("Start removing reports ....")
    $startTime = $(Get-Date)

    if ($rsServers.Count -eq 1) {
        [string]$ManagementAssemblyPath = Join-Path -Path $PSScriptRoot -ChildPath "Microsoft.Dynamics.AX.Framework.Management.dll"
        Import-Module $ManagementAssemblyPath

        Remove-AxReport -ErrorAction Continue -MaxDegreeOfParallelism 1 -ReportName $reportName -ReportServerAddress $reportServerUrls.Values[0] *>&1 |  Format-PublishOutput | Tee-Object -Variable ReportRemovalLog
        Add-Content -Path $log -Value $ReportRemovalLog
        Write-LogOutput $(($(Get-Date) - $startTime).TotalSeconds.ToString() + " Seconds.")
    }
    else {
        foreach ($rsServer in $rsServers) {
            Start-Job -Name $("RmJob" + $rsServer) -ScriptBlock {
                Param
                (
                    [Parameter(Mandatory = $true)]
                    [string]$scriptRoot,
                    [Parameter(Mandatory = $true)]
                    [string]$rsUrlArg,
                    [Parameter(Mandatory = $true)]
                    [string[]]$reportNameArg
                )

                Import-Module "$scriptRoot\AosCommon.psm1" -Force -DisableNameChecking -ErrorAction Stop
                Import-Module "$scriptRoot\Reporting.psm1" -Force -DisableNameChecking -ErrorAction Stop

                [string]$ManagementAssemblyPath = Join-Path -Path $scriptRoot -ChildPath "Microsoft.Dynamics.AX.Framework.Management.dll"
                Import-Module $ManagementAssemblyPath -ErrorAction Stop

                Remove-AxReport -ErrorAction Continue -MaxDegreeOfParallelism 1 -ReportName $reportNameArg -ReportServerAddress $rsUrlArg *>&1 | Format-PublishOutput | Out-String -OutVariable ReportRemovalLog
            } -ArgumentList $PSScriptRoot, $reportServerUrls[$rsServer], $reportName
        }

        foreach ($rsServer in $rsServers) {
            $job = Get-Job $("RmJob" + $rsServer)
            $jobResultLog = Receive-Job -Job $job -Wait -AutoRemoveJob

            if ($jobResultLog) {
                Write-LogOutput $($jobResultLog | Out-String)
            }
        }

        Write-LogOutput $(($(Get-Date) - $startTime).TotalSeconds.ToString() + " Seconds.")
    }
}

################################################################################
# Install report fonts to SSRS server
################################################################################
Function Install-Font {
    param
    (
        [Parameter(Mandatory = $true)]
        [string]$config,
        [Parameter(Mandatory = $true)]
        [string]$log,
        [Parameter(Mandatory = $true)]
        [string]$fontFilePath
    )

    Import-Module "$PSScriptRoot\AosCommon.psm1" -DisableNameChecking

    # Initialize the log file
    Initialize-Log $log

    # Start installing fonts
    Write-Output("")
    Write-LogOutput "Start installing fonts to SSRS ..."
    Write-LogOutput $("Font file path: " + $fontFilePath)
    $MangementAssemblyPath = Join-Path -Path $PSScriptRoot -ChildPath "Microsoft.Dynamics.AX.Framework.Management.dll"
    Import-Module $MangementAssemblyPath

    (Get-ChildItem -Path $fontFilePath -Filter "*.ttf").FullName | Install-ReportFont *>&1 | Tee-Object -Variable ReportFontLog
    Add-Content -Path $log -Value $ReportFontLog

    # Restart SSRS
    $serviceName = Get-SsrsServiceName
    Write-LogOutput ("The reporting service name is: $serviceName.")
    Get-Service -Name $serviceName | Restart-Service *>&1 | Tee-Object -Variable RestartSsrsLog
    Add-Content -Path $log -Value $RestartSsrsLog

    Write-LogOutput $("Completed installing fonts to SSRS.")
}

#####################################################
# Configure SSRS web service and web app protocols
#####################################################
function Config-SsrsEndpoint {
    [CmdletBinding()]
    param
    (
        [Parameter(Mandatory = $true)]
        [int]$Port,
        [Parameter(Mandatory = $true)]
        [string]$log,
        [Parameter(Mandatory = $false)]
        [string]$SslCertThumbprint
    )

    Import-Module "$PSScriptRoot\AosCommon.psm1" -DisableNameChecking

    $rsConfig = Get-RsConfig -ComputerName $env:COMPUTERNAME
    if (!$rsConfig) {
        throw "Could not find reporting service WMI object MSReportServer_ConfigurationSetting"
    }

    $lcid = [System.Globalization.CultureInfo]::GetCultureInfo('en-US').LCID
    $rsWebServiceAppName = 'ReportServerWebService'
    $reportManagerAppName = 'ReportServerWebApp'

    # Remove all existing ResvervedUrls
    $result = $rsConfig.ListReservedUrls()
    Verify-SsrsWmiCall -resultObject $result -methodName 'ListReservedurls'
    if ($result.Application) {
        for ($i = 0; $i -lt $result.Application.length; $i++) {
            $removeResult = $rsConfig.RemoveURL($result.Application[$i], $result.UrlString[$i], $lcid)
            Verify-SsrsWmiCall -resultObject $removeResult -methodName 'RemoveURL'
            Write-LogOutput $([string]::Format("Removed URL Application={0} UrlString= {1}", $result.Application[$i], $result.UrlString[$i]))
        }
    }

    $removedIpPorts = New-Object -TypeName System.Collections.Generic.HashSet[string]

    # Remove all SSL Certficate Bindings from Reporting Service
    $result = $rsConfig.ListSSLCertificateBindings($lcid)
    if ($result.Application) {
        for ($i = 0; $i -lt $result.Application.length; $i++) {
            $removeResult = $rsConfig.RemoveSSLCertificateBindings($result.Application[$i], $result.CertificateHash[$i], $result.IPAddress[$i], $result.Port[$i], $lcid)
            Verify-SsrsWmiCall -resultObject $removeResult -methodName 'RemoveSSLCertificateBindings'
            Write-LogOutput $([string]::Format("Removed SSL Binding Application={0} Certificate={1} IPAddress={2} Port={3}", $result.Application[$i], $result.CertificateHash[$i], $result.IPAddress[$i], $result.Port[$i]))

            Remove-SSLBindings -ipAddress $($result.IPAddress[$i]).ToString() -port $($result.Port[$i]).ToString() -removedIpPorts $removedIpPorts
        }
    }

    # Remove all SSL Certficate Bindings and ssl bindings from OS
    if ($SslCertThumbprint) {
        Remove-SSLBindings -ipAddress "0.0.0.0" -port $Port.ToString() -removedIpPorts $removedIpPorts
        Remove-SSLBindings -ipAddress "[::]" -port $Port.ToString() -removedIpPorts $removedIpPorts
        Remove-ReservedUrl -url "https://+:$($Port.ToString())/ReportServer/"
        Remove-ReservedUrl -url "https://+:$($Port.ToString())/Reports/"
    }

    # Reserve URL for web service and web app
    $urlString = $([string]::Format("http://+:{0}", $Port))
    if ($SslCertThumbprint) {
        $urlString = [string]::Format("https://+:{0}", $Port)
    }

    $rsConfig.SetVirtualDirectory($rsWebServiceAppName, "ReportServer", $lcid)
    $result = $rsConfig.ReserveURL($rsWebServiceAppName, $urlString, $lcid)
    Verify-SsrsWmiCall -resultObject $result -methodName 'ReserveURL'
    Write-LogOutput $([string]::Format("Reserved URL string {0} for {1}", $urlString, $rsWebServiceAppName))

    $rsConfig.SetVirtualDirectory($reportManagerAppName, "Reports", $lcid)
    $result = $rsConfig.ReserveURL($reportManagerAppName, $urlString, $lcid)
    Verify-SsrsWmiCall -resultObject $result -methodName 'ReserveURL'
    Write-LogOutput $([string]::Format("Reserved URL string {0} for {1}", $urlString, $reportManagerAppName))

    # Create SSL Certificate Bindings for web service and web app
    if ($SslCertThumbprint) {
        $ipV4Address = "0.0.0.0";
        $ipV6Address = "::";

        $result = $rsConfig.CreateSSLCertificateBinding($rsWebServiceAppName, $SslCertThumbprint, $ipV4Address, $Port, $lcid);
        Verify-SsrsWmiCall -resultObject $result -methodName 'CreateSSLCertificateBinding'
        Write-LogOutput $([string]::Format("Created SSL Certificate Binding Application={0} SslCertThumbPrint={1} IPAddress={2} Port={3}", $rsWebServiceAppName, $SslCertThumbprint, $ipV4Address, $Port))

        $result = $rsConfig.CreateSSLCertificateBinding($rsWebServiceAppName, $SslCertThumbprint, $ipV6Address, $Port, $lcid);
        Verify-SsrsWmiCall -resultObject $result -methodName 'CreateSSLCertificateBinding'
        Write-LogOutput $([string]::Format("Created SSL Certificate Binding Application={0} SslCertThumbPrint={1} IPAddress={2} Port={3}", $rsWebServiceAppName, $SslCertThumbprint, $ipV6Address, $Port))

        $result = $rsConfig.CreateSSLCertificateBinding($reportManagerAppName, $SslCertThumbprint, $ipV4Address, $Port, $lcid);
        Verify-SsrsWmiCall -resultObject $result -methodName 'CreateSSLCertificateBinding'
        Write-LogOutput $([string]::Format("Created SSL Certificate Binding Application={0} SslCertThumbPrint={1} IPAddress={2} Port={3}", $reportManagerAppName, $SslCertThumbprint, $ipV4Address, $Port))

        $result = $rsConfig.CreateSSLCertificateBinding($reportManagerAppName, $SslCertThumbprint, $ipV6Address, $Port, $lcid);
        Verify-SsrsWmiCall -resultObject $result -methodName 'CreateSSLCertificateBinding'
        Write-LogOutput $([string]::Format("Created SSL Certificate Binding Application={0} SslCertThumbPrint={1} IPAddress={2} Port={3}", $reportManagerAppName, $SslCertThumbprint, $ipV6Address, $Port))
    }

    #Restart reporting services service
    $serviceName = Get-SsrsServiceName
    Write-LogOutput ("The reporting service name is: $serviceName.")
    Restart-WindowsService -computerName $env:COMPUTERNAME -serviceName $serviceName -log $log
}

#####################################################
# Confirm SSRS web service and web app HTTPS endpoints
#####################################################
function Confirm-SsrsHttpsEndpoints {
    [CmdletBinding()]
    param
    (
        [Parameter(Mandatory = $false)]
        [string]$SslCertThumbprint,
        [Parameter(Mandatory = $true)]
        [int]$Port,
        [Parameter(Mandatory = $true)]
        [string]$log
    )

    Import-Module "$PSScriptRoot\AosCommon.psm1" -DisableNameChecking

    $rsConfig = Get-RsConfig -ComputerName $env:COMPUTERNAME
    if (!$rsConfig) {
        throw "Could not find reporting service WMI object MSReportServer_ConfigurationSetting"
    }

    $lcid = [System.Globalization.CultureInfo]::GetCultureInfo('en-US').LCID
    $rsWebServiceAppName = 'ReportServerWebService'
    $reportManagerAppName = 'ReportServerWebApp'

    # Test whether local HTTPS URL is configured
    $serviceName = Get-SsrsServiceName
    Write-LogOutput ("The reporting service name is: $serviceName.")
    Restart-WindowsService -log $log -computerName $env:COMPUTERNAME -serviceName $serviceName -ErrorAction stop
    $localHttpsReportServerUrl = $("https://localhost:" + $httpsPortValue.ToString() + "/ReportServer")

    $reconfigEndoints = $false
    $httpsEndpointExists = $true
    try {
        Test-ReportServer -reportServerUri $localHttpsReportServerUrl -forceTls12 $true -ignoreServerCertificate $true -log $log -ErrorAction Stop
    }
    catch {
        $httpsEndpointExists = $false
    }

    if (-not $httpsEndpointExists) {
        # No https endpoint, so there is no need to validate.
        $reconfigEndoints = $true
    }
    else {
        # Validate whether SSL Bindings need to be updated
        $expectedSslBindings = @{}
        $expectedSslBindings.Add($("$rsWebServiceAppName-0.0.0.0-$Port-$SslCertThumbprint"), 1)
        $expectedSslBindings.Add($("$rsWebServiceAppName-::-$Port-$SslCertThumbprint"), 1)
        $expectedSslBindings.Add($("$reportManagerAppName-0.0.0.0-$Port-$SslCertThumbprint"), 1)
        $expectedSslBindings.Add($("$reportManagerAppName-::-$Port-$SslCertThumbprint"), 1)
        
        $result = $rsConfig.ListSSLCertificateBindings($lcid)
        $foundBindingCount = 0
        if ($result.Application -and $result.Application.Length -eq $expectedSslBindings.Count) {
            for ($i = 0; $i -lt $result.Application.length; $i++) {
                $bindingString = $("$($result.Application[$i])-$($result.IPAddress[$i])-$($result.Port[$i])-$($result.CertificateHash[$i])")
                if ($expectedSslBindings.ContainsKey($bindingString)) {
                    Write-LogOutput "Found $bindingString"
                    $foundBindingCount++
                }
            }
        }

        if ($foundBindingCount -ne $expectedSslBindings.Count) {
            # SSL bindings do not have expected values, needs reconfig
            $reconfigEndoints = $true
        }
        else {
            # Remove all Http based URLs
            Write-LogOutput "SSRS reserved URLs and SSL bindings are already configured. Skip configuring."
            Remove-HttpUrl -rsConfig $rsConfig -lcid $lcid
        }
    }

    if ($reconfigEndoints) {
        Write-LogOutput "SSRS reserved URLs and SSL bindings need to be reconfigured."
        Config-SsrsEndpoint -Port $Port -log $log -SslCertThumbprint $SslCertThumbprint    
    }
}

####################################################################
# Remove SSL bindings from network configuration
####################################################################
function Remove-SSLBindings {
    param 
    (
        [Parameter(Mandatory = $true)]
        [string]$ipAddress,
        [Parameter(Mandatory = $true)]
        [string]$port,
        [Parameter(Mandatory = $true)]
        [AllowEmptyCollection()]
        [System.Collections.Generic.HashSet[string]]$removedIpPorts
    )

    if ($ipAddress -eq "::") {
        $ipAddress = "[::]"
    }

    $ipAddressWithPort = "$($ipAddress):$($port)"
    if (!$removedIpPorts.Contains($ipAddressWithPort)) {
        $showCommand = $("netsh http show sslcert ipport=$ipAddressWithPort")
        Write-LogOutput ""
        Write-LogOutput("Check SSL cert bindings from computer network config with '$showCommand")
        $netshResult = Invoke-Expression $showCommand -ErrorAction Stop
        Write-LogOutput $netshResult

        if ($netshResult.Count -ge 10) {
            $removeCommand = $("netsh http delete sslcert ipport=$ipAddressWithPort")
            Write-LogOutput("Remove SSL cert bindings from computer network config with '$removeCommand")
            $netshResult = Invoke-Expression $removeCommand -ErrorAction Stop
            $removedIpPorts.Add($ipAddressWithPort) | Out-Null
            Write-LogOutput $netshResult
        }
    }
}

####################################################################
# Remove reserved URL
####################################################################
function Remove-ReservedUrl {
    param 
    (
        [Parameter(Mandatory = $true)]
        [string]$url
    )

    $showCommand = $("netsh http show urlacl url=$url")
    Write-LogOutput ""
    Write-LogOutput $("Check reserved URL with $showCommand")
    $netshResult = Invoke-Expression $showCommand -ErrorAction Stop
    Write-LogOutput $netshResult

    if ($netshResult.Count -ge 6) {
        $removeCommand = $("netsh http delete urlacl url=$url")
        Write-LogOutput $("Remove reserved URL with $removeCommand")
        $netshResult = Invoke-Expression $removeCommand -ErrorAction Stop
        Write-LogOutput $netshResult
    }
}

####################################################################
# Remove all HTTP based URLs
####################################################################
function Remove-HttpUrl {
    param 
    (
        [Parameter(Mandatory = $true)]
        $rsConfig,
        [Parameter(Mandatory = $true)]
        $lcid
    )

    Write-LogOutput "Remove all HTTP URLs..."
    $result = $rsConfig.ListReservedUrls()
    Verify-SsrsWmiCall -resultObject $result -methodName 'ListReservedurls'
    if ($result.Application) {
        for ($i = 0; $i -lt $result.Application.length; $i++) {
            if ($result.UrlString[$i] -and $result.UrlString[$i].StartsWith("http://", [System.StringComparison]::OrdinalIgnoreCase)) {
                $removeResult = $rsConfig.RemoveURL($result.Application[$i], $result.UrlString[$i], $lcid)
                Verify-SsrsWmiCall -resultObject $removeResult -methodName 'RemoveURL'
                Write-LogOutput $([string]::Format("Removed URL Application={0} UrlString= {1}", $result.Application[$i], $result.UrlString[$i]))
            }
        }
    }    
}

####################################################################
# Verify SSRS WMI calls
####################################################################
function Verify-SsrsWmiCall {
    param
    (
        $resultObject,
        $methodName
    )

    if (!$resultObject) {
        throw $("Returned Null object when calling $methodName")
    }

    if ($resultObject.HRESULT -ne 0) {
        throw $("Error occured when calling {0}. HResult = {1} Error={2}" -f $methodName, $resultObject.HRESULT, $resultObject.Error)
    }
}


################################################################################
# Test the SSRS Server
################################################################################
Function Test-ReportServer {
    param
    (
        [Parameter(Mandatory = $true)]
        [string]$reportServerUri,
        [Parameter(Mandatory = $false)]
        [bool]$ignoreServerCertificate = $false,
        [Parameter(Mandatory = $false)]
        [bool]$forceTls12 = $false,
        [Parameter(Mandatory = $true)]
        [string]$log
    )

    # Initialize the log file
    Initialize-Log $log

    Write-LogOutput $('Checking status of Report Server URL: ' + $reportServerUri)

    # Assume a 400 response code
    $statusCode = 400

    $MaxNumberOfRetries = 5
    $retry = 1
    while ($retry -le $MaxNumberOfRetries) {
        try {
            try {
                # Invoke the web request and get the status
                $uri = New-Object -TypeName System.Uri($reportServerUri)
                [System.Net.HttpWebRequest]$request = [System.Net.HttpWebRequest]::CreateHttp($uri)
        
                $request.Timeout = 5 * 60 * 1000 
                $request.UseDefaultCredentials = $true
                $request.KeepAlive = $false
                if ($ignoreServerCertificate) {
                    $request.ServerCertificateValidationCallback = { $true }
                }

                if ($forceTls12) {
                    [System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]::Tls12
                }

                [System.Net.WebResponse]$response = $request.GetResponse()
                $statusCode = $response.StatusCode
            }
            catch [System.Net.WebException] {
                Write-LogOutput $('Failure! Status check of Report Server URL: ' + $reportServerUri)
                Write-LogOutput $($_.Exception.ToString())

                throw "An exception of type System.Net.WebException occurred when making an http request to: " + $reportServerUri + ". Refer to the log file for more details."
            }
    
            # check the status code is 200 OK 
            if ($statusCode -eq 200) {
                Write-LogOutput $('Success! Status check of Report Server URL: ' + $reportServerUri)
            }
            else {
                Write-LogOutput $('Failure! Status check of Report Server URL: ' + $reportServerUri)
                Write-LogOutput $('StatusCode value: ' + $statusCode)

                throw "Http response contains StatusCode of " + $statusCode + ". Unable to communicate with the SQL Server ReportServer service."
            }
	
            break
        }
        catch {
            if ($retry -lt $MaxNumberOfRetries) {
                $retry++
                [int]$waitSeconds = 10
                Write-LogOutput $('Wait ' + $waitSeconds + ' seconds and retry count ' + $retry + '(' + $MaxNumberOfRetries + ')')
                Start-Sleep -Seconds $waitSeconds
            }
            else {
                throw
            }
        }
    }
}

################################################################################
# Restart a windows service
################################################################################
function Restart-WindowsService {
    [CmdletBinding()]
    param
    (
        [Parameter(Mandatory = $true)]
        [string]$log,
        [Parameter(Mandatory = $true)]
        [string]$computerName,
        [Parameter(Mandatory = $true)]
        [string]$serviceName
    )

    Import-Module "$PSScriptRoot\AosCommon.psm1" -DisableNameChecking

    # Initialize log file
    Initialize-Log $log

    $retryCount = 3;

    while ($retryCount -gt 0) {
        Write-LogOutput $("Restart " + $serviceName + " service on " + $computerName + "...")
        
        try {
            Get-Service -ErrorAction Stop -Name $serviceName -ComputerName $computerName | Restart-Service -ErrorAction Stop -Force *>&1 | Tee-Object -Variable RestartServiceLog
            Add-Content -Path $log -Value $RestartServiceLog
            break
        }
        catch {
            $retryCount -= 1;
            if ($retryCount -le 0) {
                throw
            }

            Start-Sleep -Seconds 30
        }
    }

    Write-LogOutput $($serviceName + " service restarted on " + $computerName)
}

################################################################################
# Get reporting service config WMI object
################################################################################
function Get-RsConfig {
    [CmdletBinding()]
    param
    (
        [Parameter(Mandatory = $true)]
        [string]$ComputerName
    )

    Import-Module "$PSScriptRoot\AosCommon.psm1" -DisableNameChecking

    try {
        # SSRS 2022
        Write-Log "Searching SSRS 2022 WMI object"
        $rsconfig = Get-WmiObject -ErrorAction SilentlyContinue -namespace "root\Microsoft\SqlServer\ReportServer\RS_SSRS\v16\Admin" -class MSReportServer_ConfigurationSetting -ComputerName $ComputerName -Filter "InstanceName='SSRS'"
        
        if ($rsconfig) {
            Write-Log "Found SSRS 2022 WMI object"
            return $rsconfig
        }
 
        # SSRS 2019
        Write-Log "Searching SSRS 2019 WMI object"
        $rsconfig = Get-WmiObject -ErrorAction SilentlyContinue -namespace "root\Microsoft\SqlServer\ReportServer\RS_SSRS\v15\Admin" -class MSReportServer_ConfigurationSetting -ComputerName $ComputerName -Filter "InstanceName='SSRS'"
        
        if ($rsconfig) {
            Write-Log "Found SSRS 2019 WMI object"
            return $rsconfig
        }

        # SSRS 2016
        Write-Log "Searching SSRS 2016 WMI object"
        $rsconfig = Get-WmiObject -ErrorAction SilentlyContinue -namespace "root\Microsoft\SqlServer\ReportServer\RS_MSSQLSERVER\v13\Admin" -class MSReportServer_ConfigurationSetting -ComputerName $ComputerName -Filter "InstanceName='MSSQLSERVER'"
        
        if ($rsconfig) {
            Write-Log "Found SSRS 2019 WMI object"
            return $rsconfig
        }
    }
    catch {
        Write-Log $($_.Exception.ToString())
        throw "SSRS WMI object was not found."
    }

    return $rsconfig
}

################################################################################
# Set SSRS system report timeout limit
################################################################################
function Set-SsrsConfigurationInfoTimeout {
    [CmdletBinding()]
    param
    (
        [Parameter(Mandatory = $true)]
        [ValidateRange(1800, 7200)]
        [int]$timeoutInSeconds,
        [Parameter(Mandatory = $true)]
        [string]$reportServerUri,
        [Parameter(Mandatory = $true)]
        [string]$configName
    )

    Set-SsrsConfigurationSystemProperty -reportServerUri $reportServerUri -propertyName $configName -propertyValue $timeoutInSeconds
}

################################################################################
# Set SSRS execution log days kept property value
################################################################################
function Set-SsrsExecutionLogDaysKept {
    [CmdletBinding()]
    param
    (
        [Parameter(Mandatory = $true)]
        [ValidateRange(0, 60)]
        [int]$daysKept,
        [Parameter(Mandatory = $true)]
        [string]$reportServerUri
    )

    $propertyName = "ExecutionLogDaysKept"
    Set-SsrsConfigurationSystemProperty -reportServerUri $reportServerUri -propertyName $propertyName -propertyValue $daysKept
}

################################################################################
# Set SSRS system report timeout limit
################################################################################
function Set-SsrsConfigurationSystemProperty {
    [CmdletBinding()]
    param
    (
        [Parameter(Mandatory = $true)]
        [string]$reportServerUri,
        [Parameter(Mandatory = $true)]
        [string]$propertyName,
        [Parameter(Mandatory = $true)]
        [string]$propertyValue
    )

    $proxyUri = [System.IO.Path]::Combine($reportServerUri, "ReportService2005.asmx")
    $rsProxy = New-WebServiceProxy -Uri $proxyUri -UseDefaultCredential
    $systemProperties = $rsProxy.GetSystemProperties($null)
    if (-not $systemProperties) {
        throw "Could not get SSRS system properties"
    }

    $systemProperty = $systemProperties | Where-Object { $_.Name -eq $propertyName }
    if (-not $systemProperty) {
        throw "Could not find SSRS system property $propertyName"
    }

    # set system report timeout 
    $systemProperty.Value = $propertyValue
    $changeProperties = @($systemProperty)
    $rsProxy.SetSystemProperties($changeProperties)
}

################################################################################
# Get SSRS service name
################################################################################
function Get-SsrsServiceName {
    Import-Module "$PSScriptRoot\AosCommon.psm1" -DisableNameChecking

    $productNamePath = 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion'
    $productNameItem = 'ProductName'
    $ssrsPost2019ServiceName = 'SQLServerReportingServices'
    $ssrs2016ServiceName = 'ReportServer'
    $ssrsService = Get-Service -Name $ssrsPost2019ServiceName -ErrorAction SilentlyContinue
        
    if ($ssrsService) {
        Write-Log "The SSRS 2019 reporting services name is set to $ssrsService"
        $ssrsServiceName = $ssrsService.Name;
    }
    else {
        $ssrsService = Get-Service -Name $ssrs2016ServiceName -ErrorAction SilentlyContinue

        if ($ssrsService) {
            Write-Log "The SSRS 2016 reporting services name is set to $ssrsService"
            $ssrsServiceName = $ssrsService.Name;
        }
    }

    if ($ssrsServiceName) {
        Write-Log "The SSRS service name from reporting services is $ssrsServiceName."
        return $ssrsServiceName
    }
    else {
        Write-Log 'Looking up reporting services name from OS.'
        if (Test-Path -Path $productNamePath -PathType Container) {
            $productNamePathItem = Get-ItemProperty -Path $productNamePath 
            $productNameMember = Get-Member -InputObject $productNamePathItem -Name $productNameItem

            if ($productNameMember) {
                $productName = (Get-ItemProperty -Path $productNamePath -Name $productNameItem).$productNameItem
                Write-Log 'The product name is: ' + $productName
            }
        }

        if (($productName -eq 'Windows Server 2016 Datacenter') -or ($productName -eq 'Windows Server 2016 Standard')) {
            return $ssrs2016ServiceName
        }

        if (($productName -eq 'Windows Server 2019 Datacenter') -or ($productName -eq 'Windows Server 2019 Standard')) {
            return $ssrsPost2019ServiceName
        }

        if (($productName -eq 'Windows Server 2022 Datacenter') -or ($productName -eq 'Windows Server 2022 Standard')) {
            return $ssrsPost2019ServiceName
        }

        throw "The product name $productName is not a supported product"
    }
}

################################################################################
# Get Protocol and ports
################################################################################
Function Get-ReportServerUri {
    param 
    (
        [Parameter(Mandatory = $false)]
        [switch]$useHttps = $false,
        [Parameter(Mandatory = $false)]
        [int]$HttpsPort = 443
    )

    # Getting Protocol and Port for ReportServer service
    $protocol = "http://"
    $port = 80

    Write-Log $("Getting the protocol and port ...")
    if (-not [String]::IsNullOrWhiteSpace($useHttps)) {
        $useHttpsValue = $false
        if (-not [Boolean]::TryParse($useHttps, [ref]$useHttpsValue)) {
            throw "Invalid value for BiReporting.SsrsUseHttps. The value is : $($useHttps)"
        }
    
        if ($useHttpsValue) {
            $protocol = "https://"
            $port = 443
            if (-not [String]::IsNullOrWhiteSpace($httpsPort)) {
                if (-not [Int32]::TryParse($httpsPort, [ref]$port)) {
                    throw "Invalid vlaue for BiReporting.SsrsHttpsPort. The value is: $httpsPort"
                }
            }
        }
    }
    $reportServerUrls = @()
    $reportServerUrls += ($protocol, $port)

    return $reportServerUrls
}

################################################################################
# Get Reporting servers IP
################################################################################
Function Get-ReportServers {
    param
    (
        [Parameter(Mandatory = $false)]
        [string[]]$reportingServers = "127.0.0.1"
    )

    # Split report servers string
    Write-Log $('Reporting Servers: ' + $reportingServers)
    $rsServers = $reportingServers.Split(",", [System.StringSplitOptions]::RemoveEmptyEntries) | % { $_.trim() }
    $rsServers = $rsServers.Where({ -not [System.String]::IsNullOrWhiteSpace($_) })

    return $rsServers
}

################################################################################
# Validate the SSRS Service is up, if not start the service
################################################################################
Function Validate-ReportService {
    param
    (
        [Parameter(Mandatory = $false)]
        [string[]]$reportServerIP = "127.0.0.1",
        [Parameter(Mandatory = $false)]
        [switch]$UseHttps = $false,
        [Parameter(Mandatory = $false)]
        [int]$HttpsPort = 443,
        [Parameter(Mandatory = $true)]
        [string]$log
    )

    # Initialize the log file
    Initialize-Log $log

    # Getting Protocol and Port for ReportServer service
    if ($UseHttps) {
        $rsUri = Get-ReportServerUri -UseHttps -HttpsPort $HttpsPort
    }
    else {
        $rsUri = Get-ReportServerUri -HttpsPort $HttpsPort
    }

    $protocol = $rsUri[0]
    $port = $rsUri[1]
    Write-Log $("Protocol: $protocol; Port: $port")
    
    # Get report servers IP address
    $rsServers = Get-ReportServers -reportingServers $reportServerIP
    
    # Validate ReportServers 
    foreach ($rsServer in $rsServers) {
        [string]$rsUrl = $($protocol + $rsServer + ":" + $port.ToString() + "/ReportServer")
        Write-Log $("Testing: $rsUrl ...")

        #$computerName = [System.Net.Dns]::GetHostEntry($rsServer).HostName.Split('.')[0]
        $computerName = $computerName = Get-WmiObject win32_networkadapterconfiguration -filter "ipenabled = 'True'" -ComputerName $rsServer | foreach { $_.PSComputername }
        Write-Log ("Report Server Hostname: $computerName")

        # Check if the reporting service is not running, then restart it
        $serviceName = Get-SsrsServiceName
        $service = Get-Service -Name $serviceName -ComputerName $computerName
        $serviceStatus = $service.Status
        Write-Log ("The reporting service Status is: $serviceStatus.")

        if ($serviceStatus -ne 'Running') {
            $service | Set-Service -Status Running
            Write-Log 'Starting SSRS service'
            Start-Sleep -seconds 60
            if ($service.Status -eq 'Running') {
                Write-Log 'SSRS service is now Running'
            }

            # Test the SSRS report Server
            Test-ReportServer -reportServerUri $rsUrl -log $log -ErrorAction Stop
        }
    }   
}
# SIG # Begin signature block
# MIIoQwYJKoZIhvcNAQcCoIIoNDCCKDACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCfa748biK4fIL5
# dcslLjS8bMtIXDYxXU+h/PbZEpTBuaCCDXYwggX0MIID3KADAgECAhMzAAAEhV6Z
# 7A5ZL83XAAAAAASFMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjUwNjE5MTgyMTM3WhcNMjYwNjE3MTgyMTM3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDASkh1cpvuUqfbqxele7LCSHEamVNBfFE4uY1FkGsAdUF/vnjpE1dnAD9vMOqy
# 5ZO49ILhP4jiP/P2Pn9ao+5TDtKmcQ+pZdzbG7t43yRXJC3nXvTGQroodPi9USQi
# 9rI+0gwuXRKBII7L+k3kMkKLmFrsWUjzgXVCLYa6ZH7BCALAcJWZTwWPoiT4HpqQ
# hJcYLB7pfetAVCeBEVZD8itKQ6QA5/LQR+9X6dlSj4Vxta4JnpxvgSrkjXCz+tlJ
# 67ABZ551lw23RWU1uyfgCfEFhBfiyPR2WSjskPl9ap6qrf8fNQ1sGYun2p4JdXxe
# UAKf1hVa/3TQXjvPTiRXCnJPAgMBAAGjggFzMIIBbzAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUuCZyGiCuLYE0aU7j5TFqY05kko0w
# RQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEW
# MBQGA1UEBRMNMjMwMDEyKzUwNTM1OTAfBgNVHSMEGDAWgBRIbmTlUAXTgqoXNzci
# tW2oynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NybC9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3JsMGEG
# CCsGAQUFBwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDovL3d3dy5taWNyb3NvZnQu
# Y29tL3BraW9wcy9jZXJ0cy9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3J0
# MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQELBQADggIBACjmqAp2Ci4sTHZci+qk
# tEAKsFk5HNVGKyWR2rFGXsd7cggZ04H5U4SV0fAL6fOE9dLvt4I7HBHLhpGdE5Uj
# Ly4NxLTG2bDAkeAVmxmd2uKWVGKym1aarDxXfv3GCN4mRX+Pn4c+py3S/6Kkt5eS
# DAIIsrzKw3Kh2SW1hCwXX/k1v4b+NH1Fjl+i/xPJspXCFuZB4aC5FLT5fgbRKqns
# WeAdn8DsrYQhT3QXLt6Nv3/dMzv7G/Cdpbdcoul8FYl+t3dmXM+SIClC3l2ae0wO
# lNrQ42yQEycuPU5OoqLT85jsZ7+4CaScfFINlO7l7Y7r/xauqHbSPQ1r3oIC+e71
# 5s2G3ClZa3y99aYx2lnXYe1srcrIx8NAXTViiypXVn9ZGmEkfNcfDiqGQwkml5z9
# nm3pWiBZ69adaBBbAFEjyJG4y0a76bel/4sDCVvaZzLM3TFbxVO9BQrjZRtbJZbk
# C3XArpLqZSfx53SuYdddxPX8pvcqFuEu8wcUeD05t9xNbJ4TtdAECJlEi0vvBxlm
# M5tzFXy2qZeqPMXHSQYqPgZ9jvScZ6NwznFD0+33kbzyhOSz/WuGbAu4cHZG8gKn
# lQVT4uA2Diex9DMs2WHiokNknYlLoUeWXW1QrJLpqO82TLyKTbBM/oZHAdIc0kzo
# STro9b3+vjn2809D0+SOOCVZMIIHejCCBWKgAwIBAgIKYQ6Q0gAAAAAAAzANBgkq
# hkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
# EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
# IDIwMTEwHhcNMTEwNzA4MjA1OTA5WhcNMjYwNzA4MjEwOTA5WjB+MQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQg
# Q29kZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
# CgKCAgEAq/D6chAcLq3YbqqCEE00uvK2WCGfQhsqa+laUKq4BjgaBEm6f8MMHt03
# a8YS2AvwOMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc6Whe0t+bU7IKLMOv2akr
# rnoJr9eWWcpgGgXpZnboMlImEi/nqwhQz7NEt13YxC4Ddato88tt8zpcoRb0Rrrg
# OGSsbmQ1eKagYw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+lD3v++MrWhAfTVYoonpy
# 4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+EGvKhL1nkkDstrjNYxbc+/jLTswM9
# sbKvkjh+0p2ALPVOVpEhNSXDOW5kf1O6nA+tGSOEy/S6A4aN91/w0FK/jJSHvMAh
# dCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rhKEmdX4jiJV3TIUs+UsS1Vz8k
# A/DRelsv1SPjcF0PUUZ3s/gA4bysAoJf28AVs70b1FVL5zmhD+kjSbwYuER8ReTB
# w3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zdsGbiwZeBe+3W7UvnSSmn
# Eyimp31ngOaKYnhfsi+E11ecXL93KCjx7W3DKI8sj0A3T8HhhUSJxAlMxdSlQy90
# lfdu+HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS4NaIjAsCAwEAAaOCAe0w
# ggHpMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBRIbmTlUAXTgqoXNzcitW2o
# ynUClTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYD
# VR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBRyLToCMZBDuRQFTuHqp8cx0SOJNDBa
# BgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
# bC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3JsMF4GCCsG
# AQUFBwEBBFIwUDBOBggrBgEFBQcwAoZCaHR0cDovL3d3dy5taWNyb3NvZnQuY29t
# L3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3J0MIGfBgNV
# HSAEgZcwgZQwgZEGCSsGAQQBgjcuAzCBgzA/BggrBgEFBQcCARYzaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1hcnljcHMuaHRtMEAGCCsG
# AQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAHAAbwBsAGkAYwB5AF8AcwB0AGEAdABl
# AG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn8oalmOBUeRou09h0ZyKb
# C5YR4WOSmUKWfdJ5DJDBZV8uLD74w3LRbYP+vj/oCso7v0epo/Np22O/IjWll11l
# hJB9i0ZQVdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0bpdS1HXeUOeLpZMlEPXh6
# I/MTfaaQdION9MsmAkYqwooQu6SpBQyb7Wj6aC6VoCo/KmtYSWMfCWluWpiW5IP0
# wI/zRive/DvQvTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvyCInWH8MyGOLwxS3OW560
# STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn+N4sOiBpmLJZiWhub6e3dMNABQam
# ASooPoI/E01mC8CzTfXhj38cbxV9Rad25UAqZaPDXVJihsMdYzaXht/a8/jyFqGa
# J+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw3MYbBL7fQccOKO7eZS/sl/ah
# XJbYANahRr1Z85elCUtIEJmAH9AAKcWxm6U/RXceNcbSoqKfenoi+kiVH6v7RyOA
# 9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sLgOppO6/8MO0ETI7f33Vt
# Y5E90Z1WTk+/gFcioXgRMiF670EKsT/7qMykXcGhiJtXcVZOSEXAQsmbdlsKgEhr
# /Xmfwb1tbWrJUnMTDXpQzTGCGiMwghofAgEBMIGVMH4xCzAJBgNVBAYTAlVTMRMw
# EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVN
# aWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNp
# Z25pbmcgUENBIDIwMTECEzMAAASFXpnsDlkvzdcAAAAABIUwDQYJYIZIAWUDBAIB
# BQCgga4wGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIORbaQ/yhbrMNg1oDsoCwCsp
# XaRvWRJrqk8Sn2iZZKxyMEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8A
# cwBvAGYAdKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEB
# BQAEggEAdsXhvExLloqQ29poR68MeA8k459jvf4fhEFhTXCYHXMlHSd0rGWPQ2Qo
# A2h8j9flSJwb9Q5uFOTkk2mxMhGbosVKXGuyjcg79K9ISKYI4yFVci1vH3wqX0Yj
# GiytPx0CIY3ZogE/zTAMScbBSxcEDg76PUP6Ryo2xqzN6Ikge4bO0gCwPDuiqAlJ
# Rse8XcaDpWqY8DVLQYc3tH1xKJlhv1eZhK3bL7Lf8qutjhtoXC2NBkMcQyeSfr7o
# ZRK6JXJtb9/gyRnu31BOknHnq3XIzeSooYycSjhOwQeBW/6NM3lF+2GXuzsu1x14
# DnWDd5ujBbGgCmlIAcFhX9dGvJ1Q66GCF60wghepBgorBgEEAYI3AwMBMYIXmTCC
# F5UGCSqGSIb3DQEHAqCCF4YwgheCAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFaBgsq
# hkiG9w0BCRABBKCCAUkEggFFMIIBQQIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFl
# AwQCAQUABCBdvkQedoWa9ecRWIKm4YtKrDDiEDaEGhfRVeI74H09agIGaKR/10nM
# GBMyMDI1MDkwNDA3MjU0Ni44MDNaMASAAgH0oIHZpIHWMIHTMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJl
# bGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVT
# Tjo1MjFBLTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAg
# U2VydmljZaCCEfswggcoMIIFEKADAgECAhMzAAACAAvXqn8bKhdWAAEAAAIAMA0G
# CSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMB4XDTI0
# MDcyNTE4MzEyMVoXDTI1MTAyMjE4MzEyMVowgdMxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9w
# ZXJhdGlvbnMgTGltaXRlZDEnMCUGA1UECxMeblNoaWVsZCBUU1MgRVNOOjUyMUEt
# MDVFMC1EOTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNl
# MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAr1XaadKkP2TkunoTF573
# /tF7KJM9Doiv3ccv26mqnUhmv2DM59ikET4WnRfo5biFIHc6LqrIeqCgT9fT/Gks
# 5VKO90ZQW2avh/PMHnl0kZfX/I5zdVooXHbdUUkPiZfNXszWswmL9UlWo8mzyv9L
# p9TAtw/oXOYTAxdYSqOB5Uzz1Q3A8uCpNlumQNDJGDY6cSn0MlYukXklArChq6l+
# KYrl6r/WnOqXSknABpggSsJ33oL3onmDiN9YUApZwjnNh9M6kDaneSz78/YtD/2p
# Gpx9/LXELoazEUFxhyg4KdmoWGNYwdR7/id81geOER69l5dJv71S/mH+Lxb6L692
# n8uEmAVw6fVvE+c8wjgYZblZCNPAynCnDduRLdk1jswCqjqNc3X/WIzA7GGs4HUS
# 4YIrAUx8H2A94vDNiA8AWa7Z/HSwTCyIgeVbldXYM2BtxMKq3kneRoT27NQ7Y7n8
# ZTaAje7Blfju83spGP/QWYNZ1wYzYVGRyOpdA8Wmxq5V8f5r4HaG9zPcykOyJpRZ
# y+V3RGighFmsCJXAcMziO76HinwCIjImnCFKGJ/IbLjH6J7fJXqRPbg+H6rYLZ8X
# BpmXBFH4PTakZVYxB/P+EQbL5LNw0ZIM+eufxCljV4O+nHkM+zgSx8+07BVZPBKs
# looebsmhIcBO0779kehciYMCAwEAAaOCAUkwggFFMB0GA1UdDgQWBBSAJSTavgkj
# Kqge5xQOXn35fXd3OjAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJlpxtTNRnpcjBf
# BgNVHR8EWDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3Bz
# L2NybC9NaWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcmww
# bAYIKwYBBQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3Lm1pY3Jvc29m
# dC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0El
# MjAyMDEwKDEpLmNydDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUF
# BwMIMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQsFAAOCAgEAKPCG9njRtIqQ
# +fuECgxzWMsQOI3HvW7sV9PmEWCCOWlTuGCIzNi3ibdLZS0b2IDHg0yLrtdVuBi3
# FxVdesIXuzYyofIe/alTBdV4DhijLTXtB7NgOno7G12iO3t6jy1hPSquzGLry/2m
# EZBwIsSoS2D+H+3HCJxPDyhzMFqP+plltPACB/QNwZ7q+HGyZv3v8et+rQYg8sF3
# PTuWeDg3dR/zk1NawJ/dfFCDYlWNeCBCLvNPQBceMYXFRFKhcSUws7mFdIDDhZpx
# qyIKD2WDwFyNIGEezn+nd4kXRupeNEx+eSpJXylRD+1d45hb6PzOIF7BkcPtRtFW
# 2wXgkjLqtTWWlBkvzl2uNfYJ3CPZVaDyMDaaXgO+H6DirsJ4IG9ikId941+mWDej
# kj5aYn9QN6ROfo/HNHg1timwpFoUivqAFu6irWZFw5V+yLr8FLc7nbMa2lFSixzu
# 96zdnDsPImz0c6StbYyhKSlM3uDRi9UWydSKqnEbtJ6Mk+YuxvzprkuWQJYWfpPv
# ug+wTnioykVwc0yRVcsd4xMznnnRtZDGMSUEl9tMVnebYRshwZIyJTsBgLZmHM7q
# 2TFK/X9944SkIqyY22AcuLe0GqoNfASCIcZtzbZ/zP4lT2/N0pDbn2ffAzjZkhI+
# Qrqr983mQZWwZdr3Tk1MYElDThz2D0MwggdxMIIFWaADAgECAhMzAAAAFcXna54C
# m0mZAAAAAAAVMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UE
# CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
# b2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZp
# Y2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0yMTA5MzAxODIyMjVaFw0zMDA5MzAxODMy
# MjVaMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
# EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNV
# BAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIICIjANBgkqhkiG9w0B
# AQEFAAOCAg8AMIICCgKCAgEA5OGmTOe0ciELeaLL1yR5vQ7VgtP97pwHB9KpbE51
# yMo1V/YBf2xK4OK9uT4XYDP/XE/HZveVU3Fa4n5KWv64NmeFRiMMtY0Tz3cywBAY
# 6GB9alKDRLemjkZrBxTzxXb1hlDcwUTIcVxRMTegCjhuje3XD9gmU3w5YQJ6xKr9
# cmmvHaus9ja+NSZk2pg7uhp7M62AW36MEBydUv626GIl3GoPz130/o5Tz9bshVZN
# 7928jaTjkY+yOSxRnOlwaQ3KNi1wjjHINSi947SHJMPgyY9+tVSP3PoFVZhtaDua
# Rr3tpK56KTesy+uDRedGbsoy1cCGMFxPLOJiss254o2I5JasAUq7vnGpF1tnYN74
# kpEeHT39IM9zfUGaRnXNxF803RKJ1v2lIH1+/NmeRd+2ci/bfV+AutuqfjbsNkz2
# K26oElHovwUDo9Fzpk03dJQcNIIP8BDyt0cY7afomXw/TNuvXsLz1dhzPUNOwTM5
# TI4CvEJoLhDqhFFG4tG9ahhaYQFzymeiXtcodgLiMxhy16cg8ML6EgrXY28MyTZk
# i1ugpoMhXV8wdJGUlNi5UPkLiWHzNgY1GIRH29wb0f2y1BzFa/ZcUlFdEtsluq9Q
# BXpsxREdcu+N+VLEhReTwDwV2xo3xwgVGD94q0W29R6HXtqPnhZyacaue7e3Pmri
# Lq0CAwEAAaOCAd0wggHZMBIGCSsGAQQBgjcVAQQFAgMBAAEwIwYJKwYBBAGCNxUC
# BBYEFCqnUv5kxJq+gpE8RjUpzxD/LwTuMB0GA1UdDgQWBBSfpxVdAF5iXYP05dJl
# pxtTNRnpcjBcBgNVHSAEVTBTMFEGDCsGAQQBgjdMg30BATBBMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9y
# eS5odG0wEwYDVR0lBAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQCBAweCgBTAHUA
# YgBDAEEwCwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYDVR0jBBgwFoAU
# 1fZWy4/oolxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDovL2Ny
# bC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9vQ2VyQXV0XzIw
# MTAtMDYtMjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+aHR0cDov
# L3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXRfMjAxMC0w
# Ni0yMy5jcnQwDQYJKoZIhvcNAQELBQADggIBAJ1VffwqreEsH2cBMSRb4Z5yS/yp
# b+pcFLY+TkdkeLEGk5c9MTO1OdfCcTY/2mRsfNB1OW27DzHkwo/7bNGhlBgi7ulm
# ZzpTTd2YurYeeNg2LpypglYAA7AFvonoaeC6Ce5732pvvinLbtg/SHUB2RjebYIM
# 9W0jVOR4U3UkV7ndn/OOPcbzaN9l9qRWqveVtihVJ9AkvUCgvxm2EhIRXT0n4ECW
# OKz3+SmJw7wXsFSFQrP8DJ6LGYnn8AtqgcKBGUIZUnWKNsIdw2FzLixre24/LAl4
# FOmRsqlb30mjdAy87JGA0j3mSj5mO0+7hvoyGtmW9I/2kQH2zsZ0/fZMcm8Qq3Uw
# xTSwethQ/gpY3UA8x1RtnWN0SCyxTkctwRQEcb9k+SS+c23Kjgm9swFXSVRk2XPX
# fx5bRAGOWhmRaw2fpCjcZxkoJLo4S5pu+yFUa2pFEUep8beuyOiJXk+d0tBMdrVX
# VAmxaQFEfnyhYWxz/gq77EFmPWn9y8FBSX5+k77L+DvktxW/tM4+pTFRhLy/AsGC
# onsXHRWJjXD+57XQKBqJC4822rpM+Zv/Cuk0+CQ1ZyvgDbjmjJnW4SLq8CdCPSWU
# 5nR0W2rRnj7tfqAxM328y+l7vzhwRNGQ8cirOoo6CGJ/2XBjU02N7oJtpQUQwXEG
# ahC0HVUzWLOhcGbyoYIDVjCCAj4CAQEwggEBoYHZpIHWMIHTMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJl
# bGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVT
# Tjo1MjFBLTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAg
# U2VydmljZaIjCgEBMAcGBSsOAwIaAxUAjJOfLZb3ivipL3sSLlWFbLrWjmSggYMw
# gYCkfjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
# BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYD
# VQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQsF
# AAIFAOxjbB4wIhgPMjAyNTA5MDQwMTM4MzhaGA8yMDI1MDkwNTAxMzgzOFowdDA6
# BgorBgEEAYRZCgQBMSwwKjAKAgUA7GNsHgIBADAHAgEAAgI4MzAHAgEAAgITPTAK
# AgUA7GS9ngIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBCwUAA4IBAQCk2XAvTox4PvUv
# GQRQ19svohGiinxugm6TuDO+/yyHJtW/rxbCzBk0wNLlG7FXIuKbDzV5EAz+ib1X
# y2ImiSqhpZou1Gf7WOU3X5m6NWjR5E5cGDbuGUOqP54OiAUgEMorw7sx4mxzQ2Nj
# xv08e7OlpPlEKJM3iXTcb85u4dnnnUE5sRDcFSIEoVWaOtTP8GHWbWmx6rz4uPfI
# xOFT+XQRo/vUkEA/L+SzoP33fPb1J2VOREBcLmZ6gtwoHSmnoqodoTFSC6744bxd
# veLz8wknIKYfzvt8NmyOCYm484IVSbBiEQ/wLOLhOoKxlh5TjA1bW4tDwRDXDCon
# H4Z0VoVHMYIEDTCCBAkCAQEwgZMwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIw
# MTACEzMAAAIAC9eqfxsqF1YAAQAAAgAwDQYJYIZIAWUDBAIBBQCgggFKMBoGCSqG
# SIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0BCQQxIgQg7oeC8k6/0qxg
# SckhIRaQYWxvh2sJOA3e62Twa9YgfeQwgfoGCyqGSIb3DQEJEAIvMYHqMIHnMIHk
# MIG9BCDUyO3sNZ3burBNDGUCV4NfM2gH4aWuRudIk/9KAk/ZJzCBmDCBgKR+MHwx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1p
# Y3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAACAAvXqn8bKhdWAAEAAAIA
# MCIEIPBAFgb/ginhnThuqTWrVo74rG50Estjo4K2PuC/DO4NMA0GCSqGSIb3DQEB
# CwUABIICADECsmbUWXP4VInjXc/CjgyPogDcQ6Sl22nD0icdr03dFvpdVKdHNwI6
# u3IwVB6VT0ZRpODEVyDXqVFTPBYuJOJVexfuipu+xmbMptA9BLqR4n7AQ2F3dstp
# Q7/9L9zDyxTLs/vyiXKKn9UPC4Upei6Kxn0ti4QMIzRTnyBsYOnSpDyS7J3S5R92
# DN51+lNnlp/FHi02klnT/BIbT5p/gJFDiEaL86IbitRk6llJtsyYzkeCc6sPs7KS
# qazsbIlzZLiUhJtldSDX4w882ty/exJUdi4qMmXheRnhIr9TI6r6KUXr421Bh8nf
# orEjoAIaT4KPxCoEgAEjgtIEv9ijDMoLPSF7T2wZ0WCHvQJXHTDzcrozKXm6wx8W
# SUQ+1yzMOJ6aPFB+nKzktIxfP2VAQkprUEV/2xuz4nmVZzNK47vZsciqGofo92pa
# X5bfJmyRkTHcwZ1poL9b06hCEnX3+LU2hY7LDtHE5JF/7kYOG1XDuShDChIiiVTA
# ivHhMbTVjf8ojaNyZC7DlkwoCSte8FERvRDuPkSNeFF0JZjLkwTIPqAId8auj8+3
# aL3X1LCRAilRjrOHtgyLgpn4TShaZRR9T4ZLTFAuJ/iaGP9lFdgLgfeEe7kIsKHO
# hgs3cRbMSb6291KnrnehsGWJQS50qiPzupaOhVeg8E36qOIUT//+
# SIG # End signature block
