param
(
    [Parameter(Mandatory=$true)]
    $config,
    [Parameter(Mandatory=$true)]
    $log
)

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking

function Set-RegistryValues($settings)
{
    New-Item -Path HKLM:\SOFTWARE\Microsoft\Dynamics -Name Deployment -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name DeploymentDirectory -PropertyType String -Value $($settings.'Infrastructure.WebRoot') -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name BinDir -PropertyType String -Value $($settings.'Common.BinDir') -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name metadatadirectory -PropertyType String -Value $($settings.'Aos.MetadataDirectory') -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name frameworkdirectory -PropertyType String -Value $($settings.'Aos.PackageDirectory') -Force -Confirm:$false
}

function Configure-ClaimIssuers($webConfig, $settings)
{
    Write-Log "Start processing claim issuer restrictions."
    $ClaimIssuers = $settings.'Infrastructure.ClaimIssuers'
    if (![System.String]::IsNullOrWhitespace($ClaimIssuers))
    {
        $Issuers = $ClaimIssuers.Split(';')
        Write-Log "Processing claim issuer restrictions: $Issuers"

        #add missing elements to the web.config
        if ($Issuers.Count -gt 0)
        {
            $ClaimIssuerRestrictions = $webConfig.SelectNodes("configuration/claimIssuerRestrictions")
            if ($ClaimIssuerRestrictions -eq $null)
            {
                Write-Log "Web config does not contain element configuration/claimIssuerRestrictions. Creating."
                $ClaimIssuerRestrictions = $webConfig.CreateElement("claimIssuerRestrictions")
                $web.configuration.AppendChild($ClaimIssuerRestrictions)
            }

            $IssuerRestrictions = $ClaimIssuerRestrictions.issuerRestrictions
            if ($IssuerRestrictions -eq $null)
            {
                Write-Log "Web config does not contain element configuration/claimIssuerRestrictions/issuerRestrictions. Creating."
                $IssuerRestrictions = $webConfig.CreateElement("issuerRestrictions")
                $ClaimIssuerRestrictions.AppendChild($IssuerRestrictions)
            }
        }

        #add claim issuer restrictions
        foreach($Issuer in $Issuers)
        {
            $Tokens = $Issuer.Split(':')
            if ($Tokens.Count -lt 2 -or $Tokens.Count -gt 3)
            {
                throw "Claim issuer restriction is not in a valid format (expected format: issuerName:userId[,userId,userId,...[:defaultUserId]]): $Issuer"
            }

            $IssuerName = $Tokens[0]
            $AllowedUserIds = $Tokens[1]

            Write-Log "Processing issuer $IssuerName."

            $IssuerRestriction = $IssuerRestrictions.add | where { ($_.name -eq $IssuerName) }
            if ($IssuerRestriction -eq $null)
            {
                Write-Log "Creating new claim issuer restriction for issuer $IssuerName."

                $IssuerRestriction = $webConfig.CreateElement("add")
                $IssuerRestriction.SetAttribute("name", $IssuerName)
                $IssuerRestrictions.AppendChild($IssuerRestriction) | Out-Null
            }
            else
            {
                Write-Log "Claim issuer restriction already exists for issuer $IssuerName. Overriding with new restrictions."
            }

            $IssuerRestriction.SetAttribute("allowedUserIds", $AllowedUserIds)
            Write-Log "User ids $AllowedUserIds were set for $IssuerName."

            if ($Tokens.Count -eq 3)
            {
                $DefaultUserId = $Tokens[2]
                Write-Log "Setting default user id $DefaultUserId for $IssuerName."
                $IssuerRestriction.SetAttribute("defaultUserId", $DefaultUserId)
            }
        }
    }

    Write-Log "Finished processing claim issuer restrictions."
}

function Configure-MachineKeys($webConfig, $settings)
{
    Write-Log "Start processing machine keys."
    $DecryptionKey = $settings.'Infrastructure.MachineDecryptionKey'
    $ValidationKey = $settings.'Infrastructure.MachineValidationKey'

    if ($DecryptionKey -eq $null)
    {
        Write-Log "DecryptionKey not provided, skipping configuration."
        return
    }

    if ($ValidationKey -eq $null)
    {
        Write-Log "ValidationKey not provided, skipping configuration."
        return
    }

    $Web = $webConfig.SelectNodes("configuration/location[@path='.']/system.web")

    $MachineKey = $Web.machineKey

    if ($MachineKey -eq $null)
    {
        Write-Log "Creating machineKey element."

        $MachineKey = $webConfig.CreateElement("machineKey")
        $MachineKey.SetAttribute("decryption", "AES")
        $MachineKey.SetAttribute("decryptionKey", $DecryptionKey)
        $MachineKey.SetAttribute("validation", "HMACSHA256")
        $MachineKey.SetAttribute("validationKey", $ValidationKey)
        $Web.AppendChild($MachineKey) | Out-Null
    }
    else
    {
        Write-Log "Updating machineKey element."
        $MachineKey.decryption = "AES"
        $MachineKey.decryptionKey = $DecryptionKey
        $MachineKey.validation = "HMACSHA256"
        $MachineKey.validationKey = $ValidationKey
    }

    Write-Log "Finished processing machine keys."
}

function CreateFlightingCacheFolder($settings)
{
    Write-Log "Creating the Flighting Cache Folder for Application Platform Flighting"
    $webroot= $settings.'Infrastructure.WebRoot'
    $flightingcachefolder = $settings.'DataAccess.FlightingServiceCacheFolder'
    $flightingcachepath = Join-Path $webroot $flightingcachefolder

    if(!(Test-Path $flightingcachepath))
    {
        Write-Log "Creating $flightingcachepath."
        New-Item -ItemType Directory -Path $flightingcachepath -Force
    }

    #Flighting is enabled for SyncEngine.exe, it may be running by Admin,
    #so "Network Service" must be added at beginning in case W3Wp can not access it later.

    $User = "NETWORK SERVICE"
    $FullControlPermission = "FullControl"
    $Permission = (Get-Acl $flightingcachepath).Access | ?{$_.IdentityReference -match $User -and $_.FileSystemRights -match $FullControlPermission} | Select IdentityReference,FileSystemRights
    if ($Permission)
    {
        $Permission | % {Write-Log "User $($_.IdentityReference) has '$($_.FileSystemRights)' rights on folder $flightingcachepath"}
    }
    else
    {
        Write-Log "$User doesn't have full control on $flightingcachepath, adding $User"
        $Acl = Get-Acl $flightingcachepath

        $NetWorkService = New-Object System.Security.AccessControl.FileSystemAccessRule($User, $FullControlPermission, "ContainerInherit,ObjectInherit", "None", "Allow")

        $Acl.SetAccessRule($NetWorkService)
        Set-Acl $flightingcachepath $Acl

        Write-Log "added $User on $flightingcachepath"
    }
}

function Configure-FlightingCachePath($webConfig, $settings)
{
    Write-Log "Start processing Flighting Cache Path."

    #add missing elements to the web.config
    $add = $webConfig.configuration.appSettings.add | where key -eq 'DataAccess.FlightingCachePath' | select -First 1
    if ($add -eq $null)
    {
        $webRootDirectory = ($webConfig.configuration.appSettings.add | Where {$_.Key -eq 'Infrastructure.WebRoot'}).Value
        if ([System.String]::IsNullOrWhitespace($webRootDirectory))
        {
            Write-Log "Could not find web root!!!!"
        }
        else
        {
            $flightingcachefolder = $settings.'DataAccess.FlightingServiceCacheFolder'
            $flightingcachepath = Join-Path $webRootDirectory $flightingcachefolder
            $add = $webConfig.CreateElement("add")
            $add.SetAttribute("key", "DataAccess.FlightingCachePath")
            $add.SetAttribute("value", $flightingcachepath)
            $webConfig.configuration.appSettings.AppendChild($add) | Out-Null
        }
    }
}

function Update-WifServicesConfig($settings)
{
    $siteAdmin = $settings.'Provisioning.AdminPrincipalName'

    Write-Log "Reading wif.services.config at $($settings.'Infrastructure.WebRoot')"

    [xml]$wifServicesConfig = Get-Content "$($settings.'Infrastructure.WebRoot')\wif.services.config"
    [Uri]$uri = New-Object Uri $($settings.'Aad.AADLoginWsfedEndpointFormat' -f $($siteAdmin.Split('@')[1]))

    $wifServicesConfig.'system.identityModel.services'.federationConfiguration.wsFederation.issuer = "$($uri.AbsoluteUri)"

    $HostUri = New-Object Uri $settings.'Infrastructure.HostUrl'
    $wifServicesConfig.'system.identityModel.services'.federationConfiguration.cookieHandler.domain = $HostUri.DnsSafeHost

    $wifServicesConfig.Save("$($settings.'Infrastructure.WebRoot')\wif.services.config")
}

function Update-AADTenantInfo($settings, $web)
{
    Write-Log "Reading AAD metadata"

    $aadMetadataLocation = $($settings.'Aad.AADMetadataLocationFormat' -f $($settings.'Provisioning.AdminPrincipalName'.Split('@')[1]))
    Write-Log "AAD metadata location [$aadMetadataLocation]"

    $tempAadMetadataFile = [System.IO.Path]::GetTempFileName()
    Write-Log "Metadata will be stored in temporary file [$tempAadMetadataFile]"

    Write-Log "Invoking request"
    Invoke-WebRequest $aadMetadataLocation -UseBasicParsing -OutFile $tempAadMetadataFile

    Write-Log "Reading metadata from file at [$tempAadMetadataFile]"
    [xml]$aadMetadata = Get-Content $tempAadMetadataFile

    Write-Log "Removing temporary file at [$tempAadMetadataFile]"
    Remove-Item $tempAadMetadataFile -Force -Confirm:$false

    Write-Log "Extracting tenant guid from metadata"
    $url = New-Object System.Uri($aadMetadata.EntityDescriptor.entityId)

    $add = $web.configuration.appSettings.add | where key -eq 'Aad.TenantDomainGUID' | select -First 1
    $add.value = $url.AbsolutePath.Trim('/')

    Write-Log "Tenant guid updated"
}

function Update-AADTrustedIssuers($web)
{
    Write-Log "Start updating TrustedIssuers settings"

    $aadIssuerNameFormatNode = $web.configuration.appSettings.add | where key -eq 'Aad.AADIssuerNameFormat' | select -First 1
    $ppe = $aadIssuerNameFormatNode.value -like 'https://login.microsoftonline.com/{0}/'

    $aadName = if($ppe) { 'PPE' } else { 'PROD' }
    Write-Log "TrustedIssuers settings will be applied for $aadName AAD"

    $trustedIssuersNode = $web.configuration.appSettings.add | where key -eq 'Aad.TrustedIssuers' | select -First 1
    Write-Log "Current TrustedIssuers settings = '$($trustedIssuersNode.value)'"

    $trustedIssuers = @($trustedIssuersNode.value -split ';' | ? { $_ })

    $requiredTrustedIssuers = 'https://sts.windows.net/72f988bf-86f1-41af-91ab-2d7cd011db47/', # MSFT tenant
                              'https://sts.windows.net/975f013f-7f24-47e8-a7d3-abc4752bf346/'  # PME tenant

    if($ppe) {
        # For dev/test/tie environments Retail test tenant should be a trusted issuer
        $requiredTrustedIssuers += 'https://sts.windows.net/34b54b07-9b04-4369-839c-380f28f3e0fe/' # Dynamics 365 - Retail - Test (dynamics365RetailTest.onmicrosoft.com)
    } else {
        $requiredTrustedIssuers += 'https://sts.windows.net/57206206-ec82-4579-9724-0a098ed1b99f/' # Dynamics 365 - Retail (dynamics365Retail.onmicrosoft.com)
    }

    foreach($issuer in $requiredTrustedIssuers) {
        if($issuer -notin $trustedIssuers) {
            $trustedIssuers += $issuer
        }
    }

    $updatedTrustedIssuers = $trustedIssuers -join ';'
    Write-Log "Storing updated TrustedIssuers = $updatedTrustedIssuers"

    if(-not $trustedIssuersNode) {
        $trustedIssuersNode = $web.CreateElement("add")
        $trustedIssuersNode.SetAttribute('key', 'Aad.TrustedIssuers')
        $trustedIssuersNode.SetAttribute('value', $updatedTrustedIssuers)
        $web.configuration.appSettings.AppendChild($trustedIssuersNode) | Out-Null
    }
    else {
        $trustedIssuersNode.value = $updatedTrustedIssuers
    }

    $trustedIssuersMetadataAddress = 'https://login.windows.net/common/FederationMetadata/2007-06/FederationMetadata.xml'
    Write-Log "Storing $trustedIssuersMetadataAddress address"

    $trustedIssuersMetadataAddressNode = $web.configuration.appSettings.add | where key -eq 'Aad.TrustedIssuersMetadataAddress' | select -First 1
    if(-not $trustedIssuersMetadataAddressNode) {
        $trustedIssuersMetadataAddressNode = $web.CreateElement("add")
        $trustedIssuersMetadataAddressNode.SetAttribute('key', 'Aad.TrustedIssuersMetadataAddress')
        $trustedIssuersMetadataAddressNode.SetAttribute('value', $trustedIssuersMetadataAddress)
        $web.configuration.appSettings.AppendChild($trustedIssuersMetadataAddressNode) | Out-Null
    }
    else {
        $trustedIssuersMetadataAddressNode.value = $trustedIssuersMetadataAddress
    }
}

function Update-DataAccessFlightingInfo($settings, $web)
{
    Write-Log "Reading DataAccess Flighting metadata"

    $flightingEnvironment = $settings.'DataAccess.FlightingEnvironment'
    Write-Log "flighting Environment [$flightingEnvironment]"

    $flightingCertificateThumbprint = $settings.'DataAccess.FlightingCertificateThumbprint'
    Write-Log "flighting Certificate Thumbprint [$flightingCertificateThumbprint]"

    $flightingServiceCatalogID = $settings.'DataAccess.FlightingServiceCatalogID'
    Write-Log "flighting ServiceCatalogID [$flightingServiceCatalogID]"

    $add = $web.configuration.appSettings.add | where key -eq 'DataAccess.FlightingEnvironment' | select -First 1
    $add.value = $flightingEnvironment

    $add = $web.configuration.appSettings.add | where key -eq 'DataAccess.FlightingCertificateThumbprint' | select -First 1
    $add.value = $flightingCertificateThumbprint

    $add = $web.configuration.appSettings.add | where key -eq 'DataAccess.FlightingServiceCatalogID' | select -First 1
    $add.value = $flightingServiceCatalogID

    Write-Log "DataAccess Flighting Infomation updated"
}

function Update-WifConfigWithPerfSdkCustomTokenHandler($settings, $wif)
{
    Write-Log "Checking if wif.config needs to be updated with PerfSDK custom token handler"

    $updateWifConfigForPerfSdkFlagString = $settings.'WIF.UpdateWifConfigForPerfSdk'
    $updateWifConfigForPerfSdkFlagBool = $null

    if(![string]::IsNullOrEmpty($updateWifConfigForPerfSdkFlagString))
    {
        if([bool]::TryParse($updateWifConfigForPerfSdkFlagString, [ref]$updateWifConfigForPerfSdkFlagBool))
        {
            if($updateWifConfigForPerfSdkFlagBool)
            {
                Write-Log "Adding custom token handler"

                $securityTokenHandlerConfiguration = $wif.SelectSingleNode("system.identityModel/identityConfiguration/securityTokenHandlers")

                $removeNode = $wif.CreateElement("remove")
                $removeNode.SetAttribute("type","System.IdentityModel.Tokens.Saml2SecurityTokenHandler, System.IdentityModel, Version=4.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089")

                $addNode = $wif.CreateElement("add")
                $addNode.SetAttribute("type","MS.Dynamics.TestTools.CloudCommonTestUtilities.Authentication.PerfSdkSaml2TokenHandler, MS.Dynamics.TestTools.CloudCommonTestUtilities")

                $securityTokenHandlerConfiguration.AppendChild($removeNode)
                $securityTokenHandlerConfiguration.AppendChild($addNode)
            }
            else
            {
                Write-Log "WIF.UpdateWifConfigForPerfSdk is false. Custom token handler is not added"
            }
        }
        else
        {
            Write-Log "WIF.UpdateWifConfigForPerfSdk cannot be parsed as a boolean value. Custom token handler is not added"
        }
    }
    else
    {
        Write-Log "WIF.UpdateWifConfigForPerfSdk is null or empty. Custom token handler is not added"
    }
}

function Update-WebConfig($settings)
{
    Write-Log "Reading web.config at $($settings.'Infrastructure.WebRoot')"

    [xml]$web = Get-Content "$($settings.'Infrastructure.WebRoot')\web.config"

    $uri = New-Object Uri $settings.'Infrastructure.HostUrl'

    $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.FullyQualifiedDomainName' | select -First 1
    $add.value = $uri.DnsSafeHost

    $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.HostedServiceName' | select -First 1
    $add.value = $uri.DnsSafeHost.Split('.')[0]

    $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.HostName' | select -First 1
    $add.value = $uri.DnsSafeHost

    $siteAdmin = $settings.'Provisioning.AdminPrincipalName'
    $add = $web.configuration.appSettings.add | where key -eq 'Aad.AADTenantId' | select -First 1
    $add.value = $siteAdmin.Split('@')[1]

    Update-AADTenantInfo $settings $web

    Update-DataAccessFlightingInfo $settings $web

    Update-AADTrustedIssuers $web

    $ClickOnceUnpackDir = $settings.'Infrastructure.ClickonceAppsDirectory'

    Write-Log "Click-once package dir $ClickOnceUnpackDir"

    if ((Test-Path $ClickOnceUnpackDir))
    {
        $ClickOnceTargetDir = Join-Path $settings.'Infrastructure.WebRoot' "apps"
        Write-Log "Click-once package dir $ClickOnceUnpackDir exists, copying click-one packages to webroot $ClickOnceTargetDir"

        if (!(Test-Path $ClickOnceTargetDir))
        {
            New-Item $ClickOnceTargetDir -ItemType Directory -Force
        }

        Copy-Item "$ClickOnceUnpackDir\*" $ClickOnceTargetDir -Recurse -Confirm:$false -Force

        $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.ClickonceAppsDirectory' | select -First 1
        $add.value = "$(split-path $ClickOnceTargetDir -Leaf)"

        Write-Log "Providing read access to the configuration files in APPS folder under web root: required for clickonce packages to work"
        Set-WebConfigurationProperty -pspath "MACHINE/WEBROOT/APPHOST/$($settings.'Infrastructure.ApplicationName')/Apps" -filter "system.webServer/security/requestFiltering/fileExtensions/add[@fileExtension='.config']" -name "allowed" -value "True"
    }

    Configure-ClaimIssuers $web $settings

    Configure-MachineKeys $web $settings

    CreateFlightingCacheFolder $settings
    Configure-FlightingCachePath $web $settings

    $web.Save("$($settings.'Infrastructure.WebRoot')\web.config")
}

function Update-WifConfig($settings)
{
    $siteAdmin = $settings.'Provisioning.AdminPrincipalName'

    Write-Log "Reading wif.config at $($settings.'Infrastructure.WebRoot')"

    $wifConfig = (Get-Content "$($settings.'Infrastructure.WebRoot')\wif.config").Replace('TENANT_ID_PLACEHOLDER',"$($siteAdmin.Split('@')[1])")
    $wifConfig = $wifConfig.Replace('https://sts.windows.net/',$settings.'Provisioning.AdminIdentityProvider')

    Add-Type -AssemblyName "System.Xml"

    Write-Log "Adding dynamic trusted certificates to wif.config"
    [xml]$wif = Get-Content "$($settings.'Infrastructure.WebRoot')\wif.config"

    $issuerNameRegistry = $wif.SelectSingleNode("system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry")
    $authorities = $issuerNameRegistry.SelectNodes("authority")
    $TrustedCerts = $($settings.'Infrastructure.TrustedCertificates').Split(';')

    $TrustedIssuer = $null
    $TokenIssuer = $authorities | where { $_.name -eq $($settings.'Aad.AcsTokenIssuer') }
    $TokenIssuerKeys = $null

    if ($TokenIssuer -ne $null)
    {
        $TokenIssuerKeys = $TokenIssuer.SelectSingleNode("keys")
    }

    foreach($TrustedCert in $TrustedCerts)
    {
        foreach ($auth in $authorities)
        {
            $TrustedIssuer = $auth.keys.add | where thumbprint -eq $TrustedCert

            if ($TrustedIssuer -ne $null)
            {
                break
            }
        }

        if ($TrustedIssuer -eq $null)
        {
            if ($TokenIssuer -eq $null)
            {
                $TokenIssuer = $wif.CreateElement("authority")
                $TokenIssuer.SetAttribute("name",$($settings.'Aad.AcsTokenIssuer'))

                $TokenIssuerKeys = $wif.CreateElement("keys")
                $TokenIssuer.AppendChild($TokenIssuerKeys)

                $ValidIssuers = $wif.CreateElement("validIssuers")

                $newAdd = $wif.CreateElement("add")
                $newAdd.SetAttribute("name", $($settings.'Aad.AcsTokenIssuer'))
                $ValidIssuers.AppendChild($newAdd)

                $TokenIssuer.AppendChild($ValidIssuers)

                $issuerNameRegistry.AppendChild($TokenIssuer) | Out-Null
            }

            $newAdd = $wif.CreateElement("add")
            $newAdd.SetAttribute("thumbprint", $TrustedCert)
            $TokenIssuerKeys.AppendChild($newAdd)
        }
    }

    Write-Log "Removing duplicate authorities in wif.config"
    #we only dedup based on the authority name since a static authority only has one issuer/thumbprint

    $authorities = $issuerNameRegistry.SelectNodes("authority")
    $uniqueAuthorities  = New-Object 'System.Collections.Generic.HashSet[System.Xml.XmlElement]'
    foreach($auth in $authorities)
    {
        $existingAuth = $uniqueAuthorities | Where-Object {$_.name -eq $auth.name}
        if ($existingAuth -eq $null)
        {
            $newAuth = $wif.CreateElement("authority")
            $newAuth.SetAttribute("name",$auth.name)
            $newAuth.InnerXml = $auth.InnerXml
            $uniqueAuthorities.Add($newAuth) | Out-Null
        }
    }

    $type = $issuerNameRegistry.type
    $issuerNameRegistry.RemoveAll()
    $issuerNameRegistry.SetAttribute("type", $type)

    foreach($auth in $uniqueAuthorities)
    {
        $issuerNameRegistry.AppendChild($auth) | Out-Null
    }

    Update-WifConfigWithPerfSdkCustomTokenHandler $settings $wif

    $wif.Save("$($settings.'Infrastructure.WebRoot')\wif.config")
}

Initialize-Log $log
Write-Log "Decoding settings"
$settings = Decode-Settings $config

Write-Log "Updating AOS web.config"
Update-WebConfig $settings
Write-Log "AOS web.config update complete"

Write-Log "Updating AOS wif.config"
Update-WifConfig $settings
Write-Log "AOS wif.config update complete"

Write-Log "Updating AOS web.services.config"
Update-WifServicesConfig $settings
Write-Log "AOS web.services.config update complete"

#Temporary removing this:
#if these are present DBSync assumes we are in local onebox mode and behaves differently
#Set-RegistryValues $settings

Write-Log "Restarting IIS"
iisreset

Write-Log "Finished, exiting"

# SIG # Begin signature block
# MIIoKQYJKoZIhvcNAQcCoIIoGjCCKBYCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBVcsh5QkLsHju0
# KYENHbZqmC07OnCEbzxK/ANj/BkYJ6CCDXYwggX0MIID3KADAgECAhMzAAAEhV6Z
# 7A5ZL83XAAAAAASFMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjUwNjE5MTgyMTM3WhcNMjYwNjE3MTgyMTM3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDASkh1cpvuUqfbqxele7LCSHEamVNBfFE4uY1FkGsAdUF/vnjpE1dnAD9vMOqy
# 5ZO49ILhP4jiP/P2Pn9ao+5TDtKmcQ+pZdzbG7t43yRXJC3nXvTGQroodPi9USQi
# 9rI+0gwuXRKBII7L+k3kMkKLmFrsWUjzgXVCLYa6ZH7BCALAcJWZTwWPoiT4HpqQ
# hJcYLB7pfetAVCeBEVZD8itKQ6QA5/LQR+9X6dlSj4Vxta4JnpxvgSrkjXCz+tlJ
# 67ABZ551lw23RWU1uyfgCfEFhBfiyPR2WSjskPl9ap6qrf8fNQ1sGYun2p4JdXxe
# UAKf1hVa/3TQXjvPTiRXCnJPAgMBAAGjggFzMIIBbzAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUuCZyGiCuLYE0aU7j5TFqY05kko0w
# RQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEW
# MBQGA1UEBRMNMjMwMDEyKzUwNTM1OTAfBgNVHSMEGDAWgBRIbmTlUAXTgqoXNzci
# tW2oynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NybC9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3JsMGEG
# CCsGAQUFBwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDovL3d3dy5taWNyb3NvZnQu
# Y29tL3BraW9wcy9jZXJ0cy9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3J0
# MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQELBQADggIBACjmqAp2Ci4sTHZci+qk
# tEAKsFk5HNVGKyWR2rFGXsd7cggZ04H5U4SV0fAL6fOE9dLvt4I7HBHLhpGdE5Uj
# Ly4NxLTG2bDAkeAVmxmd2uKWVGKym1aarDxXfv3GCN4mRX+Pn4c+py3S/6Kkt5eS
# DAIIsrzKw3Kh2SW1hCwXX/k1v4b+NH1Fjl+i/xPJspXCFuZB4aC5FLT5fgbRKqns
# WeAdn8DsrYQhT3QXLt6Nv3/dMzv7G/Cdpbdcoul8FYl+t3dmXM+SIClC3l2ae0wO
# lNrQ42yQEycuPU5OoqLT85jsZ7+4CaScfFINlO7l7Y7r/xauqHbSPQ1r3oIC+e71
# 5s2G3ClZa3y99aYx2lnXYe1srcrIx8NAXTViiypXVn9ZGmEkfNcfDiqGQwkml5z9
# nm3pWiBZ69adaBBbAFEjyJG4y0a76bel/4sDCVvaZzLM3TFbxVO9BQrjZRtbJZbk
# C3XArpLqZSfx53SuYdddxPX8pvcqFuEu8wcUeD05t9xNbJ4TtdAECJlEi0vvBxlm
# M5tzFXy2qZeqPMXHSQYqPgZ9jvScZ6NwznFD0+33kbzyhOSz/WuGbAu4cHZG8gKn
# lQVT4uA2Diex9DMs2WHiokNknYlLoUeWXW1QrJLpqO82TLyKTbBM/oZHAdIc0kzo
# STro9b3+vjn2809D0+SOOCVZMIIHejCCBWKgAwIBAgIKYQ6Q0gAAAAAAAzANBgkq
# hkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
# EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
# IDIwMTEwHhcNMTEwNzA4MjA1OTA5WhcNMjYwNzA4MjEwOTA5WjB+MQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQg
# Q29kZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
# CgKCAgEAq/D6chAcLq3YbqqCEE00uvK2WCGfQhsqa+laUKq4BjgaBEm6f8MMHt03
# a8YS2AvwOMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc6Whe0t+bU7IKLMOv2akr
# rnoJr9eWWcpgGgXpZnboMlImEi/nqwhQz7NEt13YxC4Ddato88tt8zpcoRb0Rrrg
# OGSsbmQ1eKagYw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+lD3v++MrWhAfTVYoonpy
# 4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+EGvKhL1nkkDstrjNYxbc+/jLTswM9
# sbKvkjh+0p2ALPVOVpEhNSXDOW5kf1O6nA+tGSOEy/S6A4aN91/w0FK/jJSHvMAh
# dCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rhKEmdX4jiJV3TIUs+UsS1Vz8k
# A/DRelsv1SPjcF0PUUZ3s/gA4bysAoJf28AVs70b1FVL5zmhD+kjSbwYuER8ReTB
# w3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zdsGbiwZeBe+3W7UvnSSmn
# Eyimp31ngOaKYnhfsi+E11ecXL93KCjx7W3DKI8sj0A3T8HhhUSJxAlMxdSlQy90
# lfdu+HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS4NaIjAsCAwEAAaOCAe0w
# ggHpMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBRIbmTlUAXTgqoXNzcitW2o
# ynUClTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYD
# VR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBRyLToCMZBDuRQFTuHqp8cx0SOJNDBa
# BgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
# bC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3JsMF4GCCsG
# AQUFBwEBBFIwUDBOBggrBgEFBQcwAoZCaHR0cDovL3d3dy5taWNyb3NvZnQuY29t
# L3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3J0MIGfBgNV
# HSAEgZcwgZQwgZEGCSsGAQQBgjcuAzCBgzA/BggrBgEFBQcCARYzaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1hcnljcHMuaHRtMEAGCCsG
# AQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAHAAbwBsAGkAYwB5AF8AcwB0AGEAdABl
# AG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn8oalmOBUeRou09h0ZyKb
# C5YR4WOSmUKWfdJ5DJDBZV8uLD74w3LRbYP+vj/oCso7v0epo/Np22O/IjWll11l
# hJB9i0ZQVdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0bpdS1HXeUOeLpZMlEPXh6
# I/MTfaaQdION9MsmAkYqwooQu6SpBQyb7Wj6aC6VoCo/KmtYSWMfCWluWpiW5IP0
# wI/zRive/DvQvTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvyCInWH8MyGOLwxS3OW560
# STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn+N4sOiBpmLJZiWhub6e3dMNABQam
# ASooPoI/E01mC8CzTfXhj38cbxV9Rad25UAqZaPDXVJihsMdYzaXht/a8/jyFqGa
# J+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw3MYbBL7fQccOKO7eZS/sl/ah
# XJbYANahRr1Z85elCUtIEJmAH9AAKcWxm6U/RXceNcbSoqKfenoi+kiVH6v7RyOA
# 9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sLgOppO6/8MO0ETI7f33Vt
# Y5E90Z1WTk+/gFcioXgRMiF670EKsT/7qMykXcGhiJtXcVZOSEXAQsmbdlsKgEhr
# /Xmfwb1tbWrJUnMTDXpQzTGCGgkwghoFAgEBMIGVMH4xCzAJBgNVBAYTAlVTMRMw
# EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVN
# aWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNp
# Z25pbmcgUENBIDIwMTECEzMAAASFXpnsDlkvzdcAAAAABIUwDQYJYIZIAWUDBAIB
# BQCgga4wGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIMLYm0pglnoOe99RdZz3RM87
# CME2BG4jRKCvg1psoGogMEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8A
# cwBvAGYAdKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEB
# BQAEggEAHdXOPYEVvhOPgLvg3Gbve85rqcQRgiY8iU7j8a8K+cBslP7KhOKiEhPk
# VIaQ5cEriKmr1GGeCeHtb4hlVWD/ourTnor/QJjgZeUs4kdVIPXGp8mQ5XfkyEDK
# 1CCvQAirp2IUNNTA6QkRqUrPq4tyE6wg9W5X3YnStncJFY5sW1vqeYeN6b8ZsrSa
# El9efqDxjn2b8+KYGn2AH2gdg6bkEZcLrZmBI5FrIQZGq7h1QAjP98D5EE7yLfIk
# Alu/PXGJw1iM5gC5Bs7kL3ZZy/YYLkeNO0Q4CHngE3+jCMJMeQXKNd9gCmASCE10
# P3VcaXN4DySikArXJLAdqaaOn7BtSqGCF5MwghePBgorBgEEAYI3AwMBMYIXfzCC
# F3sGCSqGSIb3DQEHAqCCF2wwghdoAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFRBgsq
# hkiG9w0BCRABBKCCAUAEggE8MIIBOAIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFl
# AwQCAQUABCBUyCTbCHN3yuJKPdBBeqgXNkyMhDMOKUI9ISfWZoFQrQIGaKOig5wt
# GBIyMDI1MDkwNDA3MjYyNy4xOFowBIACAfSggdGkgc4wgcsxCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJTAjBgNVBAsTHE1pY3Jvc29mdCBBbWVy
# aWNhIE9wZXJhdGlvbnMxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjozNzAzLTA1
# RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZaCC
# EeowggcgMIIFCKADAgECAhMzAAACCkeyFNnusrMPAAEAAAIKMA0GCSqGSIb3DQEB
# CwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
# EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNV
# BAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMB4XDTI1MDEzMDE5NDI1
# N1oXDTI2MDQyMjE5NDI1N1owgcsxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJTAjBgNVBAsTHE1pY3Jvc29mdCBBbWVyaWNhIE9wZXJhdGlvbnMx
# JzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjozNzAzLTA1RTAtRDk0NzElMCMGA1UE
# AxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCAiIwDQYJKoZIhvcNAQEB
# BQADggIPADCCAgoCggIBALLs3PASlvsGnACT0snTRerfRNxONzA1nzRPEC0BuFJo
# naTCckDUC78Oj398QGMAe/oya23d+Y8N4gmdDtUF4irxFx+NyOts8iDfADm/kxB+
# U81IE069xdE/59mcDLQQsPN+ckecKRk2xBRnsYXsvFQtMo5hjZgnDhOuZweiGRjo
# OMJnLGGqYZDDB1uOg9ZFti7+jMV6b/J8k/KNUGqXXTrxtxWHnwDxzkIPpNY38ve7
# 43L7s7z4O96vqoFPjgTLul89kxnUeLvK8Iu/ksbNIHqjY4PkYhnLvPiSHxRgd3OO
# f1bH5BnbbfdIlAq1mGkju4d/egxODTNqdB/PuaC515+YYGtASDWjHef7dOLOHQP3
# NWY1A/2wWOw9C00Zn0gP0fwD6QUifHiatLEDZLIYw5BzGUYzfSS0gozeKoK4URT0
# IdUyk33j/V+XhPo+icn7+hYmoWVq1i4wq/Zny6KmucVwDEKk6wMhUE70rW8d4AyJ
# BBSVwT0IPnfojVgueY7aodqA8+ZeF04asdScJS2inbV6W6QeHvmr/nMf46c16Snm
# 52mNA1kK+JgBl0ewueRTQ19QCvqvVSoNxKrXJ/lGLCKYHxKOx2YeWXiA+OapWLT+
# uaczWgARCbc/JZxNBCJtguK4o3tjiKjlslNXCb69FFuknlQv8PfcL//uqcdt6hYF
# AgMBAAGjggFJMIIBRTAdBgNVHQ4EFgQUAq8XQQSPgDI99jxb+quwC9+1nCQwHwYD
# VR0jBBgwFoAUn6cVXQBeYl2D9OXSZacbUzUZ6XIwXwYDVR0fBFgwVjBUoFKgUIZO
# aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWljcm9zb2Z0JTIw
# VGltZS1TdGFtcCUyMFBDQSUyMDIwMTAoMSkuY3JsMGwGCCsGAQUFBwEBBGAwXjBc
# BggrBgEFBQcwAoZQaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0
# cy9NaWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcnQwDAYD
# VR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAOBgNVHQ8BAf8EBAMC
# B4AwDQYJKoZIhvcNAQELBQADggIBALCTc9bQLKM5qpcIw66xN17t6nYLA//FXUne
# j0oCIpJ2rxrZRHlEIdiGcmx56bB4xQfaQ15+mcCHYGfwfaikPO0MmSPw5hF38pDK
# wTXY3Bpjco7ABbxGDw51e/l9Rkiw5aCmuQJAzRWgUd1dgTGQk3cTMFtQJTDfJviA
# AU8rnpP7pw+kjwwkskJBcc0yC2LZBmFf0qR9EB98VgwVymxsj6jOILkGuuTffrVw
# kUeGAQlHQnjaeplMZKBSkMYdJkBEao1wBXmH45AMt4XandRHH8/oxxSDWcnaAw9g
# GwP5vB30HRz9chn0y3e6OUQs+mSKOMJ1JnGHO7YzdJlYmvnu5j4UL4uym8g4fU6m
# h9qeHckYOiw1wAS73JQxYqwQEyeAPpVZBJBhnpf0XzTMKh5nBOrSL0C6OdjPPHlb
# Z8SBl6NG7orUYIBKbO02VgmUAKOpp9ZGh9OqQmFX8na/3tfeu4O9+m465ChS1UDB
# esOwbY04G69Wyjkp1bniEFKzP/j45EHiouerj8Y21qNQeispEocY6AWjwMppcb5Q
# 0A3CEY3EdsgtJrn0/ETEyAKMFE/fzbzIYqyq5ITPMs1afPlp/1mZ4fjzT1/Ai20j
# jUmS6Jj1fqGYyYti/w5jfi+84r7PLGRL70FQtUA/qC37JodoGWuWfwaLj90GvbLp
# Kuv/nqDQMIIHcTCCBVmgAwIBAgITMwAAABXF52ueAptJmQAAAAAAFTANBgkqhkiG
# 9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAO
# BgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEy
# MDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIw
# MTAwHhcNMjEwOTMwMTgyMjI1WhcNMzAwOTMwMTgzMjI1WjB8MQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGlt
# ZS1TdGFtcCBQQ0EgMjAxMDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIB
# AOThpkzntHIhC3miy9ckeb0O1YLT/e6cBwfSqWxOdcjKNVf2AX9sSuDivbk+F2Az
# /1xPx2b3lVNxWuJ+Slr+uDZnhUYjDLWNE893MsAQGOhgfWpSg0S3po5GawcU88V2
# 9YZQ3MFEyHFcUTE3oAo4bo3t1w/YJlN8OWECesSq/XJprx2rrPY2vjUmZNqYO7oa
# ezOtgFt+jBAcnVL+tuhiJdxqD89d9P6OU8/W7IVWTe/dvI2k45GPsjksUZzpcGkN
# yjYtcI4xyDUoveO0hyTD4MmPfrVUj9z6BVWYbWg7mka97aSueik3rMvrg0XnRm7K
# MtXAhjBcTyziYrLNueKNiOSWrAFKu75xqRdbZ2De+JKRHh09/SDPc31BmkZ1zcRf
# NN0Sidb9pSB9fvzZnkXftnIv231fgLrbqn427DZM9ituqBJR6L8FA6PRc6ZNN3SU
# HDSCD/AQ8rdHGO2n6Jl8P0zbr17C89XYcz1DTsEzOUyOArxCaC4Q6oRRRuLRvWoY
# WmEBc8pnol7XKHYC4jMYctenIPDC+hIK12NvDMk2ZItboKaDIV1fMHSRlJTYuVD5
# C4lh8zYGNRiER9vcG9H9stQcxWv2XFJRXRLbJbqvUAV6bMURHXLvjflSxIUXk8A8
# FdsaN8cIFRg/eKtFtvUeh17aj54WcmnGrnu3tz5q4i6tAgMBAAGjggHdMIIB2TAS
# BgkrBgEEAYI3FQEEBQIDAQABMCMGCSsGAQQBgjcVAgQWBBQqp1L+ZMSavoKRPEY1
# Kc8Q/y8E7jAdBgNVHQ4EFgQUn6cVXQBeYl2D9OXSZacbUzUZ6XIwXAYDVR0gBFUw
# UzBRBgwrBgEEAYI3TIN9AQEwQTA/BggrBgEFBQcCARYzaHR0cDovL3d3dy5taWNy
# b3NvZnQuY29tL3BraW9wcy9Eb2NzL1JlcG9zaXRvcnkuaHRtMBMGA1UdJQQMMAoG
# CCsGAQUFBwMIMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1UdDwQEAwIB
# hjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFNX2VsuP6KJcYmjRPZSQW9fO
# mhjEMFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9w
# a2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNybDBaBggr
# BgEFBQcBAQROMEwwSgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cubWljcm9zb2Z0LmNv
# bS9wa2kvY2VydHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3J0MA0GCSqGSIb3
# DQEBCwUAA4ICAQCdVX38Kq3hLB9nATEkW+Geckv8qW/qXBS2Pk5HZHixBpOXPTEz
# tTnXwnE2P9pkbHzQdTltuw8x5MKP+2zRoZQYIu7pZmc6U03dmLq2HnjYNi6cqYJW
# AAOwBb6J6Gngugnue99qb74py27YP0h1AdkY3m2CDPVtI1TkeFN1JFe53Z/zjj3G
# 82jfZfakVqr3lbYoVSfQJL1AoL8ZthISEV09J+BAljis9/kpicO8F7BUhUKz/Aye
# ixmJ5/ALaoHCgRlCGVJ1ijbCHcNhcy4sa3tuPywJeBTpkbKpW99Jo3QMvOyRgNI9
# 5ko+ZjtPu4b6MhrZlvSP9pEB9s7GdP32THJvEKt1MMU0sHrYUP4KWN1APMdUbZ1j
# dEgssU5HLcEUBHG/ZPkkvnNtyo4JvbMBV0lUZNlz138eW0QBjloZkWsNn6Qo3GcZ
# KCS6OEuabvshVGtqRRFHqfG3rsjoiV5PndLQTHa1V1QJsWkBRH58oWFsc/4Ku+xB
# Zj1p/cvBQUl+fpO+y/g75LcVv7TOPqUxUYS8vwLBgqJ7Fx0ViY1w/ue10CgaiQuP
# Ntq6TPmb/wrpNPgkNWcr4A245oyZ1uEi6vAnQj0llOZ0dFtq0Z4+7X6gMTN9vMvp
# e784cETRkPHIqzqKOghif9lwY1NNje6CbaUFEMFxBmoQtB1VM1izoXBm8qGCA00w
# ggI1AgEBMIH5oYHRpIHOMIHLMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
# Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
# cmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1lcmljYSBPcGVyYXRpb25zMScw
# JQYDVQQLEx5uU2hpZWxkIFRTUyBFU046MzcwMy0wNUUwLUQ5NDcxJTAjBgNVBAMT
# HE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2WiIwoBATAHBgUrDgMCGgMVANEA
# xUmUDpsqr3dWe7dSQmCbkeVhoIGDMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNV
# BAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
# c29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAg
# UENBIDIwMTAwDQYJKoZIhvcNAQELBQACBQDsYzdiMCIYDzIwMjUwOTAzMjE1MzM4
# WhgPMjAyNTA5MDQyMTUzMzhaMHQwOgYKKwYBBAGEWQoEATEsMCowCgIFAOxjN2IC
# AQAwBwIBAAICJBcwBwIBAAICEm0wCgIFAOxkiOICAQAwNgYKKwYBBAGEWQoEAjEo
# MCYwDAYKKwYBBAGEWQoDAqAKMAgCAQACAwehIKEKMAgCAQACAwGGoDANBgkqhkiG
# 9w0BAQsFAAOCAQEAUEI8GD2uW9kO72roaYXdRdfQHvMVQKqANBOI9LdtI4RBWRgS
# LGlxlIBMffQqT+c4plVuCPU3pcwMSyHpVelnbpcv8VelihM2JCqnee5cYSHKCAZd
# CnKiiP18KavGDPO++vXyblVO9ykPi6IlXHUrm5+oTIJLqTZyqMb6ni45MRxrM8mQ
# TO9qc13lu+aJsd8URxbTtSYJvUmyKAkOdJyXpShqAlbJK4eGRZ92Wzk3l02zsA31
# tZAYJ0CxZvoXtHdPDHGXYRPVjK5Fk7Mutb4U0r5hPogQGlaulFglgTLXXxYFefi1
# d82CZ+mxjEHItkXPIhNx44pxmF37itGh7f7HQTGCBA0wggQJAgEBMIGTMHwxCzAJ
# BgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jv
# c29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAACCkeyFNnusrMPAAEAAAIKMA0G
# CWCGSAFlAwQCAQUAoIIBSjAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwLwYJ
# KoZIhvcNAQkEMSIEILuLtlPMy93mQxgA7ebbG6YZlc1ww6a+7/CNGsEhmMdUMIH6
# BgsqhkiG9w0BCRACLzGB6jCB5zCB5DCBvQQgTZrL/LR6kr9fdnTcpNyWioTy70hd
# G107sx7HSHD35JowgZgwgYCkfjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2Fz
# aGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
# cnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAx
# MAITMwAAAgpHshTZ7rKzDwABAAACCjAiBCBzxdAEBBLRN+Jdmg125NTrzCZJWM2d
# j4IgTp9jP9ga4zANBgkqhkiG9w0BAQsFAASCAgAcdlXayT74XDNfpLkBBZU55kEE
# DFzbBFGq7KwAIhUQx9EESEzhjhfnnYWSlDAAZ3+8CoFXw9FJ5UjnVJkfpnYra8JI
# MuEC9ZgWTJgRynC0twoQ/zdBiRcB0Y4JIPgKw52VDG8D12Pj/GQXD+TLmByF2o9E
# ymyYeCZI0soEK3w9AuNiAEl7KXDuixUmdZ37z8sOKftf4AX0IJtJWdzTpWRvIxXT
# sfBWrn428olzYoexMx3Pe82lX5Fx9Yro2jHWeA5X0PZta4JX0VmL+dHmuiG7kJFY
# ZQHY5ZISVxHhw/ddMBj0SCdck3wlcqqZMHrqVza1PW5OAZJdfxz/jy2rVduE1LEb
# B0jjrM18orqM3sOzlkK6vkRUBbeuJxeUkImYzagBTmd0cvV7JiXce0DYOimMNg2P
# Sl9P6puA+LVxJ5PYrnpgDZlPEo5PvSj0G3YL61i8IZ6hYk64EnuivUi/mT/vS0Ql
# bUoYbIYxOy7AfY2c+/t1qqoiQF1rh3qC04HfM3fe0vo0mRbU1Gn7inSdmaIJiRWh
# opZWRkxEOXvTt7nmCfibIq3gSnlcUGm/EtWETSWuyMSwXG++/00yrXnBjBxRSwXw
# 5OIsLTSQ+K3zA8ATcRhfKLtEcR+wDEe02swBLhhFZxVP4jqcYYirgHOipatLrlbW
# YqLX5C2QAc2iTEIQkg==
# SIG # End signature block
