﻿<#
.SYNOPSIS
    Backup the AOS service files.

.DESCRIPTION

    Copyright © 2019 Microsoft. All rights reserved.
#>
[CmdletBinding()]
param
(
    [Parameter(Mandatory = $false, HelpMessage = "The path to the directory in which to write log files.")]
    [string]$LogDir
)

function PrepareDeltaDBsyncReportDeployment([string]$metadataPackagePath)
{
    $MetadataParentPath = Split-Path -Path $metadataPackagePath -Parent
    $deltaSyncFolder = Join-Path -Path $MetadataParentPath -ChildPath "DeltaSync"

    if (!(Test-Path -Path $deltaSyncFolder))
    {
        Write-ServicingLog "Creating delta sync folder." -Vrb
        New-Item -Path $deltaSyncFolder -ItemType Directory -Force | Out-Null
    }
    else
    {
        Write-ServicingLog "Deleting delta sync folder contents." -Vrb
        Get-ChildItem -Path $deltaSyncFolder | Remove-Item -Force -Recurse
    }

    Write-ServicingLog "Copying MD files from metadata package folder to delta sync folder." -Vrb
    $roboCopyLogFile = $null
    if ((-not [string]::IsNullOrWhiteSpace($LogDir)) -and (Test-Path -Path $LogDir))
    {
        $roboCopyLogFile = Join-Path $LogDir "DeltaSyncCopy.log"
        Write-ServicingLog "Copying MD files log: $roboCopyLogFile." -Vrb
    }

    Robocopy.exe $metadataPackagePath $deltaSyncFolder /s /W:5 /R:3 /nfl /ndl *AXSecurity*.md *AXTable*.md *AXView*.md *AXEdt*.md *.rdl *> $roboCopyLogFile
    $roboCopyExitCode = $LASTEXITCODE

    if (($roboCopyExitCode -ge 8) -or ($roboCopyExitCode -lt 0))
    {
        throw "Failed to copy MD files with RoboCopy exit code $roboCopyExitCode."
    }
    Write-ServicingLog "Copying MD files completed with RoboCopy exit code $roboCopyExitCode." -Vrb
}

# ALM Service keeps a "clean" copy of packages folder (i.e. only standard AX vanilla code) to start from on every build
# We want AOS service to backup the "Clean" state packages so rollback doesn't put custom code back
# after removing the ALM service's backup in the AOS update step.
# We can't do this in ALM Service script due to these inter-dependencies between ALM and AOS.
# This is a compromise solution awaiting ALM to do "clean" differently.
function Restore-ALMPackagesBackup
{
    $ALMPackagesCopyPath = Get-ALMPackageCopyPath
    if ($ALMPackagesCopyPath)
    {
        # Copy latest DevALMInstrumentor.ps1 before using it. This ensures we use the fix which prevents locking 
        # Microsoft.Dynamics.ApplicationPlatform.Instrumentation.dll. 
        $NewInstrumentationScript = "$PSScriptRoot\..\..\ALMService\Code\DevALMInstrumentor.ps1"
        if (Test-Path -Path $NewInstrumentationScript  -PathType leaf)
        {
            Write-ServicingLog "Copying latest DevALMInstrumentor.ps1 from $($NewInstrumentationScript) to $($env:DynamicsSDK) before using it." -Vrb
            Copy-Item $NewInstrumentationScript -Destination $env:DynamicsSDK -Force
        }

        $ALMBackupFlag = Join-Path -Path $ALMPackagesCopyPath -ChildPath "Packages\BackupComplete.txt"

        # Check if a completed backup exists.
        if (Test-Path -Path $ALMBackupFlag)
        {
            Write-ServicingLog "Restoring packages from ALM Service backup at $($ALMPackagesCopyPath)..."
            Write-ServicingLog "Calling $($ALMPackagesCopyRestore) script to restore packages..." -Vrb
            $ALMPackagesCopyRestore = Join-Path -Path $env:DynamicsSDK -ChildPath "PrepareForBuild.ps1"
            & $ALMPackagesCopyRestore
            Write-ServicingLog "Call to $($ALMPackagesCopyRestore) completed." -Vrb

            # PrepareForBuild script shuts down IIS service entirely, even though individual sites are already stopped.
            # This can break subsequent IIS web administration calls during servicing.
            # Calling Get-AosWebSitePhysicalPath will restart the service if needed
            # and leave the websites in the state they were in prior to IIS service stop
            $ALMTestWebRoot = Get-AosWebSitePhysicalPath
            Write-ServicingLog "Call to Get-AosWebSitePhysicalPath returned: $($ALMTestWebRoot)" -Vrb
        }
        else
        {
            Write-ServicingLog "Warning: No complete ALM Service backup found at $($ALMPackagesCopyPath). Skipping restore."
        }
    }
}

$ErrorActionPreference = "Stop"
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\AosEnvironmentUtilities.psm1" -Force -DisableNameChecking

# Initialize exit code.
[int]$ExitCode = 0

# Initialize the log file to use with Write-ServicingLog.
Set-ServicingLog -LogDir $LogDir -LogFileName "AosService-Backup_$([DateTime]::UtcNow.ToString("yyyyMMddHHmmss")).log"

try
{
    Write-ServicingLog "Starting AOS backup..."

    # $RunbookBackupFolder and $PSScriptRoot variables are populated directly from runbook.
    $BackupFolder = $RunbookBackupFolder

    if ([string]::IsNullOrEmpty($BackupFolder))
    {
        $BackupFolder = Join-Path -Path $PSScriptRoot -ChildPath "ManualAosServiceBackup"
    }

    # Determine if this is a developer/demo machine.
    $DeveloperBox = Get-DevToolsInstalled

    Write-ServicingLog "Backup folder: $($BackupFolder)"

    $webrootBackupFolder = Join-Path -Path $BackupFolder -ChildPath "webroot"
    $packageBackupFolder = Join-Path -Path $BackupFolder -ChildPath "packages"
    $webrootBackupFilePath = Join-Path -Path $webrootBackupFolder -ChildPath "webroot.zip"
    $packageBackupFilePath = Join-Path -Path $packageBackupFolder -ChildPath "packages.zip"
    $webrootBackupSymlinkFolder = Join-Path -Path $webrootBackupFolder -ChildPath "symlink"
    $packageBackupSymlinkFolder = Join-Path -Path $packageBackupFolder -ChildPath "symlink"

    $webroot = Get-AosWebSitePhysicalPath
    $packagePath = Get-AosPackageDirectory

    if (!(Test-Path -Path $webrootBackupFolder))
    {
        Write-ServicingLog "Creating new backup folder: $($webrootBackupFolder)" -Vrb
        New-Item -ItemType Directory -Path $webrootBackupFolder | Out-Null
    }

    if (!(Test-Path -Path $packageBackupFolder))
    {
        Write-ServicingLog "Creating new backup folder: $($packageBackupFolder)" -Vrb
        New-Item -ItemType Directory -Path $packageBackupFolder | Out-Null
    }

    if (!(Test-Path -Path $webrootBackupSymlinkFolder))
    {
        Write-ServicingLog "Creating new backup folder: $($webrootBackupSymlinkFolder)" -Vrb
        New-Item -ItemType Directory -Path $webrootBackupSymlinkFolder | Out-Null
    }

    if (!(Test-Path -Path $packageBackupSymlinkFolder))
    {
        Write-ServicingLog "Creating new backup folder: $($packageBackupSymlinkFolder)" -Vrb
        New-Item -ItemType Directory -Path $packageBackupSymlinkFolder | Out-Null
    }

    # Backup is not needed for dev/build/demo/vhd env as we don't support roll back in those environments.
    # This will save a lot fo disk space and time for those env when applying packages.
    if (!$DeveloperBox)
    {
        try
        {
            Write-ServicingLog "Creating backup of AOS webroot symlinks..."
            Copy-SymbolicLinks -SourcePath $webroot -DestinationPath $webrootBackupSymlinkFolder -Move

            $exclude = ""
            Write-ServicingLog "Creating backup of AOS webroot files..."
            Create-ZipFiles -sourceFolder:$webroot -destFile:$webrootBackupFilePath -filetypesExcluded:$exclude -zipLogDir:$LogDir

            # Restore AOS packages from complete ALM service backup if one is found.
            # Do this before creating backup of AOS package files.
            Restore-ALMPackagesBackup

            $exclude = "*.bak"
            Write-ServicingLog "Creating backup of AOS packages symlinks..."
            Copy-SymbolicLinks -SourcePath $packagePath -DestinationPath $packageBackupSymlinkFolder -Move

            Write-ServicingLog "Creating backup of AOS packages..."
            Create-ZipFiles -sourceFolder:$packagePath -destFile:$packageBackupFilePath -filetypesExcluded:$exclude -zipLogDir:$LogDir
        }
        catch
        {
            throw $_
        }
        finally
        {
            # Restore the symlinks after the backup regardless of error.
            Write-ServicingLog "Restoring AOS packages symlinks..." -Vrb
            Copy-SymbolicLinks -SourcePath $packageBackupSymlinkFolder -DestinationPath $packagePath

            Write-ServicingLog "Restoring AOS webroot symlinks..." -Vrb
            Copy-SymbolicLinks -SourcePath $webrootBackupSymlinkFolder -DestinationPath $webroot
        }
    }
    else
    {
        Write-ServicingLog "Skipping backup of AOS files as developer tools are installed."

        # Restore AOS packages from complete ALM service backup if one is found.
        Restore-ALMPackagesBackup
    }

    Write-ServicingLog "Preparing delta DbSync report deployment..."
    PrepareDeltaDBsyncReportDeployment -metadataPackagePath:$packagePath
}
catch
{
    # Ensure non-zero exit code if an exception is caught and no exit code set.
    if ($ExitCode -eq 0)
    {
        $ExitCode = 1024
    }

    $ErrorMessage = "Error during AOS backup: $($_)"

    Write-ServicingLog $ErrorMessage
    Write-ServicingLog $($_) -Vrb
    Write-ServicingLog "AOS backup script failed with exit code: $($ExitCode)."

    # Use throw to indicate error to AXUpdateInstaller.
    # In case of exceptions, the output is not captured, so only the error message and
    # log file contents will be available for diagnostics.
    throw "$($ErrorMessage) [Log: $(Get-ServicingLog)]"
}

Write-ServicingLog "AOS backup script completed with exit code: $($ExitCode)."
exit $ExitCode
# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAgK9bU5hKrnd3Z
# NU61V7Zupac4qdvaiHdSkbPSvEt4dqCCDYEwggX/MIID56ADAgECAhMzAAABh3IX
# chVZQMcJAAAAAAGHMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAwMzA0MTgzOTQ3WhcNMjEwMzAzMTgzOTQ3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDOt8kLc7P3T7MKIhouYHewMFmnq8Ayu7FOhZCQabVwBp2VS4WyB2Qe4TQBT8aB
# znANDEPjHKNdPT8Xz5cNali6XHefS8i/WXtF0vSsP8NEv6mBHuA2p1fw2wB/F0dH
# sJ3GfZ5c0sPJjklsiYqPw59xJ54kM91IOgiO2OUzjNAljPibjCWfH7UzQ1TPHc4d
# weils8GEIrbBRb7IWwiObL12jWT4Yh71NQgvJ9Fn6+UhD9x2uk3dLj84vwt1NuFQ
# itKJxIV0fVsRNR3abQVOLqpDugbr0SzNL6o8xzOHL5OXiGGwg6ekiXA1/2XXY7yV
# Fc39tledDtZjSjNbex1zzwSXAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUhov4ZyO96axkJdMjpzu2zVXOJcsw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDU4Mzg1MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAixmy
# S6E6vprWD9KFNIB9G5zyMuIjZAOuUJ1EK/Vlg6Fb3ZHXjjUwATKIcXbFuFC6Wr4K
# NrU4DY/sBVqmab5AC/je3bpUpjtxpEyqUqtPc30wEg/rO9vmKmqKoLPT37svc2NV
# BmGNl+85qO4fV/w7Cx7J0Bbqk19KcRNdjt6eKoTnTPHBHlVHQIHZpMxacbFOAkJr
# qAVkYZdz7ikNXTxV+GRb36tC4ByMNxE2DF7vFdvaiZP0CVZ5ByJ2gAhXMdK9+usx
# zVk913qKde1OAuWdv+rndqkAIm8fUlRnr4saSCg7cIbUwCCf116wUJ7EuJDg0vHe
# yhnCeHnBbyH3RZkHEi2ofmfgnFISJZDdMAeVZGVOh20Jp50XBzqokpPzeZ6zc1/g
# yILNyiVgE+RPkjnUQshd1f1PMgn3tns2Cz7bJiVUaqEO3n9qRFgy5JuLae6UweGf
# AeOo3dgLZxikKzYs3hDMaEtJq8IP71cX7QXe6lnMmXU/Hdfz2p897Zd+kU+vZvKI
# 3cwLfuVQgK2RZ2z+Kc3K3dRPz2rXycK5XCuRZmvGab/WbrZiC7wJQapgBodltMI5
# GMdFrBg9IeF7/rP4EqVQXeKtevTlZXjpuNhhjuR+2DMt/dWufjXpiW91bo3aH6Ea
# jOALXmoxgltCp1K7hrS6gmsvj94cLRf50QQ4U8Qwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAYdyF3IVWUDHCQAAAAABhzAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgrmdOsO0I
# TX45XXG90MF9BCxtQ+vutKZN3ujMzlux4+kwQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQBjrDnCQdqOa/p5M6okYAOzFflY9RlUb+3yF/oWofDT
# uB7byguCTQHzPAU8nWO4emMLpqRghJOAh21wthzFnenBqsUaAWHWGlM2UNB9rgSo
# PjFDRYaZOufhA/PglybL1+skblwqBX/jhUX8l5h1tyo0xOihx/oHxqYgtZYrgoFk
# ZWUWnUsHzMQPSHy8BTEKGb03ZmEDGiSD+HHSLH68T+kxX3wro0aTuv0B7zRPI+li
# y3YJaHR/tLKQrzsJpFgv3mcRFyddhECLapX4SnrgIP82E0E/cOvicZzIdxfptMsw
# NyG4m3BtWMcjjrYbbZ3M47UqXHGw+OA80SoSi1ejRqr0oYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIEuNC719KKIy3e0xMYxVi+uBfe/PnBNVwdwWA7VU
# QyTQAgZfs/W5NoAYEzIwMjAxMTIwMTkxMzE5LjQ3NVowBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjpGN0E2LUUyNTEtMTUwQTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABJYvei2xyJjHdAAAA
# AAElMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTE5MTIxOTAxMTQ1OFoXDTIxMDMxNzAxMTQ1OFowgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpGN0E2
# LUUyNTEtMTUwQTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBANB7H2N2YFvs4cnBJiYx
# Sitk3ABy/xXLfpOUm7NxXHsb6UWq3bONY4yVI4ySbVegC4nxVnlKEF50ANcMYMrE
# c1mEu7cRbzHmi38g6TqLMtOUAW28hc6DNez8do4zvZccrKQxkcB0v9+lm0BIzk9q
# Waxdfg6XyVeSb2NHnkrnoLur36ENT7a2MYdoTVlaVpuU1RcGFpmC0IkJ3rRTJm+A
# jv+43Nxp+PT9XDZtqK32cMBV3bjK39cJmcdjfJftmweMi4emyX4+kNdqLUPB72nS
# vIJmyX1I4wd7G0gd72qVNu1Zgnxa1Yugf10QxDFUueY88M5WYGPstmFKOLfw31Wn
# P8UCAwEAAaOCARswggEXMB0GA1UdDgQWBBTzqsrlByb5ATk0FcYI8iIIF0Mk+DAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQCTHFk8YSAiACGypk1NmTnxXW9CInmN
# rbEeXlOoYDofCPlKKguDcVIuJOYZX4G0WWlhS2Sd4HiOtmy42ky19tMx0bun/EDI
# hW3C9edNeoqUIPVP0tyv3ilV53McYnMvVNg1DJkkGi4J/OSCTNxw64U595Y9+cxO
# IjlQFbk52ajIc9BYNIYehuhbV1Mqpd4m25UNNhsdMqzjON8IEwWObKVG7nZmmLP7
# 0wF5GPiIB6i7QX/fG8jN6mggqBRYJn2aZWJYSRXAK1MZtXx4rvcp4QTS18xT9hjZ
# SagY9zxjBu6sMR96V6Atb5geR+twYAaV+0Kaq0504t6CEugbRRvH8HuxMIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpG
# N0E2LUUyNTEtMTUwQTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUARdMv4VBtzYb7cxde8hEpWvahcKeggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AONiaGwwIhgPMjAyMDExMjAyMDA4MTJaGA8yMDIwMTEyMTIwMDgxMlowdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA42JobAIBADAKAgEAAgIgsQIB/zAHAgEAAgIRnTAK
# AgUA42O57AIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBADjQibTspfGjwa8E
# AKPXk5cqIsvM1UeXYN+l20Rv3eQETDQ5lJrEGbpnuiUxhuSbb9QSfX1lxpYfqxeD
# I36YDcVKe2vXp+QNHLyyjwb5k9JqDcUi++VS/UVhrqOQtRzC3b4HPdWwfRRlSrxs
# 6eGQ/rSxXd8a/jfNJUhSLmOyY8UTMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAEli96LbHImMd0AAAAAASUwDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQgZTFHx/TpalUB3Bz/Y4CeOltmpvIDZ7FHLYKlm77syUYwgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCBd38ayLm8wX/qJfYIOH5V+YvlG+poWQXCW6LKN
# 70H3DjCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# JYvei2xyJjHdAAAAAAElMCIEIGUv4mVaWHZFPaXIT+rEBijop/T2Q4KhKJ7kCZZH
# MY4/MA0GCSqGSIb3DQEBCwUABIIBADmM6a5Cwfuqz9dOVEKxad4mlFqgm1nUS3vf
# 9tYGd/Wi4s/JSOdu1Rkvooduj8+9T/Gm2yrlbnFpRC42TkY1ucuyKaadzgwr9/tL
# wXZuWil2n4yYqFJ2TaL+IWu44TtK1TJ/S0xFWWN+P4OtjsUBwBF5ePI+1MRfs5yr
# HaaqrpCyrvZ2VvZtocnb/19PWJ08t+Rrgeo+AOj/Oq2aug2LiV9hfNBVGB89E7Ve
# faceioV3UG4+hbau0YamKKwiQY2gGfGMxQSuKSsQaI0DSiTUjsDs0E97+5qaeWev
# NOm3baV03ZKbm2mRM6VrE/sEDuZ/GGRm9BHm5hK5Wfm2Hd0oQcE=
# SIG # End signature block
