﻿<#
.SYNOPSIS
    Create the AOS staging environment.

.DESCRIPTION

    Copyright © 2019 Microsoft. All rights reserved.
#>
[CmdletBinding()]
param
(
    [Parameter(Mandatory = $false, HelpMessage = "The path to the directory in which to write log files.")]
    [string]$LogDir,
    
    [Parameter(Mandatory = $false, HelpMessage = "The ID of the runbook this is executing as part of.")]
    [string]$RunbookId,

    [Parameter(Mandatory = $false, HelpMessage = "The activity id for current operation.")]        
    [string]$ActivityId,

    [Parameter(Mandatory=$false, HelpMessage = "The secondary database connection info if exists.")]
	[hashtable]$AxDRConnectionString,

    [Parameter(Mandatory=$false, HelpMessage = "The servicing features enabled for this package.")]
	[string[]]$EnabledServicingFeatures,

    [Parameter(Mandatory=$false, HelpMessage = "The global update step VMs for each service model.")]
	[hashtable]$GlobalUpdateVMs,

    [Parameter(Mandatory=$false)]
    [hashtable]$Credentials
)

function PrepareDeltaDBsyncReportDeployment([string]$metadataPackagePath)
{
    $MetadataParentPath = Split-Path -Path $metadataPackagePath -Parent
    $deltaSyncFolder = Join-Path -Path $MetadataParentPath -ChildPath "DeltaSync"

    if (!(Test-Path -Path $deltaSyncFolder))
    {
        Write-ServicingLog "Creating delta sync folder." -Vrb
        New-Item -Path $deltaSyncFolder -ItemType Directory -Force | Out-Null
    }
    else
    {
        Write-ServicingLog "Deleting delta sync folder contents." -Vrb
        Get-ChildItem -Path $deltaSyncFolder | Remove-Item -Force -Recurse
    }

    Write-ServicingLog "Copying MD files from metadata package folder to delta sync folder." -Vrb
    Robocopy.exe $metadataPackagePath $deltaSyncFolder /s *AXSecurity*.md *AXTable*.md *AXView*.md *AXEdt*.md *.rdl  >$null
    Write-ServicingLog "Copying MD files completed with RoboCopy exit code $($LASTEXITCODE)." -Vrb
}

$ErrorActionPreference = "Stop"
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\AosEnvironmentUtilities.psm1" -Force -DisableNameChecking

# Initialize the log file to use with Write-ServicingLog.
Set-ServicingLog -LogDir $LogDir

# Initialize exit code.
[int]$ExitCode = 0

# Get the runbook ID
if ([string]::IsNullOrWhiteSpace($RunbookId))
{
    $RunbookId = Get-RunbookId

    #If the runbook ID still wasn't found, set it to a unique value
    if ([string]::IsNullOrWhiteSpace($RunbookId))
    {
        $runbookIdPlaceholder = [DateTime]::UtcNow.Ticks.ToString()
        Write-ServicingLog "Warning: Runbook ID was not specified and could not be determined. Using [$runbookIdPlaceholder] as a temporary Runbook ID."
        $RunbookId = $runbookIdPlaceholder
    }
}

# Progress breadcrumb names
$copyStagingStarted = "CopyToStagingStarted"
$copyStagingCompleted = "CopyToStagingCompleted"
$prepareDeltaSyncCompleted = "PrepareDeltaSyncReportCompleted"
$upgradeConfigCompleted = "UpgradeConfigAosServiceCompleted"
$updateAosCompleted = "UpdateAosServiceCompleted"
$overallCompleted = "CreateAOSStagingEnvCompleted"

try
{
    Write-ServicingLog "Starting AOS create staging environment script..."

    Disable-CopyStagingTask
    Assert-CopyStagingTaskNotRunning

    $aosWebServicePath = Get-AosServicePath
    $aosWebServiceStagingPath = Get-AosServiceStagingPath
    $packagePath = Join-Path -Path $aosWebServiceStagingPath -ChildPath "PackagesLocalDirectory"
    
    $progressFileNameFormat = "AOSStagingProgress_{0}.json"
    $script:progressFilePath = Join-Path -Path $aosWebServiceStagingPath -ChildPath ($progressFileNameFormat -f $RunbookId)

    if (!(Test-Path -Path $aosWebServiceStagingPath))
    {
        Write-ServicingLog "Creating AOS staging folder." -Vrb
        New-Item -Path $aosWebServiceStagingPath -ItemType Directory -Force | Out-Null
    }
    else
    {
        Write-ServicingLog "Detected AOS Staging folder '$aosWebServiceStagingPath'."
    }

    Initialize-ScriptProgress -progressFile $script:progressFilePath -Scope "Global" -ErrorAction Continue
    Add-ScriptProgress $copyStagingStarted -ErrorAction Continue
    
    if (Test-ScriptProgress $copyStagingCompleted -ErrorAction SilentlyContinue)
    {
        Write-ServicingLog "Detected the previous copy to staging completed at [$(Get-ScriptProgress $copyStagingCompleted)] for runbook [$RunbookId]. Skipping copy."
    }
    else
    {
        Write-ServicingLog "Copying AOS folder to AOS staging folder."
        $robocopyLogFile = Join-Path $LogDir ("Robocopy_{0}.txt" -f (Get-Date).ToString("yyyyMMddHHMMss"))

        $roboCopyExitCode = Copy-FullFolder -SourcePath $aosWebServicePath -DestinationPath $aosWebServiceStagingPath -LogFile $robocopyLogFile 
        Write-ServicingLog "Copying AOS folder to AOS staging folder completed with exit code: $roboCopyExitCode"
        Add-ScriptProgress $copyStagingCompleted -ErrorAction Continue
    }

    # Attempt to clean up any progress files copied over from previous runs
    Get-ChildItem $aosWebServiceStagingPath -Filter ($progressFileNameFormat -f "*") -File | ForEach-Object {
        if ($_.Name -ne ($progressFileNameFormat -f $RunbookId))
        {
            Write-Host "Removing progress file from a previous run [$($_.Name)]"
            Remove-Item -Path $_.FullName -Force -ErrorAction Continue
        }
    }
    
    # Prepare delta dbsyc / report deployment
    if (Test-ScriptProgress $prepareDeltaSyncCompleted)
    {
        Write-ServicingLog "Detected PrepareDeltaDBSyncReportDeployment previously completed at [$(Get-ScriptProgress $prepareDeltaSyncCompleted)] for runbook [$RunbookId]. Skipping."
    }
    else 
    {
        Write-ServicingLog "Invoking PrepareDeltaDBSyncReportDeployment function."
        PrepareDeltaDBsyncReportDeployment -metadataPackagePath:$packagePath
        Add-ScriptProgress $prepareDeltaSyncCompleted -ErrorAction Continue
    }
    
    # Update the web.config
    if (Test-ScriptProgress $upgradeConfigCompleted)
    {
        Write-ServicingLog "Detected updating AOSService configuration previously completed at [$(Get-ScriptProgress $upgradeConfigCompleted)] for runbook [$RunbookId]. Skipping."
    }
    else 
    {
        Write-ServicingLog "Invoking AutoUpgradeConfigAosServicePlatformUpdate3.ps1 script."
        Invoke-Expression "$PSScriptRoot\AutoUpgradeConfigAosServicePlatformUpdate3.ps1 -useStaging"
        Write-ServicingLog "Completed AutoUpgradeConfigAosServicePlatformUpdate3.ps1 script with exit code: $($LASTEXITCODE)." -Vrb
        Add-ScriptProgress $upgradeConfigCompleted -ErrorAction Continue
    }

    # Update AOSService
    if (Test-ScriptProgress $updateAosCompleted)
    {
        Write-ServicingLog "Detected updating AOSService previously completed at [$(Get-ScriptProgress $updateAosCompleted)] for runbook [$RunbookId]. Skipping."
    }
    else
    {
        # Do not pass LogDir to AutoUpdateAosService to keep output going to the current log file.
        Write-ServicingLog "Invoking AutoUpdateAosService.ps1 script."
        Invoke-Expression "$PSScriptRoot\AutoUpdateAosService.ps1 -useStaging"
        Write-ServicingLog "Completed AutoUpdateAosService.ps1 script with exit code: $($LASTEXITCODE)." -Vrb
        Add-ScriptProgress $updateAosCompleted -ErrorAction Continue
    }

    # Indicates the script logic has completed - ensure this stays the last line of the execution block.
    Add-ScriptProgress $overallCompleted -ErrorAction Continue
}
catch
{
    # Ensure non-zero exit code if an exception is caught and no exit code set.
    if ($ExitCode -eq 0)
    {
        $ExitCode = 1024
    }

    $ErrorMessage = "Error during AOS create staging: $($_)"

    Write-ServicingLog $ErrorMessage
    Write-ServicingLog $($_) -Vrb
    Write-ServicingLog "AOS create staging environment script failed with exit code: $($ExitCode)."

    # Use throw to indicate error to AXUpdateInstaller.
    # In case of exceptions, the output is not captured, so only the error message and
    # log file contents will be available for diagnostics.
    throw "$($ErrorMessage) [Log: $(Get-ServicingLog)]"
}
finally
{
    # Back up the progress file for this script's execution.
    $backupDir = if ([string]::IsNullOrEmpty($LogDir)) { $PSScriptRoot } else { $LogDir }
    $logTimeSuffix = [DateTime]::UtcNow.ToString('yyyyMMdd-HHmmss')
    $scriptProgressFileBackup = Join-Path -Path $backupDir -ChildPath "CreateAOSStagingEnv-ProgressBackup-$logTimeSuffix.json"
    Copy-ScriptProgressFile -Destination $scriptProgressFileBackup -ErrorAction Continue
}

Write-ServicingLog "AOS create staging environment script completed with exit code: $($ExitCode)."

$machineName = $env:COMPUTERNAME
if ($GlobalUpdateVMs -and $GlobalUpdateVMs['AOSService'] -eq $machineName)
{
    Write-ServicingLog "Running DB sync precheck..."

    # Run DB Sync PreCheck
    $precheckScriptPath = "$PSScriptRoot\DBSyncPrecheck.ps1"
    $precheckScriptPath = Resolve-Path -Path $precheckScriptPath
    & $precheckScriptPath -LogDir $LogDir -ActivityId $ActivityId -RunbookId $RunbookId -AxDRConnectionString $AxDRConnectionString -EnabledServicingFeatures $EnabledServicingFeatures -Credentials $Credentials -PreDowntime
    Write-ServicingLog "DB sync precheck completed."
}

exit $ExitCode
# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCANzfGR69dTmu4N
# agG3gv4EVIU7zNnpQ3QVhosX9+z71KCCDYEwggX/MIID56ADAgECAhMzAAABh3IX
# chVZQMcJAAAAAAGHMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAwMzA0MTgzOTQ3WhcNMjEwMzAzMTgzOTQ3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDOt8kLc7P3T7MKIhouYHewMFmnq8Ayu7FOhZCQabVwBp2VS4WyB2Qe4TQBT8aB
# znANDEPjHKNdPT8Xz5cNali6XHefS8i/WXtF0vSsP8NEv6mBHuA2p1fw2wB/F0dH
# sJ3GfZ5c0sPJjklsiYqPw59xJ54kM91IOgiO2OUzjNAljPibjCWfH7UzQ1TPHc4d
# weils8GEIrbBRb7IWwiObL12jWT4Yh71NQgvJ9Fn6+UhD9x2uk3dLj84vwt1NuFQ
# itKJxIV0fVsRNR3abQVOLqpDugbr0SzNL6o8xzOHL5OXiGGwg6ekiXA1/2XXY7yV
# Fc39tledDtZjSjNbex1zzwSXAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUhov4ZyO96axkJdMjpzu2zVXOJcsw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDU4Mzg1MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAixmy
# S6E6vprWD9KFNIB9G5zyMuIjZAOuUJ1EK/Vlg6Fb3ZHXjjUwATKIcXbFuFC6Wr4K
# NrU4DY/sBVqmab5AC/je3bpUpjtxpEyqUqtPc30wEg/rO9vmKmqKoLPT37svc2NV
# BmGNl+85qO4fV/w7Cx7J0Bbqk19KcRNdjt6eKoTnTPHBHlVHQIHZpMxacbFOAkJr
# qAVkYZdz7ikNXTxV+GRb36tC4ByMNxE2DF7vFdvaiZP0CVZ5ByJ2gAhXMdK9+usx
# zVk913qKde1OAuWdv+rndqkAIm8fUlRnr4saSCg7cIbUwCCf116wUJ7EuJDg0vHe
# yhnCeHnBbyH3RZkHEi2ofmfgnFISJZDdMAeVZGVOh20Jp50XBzqokpPzeZ6zc1/g
# yILNyiVgE+RPkjnUQshd1f1PMgn3tns2Cz7bJiVUaqEO3n9qRFgy5JuLae6UweGf
# AeOo3dgLZxikKzYs3hDMaEtJq8IP71cX7QXe6lnMmXU/Hdfz2p897Zd+kU+vZvKI
# 3cwLfuVQgK2RZ2z+Kc3K3dRPz2rXycK5XCuRZmvGab/WbrZiC7wJQapgBodltMI5
# GMdFrBg9IeF7/rP4EqVQXeKtevTlZXjpuNhhjuR+2DMt/dWufjXpiW91bo3aH6Ea
# jOALXmoxgltCp1K7hrS6gmsvj94cLRf50QQ4U8Qwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAYdyF3IVWUDHCQAAAAABhzAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQghJ1MNuax
# ywvU5MHyzsR7zUO6otzKgItvFaoCqeQD2q4wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQCn73fGYLTMVoKt+9dz3YfBxEJtJfNJLh1ksc5vyXZM
# 1D9GUYqauKH550E1h2+c5PWh8zG07Han+4qHZHoksyl9o8a9P6GwxBJva3OIUBSs
# UjGG5w995DVL55zGGn3D7pSi11Hj5/knvCBxTEQ402nDUYcmFGTRqT8Yj4lh9PO9
# dqPCIkPjdJ4kFFKmldfmEpwy0y9hJaLr8ASMe5ooFa0A9kIlDOvSEZlJ2r970QZB
# Q7HOgWq47X1Rmgvf1EvwFPneOaTfkomqgYGvmLigpnDvL1c6Y9LRZF+0timR05BX
# LkEJH3q+L6nWVzLihIgynnxgFrr/sF36RxqSAqVR0x0woYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIG+ksJ36tpX6rq3MGduIbxgcWyD27FQhbNDXgBTw
# vPM7AgZfs/a3udcYEzIwMjAxMTIwMTkxMzE4LjgxM1owBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjowQTU2LUUzMjktNEQ0RDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABJy9uo++RqBmoAAAA
# AAEnMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTE5MTIxOTAxMTQ1OVoXDTIxMDMxNzAxMTQ1OVowgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjowQTU2
# LUUzMjktNEQ0RDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAPgB3nERnk6fS40vvWeD
# 3HCgM9Ep4xTIQiPnJXE9E+HkZVtTsPemoOyhfNAyF95E/rUvXOVTUcJFL7Xb16jT
# KPXONsCWY8DCixSDIiid6xa30TiEWVcIZRwiDlcx29D467OTav5rA1G6TwAEY5rQ
# jhUHLrOoJgfJfakZq6IHjd+slI0/qlys7QIGakFk2OB6mh/ln/nS8G4kNRK6Do4g
# xDtnBSFLNfhsSZlRSMDJwFvrZ2FCkaoexd7rKlUNOAAScY411IEqQeI1PwfRm3aW
# bS8IvAfJPC2Ah2LrtP8sKn5faaU8epexje7vZfcZif/cbxgUKStJzqbdvTBNc93n
# /Z8CAwEAAaOCARswggEXMB0GA1UdDgQWBBTl9JZVgF85MSRbYlOJXbhY022V8jAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQAKyo180VXHBqVnjZwQy7NlzXbo2+W5
# qfHxR7ANV5RBkRkdGamkwUcDNL+DpHObFPJHa0oTeYKE0Zbl1MvvfS8RtGGdhGYG
# CJf+BPd/gBCs4+dkZdjvOzNyuVuDPGlqQ5f7HS7iuQ/cCyGHcHYJ0nXVewF2Lk+J
# lrWykHpTlLwPXmCpNR+gieItPi/UMF2RYTGwojW+yIVwNyMYnjFGUxEX5/DtJjRZ
# mg7PBHMrENN2DgO6wBelp4ptyH2KK2EsWT+8jFCuoKv+eJby0QD55LN5f8SrUPRn
# K86fh7aVOfCglQofo5ABZIGiDIrg4JsV4k6p0oBSIFOAcqRAhiH+1spCMIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjow
# QTU2LUUzMjktNEQ0RDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUAs5W4TmyDHMRM7iz6mgGojqvXHzOggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AONiaTwwIhgPMjAyMDExMjAyMDExNDBaGA8yMDIwMTEyMTIwMTE0MFowdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA42JpPAIBADAKAgEAAgIfvQIB/zAHAgEAAgIRxTAK
# AgUA42O6vAIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBAIS/2M49L36GBFys
# joumQutknPPJ23eHv66FFRw5Idt75CbKi7gDYLX+rIr9vWUgfcMrmiB9kWSKovO3
# MT9gQmPcYOBpMuK+/0KeY5u4YywItyoeVhhjU1EWm5XD8EmSl4bDSaAR19H4D4Ml
# /dxmStDoXPdCGSdfa+VS+5nWFdNPMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAEnL26j75GoGagAAAAAAScwDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQgkAaO+Hp8PjGWetxw35U6nmMFW1FJElZQWT8pPRyMKvYwgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCAbkuhLEoYdahb/BUyVszO2VDi6kB3MSaof/+8u
# 7SM+IjCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# Jy9uo++RqBmoAAAAAAEnMCIEIP3bZyhVI960OiLQe2eMV27gvrZ621PYlH9vndfQ
# sr45MA0GCSqGSIb3DQEBCwUABIIBAE/JjsHLjrpGRGlSaQ7YmZV9QXKflcnjZvLt
# 3sa3Ii/mVipHpNU81kZD6f3+Gf0Q4EyxvclqrcTKqTNzm/xeEeJqUFArxxsoAW73
# ADrI9McTnZK90NkZPNyg6Ly3KFi6zQbsdjwsciE0DEbunAZkrOs9AbbkzzLOJFAF
# 4hUTxN+vQTAjSid36NtoKOl6RNf3WseGoBj7Id7g9m6qX4kB/a25E8hWosO3jCSX
# Gdzk7R0Pf7LC+X5QKvItstnuD/pnJ9RQpOoaE/L6J8bbB/aAQBcKOPJjtu0LQavR
# VCdG7E3acb8ZrOW+YRYmOb7+qK1tti3kNEu0lzTq3TT3qumZjL0=
# SIG # End signature block
