﻿[CmdletBinding()]
param
(
    [Parameter(Mandatory=$true)]
    [string]$ServiceModelXml,

    [Parameter(Mandatory=$true)]
    [string]$log,

    [Parameter(Mandatory=$false)]
    [string]$config
)

Import-Module WebAdministration

#region configuration generators
function Write-Storage-Emulator-Configuration([string]$logDir)
{
    $registrykey="HKLM:\SOFTWARE\Wow6432Node\Microsoft\Windows Azure Storage Emulator\"
    $registrykeyvaluename="InstallPath"
    $emulatorInstallPath=Get-RegistryKeyValue -registryKey:$registrykey -registryKeyValueName:$registrykeyvaluename
    if([string]::IsNullOrEmpty($emulatorInstallPath)){
        Write-Log "Skipping Azure sdk configuration as the Azure emulator is not installed."
        break
    }

    $azurestorageemulator= join-path $emulatorInstallPath "azurestorageemulator.exe"

    $standardout=join-path $global:logdir "azurestorageemulator.stopprocess.output"
    $standarderr=join-path $global:logdir "azurestorageemulator.stopprocess.error"
    $script="`
    WindowsProcess StopEmulatorProcesses `
    { `
        Ensure=`"Absent`" `
        Arguments=`"start /inprocess`" `
        Path=`"$azurestorageemulator`" `
        DependsOn=@() `
    }"

    Write-Log "Creating the configuration to stop any storage emulator process(es)"
    Write-Content $script

    $script = @"
    DynamicsScheduledTask StartAzureStorageEmulator_ScheduledTask
    {
        TaskName = "DynamicsStartAzureStorageEmulator"
        Ensure = "Absent"
    }
"@
    Write-Content $script

}

function Write-Miscellaneous-Processes-Configuration([string]$webroot,[string]$logDir)
{
    $binarypathname=join-path $webroot "bin\Microsoft.Dynamics.AX.Deployment.Setup.exe"
    $standardoutdir=join-path $logDir "Microsoft.Dynamics.AX.Deployment.Setup.exe.output"
    $standarderrdir=join-path $logDir "Microsoft.Dynamics.AX.Deployment.Setup.exe.error"

    $script="`
    DynamicsProcess StopDeploymentSetupProcess `
    { `
        Ensure=`"Absent`" `
        Arguments=`"`" `
        File=`"$binarypathname`" `
        StandardErrorPath=`"$standardoutdir`" `
        StandardOutputPath=`"$standarderrdir`" `
        DependsOn=@(`"[WindowsProcess]StopEmulatorProcesses`") `
        MonitoringAssembly=`"$global:telemetrydll`" `
    } "

    Write-Log "Creating the configuration to stop the deployment setup process"
    Write-Content $script
}

function Write-BatchService-Configuration([string]$webroot,[string]$logDir)
{
    $webconfig= join-path $webroot "web.config"
    $binarypathname=join-path $webroot "bin\Batch.exe"
    $standardout=join-path $logDir "batch.exe.output"
    $standarderr=join-path $logDir "batch.exe.error"
    $deleteBatchServiceDependency=""
    if(Test-Path $binarypathname){
        $script="`
        WindowsProcess StopBatchProcess `
        { `
            Ensure=`"Absent`" `
            Arguments='-service $webconfig' `
            Path=`"$binarypathname`" `
        } "

        Write-Log "Creating the configuration to stop the Dynamics AX Batch process"
        Write-Content $script
        $deleteBatchServiceDependency="`"[WindowsProcess]StopBatchProcess`""
    }
    
    $servicepath=join-path $webroot "bin\Batch.exe -service $webconfig"
    $script = "`
    Service DeleteBatchService `
    { `
        Ensure=`"Absent`" `
        Name=`"DynamicsAxBatch`" `
        State=`"Stopped`" `
        DisplayName=`"Dynamics AX Batch Management`" `
        Path='$servicepath' `
        DependsOn=@($deleteBatchServiceDependency) `
    }"

    Write-Log "Creating the configuration to delete the Dynamics AX Batch service"
    Write-Content $script

    $script = @"
    DynamicsScheduledTask StartBatch_ScheduledTask
    {
        TaskName = "DynamicsStartBatch"
        Ensure = "Absent"
    }
"@
    Write-Content $script
}

function Write-WebSite-Configuration([string]$websitename,[string]$webroot,[string]$apppoolname)
{
    $source=$env:TEMP
    $script="`
    xWebSite DeleteAosWebSite `
    { `
        Ensure=`"Absent`" `
        Name=`"$websiteName`" `
        PhysicalPath=`"$webRoot`" `
        State=`"Stopped`" `
    }"

    Write-Log "Creating the configuration to delete the AOS web site"
    Write-Content $script

    if(![string]::IsNullOrEmpty($apppoolname)){
         # delete the aos apppool if it exists
        $script="`
        xWebAppPool DeleteAosAppPool `
        { `
            Ensure=`"Absent`" `
            State=`"Stopped`" `
            Name=`"$apppoolname`" `
            DependsOn=@(`"[xWebSite]DeleteAosWebSite`") `
        }"

        Write-Log "Creating the configuration to delete the AOS web app pool"
        Write-Content $script
    }

    $script= @"
    xWebAppPool DeleteProductConfigurationAppPool
    {
        Ensure="Absent"
        State="Stopped"
        Name="ProductConfiguration"
        DependsOn=@("[xWebSite]DeleteAosWebSite")
    }
"@
    Write-Log "Creating the configuration to delete the product configuration web app pool"
    Write-Content $script

    $script = "
    File DeleteWebSitePhysicalPath`
    { `
        Ensure=`"Absent`" `
        DestinationPath=`"$webRoot`" `
        Recurse=`$true `
        SourcePath=`"`" `
        Type=`"Directory`" `
        DependsOn=@(`"[xWebSite]DeleteAosWebSite`", `"[Service]DeleteBatchService`") `
        Force=`$true `
    }"

     Write-Log "Creating the configuration to delete the AOS web site files"
     Write-Content $script
    
}

function Write-Packages-Configuration([string]$packagedir)
{
    $script = "
    File DeletePackages`
    { `
        Ensure=`"Absent`" `
        DestinationPath=`"$packageDir`" `
        Recurse=`$true `
        SourcePath=`"`" `
        Type=`"Directory`" `
        Force=`$true `
        DependsOn=@(`"[xWebSite]DeleteAosWebSite`") `
    }"

    Write-Log "Creating the configuration to delete the Dynamics packages"
    Write-Content $script
}

function Write-Database-Configuration([string]$dbName,[string]$dbServer,[string]$dbUser,[string]$dbPwd,[string]$logDir)
{
    # for dropping the db, use the temp folder name as the package dir as it is not used
    $packageDir=$env:TEMP

    $script="`
    DynamicsDatabase DropDatabase `
    { `
        Ensure=`"Absent`" `
        PackageDirectory=`"$packageDir`" `
        DatabaseName=`"$dbName`" `
        DatabaseServer=`"$dbServer`" `
        UserName=`"$dbUser`" `
        Password=`"$dbPwd`" `
        DependsOn=@(`"[File]DeletePackages`") `
        MonitoringAssembly=`"$global:telemetrydll`" `
    }"

    Write-Log "Creating the configuration to drop the AX database"
    Write-Content $script
}

function Write-LCM-Configuration($thumbprint)
{
    Write-Verbose "Creating the LCM configuration to set the encryption certificate thumbprint..."
    $script="`
    LocalConfigurationManager 
    {
        CertificateId=`"$thumbprint`"
    }"
    
    write-content $script
}
#endregion

#region helper functions
function Initialize-Log([string]$logdir,[string]$logfile)
{
    if(-not (Test-Path $logdir)){
        New-Item -ItemType Directory -Path $logdir
    }
    
    if(Test-Path $logfile){
        Remove-Item $logfile -Force
    }

    New-item $logfile -ItemType File -Force
}

function Write-Log([string]$message)
{
    Add-Content $global:logfile $message
    Write-Host $message
}

function Write-Header
{
    $datetime=Get-Date
    $header="`
###
# ==++==
#
# Copyright (c) Microsoft Corporation. All rights reserved.
#
# Generated date: $datetime
#
# This file is auto-generated by a tool. Any changes made to this file will be lost.
#
# This file will be compiled to generate a Windows DSC configuration that will remove 
# the Dynamics AOS service on this machine.
###

#region service configuration
Configuration RemoveAosServiceConfiguration { `
`
Import-DscResource -Module xWebAdministration `
Import-DscResource -Module xDynamics `
Import-DscResource -Module PSDesiredStateConfiguration `

Node `"localhost`"{"

    Write-Content $header
}

function Write-Footer([string]$outputpath)
{
    $footer="}`
    } `

`$configData=@{
    AllNodes=@(
        @{NodeName=`"localhost`";CertificateFile=`"$global:encryptioncertpublickeyfile`";Thumbprint=`"$global:encryptioncertthumbprint`";PSDscAllowDomainUser=`$true; }
    )
}`

#endregion

# generate the MOF file `    
RemoveAosServiceConfiguration -OutputPath:$outputpath -ConfigurationData:`$configData"

    Write-Content $footer
}

function Write-Content([string]$content)
{
   Add-Content $global:dscconfigfile -Value $content
}

function Get-RegistryKeyValue([string]$registryKey,[string]$registryKeyValueName,[string]$defaultValue=[string]::Empty)
{
    $value=(Get-ItemProperty "$registryKey").$registryKeyValueName
    if([string]::IsNullOrEmpty($value)){
        return $defaultValue
    }

    return $value
}

function Copy-CustomDSCResources([string]$grandparentDir)
{
    if(Test-Path "$grandparentDir\Scripts\xDynamics"){
        Write-Log "Copying the custom DSC resources"
        Copy-Item -Path "$grandparentDir\Scripts\xDynamics" -Destination "$env:ProgramFiles\WindowsPowerShell\Modules" -Force -Recurse -Verbose
        Get-ChildItem -Path "$env:ProgramFiles\WindowsPowerShell\Modules" -Recurse | Unblock-File -Verbose  
    }else{
        Write-Log "No custom DSC resources to copy"
    }
}

function Save-EncryptionCertificate-PublicKey($certificate)
{
    Write-Log "Saving the encryption cert public key to file..."
    $global:encryptioncertpublickeyfile=join-path $global:logdir "EncryptCert.cer"
    Export-Certificate -cert:$certificate -FilePath $global:encryptioncertpublickeyfile -Force -Type CERT | Out-Null
}

function Get-EncryptionCertificate-Thumbprint
{
    $subject="MicrosoftDynamicsAXDSCEncryptionCert"

    #get or create a new self-signed encryption certificate to secure sensitive info in the MOF files
    $cert=Get-Make-Encryption-Cert -subject:$subject
    Save-EncryptionCertificate-PublicKey -certificate:$cert
    $cert.Thumbprint
}

function Get-Make-Encryption-Cert([string]$subject)
{
    Write-Log "Checking if a self-signed encryption cert with subject '$subject' exists..."
    $formattersubject="CN=$subject"
    $cert=Get-ChildItem Cert:\LocalMachine\My|where {$_.Subject -eq $formattersubject}
    if($cert -ne $null) # if cert existed make sure it is valid
    {
        if(!(Is-ValidCert -certificate:$cert))
        {
            Write-Log "Dynamics DSC self-signed encryption cert is expired. Generating a new self-signed encryption certificate..."
            Write-Log "Deleting the invalid self-signed encryption certificate with subject '$cert.subject'... "
            $thumbprint=$cert.Thumbprint
            Remove-Item -Path Cert:\LocalMachine\My\$thumbprint -Force -DeleteKey |out-null
            $cert=Make-Certificate -subject:$subject
        }
    }
    else
    {
        $cert=Make-Certificate -subject:$subject 
    }

    $cert
}

function Is-ValidCert($certificate)
{
    $subject=$certificate.Subject
    Write-Log "Checking if the certificate '$subject' is valid..."
    $thumbprint=$certificate.Thumbprint
    $cert=Get-ChildItem -Path Cert:\LocalMachine\My\$thumbprint
    if($cert -ne $null)
    {
        if($cert.NotAfter -lt (Get-Date)) #expired
        {
            return $false
        }
        else
        {
            return $true
        }
    }

    #if cert is not found, return false
    return $false
}

function Make-Certificate([string]$subject)
{
     Write-Log "Creating a new DSC self-signed encryption certificate with subject '$subject'..."
     return New-SelfSignedCertificate -DnsName $subject -CertStoreLocation cert:\LocalMachine\My
}

function Output-CurrentEnvironmentState([string]$websitePath, [string]$packagePath)
{
    $handleExePath=join-path $env:SystemDrive "SysInternals\Handle.exe"
    $openHandlesLog=join-path $global:logDir "aosservice-uninstallation-openhandles.log"
    if(-not (Test-Path $handleExePath))
    {
        return
    }
    
    #dump any handles to files in the website directory
    if(Test-Path $websitePath)
    {
        Write-Log "AOS WebRoot still exists at $websitePath. Dumping open handles to $openHandlesLog..."
        & "$handleExePath" -AcceptEula $websitePath | Out-File $openHandlesLog -Append
    }
    
    #dump any handles to files in the package directory
    if(Test-Path $packagePath)
    {
        Write-Log "AOS packages directory still exists at $packagePath. Dumping open handles to $openHandlesLog..."
        & "$handleExePath" -AcceptEula $packagePath | Out-File $openHandlesLog -Append
    }
}

#endregion

#region Main
$parentdir=Split-Path -parent $PSCommandPath
$grandparentdir=Split-Path -parent $parentdir
$global:logfile=$log
$global:logdir=[System.IO.Path]::GetDirectoryName($log)

Initialize-Log -logdir:$global:logdir -logfile:$log
Copy-CustomDSCResources -grandparentDir:$grandparentdir

$global:decodedservicemodelxml=[System.Text.Encoding]::UTF8.GetString([System.Convert]::FromBase64String($servicemodelxml))

$global:dscconfigfile=join-path $global:logdir "Uninstall.ps1"

if(Test-Path $global:dscconfigfile){
    Remove-Item $global:dscconfigfile -Force
}

$outputpath=join-path $global:logdir "Uninstall"
$etwpath=join-path $grandparentdir "ETWManifest"
$global:telemetrydll = join-path $etwpath "Microsoft.Dynamics.AX7Deployment.Instrumentation.dll"

if(-not (Test-Path $global:telemetrydll)){
    throw "The deployment telemetry assembly does not exist"
}

[System.Xml.XmlDocument] $xd = new-object System.Xml.XmlDocument
$xd.LoadXml($global:decodedservicemodelxml)
$ns=New-Object System.Xml.XmlNamespaceManager($xd.NameTable)
$ns.AddNamespace("ns",$xd.DocumentElement.NamespaceURI)

if($env:USERDOMAIN -eq $env:COMPUTERNAME){
    $global:domain="builtin"
}else{
    $global:domain=$env:USERDOMAIN
}

$global:username=$env:USERNAME

$global:encryptioncertthumbprint=Get-EncryptionCertificate-Thumbprint

[string]$websiteName=$xd.SelectSingleNode("//ns:ServiceModel",$ns).getAttribute("Name")
[string]$webRoot=$xd.SelectSingleNode("//ns:Setting[@Name='Infrastructure.WebRoot']",$ns).getAttribute("Value")
Write-Log "The web root is $webRoot"
[string]$packageDir=$xd.SelectSingleNode("//ns:WorkingFolder[@Name='PackagesLocalDirectory']",$ns).getAttribute("Directory")
Write-Log "The package directory is $packageDir"

$website=Get-Website $websiteName
if($website -ne $null)
{
    $apppool=$website.applicationPool
}

$uninstallAttempt=1
while($uninstallAttempt -le 3 -and !$success)
{
    if($uninstallAttempt -gt 1)
    {
        $retryAttempt=$uninstallAttempt-1
        Write-Log "Retry attempt $retryAttempt`: Retrying AOS service uninstallation..."
    }

    # create the configuration file
    Write-Header 
    Write-LCM-Configuration -thumbprint:$global:encryptioncertthumbprint
    Write-Storage-Emulator-Configuration -logDir:$log
    Write-Miscellaneous-Processes-Configuration -webroot:$webroot -logDir:$log
    Write-BatchService-Configuration -webroot:$webroot -logDir:$log
    Write-WebSite-Configuration -websitename:$websiteName -webroot:$webroot -apppoolname:$apppool
    Write-Packages-Configuration -packagedir:$packageDir

    Write-Footer -outputpath:$outputpath
    #endregion

    #region execute the configuration
    Write-Log "Executing the configuration.."
    & $global:dscconfigfile
    [bool]$success=$false
    $dscConfigApplied = $false
    try
    {
        $dscConfigApplied = $false
        Set-Location $outputpath
        Write-Log ("PSModulePath is currently: "+$env:PSModulePath)

        Write-Log "Setting up LCM to decrypt credentials..."
        Set-DscLocalConfigurationManager -path "$outputpath" -Verbose *>&1 | Tee-Object $log
        
        try
        {
            Write-Log("Dumping available DSC resources before applying the configuration...")
            $availableDSCResourceLog=join-path $global:logdir "aosservice-uninstallation-availabledscresources.log"
            Get-DSCResource -Name * | Format-List | Out-File -FilePath $availableDSCResourceLog
        }
        catch
        {
            Write-Log "Failed to get DSC resources, Error: $_"
        }
        
        Write-Log "Applying the configuration..."
        Start-DscConfiguration -wait -Verbose -path "$outputpath" -Force *>&1 | Tee-Object $log
        $dscConfigApplied = $true
    }
    catch
    {
        Write-Log "Uninstall attempt $uninstallAttempt`: Error: $_"
    }

    $configstatuslog=join-path $global:logdir "aosservice-uninstallation-status.log"
    $ConfigStatus = Get-DscConfigurationStatus
    $ConfigStatus | Format-List -Property * | Out-File -FilePath $configstatuslog -Force
    if($ConfigStatus.Status -ieq 'Success' -and $dscConfigApplied -eq $true)
    {
        $success=$true
    }
    else
    {
        Output-CurrentEnvironmentState $webRoot $packageDir
        Move-Item $global:dscconfigfile (join-path $global:logdir "Uninstall_Attempt_$uninstallAttempt.ps1")
        $uninstallAttempt++
    }
}

if($success)
{
    Write-Log "Configuration applied."
    return 0
}
else
{
    throw "AOS uninstallation did not complete after 3 retries, Message: $($ConfigJob.StatusMessage), see log for details."    
}

#endregion

# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAfYjrp8MucsvnP
# mhosShlB1xRhnlNlyhBUqLPqcZEgqKCCDYEwggX/MIID56ADAgECAhMzAAABh3IX
# chVZQMcJAAAAAAGHMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAwMzA0MTgzOTQ3WhcNMjEwMzAzMTgzOTQ3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDOt8kLc7P3T7MKIhouYHewMFmnq8Ayu7FOhZCQabVwBp2VS4WyB2Qe4TQBT8aB
# znANDEPjHKNdPT8Xz5cNali6XHefS8i/WXtF0vSsP8NEv6mBHuA2p1fw2wB/F0dH
# sJ3GfZ5c0sPJjklsiYqPw59xJ54kM91IOgiO2OUzjNAljPibjCWfH7UzQ1TPHc4d
# weils8GEIrbBRb7IWwiObL12jWT4Yh71NQgvJ9Fn6+UhD9x2uk3dLj84vwt1NuFQ
# itKJxIV0fVsRNR3abQVOLqpDugbr0SzNL6o8xzOHL5OXiGGwg6ekiXA1/2XXY7yV
# Fc39tledDtZjSjNbex1zzwSXAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUhov4ZyO96axkJdMjpzu2zVXOJcsw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDU4Mzg1MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAixmy
# S6E6vprWD9KFNIB9G5zyMuIjZAOuUJ1EK/Vlg6Fb3ZHXjjUwATKIcXbFuFC6Wr4K
# NrU4DY/sBVqmab5AC/je3bpUpjtxpEyqUqtPc30wEg/rO9vmKmqKoLPT37svc2NV
# BmGNl+85qO4fV/w7Cx7J0Bbqk19KcRNdjt6eKoTnTPHBHlVHQIHZpMxacbFOAkJr
# qAVkYZdz7ikNXTxV+GRb36tC4ByMNxE2DF7vFdvaiZP0CVZ5ByJ2gAhXMdK9+usx
# zVk913qKde1OAuWdv+rndqkAIm8fUlRnr4saSCg7cIbUwCCf116wUJ7EuJDg0vHe
# yhnCeHnBbyH3RZkHEi2ofmfgnFISJZDdMAeVZGVOh20Jp50XBzqokpPzeZ6zc1/g
# yILNyiVgE+RPkjnUQshd1f1PMgn3tns2Cz7bJiVUaqEO3n9qRFgy5JuLae6UweGf
# AeOo3dgLZxikKzYs3hDMaEtJq8IP71cX7QXe6lnMmXU/Hdfz2p897Zd+kU+vZvKI
# 3cwLfuVQgK2RZ2z+Kc3K3dRPz2rXycK5XCuRZmvGab/WbrZiC7wJQapgBodltMI5
# GMdFrBg9IeF7/rP4EqVQXeKtevTlZXjpuNhhjuR+2DMt/dWufjXpiW91bo3aH6Ea
# jOALXmoxgltCp1K7hrS6gmsvj94cLRf50QQ4U8Qwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAYdyF3IVWUDHCQAAAAABhzAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgg7Owaixk
# M5EDzafPQCiaXwrV6Wgf9p2NIOssY20/Mr8wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQCzs4cAttpI4Jjzg+071U/yIjsax8kx0eHsJkniRjs7
# gcZl+yBwW11LolJONjZq5mzI9LcOkkrQ/y/Y3e0QvEAcEt917dfouH0k2ttGGGPE
# xEEkg8v3eEyGTrqisD6TPDFXbShdbB1kO7J6Y++xQtkXaGLGwD93qw39jnVKQrVg
# hqTLukbTPjOo6HwcHq6pCC1ML3/WIj25VYFjRUkzDP1gUMdHd90sqsF3MdLOT0Ml
# FPRnyVIxXwMIm9azkkRd2APX8Gj6Ar9IP/4V9oParokKTOxSRhtufsVv5IdjwORB
# BlaLNvnKxTuWdeUVMRz5+a8kjOVpi4DHoJche2rhgUOJoYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIL0t4Kmb8kb1DoLIi02DiPeBhU3mLOpm+aCPeNbB
# OZ5wAgZfu+uDxoMYEzIwMjAxMjE3MDgwODIyLjkwOFowBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjozMkJELUUzRDUtM0IxRDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABLqjSGQeT9GvoAAAA
# AAEuMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTE5MTIxOTAxMTUwNVoXDTIxMDMxNzAxMTUwNVowgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjozMkJE
# LUUzRDUtM0IxRDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAK7TTKJRU196LFIjMQ9q
# /UjpPhz43m5RnHgHAVp2YGni74+ltsYoO1nZ58rTbJhCQ8GYHy8B4devgbqqYPQN
# U3i+drpEtEcNLbsMr4MEq3SM+vO3a6QMFd1lDRy7IQLPJNLKvcM69Nt7ku1YyM5N
# nPNDcRJsnUb/8Yx/zcW5cWjnoj8s9fQ93BPf/J74qM1ql2CdzQV74PBisMP/tppA
# nSuNwo8I7+uWr6vfpBynSWDvJeMDrcsa62Xsm7DbB1NnSsPGAGt3RzlBV9KViciz
# e4U3fo4chdoB2+QLu17PaEmj07qq700CG5XJkpEYOjedNFiByApF7YRvQrOZQ07Q
# YiMCAwEAAaOCARswggEXMB0GA1UdDgQWBBSGmokmTguJN7uqSTQ1UhLwt1RObDAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQCN4ARqpzCuutNqY2nWJDDXj35iaidl
# gtJ/bspYsAX8atJl19IfUKIzTuuSVU3caXZ6/YvMMYMcbsNa/4J28us23K6PWZAl
# jIj0G8QtwDMlQHjrKnrcr4FBAz6ZqvB6SrN3/Wbb0QSK/OlxsU0mfD7z87R2JM4g
# wKJvH6EILuAEtjwUGSB1NKm3Twrm51fCD0jxvWxzaUS2etvMPrh8DNrrHLJBR3UH
# vg/NXS2IzdQn20xjjsW0BUAiTf+NCRpxUvu/j80Nb1++vnejibfpQJ2IlXiJdIi+
# Hb+OL3XOr8MaDDSYOaRFAIfcoq3VPi4BkvSC8QGrvhjAZafkE7R6L5FJMIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjoz
# MkJELUUzRDUtM0IxRDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUA+1/CN6BILeU1yDGo+b6WkpLoQpuggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AOOFY/AwIhgPMjAyMDEyMTcwODU4MjRaGA8yMDIwMTIxODA4NTgyNFowdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA44Vj8AIBADAKAgEAAgImRQIB/zAHAgEAAgITjDAK
# AgUA44a1cAIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBABphlMcnYjhA0+uG
# PO7KmhT1uAi5KompJhV+/KKDWGCCzpfCfojjhAYDT64pclOFb/DePV0UJgH8qXsj
# Fti6NHTChtutoKedBaQI/O2kHV9Jt8Rw6seSi1bigFXOvFnw01rh2buY/zWmOwOV
# RYH4ALLMOXWYJN9qRQdXeo0LEfNTMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAEuqNIZB5P0a+gAAAAAAS4wDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQg6ZgOEKhAoLmTlRghYjfxU8TqLORfm7AgDZAXkIhRZA0wgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCDa/s3O8YhWiqpVN0kTeK+x2m0RAh17JpR6DiFo
# TILJKTCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# LqjSGQeT9GvoAAAAAAEuMCIEICJXWP5hlwCqjWy3iLZHWkOK/zE+IdXHOVC8S86/
# 0+vXMA0GCSqGSIb3DQEBCwUABIIBAI29AP7MbSs4p6GevXwzpTEJkf2thmPvH+3V
# 0Mt2VXcZ95/shFWQ0RCk9DkG/cgvqD08/75mD6ABtNW6qDeVQKByXN1+kH86J/gH
# 3ptdhlEi6/PIIqcHMLPY+Gupi9kMzadm6i1kPDnTl+xwPobfC1/4/8PVkUpPLTsl
# 3I9ShlpirysJXRVcrxtPMsLukum2zFBkw6bqEFE6SeJFJ7AUlg/p5Q9qYOF1pg2p
# W21tI1/BvuU7HA2bssMdaV3G9IyABBkSrtU46QqNHRb9f2kSJA+0F3RWiYXsrQIM
# aRUK/6qVjFODEpFrbNN+62xMlVFy3VHgxYpBE/ryt/RbNu6VSYw=
# SIG # End signature block
