﻿<#
.SYNOPSIS
    
Provides operations and functions for the retrieval of KeyVault Keys and Passwords as
well as some other utilities for manipulating the data that is returned.

.DESCRIPTION

Provides a common module for Unified Operations Setup to retrieve and manipulate Azure Secret/Keys.

.PARAMETER VaultName

The Name of the KeyVault you want to retrieve the key from. Azure specifies that a 
KeyVault name must match this regex "^[a-zA-Z0-9-]{3,24}$".

.PARAMETER ApplicationID

The ApplicationID of the ServicePrincipal we are going to use to authenticate with.

.PARAMETER TenantID

The TenantID guid that Application is under.

.PARAMETER CertificateThumprint

The Thumprint of the certificate that is associated with the Application.

.EXAMPLE

C:\PS> Import-Module ".\KeyVault.psm1" -ArgumentList ("<KeyVaultName>", "<ApplicationID>", "<TenantID>", "<Thumbprint>")

#>
[CmdletBinding()]
param
(
    [Parameter(Mandatory=$False, Position = 0)]
    [System.String]
    $VaultName,

    [Parameter(Mandatory=$False, Position = 1)]
    [System.String]
    $ApplicationID,

    [Parameter(Mandatory=$False, Position = 2)]
    [System.String]
    $CertificateThumprint
)

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking

if (![System.String]::IsNullOrWhiteSpace($VaultName))
{
    $Script:VaultName = $VaultName
}
else
{
    $Script:VaultName = "default"
}

if (![System.String]::IsNullOrWhiteSpace($ApplicationID) -and 
    ![System.String]::IsNullOrWhiteSpace($CertificateThumprint))
{
    $Script:ApplicationID = $ApplicationID
    $Script:CertificateThumprint = $CertificateThumprint 
    Import-Module "$PSScriptRoot\DynamicsVault.psm1" -Force -DisableNameChecking
}

function Get-KeyVaultKey
{
    <#
    .SYNOPSIS
    
    Retrieves a specified Key from a KeyVault.

    .DESCRIPTION

    Retrieves a specified Key from KeyVault using the Azure PowerShell Modules and wraps
    the function up to provide an easier call. 

    .PARAMETER Name

    The Name of the Key in the selected vault we want to retrieve.

    .PARAMETER Version

    The Version of the Key we want, this is an optional field. The default Version
    is the latest one in KeyVault.

    .PARAMETER VaultName

    The Name of the KeyVault you want to retrieve the key from. This parameter is optional,
    it will default to the $Script:VaultName parameter. Azure specifies that a KeyVault name
    must match this regex "^[a-zA-Z0-9-]{3,24}$".

    .EXAMPLE

    Retrieves the latest version of the Key.
    
    C:\PS> Get-KeyVaultKey "KeyName"

    .EXAMPLE
    
    Retrieves the specific version of the Key.

    C:\PS> Get-KeyVaultKey "KeyName" "KeyVersion"

    .EXAMPLE

    Retrieves the latest version of the Key from a different KeyVault than what was passed
    from the script parameters.

    C:\PS> Get-KeyVaultKey -Name "KeyName" -VaultName "VaultName"
    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateLength(1,127)]
        [System.String]
        $Name,

        [Parameter(ParameterSetName = "Version", Mandatory=$False, Position = 1)]
        [System.String]
        $Version,

        [Parameter(Mandatory=$False, Position = 2)]
        [ValidatePattern("^[a-zA-Z0-9-]{3,24}$")]
        [System.String]
        $VaultName = $Script:VaultName
    )


    switch ($PsCmdlet.ParameterSetName)
    {
        "Version" 
        { 
            $keyInformation = Get-AzureKeyVaultKey -VaultName $VaultName -Name $Name -Version $Version
        }
        default
        {
            $keyInformation = Get-AzureKeyVaultKey -VaultName $VaultName -Name $Name
        }
    }
    return $keyInformation
}

function Get-KeyVaultSecret
{
    <#
    .SYNOPSIS
    
    Retrieves a specified secret from a KeyVault.

    .DESCRIPTION

    Retrieves a specified secret from KeyVault using the Azure PowerShell Modules and wraps
    the function up to provide an easier call. 

    .PARAMETER VaultUri

    The Name of the secret in the selected vault we want to retrieve. The assumption is the
    URI is formatted as such "Vault://SecretName/SecretVersion", if it does not match this
    pattern we assume the secret is the VaultUri parameter.

    .PARAMETER VaultName

    The Name of the KeyVault you want to retrieve the secret from. This parameter is optional,
    it will default to the $Script:VaultName parameter. Azure specifies that a KeyVault name
    must match this regex "^[a-zA-Z0-9-]{3,24}$".

    .EXAMPLE

    Retrieves the latest version of the secret.
    
    C:\PS> Get-KeyVaultSecret "VaultUri"

    .EXAMPLE

    Retrieves the latest version of the secret from a different KeyVault than what was passed
    from the script parameters.

    C:\PS> Get-KeyVaultSecret -VaultUri "Vault://SecretName/SecretVersion" -VaultName "VaultName"
    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [System.String]
        $VaultUri,

        [Parameter(Mandatory=$False, Position = 1)]
        [ValidatePattern("^[a-zA-Z0-9-]{3,24}$")]
        [System.String]
        $VaultName = $Script:VaultName
    )

    if (!(Test-ValidKeyVaultUri -VaultUri $VaultUri))
    {
        return $VaultUri
    }
    else
    {
        [System.Uri]$Uri = [System.Uri]::new($VaultUri)

        $secretName = $Uri.Host

        if ($Uri.Segments.Count -ge 2)
        {
            $secret = Get-VaultCredential -ClientId $Script:ApplicationID -CertificateThumbprint $Script:CertificateThumprint -VaultName $VaultName -SecretName $secretName -Version $Uri.Segments[1]
        }
        else
        {
            $secret = Get-VaultCredential -ClientId $Script:ApplicationID -CertificateThumbprint $Script:CertificateThumprint -VaultName $VaultName -SecretName $secretName
        }

        return $secret
    }
}

function Get-CertificatefromBase64String
{
    <#
    .SYNOPSIS

    Takes in a Base64 String representation of a Private Certificate (PFX) and the
    password associated with it and loads the certificate into a X509Certificate2 object.

    .DESCRIPTION

    Upon receiving the certificate information from KeyVault this allows us to convert
    the Base64 string and the password and load a certificate into a X509Certificate2 object.

    .PARAMETER Base64String

    The Base64 string representation of the PFX file.

    .PARAMETER CertificatePassword

    The password of the PFX Certificate.

    .EXAMPLE

    C:\PS> Create-CertificatefromBase64String "<Base64String>" "<Password>"

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $Base64String,

        [Parameter(Mandatory=$False, Position = 1)]
        [System.Security.SecureString]
        $CertificatePassword
    )

    $certificateBytes = [System.Convert]::FromBase64String($Base64String)

    if ($CertificatePassword -ne $null)
    { 
        return [System.Security.Cryptography.X509Certificates.X509Certificate2]::new($CertificateBytes, $CertificatePassword)       
    }
    else
    {
        return [System.Security.Cryptography.X509Certificates.X509Certificate2]::new($CertificateBytes)
    }
}

function Get-CertificateFromJsonString
{
    <#
    .SYNOPSIS

    Takes in a JSON String representation of a Certificate and attributes 
    and loads the certificate in memory.

    .DESCRIPTION

    This functions makes the assumption that the JSON file is formatted like this.

    {
        "data": "<Base64String>",
        "dataType" :"pfx",
        "password": "<CertificatePassword>"
    }

    This is following the same format that the ARM Templates expect Certificates for 
    VM Deployments.

    .PARAMETER JsonString

    The Json String string representation of the Certificate file and its attributes.

    .PARAMETER CertificateType

    This is a reference object so that the caller of the function can determine the
    type of the certificate for example (Public or Private key).

    .EXAMPLE

    C:\PS> [System.String]$certificateType = [System.String]::Empty
    C:\PS> $jsonObject = @"
    C:\PS> {
    C:\PS> "data": "<Base64EncodedCert>",
    C:\PS> "dataType" :"pfx",
    C:\PS> "password": "<CertificatePassword>"
    C:\PS> }
    C:\PS> Get-CertificateFromJsonString $jsonObject ([ref]$certificateType)

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $JsonString,

        [Parameter(Mandatory=$True, Position = 1)]
        [Ref]
        $CertificateType
    )

    $jsonObject = $JsonString | ConvertFrom-Json

    $CertificateType.Value = $jsonObject.dataType
    [System.Security.SecureString]$certificatePassword = $null

    if (![System.String]::IsNullOrWhiteSpace($jsonObject.password))
    {
        $certificatePassword = ConvertTo-SecureString $jsonObject.password -AsPlainText -Force
    }

    $certificate = Get-CertificatefromBase64String -Base64String $jsonObject.data -CertificatePassword $certificatePassword

    return $certificate
}

function Get-CertificateFromVault
{
    <#
    .SYNOPSIS

    Takes in two Vault Uri's. One is for the Certificate file (base64 encoded) and the
    second the password for the Certificate (Optional).

    .DESCRIPTION

    Once the information for the certificate has been retrieved we take the information and 
    create a x509Certificate2 object to be consumed.

    .PARAMETER CertUri

    The Uri of the certificate file we want.

    .PARAMETER CertPasswordUri

    The Uri of the certificate password that corresponds to the Certificate

    .PARAMETER VaultName

    The name of the Vault to retrieve the secret information.

    .EXAMPLE

    # Uses default VaultName
    C:\PS> Get-CertificateFromVault "vault://certSecretName" "vault://certSecretPasswordName"

    # overrides default VaultName
    C:\PS> Get-CertificateFromVault "vault://certSecretName" "vault://certSecretPasswordName" "VaultName"

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $CertUri,

        [Parameter(Mandatory=$false, Position = 1)]
        [System.String]
        $CertPasswordUri,

        [Parameter(Mandatory=$False, Position = 2)]
        [System.String]
        $VaultName = $Script:VaultName
    )

    $base64Cert = Get-KeyVaultSecret -VaultUri $CertUri -VaultName $VaultName

    if (![System.String]::IsNullOrWhiteSpace($CertPasswordUri))
    {
        $certificatePassword = ConvertTo-SecureString -String $(Get-KeyVaultSecret -VaultUri $CertPasswordUri -VaultName $VaultName) -AsPlainText -Force
    }

    return Get-CertificatefromBase64String -Base64String $base64Cert -CertificatePassword $certificatePassword
}

function Get-Base64UTF8String
{
    <#
    .SYNOPSIS

    Takes in a UTF8 Encoded Base64 String and converts it to native string.

    .DESCRIPTION

    This is a helper function to unwrap the Private Key that is stored for the
    ARM Template deployment.

    Azure requires this JSON Structure to be wrapped like this 

    $JsonBlob = @{
        "data": "<Base64String>",
        "dataType" :"pfx",
        "password": "<CertificatePassword>"
    }@

    $ContentBytes = [System.Text.Encoding]::UTF8.GetBytes($JSONBlob)
    $Content = [System.Convert]::ToBase64String($ContentBytes)

    This will unwrap the JSON File.

    .PARAMETER Base64String

    The UTF8 Base64String Object to unwrap.

    .EXAMPLE

    C:\PS> Read-Base64UTF8String <UTF8 Encoded Base64String>

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $Base64String
    )

    $ReceivedBytes = [System.Convert]::FromBase64String($Base64String)

    return [System.Text.Encoding]::UTF8.GetString($ReceivedBytes)
}

function Create-FilefromBase64String
{
    <#
    .SYNOPSIS
    
    Takes in a Base64 String representation of a File and
    Creates a file to the specified path.
    
    .DESCRIPTION

    Takes in a Base64 String representation of a File and
    Creates a file to the specified path.

    .PARAMETER Base64String

    The Base64 string representation of the PFX file. 

    .PARAMETER DestinationPath

    The Path of the file we want to save too. This includes the path and 
    file name i.e. (C:\Temp\Cert.pfx)

    .EXAMPLE

    C:\PS> Create-FilefromBase64String "<Base64String>" "C:\Temp\Cert.pfx"

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $Base64String,

        [Parameter(Mandatory=$True, Position = 1)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $DestinationPath
    )
    
    $pfxBytes = [System.Convert]::FromBase64String($Base64String)
    
    [System.IO.File]::WriteAllBytes($DestinationPath, $pfxBytes)
}

function Test-ValidKeyVaultUri
{
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [System.String]
        $VaultUri
    )

    if (![System.Uri]::IsWellFormedUriString($VaultUri, [System.UriKind]::Absolute))
    {
        return $false
    }

    [System.Uri]$Uri = [System.Uri]::new($VaultUri)

    if ($Uri.Scheme -ne "vault")
    {
        return $false
    }

    return $true
}
# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCYIEPfj2Yy48kJ
# gWEKNntz2Qhws7AgUPUErSkZHt+v0qCCDYEwggX/MIID56ADAgECAhMzAAAB32vw
# LpKnSrTQAAAAAAHfMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAxMjE1MjEzMTQ1WhcNMjExMjAyMjEzMTQ1WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQC2uxlZEACjqfHkuFyoCwfL25ofI9DZWKt4wEj3JBQ48GPt1UsDv834CcoUUPMn
# s/6CtPoaQ4Thy/kbOOg/zJAnrJeiMQqRe2Lsdb/NSI2gXXX9lad1/yPUDOXo4GNw
# PjXq1JZi+HZV91bUr6ZjzePj1g+bepsqd/HC1XScj0fT3aAxLRykJSzExEBmU9eS
# yuOwUuq+CriudQtWGMdJU650v/KmzfM46Y6lo/MCnnpvz3zEL7PMdUdwqj/nYhGG
# 3UVILxX7tAdMbz7LN+6WOIpT1A41rwaoOVnv+8Ua94HwhjZmu1S73yeV7RZZNxoh
# EegJi9YYssXa7UZUUkCCA+KnAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUOPbML8IdkNGtCfMmVPtvI6VZ8+Mw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDYzMDA5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAnnqH
# tDyYUFaVAkvAK0eqq6nhoL95SZQu3RnpZ7tdQ89QR3++7A+4hrr7V4xxmkB5BObS
# 0YK+MALE02atjwWgPdpYQ68WdLGroJZHkbZdgERG+7tETFl3aKF4KpoSaGOskZXp
# TPnCaMo2PXoAMVMGpsQEQswimZq3IQ3nRQfBlJ0PoMMcN/+Pks8ZTL1BoPYsJpok
# t6cql59q6CypZYIwgyJ892HpttybHKg1ZtQLUlSXccRMlugPgEcNZJagPEgPYni4
# b11snjRAgf0dyQ0zI9aLXqTxWUU5pCIFiPT0b2wsxzRqCtyGqpkGM8P9GazO8eao
# mVItCYBcJSByBx/pS0cSYwBBHAZxJODUqxSXoSGDvmTfqUJXntnWkL4okok1FiCD
# Z4jpyXOQunb6egIXvkgQ7jb2uO26Ow0m8RwleDvhOMrnHsupiOPbozKroSa6paFt
# VSh89abUSooR8QdZciemmoFhcWkEwFg4spzvYNP4nIs193261WyTaRMZoceGun7G
# CT2Rl653uUj+F+g94c63AhzSq4khdL4HlFIP2ePv29smfUnHtGq6yYFDLnT0q/Y+
# Di3jwloF8EWkkHRtSuXlFUbTmwr/lDDgbpZiKhLS7CBTDj32I0L5i532+uHczw82
# oZDmYmYmIUSMbZOgS65h797rj5JJ6OkeEUJoAVwwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAd9r8C6Sp0q00AAAAAAB3zAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgvG+WBvkA
# 14iO9myC8yOWc0R8jNV8VZprDIn9UoQxvSAwQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQCVClglFF7TDZ5NecrY0kBalDKKnyks7IS2ta35GcM4
# Jt7/VN9kv0AlO5hDy2xY5kzsl8HaNw8NAN9ANGyiVKTcNsHw+Z614Q4a/LSv+naI
# 4qVqsDx3BON8vAGdfrzumgSIh2vrqnJDO/I5o+8hM3SjTekQTakj7UxWK9fWSCq3
# +eFXenGWRpFm0pEPNBea24uFKwWd6Nc6qln34DhRZXB85cTzswMpvoVkFycFawUb
# Sk1zCxGpM6LX8/XL1cJLjbs4YJ5gUWbX7PCzpEmVfnrftIzdewt5YsUDq+6hQwuv
# g3AY/9Mvqt8KB3qslnHeE9JuD4oKBH8MfonYKWiyQtmfoYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIKqG+jMW6dvxJpXVLG96ZwaIkMIK5h1LcKVdouox
# F/oIAgZhHrHUkAkYEzIwMjEwODIzMDcyMTAwLjk5NVowBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjpEOURFLUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABYfWiM16gKiRpAAAA
# AAFhMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTIxMDExNDE5MDIyMVoXDTIyMDQxMTE5MDIyMVowgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpEOURF
# LUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJeInahBrU//GzTqhxUy
# AC8UXct6UJCkb2xEZKV3gjggmLAheBrxJk7tH+Pw2tTcyarLRfmV2xo5oBk5pW/O
# cDc/n/TcTeQU6JIN5PlTcn0C9RlKQ6t9OuU/WAyAxGTjKE4ENnUjXtxiNlD/K2ZG
# MLvjpROBKh7TtkUJK6ZGWw/uTRabNBxRg13TvjkGHXEUEDJ8imacw9BCeR9L6und
# r32tj4duOFIHD8m1es3SNN98Zq4IDBP3Ccb+HQgxpbeHIUlK0y6zmzIkvfN73Zxw
# fGvFv0/Max79WJY0cD8poCnZFijciWrf0eD1T2/+7HgewzrdxPdSFockUQ8QovID
# IYkCAwEAAaOCARswggEXMB0GA1UdDgQWBBRWHpqd1hv71SVj5LAdPfNE7PhLLzAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQAQTA9bqVBmx5TTMhzj+Q8zWkPQXgCc
# SQiqy2YYWF0hWr5GEiN2LtA+EWdu1y8oysZau4CP7SzM8VTSq31CLJiOy39Z4RvE
# q2mr0EftFvmX2CxQ7ZyzrkhWMZaZQLkYbH5oabIFwndW34nh980BOY395tfnNS/Y
# 6N0f+jXdoFn7fI2c43TFYsUqIPWjOHJloMektlD6/uS6Zn4xse/lItFm+fWOcB2A
# xyXEB3ZREeSg9j7+GoEl1xT/iJuV/So7TlWdwyacQu4lv3MBsvxzRIbKhZwrDYog
# moyJ+rwgQB8mKS4/M1SDRtIptamoTFJ56Tk6DuUXx1JudToelgjEZPa5MIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpE
# OURFLUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUAFW5ShAw5ekTEXvL/4V1s0rbDz3mggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AOTNJLYwIhgPMjAyMTA4MjIyMzMyMDZaGA8yMDIxMDgyMzIzMzIwNlowdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA5M0ktgIBADAKAgEAAgIUtgIB/zAHAgEAAgIRHjAK
# AgUA5M52NgIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBAKKmsqxZtAZnR0OA
# w7NGBFN/Js6YhRew8zhb6/DpO3vUAqrKjQ1Ly7UoKm48GMWNC59qUUBaZcH0NvAu
# GK6/flzwccLYgICDOaJ7QUY81DkIqpWExWH6pMRj5iYBw/EhI6WGUjoXqPw7kl+8
# 5lXQ7cHGwgZeCPu75LvfwDslY7rcMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAFh9aIzXqAqJGkAAAAAAWEwDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQgCenvRj4pd4xrw01DyGTf0hIR/z3+xrRmoWuT8IOHNiEwgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCBhz4un6mkSLd/zA+0N5YLDGp4vW/VBtNW/lpmh
# tAk4bzCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# YfWiM16gKiRpAAAAAAFhMCIEIEbIt6Ygk+/l43GhkFC+e8Aa0iWxxDdy3EcJIAcP
# Z/ThMA0GCSqGSIb3DQEBCwUABIIBAAWrjT+Fm0czAEvAF+issw5g5kz/Qon7xIDO
# kvKG+c64E2QGsFzkEeeF3XUJ3eu2vWuLvpfjlJakQ1rni5LLDjK45FM+8JPqDJHt
# 881IItm4kOLBDMmzNNa1u5FdKNz8+dnfmqrcjnahzhdGxWHnkjWWuQIXNA0q4R7F
# pzDAyyp8oBNz58YofFZh6GNv6k75ni2PHWavTtrEpDk0hW2jDhEMujo+R1mQqHfS
# WS0e6mRVcA0VlsGbB02soI+FmZyyGOydaE07lHlFAuRJliW+7f1O/+M+enQEItQx
# KkvLhhxUoVSUcy/opCK9PpMZJ7/zxiYKRTQOg09jjeldw4cCWGQ=
# SIG # End signature block
