﻿[CmdletBinding()]
param
(
    [Parameter(Mandatory=$false)]
    [string]$LogDir,

    [parameter(Mandatory=$false)]
    [boolean] $useServiceFabric=$false,

    [Parameter(Mandatory = $false)]
    [string]$ActivityId,

    [Parameter(Mandatory = $false)]
    [string]$RunbookId,

    [Parameter(Mandatory=$false)]
	[hashtable]$AxDRConnectionString,

    [Parameter(Mandatory=$false)]
	[string[]]$EnabledServicingFeatures,

    [Parameter(Mandatory=$false)]
	[Switch]$PreDowntime,

    [Parameter(Mandatory=$false)]
    [hashtable]$Credentials
)

# Pre-defined list of error messages for known customer induced issues
$knownErrorHash = @{
    "The CREATE UNIQUE INDEX statement terminated because a duplicate key" = 131079
    "Please drop the original field, sync the table" = 131089
    "extension cannot be applied to the base element" = 131090
    "is missing the following dependencies" = 131091
    "Method not found" = 131092
}

# Check pre check status
function IsPreCheckSucceeded ([string]$ActivityId)
{
  if(!$ActivityId)
  {
    return $false
  }

  $axDBUser = Get-DataAccessSqlUsr

  if($Credentials -and $Credentials[$($axDBUser)])  {
     $axDBPwd = $Credentials[$($axDBUser)]
     EventWrite-RunbookScriptTrace -Message "Using axdbadmin password from DSU parameters." -Component "servicing" -RunbookId $RunbookId| Out-Null
  }
  else{
     $axDBPwd = Get-DataAccessSqlPwd
     EventWrite-RunbookScriptTrace -Message "Using axdbadmin password from Web config." -Component "servicing" -RunbookId $RunbookId| Out-Null
  }
  
  $axDBDatabaseName = Get-DataAccessDatabase
  $axDBServer = Get-DataAccessDbServer

    $target_sqlParams = @{
		'Database' = $axDBDatabaseName
		'UserName' = $axDBUser
		'Password' = $axDBPwd
		'ServerInstance' = $axDBServer
		'EncryptConnection' = !(IsLocalServer($axDBServer))
		'QueryTimeout' = 0
	    }

    $target_sqlParams.Query = "Select top 1 Success from DBSyncExecStats where SyncStep = 'PreCheck' and RelatedActivityId = '$ActivityId' order by StartDateTime desc"

    try{
        # during first precheck on this environment table will not be exists so catch exception and consider running precheck.
        $result = Invoke-SqlCmd @target_sqlParams
    }
    catch
    {
        $warningMessage = "[$(Get-Date)]:Get pre check status failed, activity id: $ActivityId, Exception: [$($_.Exception)]"
        Write-Host $warningMessage
        EventWrite-RunbookScriptTrace -Message $warningMessage -Component "servicing" -RunbookId $RunbookId| Out-Null
    }

    if ($result -ne $null)
    {
        return $result.Success
    }

    return $false
}

# Run db sync precheck to evaluate DB Sync would fail or succeed.
function RunDBSyncPreCheck([string]$ActivityId, [hashtable]$AxDRConnectionString, [string[]]$EnabledServicingFeatures)
{
    if (!$EnabledServicingFeatures -or $EnabledServicingFeatures -notcontains "DBSyncPreCheck")
    { 
       EventWrite-RunbookScriptTrace -Message "DBSyncPreCheck feature is disabled. Skipping db sync pre check, activity id: $ActivityId" -Component "servicing"  -RunbookId $RunbookId| Out-Null
       return $false
    }

    # Check if previous precheck status for activity id
    if($ActivityId)
    {
        $PreCheckResult = IsPreCheckSucceeded -ActivityId $ActivityId
        if($PreCheckResult -eq $true) 
        {
           # For prod scenario precheck will run during preparation, so we do not need to run it again.
           EventWrite-RunbookScriptTrace -Message "[$(Get-Date)]: Pre service check already succeeded for activity id: $ActivityId, Skipping precheck..." -Component "servicing" -RunbookId $RunbookId| Out-Null
           return $false
        }
     }

    # Run Precheck
    $currentDateTime = [DateTime]::UtcNow.ToString("yyyyMMddHHmmss")
    $logfile = Join-Path $LogDir "dbsyncprecheck_$currentDateTime.log"
    $errorfile = Join-Path $LogDir "dbsyncprecheck.error_$currentDateTime.log"
    Write-Host "Precheck Started..."
   
    if($PreDowntime -eq $true)
    {
      $aosWebServiceStagingPath = Get-AosServiceStagingPath
      $webroot = Join-Path -Path $aosWebServiceStagingPath -ChildPath "WebRoot"
      $PUVersion = Get-ProductPlatformBuildVersion -webroot:$webroot

      $metadatadir = Join-Path -Path $aosWebServiceStagingPath -ChildPath "PackagesLocalDirectory"
    }
    else
    {
      $PUVersion = (Get-ProductPlatformBuildVersion)
      $metadatadir = $(Get-AOSPackageDirectory)
    }

    Write-Host "AOS platform version is: $PUVersion"

    # If PU version >= PU34 then run precheck on secondary DB if exists.
    $runPrecheckOnSecondaryDB = $false
    if($PUVersion -ge 5629 -and $AxDRConnectionString -and $AxDRConnectionString['AxDRServerName'] -and $AxDRConnectionString['AxDRDatabaseName'])
    {
        $runPrecheckOnSecondaryDB =  $true
        $sqlServer =  $AxDRConnectionString['AxDRServerName']
        $sqlDB =  $AxDRConnectionString['AxDRDatabaseName']
        EventWrite-RunbookScriptTrace -Message "[$(Get-Date)]: Running precheck on secondary DB" -Component "servicing" -RunbookId $RunbookId| Out-Null
    }

    # If platform version > PU35 then run precheck with deployment setup exe
    if ($PUVersion -gt 5644)
    {
        EventWrite-RunbookScriptTrace -Message "Running precheck with deployment setup exe, activity id $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null
        $deploymentSetupParameter = "-setupmode preservicecheck -syncmode precheck"
        
        if($ActivityId) 
        {
            $deploymentSetupParameter += " -activityid $ActivityId"
        }

        $deploySetupPs1 = Resolve-Path -Path $deploySetupPs1
        & $deploySetupPs1 -deploymentSetupParameter $deploymentSetupParameter -logfile $logfile -errorfile $errorfile -axDRServerName $sqlServer -axDRDatabaseName $sqlDB -isPreDowntime $PreDowntime | Out-Null
        
        EventWrite-RunbookScriptTrace -Message "[$(Get-Date)]: Pre service check completed using deployment setup for activity id $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null
        return $true
    }
    
    EventWrite-RunbookScriptTrace -Message "Running precheck on Metadata dir: $metadatadir" -Component "servicing" -RunbookId $RunbookId| Out-Null

    # If platform version <= PU35 Run precheck with sync engine exe
    EventWrite-RunbookScriptTrace -Message "Running precheck with sync engine exe, activity id $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null

    $sqlUser = Get-DataAccessSqlUsr
    if($Credentials -and $Credentials[$($sqlUser)]) {
      $sqlPwd = $Credentials[$($sqlUser)]
    }
    else {
      $sqlPwd = Get-DataAccessSqlPwd
    }

    if($runPrecheckOnSecondaryDB -eq $false){
        $sqlServer = Get-DataAccessDbServer
        $sqlDB = Get-DataAccessDatabase
    }

    $connectionString = "Data Source=$($sqlServer); " +
        "Integrated Security=False; " +
        "User Id=$($sqlUser); " +
        "Password='$($sqlPwd)'; " +
        "Initial Catalog=$($sqlDB)"
    
    $command = "$metadatadir\bin\syncengine.exe"
    $commandParameter = "-syncmode=precheck"
    $commandParameter += " -metadatabinaries=$metadatadir"
    $commandParameter +=  " -connect=`"$connectionString`""
    $commandParameter +=  " -enableParallelSync"
                
    if($ActivityId)
    {
        $commandParameter += " -relatedactivityid=$ActivityId"
    }

    $process = Start-Process $command $commandParameter -PassThru -Wait -RedirectStandardOutput "$logfile" -RedirectStandardError "$errorfile"
    Write-Host "Log file location: $logfile"
        
    if ($process.ExitCode -ne 0) {
        $deploymentSetupError = Get-Content "$errorfile"
        throw $deploymentSetupError
    }

    EventWrite-RunbookScriptTrace -Message "[$(Get-Date)]: Pre service check completed using sync engine for activity id $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null
    return $true
}

$ErrorActionPreference = "stop"
Import-Module "$PSScriptRoot\AosEnvironmentUtilities.psm1" -Force -DisableNameChecking
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -ArgumentList $useServiceFabric -DisableNameChecking

if(!$LogDir)
{
    $LogDir = $PSScriptRoot
}

$deploySetupPs1 = "$PSScriptRoot\TriggerDeploymentSetupEngine.ps1"

# PreCheck
try
{
    # This returns false if pre check skipped.
    $result = RunDBSyncPreCheck -ActivityId $ActivityId -AxDRConnectionString $AxDRConnectionString -EnabledServicingFeatures $EnabledServicingFeatures
   
    if($result -eq $true)
    {   
        $message = "[$(Get-Date)]: Pre service check succeeded for activity id: $ActivityId."
    }
    else
    {
      $message = "[$(Get-Date)]: Pre service check did not run, activity id: $ActivityId."
    }

    Write-Host $message
    EventWrite-RunbookScriptTrace -Message $message -Component "servicing" -RunbookId $RunbookId| Out-Null
}
catch
{
    $errorMessage = "[$(Get-Date)]: Pre service check failed for activity id: $ActivityId, Exception: [$($_.Exception)]"    
    EventWrite-RunbookScriptTrace -Message $errorMessage -Component "servicing" -RunbookId $RunbookId | Out-Null

    if ($EnabledServicingFeatures -and $EnabledServicingFeatures -contains "FailDBSyncOnPrecheckFailure")
    { 
       EventWrite-RunbookScriptTrace -Message "FailDBSyncOnPrecheckFailure feature is enabled and db sync pre-check is failed, activity id: $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null

       # Only fail in dbsync pre-check if the error message contains one of the known error strings
       foreach ($knownError in $knownErrorHash.GetEnumerator())
       {
          if($errorMessage.Contains($knownError.Name))
          {
            # In case precheck failure due to a known customer-induced error scenario we do not require PITR during rollback. In precheck success scenario full sync will run and that will calculate PITR required or not.
            Write-IsPITRRequiredDuringRollback -PitrRequired 'false'

            EventWrite-RunbookScriptTrace -Message "DBSync pre-check failed due to a known customer induced error, error code: $($knownError.Value), activity id: $ActivityId" -Component "servicing" -RunbookId $RunbookId| Out-Null
            throw $errorMessage
          }
       }
    }
}
# SIG # Begin signature block
# MIIjgQYJKoZIhvcNAQcCoIIjcjCCI24CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDf8pNtqZY7VDtd
# JiquVayV4+X8rN+4YqEWHTNh3X0wYqCCDYEwggX/MIID56ADAgECAhMzAAAB32vw
# LpKnSrTQAAAAAAHfMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAxMjE1MjEzMTQ1WhcNMjExMjAyMjEzMTQ1WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQC2uxlZEACjqfHkuFyoCwfL25ofI9DZWKt4wEj3JBQ48GPt1UsDv834CcoUUPMn
# s/6CtPoaQ4Thy/kbOOg/zJAnrJeiMQqRe2Lsdb/NSI2gXXX9lad1/yPUDOXo4GNw
# PjXq1JZi+HZV91bUr6ZjzePj1g+bepsqd/HC1XScj0fT3aAxLRykJSzExEBmU9eS
# yuOwUuq+CriudQtWGMdJU650v/KmzfM46Y6lo/MCnnpvz3zEL7PMdUdwqj/nYhGG
# 3UVILxX7tAdMbz7LN+6WOIpT1A41rwaoOVnv+8Ua94HwhjZmu1S73yeV7RZZNxoh
# EegJi9YYssXa7UZUUkCCA+KnAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUOPbML8IdkNGtCfMmVPtvI6VZ8+Mw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDYzMDA5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAnnqH
# tDyYUFaVAkvAK0eqq6nhoL95SZQu3RnpZ7tdQ89QR3++7A+4hrr7V4xxmkB5BObS
# 0YK+MALE02atjwWgPdpYQ68WdLGroJZHkbZdgERG+7tETFl3aKF4KpoSaGOskZXp
# TPnCaMo2PXoAMVMGpsQEQswimZq3IQ3nRQfBlJ0PoMMcN/+Pks8ZTL1BoPYsJpok
# t6cql59q6CypZYIwgyJ892HpttybHKg1ZtQLUlSXccRMlugPgEcNZJagPEgPYni4
# b11snjRAgf0dyQ0zI9aLXqTxWUU5pCIFiPT0b2wsxzRqCtyGqpkGM8P9GazO8eao
# mVItCYBcJSByBx/pS0cSYwBBHAZxJODUqxSXoSGDvmTfqUJXntnWkL4okok1FiCD
# Z4jpyXOQunb6egIXvkgQ7jb2uO26Ow0m8RwleDvhOMrnHsupiOPbozKroSa6paFt
# VSh89abUSooR8QdZciemmoFhcWkEwFg4spzvYNP4nIs193261WyTaRMZoceGun7G
# CT2Rl653uUj+F+g94c63AhzSq4khdL4HlFIP2ePv29smfUnHtGq6yYFDLnT0q/Y+
# Di3jwloF8EWkkHRtSuXlFUbTmwr/lDDgbpZiKhLS7CBTDj32I0L5i532+uHczw82
# oZDmYmYmIUSMbZOgS65h797rj5JJ6OkeEUJoAVwwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVVjCCFVICAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAd9r8C6Sp0q00AAAAAAB3zAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgLPWykckg
# eyzPZUVRJPlSm+4ktAGi5YWFWPIyVder8t4wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQA0C2AK5JRAEGrycImUibefz4SXlEo0bLHafd8eXnif
# sKDiliK4E7RBCo1pLoqPcLgvITxzEXC8PnDY6yU3bTEWdBqssRUrDFebNa7cCXQp
# Z4wrh9BJsP1Zi+eHM5RT3ZzI+CkyznnhmZNSk/iyjsvlNWgS+inFanqUIoynzWSB
# gYK8Vrh7en7zJmkDpUSA1delGgd5TVcf5AH44NmqBUSb7CYxWLi04eyU4MtNiF6A
# ubLsGsoHPii7fGNuFradpcsbW7fPsboldaQQKIyd6qZRevOFvlh9RSfTcdQZzxhL
# SCjzm1UkOLmzIKhO2HI4bYpqTwUo+imIABbp29MYaqDpoYIS4DCCEtwGCisGAQQB
# gjcDAwExghLMMIISyAYJKoZIhvcNAQcCoIISuTCCErUCAQMxDzANBglghkgBZQME
# AgEFADCCAU8GCyqGSIb3DQEJEAEEoIIBPgSCATowggE2AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIDcJyXrcCohvfoRXaaty8c6Hrek0L4ZQ06eoiYEw
# XT3HAgZg+YSg8EEYETIwMjEwNzMwMTc0NzMzLjNaMASAAgH0oIHQpIHNMIHKMQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQLExxNaWNy
# b3NvZnQgQW1lcmljYSBPcGVyYXRpb25zMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVT
# TjpFNUE2LUUyN0MtNTkyRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAg
# U2VydmljZaCCDjkwggTxMIID2aADAgECAhMzAAABR52P8ebeMYNZAAAAAAFHMA0G
# CSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMB4XDTIw
# MTExMjE4MjU1NVoXDTIyMDIxMTE4MjU1NVowgcoxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xJTAjBgNVBAsTHE1pY3Jvc29mdCBBbWVyaWNhIE9w
# ZXJhdGlvbnMxJjAkBgNVBAsTHVRoYWxlcyBUU1MgRVNOOkU1QTYtRTI3Qy01OTJF
# MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNlMIIBIjANBgkq
# hkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEArQUDTOl99ihZPwSfGGqa38xLei49+Bvl
# S484HxIDhklXdTLu5wqYStCuKMj68yLYDUYweIFIiquhs4MXQx4GT4ExL5ue87Gr
# HMhq8m1pH91Va/G6n9jyIBr8CgzRIFXMoLBG7lsF8/S4ujOwDqA+EwfH5eAE5vi+
# 2+PpWA1DCWDC86YiczO+OWtFx4q4lGjqWFn5MDvR8+rn/pNh6u8hsoLh/J2jyzJ2
# H5mCLdj1wMFlbxuDA+GG41wVWdeLnoe2JamUzLmt5/ZE9QmQ7B78/qOfJ7KGpl0c
# ERTm+yw41VfvnGSGSztvbrIiNB+/bW930r4fbVO0AuwoqzQgWGoDSQIDAQABo4IB
# GzCCARcwHQYDVR0OBBYEFLT43G/eZKGGVxsvQz8TePXUgrC6MB8GA1UdIwQYMBaA
# FNVjOlyKMZDzQ3t8RhvFM2hahW1VMFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6Ly9j
# cmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1RpbVN0YVBDQV8y
# MDEwLTA3LTAxLmNybDBaBggrBgEFBQcBAQROMEwwSgYIKwYBBQUHMAKGPmh0dHA6
# Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljVGltU3RhUENBXzIwMTAt
# MDctMDEuY3J0MAwGA1UdEwEB/wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwgwDQYJ
# KoZIhvcNAQELBQADggEBAFDEDso1fnmtpBLV6g9HDnw9mdjxVCMuZC5BlOd0QxnH
# 4ZcyEfUJU4GS6kxIYCy7gksuo3Jmvpz4O3uV4NaKgmXGSUcLUT80tRevzOmEd+R9
# 26zdJmlZz65q0PdZJH7Dag07blg4gCAX5DRIAqUGQm+DldxzLuS/Hmc4OXVGie1x
# PiwbqYSUixSbWm8WLeH5AkMZ0w9s+dGN9mbl4jZhRYmu6lavt2HN5pltNvvylh/D
# 9Rz4qGyRvHKxmIEhJMbZFFxnaLrcBllvPCkfUUXDP+Bii6rdjkPAFhEG+7IYDPKc
# Tb7t9E8Xo0QKWuFgHzgHo0xuPiswZuu141WdiJVwD5IwggZxMIIEWaADAgECAgph
# CYEqAAAAAAACMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UE
# CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
# b2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZp
# Y2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0xMDA3MDEyMTM2NTVaFw0yNTA3MDEyMTQ2
# NTVaMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
# EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNV
# BAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIIBIjANBgkqhkiG9w0B
# AQEFAAOCAQ8AMIIBCgKCAQEAqR0NvHcRijog7PwTl/X6f2mUa3RUENWlCgCChfvt
# fGhLLF/Fw+Vhwna3PmYrW/AVUycEMR9BGxqVHc4JE458YTBZsTBED/FgiIRUQwzX
# Tbg4CLNC3ZOs1nMwVyaCo0UN0Or1R4HNvyRgMlhgRvJYR4YyhB50YWeRX4FUsc+T
# TJLBxKZd0WETbijGGvmGgLvfYfxGwScdJGcSchohiq9LZIlQYrFd/XcfPfBXday9
# ikJNQFHRD5wGPmd/9WbAA5ZEfu/QS/1u5ZrKsajyeioKMfDaTgaRtogINeh4HLDp
# mc085y9Euqf03GS9pAHBIAmTeM38vMDJRF1eFpwBBU8iTQIDAQABo4IB5jCCAeIw
# EAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFNVjOlyKMZDzQ3t8RhvFM2hahW1V
# MBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1UdDwQEAwIBhjAPBgNVHRMB
# Af8EBTADAQH/MB8GA1UdIwQYMBaAFNX2VsuP6KJcYmjRPZSQW9fOmhjEMFYGA1Ud
# HwRPME0wS6BJoEeGRWh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3By
# b2R1Y3RzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNybDBaBggrBgEFBQcBAQRO
# MEwwSgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2Vy
# dHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3J0MIGgBgNVHSABAf8EgZUwgZIw
# gY8GCSsGAQQBgjcuAzCBgTA9BggrBgEFBQcCARYxaHR0cDovL3d3dy5taWNyb3Nv
# ZnQuY29tL1BLSS9kb2NzL0NQUy9kZWZhdWx0Lmh0bTBABggrBgEFBQcCAjA0HjIg
# HQBMAGUAZwBhAGwAXwBQAG8AbABpAGMAeQBfAFMAdABhAHQAZQBtAGUAbgB0AC4g
# HTANBgkqhkiG9w0BAQsFAAOCAgEAB+aIUQ3ixuCYP4FxAz2do6Ehb7Prpsz1Mb7P
# BeKp/vpXbRkws8LFZslq3/Xn8Hi9x6ieJeP5vO1rVFcIK1GCRBL7uVOMzPRgEop2
# zEBAQZvcXBf/XPleFzWYJFZLdO9CEMivv3/Gf/I3fVo/HPKZeUqRUgCvOA8X9S95
# gWXZqbVr5MfO9sp6AG9LMEQkIjzP7QOllo9ZKby2/QThcJ8ySif9Va8v/rbljjO7
# Yl+a21dA6fHOmWaQjP9qYn/dxUoLkSbiOewZSnFjnXshbcOco6I8+n99lmqQeKZt
# 0uGc+R38ONiU9MalCpaGpL2eGq4EQoO4tYCbIjggtSXlZOz39L9+Y1klD3ouOVd2
# onGqBooPiRa6YacRy5rYDkeagMXQzafQ732D8OE7cQnfXXSYIghh2rBQHm+98eEA
# 3+cxB6STOvdlR3jo+KhIq/fecn5ha293qYHLpwmsObvsxsvYgrRyzR30uIUBHoD7
# G4kqVDmyW9rIDVWZeodzOwjmmC3qjeAzLhIp9cAvVCch98isTtoouLGp25ayp0Ki
# yc8ZQU3ghvkqmqMRZjDTu3QyS99je/WZii8bxyGvWbWu3EQ8l1Bx16HSxVXjad5X
# wdHeMMD9zOZN+w2/XU/pnR4ZOC+8z1gFLu8NoFA12u8JJxzVs341Hgi62jbb01+P
# 3nSISRKhggLLMIICNAIBATCB+KGB0KSBzTCByjELMAkGA1UEBhMCVVMxEzARBgNV
# BAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
# c29mdCBDb3Jwb3JhdGlvbjElMCMGA1UECxMcTWljcm9zb2Z0IEFtZXJpY2EgT3Bl
# cmF0aW9uczEmMCQGA1UECxMdVGhhbGVzIFRTUyBFU046RTVBNi1FMjdDLTU5MkUx
# JTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2WiIwoBATAHBgUr
# DgMCGgMVAKunwbRBDaHDQEjKi9N8xiUtMGXdoIGDMIGApH4wfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTAwDQYJKoZIhvcNAQEFBQACBQDkro7WMCIYDzIwMjEw
# NzMwMjI0NDM4WhgPMjAyMTA3MzEyMjQ0MzhaMHQwOgYKKwYBBAGEWQoEATEsMCow
# CgIFAOSujtYCAQAwBwIBAAICDjkwBwIBAAICEfIwCgIFAOSv4FYCAQAwNgYKKwYB
# BAGEWQoEAjEoMCYwDAYKKwYBBAGEWQoDAqAKMAgCAQACAwehIKEKMAgCAQACAwGG
# oDANBgkqhkiG9w0BAQUFAAOBgQBYlnKBzvwosm8bCyMjRL0g219Hs+oUDKdqNMSi
# 8U3loByZEQTF1Vj0zrAItMfg+d1XQ8jJpbgwN6UZodGx+GmsnHvQR3AsT6XT4H9I
# mOZnE0HriO31tt3Vg6Vg/hwvgEmbrTUw1jJ36s8P8+feWgz3NSwQUMxQE3IJRvwv
# xQratjGCAw0wggMJAgEBMIGTMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# AhMzAAABR52P8ebeMYNZAAAAAAFHMA0GCWCGSAFlAwQCAQUAoIIBSjAaBgkqhkiG
# 9w0BCQMxDQYLKoZIhvcNAQkQAQQwLwYJKoZIhvcNAQkEMSIEILlLwNjTG06ZnZ3O
# jpP7zgaI/JZxAsL7MdK+ifVgg5IdMIH6BgsqhkiG9w0BCRACLzGB6jCB5zCB5DCB
# vQQge9s8EgOUz7oGyImTv9h9Ya4rAFSLcMw7HG9FqooY6YUwgZgwgYCkfjB8MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNy
# b3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAUedj/Hm3jGDWQAAAAABRzAi
# BCBOu8zfVjWTfZNYBUSOINEyKw0b/Ff2xszmyeEKUlOs0DANBgkqhkiG9w0BAQsF
# AASCAQAjXl/CfMdJscPPuq1sWBx9EfUk3fTdwfr+1tyFxvpTVTcceRvjIW4Usg7l
# +O+ZsW7G+cnqtbDEwRww5qxhjUL/XupsnpDq0xbP+NtjZHDQ+XwugXc7z9MTkcVH
# KZvDvlA6qJJP38PXEQAW1lh0HGv6a9uZ3eMnc9eDfALtxU0O9IxL7NSJzhZUNLD2
# P3WkXa1Kx/omlytwr9VYTvRz4QeZQo8A3vIOSIplDhgHi0yls7AJGmMBIsl+473c
# fqJgXz960mm4w1ehWGT9zCkOM+xayWA3HZLjVVCCARMy5ohcGRyn/InSyCDxJFJr
# gxd8Je/QEPDXn64IKuewXffYdR1h
# SIG # End signature block
