﻿param
(
    [parameter(Position=0,Mandatory=$false)][boolean] $useServiceFabric=$false 
)

Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1"  -ArgumentList $useServiceFabric -DisableNameChecking

$script:EnvDllsLoaded = $false
function Get-ApplicationEnvironment
{
    $ErrorActionPreference = 'Stop'

    if (-not $script:EnvDllsLoaded)
    {        
        $webroot = Get-AosWebSitePhysicalPath
        $InstalledPlatformBuildVersion = Get-ProductPlatformBuildFullVersion
        $isKeyVault = Get-IsMSIKeyVault

        if (($isKeyVault -eq $true) -and ($InstalledPlatformBuildVersion -ne $null) -and ($InstalledPlatformBuildVersion.Build -ge 5644))
        {        
            # Get-PackageRoot comes from CommonRollbackUtilities.psm1. it resolves the root of the deployable package
            $packageRoot = Get-PackageRoot
            $packageTemp = ""

            if(![string]::IsNullOrEmpty($packageRoot) -and (Test-Path $packageRoot))
            {
                $packageTemp = Join-Path $packageRoot "RunbookWorkingFolder\KeyVaultAssemblyTemp"
            }

            if ([string]::IsNullOrEmpty($packageTemp))
            {   
                $suffix = $(Get-Date).ToString("yyyyMMdd_HHmmss")
                $dirName = "AssemblyTemp_$suffix"
                $packageTemp = Join-Path $env:Temp "dirName"
            }

            if(!(Test-Path $packageTemp -PathType Container))
            {
                New-Item $packageTemp -ItemType Container | Out-Null
            }
        
            #need to load the Microsoft.Dynamics.ApplicationPlatform.Environment.dll and all the dll it referenced
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Configuration.Base.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.Instrumentation.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Framework.EncryptionEngine.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.BusinessPlatform.SharedTypes.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.KeyVaultHelper.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Carbon.Flighting.Runtime.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Carbon.CertificateCommon.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Commerce.Flighting.Common.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Flighting.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.XppServices.Instrumentation.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.CE.VaultSDK.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.CE.VaultSDK.ADALv2.AuthHelper.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.KeyVault.Core.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.KeyVault.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.Common.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.Common.NetFramework.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Hyak.Common.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Authentication.Instrumentation.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.Extensions.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.Extensions.Desktop.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.IdentityModel.Clients.ActiveDirectory.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Practices.EnterpriseLibrary.TransientFaultHandling.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Newtonsoft.Json.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.SharedUtility.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp            

            try
            {
                $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
                $fileTempPath =  Join-Path $packageTemp $(Get-Item $dllPath).Name
                if (Test-Path $fileTempPath)
                {
                    # Load file from temp folder
                    [System.Reflection.Assembly]::LoadFrom($fileTempPath)
                }
            }
            catch
            {    
                #write-log "Failed to load from temp folder with Exception: [$($_.Exception)]"
                throw $_
            }
        }              
        else
        {
            #need to load the Microsoft.Dynamics.ApplicationPlatform.Environment.dll and all the dll it referenced
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
            Load-DllinMemory -dllPath $dllPath
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Configuration.Base.dll'
            Load-DllinMemory -dllPath $dllPath
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.Instrumentation.dll'
            if (Test-Path $dllPath)
            {
                Load-DllinMemory -dllPath $dllPath
            }
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Framework.EncryptionEngine.dll'
            Load-DllinMemory -dllPath $dllPath
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.BusinessPlatform.SharedTypes.dll'
            Load-DllinMemory -dllPath $dllPath  
        }

        $script:EnvDllsLoaded = $true
    }

    $config = [Microsoft.Dynamics.ApplicationPlatform.Environment.EnvironmentFactory]::GetApplicationEnvironment()
    
    return $config
}

function Load-DllinMemory([string] $dllPath)
{
    #try catch as not all dll exist in RTM version, some dependency/dll are introduced at update 1 or later
    #powershell cannot unload dll once it's loaded, the trick is to create an in-memory copy of the dll than load it
    #after the loading of in-memory dll, the physical dll stay unlocked

    try
    {
        if (Test-Path $dllPath)
        {
            $bytes = [System.IO.File]::ReadAllBytes($dllPath)
            [System.Reflection.Assembly]::Load($bytes) | Out-Null
        }
    }
    catch
    {}
}

function CopyDllToTempFolder([string] $dllPath, [string] $packageTemp)
{
    #try catch as not all dll exist in RTM version, some dependency/dll are introduced at update 1 or later    
    $fileTempPath =  Join-Path $packageTemp $(Get-Item $dllPath).Name
    try
    {
        if ((Test-Path $dllPath) -And !$(Test-Path $fileTempPath))
        {
            $Buffer = New-Object 'byte[]' 1024
            # Open file in read mode
            $SourceFile = [System.IO.File]::Open($dllPath, [System.IO.FileMode]::Open, [System.IO.FileAccess]::Read, [System.IO.FileShare]::Read)
            # Create the new file
            $DestinationFile = [System.IO.File]::Open($fileTempPath, [System.IO.FileMode]::CreateNew)
            try{
                # Copy the contents of the source file to the destination
                while(($readLength = $SourceFile.Read($Buffer, 0, $Buffer.Length)) -gt 0)
                {
                    $DestinationFile.Write($Buffer, 0, $readLength)
                }
            }
            catch{
                throw $_
            }
            finally{
                $SourceFile.Close()
                $DestinationFile.Close()
            }
        }
    }
    catch
    {
        Write-Log "Failed to copy file to temp folder with Exception: [$($_.Exception)]"
        throw $_
    }    
}

function Get-IsMSIKeyVault
{
    $webroot = Get-AosWebSitePhysicalPath
    $webconfig=join-path $webroot "web.config"
    $isMSIKeyVault=$false
    
    [System.Xml.XmlDocument] $xd=new-object System.Xml.XmlDocument
    $xd.Load($webconfig)
    $ns=New-Object System.Xml.XmlNamespaceManager($xd.NameTable)
    $ns.AddNamespace("ns",$xd.DocumentElement.NamespaceURI)

    $key="CertificateHandler.IsMSIKeyVault"

    $isKeyVaultValue = $xd.SelectSingleNode("//ns:add[@key='$key']",$ns)

    if($isKeyVaultValue -ne $null)
    {        
        $isMSIKeyVault = $isKeyVaultValue.GetAttribute("value")
    }

    return $isMSIKeyVault 
}

function GenerateMetadataModuleInstallationInfo
{
    try
    {
        $ErrorActionPreference = 'Stop'

        write-output "Creating Metadata Module Installation Info."
        
        $packagePath = Get-AOSPackageDirectory
        $CommonBin = $(Get-CommonBinDir)

        $dllPath = Join-Path $CommonBin 'bin\Microsoft.Dynamics.AX.AXInstallationInfo.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $CommonBin 'bin\Microsoft.Dynamics.AX.Metadata.Storage.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $CommonBin 'bin\Microsoft.Dynamics.AX.Metadata.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $CommonBin 'bin\Microsoft.Dynamics.AX.Metadata.Core.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $CommonBin 'bin\Microsoft.Dynamics.ApplicationPlatform.XppServices.Instrumentation.dll'
        Load-DllinMemory -dllPath $dllPath

        [Microsoft.Dynamics.AX.AXInstallationInfo.AXInstallationInfo]::ScanMetadataModelInRuntimePackage($packagePath)
    }
    catch
    {}
}

function Get-IsAppFallOrLater([Parameter(Mandatory=$false)] $webroot)
{
    $version = Get-ApplicationReleaseFromAOS -webroot:$webroot

    return $($version -ne "RTW")
}

function Get-ApplicationReleaseFromPackage
{
    $packageRoot = split-Path -parent "$(split-Path -parent $PSScriptRoot)" 
    $PackageInstallationInfo = "$packageRoot\HotfixInstallationInfo.xml"
    $XPath = '/HotfixInstallationInfo/ServiceModelList/ComponentModel/Release[../ServiceModelGroup = "Application"]'
    $ApplicationRelease =  Select-Xml -Path $PackageInstallationInfo -XPath $Xpath 
    $MRXPath = '/HotfixInstallationInfo/ServiceModelList/ComponentModel[Name = "MRApplicationService"]'
    $MRApplicationServiceModel =  Select-Xml -Path $PackageInstallationInfo -XPath $MRXpath 

    if($ApplicationRelease.Count -ge 1)
    {
        return $ApplicationRelease[0].Node.InnerText
    }
    elseif ($MRApplicationServiceModel -eq $null)
    {
        return ""
    }
    else
    {
        return "RTW"
    }
}

function Get-IsModulePartOfPlatformAsBinary ([string] $packageNugetFile)
{
    [Void][Reflection.Assembly]::LoadWithPartialName('System.IO.Compression.FileSystem') 

    try
    {
        $PackFiles = [IO.Compression.ZipFile]::OpenRead($packageNugetFile)
        $PackageSpec =  $PackFiles.Entries | where {($_.Name -like '*.nuspec')}

        if(!($PackageSpec))
        {
            Throw "Unable to get package information"
        }

        [System.Xml.XmlDocument] $xmlDoc = New-Object System.Xml.XmlDocument
        $XmlDoc.Load($PackageSpec.Open())

        $PackFiles.Dispose()

        $Description = $xmlDoc.GetElementsByTagName('description').Item(0).InnerText

        if($Description.Contains("[Platform Package]"))
        {
            return $true
        }
        else
        {
            return $false
        }        
    }
    catch
    {
        return $false
    }
}

function Get-IsModulePartOfApplicationAsBinary ([string] $PackageNugetFilePath)
{
    [Void][Reflection.Assembly]::LoadWithPartialName('System.IO.Compression.FileSystem') 

    try
    {
        $PackFiles = [IO.Compression.ZipFile]::OpenRead($PackageNugetFilePath)
        $PackageSpec =  $PackFiles.Entries | Where-Object {($_.Name -like '*.nuspec')}

        if(!($PackageSpec))
        {
            Throw "Unable to get package information"
        }

        [System.Xml.XmlDocument] $xmlDoc=new-object System.Xml.XmlDocument
        $XmlDoc.Load($PackageSpec.Open())

        $PackFiles.Dispose()

        $Description = $xmlDoc.GetElementsByTagName('description').Item(0).InnerText

        if($Description.Contains("[Application Package]"))
        {
            return $true
        }
        else
        {
            return $false
        }
    }
    catch
    {
        return $false
    }
}

function Get-IsPlatformUpdate3OrLater([Parameter(Mandatory=$false)] $webroot)
{
    if (!$webroot) {
    $webroot = Get-AosWebSitePhysicalPath
    }

    #must use job or process to load the production information provider dll or it'll lock it self
    #in memory copy is not usable as this dll have some special hard coded reference dll which won't resolve when loaded in memory.
    $job = Start-Job -ScriptBlock  {
        param($webrootBlock)
        $VersionDLLPath = Join-Path $webrootBlock 'bin\Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.dll'
        Add-Type -Path $VersionDLLPath
        $provider = [Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.ProductInfoProvider]::get_Provider();
        $version = $provider.get_PlatformVersion();
        $version
    } -ArgumentList $webroot
    Wait-Job -Job $job | Out-Null
    $version = Receive-Job -Job $job

    return $(($version -ne '7.0') -and ($version -ne '7.0.0.0') -and (![string]::IsNullOrEmpty($version)))    
}

function Get-IsPackageContainUpgradeBinaryForUpdate3
{
    $sourcePath = [IO.Path]::Combine($(split-Path -parent $PSScriptRoot), "Packages")
    #using a framework package from platform which customer cannot generate to identify if it's from platform update 3
    $files=get-childitem -Path:$sourcePath dynamicsax-framework-bin.*.nupkg
    foreach ($packageFile in $files) 
    {
        if(Get-IsModulePartOfPlatformAsBinary -packageNugetFile $packageFile.FullName)
        {
            return $true
        }
    }

    return $false
}

function Get-ProductPlatformBuildVersion([Parameter(Mandatory=$false)] $webroot)
{
    if (!$webroot) {
       $webroot = Get-AosWebSitePhysicalPath
    }

    #must use job or process to load the production information provider dll or it'll lock it self
    #in memory copy is not usable as this dll have some special hard coded reference dll which won't resolve when loaded in memory.
    $job = Start-Job -ScriptBlock  {
        param($webrootBlock)
        $VersionDLLPath = Join-Path $webrootBlock 'bin\Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.dll'
        Add-Type -Path $VersionDLLPath
        $provider = [Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.ProductInfoProvider]::get_Provider();
        $version = $provider.get_PlatformBuildVersion();
        $version
    } -ArgumentList $webroot
    Wait-Job -Job $job | Out-Null
    $version = Receive-Job -Job $job
   
    $build = [System.Version]::new()
    $releaseBuild
    if([System.Version]::TryParse($Version, [ref]$build))
    {
        $releaseBuild = $build.Build
    }
    else
    {
        #default to 0 from 7.0.0.0 
        $releaseBuild = 0
    }   

    return  $releaseBuild
}

function Get-ProductPlatformBuildFullVersion
{
    $webroot = Get-AosWebSitePhysicalPath

    #must use job or process to load the production information provider dll or it'll lock it self
    #in memory copy is not usable as this dll have some special hard coded reference dll which won't resolve when loaded in memory.
    $job = Start-Job -ScriptBlock  {
        param($webrootBlock)
        $VersionDLLPath = Join-Path $webrootBlock 'bin\Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.dll'
        Add-Type -Path $VersionDLLPath
        $provider = [Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.ProductInfoProvider]::get_Provider();
        $version = $provider.get_PlatformBuildVersion();
        $version
    } -ArgumentList $webroot
    Wait-Job -Job $job | Out-Null
    $version = Receive-Job -Job $job
   
    $build = [System.Version]::new()
    
    if([System.Version]::TryParse($Version, [ref]$build))
    {
        return $build
    }
    
    return $null
}

function Get-ProductApplicationVersion ([Parameter(Mandatory=$false)] $webroot)
{
    if($webroot -eq $null)
    {
        $webroot = Get-AosWebSitePhysicalPath
    }

    #must use job or process to load the production information provider dll or it'll lock it self
    #in memory copy is not usable as this dll have some special hard coded reference dll which won't resolve when loaded in memory.
    $job = Start-Job -ScriptBlock  {
        param($webrootBlock)
        $VersionDLLPath = Join-Path $webrootBlock 'bin\Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.dll'
        Add-Type -Path $VersionDLLPath
        $provider = [Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.ProductInfoProvider]::get_Provider();
        $version = $provider.get_ApplicationBuildVersion();
        $version
    } -ArgumentList $webroot
    Wait-Job -Job $job | Out-Null
    $version = Receive-Job -Job $job    
    
    return  $version
}

function Get-DataAccessSqlPwd
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.SqlPwd
}

function Get-DataAccessDatabase
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.Database
}

function Get-DataAccessSqlUsr
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.SqlUser
}

function Get-DataAccessDbServer
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.DbServer
}


function Get-AOSPackageDirectory
{
    $Config = Get-ApplicationEnvironment
    return $Config.Aos.PackageDirectory
}

function Get-CommonBinDir
{
    $Config = Get-ApplicationEnvironment
    return $Config.Common.BinDir
}

function Get-BiReportingPersistentVirtualMachineIPAddressSSRS
{
    $Config = Get-ApplicationEnvironment
    return $Config.BiReporting.PersistentVirtualMachineIPAddressSSRS
}

function Get-BiReportingReportingServers
{
    $Config = Get-ApplicationEnvironment
    $reportingServers = $Config.BiReporting.ReportingServers
    if ([System.String]::IsNullOrWhiteSpace($reportingServers))
    {
        $reportingServers = $Config.BiReporting.PersistentVirtualMachineIPAddressSSRS
    }

    return $reportingServers
}

function Get-UseReportingCluster
{
    $webRoot = Get-AosWebSitePhysicalPath
    $webConfigPath = join-path $webRoot "web.config"
    
    if(Test-Path $webConfigPath){
        [System.Xml.XmlDocument] $webConfig = new-object System.Xml.XmlDocument
        $webConfig.Load($webConfigPath)

        $ns = New-Object System.Xml.XmlNamespaceManager($webConfig.NameTable)
        $ns.AddNamespace("ns",$webConfig.DocumentElement.NamespaceURI)
        $key = "UseReportingCluster"
        $userClusterKey = $webConfig.SelectSingleNode("//ns:add[@key='$key']",$ns)

        if($null -ne $userClusterKey){
            if($userClusterKey.Value -eq $true){
                return $true
            }
        }
    }

    return $false
}

function Get-InfrastructureClickonceAppsDirectory
{
    $Config = Get-ApplicationEnvironment
    return $Config.Infrastructure.ClickonceAppsDirectory
}

function Get-DevToolsInstalled
{
    $webroot = Get-AosWebSitePhysicalPath
    $webconfig=join-path $webroot "web.config"
    $DevInstall=$false
    
    [System.Xml.XmlDocument] $xd=new-object System.Xml.XmlDocument
    $xd.Load($webconfig)
    $ns=New-Object System.Xml.XmlNamespaceManager($xd.NameTable)
    $ns.AddNamespace("ns",$xd.DocumentElement.NamespaceURI)

    $key="Infrastructure.VSToolsCount"

    $VScount = $xd.SelectSingleNode("//ns:add[@key='$key']",$ns)

    if($VScount -ne $null){
        
        if($VScount.GetAttribute("value") -gt 0)
        {
            $DevInstall=$true
        }
    }
    return $DevInstall 
}

function Get-ProductVersionMajorMinor
{
    [string]  $productVersionMajorMinorString = '7.0'
    return $productVersionMajorMinorString
}

function Get-IsRetailProductSku
{
    $productVersion = Get-ProductVersionMajorMinor    
    $retailHQConfigurationLocationRegistryPath = "HKLM:\SOFTWARE\Microsoft\Dynamics\$productVersion\Setup\Metadata"
    if(Test-Path $retailHQConfigurationLocationRegistryPath)
    {
        $retailProductSku = Get-ItemProperty -Path $retailHQConfigurationLocationRegistryPath -Name 'ProductSku' -ErrorAction SilentlyContinue
        
        if($retailProductSku -and $retailProductSku.ProductSku -and ($retailProductSku.ProductSku -eq 'Dynamics365ForRetail'))
        {
            return $true
        }
    }

    return $false
}

function Get-IsOverlayeringDisabled
{
    $productVersion = Get-ProductVersionMajorMinor    
    $retailHQConfigurationLocationRegistryPath = "HKLM:\SOFTWARE\Microsoft\Dynamics\$productVersion\Setup\Metadata"
    if(Test-Path $retailHQConfigurationLocationRegistryPath)
    {
        $isOverlayeringDisabled = Get-ItemProperty -Path $retailHQConfigurationLocationRegistryPath -Name 'DisableOverlayering' -ErrorAction SilentlyContinue
        
        if($isOverlayeringDisabled -and $isOverlayeringDisabled.DisableOverlayering -and ($isOverlayeringDisabled.DisableOverlayering -like 'true'))
        {
            return $true
        }
    }

    return $false
}

function Get-IsAppSealed ([Parameter(Mandatory=$false)] $webroot)
{
    if($webroot -eq $null)
    {
        $webroot = Get-AosWebSitePhysicalPath
    }

    $Version = (Get-ProductApplicationVersion -webroot:$webroot)
    $build = [System.Version]::new()
    $minSealedVersion = new-object System.Version "8.1"

    if([System.Version]::TryParse($Version, [ref]$build))
    {
        # Any build major/minor 8.1+ is considered sealed.
        if($build.CompareTo($minSealedVersion) -ge 0)
        {
           return $true
        }

        # Sealing logic for 8.0 and earlier
        if($build.Major -ge "8")
        {
            ##Bug 240512: Receipt designer bits are not updated with the uptake of latest binary hotfix
            ##The appseal have a different meaning during package deployment than development
            ##if the appSeal == true we expect full binary hotfix, other wise we expect seperate x++ hotfix and special handle the remaining mofule (ie, clickonce... package)
            ##the app8.0, even though it's sealed from dev point of view, it's still releasing seperate x++ hotfix and the special handling code path for none sealed need to be used during package deployment
            if($build.Minor -ge "1") 
            {
                return $true
            }
        }
        elseif($build.Minor -ge "2")
        {
            if($build.Build.ToString().StartsWith("2"))
            {
                return $true
            }
        }
    }
    
    return $false
    
}

function Get-DependencyAXModelList([string] $sourcePath, [string] $metaPackageName)
{
    $microsoftPackages = @()
    [Void][Reflection.Assembly]::LoadWithPartialName('System.IO.Compression.FileSystem')
    $zipFile = Get-Item $sourcePath\$metaPackageName*.nupkg
    if($zipFile -ne $null)
    {
        $PackFiles = [IO.Compression.ZipFile]::OpenRead($zipFile).Entries
        $PackageSpec =  $PackFiles | where {($_.Name -like "*.nuspec")}

        if(!($PackageSpec))
        {
            Throw "Unable to get package information for $metaPackageName"
        }

        [System.Xml.XmlDocument] $xmlDoc=new-object System.Xml.XmlDocument
        $XmlDoc.Load($PackageSpec.Open())

        $Dependencies = $xmlDoc.GetElementsByTagName('dependency').id
        
        foreach($d in $Dependencies)
        {
            $microsoftPackages += $d
        }
    }
    
    return $microsoftPackages
}

function Get-ALMPackageCopyPath
{
    [string]$ALMBackupPath = ""

    # Is the ALM Service registered (environment variable "DynamicsSDK" would be set)
    if ($env:DynamicsSDK)
    {
        $RegPath = "HKLM:\SOFTWARE\Microsoft\Dynamics\AX\7.0\SDK"

        [string]$ALMBackupPath = $null
        
        # Do not fail when registry key is not found
        try
        {
            # Get the Dynamics SDK (ALM Service) registry key (throws if not found).
            $RegKey = Get-ItemProperty -Path $RegPath
            if ($RegKey -ne $null)
            {
                # Check if backup path set in registry
                $ALMBackupPath = $RegKey.BackupPath
            }
        }
        catch
        {
        }
        
        # If path not found in registry, check default paths
        if (-not $ALMBackupPath)
        {
            if (Test-Path "I:\DynamicsBackup")
            {
                $ALMBackupPath = "I:\DynamicsBackup"
            }
            elseif (Test-Path "$($env:SystemDrive)\DynamicsBackup")
            {
                $ALMBackupPath = "$($env:SystemDrive)\DynamicsBackup"
            }
        }
    }

    return $ALMBackupPath
}

function Get-EnvironmentId
{
    $Config = Get-ApplicationEnvironment
    return $Config.LCS.LCSEnvironmentId
}

function Get-DataAccessFlightingCachePath
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.FlightingCachePath
}

function Get-DataAccessFlightingEnvironment
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.FlightingEnvironment
}

function Get-DataAccessFlightingCertificateThumbprint
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.FlightingCertificateThumbprint
}

function Get-DataAccessFlightingServiceCatalogID
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.FlightingServiceCatalogID
}

function Get-TenantDomainGUID
{
    $Config = Get-ApplicationEnvironment
    return $Config.Aad.TenantDomainGUID
}

function Get-AADTenantId
{
    $Config = Get-ApplicationEnvironment
    return $Config.Aad.AADTenantId
}

function Get-S2SCertificate
{
    try
    {
        $config = Get-ApplicationEnvironment
        $certHandlerSetttings = $config.CertificateHandler.GetCertificateHandlerSettings()

        $certificateHandler = [Microsoft.Dynamics.AX.Configuration.CertificateHandler.CertificateHandlerFactory]::CreateCertificateHandler($certHandlerSetttings)
    
        $s2sCertificate = $certificateHandler.GetSingleCertificateForId($config.Infrastructure.S2SCertThumbprint)

        return $s2sCertificate
    }
    catch
    {
        return $null
    }
}

function Load-FlightingDlls
{
    $webroot = Get-AosWebSitePhysicalPath

    Load-DllinMemory "$webroot\bin\Microsoft.Carbon.CertificateCommon.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Carbon.Flighting.Runtime.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Commerce.Flighting.Common.dll"

    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.BusinessPlatform.SharedTypes.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.ApplicationPlatform.XppServices.Instrumentation.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.ApplicationPlatform.Services.Instrumentation.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.AX.Data.Sql.Shared.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.ApplicationPlatform.Flighting.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.ApplicationPlatform.AOSFlighting.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.ApplicationPlatform.PerformanceCounters.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Practices.EnterpriseLibrary.TransientFaultHandling.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Practices.EnterpriseLibrary.TransientFaultHandling.Data.dll"
    Load-DllinMemory "$webroot\bin\Newtonsoft.Json.dll"

    # Contained in XppServices.Instrumentation in earlier releases
    $dbSyncInstrumentationPath = "$webroot\bin\Microsoft.Dynamics.ApplicationPlatform.DatabaseSynchronize.Instrumentation.dll"
    if (Test-Path $dbSyncInstrumentationPath)
    {
        Load-DllinMemory $dbSyncInstrumentationPath
    }
}

$script:FlightingProvider = $null
function Get-IsFlightEnabled([string] $flightName)
{
    $ErrorActionPreference = 'Stop'

    if ([String]::IsNullOrWhiteSpace($flightName))
    {
        return $false
    }

    if ($script:FlightingProvider -eq $null)
    {
        $sqlPwd    = Get-DataAccessSqlPwd
        $sqlUser   = Get-DataAccessSqlUsr
        $sqlServer = Get-DataAccessDbServer
        $sqlDB     = Get-DataAccessDatabase
        $connectionString = "Data Source=$sqlServer; " +
            "Integrated Security=False; " +
            "User Id=$sqlUser; " +
            "Password=`"$sqlPwd`"; " +
            "Initial Catalog=$sqlDB"

        $flightingCacheFolder = Get-DataAccessFlightingCachePath
        if ([String]::IsNullOrWhiteSpace($flightingCacheFolder))
        {
            $flightingCacheFolder = Get-AosWebSitePhysicalPath # webroot
        }

        Load-FlightingDlls

        $flightingConfiguration = New-Object -TypeName Microsoft.Dynamics.ApplicationPlatform.AOSFlighting.StaticFlightingServiceConfiguration -ArgumentList `
            (Get-DataAccessFlightingEnvironment), `
            (Get-DataAccessFlightingCertificateThumbprint), `
            (Get-DataAccessFlightingServiceCatalogID), `
            $flightingCacheFolder, `
            $null, # object extraInformation `
            $connectionString, `
            (60 * 60) # int refreshDataIntervalInSec

        $script:FlightingProvider = [Microsoft.Dynamics.ApplicationPlatform.AOSFlighting.AOSFlightingSystem]::InitializeFlightService($flightingConfiguration)
        $script:FlightingProvider.AddInformationToCommonContext("TenantID", (Get-TenantDomainGUID))
        $script:FlightingProvider.AddInformationToCommonContext("EnvironmentId", (Get-EnvironmentId))

        Write-Log "Flighting provider is a $($script:FlightingProvider.GetType().Name)."
    }

    return $script:FlightingProvider.IsFeatureEnabled($flightName)
}

function IsLocalServer([string] $serverName)
{
    if ($serverName -eq $null) { return $false }
    
    if ("." -eq $serverName) { return $true }
    if ("(local)" -eq $serverName) { return $true }
    if ("localhost" -eq $serverName) { return $true }
    if ("127.0.0.1" -eq $serverName) { return $true }
    if ([System.Environment]::MachineName -eq $serverName.Split('.')[0]) { return $true }

    return $false
}

function Invoke-XppStaticMethod()
{
    Param(
        [Parameter(Mandatory=$true)]
        [ValidatePattern('^[a-zA-Z0-9_]+$')]
        [String]
        $ClassName,

        [Parameter(Mandatory=$true)]
        [ValidatePattern('^[a-zA-Z0-9_]+$')]
        [String]
        $MethodName,

        [Parameter(Mandatory=$true)]
        [ValidateNotNullOrEmpty()]
        [String]
        $RedirectStandardError
    )

    $commandParameter = "-bindir `"$(Get-CommonBinDir)`""
    $commandParameter += " -metadatadir `"$(Get-AOSPackageDirectory)`""
    $commandParameter += " -sqluser `"$(Get-DataAccessSqlUsr)`""
    $commandParameter += " -sqlserver `"$(Get-DataAccessDbServer)`""
    $commandParameter += " -sqldatabase `"$(Get-DataAccessDatabase)`""
    $commandParameter += " -sqlpwd `"$(Get-DataAccessSqlPwd)`""
    $commandParameter += " -setupmode runstaticxppmethod"
    $commandParameter += " -classname $ClassName"
    $commandParameter += " -methodname $MethodName"

    $webroot = Get-AosWebSitePhysicalPath
    $startProcessArgs = @{
        FilePath = "$webroot\bin\Microsoft.Dynamics.AX.Deployment.Setup.exe"
        ArgumentList = $commandParameter
        PassThru = $true
        Wait = $true
        RedirectStandardError = $RedirectStandardError
    }

    $process = Start-Process @startProcessArgs
    
    if ($process.ExitCode -ne 0)
    {
        $errorContent = Get-Content "$RedirectStandardError"
        if ($errorContent -eq $null)
        {
            $errorContent = "Process terminated with exit code $($process.ExitCode) (error stream is empty)."
        }
        throw $errorContent
    }
}

if(!$useServiceFabric)
{
    if (Test-Path "$($PSScriptRoot)\NonAdminDevToolsInterject.ps1")
    {
        & "$($PSScriptRoot)\NonAdminDevToolsInterject.ps1"
    }
}

function Get-HotfixInstallationInfo
{
    $packageRoot = split-Path -parent "$(split-Path -parent $PSScriptRoot)"

    # Using job to load the AXInstallationInfo dll to avoid file locks
    $job = Start-Job -ScriptBlock  {
        param($packageRootBlock)
        $axInstallationInfoDLLPath = Join-Path $packageRootBlock "Microsoft.Dynamics.AX.AXInstallationInfo.dll"
        $hotfixInstallationInfoFilePath = Join-Path $packageRootBlock "HotfixInstallationInfo.xml"
        try
        {
            if (Test-Path $axInstallationInfoDLLPath)
            {
                # Load AXInstallationInfo.dll
                [void][System.Reflection.Assembly]::LoadFrom($axInstallationInfoDLLPath)
                $hotfixInstallationInfo = New-Object Microsoft.Dynamics.AX.AXInstallationInfo.HotfixInstallationInfo

                if (Test-Path $hotfixInstallationInfoFilePath)
                {
                    # Initialize HotfixInstallationInfo from HotfixInstallationInfo.xml
                    $hotfixInstallationInfo.InitFromFile($hotfixInstallationInfoFilePath)
                    # Get HotfixInstallationInfo
                    $hotfixInstallationInfo
                }
            }
        }
        catch
        {
            Write-Warning "Failed to get HotfixInstallationInfo instance from deployable package"
            return $null
        }
    } -ArgumentList $packageRoot
    Wait-Job -Job $job | Out-Null
    $hotfixInstallationInfo  = Receive-Job -Job $job

    return $hotfixInstallationInfo
}

function Get-IsBinaryPackageTypeFromPackage
{
    $hotfixInstallationInfo = Get-HotfixInstallationInfo
    # Get Package Type from HotfixInstallationInfo for the package being deployed
    $packageType = $hotfixInstallationInfo.Type
    if(![string]::IsNullOrEmpty($packageType))
    {
        return ($packageType -eq 'PlatAppBinaryHotfixPackage' -or $packageType -eq 'MergedPackage')
    }

    Write-Warning "Failed to get package type from deployable package"
    return $false
}

function Get-MetadataModuleListFromPackage
{
    $hotfixInstallationInfo = Get-HotfixInstallationInfo
    # Get Metadata Module List from HotfixInstallationInfo of the package being deployed
    $metadataModuleList = $hotfixInstallationInfo.MetadataModuleList
    if(![string]::IsNullOrEmpty($metadataModuleList))
    {
        return $metadataModuleList
    }

    Write-Warning "Failed to get metadata module list from HotfixInstallationInfo.xml."
    return ""
}

function Get-PackageNamesFromDLL([Parameter(Mandatory=$false)] $webroot, [Parameter(Mandatory=$true)][string] $productInfoDLL)
{
    if (!$webroot) {
    $webroot = Get-AosWebSitePhysicalPath
    }

    # Using job to load the Product Information dll to avoid file locks
    $job = Start-Job -ScriptBlock  {
        param($webrootBlock, $productInfoDLLBlock)
        $metadataCoreDLLPath = Join-Path $webrootBlock "bin\Microsoft.Dynamics.AX.Metadata.Core.dll"
        $productInfoDLLPath = Join-Path $webrootBlock $productInfoDLLBlock
        try
        {
            if (Test-Path $metadataCoreDLLPath)
            {
                # Load Metadata.Core.dll
                [void][System.Reflection.Assembly]::LoadFrom($metadataCoreDLLPath)
                if (Test-Path $productInfoDLLPath)
                {
                    # Get package names from Product Information dll
                    $packageNames = [Microsoft.Dynamics.AX.Metadata.Core.CoreHelper]::GetPackagesNames($productInfoDLLPath, $null)
                    $packageNames
                }
            }
        }
        catch
        {
            Write-Warning "Failed to get package names from product information dll."
            return $null
        }
    } -ArgumentList $webroot,$productInfoDLL
    Wait-Job -Job $job | Out-Null
    $packageNames  = Receive-Job -Job $job
    return $packageNames
}

Export-ModuleMember -Function Get-DataAccessSqlPwd
Export-ModuleMember -Function Get-DataAccessDatabase
Export-ModuleMember -Function Get-DataAccessSqlUsr
Export-ModuleMember -Function Get-DataAccessDbServer
Export-ModuleMember -Function Get-AOSPackageDirectory
Export-ModuleMember -Function Get-CommonBinDir
Export-ModuleMember -Function Get-BiReportingPersistentVirtualMachineIPAddressSSRS
Export-ModuleMember -Function Get-BiReportingReportingServers
Export-ModuleMember -Function Get-UseReportingCluster
Export-ModuleMember -Function Get-InfrastructureClickonceAppsDirectory
Export-ModuleMember -Function Get-DevToolsInstalled
Export-ModuleMember -Function Get-IsModulePartOfPlatformAsBinary
Export-ModuleMember -Function Get-IsAppFallOrLater
Export-ModuleMember -Function Get-IsPlatformUpdate3OrLater
Export-ModuleMember -Function Get-IsPackageContainUpgradeBinaryForUpdate3
Export-ModuleMember -Function Get-ProductPlatformBuildVersion
Export-ModuleMember -Function Get-ProductApplicationVersion
Export-ModuleMember -Function Get-ProductPlatformBuildFullVersion
Export-ModuleMember -Function Get-ApplicationReleaseFromPackage
Export-ModuleMember -Function Get-ProductVersionMajorMinor
Export-ModuleMember -Function Get-IsRetailProductSku
Export-ModuleMember -Function Get-IsOverlayeringDisabled
Export-ModuleMember -Function Get-DependencyAXModelList
Export-ModuleMember -Function Get-IsAppSealed
Export-ModuleMember -Function Get-IsModulePartOfApplicationAsBinary
Export-ModuleMember -Function Get-ALMPackageCopyPath
Export-ModuleMember -Function Get-IsFlightEnabled
Export-ModuleMember -Function GenerateMetadataModuleInstallationInfo
Export-ModuleMember -Function Get-EnvironmentId
Export-ModuleMember -Function Get-AADTenantId
Export-ModuleMember -Function Get-TenantDomainGUID
Export-ModuleMember -Function Get-S2SCertificate
Export-ModuleMember -Function IsLocalServer
Export-ModuleMember -Function Invoke-XppStaticMethod
Export-ModuleMember -Function Get-IsBinaryPackageTypeFromPackage
Export-ModuleMember -Function Get-MetadataModuleListFromPackage
Export-ModuleMember -Function Get-PackageNamesFromDLL

# SIG # Begin signature block
# MIIjhgYJKoZIhvcNAQcCoIIjdzCCI3MCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCl/ccGLqnEnnJv
# 1HCBSSXueHFuMCYkbfotpy0/Dx7f8KCCDYEwggX/MIID56ADAgECAhMzAAAB32vw
# LpKnSrTQAAAAAAHfMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAxMjE1MjEzMTQ1WhcNMjExMjAyMjEzMTQ1WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQC2uxlZEACjqfHkuFyoCwfL25ofI9DZWKt4wEj3JBQ48GPt1UsDv834CcoUUPMn
# s/6CtPoaQ4Thy/kbOOg/zJAnrJeiMQqRe2Lsdb/NSI2gXXX9lad1/yPUDOXo4GNw
# PjXq1JZi+HZV91bUr6ZjzePj1g+bepsqd/HC1XScj0fT3aAxLRykJSzExEBmU9eS
# yuOwUuq+CriudQtWGMdJU650v/KmzfM46Y6lo/MCnnpvz3zEL7PMdUdwqj/nYhGG
# 3UVILxX7tAdMbz7LN+6WOIpT1A41rwaoOVnv+8Ua94HwhjZmu1S73yeV7RZZNxoh
# EegJi9YYssXa7UZUUkCCA+KnAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUOPbML8IdkNGtCfMmVPtvI6VZ8+Mw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDYzMDA5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAnnqH
# tDyYUFaVAkvAK0eqq6nhoL95SZQu3RnpZ7tdQ89QR3++7A+4hrr7V4xxmkB5BObS
# 0YK+MALE02atjwWgPdpYQ68WdLGroJZHkbZdgERG+7tETFl3aKF4KpoSaGOskZXp
# TPnCaMo2PXoAMVMGpsQEQswimZq3IQ3nRQfBlJ0PoMMcN/+Pks8ZTL1BoPYsJpok
# t6cql59q6CypZYIwgyJ892HpttybHKg1ZtQLUlSXccRMlugPgEcNZJagPEgPYni4
# b11snjRAgf0dyQ0zI9aLXqTxWUU5pCIFiPT0b2wsxzRqCtyGqpkGM8P9GazO8eao
# mVItCYBcJSByBx/pS0cSYwBBHAZxJODUqxSXoSGDvmTfqUJXntnWkL4okok1FiCD
# Z4jpyXOQunb6egIXvkgQ7jb2uO26Ow0m8RwleDvhOMrnHsupiOPbozKroSa6paFt
# VSh89abUSooR8QdZciemmoFhcWkEwFg4spzvYNP4nIs193261WyTaRMZoceGun7G
# CT2Rl653uUj+F+g94c63AhzSq4khdL4HlFIP2ePv29smfUnHtGq6yYFDLnT0q/Y+
# Di3jwloF8EWkkHRtSuXlFUbTmwr/lDDgbpZiKhLS7CBTDj32I0L5i532+uHczw82
# oZDmYmYmIUSMbZOgS65h797rj5JJ6OkeEUJoAVwwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVWzCCFVcCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAd9r8C6Sp0q00AAAAAAB3zAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgCYg9BufB
# gGGHNQISK57qCPJTpXid15Fn+5byUuWmj44wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQCzxCZfU2dI0J8kfq4AFofPhGX37aa8KzoLko8q60dd
# HX2+YuxEEMazAj2C1abVEM/V9XSuVj9wykygjht561HcYXyVY/5Io/gIBm67GKlZ
# FvWXsFA/MpTqS/r7MTKn9SLF5cLXTniNjQhN9VS8Bs+/vNzfrSOks0zlHpedK1No
# CqxqZK0iEg6gVDXSj1lD+/wMSmwNjltwdusnDE0o6yf4FAKhd3+4zpxvqZpqHsV3
# G7C2v1y+M2E+45ZJQRbSZdD2MjtQN+3TKFo8n8QrfzD81WpJo2LcjZLPwy96HeQR
# 1WO5PBp50/3aalSKd/pbPzaTcg6dpmIRL9F+Y+I71XRFoYIS5TCCEuEGCisGAQQB
# gjcDAwExghLRMIISzQYJKoZIhvcNAQcCoIISvjCCEroCAQMxDzANBglghkgBZQME
# AgEFADCCAVEGCyqGSIb3DQEJEAEEoIIBQASCATwwggE4AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIF+IXPDrUmWzL7bJxt36rnDtLatlp57qQS85I7BG
# QKOfAgZg+YTL/T0YEzIwMjEwNzMwMTc0NzMyLjE4N1owBIACAfSggdCkgc0wgcox
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJTAjBgNVBAsTHE1p
# Y3Jvc29mdCBBbWVyaWNhIE9wZXJhdGlvbnMxJjAkBgNVBAsTHVRoYWxlcyBUU1Mg
# RVNOOjhBODItRTM0Ri05RERBMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFt
# cCBTZXJ2aWNloIIOPDCCBPEwggPZoAMCAQICEzMAAAFLT7KmSNXkwlEAAAAAAUsw
# DQYJKoZIhvcNAQELBQAwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
# b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwHhcN
# MjAxMTEyMTgyNTU5WhcNMjIwMjExMTgyNTU5WjCByjELMAkGA1UEBhMCVVMxEzAR
# BgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
# Y3Jvc29mdCBDb3Jwb3JhdGlvbjElMCMGA1UECxMcTWljcm9zb2Z0IEFtZXJpY2Eg
# T3BlcmF0aW9uczEmMCQGA1UECxMdVGhhbGVzIFRTUyBFU046OEE4Mi1FMzRGLTlE
# REExJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2UwggEiMA0G
# CSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQChNnpQx3YuJr/ivobPoLtpQ9egUFl8
# THdWZ6SAKIJdtP3L24D3/d63ommmjZjCyrQm+j/1tHDAwjQGuOwYvn79ecPCQfAB
# 91JnEp/wP4BMF2SXyMf8k9R84RthIdfGHPXTWqzpCCfNWolVEcUVm8Ad/r1LrikR
# O+4KKo6slDQJKsgKApfBU/9J7Rudvhw1rEQw0Nk1BRGWjrIp7/uWoUIfR4rcl6U1
# utOiYIonC87PPpAJQXGRsDdKnVFF4NpWvMiyeuksn5t/Otwz82sGlne/HNQpmMzi
# gR8cZ8eXEDJJNIZxov9WAHHj28gUE29D8ivAT706ihxvTv50ZY8W51uxAgMBAAGj
# ggEbMIIBFzAdBgNVHQ4EFgQUUqpqftASlue6K3LePlTTn01K68YwHwYDVR0jBBgw
# FoAU1WM6XIoxkPNDe3xGG8UzaFqFbVUwVgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDov
# L2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljVGltU3RhUENB
# XzIwMTAtMDctMDEuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+aHR0
# cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNUaW1TdGFQQ0FfMjAx
# MC0wNy0wMS5jcnQwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDCDAN
# BgkqhkiG9w0BAQsFAAOCAQEAFtq51Zc/O1AfJK4tEB2Nr8bGEVD5qQ8l8gXIQMrM
# ZYtddHH+cGiqgF/4GmvmPfl5FAYh+gf/8Yd3q4/iD2+K4LtJbs/3v6mpyBl1mQ4v
# usK65dAypWmiT1W3FiXjsmCIkjSDDsKLFBYH5yGFnNFOEMgL+O7u4osH42f80nc2
# WdnZV6+OvW035XPV6ZttUBfFWHdIbUkdOG1O2n4yJm10OfacItZ08fzgMMqE+f/S
# TgVWNCHbR2EYqTWayrGP69jMwtVD9BGGTWti1XjpvE6yKdO8H9nuRi3L+C6jYntf
# aEmBTbnTFEV+kRx1CNcpSb9os86CAUehZU1aRzQ6CQ/pjzCCBnEwggRZoAMCAQIC
# CmEJgSoAAAAAAAIwDQYJKoZIhvcNAQELBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jvc29mdCBSb290IENlcnRp
# ZmljYXRlIEF1dGhvcml0eSAyMDEwMB4XDTEwMDcwMTIxMzY1NVoXDTI1MDcwMTIx
# NDY1NVowfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
# BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQG
# A1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwggEiMA0GCSqGSIb3
# DQEBAQUAA4IBDwAwggEKAoIBAQCpHQ28dxGKOiDs/BOX9fp/aZRrdFQQ1aUKAIKF
# ++18aEssX8XD5WHCdrc+Zitb8BVTJwQxH0EbGpUdzgkTjnxhMFmxMEQP8WCIhFRD
# DNdNuDgIs0Ldk6zWczBXJoKjRQ3Q6vVHgc2/JGAyWGBG8lhHhjKEHnRhZ5FfgVSx
# z5NMksHEpl3RYRNuKMYa+YaAu99h/EbBJx0kZxJyGiGKr0tkiVBisV39dx898Fd1
# rL2KQk1AUdEPnAY+Z3/1ZsADlkR+79BL/W7lmsqxqPJ6Kgox8NpOBpG2iAg16Hgc
# sOmZzTznL0S6p/TcZL2kAcEgCZN4zfy8wMlEXV4WnAEFTyJNAgMBAAGjggHmMIIB
# 4jAQBgkrBgEEAYI3FQEEAwIBADAdBgNVHQ4EFgQU1WM6XIoxkPNDe3xGG8UzaFqF
# bVUwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8GA1Ud
# EwEB/wQFMAMBAf8wHwYDVR0jBBgwFoAU1fZWy4/oolxiaNE9lJBb186aGMQwVgYD
# VR0fBE8wTTBLoEmgR4ZFaHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwv
# cHJvZHVjdHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3JsMFoGCCsGAQUFBwEB
# BE4wTDBKBggrBgEFBQcwAoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9j
# ZXJ0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcnQwgaAGA1UdIAEB/wSBlTCB
# kjCBjwYJKwYBBAGCNy4DMIGBMD0GCCsGAQUFBwIBFjFodHRwOi8vd3d3Lm1pY3Jv
# c29mdC5jb20vUEtJL2RvY3MvQ1BTL2RlZmF1bHQuaHRtMEAGCCsGAQUFBwICMDQe
# MiAdAEwAZQBnAGEAbABfAFAAbwBsAGkAYwB5AF8AUwB0AGEAdABlAG0AZQBuAHQA
# LiAdMA0GCSqGSIb3DQEBCwUAA4ICAQAH5ohRDeLG4Jg/gXEDPZ2joSFvs+umzPUx
# vs8F4qn++ldtGTCzwsVmyWrf9efweL3HqJ4l4/m87WtUVwgrUYJEEvu5U4zM9GAS
# inbMQEBBm9xcF/9c+V4XNZgkVkt070IQyK+/f8Z/8jd9Wj8c8pl5SpFSAK84Dxf1
# L3mBZdmptWvkx872ynoAb0swRCQiPM/tA6WWj1kpvLb9BOFwnzJKJ/1Vry/+tuWO
# M7tiX5rbV0Dp8c6ZZpCM/2pif93FSguRJuI57BlKcWOdeyFtw5yjojz6f32WapB4
# pm3S4Zz5Hfw42JT0xqUKloakvZ4argRCg7i1gJsiOCC1JeVk7Pf0v35jWSUPei45
# V3aicaoGig+JFrphpxHLmtgOR5qAxdDNp9DvfYPw4TtxCd9ddJgiCGHasFAeb73x
# 4QDf5zEHpJM692VHeOj4qEir995yfmFrb3epgcunCaw5u+zGy9iCtHLNHfS4hQEe
# gPsbiSpUObJb2sgNVZl6h3M7COaYLeqN4DMuEin1wC9UJyH3yKxO2ii4sanblrKn
# QqLJzxlBTeCG+SqaoxFmMNO7dDJL32N79ZmKLxvHIa9Zta7cRDyXUHHXodLFVeNp
# 3lfB0d4wwP3M5k37Db9dT+mdHhk4L7zPWAUu7w2gUDXa7wknHNWzfjUeCLraNtvT
# X4/edIhJEqGCAs4wggI3AgEBMIH4oYHQpIHNMIHKMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1lcmljYSBP
# cGVyYXRpb25zMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjo4QTgyLUUzNEYtOURE
# QTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZaIjCgEBMAcG
# BSsOAwIaAxUAkToz97fseHxNOUSQ5O/bBVSF+e6ggYMwgYCkfjB8MQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQg
# VGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIFAOSujvwwIhgPMjAy
# MTA3MzAyMjQ1MTZaGA8yMDIxMDczMTIyNDUxNlowdzA9BgorBgEEAYRZCgQBMS8w
# LTAKAgUA5K6O/AIBADAKAgEAAgIj2gIB/zAHAgEAAgISKDAKAgUA5K/gfAIBADA2
# BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIBAAIDB6EgoQowCAIB
# AAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBAFV8PdFGgbCKsA4fEHilmFZEcMi5i2kc
# kGKIst89uCugL7eII+kXYmO9lsAR9VzA0XRFgV+t9z9IhiEOUS8pPtAXa9XfIU14
# FKqVTVJe6k2ahgfzi0VK9uNRLGdChTOwskKK9cvv0NRHIzSs0Xtn2yuZ7gkyhAkM
# HGqwXdJ5qvc8MYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgT
# Cldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29m
# dCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENB
# IDIwMTACEzMAAAFLT7KmSNXkwlEAAAAAAUswDQYJYIZIAWUDBAIBBQCgggFKMBoG
# CSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0BCQQxIgQg0Th697Sg
# 1C7Ls4IRN2Vya3Syq5p4RSqXjq7LptRq100wgfoGCyqGSIb3DQEJEAIvMYHqMIHn
# MIHkMIG9BCBr9u6EInnsZYEts/Fj/rIFv0YZW1ynhXKOP2hVPUU5IzCBmDCBgKR+
# MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMT
# HU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAABS0+ypkjV5MJRAAAA
# AAFLMCIEIGo9H9oOywJCtW984455f4Tit2hJpvIzrUAtg69H95glMA0GCSqGSIb3
# DQEBCwUABIIBAGnKR/OqQ39Sx6xYcMXtnHtovjVlCpxre03jOaDrw2jEDJy4q0CY
# D62JgRKwafFLjVQ8Eq4nikNWuGw8bQM1EjRZu1Tq/KY99PkV4y1aZyDqwLtsMA4t
# Vql72b0tSVo/Esld4bRVkG1L5B5NY2vk43ZJDPZ5kdGRUma63rOrlg8CVTrZcUN7
# ZcsKNzLeBgzyS23xKyphrMNjDHKirg+cdKRmq8JtbWhcOETed/fGybarkzLxTfBd
# wG7+ekqBzBHc0tWwzIzGs0exE2TSftTLzwW0MrknPnuzl58jWHQE5ut76MTpwW/R
# 0NEckn7wImnRxvFXQp7cUUoT4zg1T48+EhA=
# SIG # End signature block
