Param([string]$logDir)

$OdbcTester = @"
using System;
using System.Runtime.InteropServices;

namespace UpdateOdbc
{
    public static class TestOdbcDriverManagerAPI
    {
        [DllImport("odbc32.dll", CharSet = CharSet.Unicode)]
        extern static short SQLAllocHandle(short HandleType, IntPtr InputHandle, out IntPtr OutputHandle);

        [DllImport("odbc32.dll")]
        private static extern short SQLFreeHandle(short hType, IntPtr handle);

        private const short SQL_SUCCESS = 0;
        private const short SQL_HANDLE_ENV = 1;

        public static bool TestAllocEnvHandle()
        {
            bool isSuccess = false;
            IntPtr henv = IntPtr.Zero;

            try
            {
                if ((SQL_SUCCESS == SQLAllocHandle(SQL_HANDLE_ENV, henv, out henv))
                    && (henv != IntPtr.Zero))
                {
                    isSuccess = true;
                }
            }
            catch (Exception)
            {
                isSuccess = false;
            }
            finally
            {
                if (henv != IntPtr.Zero)
                {
                    SQLFreeHandle(SQL_HANDLE_ENV, henv);
                }
            }

            return isSuccess;
        }
    }

    public static class TestOdbc17DriverAPI
    {
        [DllImport("msodbcsql17.dll", CharSet = CharSet.Unicode)]
        extern static short SQLAllocHandle(short HandleType, IntPtr InputHandle, out IntPtr OutputHandle);

        [DllImport("msodbcsql17.dll")]
        private static extern short SQLFreeHandle(short hType, IntPtr handle);

        private const short SQL_SUCCESS = 0;
        private const short SQL_HANDLE_ENV = 1;

        public static bool TestAllocEnvHandle()
        {
            bool isSuccess = false;
            IntPtr henv = IntPtr.Zero;

            try
            {
                if ((SQL_SUCCESS == SQLAllocHandle(SQL_HANDLE_ENV, henv, out henv))
                    && (henv != IntPtr.Zero))
                {
                    isSuccess = true;
                }
            }
            catch (Exception)
            {
                isSuccess = false;
            }
            finally
            {
                if (henv != IntPtr.Zero)
                {
                    SQLFreeHandle(SQL_HANDLE_ENV, henv);
                }
            }

            return isSuccess;
        }
    }
}
"@

$driverName = "ODBC Driver 13 for SQL Server"
$odbcMSIPath = Join-Path $PSScriptRoot "msodbcsql.msi"
$odbcMSIPath_v17 = Join-Path $PSScriptRoot "msodbcsql_17.msi"
$minimumVersion = New-Object System.Version("2015.131.4413.46")
$v171_Version = New-Object System.Version("2017.171.0.1")

if (Test-Path $odbcMSIPath_v17)
{
    $minimumVersion = New-Object System.Version("2017.172.0.1")
    $odbcMSIPath = $odbcMSIPath_v17
    $driverName = "ODBC Driver 17 for SQL Server"
}

$installDriver = $true
$unInstallDriver = $false
$driver = $null
if(Test-Path $odbcMSIPath)
{
  try
  {
      $driver = Get-ODBCDriver -Driver $driverName -Platform 64-bit -ErrorAction Stop
  }
  catch
  {
      $driver = $null
      Write-Output "Failed to get driver `"$driverName`""
  }

  if($driver)
  {
      $driverPath = $driver.Attribute["Driver"]
      $driverPath = [System.Environment]::ExpandEnvironmentVariables($driverPath)
      if(Test-Path $driverPath)
      {
          $driverVersionInfo = [System.Diagnostics.FileVersionInfo]::GetVersionInfo($driverPath)
          $installedVersion = New-Object System.Version($driverVersionInfo.FileVersionRaw)
          Write-Output ("Found installed driver `"$driverName`" version `"$($installedVersion.ToString())`"")
          if ($installedVersion.CompareTo($v171_Version) -eq 0)
          {
              Write-Output ("Installed version of `"$($installedVersion.ToString())`", need to uninstall it and reinstall latest one.")

              $programName = "Microsoft ODBC Driver 17 for SQL Server"
              $app = Get-WmiObject -Class Win32_Product -Filter ("Name = '" + $programName + "'")

              # Uninstallation needs Admins
              if ($app)
              {
                  $uninstallRet = $app.Uninstall()
                  if($uninstallRet."ReturnValue" -ne 0)
                  {
                      throw "Failed to uninstall `"$($installedVersion.ToString())`". See for details"
                  }
              }

              $installDriver = $true
          }
          elseif ($installedVersion.CompareTo($minimumVersion) -ge 0)
          {
              if ($driverName -match "17")
              {
                  Write-Output "Installed version of `"$driverName`" validate runtime API"

                  $shortPath = Split-Path -Path $driverPath -Parent
                  $shortPath += "\1033\msodbcsqlr17.rll"
                  if (-Not(Test-Path $shortPath))
                  {
                      $installDriver = $true
                      $unInstallDriver = $true
                      Write-Output "Installed version of `"$driverName`" is not installed correctly. Need to reinstall it!"
                  }
                  else
                  {
                      Add-Type -TypeDefinition $OdbcTester -Language CSharp
                      $validateDriverManager = [UpdateOdbc.TestOdbcDriverManagerAPI]::TestAllocEnvHandle()
                      $validateDriver = [UpdateOdbc.TestOdbc17DriverAPI]::TestAllocEnvHandle()
                      if ($validateDriverManager -and $validateDriver)
                      {
                          $installDriver = $false
                          $unInstallDriver = $false
                          Write-Output "Installed version of `"$driverName`" validate runtime API successfully! Skipping install of updated driver."
                      }
                      else
                      {
                          $installDriver = $true
                          $unInstallDriver = $true
                          Write-Output "Installed version of `"$driverName`" is not installed correctly. API Test failed! Need to reinstall it."
                      }
                  }
              }
              else
              {
                  Write-Output ("Installed version of `"$driverName`" is greater or equal to the miniumum supported version `""+$minimumVersion.ToString() +"`". Skipping install of updated driver.")

                  $installDriver = $false
              }
          }
          else
          {
              Write-Output ("Installed version of `"$driverName`" is lower than miniumum supported version `""+$minimumVersion.ToString() +"`". Installing updated version...")
          }
      }
      else
      {
          Write-Output "Driver `"$driverName`" registered but the file was not found. Installing..."
      }
  }
  else
  {
    Write-Output "$driverName not found! Installing..."
  }

  if($installDriver)
  {
    # Initialize logs
    if (!$logdir -or $logdir -eq $null)
    {
        $logdir = $PSScriptRoot 
    }
    if ($unInstallDriver)
    {
        $odbcUninstallOutputLog = Join-Path $logdir "MSODBC_Uninstall.log"
        $msiExecArgs="/x `"$odbcMSIPath`" /quiet /passive /norestart /log `"$odbcUninstallOutputLog`" IACCEPTMSODBCSQLLICENSETERMS=YES"
        $msiExecProcess = Start-Process "msiexec.exe" $msiExecArgs -PassThru -Wait
        if($msiExecProcess.ExitCode -ne 0)
        {
            throw "Failed to uninstall `"$driverName`". See $odbcUninstallOutputLog for details"
        }
        else
        {
            Write-Output "Uninstall of `"$driverName`" completed successfully. Logs can be found at $odbcUninstallOutputLog"
        }
    }
    $odbcInstallOutputLog = Join-Path $logdir "MSODBC_Install.log"
    $msiExecArgs="/i `"$odbcMSIPath`" /quiet /passive /norestart /log `"$odbcInstallOutputLog`" IACCEPTMSODBCSQLLICENSETERMS=YES"
    $msiExecProcess = Start-Process "msiexec.exe" $msiExecArgs -PassThru -Wait
    if($msiExecProcess.ExitCode -ne 0)
    {
        throw "Failed to install `"$driverName`". See $odbcInstallOutputLog for details"
    }
    else
    {
        Write-Output "Install of `"$driverName`" completed successfully. Logs can be found at $odbcInstallOutputLog"
    }
  }
}
else
{
  Write-Output "Updated ODBC Driver MSI not found in package. No update required."
}
# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDVnUlPXG7oWgP0
# MWRNr8ubdOJ92Zybsm4Ybk4fFAIuhqCCDYEwggX/MIID56ADAgECAhMzAAAB32vw
# LpKnSrTQAAAAAAHfMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAxMjE1MjEzMTQ1WhcNMjExMjAyMjEzMTQ1WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQC2uxlZEACjqfHkuFyoCwfL25ofI9DZWKt4wEj3JBQ48GPt1UsDv834CcoUUPMn
# s/6CtPoaQ4Thy/kbOOg/zJAnrJeiMQqRe2Lsdb/NSI2gXXX9lad1/yPUDOXo4GNw
# PjXq1JZi+HZV91bUr6ZjzePj1g+bepsqd/HC1XScj0fT3aAxLRykJSzExEBmU9eS
# yuOwUuq+CriudQtWGMdJU650v/KmzfM46Y6lo/MCnnpvz3zEL7PMdUdwqj/nYhGG
# 3UVILxX7tAdMbz7LN+6WOIpT1A41rwaoOVnv+8Ua94HwhjZmu1S73yeV7RZZNxoh
# EegJi9YYssXa7UZUUkCCA+KnAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUOPbML8IdkNGtCfMmVPtvI6VZ8+Mw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDYzMDA5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAnnqH
# tDyYUFaVAkvAK0eqq6nhoL95SZQu3RnpZ7tdQ89QR3++7A+4hrr7V4xxmkB5BObS
# 0YK+MALE02atjwWgPdpYQ68WdLGroJZHkbZdgERG+7tETFl3aKF4KpoSaGOskZXp
# TPnCaMo2PXoAMVMGpsQEQswimZq3IQ3nRQfBlJ0PoMMcN/+Pks8ZTL1BoPYsJpok
# t6cql59q6CypZYIwgyJ892HpttybHKg1ZtQLUlSXccRMlugPgEcNZJagPEgPYni4
# b11snjRAgf0dyQ0zI9aLXqTxWUU5pCIFiPT0b2wsxzRqCtyGqpkGM8P9GazO8eao
# mVItCYBcJSByBx/pS0cSYwBBHAZxJODUqxSXoSGDvmTfqUJXntnWkL4okok1FiCD
# Z4jpyXOQunb6egIXvkgQ7jb2uO26Ow0m8RwleDvhOMrnHsupiOPbozKroSa6paFt
# VSh89abUSooR8QdZciemmoFhcWkEwFg4spzvYNP4nIs193261WyTaRMZoceGun7G
# CT2Rl653uUj+F+g94c63AhzSq4khdL4HlFIP2ePv29smfUnHtGq6yYFDLnT0q/Y+
# Di3jwloF8EWkkHRtSuXlFUbTmwr/lDDgbpZiKhLS7CBTDj32I0L5i532+uHczw82
# oZDmYmYmIUSMbZOgS65h797rj5JJ6OkeEUJoAVwwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAd9r8C6Sp0q00AAAAAAB3zAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgn061YxDd
# XoXeRKbGoFR5u4Q4BTAVjW53btuL35tIpcowQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQB7gc/NeBeu1syx27uNtd16RFO8cMAoPD9kgUUGl0UB
# amfA5yHH112fGUUUcTI727WMrsSignAk0dQqB4Z5SOOadXWuGQQEDBSsbvD8mWJ7
# oZpyiwr5i1hy+YK8bJ+SkGU/+mlTmfdefrgw54QCbyukj2gdtcEk16QGlkN2PdEN
# qJk0S8RbsJIITr5kB9qQ5M76mUAtd+oWjquBln0E9BFF9kFIsFBW09WcIHGIG/J2
# lyp/3diTQgpLaMvYlZrPNHcaEQxt9oJiaNTP3LN+bf4lMyhsgn1/LDy+Bn2Q0GEh
# ltoZbURRpkrD8rjeLvUf0Km7lNG4k/5Cvcwxh2H+6jp6oYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIPioALiTM4hvlUJyLbeaECqC9lSqbnPJg9CcrENi
# PjkkAgZg+wotb9wYEzIwMjEwNzMwMTc0NzQwLjQ3NVowBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjpDNEJELUUzN0YtNUZGQzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABV0QHYtxv6L4qAAAA
# AAFXMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTIxMDExNDE5MDIxM1oXDTIyMDQxMTE5MDIxM1owgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpDNEJE
# LUUzN0YtNUZGQzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAN5tA6dUZvnnwL9qQtXc
# wPANhB4ez+5CQrePp/Z8TH4NBr5vAfGMo0lV/lidBatKTgHErOuKH11xVAfBehHJ
# vH9T/OhOc83CJs9bzDhrld0Jdy3eJyC0yBdxVeucS+2a2ZBd50wBg/5/2YjQ2ylf
# D0dxKK6tQLxdODTuadQMbda05lPGnWGwZ3niSgIKVRgqqCVlhHzwNtRh1AH+Zxbf
# Se7t8z3oEKAdTAy7SsP8ykht3srjdh0BykPFdpaAgqwWCJJJmGk0gArSvHC8+vXt
# Go3MJhWQRe5JtzdD5kdaKH9uc9gnShsXyDEhGZjx3+b8cuqEO8bHv0WPX9MREfrf
# xvkCAwEAAaOCARswggEXMB0GA1UdDgQWBBRdMXu76DghnU/kPTMKdFkR9oCp2TAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQAld3kAgG6XWiZyvdibLRmWr7yb6RSy
# cjVDg8tcCitS01sTVp4T8Ad2QeYfJWfK6DMEk7QRBfKgdN7oE8dXtmQVL+JcxLj0
# pUuy4NB5RchcteD5dRnTfKlRi8vgKUaxDcoFIzNEUz1EHpopeagDb4/uI9Uj5tIu
# wlik/qrv/sHAw7kM4gELLNOgdev9Z/7xo1JIwfe0eoQM3wxcCFLuf8S9OncttaFA
# WHtEER8IvgRAgLJ/WnluFz68+hrDfRyX/qqWSPIE0voE6qFx1z8UvLwKpm65QNyN
# DRMp/VmCpqRZrxB1o0RY7P+n4jSNGvbk2bR70kKt/dogFFRBHVVuUxf+MIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpD
# NEJELUUzN0YtNUZGQzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUAES34SWJ7DfbSG/gbIQwTrzgZ8PKggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AOSuGkkwIhgPMjAyMTA3MzAxMDI3MjFaGA8yMDIxMDczMTEwMjcyMVowdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA5K4aSQIBADAKAgEAAgIjhwIB/zAHAgEAAgIRSjAK
# AgUA5K9ryQIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBAKu4+A8ofJTWvvgX
# adZkhS2Qvnj8DdtIJrLDHIkVm/GPiM2qNhI0uDIijiadCG/OCGbjmNUp6QCVAPyE
# DlpmUbt++NMojsXw5Y/FFbSffoy/Q4mtwJzkbZGT6/KK+1wgzg0D9eSec2FFcc+y
# T4uxnczsHYmMi7TfOJm9+rZXdo8wMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAFXRAdi3G/ovioAAAAAAVcwDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQgc26oMfBaSPew8/QMPDA+f7PrqRKlGgXKzEdv1LWEDQcwgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCAsWo0NQ6vzuuupUsZEMSJ4UsRjtQw2dFxZWkHt
# qRygEzCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# V0QHYtxv6L4qAAAAAAFXMCIEIMWRWMnmFG5HdQVZ4OPA/YHYAEp2UYo6/c5IolNZ
# GwBYMA0GCSqGSIb3DQEBCwUABIIBADm91I0NM6Uyp4xJVCUnmdyByYvQMZVZ8xLG
# nt3adVa0mAcz7jffbJhLEKf21HSrQFAUFTtYT7xKtJG/LVxKNXlsDoQZrnqNu0DX
# 7VA6cii8oTV4TB0EpLwl3NgnbVHeMwQkyTHBiOHHSFXVSz9VnUWwgbeWv2sUhNbt
# RtHPY76Yvia/GrwxpTAed3g2o00+GdTQKTliyu9hnZVLR4+aDNeztfvKrZ2I3e1H
# PpCjhbDPGQ0daBdiXiwzmPXWZH1feVw2UF3HG6LyPUj6Ir06RoqGCQ1DDN0/byzt
# mGaBLEsyMaBm+V1PvBf+0BRpB2ESqwnIvFwOj+GbHv/xBY1hpYc=
# SIG # End signature block
