#######################################################
###              Restore Admin Account             ####
#######################################################

PARAM (
    [Parameter(Mandatory=$true)]
    $log,

    [Parameter(Mandatory=$false)]
    $UserInfoTableName = "USERINFO"
)

# Import modules and initialize the log
Import-Module WebAdministration
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking
Initialize-Log $log

function Add-AdminAccount()
{
    $expectedAdminSID = Get-Sid -Email $AdminPrincipalName -Domain $AdminIdentityProvider
    Write-LogWithoutEmail "Inserting a new Admin account record with SID='$($expectedAdminSID)', NetworkDomain='$($AdminIdentityProvider)', and NetworkAlias='$($AdminPrincipalName)'..."
    $target_sqlParams.Query = @"
    INSERT INTO $($UserInfoTableName) (ID, NAME, ENABLE, STATUSLINEINFO, TOOLBARINFO, DEBUGINFO, AUTOINFO, AUTOUPDATE, GARBAGECOLLECTLIMIT, HISTORYLIMIT, MESSAGELIMIT, GENERALINFO, SHOWSTATUSLINE, SHOWTOOLBAR, SHOWAOTLAYER, CONFIRMDELETE, REPORTFONTSIZE, FORMFONTSIZE, PROPERTYFONTSIZE, COMPANY, COMPILERWARNINGLEVEL, SID, NETWORKDOMAIN, NETWORKALIAS, LANGUAGE, HELPLANGUAGE, PREFERREDTIMEZONE, NOTIFYTIMEZONEMISMATCH, ACCOUNTTYPE, DEFAULTPARTITION, INTERACTIVELOGON)  
    VALUES ('Admin', 'Admin', 1, -538365, -1, 12, -1, 6, 20, 5, 1000, -8209, 1, 1, 4, -1, 9, 9, 9, 'DAT', 3, '$($expectedAdminSID)', '$($AdminIdentityProvider)', '$($AdminPrincipalName)', 'en-US', 'en-us', 58, 1, 2, 1, 1)
"@
    Invoke-SqlQueryWithLog $target_sqlParams | out-null
    Write-LogWithoutEmail "Added a new Admin record to $($UserInfoTableName) table."
}

function Ensure-AdminAccountIsEnabled($AdminAccount)
{
    if ($AdminAccount.ENABLE -ne 1)
    {
        Write-LogWithoutEmail "Admin account is disabled. Enabling Admin account..."
        $target_sqlParams.Query = @"
            UPDATE $($UserInfoTableName) set ENABLE = 1 WHERE ID = 'Admin'
"@
        Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
        Write-LogWithoutEmail "Finished enabling Admin account."
    }
    else
    {
        Write-LogWithoutEmail "Validated that Admin account is enabled."
    }
}

function Ensure-AdminAccountHasCorrectNetworkDomain($AdminAccount)
{
    if ($AdminAccount.NETWORKDOMAIN -ne $AdminIdentityProvider)
    {
        Write-LogWithoutEmail "The Admin account's NetworkDomain '$($AdminAccount.NETWORKDOMAIN)' in $($UserInfoTableName) does not match default provisioning admin identity provider '$($AdminIdentityProvider)'."
        $target_sqlParams.Query = @"
            UPDATE $($UserInfoTableName) set NETWORKDOMAIN = '$AdminIdentityProvider' WHERE ID = 'Admin'
"@
        Write-LogWithoutEmail "Updated the Admin account's NetworkDomain to '$($AdminIdentityProvider)' in $($UserInfoTableName)."
        Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
    }
    else
    {
        Write-LogWithoutEmail "Admin network domain is correct."
    }
}

function Ensure-AdminAccountHasCorrectSID($AdminAccount)
{
    $expectedAdminSID = Get-Sid -Email $AdminPrincipalName -Domain $AdminIdentityProvider
    if ([string]::IsNullOrEmpty($expectedAdminSID))
    {
        Write-LogWithoutEmail "Expected admin SID is null or empty. Ignoring database check for admin SID."
    }
    elseif ($AdminAccount.SID -ne $expectedAdminSID)
    {
        Write-LogWithoutEmail "SID in database '$($AdminAccount.SID)' does not match generated SID '$($expectedAdminSID)'. Updating SID in database..."
        $target_sqlParams.Query = @"
            UPDATE $($UserInfoTableName) set SID = '$($expectedAdminSID)' WHERE ID = 'Admin'
"@
        Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
        Write-LogWithoutEmail "Finished updating the SID."
    }
    else
    {
        Write-LogWithoutEmail "SID in database matches the generated SID."
    }
}

function Ensure-NoDuplicateAdminAccounts()
{
    $adminUserCount = Get-UserCount -Id "Admin"
    # Validate that ADMIN account is unique.
    if ($adminUserCount -gt 1)
    {
        Write-LogWithoutEmail "$($adminUserCount) admin records found. Removing duplicates."
        $target_sqlParams.Query = @"
        DECLARE @expectedCount int = 0,
        @adminPrincipalName varchar(100) = '$($AdminPrincipalName)'
            SELECT @expectedCount = COUNT(*) FROM $($UserInfoTableName) WHERE ID = 'Admin' and NETWORKALIAS = @adminPrincipalName;
    
                    IF @expectedCount >= 1 
                    BEGIN
                        DELETE $($UserInfoTableName) WHERE ID = 'Admin' and NETWORKALIAS != @adminPrincipalName;
                        IF @expectedCount > 1 
                        BEGIN
                            DELETE TOP(@expectedCount - 1) $($UserInfoTableName) WHERE ID = 'Admin'
                        END;
                    END;

                    ELSE
                    BEGIN
                        DELETE TOP($($adminUserCount) - 1) $($UserInfoTableName) WHERE ID = 'Admin'
                    END;
"@
        Invoke-SqlQueryWithLog $target_sqlParams | out-null
        Write-LogWithoutEmail "Removed the duplicate Admin records."
    }
    else
    {
        Write-LogWithoutEmail "No duplicate Admin records found in $($UserInfoTableName) table."
    }
}

function Ensure-NoDuplicateNetworkAliasAccounts($NetworkAlias)
{
    # Check if db has duplicate networkalias with admin account
    $adminUserCount = Get-UserCount -NetworkAlias $NetworkAlias

    # Validate that ADMIN networkalias account is not duplicated.
    if ($adminUserCount -gt 1)
    {
        $target_sqlParams.Query = @"
        DELETE $($UserInfoTableName) WHERE NETWORKALIAS = '$($NetworkAlias)' AND ID != 'Admin'
"@
        Invoke-SqlQueryWithLog $target_sqlParams | out-null
        Write-LogWithoutEmail "Removed the non Admin records where NetworkAlias was set to '$($NetworkAlias)'."
    }
}

function Get-AdminAccount()
{
    $target_sqlParams.Query = @"
    SELECT	ENABLE,
            NETWORKALIAS,
            IDENTITYPROVIDER,
            NETWORKDOMAIN,
            SID 
    FROM $($UserInfoTableName) WHERE ID = 'Admin'
"@
    return Invoke-SqlQueryWithLog $target_sqlParams
}

function Get-Sid($Email, $Domain)
{
    $sid = $null
    $dllPath = Join-Path $webrootBinPath "Microsoft.Dynamics.AX.Security.SidGenerator.dll"
    if (Test-Path $dllPath)
    {
        Write-LogWithoutEmail "Loading Microsoft.Dynamics.AX.Security.SidGenerator.dll from path '$($dllPath)' to generate SID..."
        $assembly = [Reflection.Assembly]::LoadFile($dllPath)

        Write-LogWithoutEmail "Loading SidGenerator type from assembly..."
        $sidGeneratorType = $assembly.GetType("Microsoft.Dynamics.Ax.Security.SidGenerator")

        Write-LogWithoutEmail "Attempting to find new SidGenerator method 'GenerateSIDByDefaultAlgo' (available in PU37+)..."
        $newGenerateSidMethod = $sidGeneratorType.GetMethod("GenerateSIDByDefaultAlgo")
        if ($null -ne $newGenerateSidMethod)
        {
            Write-LogWithoutEmail "New SidGenerator::GenerateSIDByDefaultAlgo(...) method found - using it."
            Write-LogWithoutEmail "Generating SID using email '$($Email)' and domain '$($Domain)'..."
            $sid = [Microsoft.Dynamics.Ax.Security.SidGenerator]::GenerateSIDByDefaultAlgo($Email, $Domain)
        }
        else
        {
            Write-LogWithoutEmail "Could not find method SidGenerator::GenerateSIDByDefaultAlgo(...) - will use old SidGenerator::Generate(email, domain) method instead."
            Write-LogWithoutEmail "Generating SID using email '$($Email)' and domain '$($Domain)'..."
            $sid = [Microsoft.Dynamics.Ax.Security.SidGenerator]::Generate($Email, $Domain)
        }

        Write-LogWithoutEmail "Generated SID: '$($sid)'"
    }
    else
    {
        Write-LogWithoutEmail "Unable to find SIDGenerator assembly '$($dllPath)'. Cannot generate SID."
    }

    return $sid
}

function Get-UserCount(
    [Parameter(Mandatory=$true, ParameterSetName="Id")]
    [String]
    $Id,

    [Parameter(Mandatory=$true, ParameterSetName="NetworkAlias")]
    [String]
    $NetworkAlias
)
{
    if (-not [string]::IsNullOrEmpty($Id))
    {
        $whereClause = "ID = '$($Id)'"
    }
    else
    {
        $whereClause = "NETWORKALIAS = '$($NetworkAlias)'"
    }

    $target_sqlParams.Query = @"
        SELECT COUNT(*) AS UserCount FROM $($UserInfoTableName) WHERE $($whereClause)
"@
    $userCountResults = Invoke-SqlQueryWithLog $target_sqlParams

    [int]$count = $userCountResults.UserCount
    Write-LogWithoutEmail "Number of users with $($whereClause): $($count)"
    return $count
}

function Initialize-Settings()
{
    $webroot = Get-AosWebSitePhysicalPath
    Write-LogWithoutEmail "The Service drive path of the webroot on the aos machine is: $($webroot)"

    $script:webrootBinPath = Join-Path $webroot 'bin'
    $environmentDllPath = Join-Path $webrootBinPath 'Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
    Add-Type -Path $environmentDllPath

    # Load application configuration
    $script:config = [Microsoft.Dynamics.ApplicationPlatform.Environment.EnvironmentFactory]::GetApplicationEnvironment()

    $script:target_sqlParams = @{
        'Database'       = $($config.DataAccess.Database)
        'UserName'       = $($config.DataAccess.SqlUser)
        'Password'       = $($config.DataAccess.SqlPwd)
        'ServerInstance' = $($config.DataAccess.DbServer)
        'Query'          = ''
        'QueryTimeout'   = 0 }


    $script:TenantId = $($config.Aad.AADTenantId)
    $script:AdminPrincipalName = $($config.Provisioning.AdminPrincipalName)
    $script:AdminIdentityProvider = $($config.Provisioning.AdminIdentityProvider)
    if (-not $AdminIdentityProvider) { $script:AdminIdentityProvider = 'https://sts.windows.net/' }

    if ($AdminPrincipalName -eq $null)
    {
        throw "Admin principal name in web.config is null so stopping restore of Admin account. Please check Admin account in DB manually and update the web.config for Admin principal name."
    }
}

function Invoke-SqlQueryWithLog($params)
{
    Write-LogWithoutEmail "Invoking SQL query:
$($params.Query)"
    try
    {
        return Invoke-SqlCmd @params -ErrorAction Stop
    }
    catch
    {
        Write-LogWithoutEmail "An exception occurred: $($_.Exception)"
    }
}

function Remove-DuplicateNetworkAliasAccounts($UserCount, $NetworkAlias)
{
    # Validate that Admin networkalias account is not duplicated.
    if ($UserCount -gt 1)
    {
        Write-LogWithoutEmail "Found $($UserCount) accounts with NetworkAlias '$($AdminPrincipalName)'. Removing duplicates (leaving only one record)..."

        # Remove duplicated records
        $target_sqlParams.Query = @"
            DELETE TOP($($UserCount-1)) FROM $($UserInfoTableName) WHERE NETWORKALIAS = '$AdminPrincipalName' 
"@
        Invoke-SqlQueryWithLog $target_sqlParams | out-null
        Write-LogWithoutEmail "Removed duplicated records for NetworkAlias '$($AdminPrincipalName)'."
    }
}

function Rename-AdminAccount()
{
    $PreviousAdminId = "OldAdmin{0:yyyyMMddHHmm}" -f [DateTime]::UtcNow

    # Update id of current admin account to different id
    $target_sqlParams.Query = @"
    UPDATE $($UserInfoTableName) set ID = '$PreviousAdminId' WHERE ID = 'Admin'
"@
    Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
    Write-LogWithoutEmail "Updated id of existing Admin account. New id is '$($PreviousAdminId)'."
}

function Set-AccountAsAdmin($NetworkAlias)
{
    Write-LogWithoutEmail "Updating ID (to 'Admin') and enabling account with NetworkAlias '$($NetworkAlias)'..."

    # Enable and update id of the user with networkalias of admin principal name to 'Admin' 
    $target_sqlParams.Query = @"
        UPDATE $($UserInfoTableName) set Enable = 1, ID = 'Admin' WHERE NETWORKALIAS = '$NetworkAlias'
"@
    Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
    Write-LogWithoutEmail "Updated id of account with NetworkAlias '$($NetworkAlias)'."
}

function Test-DoesNetworkAliasMatchTenantId($NetworkAlias)
{
    if (-not [string]::IsNullOrEmpty($NetworkAlias))
    {
        $NetworkAliasDomainSuffix = $NetworkAlias.SubString($NetworkAlias.IndexOf('@') + 1)

        # Validate that admin is the same provisioning admin and has the same domain as the tenant
        if ($NetworkAliasDomainSuffix -ne $TenantId)
        {
            Write-LogWithoutEmail "Admin in $($UserInfoTableName) has domain '$($NetworkAliasDomainSuffix)' in the NetworkAlias field, which doesn't match the 'Aad.AADTenantId' value in web.config: '$($TenantId)'."
        }
    }
}

function Write-LogWithoutEmail($LogMessage)
{
    # Regular expression from http://www.regular-expressions.info/email.html
    # Except we're not allowing ' to be the first character in the email address
    # since that messes with logging where we have a string like '<email address>' (single quotes included)
    # $1 in <email>@$1 is for the <Domain> capture group
    # meaning an email username@domain.com will be replaced with <email>@domain.com
    $LogMessageWithoutEmail = $LogMessage -replace "[a-z0-9!#\$%&*+/=?^_`{|}~-]+(?:\.[a-z0-9!#\$%&'*+/=?^_`{|}~-]+)*@(?<Domain>(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9]))?", '<email>@$1'
    Write-Log $LogMessageWithoutEmail
}

try
{
    Write-LogWithoutEmail "Restore Admin Account: Start"
    $error.Clear()

    Initialize-Settings

    Ensure-NoDuplicateAdminAccounts

    $adminUserCount = Get-UserCount -Id "Admin"
    if ($adminUserCount -eq 0)
    {
        Write-LogWithoutEmail "$($UserInfoTableName) has no Admin account. Need to create Admin account."
        Add-AdminAccount
    }
    else
    {
        Write-LogWithoutEmail "Validated that Admin account exists."
        
        $adminAccount = Get-AdminAccount
        $adminAccountNetworkAlias = $adminAccount.NETWORKALIAS
        Test-DoesNetworkAliasMatchTenantId -NetworkAlias $adminAccountNetworkAlias

        if ($adminAccountNetworkAlias -eq $AdminPrincipalName)
        {
            Write-LogWithoutEmail "Validated that admin network alias '$($adminAccountNetworkAlias)' matches the admin principal name '$($AdminPrincipalName)' in the config."
            Ensure-NoDuplicateNetworkAliasAccounts -NetworkAlias $AdminPrincipalName
        }
        else
        {
            Write-LogWithoutEmail "Admin network alias '$($adminAccountNetworkAlias)' in $($UserInfoTableName) is not the same as default provisioning admin principal name '$($AdminPrincipalName)'. Will update id of existing Admin account."
            Rename-AdminAccount

            $adminNetworkAliasUserCount = Get-UserCount -NetworkAlias $AdminPrincipalName

            # Validate if there is non admin account in USERINFO has same NETWORKALIAS with default provisioning admin principal name.
            if ($adminNetworkAliasUserCount -ge 1)
            {
                Remove-DuplicateNetworkAliasAccounts -UserCount $adminNetworkAliasUserCount -NetworkAlias $AdminPrincipalName
                Set-AccountAsAdmin -NetworkAlias $AdminPrincipalName
            }
            else
            {
                Write-LogWithoutEmail "No account found with NetworkAlias '$($AdminPrincipalName)'. Will create new Admin account record."
                Add-AdminAccount
            }
        }
    }

    $adminAccount = Get-AdminAccount
    Ensure-AdminAccountIsEnabled -AdminAccount $adminAccount
    Ensure-AdminAccountHasCorrectSID -AdminAccount $adminAccount
    Ensure-AdminAccountHasCorrectNetworkDomain -AdminAccount $adminAccount
}
Catch
{
    # Write the exception and that this process failed.  
    # Do not throw as this should not cause the deployment or update to fail.
    Write-Exception $_
    Write-LogWithoutEmail "Restore Admin Account failed, see $log for details."
}
Finally
{
    Write-LogWithoutEmail "Restore Admin Account: Stop"
}

# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBixKloG51Xld+P
# iHfYWRDWZcIge70qnkaDkV/xoMTj2KCCDYEwggX/MIID56ADAgECAhMzAAAB32vw
# LpKnSrTQAAAAAAHfMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAxMjE1MjEzMTQ1WhcNMjExMjAyMjEzMTQ1WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQC2uxlZEACjqfHkuFyoCwfL25ofI9DZWKt4wEj3JBQ48GPt1UsDv834CcoUUPMn
# s/6CtPoaQ4Thy/kbOOg/zJAnrJeiMQqRe2Lsdb/NSI2gXXX9lad1/yPUDOXo4GNw
# PjXq1JZi+HZV91bUr6ZjzePj1g+bepsqd/HC1XScj0fT3aAxLRykJSzExEBmU9eS
# yuOwUuq+CriudQtWGMdJU650v/KmzfM46Y6lo/MCnnpvz3zEL7PMdUdwqj/nYhGG
# 3UVILxX7tAdMbz7LN+6WOIpT1A41rwaoOVnv+8Ua94HwhjZmu1S73yeV7RZZNxoh
# EegJi9YYssXa7UZUUkCCA+KnAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUOPbML8IdkNGtCfMmVPtvI6VZ8+Mw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDYzMDA5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAnnqH
# tDyYUFaVAkvAK0eqq6nhoL95SZQu3RnpZ7tdQ89QR3++7A+4hrr7V4xxmkB5BObS
# 0YK+MALE02atjwWgPdpYQ68WdLGroJZHkbZdgERG+7tETFl3aKF4KpoSaGOskZXp
# TPnCaMo2PXoAMVMGpsQEQswimZq3IQ3nRQfBlJ0PoMMcN/+Pks8ZTL1BoPYsJpok
# t6cql59q6CypZYIwgyJ892HpttybHKg1ZtQLUlSXccRMlugPgEcNZJagPEgPYni4
# b11snjRAgf0dyQ0zI9aLXqTxWUU5pCIFiPT0b2wsxzRqCtyGqpkGM8P9GazO8eao
# mVItCYBcJSByBx/pS0cSYwBBHAZxJODUqxSXoSGDvmTfqUJXntnWkL4okok1FiCD
# Z4jpyXOQunb6egIXvkgQ7jb2uO26Ow0m8RwleDvhOMrnHsupiOPbozKroSa6paFt
# VSh89abUSooR8QdZciemmoFhcWkEwFg4spzvYNP4nIs193261WyTaRMZoceGun7G
# CT2Rl653uUj+F+g94c63AhzSq4khdL4HlFIP2ePv29smfUnHtGq6yYFDLnT0q/Y+
# Di3jwloF8EWkkHRtSuXlFUbTmwr/lDDgbpZiKhLS7CBTDj32I0L5i532+uHczw82
# oZDmYmYmIUSMbZOgS65h797rj5JJ6OkeEUJoAVwwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAd9r8C6Sp0q00AAAAAAB3zAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgJjvBas/0
# P1Jmru8VRxEBFLN6+mGZl1/3GCRpMdk4vv4wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQA7XPy8R28Wt0fG/yaD2385jC/kVHzf4g1+OWh9xDJ6
# I2qyQZpiyN1v/YCI6eCbRmbT6aCvKJ8x/GEvuuoDRH460MOcSZvHX3hQ24VjbYNV
# v8iM/i9Dp1suwn7L2yr79E57XGovRcQnLhYVWbLViBm3eBsf9doPa42pGVKrd4s/
# mYkWLbckVcHw710AXlkh1/Xaan1wZ/zPZxzjHThikUMevxILBpogfVWl97DYYje7
# yyo7uOcEny96f08aNP276sZuT4t++vZva0bonCpoXL0fubaNP/IBQVPnqxEUXJRf
# wwdz7hIK/0y5h6qRqtVYCNuuP3Mz3io/k1OWN80Mqr8hoYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEILHzCfnTZ0GXj0CFPgxY4beSmfoj2V4qXljJInTH
# UX+eAgZg+wr5zp0YEzIwMjEwNzMwMTc0NzM2LjIyNVowBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjowQTU2LUUzMjktNEQ0RDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABW3ywujRnN8GnAAAA
# AAFbMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTIxMDExNDE5MDIxNloXDTIyMDQxMTE5MDIxNlowgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjowQTU2
# LUUzMjktNEQ0RDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMgkf6Xs9dqhesumLltn
# l6lwjiD1jh+Ipz/6j5q5CQzSnbaVuo4KiCiSpr5WtqqVlD7nT/3WX6V6vcpNQV5c
# dtVVwafNpLn3yF+fRNoUWh1Q9u8XGiSX8YzVS8q68JPFiRO4HMzMpLCaSjcfQZId
# 6CiukyLQruKnSFwdGhMxE7GCayaQ8ZDyEPHs/C2x4AAYMFsVOssSdR8jb8fzAek3
# SNlZtVKd0Kb8io+3XkQ54MvUXV9cVL1/eDdXVVBBqOhHzoJsy+c2y/s3W+gEX8Qb
# 9O/bjBkR6hIaOwEAw7Nu40/TMVfwXJ7g5R/HNXCt7c4IajNN4W+CugeysLnYbqRm
# W+kCAwEAAaOCARswggEXMB0GA1UdDgQWBBRl5y01iG23UyBdTH/15TnJmLqrLjAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQCnM2s7phMamc4QdVolrO1ZXRiDMUVd
# gu9/yq8g7kIVl+fklUV2Vlout6+fpOqAGnewMtwenFtagVhVJ8Hau8Nwk+IAhB0B
# 04DobNDw7v4KETARf8KN8gTH6B7RjHhreMDWg7icV0Dsoj8MIA8AirWlwf4nr8pK
# H0n2rETseBJDWc3dbU0ITJEH1RzFhGkW7IzNPQCO165Tp7NLnXp4maZzoVx8PyiO
# NO6fyDZr0yqVuh9OqWH+fPZYQ/YYFyhxy+hHWOuqYpc83Phn1vA0Ae1+Wn4bne6Z
# GjPxRI6sxsMIkdBXD0HJLyN7YfSrbOVAYwjYWOHresGZuvoEaEgDRWUrMIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjow
# QTU2LUUzMjktNEQ0RDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUACrtBbqYy0r+YGLtUaFVRW/Yh7qaggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AOSuGxAwIhgPMjAyMTA3MzAxMDMwNDBaGA8yMDIxMDczMTEwMzA0MFowdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA5K4bEAIBADAKAgEAAgIlswIB/zAHAgEAAgIRKTAK
# AgUA5K9skAIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBAJ+RkAg7UwpT7ZZi
# U3NGWAYbRSZA406bCKKK8s72DLd/f/tV5PxPU6Z85QhzaExgO5ADpRumxP+bieXT
# B7bCkMoLcWoHyIT3zlRya+WkwX7wAt2IGcy8fnrWxgyau6w5h9EAmmyEc9pKtPTe
# WjVvGstmTZc++xvItXnAX00ewtJ+MYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAFbfLC6NGc3wacAAAAAAVswDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQgmElz/f7/uTcjvXX2OMdto6j/Q6yghx8FgwPp//30MSUwgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCDJIuCpKGMRh4lCGucGPHCNJ7jq9MTbe3mQ2FtS
# ZLCFGTCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# W3ywujRnN8GnAAAAAAFbMCIEIMAEMFdAKYAcZ7gspCd35dmED425Q2+wwSaWH+sN
# fs2NMA0GCSqGSIb3DQEBCwUABIIBAECOAUoupbjhdEWrzeICo907mYBmVzwcWDzE
# gd7RtG7effgbI2BrHkqXRA7fXcbjcLSGbps30CquzvuywC2XC3MeDHzAI/CteNl+
# i1ZNaw3u8wmrFKh9kDvC3bI9M0uKfiM0VFXSLLNtQw8NoKFlRaEpx1q0Ymy43YaA
# br3oK7viUYCsAmqAEB8Clp3ly32elbmjpIlZxaUBHmAMNoXWzdcH64yXCco82C00
# SarSgWDt9HALXj/kIrPBih8/X0u29DGu0OdWVXaetXISpkUnLdEDFNpX8PaEu+tA
# A4sYrhQpbaJ2EHV8HSYV+T76I9E4gDHhCb75lv3hWVj2ngL5pG8=
# SIG # End signature block
