﻿<#
.SYNOPSIS
    Stops the AOS service.

.DESCRIPTION

    Copyright © 2019 Microsoft. All rights reserved.
#>
[CmdletBinding()]
param
(
    [Parameter(Mandatory = $false, HelpMessage = "The path to the directory in which to write log files.")]
    [string]$LogDir
)

$ErrorActionPreference = "Stop"
Import-Module WebAdministration
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\CommonFaultHandling.psm1" -Force

function checkAosKernelFileLocked
{
    Param
    ( 
    [parameter(mandatory=$true,position=0)]
    [string]
    $AosKernelPathParam,
    [parameter(mandatory=$true, position=1)]
    [Int]
    $MaxWaitSecParam 
    )

    $AosKernelLockedFlag = $true

    $StopWatch = [System.Diagnostics.StopWatch]::StartNew()

    while ($AosKernelLockedFlag -and $StopWatch.Elapsed.TotalSeconds -lt $MaxWaitSecParam)
    {
        try
        {
            [IO.File]::OpenWrite($AosKernelPathParam).close()
            $AosKernelLockedFlag = $false
        }
        catch
        {
            Write-ServicingLog "Failed to open $($AosKernelPathParam) for write: $($_.Exception.Message)" -Vrb
            Start-Sleep -Seconds 1
        }
    }

    $StopWatch.Stop()
    return $AosKernelLockedFlag
}

# Initialize exit code.
[int]$ExitCode = 0

# Initialize the log file to use with Write-ServicingLog.
Set-ServicingLog -LogDir $LogDir

try
{
    Write-ServicingLog "Stopping AOS..."

    # For non-admin developer machine scenarios, import module to replace functions that
    # will not work within running as administrator.
    if (Test-Path "$($PSScriptRoot)\NonAdminDevToolsInterject.ps1")
    {
        & "$($PSScriptRoot)\NonAdminDevToolsInterject.ps1"
    }

    # Sometimes Get-Process will throw an exception even when called with a
    # SilentlyContinue error action.
    try
    {
        $devenv = Get-Process -Name "devenv" -ErrorAction SilentlyContinue
    }
    catch
    {
        Write-ServicingLog "Warning: Unable to get devenv processes: $($_)" -Vrb
    }

    # If any devenv processes are found, throw an exception.
    if ($devenv)
    {
        throw "Please close all instance of Visual Studio to continue with the installation."
    }

    # Determine if running in admin or non-admin mode.
    $IsAdmin = Test-IsRunningAsAdministrator

    # Get IIS service and start it if it is not already running.
    $IisService = Get-Service -Name "W3SVC"
    if ($IisService -and $IisService.Status -ine "Running")
    {
        Write-ServicingLog "IIS service is not running, starting IIS Service..."
        Start-Service -Name "W3SVC"
        Write-ServicingLog "IIS service started."
    }

    # Get the AOS web site and app pool and make sure they are both started.
    $websiteName = Get-AosWebsiteName
    $appPoolName = Get-AosAppPoolName
    if ((![string]::IsNullOrWhitespace($websiteName)) -and (![string]::IsNullOrWhitespace($appPoolName)))
    {
        Write-ServicingLog "Stopping IIS AOS web site."
        Stop-Website -Name $websiteName

        # Check if in stopped state before stopping to avoid error.
        $AosWebAppPoolState = Get-WebAppPoolState -Name $appPoolName
        if ($AosWebAppPoolState.Value -ine "Stopped")
        {
            Write-ServicingLog "Stopping IIS AOS application pool."
            Stop-WebAppPool -Name $appPoolName
        }

        # Set properties on web site and app pool, so they do not restart during reboots / IISRESETs.
        # Note: Stopping / starting via IIS mananger GUI will set these same properties.
        # Note: This is not available for non-admin users.
        if ($IsAdmin)
        {
            Write-ServicingLog "Disabling IIS auto start properties for the AOS website and application pool." -Vrb
            Set-ItemProperty -Path "IIS:\Sites\$webSiteName" -Name serverAutoStart -Value $false
            Set-ItemProperty -Path "IIS:\AppPools\$appPoolName" -Name autoStart -Value $false
        }

        $productConfigurationPoolName = Get-ProductConfigurationAppPoolName
        if (![string]::IsNullOrWhitespace($productConfigurationPoolName))
        {
            # Check if in stopped state before stopping to avoid error.
            $ProductConfigurationAppPoolState = Get-WebAppPoolState $productConfigurationPoolName
            if ($ProductConfigurationAppPoolState.Value -ine "Stopped")
            {
                Write-ServicingLog "Stopping IIS product configuration application pool."
                Stop-WebAppPool -Name $productConfigurationPoolName
            }

            # Set property on app pool, so it does not restart during reboots / IISRESETs.
            # Note: Stopping / starting via IIS mananger GUI will set the same property.
            # Note: This is not available for non-admin users.
            if ($IsAdmin)
            {
                Write-ServicingLog "Disabling IIS auto start property for the product configuration application pool." -Vrb
                Set-ItemProperty -Path "IIS:\AppPools\$productConfigurationPoolName" -Name autoStart -Value $false
            }
        }
    }

    # Try to get a list of all xppc* processes and attempt to stop them.
    try
    {
        $XppcProcesses = @(Get-Process -Name "xppc*" -ErrorAction SilentlyContinue)
        if ($XppcProcesses.Count -gt 0)
        {
            Write-ServicingLog "Stopping $($XppcProcesses.Count) 'xppc*' processes..." -Vrb
            foreach ($XppcProcess in $XppcProcesses)
            {
                Write-ServicingLog "Stopping $($XppcProcess.Name) process ID $($XppcProcess.Id)..." -Vrb
                try
                {
                    $XppcProcess | Stop-Process -Force
                }
                catch
                {
                    Write-ServicingLog "Warning: Failed to stop $($XppcProcess.Name) process ID $($XppcProcess.Id): $($_)" -Vrb
                }
            }
        }
    }
    catch
    {
        Write-ServicingLog "Warning: Unable to get xppc processes to stop: $($_)" -Vrb
    }

    # Stop the batch service and set the startup type to disabled, so it does not get started on reboot.
    Write-ServicingLog "Stopping and disabling the batch service."
    
    # Logger to pass to the retry function
    $logger = {param($message) Write-ServicingLog $message}

    # Block to kill batch in the last iteration of the retry function
    $killBatchProcess = {
        #Kill the batch process if still running at the end of the retries.
        $batchSvcName = "DynamicsAxBatch"
        $batchProcess = Get-WmiObject Win32_Service | where {$_.name -eq $batchSvcName}
        if($batchProcess)
        {
          $batchProcessInstance = $batchProcess | Format-List | Out-String
          Write-Output "Found Batch service: $batchProcessInstance"

          $batchProcessId = $batchProcess.ProcessId

          if($batchProcessId -gt 0)
          {
            Write-Output "Stopping Batch process ID: $batchProcessId"
            Stop-Process -Id $batchProcessId -Force
            Start-Sleep 5
          }

          #Final attempt to stop/disable batch
          Write-Output "Final attempt to stop and disable batch after killing the batch process"
          Stop-ServiceAndDisableStartup -ServiceName $batchSvcName
        }
        else
        {
          throw "Unable to find batch service"
        }
      }

    # Attempt to stop batch for up to 7 minutes with incrimental backoff of 4n seconds between attempts
    $output=""
    Invoke-RetryWithBackoff -Command { Stop-ServiceAndDisableStartup -ServiceName "DynamicsAxBatch" } `
                            -Logger $logger `
                            -TimeoutSeconds 420 `
                            -RetryIntervalSec 4 `
                            -TimeoutAction $killBatchProcess `
                            -CommandOutput ([ref]$output) `
                            -TreatTimeoutActionAsSuccess

    if($output -ne "")
    {
       Write-ServicingLog $output
    }

    # Get the AOS web root and find the path to the AOS kernel DLL and check that there are no file
    # locks on it. Throw an exception if there is a lock and it does not go away within the timeout.
    $webroot = Get-AosWebSitePhysicalPath
    $AosKernelPath = Join-Path -Path $webroot -ChildPath "bin\AOSKernel.dll"
    if (Test-Path -Path $AosKernelPath -ErrorAction SilentlyContinue)
    {
        Write-ServicingLog "Validating that $($AosKernelPath) is not locked..." -Vrb

        $AosKernelLocked = $true
        $MaxWaitSec = 300

        $AosKernelLocked = checkAosKernelFileLocked -AosKernelPathParam $AosKernelPath -MaxWaitSecParam $MaxWaitSec

        if ($AosKernelLocked)
        {            
            $w3wpName = "w3wp"
            $axhostName = "axhost"
            $batchname = "Batch"
            $w3wpOraxHostTerminated  = $false

            # Get the locking processes
            Get-Process | foreach {
                $processVar = $_;
                $_.Modules | foreach {
                    if($_.FileName -eq $AosKernelPath)
                    {                        
                        # Terminate w3wp or axhost or batch process
                        if ($processVar.Name -eq $w3wpName -or $processVar.Name -eq $axhostName -or $processVar.Name -eq $batchname)
                        {
                            Write-ServicingLog "Terminate the locking process: $($processVar.Name) PID:$($processVar.id) Locked file:$($AosKernelPath), and try to continue." -Vrb
                            Stop-Process -Id $processVar.Id -Force                            
                            $w3wpOraxHostTerminated = $true
                        }
                    }
                }
            }

            if ($w3wpOraxHostTerminated)
            {
                # Check again for file locking.
                Write-ServicingLog "Revalidating that $($AosKernelPath) is not locked..." -Vrb
                $AosKernelLocked = checkAosKernelFileLocked -AosKernelPathParam $AosKernelPath -MaxWaitSecParam $MaxWaitSec
            }

            # still locked
            if ($AosKernelLocked)
            {
                $LockingProcessFound = $false
                # Get the locking processes
                Get-Process | foreach {
                    $processVar = $_;
                    $_.Modules | foreach {
                        if($_.FileName -eq $AosKernelPath)
                        {
                            $LockingProcessFound = $true
                            Write-ServicingLog "Locking process: $($processVar.Name) PID:$($processVar.id) Locked file:$($AosKernelPath)" -Vrb
                        }
                    }
                }

                $ErrorMessage = "File locks found on $($AosKernelPath) for more than $($MaxWaitSec) seconds."
                if ($LockingProcessFound)
                {
                    $ErrorMessage += " Check previous logs for identified locking processes."
                }
                else
                {
                    $ErrorMessage += " No locking process was able to be automatically identified. Please manually check which processes are keeping a lock on the file."
                }

                throw $ErrorMessage
            }
            else
            {
                Write-ServicingLog "No file locks found on $($AosKernelPath) after terminating processes." -Vrb
            }
        }
        else
        {
            Write-ServicingLog "No file locks found on $($AosKernelPath)." -Vrb
        }
    }
    else
    {
        Write-ServicingLog "Warning: No AOS kernel DLL found at: $($AosKernelPath)." -Vrb
    }
}
catch
{
    # Ensure non-zero exit code if an exception is caught and no exit code set.
    if ($ExitCode -eq 0)
    {
        $ExitCode = 1024
    }

    $ErrorMessage = "Error during AOS stop: $($_)"

    Write-ServicingLog $ErrorMessage
    Write-ServicingLog $($_) -Vrb
    Write-ServicingLog "AOS stop script failed with exit code: $($ExitCode)."

    # Use throw to indicate error to AXUpdateInstaller.
    # In case of exceptions, the output is not captured, so only the error message and
    # log file contents will be available for diagnostics.
    throw "$($ErrorMessage) [Log: $(Get-ServicingLog)]"
}

Write-ServicingLog "AOS stop script completed with exit code: $($ExitCode)."
exit $ExitCode
# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCA25bYQ0dQc0VH6
# EKu4W9dTbOuaZqZwvKZRVHlVmKgNb6CCDYEwggX/MIID56ADAgECAhMzAAAB32vw
# LpKnSrTQAAAAAAHfMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAxMjE1MjEzMTQ1WhcNMjExMjAyMjEzMTQ1WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQC2uxlZEACjqfHkuFyoCwfL25ofI9DZWKt4wEj3JBQ48GPt1UsDv834CcoUUPMn
# s/6CtPoaQ4Thy/kbOOg/zJAnrJeiMQqRe2Lsdb/NSI2gXXX9lad1/yPUDOXo4GNw
# PjXq1JZi+HZV91bUr6ZjzePj1g+bepsqd/HC1XScj0fT3aAxLRykJSzExEBmU9eS
# yuOwUuq+CriudQtWGMdJU650v/KmzfM46Y6lo/MCnnpvz3zEL7PMdUdwqj/nYhGG
# 3UVILxX7tAdMbz7LN+6WOIpT1A41rwaoOVnv+8Ua94HwhjZmu1S73yeV7RZZNxoh
# EegJi9YYssXa7UZUUkCCA+KnAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUOPbML8IdkNGtCfMmVPtvI6VZ8+Mw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDYzMDA5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAnnqH
# tDyYUFaVAkvAK0eqq6nhoL95SZQu3RnpZ7tdQ89QR3++7A+4hrr7V4xxmkB5BObS
# 0YK+MALE02atjwWgPdpYQ68WdLGroJZHkbZdgERG+7tETFl3aKF4KpoSaGOskZXp
# TPnCaMo2PXoAMVMGpsQEQswimZq3IQ3nRQfBlJ0PoMMcN/+Pks8ZTL1BoPYsJpok
# t6cql59q6CypZYIwgyJ892HpttybHKg1ZtQLUlSXccRMlugPgEcNZJagPEgPYni4
# b11snjRAgf0dyQ0zI9aLXqTxWUU5pCIFiPT0b2wsxzRqCtyGqpkGM8P9GazO8eao
# mVItCYBcJSByBx/pS0cSYwBBHAZxJODUqxSXoSGDvmTfqUJXntnWkL4okok1FiCD
# Z4jpyXOQunb6egIXvkgQ7jb2uO26Ow0m8RwleDvhOMrnHsupiOPbozKroSa6paFt
# VSh89abUSooR8QdZciemmoFhcWkEwFg4spzvYNP4nIs193261WyTaRMZoceGun7G
# CT2Rl653uUj+F+g94c63AhzSq4khdL4HlFIP2ePv29smfUnHtGq6yYFDLnT0q/Y+
# Di3jwloF8EWkkHRtSuXlFUbTmwr/lDDgbpZiKhLS7CBTDj32I0L5i532+uHczw82
# oZDmYmYmIUSMbZOgS65h797rj5JJ6OkeEUJoAVwwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAd9r8C6Sp0q00AAAAAAB3zAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgZ4bxuDkH
# KXOPNZ8537UhGsnZlncfvPMVrQtkGcU5rxIwQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQCsOLzUpOBQDcuOrCEZ8oNHbp6fBd5rxQjkowrZC0UD
# tCljELNfN6O1R8E+aqIe8XGH9pGp7i9VNdImvku4WEHpum9/k1H+YAkHRTwR98r7
# BIzK0t2XqujTiEFTjcYXm8SMgBqtH0VVvqiCeL4CX/og1GfHSz1jzzSlLOcZ/0Hp
# E1EvVUXgdwlqSjCApVvtuTmUtefsbJTF4ac3IiGnENnCF6R5tnHl0kNTEMmfsmkB
# 5GAEv0W6PJZR9MHBc4ec+u+IU+7M+Z7BzP4DpocLCJoToC6b+kcqftqLwPQkrvqz
# heW7KZFfN/UxmDRkRIeY/5oemhDDd1d6GSxA1KoZAbOvoYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIIpg3f4FFbXNpbYKsZpWR/TT7CRzBB/VT7QmHJmV
# lerWAgZg+vX+Yf4YEzIwMjEwNzMwMTc0NzM1LjYxM1owBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjo0RDJGLUUzREQtQkVFRjElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABX8OuZVblU1jsAAAA
# AAFfMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTIxMDExNDE5MDIxOVoXDTIyMDQxMTE5MDIxOVowgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjo0RDJG
# LUUzREQtQkVFRjElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBALw9efmC2WQ9uaaw7k4g
# xHSSCEoJLk22FTAaF8jYbAMkQC6DQF0WPnIheLM1ERTuQ9FWbglf0mXbDd2KjezR
# Nlz53ycJIReiGUQOnw5vd4TgjLUxL17g3K0MP2nNhY/LyP98Ml/40X905egDbiIn
# dZdtHiDb1xfY17a7v1j9o3muc+MCgFt9fO+U4CDNUpMMMQJFr/9QlU4YdJawjbyK
# fK3Ltvqfq3lvgK0/HphiDtX5ch3beGNBKowKSTXhft8pwuXQProutWgB5PZmAN8X
# ZhACo4jWi/a0zgAJJcBqoXvS6InrWcH/Eqi/qVaj8Vs56/Z/6kaYZZu/1mSzLn5E
# ALMCAwEAAaOCARswggEXMB0GA1UdDgQWBBQl7OnTlc0rgZ7Fd7qlDFguYTU49TAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQAOgtfZLJYSbsE3W73nd0hLnqQqHSFl
# 2spHxzeXxM4uJT2uAVk/SLVzzjvZemUDBeOedKeXG8hctprpoQMpU3gbsNUnUaDe
# sDcmR+eELCwYa+VBkUCqsIGJmQlDwuDwNa67kyCEPyPW59Yu2w/djNrwNWSjtuRw
# fUFoDkjYyDjnXD0josi67qxJgW8rRqjl9a62hGzlzgE+aVLTT5IhK5z2X62Lph8j
# 9f4XjtCPnyeFKFmgBWHPY1HbbjUHfg91StCLxueH2LjZoQETWOJ+pxElicXwVP5B
# 0wlWkiauwug3rTKnDb5WKUb2llsnQgaegV+MQjMI7K6v+spvsMgRjPlhMIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjo0
# RDJGLUUzREQtQkVFRjElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUA+gfSqjdAndOFEaXOQyBCdupmQoeggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AOSursQwIhgPMjAyMTA3MzAyMTAwNTJaGA8yMDIxMDczMTIxMDA1MlowdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA5K6uxAIBADAKAgEAAgIXmAIB/zAHAgEAAgIRqzAK
# AgUA5LAARAIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBADKtUarllM+ysnd+
# wLu40NMxM5F3pm7tKpQmGSWHDI3ugW7pIQeNCoCK0bNGlpOZ6M0EMG3qhEfFVz8Z
# 31LGxINoK814UP46c/rNg+JCy1DETQH3NAL9NcyF68lfizSW1CiEtenScdWSaGrw
# E+zaGvqWSnouNk/qoSgz7tWYsfHEMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAFfw65lVuVTWOwAAAAAAV8wDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQgmATrd8XTrnAPmnQ7NmcwdUdKM9wongyWDosXfHJC0WUwgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCDQzXq1KxGsLuj0szktrnlhIRqmbwp5bVGc6Bu6
# hglMXDCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# X8OuZVblU1jsAAAAAAFfMCIEILf4KeYUqZGMl7dxJGHjgbP6DQIswWYcqkRKzrlY
# eP0RMA0GCSqGSIb3DQEBCwUABIIBACNGgT2Vs3tuNO4w7CQyz0/Q96IBHVdBGHWl
# 3PQI6U7n8LlTf2RG+7VtgSCFfw4GVPe5b4uUf6m3xjeQ3xS6ICbxipwgEH5dNfZB
# wC6qclK0B5XHe4mbXfQt7XXXB98ZRebVPZ+tMnitpVVMOQOq475LVqBEVvNH/Ad2
# Sz1ZfmQ+mMlMCh1sJ7n4Ol/RTzOclQ+2Yv3sBZrunE5TjJ8nj0JnattJ93E9BJrg
# aRLUOriiLGXgXnYtjoV/jfhQCW/3/KWuLF3ZuSmQTrGAMxOJrQ3u5aZKCng50XGS
# DzMHlGB4u1sSk465cgbsKSTdQF6iU1pzRrtmhhN7BsLVENXbpVg=
# SIG # End signature block
