﻿[CmdletBinding()]
param
(
	# Directory to place logs. Passed in when upgrade is triggered through LCS.
    [string]
	$LogDir
)

Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\AOSEnvironmentUtilities.psm1" -DisableNameChecking

function Get-ApplicationEnvironment
{
    $ErrorActionPreference = 'Stop'
	       
    $webroot = Get-AosWebSitePhysicalPath
    $InstalledPlatformBuildVersion = Get-ProductPlatformBuildFullVersion
	$isKeyVault = Get-IsMSIKeyVault

    if (($isKeyVault -eq $true) -and ($InstalledPlatformBuildVersion -ne $null) -and ($InstalledPlatformBuildVersion.Build -ge 5644))
	{        
        # Get-PackageRoot comes from CommonRollbackUtilities.psm1. it resolves the root of the deployable package
        $packageRoot = Get-PackageRoot
        $packageTemp = ""
        if(![string]::IsNullOrEmpty($packageRoot) -and (Test-Path $packageRoot))
        {
            $packageTemp = Join-Path $packageRoot "RunbookWorkingFolder\KeyVaultAssemblyTemp"
        }

        if ([string]::IsNullOrEmpty($packageTemp))
        {   
            $suffix = $(Get-Date).ToString("yyyyMMdd_HHmmss")
            $dirName = "AssemblyTemp_$suffix"
            $packageTemp = Join-Path $env:Temp "dirName"
        }

        if(!(Test-Path $packageTemp -PathType Container))
        {
            New-Item $packageTemp -ItemType Container | Out-Null
        }
        
        #need to load the Microsoft.Dynamics.ApplicationPlatform.Environment.dll and all the dll it referenced
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Configuration.Base.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.Instrumentation.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Framework.EncryptionEngine.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.BusinessPlatform.SharedTypes.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.KeyVaultHelper.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Carbon.Flighting.Runtime.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Carbon.CertificateCommon.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Commerce.Flighting.Common.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Flighting.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.XppServices.Instrumentation.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.CE.VaultSDK.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.CE.VaultSDK.ADALv2.AuthHelper.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.KeyVault.Core.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.KeyVault.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.Common.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.Common.NetFramework.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Hyak.Common.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Authentication.Instrumentation.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.Extensions.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.Extensions.Desktop.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.IdentityModel.Clients.ActiveDirectory.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Practices.EnterpriseLibrary.TransientFaultHandling.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Newtonsoft.Json.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.SharedUtility.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp    
                
        try
        {
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
            $fileTempPath =  Join-Path $packageTemp $(Get-Item $dllPath).Name
	        if (Test-Path $fileTempPath)
            {
                # Load file from temp folder
	            [System.Reflection.Assembly]::LoadFrom($fileTempPath)
            }
        }
        catch
        {    
            #write-log "Failed to load from temp folder with Exception: [$($_.Exception)]"
            throw $_
        }
	}              
    else
    {
        #need to load the Microsoft.Dynamics.ApplicationPlatform.Environment.dll and all the dll it referenced
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Configuration.Base.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.Instrumentation.dll'
        if (Test-Path $dllPath)
        {
            Load-DllinMemory -dllPath $dllPath
        }
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Framework.EncryptionEngine.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.BusinessPlatform.SharedTypes.dll'
        Load-DllinMemory -dllPath $dllPath  
	}

    $config = [Microsoft.Dynamics.ApplicationPlatform.Environment.EnvironmentFactory]::GetApplicationEnvironment()
    
    return $config
}

function Load-DllinMemory([string] $dllPath)
{
    #try catch as not all dll exist in RTM version, some dependency/dll are introduced at update 1 or later
    #powershell cannot unload dll once it's loaded, the trick is to create an in-memory copy of the dll than load it
    #after the loading of in-memory dll, the physical dll stay unlocked

    try
    {
	    if (Test-Path $dllPath)
        {
            $bytes = [System.IO.File]::ReadAllBytes($dllPath)
            [System.Reflection.Assembly]::Load($bytes) | Out-Null
        }
    }
    catch
    {}
}

function CopyDllToTempFolder([string] $dllPath, [string] $packageTemp)
{
    #try catch as not all dll exist in RTM version, some dependency/dll are introduced at update 1 or later    
    $fileTempPath =  Join-Path $packageTemp $(Get-Item $dllPath).Name
    try
    {
	    if ((Test-Path $dllPath) -And !$(Test-Path $fileTempPath))
        {
            $Buffer = New-Object 'byte[]' 1024
            # Open file in read mode
            $SourceFile = [System.IO.File]::Open($dllPath, [System.IO.FileMode]::Open, [System.IO.FileAccess]::Read, [System.IO.FileShare]::Read)
            # Create the new file
            $DestinationFile = [System.IO.File]::Open($fileTempPath, [System.IO.FileMode]::CreateNew)
            try{
                # Copy the contents of the source file to the destination
                while(($readLength = $SourceFile.Read($Buffer, 0, $Buffer.Length)) -gt 0)
                {
                    $DestinationFile.Write($Buffer, 0, $readLength)
                }
            }
            catch{
                throw $_
            }
            finally{
                $SourceFile.Close()
                $DestinationFile.Close()
            }
        }
    }
    catch
    {
        Write-Log "Failed to copy file to temp folder with Exception: [$($_.Exception)]"
        throw $_
    }    
}

function Get-IsMSIKeyVault
{
    $webroot = Get-AosWebSitePhysicalPath
    $webconfig=join-path $webroot "web.config"
    $isMSIKeyVault=$false
    
    [System.Xml.XmlDocument] $xd=new-object System.Xml.XmlDocument
    $xd.Load($webconfig)
    $ns=New-Object System.Xml.XmlNamespaceManager($xd.NameTable)
    $ns.AddNamespace("ns",$xd.DocumentElement.NamespaceURI)

    $key="CertificateHandler.IsMSIKeyVault"

    $isKeyVaultValue = $xd.SelectSingleNode("//ns:add[@key='$key']",$ns)

    if($isKeyVaultValue -ne $null)
    {        
        $isMSIKeyVault = $isKeyVaultValue.GetAttribute("value")
    }

    return $isMSIKeyVault 
}

Start-Sleep -s 20

Import-Module "$PSScriptRoot\AOSCommon.psm1" -DisableNameChecking

$datetime=get-date -Format "MMddyyyyhhmmss"

if(!$LogDir)
{
    $LogDir = $PSScriptRoot
}

$log=join-path $LogDir "AutoRunDVT_AOS_$datetime.log"
  
Initialize-Log $log

$TestList = Get-ChildItem "$PSScriptRoot\DVT" -Filter 'Validate*.ps1'

$config = Get-ApplicationEnvironment

$ServiceName = $config.Infrastructure.ApplicationName
$endpointValue = $config.Infrastructure.HostUrl
$batchService = 'DynamicsAxBatch'
$serviceState = 'Running'
$appPoolState = 'Started'
$installPath = $config.Infrastructure.AOSWebRoot
$manifestPath = Join-Path $installPath 'web.config'
$appPoolName = $config.Infrastructure.ApplicationName
#Need to update the environmentassembly to add this value.
[xml]$WebConfig = Get-Content "$(join-path $installPath "web.config")"
$IsPrivateAOSInstance = $($WebConfig.configuration.appSettings.add | where { $_.key -eq 'Infrastructure.IsPrivateAOSInstance' }).value
$ValidateBatch = ([System.String]::IsNullOrWhiteSpace($IsPrivateAOSInstance) -or ![System.Convert]::ToBoolean($IsPrivateAOSInstance))

if (!$env:SERVICEDRIVE)
{
	Write-Log '%SERVICEDRIVE% was not defined. Exiting.' -Warning
	exit
}

Write-Log "Running DVT Process for $ServiceName"

$DiagnosticsPath = Join-Path -Path $env:SERVICEDRIVE -ChildPath "DynamicsDiagnostics\$ServiceName"

foreach ($test in $testList)
{
    Write-Log "Processing $test"    
    
    $DVTLocalRoot = Join-Path -Path $DiagnosticsPath -ChildPath ([System.IO.Path]::GetFileNameWithoutExtension($test))

    $DVTLocalBin = Join-Path $DVTLocalRoot "input"
    if(-not (Test-Path -Path $DVTLocalBin))
    {
        Write-Log "Creating DVT local bin at: '$DVTLocalBin'"
        New-Item -Path $DVTLocalBin -Type Directory -Force | Out-Null
    }   
        
    $DVTScript = Join-Path -Path "$PSScriptRoot\DVT" -ChildPath $test
    Write-Log "Copy DVT Script '$DVTScript' to $DVTLocalBin"
    Copy-Item -Path $DVTScript -Destination $DVTLocalBin -Recurse -Force | Out-Null

    Write-Log "Copy AosCommon.psm1 to DVTLocalBin"
    $AosCommon = Join-Path -Path $PSScriptRoot -ChildPath "AosCommon.psm1"
    Copy-Item -Path $AosCommon -Destination $DVTLocalBin -Recurse -Force | Out-Null

        
    [string]$DVTOutputBin = Join-Path -Path $DVTLocalRoot -ChildPath "Output"

    ####################################################################################
    ## DVT input XML Template
    ####################################################################################
    [xml]$xmlTemplate = "<?xml version=`"1.0`"?>
    <DVTParameters xmlns:xsi=`"http://www.w3.org/2001/XMLSchema-instance`" xmlns:xsd=`"http://www.w3.org/2001/XMLSchema`">
        <AosWebRootPath>$installPath</AosWebRootPath>
        <ServiceName>$ServiceName</ServiceName>        
        <OutputPath>$DVTOutputBin</OutputPath>
        <EndPoint>$endpointValue</EndPoint>
        <InstallPath>$installPath</InstallPath>
        <ManifestPath>$manifestPath</ManifestPath>
        <BatchService>$batchService</BatchService>
        <ServiceState>$serviceState</ServiceState>
        <AppPoolName>$appPoolName</AppPoolName>
        <AppPoolState>$appPoolState</AppPoolState>
        <ValidateBatch>$ValidateBatch</ValidateBatch>
    </DVTParameters>"

    $XMLInputPath = Join-Path -Path $DVTLocalBin -ChildPath "$([System.IO.Path]::GetFileNameWithoutExtension($test)).xml"
    Write-Log "Executing DVT XML at: $XMLInputPath"
    $xmlTemplate.InnerXml | Out-File -FilePath $XMLInputPath -Force -Encoding utf8
    
    ####################################################################################
    ## Execute DVT Script
    ####################################################################################
    try
    {
        $DVTLocalScript = Join-Path -Path $DVTLocalBin -ChildPath $test
        if(Test-Path -Path $DVTLocalScript)
        {
            Write-Log "Executing DVT Script: $DVTLocalScript"
            $commandArgs = @{
                "InputXML" = $XMLInputPath;                
                "Log" = $Log
            }
    
            & $DVTLocalScript @commandArgs
            
        }
        else
        {
            throw "$DVTLocalScript was not found."
        }
    }
    catch
    {     
        Write-Exception $_       
        throw "DVT Script Failed, see $log for details."
    }
}    

# SIG # Begin signature block
# MIIjgwYJKoZIhvcNAQcCoIIjdDCCI3ACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCA86YOltOSTzBet
# okCiOQ7oUgszHKt0kwcQFZAZfzQiC6CCDYEwggX/MIID56ADAgECAhMzAAAB32vw
# LpKnSrTQAAAAAAHfMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAxMjE1MjEzMTQ1WhcNMjExMjAyMjEzMTQ1WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQC2uxlZEACjqfHkuFyoCwfL25ofI9DZWKt4wEj3JBQ48GPt1UsDv834CcoUUPMn
# s/6CtPoaQ4Thy/kbOOg/zJAnrJeiMQqRe2Lsdb/NSI2gXXX9lad1/yPUDOXo4GNw
# PjXq1JZi+HZV91bUr6ZjzePj1g+bepsqd/HC1XScj0fT3aAxLRykJSzExEBmU9eS
# yuOwUuq+CriudQtWGMdJU650v/KmzfM46Y6lo/MCnnpvz3zEL7PMdUdwqj/nYhGG
# 3UVILxX7tAdMbz7LN+6WOIpT1A41rwaoOVnv+8Ua94HwhjZmu1S73yeV7RZZNxoh
# EegJi9YYssXa7UZUUkCCA+KnAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUOPbML8IdkNGtCfMmVPtvI6VZ8+Mw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDYzMDA5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAnnqH
# tDyYUFaVAkvAK0eqq6nhoL95SZQu3RnpZ7tdQ89QR3++7A+4hrr7V4xxmkB5BObS
# 0YK+MALE02atjwWgPdpYQ68WdLGroJZHkbZdgERG+7tETFl3aKF4KpoSaGOskZXp
# TPnCaMo2PXoAMVMGpsQEQswimZq3IQ3nRQfBlJ0PoMMcN/+Pks8ZTL1BoPYsJpok
# t6cql59q6CypZYIwgyJ892HpttybHKg1ZtQLUlSXccRMlugPgEcNZJagPEgPYni4
# b11snjRAgf0dyQ0zI9aLXqTxWUU5pCIFiPT0b2wsxzRqCtyGqpkGM8P9GazO8eao
# mVItCYBcJSByBx/pS0cSYwBBHAZxJODUqxSXoSGDvmTfqUJXntnWkL4okok1FiCD
# Z4jpyXOQunb6egIXvkgQ7jb2uO26Ow0m8RwleDvhOMrnHsupiOPbozKroSa6paFt
# VSh89abUSooR8QdZciemmoFhcWkEwFg4spzvYNP4nIs193261WyTaRMZoceGun7G
# CT2Rl653uUj+F+g94c63AhzSq4khdL4HlFIP2ePv29smfUnHtGq6yYFDLnT0q/Y+
# Di3jwloF8EWkkHRtSuXlFUbTmwr/lDDgbpZiKhLS7CBTDj32I0L5i532+uHczw82
# oZDmYmYmIUSMbZOgS65h797rj5JJ6OkeEUJoAVwwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVWDCCFVQCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAd9r8C6Sp0q00AAAAAAB3zAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgTyzjztgi
# Vrky53xY9c0M+0qxUAh/DzUpgO8DId4sak0wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQAz6uzc5wlsS5dS4O4y9me2QzGa/Unu1p7ASg+FrvGY
# gS2rhsRdhN26vU02a3pRr80ollJXyxQK+HGyFCWGhRq25xnBjwUdi+kymnfqjxNw
# KtsscaEZas4raEk6nF9tQkUDuhtfRLYsGzx1NL+cj9T8h+8GdjHnnkXJvfbQIhPs
# 9IIUyBmsqZ3P26TEQn+XWnQwhYViMcyoe8PQjCjhRRfbY49Nim4nDk2vx9R/VD28
# ga61vqjLasBRgBEOT5iPBt+c4XWKN82f7Wt89V1lrKNnK10TKWg/RGKrcqiR4SNY
# skdBeEIxOQxA+rKjRyImpQ8soXppnByzYtRPCGJ2IQWmoYIS4jCCEt4GCisGAQQB
# gjcDAwExghLOMIISygYJKoZIhvcNAQcCoIISuzCCErcCAQMxDzANBglghkgBZQME
# AgEFADCCAVEGCyqGSIb3DQEJEAEEoIIBQASCATwwggE4AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIOpnOUV1aM2M20BO/sOQliEAoPlF/LRoK+S+DLSh
# b5x4AgZg+YSfUq4YEzIwMjEwNzMwMTc0NzQyLjk2N1owBIACAfSggdCkgc0wgcox
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJTAjBgNVBAsTHE1p
# Y3Jvc29mdCBBbWVyaWNhIE9wZXJhdGlvbnMxJjAkBgNVBAsTHVRoYWxlcyBUU1Mg
# RVNOOjQ5QkMtRTM3QS0yMzNDMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFt
# cCBTZXJ2aWNloIIOOTCCBPEwggPZoAMCAQICEzMAAAFJgAhKuwmgMwsAAAAAAUkw
# DQYJKoZIhvcNAQELBQAwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
# b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwHhcN
# MjAxMTEyMTgyNTU3WhcNMjIwMjExMTgyNTU3WjCByjELMAkGA1UEBhMCVVMxEzAR
# BgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
# Y3Jvc29mdCBDb3Jwb3JhdGlvbjElMCMGA1UECxMcTWljcm9zb2Z0IEFtZXJpY2Eg
# T3BlcmF0aW9uczEmMCQGA1UECxMdVGhhbGVzIFRTUyBFU046NDlCQy1FMzdBLTIz
# M0MxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2UwggEiMA0G
# CSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvE/uJD4XYdtp6OSoZPkolG9p3CWcw
# Lle1XkQMluEejNzIQMeWMsd8ZbujdfjJfWG/c3SOmZBwUAWEfVSdlCaMayt8gQHk
# KFikoD/bY1Q4y7Rfda7sCJw8CXf5wfLqzsGMvKkhtBFGOhqN/YqQm5j7B0c9qq12
# 8i40lrrspOm31Vel+UAqlVt1L7Jb5MGKMWmEaoQpgvLGQq9NPBDMdgVjm1XwFFVc
# peBRWWn3Vb0UCWA6tqRuFLLaOsheYCA/jw6zw3+UwITm3JmnQVMIr9HALgvKY2uS
# 7lnSKiEaKRjb1oB1v0U0s8WPzkgbVpsyro+Uml2v7VreagzQzwvR+dWtAgMBAAGj
# ggEbMIIBFzAdBgNVHQ4EFgQUVnea8aPvuLS8NTXWT8mpc+pvJIEwHwYDVR0jBBgw
# FoAU1WM6XIoxkPNDe3xGG8UzaFqFbVUwVgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDov
# L2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljVGltU3RhUENB
# XzIwMTAtMDctMDEuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+aHR0
# cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNUaW1TdGFQQ0FfMjAx
# MC0wNy0wMS5jcnQwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDCDAN
# BgkqhkiG9w0BAQsFAAOCAQEAEN54Cz4g7OBKqc8iwqLzNdQj2OCTxKmH+jr3Ayp+
# AY/1qw4d77A/4WCP8g8PdToYiC47UXC6Fd2epJ07Olen50f88rFAz49H5BV7XlwP
# jiyE1ZU0vLKHiCcB2mibalui7W0dtg4W4bIqi7UlQkhBLERS5nn+zHYQg/rFQUQv
# vJrKpx2NM0MFgv2hki4B3JkDUfFwoHxYbAAJR1UtXaH+0PG1BW5yL1DLs451q7D/
# RsHGmvx1M6+RKSr3qCUicbfQEa8vaP+nKJ0T/Da5vSqpSKocfD8dwM3Unn0tpoC+
# lKmqQMDbllghGs7NVhps+9xG95s7beCMr3AuUZG/E6RQaTCCBnEwggRZoAMCAQIC
# CmEJgSoAAAAAAAIwDQYJKoZIhvcNAQELBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jvc29mdCBSb290IENlcnRp
# ZmljYXRlIEF1dGhvcml0eSAyMDEwMB4XDTEwMDcwMTIxMzY1NVoXDTI1MDcwMTIx
# NDY1NVowfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
# BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQG
# A1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwggEiMA0GCSqGSIb3
# DQEBAQUAA4IBDwAwggEKAoIBAQCpHQ28dxGKOiDs/BOX9fp/aZRrdFQQ1aUKAIKF
# ++18aEssX8XD5WHCdrc+Zitb8BVTJwQxH0EbGpUdzgkTjnxhMFmxMEQP8WCIhFRD
# DNdNuDgIs0Ldk6zWczBXJoKjRQ3Q6vVHgc2/JGAyWGBG8lhHhjKEHnRhZ5FfgVSx
# z5NMksHEpl3RYRNuKMYa+YaAu99h/EbBJx0kZxJyGiGKr0tkiVBisV39dx898Fd1
# rL2KQk1AUdEPnAY+Z3/1ZsADlkR+79BL/W7lmsqxqPJ6Kgox8NpOBpG2iAg16Hgc
# sOmZzTznL0S6p/TcZL2kAcEgCZN4zfy8wMlEXV4WnAEFTyJNAgMBAAGjggHmMIIB
# 4jAQBgkrBgEEAYI3FQEEAwIBADAdBgNVHQ4EFgQU1WM6XIoxkPNDe3xGG8UzaFqF
# bVUwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8GA1Ud
# EwEB/wQFMAMBAf8wHwYDVR0jBBgwFoAU1fZWy4/oolxiaNE9lJBb186aGMQwVgYD
# VR0fBE8wTTBLoEmgR4ZFaHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwv
# cHJvZHVjdHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3JsMFoGCCsGAQUFBwEB
# BE4wTDBKBggrBgEFBQcwAoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9j
# ZXJ0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcnQwgaAGA1UdIAEB/wSBlTCB
# kjCBjwYJKwYBBAGCNy4DMIGBMD0GCCsGAQUFBwIBFjFodHRwOi8vd3d3Lm1pY3Jv
# c29mdC5jb20vUEtJL2RvY3MvQ1BTL2RlZmF1bHQuaHRtMEAGCCsGAQUFBwICMDQe
# MiAdAEwAZQBnAGEAbABfAFAAbwBsAGkAYwB5AF8AUwB0AGEAdABlAG0AZQBuAHQA
# LiAdMA0GCSqGSIb3DQEBCwUAA4ICAQAH5ohRDeLG4Jg/gXEDPZ2joSFvs+umzPUx
# vs8F4qn++ldtGTCzwsVmyWrf9efweL3HqJ4l4/m87WtUVwgrUYJEEvu5U4zM9GAS
# inbMQEBBm9xcF/9c+V4XNZgkVkt070IQyK+/f8Z/8jd9Wj8c8pl5SpFSAK84Dxf1
# L3mBZdmptWvkx872ynoAb0swRCQiPM/tA6WWj1kpvLb9BOFwnzJKJ/1Vry/+tuWO
# M7tiX5rbV0Dp8c6ZZpCM/2pif93FSguRJuI57BlKcWOdeyFtw5yjojz6f32WapB4
# pm3S4Zz5Hfw42JT0xqUKloakvZ4argRCg7i1gJsiOCC1JeVk7Pf0v35jWSUPei45
# V3aicaoGig+JFrphpxHLmtgOR5qAxdDNp9DvfYPw4TtxCd9ddJgiCGHasFAeb73x
# 4QDf5zEHpJM692VHeOj4qEir995yfmFrb3epgcunCaw5u+zGy9iCtHLNHfS4hQEe
# gPsbiSpUObJb2sgNVZl6h3M7COaYLeqN4DMuEin1wC9UJyH3yKxO2ii4sanblrKn
# QqLJzxlBTeCG+SqaoxFmMNO7dDJL32N79ZmKLxvHIa9Zta7cRDyXUHHXodLFVeNp
# 3lfB0d4wwP3M5k37Db9dT+mdHhk4L7zPWAUu7w2gUDXa7wknHNWzfjUeCLraNtvT
# X4/edIhJEqGCAsswggI0AgEBMIH4oYHQpIHNMIHKMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1lcmljYSBP
# cGVyYXRpb25zMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjo0OUJDLUUzN0EtMjMz
# QzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZaIjCgEBMAcG
# BSsOAwIaAxUAP+Wxrucu9GSImwAdD52BRGupqHeggYMwgYCkfjB8MQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQg
# VGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIFAOSujuAwIhgPMjAy
# MTA3MzAyMjQ0NDhaGA8yMDIxMDczMTIyNDQ0OFowdDA6BgorBgEEAYRZCgQBMSww
# KjAKAgUA5K6O4AIBADAHAgEAAgIBYTAHAgEAAgITQzAKAgUA5K/gYAIBADA2Bgor
# BgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIBAAIDB6EgoQowCAIBAAID
# AYagMA0GCSqGSIb3DQEBBQUAA4GBAI7YJ927S6QRm3xpW+SBeBzbUH5hyDcYntI+
# ux+VXWi+PAWMHcFh+K6+JA5is8ItSQob5aSfrCSj3i25NHNszZ9n3sFsTzw9duIH
# IUk/lIJfJJqvWixzl5lh/KMiDaeeTAFSnoiWDCyXQ6O/rycMh32qCoBrVbfyLCDq
# 1D4aQJFKMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIw
# MTACEzMAAAFJgAhKuwmgMwsAAAAAAUkwDQYJYIZIAWUDBAIBBQCgggFKMBoGCSqG
# SIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0BCQQxIgQg6o+KYO4LFp/C
# sxd62sdHRAIKsZ0YhBaJiCo7no+lQ7YwgfoGCyqGSIb3DQEJEAIvMYHqMIHnMIHk
# MIG9BCAolfr8WH1478zdhngQdSqc7DQL0sZx0OXG9a0fueihsjCBmDCBgKR+MHwx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1p
# Y3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAABSYAISrsJoDMLAAAAAAFJ
# MCIEIH+Uu0YKM6lJ54ZXikYz3MQFaHNupVvVqS6nDS7wCvxvMA0GCSqGSIb3DQEB
# CwUABIIBAH9h1lxJr/rqYR0ZMr7cMG7siLlb6ZsgYdGfq4rcjSRXdJIRCZGN+XdR
# JmUHe21z4Giln1GTk9IxIsCZyLFIdo5ZPxfNb/O02iCJacxDYi87l+OYP459IvTM
# 5ARUWmaU0idEN1RPoTTCo0hs5JhiIiMS/GrKoVt7C7QKfAuv5jxWB4d1vNCslv9Z
# kUQLU9OJV9pebc4jR+kGkrCvzl/fFInzb1MyjGzXMAQyaEOk/qACECRQraD/Gwj5
# bSJxXNlJtG1V3+aUKkWs+Ggw1P7v1owwh/3DREz9qffTkav8iSEuVpKNNorXJKZ8
# iNXBYJFsH8yLF6LbD/na2ZzsCfpSzZ0=
# SIG # End signature block
