﻿<#
.SYNOPSIS
    Swap the AOS staging environment to production.

.DESCRIPTION

    Copyright © 2019 Microsoft. All rights reserved.
#>
[CmdletBinding()]
param
(
    [Parameter(Mandatory = $false, HelpMessage = "The path to the directory in which to write log files.")]
    [string]$LogDir,
    [Parameter(Mandatory = $false, HelpMessage = "The ID of the runbook this is executing as part of.")]
    [string]$RunbookId
)

$ErrorActionPreference = "Stop"
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\AosEnvironmentUtilities.psm1" -Force -DisableNameChecking

# Initialize the log file to use with Write-ServicingLog.
Set-ServicingLog -LogDir $LogDir

# Initialize exit code.
[int]$ExitCode = 0

# Get the runbook ID
if ([string]::IsNullOrWhiteSpace($RunbookId))
{
    $RunbookId = Get-RunbookId

    #If the runbook ID still wasn't found, set it to a unique value
    if ([string]::IsNullOrWhiteSpace($RunbookId))
    {
        $runbookIdPlaceholder = [DateTime]::UtcNow.Ticks.ToString()
        Write-ServicingLog "Warning: Runbook ID was not specified and could not be determined. Using [$runbookIdPlaceholder] as a temporary Runbook ID."
        $RunbookId = $runbookIdPlaceholder
    }
}

# Progress breadcrumb names
$createStagingCompleted = "CreateAOSStagingEnvCompleted"
$swapStagingStarted = "SwapAOSStagingEnvStarted"
$swapStagingCompleted = "SwapAOSStagingEnvCompleted"

try
{
    <#
        Directories used in this script:
        - <service volume>\AOSServiceStaging
            The initial staging directory pre-swap
        - <service volume>\AOSServiceBackup
            The post-swap backup of the original AOSService folder
        - <service volume>\AOSService
            The current AOSService instance
        - <deployable package root>\RunbookWorkingFolder\<runbook id>\<machine>\AOSService\<step>\Backup\AOSServiceBackup
            The backup location on completion of SwapAOSStagingEnv.ps1
    #>

    Write-ServicingLog "Starting AOS swap staging environment script..."

    # If Copy Staging Folder task is not disabled, the content of AOSServiceStaging folder may not exactly match the AOSService folder
    Assert-CopyStagingTaskNotRunning

    $aosWebServicePath = Get-AosServicePath
    $stagingFolder = Get-AosServiceStagingPath
    $backupFolder = Get-AosServiceBackupPath

    #region Initialize script progress tracking
    $progressFileName = "AOSStagingProgress_$($RunbookId).json"
    $progressFileInitial = Join-Path -Path $stagingFolder -ChildPath $progressFileName
    $progressFilePostSwap = Join-Path -Path $aosWebServicePath -ChildPath $progressFileName

    if (Test-Path $progressFilePostSwap)
    {
        # The swap completed so the progress file from the initial creation of AOSServiceStaging should now be in AOSService
        Initialize-ScriptProgress -progressFile $progressFilePostSwap -ErrorAction Continue
    }
    elseif (Test-Path $progressFileInitial)
    {
        # The swap was not started or incomplete so the progress file should still exist in AOSServiceStaging
        Initialize-ScriptProgress -progressFile $progressFileInitial -ErrorAction Continue
    }
    else
    {
        # Not in the expected location. Log warning - this fails later on when checking for successful completion of the staging directory.
        Write-ServicingLog "Warning: unable to detect AOSService staging progress file [$progressFileName] for staging creation."   
    }
    #endregion

    if (Test-Path -Path $stagingFolder)
    {
        # Check that creation of the staging folder completed successfully.
        Write-ServicingLog "Checking for successful AOS staging environment creation..."
        if (Test-ScriptProgress $createStagingCompleted)
        {
            Write-ServicingLog "Detected successful creation of the AOS Staging environment at [$(Get-ScriptProgress $createStagingCompleted)] for runbook [$RunbookId]"
        }
        else 
        {
            # Log warning if the completion breadcrumb was not found 
            # Leave it to the parent runbook to fail on unfulfilled step dependencies
            Write-ServicingLog "Warning: No completion detected for pre-req step CreateAOSStagingEnv."
        }

        Add-ScriptProgress $swapStagingStarted -ErrorAction Continue

        Write-ServicingLog "Terminating watchdog process if running."
        KillWatchdogProcess

        # Ensure AOS service is stopped.
        # This is a mitigation in case the machine was rebooted or the AOS service started while deployable package.
        Write-ServicingLog "Calling script to stop the AOS..."
        & "$PSScriptRoot\AutoStopAOS.ps1"

        Write-ServicingLog "Terminating other user sessions..."
        KillAllOtherUserSession

        Write-ServicingLog "Terminating processes locking files under AOS service folder..."
        KillProcessLockingFolder -folder $aosWebServicePath

        Write-ServicingLog "Terminating watchdog process if running."
        KillWatchdogProcess

        if (Test-Path -Path $aosWebServicePath)
        {
            if (Test-Path -Path $backupFolder)
            {
                Write-ServicingLog "Removing existing $($backupFolder) backup folder..."
                Get-ChildItem -Path $backupFolder -Recurse | Remove-Item -Force -Recurse
                Remove-Item -Path $backupFolder -Force
            }

            Write-ServicingLog "Renaming $($aosWebServicePath) to AosServiceBackup..."
            Rename-Item -Path $aosWebServicePath -NewName "AosServiceBackup"
        }

        Write-ServicingLog "Renaming $($stagingFolder) to AosService..."
        Rename-Item -Path $stagingFolder -NewName "AosService"

        # $RunbookBackupFolder and $PSScriptRoot variables are populated directly from runbook.
        $DPBackupFolder = "$RunbookBackupFolder"

        if ([string]::IsNullOrEmpty($DPBackupFolder))
        {
            $DPBackupFolder = Join-Path -Path $PSScriptRoot -ChildPath "ManualAosServiceBackup"
        }

        Write-ServicingLog "Moving $($backupFolder) to $($DPBackupFolder)..."
        Move-Item -Path $backupFolder -Destination $DPBackupFolder -Force

        # Generate installation records.
        Write-ServicingLog "Generating installation records..."
        GenerateMetadataModuleInstallationInfo

        # If the script executed this far, the progress file is now in the swapped location. 
        # Re-initialize progres tracking to the swapped location and write completion.
        Initialize-ScriptProgress -progressFile $progressFilePostSwap -ErrorAction Continue
        Add-ScriptProgress $swapStagingCompleted -ErrorAction Continue
    }
    elseif (Test-ScriptProgress $swapStagingCompleted -ErrorAction SilentlyContinue)
    {
        Write-ServicingLog "Detected staging swap completed at [$(Get-ScriptProgress -ProgressStep $swapStagingCompleted)] for runbook [$($RunbookId)]"
    }
    else
    {
        Write-ServicingLog "Warning: No staging folder found at: $stagingFolder"
    }

    # Ensure that a log path exists.
    [string]$logPath = $LogDir
    if ([String]::IsNullOrWhiteSpace($logPath))
    {
        $logPath = "$PSScriptRoot"
    }

    #region Change IIS Ping Interval on config change
    if (Test-Path -Path "$PSScriptRoot\ChangeIISPingInterval.ps1")
    {
        $dt = [System.DateTime]::Now.ToString("yyyyMMddHHmmss");
        $logFile = Join-Path -Path $logPath -ChildPath "ChangeIISPingInterval.$dt.log"
        Write-ServicingLog "Change IIS Ping Interval..."
        Invoke-Expression "$PSScriptRoot\ChangeIISPingInterval.ps1 -log:`"$logFile`" -appPoolName:$(Get-AosAppPoolName)"
    }
    #endregion

    #region Disable IIS restart on config change
    if (Test-Path -Path "$PSScriptRoot\SetIISDisallowRotationOnConfigChange.ps1")
    {
        $dt = [System.DateTime]::Now.ToString("yyyyMMddHHmmss");
        $logFile = Join-Path -Path $logPath -ChildPath "SetIISDisallowRotationOnConfigChange.$dt.log"
        Write-ServicingLog "Disabling rotation on config change..."
        Invoke-Expression "$PSScriptRoot\SetIISDisallowRotationOnConfigChange.ps1 -log:`"$logFile`" -appPoolName:$(Get-AosAppPoolName) -disallowRotationOnConfigChangeValue:$true"
    }
    #endregion
}
catch
{
    # Ensure non-zero exit code if an exception is caught and no exit code set.
    if ($ExitCode -eq 0)
    {
        $ExitCode = 1024
    }

    $ErrorMessage = "Error during AOS swap staging: $($_)"

    Write-ServicingLog $ErrorMessage
    Write-ServicingLog $($_) -Vrb
    Write-ServicingLog "AOS swap staging environment script failed with exit code: $($ExitCode)."

    # Use throw to indicate error to AXUpdateInstaller.
    # In case of exceptions, the output is not captured, so only the error message and
    # log file contents will be available for diagnostics.
    throw "$($ErrorMessage) [Log: $(Get-ServicingLog)]"
}
finally
{
    # Back up the progress file for this script's execution.
    $backupDir = if ([string]::IsNullOrEmpty($LogDir)) { $PSScriptRoot } else { $LogDir }
    $logTimeSuffix = [DateTime]::UtcNow.ToString('yyyyMMdd-HHmmss')
    $scriptProgressFileBackup = Join-Path -Path $backupDir -ChildPath "SwapAOSStagingEnv-ProgressBackup-$logTimeSuffix.json"
    Copy-ScriptProgressFile -Destination $scriptProgressFileBackup -ErrorAction Continue
    
    Write-ServicingLog "Attempting to reset IIS"
    Reset-IIS
    
    if($?)
    {
        Write-ServicingLog "Successfully reset IIS"
    }
}

Write-ServicingLog "AOS swap staging environment script completed with exit code: $($ExitCode)."
exit $ExitCode
# SIG # Begin signature block
# MIInqQYJKoZIhvcNAQcCoIInmjCCJ5YCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAtS2vLNQBZhCok
# pk+5G0vxYkh4u8Ga+aGTZlmcXCf9D6CCDYEwggX/MIID56ADAgECAhMzAAACzI61
# lqa90clOAAAAAALMMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjIwNTEyMjA0NjAxWhcNMjMwNTExMjA0NjAxWjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQCiTbHs68bADvNud97NzcdP0zh0mRr4VpDv68KobjQFybVAuVgiINf9aG2zQtWK
# No6+2X2Ix65KGcBXuZyEi0oBUAAGnIe5O5q/Y0Ij0WwDyMWaVad2Te4r1Eic3HWH
# UfiiNjF0ETHKg3qa7DCyUqwsR9q5SaXuHlYCwM+m59Nl3jKnYnKLLfzhl13wImV9
# DF8N76ANkRyK6BYoc9I6hHF2MCTQYWbQ4fXgzKhgzj4zeabWgfu+ZJCiFLkogvc0
# RVb0x3DtyxMbl/3e45Eu+sn/x6EVwbJZVvtQYcmdGF1yAYht+JnNmWwAxL8MgHMz
# xEcoY1Q1JtstiY3+u3ulGMvhAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUiLhHjTKWzIqVIp+sM2rOHH11rfQw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDcwNTI5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAeA8D
# sOAHS53MTIHYu8bbXrO6yQtRD6JfyMWeXaLu3Nc8PDnFc1efYq/F3MGx/aiwNbcs
# J2MU7BKNWTP5JQVBA2GNIeR3mScXqnOsv1XqXPvZeISDVWLaBQzceItdIwgo6B13
# vxlkkSYMvB0Dr3Yw7/W9U4Wk5K/RDOnIGvmKqKi3AwyxlV1mpefy729FKaWT7edB
# d3I4+hldMY8sdfDPjWRtJzjMjXZs41OUOwtHccPazjjC7KndzvZHx/0VWL8n0NT/
# 404vftnXKifMZkS4p2sB3oK+6kCcsyWsgS/3eYGw1Fe4MOnin1RhgrW1rHPODJTG
# AUOmW4wc3Q6KKr2zve7sMDZe9tfylonPwhk971rX8qGw6LkrGFv31IJeJSe/aUbG
# dUDPkbrABbVvPElgoj5eP3REqx5jdfkQw7tOdWkhn0jDUh2uQen9Atj3RkJyHuR0
# GUsJVMWFJdkIO/gFwzoOGlHNsmxvpANV86/1qgb1oZXdrURpzJp53MsDaBY/pxOc
# J0Cvg6uWs3kQWgKk5aBzvsX95BzdItHTpVMtVPW4q41XEvbFmUP1n6oL5rdNdrTM
# j/HXMRk1KCksax1Vxo3qv+13cCsZAaQNaIAvt5LvkshZkDZIP//0Hnq7NnWeYR3z
# 4oFiw9N2n3bb9baQWuWPswG0Dq9YT9kb+Cs4qIIwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIZfjCCGXoCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAsyOtZamvdHJTgAAAAACzDAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQg57lYPqiR
# sUdyIJjtb+Z6QpvQu45zE/pkKEKXDwby3lQwQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQAyycYBxwP/KcUdqEFVrFspdul+6RpxJ50cPP83/zqx
# +VEnEel8KECFs4y9hE/FwJufW8esbgUtL6CbUZ28OdmZR5onOvfqnzPgsus55URz
# GmKVoB9qk8fHt2ZrP9nAC2ac0a4DMtDLR4Ovf7eNFV35N2E2/kEjmTbI2w2GS/Sb
# ugL3INkFTuZdBeTCRPxDVmnbRLw+sZ4CXXu0J8xmEn41pOrhwhr5KcbAzgYvFjz6
# yWZW9Xzam81vX34fHSc5GNv1UFxM2bIVcL2piSdVrniekjEWAk1BuTmgmlH8G6Yl
# izsd66g8rTsCLZSA7KttCJP+B8Eht+Ae467ucsxySX0toYIXCDCCFwQGCisGAQQB
# gjcDAwExghb0MIIW8AYJKoZIhvcNAQcCoIIW4TCCFt0CAQMxDzANBglghkgBZQME
# AgEFADCCAVQGCyqGSIb3DQEJEAEEoIIBQwSCAT8wggE7AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIFqxTJ2vg6ciSR5+Kzl73xwFe8hDg1B9UVYv13VK
# M/lvAgZjoaDH77gYEjIwMjMwMTA2MjE0NTE0Ljc1WjAEgAIB9KCB1KSB0TCBzjEL
# MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1v
# bmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEpMCcGA1UECxMgTWlj
# cm9zb2Z0IE9wZXJhdGlvbnMgUHVlcnRvIFJpY28xJjAkBgNVBAsTHVRoYWxlcyBU
# U1MgRVNOOkY4N0EtRTM3NC1EN0I5MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1T
# dGFtcCBTZXJ2aWNloIIRXDCCBxAwggT4oAMCAQICEzMAAAGuqgtcszSllRoAAQAA
# Aa4wDQYJKoZIhvcNAQELBQAwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
# bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
# b3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAw
# HhcNMjIwMzAyMTg1MTM3WhcNMjMwNTExMTg1MTM3WjCBzjELMAkGA1UEBhMCVVMx
# EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoT
# FU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEpMCcGA1UECxMgTWljcm9zb2Z0IE9wZXJh
# dGlvbnMgUHVlcnRvIFJpY28xJjAkBgNVBAsTHVRoYWxlcyBUU1MgRVNOOkY4N0Et
# RTM3NC1EN0I5MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNl
# MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAk4wa8SE1DAsdpy3Oc+lj
# wmDmojxCyCnSaGXYbbO1U+ieriCw4x7m72nl/Xs8gzUpeNRoo2Xd2Odyrb0uKqaq
# doo5GCA8c0STtD61qXkjJz5LyT9HfWAIa3iq9BWoEtA2K/E66RR9qkbjUtN0sd4z
# i7AieT5CsZAfYrjCM22JSmKsXY90JxuRfIAsSnJPZGvDMmbNyZt0KxxjQ3dEfGsx
# 5ZDeTuw23jU0Fk5P7ikKaTDxSSAqJIlczMqzfwzFSrH86VLzR0sNMd35l6LVLX+p
# sK1MbM2bRuPqp+SVQzckUAXUktfDC+qBlF0NBTrbbjC0afBqVNo4jRHR5f5ytw+l
# cYHbsQiBhT7SWjZofv1I2uw9YRx0EgJ3TJ+EVTaeJUl6kbORd60m9sXFbeI3uxyM
# t/D9LpRcXvC0TN041dWIjk/ZQzvv0/oQhn6DzUTYxZfxeMtXK8iy/PJyQngUWL6H
# XI8T6/NyQ/HMc6yItpp+5yzIyMBoAzxbBr7TYG6MQ7KV8tLKTSK/0i9Ij1mQlb+A
# u9DjZTT5TTflmFSEKpsoRYQwivbJratimtQwQpxd/hH3stU8F+wmduQ1S5ulQDgr
# WLuKNDWmRSW35hD/fia0TLt5KKBWlXOep+s1V6sK8cbkjB94VWE81sDArqUERDb2
# cxiNFePhAvK+YpGao4kz/DUCAwEAAaOCATYwggEyMB0GA1UdDgQWBBTTMG/fvyhg
# isGprXT+/O1kOmFR7jAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJlpxtTNRnpcjBf
# BgNVHR8EWDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3Bz
# L2NybC9NaWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcmww
# bAYIKwYBBQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3Lm1pY3Jvc29m
# dC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0El
# MjAyMDEwKDEpLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUFBwMI
# MA0GCSqGSIb3DQEBCwUAA4ICAQBdv5Hw/VUARA48rTMDEAMhc/hwlCZqu2NUUswS
# QtiHf08W1Vu3zhG/RDUZJNiaE/x/846+eYLl6PDc1zVVGLvitYZQhO/Xxaqvx4G8
# BJ3h4MDEVsDySc46b9nJKQwMNh1vrvfxpDTK+p/sBZyGA+e0Jz+eE1qlImaPNSR7
# sS+MHx6LQGdjTGX4BBxLEkb9Weyb0jA56vwTWaJUth8+f18gN1pq/Vur2L6Cdl/W
# FLtqkanFuK0ImvUoYPiMjIAGTEeF6g86GG1CbW7OcTtuUrEfylTtbYD56qCCw2Qz
# dUHSevNFkGqbhKYFI2E4/PLeh86YtxEr9qWg4Cvqd6GLyLmWGZODUuQ4DEKEvAe+
# W6IJj0r7a8im3jyKgr+H63PlGBV1v5LzHCfvbyU3wo+SQHZFrmKJyu+2ADnnBJR2
# HoUXFfF5L5uyAFrKftnJp9OkMzsFA4FjBqh2y5V/leAavIbHziThHnyY/AHdDT0J
# EAazfk063pOs9epzKU27pnPzFNANxomnnikrI6hbmIgMWOkud5dMSO1YIUKACjjN
# un0I0hOn3so+dzeBlVoy8SlTxKntVnA31yRHZYMrI6MOCEhx+4UlMs52Q64wsaxY
# 92djqJ21ZzZtQNBrZBvOY1JnIW2ESmvBDYaaBoZsYq5hVWpSP9i3bUcPQ8F4Mjkx
# qXxJzDCCB3EwggVZoAMCAQICEzMAAAAVxedrngKbSZkAAAAAABUwDQYJKoZIhvcN
# AQELBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
# VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAw
# BgNVBAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eSAyMDEw
# MB4XDTIxMDkzMDE4MjIyNVoXDTMwMDkzMDE4MzIyNVowfDELMAkGA1UEBhMCVVMx
# EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoT
# FU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgUENBIDIwMTAwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDk
# 4aZM57RyIQt5osvXJHm9DtWC0/3unAcH0qlsTnXIyjVX9gF/bErg4r25PhdgM/9c
# T8dm95VTcVrifkpa/rg2Z4VGIwy1jRPPdzLAEBjoYH1qUoNEt6aORmsHFPPFdvWG
# UNzBRMhxXFExN6AKOG6N7dcP2CZTfDlhAnrEqv1yaa8dq6z2Nr41JmTamDu6Gnsz
# rYBbfowQHJ1S/rboYiXcag/PXfT+jlPP1uyFVk3v3byNpOORj7I5LFGc6XBpDco2
# LXCOMcg1KL3jtIckw+DJj361VI/c+gVVmG1oO5pGve2krnopN6zL64NF50ZuyjLV
# wIYwXE8s4mKyzbnijYjklqwBSru+cakXW2dg3viSkR4dPf0gz3N9QZpGdc3EXzTd
# EonW/aUgfX782Z5F37ZyL9t9X4C626p+Nuw2TPYrbqgSUei/BQOj0XOmTTd0lBw0
# gg/wEPK3Rxjtp+iZfD9M269ewvPV2HM9Q07BMzlMjgK8QmguEOqEUUbi0b1qGFph
# AXPKZ6Je1yh2AuIzGHLXpyDwwvoSCtdjbwzJNmSLW6CmgyFdXzB0kZSU2LlQ+QuJ
# YfM2BjUYhEfb3BvR/bLUHMVr9lxSUV0S2yW6r1AFemzFER1y7435UsSFF5PAPBXb
# GjfHCBUYP3irRbb1Hode2o+eFnJpxq57t7c+auIurQIDAQABo4IB3TCCAdkwEgYJ
# KwYBBAGCNxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQUKqdS/mTEmr6CkTxGNSnP
# EP8vBO4wHQYDVR0OBBYEFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMFwGA1UdIARVMFMw
# UQYMKwYBBAGCN0yDfQEBMEEwPwYIKwYBBQUHAgEWM2h0dHA6Ly93d3cubWljcm9z
# b2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5Lmh0bTATBgNVHSUEDDAKBggr
# BgEFBQcDCDAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYw
# DwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvXzpoY
# xDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtp
# L2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYIKwYB
# BQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20v
# cGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDANBgkqhkiG9w0B
# AQsFAAOCAgEAnVV9/Cqt4SwfZwExJFvhnnJL/Klv6lwUtj5OR2R4sQaTlz0xM7U5
# 18JxNj/aZGx80HU5bbsPMeTCj/ts0aGUGCLu6WZnOlNN3Zi6th542DYunKmCVgAD
# sAW+iehp4LoJ7nvfam++Kctu2D9IdQHZGN5tggz1bSNU5HhTdSRXud2f8449xvNo
# 32X2pFaq95W2KFUn0CS9QKC/GbYSEhFdPSfgQJY4rPf5KYnDvBewVIVCs/wMnosZ
# iefwC2qBwoEZQhlSdYo2wh3DYXMuLGt7bj8sCXgU6ZGyqVvfSaN0DLzskYDSPeZK
# PmY7T7uG+jIa2Zb0j/aRAfbOxnT99kxybxCrdTDFNLB62FD+CljdQDzHVG2dY3RI
# LLFORy3BFARxv2T5JL5zbcqOCb2zAVdJVGTZc9d/HltEAY5aGZFrDZ+kKNxnGSgk
# ujhLmm77IVRrakURR6nxt67I6IleT53S0Ex2tVdUCbFpAUR+fKFhbHP+CrvsQWY9
# af3LwUFJfn6Tvsv4O+S3Fb+0zj6lMVGEvL8CwYKiexcdFYmNcP7ntdAoGokLjzba
# ukz5m/8K6TT4JDVnK+ANuOaMmdbhIurwJ0I9JZTmdHRbatGePu1+oDEzfbzL6Xu/
# OHBE0ZDxyKs6ijoIYn/ZcGNTTY3ugm2lBRDBcQZqELQdVTNYs6FwZvKhggLPMIIC
# OAIBATCB/KGB1KSB0TCBzjELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
# b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEpMCcGA1UECxMgTWljcm9zb2Z0IE9wZXJhdGlvbnMgUHVlcnRvIFJpY28x
# JjAkBgNVBAsTHVRoYWxlcyBUU1MgRVNOOkY4N0EtRTM3NC1EN0I5MSUwIwYDVQQD
# ExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQC8
# mrCT/GfJyBXkZ3LlvgjAT9Na46CBgzCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1w
# IFBDQSAyMDEwMA0GCSqGSIb3DQEBBQUAAgUA52KIfTAiGA8yMDIzMDEwNjE1NDYw
# NVoYDzIwMjMwMTA3MTU0NjA1WjB0MDoGCisGAQQBhFkKBAExLDAqMAoCBQDnYoh9
# AgEAMAcCAQACAhd2MAcCAQACAhEVMAoCBQDnY9n9AgEAMDYGCisGAQQBhFkKBAIx
# KDAmMAwGCisGAQQBhFkKAwKgCjAIAgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZI
# hvcNAQEFBQADgYEAHlx+YATlPrrNWdzkotuYfRgjVAxXMO+PvwNrQUQLTiMgRKNK
# KanYxFnos0XqdG3sApJNpQItYQ5sKUldK6hHtIA1YKT4VX3ksbXg8ARAZ1lcpgP4
# TYFx8f+sXq/89DKHmLMdliFm90Iy1PfUlspDJXLj8OAvG5L18kkLR7BA4hwxggQN
# MIIECQIBATCBkzB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
# MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
# MSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAa6q
# C1yzNKWVGgABAAABrjANBglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkDMQ0G
# CyqGSIb3DQEJEAEEMC8GCSqGSIb3DQEJBDEiBCCXQ6xMvMSnger2QUbISm7Rsnz1
# vSVxk9Dju01xtevMcTCB+gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EIEkoHT64
# jMNaoe6fT2apNTy46Dq17DTK7W5DSJT8Um9oMIGYMIGApH4wfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAGuqgtcszSllRoAAQAAAa4wIgQgaRU1geSU
# t6i6HU6ShSDMYp7HFlJHfq4ECPhHOYwfM6wwDQYJKoZIhvcNAQELBQAEggIAXCi9
# 8JOjix26G1s90A7pK3IGSD5gB+N1Wx72F82Dm064LVT6ZAXE92s6h/HcbSeJZ8Uc
# v89QUevGpID7w26KbD7tQoMpf7dpGJqzTF6bo2eRwzxButGoq8gzoF2dUsFH9/zY
# P5BLx/TjSXNe6mjwFvwMRKR9X+vcBdpS11Enr9aNHqSAl91rlsVC/Khw55lvHqT0
# yfD7q8OKZr+JJ73d2dk3FBeMXnIhwD5YU5A8Xq0IimcjrAzClFCthGXccHZ1FIbN
# rWmNNsHMyru+z0paS4tCInUJmzYBIJe4EFMO8VNDsx5JOMj2lr3OVzffb7KTEfvb
# nqR8/J5/HiZBVsfIkQuSAfFvP+oVG2sl+h6x9QbJSv5F6d8fm5x8WDS7LPG5TTyF
# cVKczGhIUG26Ndb32PMIDPJfRjf3e3iNrQ5+leq48hjh+6gLXsaBuZ1uNjUoJmg5
# LTbAqrihRZ7Dy/Lagz5TBimObWYrwAssnBGE9lWwNyZlEtjjPNhkSa27y3x0rR/9
# re3UOWYgn0PgAE3wfxTxXkQfqtuEN0x91XLs0/bGIh53CKd7TYxaj2maePs9YiSP
# 8xLujmGcE1ecJlIc1sTqPYrbPAZi6t1MKqn79TmgpBvUekM/p9QXFnqkx4FXx+pc
# 1zNr3X6jo4WCAGc+xKnq94K54YgfvBwV8kP3wwk=
# SIG # End signature block
