﻿function Add-Cert($pfx, $credentialXml)
{
	Write-Output "Start processing certificate '$pfx.Name'." 
    $certInstance = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2

	# Get the password for certificate.
	$pathNode = $credentialXml.Credentials.Certificates.Certificate | where { $_.Path -eq "Deployment\Onebox\"+$pfx.Name}
	if($pathNode.Count -gt 0)
	{
		$password = $pathNode[0].Password
	}
	else
	{
		$password = $pathNode.Password
	}
 
	if($password -ne $null -and [string]::IsNullOrEmpty($password) -eq $false )
	{
		$certInstance.import($pfx.FullName, $password, "Exportable,PersistKeySet,MachineKeySet"); 
		$certTP = $certInstance.Thumbprint;
        Add-CertToStore -certTP:$certTP -certInstance:$certInstance -pfx:$pfx
	}
}

function Add-CertToStore([string]$certTP,$certInstance,$pfx)
{
    $store = Get-Item Cert:\LocalMachine\My
    $store.Open("ReadWrite")

    $trustedStore = Get-Item Cert:\LocalMachine\Root
    $trustedStore.Open("ReadWrite")
    
    # add certificate to trusted root store if not present
	$trustedRootCert = Get-ChildItem "Cert:\LocalMachine\Root" | where {$_.Thumbprint -eq "$certTP"}
	if($trustedRootCert -eq $null -or [string]::IsNullOrEmpty($trustedRootCert) -eq $true)
	{
		Write-output "Adding certificate $pfx.Name to the machine trusted root store."
		$trustedStore.Add($certInstance)
	}

    # add certificate to local machine store if not present
	$localStoreCert = Get-ChildItem "Cert:\LocalMachine\My" |where {$_.Thumbprint -eq "$certTP"}
	if ($localStoreCert -eq $null -or $localStoreCert -eq "") 
	{
		Write-output "Adding certificate $pfx.Name to the local machine store."
		$store.Add($certInstance)

        # Grant permissin to Network Service to the private key.
		$sslCertPrivKey = $certInstance.PrivateKey.CspKeyContainerInfo.UniqueKeyContainerName
		$privateKeyCertFile = Get-Item -path "$ENV:ProgramData\Microsoft\Crypto\RSA\MachineKeys\*"  | where {$_.Name -eq $sslCertPrivKey} 
		$privateKeyAcl = (Get-Item -Path $privateKeyCertFile.FullName).GetAccessControl("Access") 
		$permission = "NT AUTHORITY\NETWORK SERVICE","Read","Allow" 
		$accessRule = new-object System.Security.AccessControl.FileSystemAccessRule $permission 
		$privateKeyAcl.AddAccessRule($accessRule) 
		Set-Acl $privateKeyCertFile.FullName $privateKeyAcl
		Write-output "Network Service account is granted access to the private key of certificate $pfx.Name"

		# Configure the SSL certificate "star1.cloud.onebox.dynamics.com.pfx" to the port 443
		try
		{
			if($pfx.Name -eq "star1.cloud.onebox.dynamics.com.pfx")
			{
				get-item cert:\LocalMachine\My\$certTP | new-item IIS:\SslBindings\0.0.0.0!443 -ErrorAction SilentlyContinue
			}
		}
		catch 
		{
			Write-output "SSL is configured for the AOS website."
			$_.Exception.Message    
		}
    }

    $store.Close()
    $trustedStore.Close()
}

function Remove-CertFromStore($certInstance)
{
    $store = Get-Item Cert:\LocalMachine\My
    $store.Open("ReadWrite")

    $trustedStore = Get-Item Cert:\LocalMachine\Root
    $trustedStore.Open("ReadWrite")

    # remove certificate from the trusted root store if present
	$trustedRootCert = Get-ChildItem "Cert:\LocalMachine\Root" | where {$_.Thumbprint -eq "$certTP"}
	if($trustedRootCert -ne $null -or [string]::IsNullOrEmpty($trustedRootCert) -eq $false)
	{
		Write-output "Removing certificate $pfx.Name from the machine trusted root store."
		$trustedStore.Remove($certInstance)
	}

    # remove certificate from the local machine store if present
	$localStoreCert = Get-ChildItem "Cert:\LocalMachine\My" |where {$_.Thumbprint -eq "$certTP"}
	if ($localStoreCert -ne $null -or [string]::IsNullOrEmpty($localStoreCert) -eq $false) 
	{
		Write-output "Removing certificate $pfx.Name from the local machine store."
		$store.Remove($certInstance)
    }

    $store.Close()
    $trustedStore.Close()
}

function Remove-Cert($pfx, $credentialXml)
{
    $certInstance = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2

    # Get the password for certificate.
    $pathNode = $credentialXml.Credentials.Certificates.Certificate | where { $_.Path -eq "Deployment\Onebox\"+$pfx.Name}
    
    if($pathNode.Count -gt 0)
    {
        $password = $pathNode[0].Password
    }
    else
    {
        $password = $pathNode.Password
    }
 
    if($password -ne $null -and [string]::IsNullOrEmpty($password) -eq $false )
    {
       $certInstance.import($pfx.FullName, $password, "Exportable,PersistKeySet,MachineKeySet"); 
       $certTP = $certInstance.Thumbprint;
       
       Remove-CertFromStore -certInstance:$certInstance
    }
}

function AddOrRemoveWebsiteCerts([string] $rootPath, [bool] $remove)
{
	# Add all certificates to store.
	$credentialXml = [xml](Get-Content "$rootPath\DObind\Deployment\Onebox\Credentials.xml")

	# Get all the available certificate files.
	$allPFX = Get-ChildItem "$rootPath\DObind\Deployment\Onebox\*.pfx"

	# Loop through all each certificate and install it if not installed.
	foreach($pfx in $allPFX)
	{
		if($remove)
		{
			Remove-Cert $pfx $credentialXml
		}
		else
		{
			Add-Cert $pfx $credentialXml
		}
	}
}

function Add-WebsiteCerts([string] $rootPath)
{
	AddOrRemoveWebsiteCerts $rootPath $false
}

function Remove-WebsiteCerts([string] $rootPath)
{
	AddOrRemoveWebsiteCerts $rootPath $true
}

function Remove-ExistingAppPools([string] $appPoolName)
{
	$machineName = $env:COMPUTERNAME;
    $serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)
    $existingAppPool = $serverManager.ApplicationPools | Where-Object{$_.Name -eq "$appPoolName"}
    if($existingAppPool -ne $null)
    {
        Write-Output "Removing app pool '$appPoolName'."
        $serverManager.ApplicationPools.Remove($existingAppPool)
		$serverManager.CommitChanges()
    }
}

function Remove-ExistingWebsites([string] $siteName, $port, $hostHeaderStrs)
{
	$machineName = $env:COMPUTERNAME;
	$serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)
    $existingSite = $serverManager.Sites | Where-Object{$_.Name -eq "$siteName"}
    if($existingSite -ne $null)
    {
        Write-Output "Removing website '$siteName'."
        $serverManager.Sites.Remove($existingSite)
		$serverManager.CommitChanges()
    }
 
    #remove anything already using that port
    [Microsoft.Web.Administration.Site]$site
    foreach($site in $serverManager.Sites) 
    {
        foreach($hostHeader in $hostHeaderStrs)
		{
            foreach($binding in $site.Bindings)
            {
                if($binding.EndPoint.Port -eq $port -and $binding.EndPoint.Address -eq $hostHeader)
                {
                    Write-Warning "Warning: Found an existing site '$($site.Name)' already using port $port; removing it."
				    $serverManager.Sites.Remove($site)
					$serverManager.CommitChanges()
				    Write-Output "Website $($site.Name) removed."
                }
            }
        }
    }
}

function New-AppPool([string] $appPoolName, [string] $user, [string] $password, [int] $pingTimeoutSeconds)
{
	$machineName = $env:COMPUTERNAME;
	$serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)

    Write-output "Creating an app pool named '$appPoolName' under v4.0 runtime, default (Integrated) pipeline."
    $pool = $serverManager.ApplicationPools.Add($appPoolName);
    $pool.managedRuntimeVersion = "v4.0"
    $pool.processModel.identityType = [Microsoft.Web.Administration.ProcessModelIdentityType]::NetworkService
	$pool.processModel.PingResponseTime = [System.TimeSpan]::FromSeconds($pingTimeoutSeconds)
	$pool.Failure.RapidFailProtection =$false
	
	if ($user -ne $null -AND $password -ne $null -AND $user -ne "" -AND $password -ne "") {
	    Write-Output "Setting AppPool to run as user '$user'."
		$pool.processmodel.identityType = [Microsoft.Web.Administration.ProcessModelIdentityType]::SpecificUser
		$pool.processmodel.username = $user
		$pool.processmodel.password = $password
	} 
	
    $serverManager.CommitChanges()

	for($tryCount= 30; $tryCount -gt 0; --$tryCount)
	{
		if ($serverManager.ApplicationPools[$appPoolName].State -eq [Microsoft.Web.Administration.ObjectState]::Started)
		{
			Write-Output "The AppPool was created and started successfully."
			return
		}

		Start-Sleep -Milliseconds 100
	}

	throw "App pool '$appPoolName' was created but did not start automatically."
}

function New-Website([string] $siteName, [string] $aosSitePath, [string] $port, $hostHeaderStrs,$cert, [string] $appPoolName)
{
	$machineName = $env:COMPUTERNAME
	$serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)
	$bindingArguments = "*:443:"+$hostHeaderStrs[0]
	Write-Output "Creating website. Site name: '$siteName'; Bindings: '$bindingArguments'; Website root: '$aosSitePath'."
	$website = $serverManager.Sites.Add($siteName, "https", $bindingArguments, $aosSitePath)
	for($i = 1; $i -lt $hostHeaderStrs.Length ; $i++)
	{
		$bindingArguments = "*:443:"+$hostHeaderStrs[$i]
		Write-Output "Adding binding '$bindingArguments' to the AOS website."
        $website.Bindings.Add($bindingArguments, $cert.GetCertHash(), "My")
	}

	Write-Output "Setting the website app pool to '$appPoolName'."
	# Set the app pool for the website
	$website.Applications[0].ApplicationPoolName = $appPoolName
    $serverManager.CommitChanges()

    
    Set-WebConfigurationProperty -pspath "MACHINE/WEBROOT/APPHOST/$siteName/Apps"  -filter "system.webServer/security/requestFiltering/fileExtensions/add[@fileExtension='.config']" -name "allowed" -value "True"


	for($tryCount= 100; $tryCount -gt 0; --$tryCount)
	{
		if ($serverManager.Sites[$siteName].State -eq [Microsoft.Web.Administration.ObjectState]::Started)
		{
			Write-Output "AOS website was created and started successfully."
			return
		}

		Start-Sleep -Milliseconds 100
	}
	
	throw "The AOS website '$siteName' was created but did not start after waiting for 10 seconds. Check the event log for further details."
}

function New-AosWebsite([string] $rootPath, [string] $siteName, [string] $aosSitePath, [string] $appPoolName, [string] $port, $hostHeaderStrs, [string] $user, [string] $userPassword, [string] $certTP, [int] $pingTimeoutSeconds = 600)
{
	# start IIS.
	Invoke-Command -script {iisreset.exe /start}

    # Cleanup old websites and app pools
	Remove-ExistingWebsites $siteName $appPoolName $port
	Remove-ExistingAppPools $appPoolName

	# Add certs
	Add-WebsiteCerts $rootPath

	# Create new app Pool
	New-AppPool $appPoolName $user $userPassword $pingTimeoutSeconds
	
	# Create the new site
	$cert = Get-ChildItem "Cert:\LocalMachine\My" |where {$_.Thumbprint -eq "$certTP"}
	New-Website $siteName $aosSitePath $port $hostHeaderStrs $cert $appPoolName
}

function Remove-Website([string] $rootPath, [string] $websiteName, [string] $appPoolName)
{
	# Cleanup certs
	Write-Output "Removing certificates."
	Remove-WebsiteCerts $rootPath

	# Cleanup old websites and app pools
	Write-Output "Remvoing website '$websiteName'."
	Remove-ExistingWebsites $websiteName $appPoolName 0

	Write-Output "Remvoing App pool '$appPoolName'."
	Remove-ExistingAppPools $appPoolName
}

function Delete-Service([string]$serviceName)
{
    try
    {
        $service = Get-WmiObject -Class Win32_Service -Filter "Name='$serviceName'" -ErrorAction SilentlyContinue
	    if($service -ne $null)
	    {
		    # stop and delete the batch Service
            Write-output "Stopping service '$serviceName'."
		    Stop-Service "$serviceName" -ErrorAction SilentlyContinue -Force
            $service = Get-Service $serviceName
            while($service -ne $null -and $service.Status -ne "Stopped")
            {
                $service = Get-Service $serviceName
                Write-output "Sleeping for 5 minutes while waiting for the service '$serviceName' to stop."
                Start-Sleep 5
            }

            Write-output "Deleting service '$serviceName'."
		    sc.exe delete "$serviceName" 
	    }
    }
    catch
    {
    }
}

function Terminate-process
{
    # kill process
	$processes = @("batch","batch.exe","mmc")
	foreach($process in $processes)
	{
		$handle = get-process "$process" -ErrorAction SilentlyContinue
		if($handle -ne $null)
		{
			try
			{
                # special case for Event viewer
                if($process -eq "mmc" -and $handle.MainWindowTitle -eq "Event Viewer")
                {
                    $procName = $handle.MainWindowTitle
                    Write-output "Terminating process '$procName'."
				    $handle.kill()
                }
                else
                {
                    Write-output "Terminating process '$handle'."
                    $handle.Kill()
                }
			}
			catch
			{
			}
		}
	}

    # stop and delete Dynamics Rainier services
    Delete-Service -serviceName:"DynamicsAxBatch"
}

function Register-RestartTask([string]$deploymentDir,[string]$logDir,[bool]$deployChocolateyPackage,[string]$packageDir)
{
    Write-Output "Registering a scheduled task to auto start Azure storage emulator upon machine restart."
    $localMachineName = $env:COMPUTERNAME
    $batchFilename = [IO.Path]::Combine("$deploymentDir","AOSDeploy.cmd")
    Write-Output "Creating scheduler task batch file '$batchFilename'."

    "$deploymentDir\DoBind\aosdeploy.exe /action:Restart /rd:`"$deploymentDir`" /ld:`"$logDir`" /td:`"$packageDir`" /chocopkg:$deployChocolateyPackage"|Set-Content $batchFilename

    $taskName = "DynamicsRainierRestartTask"
    & schtasks /create /ru SYSTEM /sc ONSTART /tn $taskName /tr "cmd /c $batchFilename" /F /RL HIGHEST
    if (!$?)
    {
        throw "There was an error creating the task $taskName."
    }
}

function Delete-RestartTask
{
    $taskName = "DynamicsRainierRestartTask"
    Write-Output "Deleting scheduled task '$taskName'."
    $SchTsk = Get-ScheduledTask | Select Name, State | ? {$_.Name -eq $taskName}
    if($SchTsk -ne $null)
    {
        & schtasks /delete /tn $taskName /F
    }
}

function Configure-ProductConfiguration([string]$sitename)
{
    Write-output "Starting the configuration of the production configuration application."
    $aossite=Get-Website -Name $sitename
    if($aossite -eq $null){
        $message="The website '$sitename' does not exist."
        Write-Output $message
        throw $message
    }

    $webroot=$aossite.physicalPath
    $productconfigurationdir=join-path $webRoot "productconfiguration"

    if(!(Test-Path "$productconfigurationdir")){
        Write-Output "Skipping product configuration as the product configuration directory does not exist under the aos web root $webroot."
        return
    }

    $pcapppool="productconfiguration"
    Write-output "Removing web app pool $pcapppool."
    Remove-WebAppPool -Name:$pcapppool -ErrorAction SilentlyContinue
    write-output "Creating web app pool $pcapppool."
    $newapppool=New-WebAppPool -Name:$pcapppool -ErrorAction SilentlyContinue
    write-output "Setting the identity for the web app pool '$pcapppool' to NetworkService."
    $newapppool.processModel.identityType="NetworkService"
    $newapppool|Set-Item
    write-output "Starting web app pool $pcapppool."
    Start-WebAppPool -Name:$pcapppool

    Write-Output "Deleting web application 'ProductConfiguration' under web site '$sitename'."
    Remove-WebApplication -Site:$aossite.Name -Name:"ProductConfiguration" -ErrorAction SilentlyContinue
    Write-Output "Creating new web application 'ProductConfiguration' under web site '$sitename' with application pool `"$pcapppool`"."
    New-WebApplication -Site:$aossite.Name -Name:"ProductConfiguration" -PhysicalPath:$productconfigurationdir -ApplicationPool:$pcapppool -Force 
    Write-Output "Product configuration application successfully configured."
}

Export-ModuleMember -Function New-AosWebsite,Remove-Website,Terminate-process,Register-RestartTask,Delete-RestartTask,Delete-Service,Configure-ProductConfiguration,Add-WebsiteCerts,Remove-WebsiteCerts
# SIG # Begin signature block
# MIInkwYJKoZIhvcNAQcCoIInhDCCJ4ACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCD4vYul+/4MT2Y/
# shNR1m8EcMqg8PK/RZCg1SpB/rUJ2qCCDXYwggX0MIID3KADAgECAhMzAAACy7d1
# OfsCcUI2AAAAAALLMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjIwNTEyMjA0NTU5WhcNMjMwNTExMjA0NTU5WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQC3sN0WcdGpGXPZIb5iNfFB0xZ8rnJvYnxD6Uf2BHXglpbTEfoe+mO//oLWkRxA
# wppditsSVOD0oglKbtnh9Wp2DARLcxbGaW4YanOWSB1LyLRpHnnQ5POlh2U5trg4
# 3gQjvlNZlQB3lL+zrPtbNvMA7E0Wkmo+Z6YFnsf7aek+KGzaGboAeFO4uKZjQXY5
# RmMzE70Bwaz7hvA05jDURdRKH0i/1yK96TDuP7JyRFLOvA3UXNWz00R9w7ppMDcN
# lXtrmbPigv3xE9FfpfmJRtiOZQKd73K72Wujmj6/Su3+DBTpOq7NgdntW2lJfX3X
# a6oe4F9Pk9xRhkwHsk7Ju9E/AgMBAAGjggFzMIIBbzAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUrg/nt/gj+BBLd1jZWYhok7v5/w4w
# RQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEW
# MBQGA1UEBRMNMjMwMDEyKzQ3MDUyODAfBgNVHSMEGDAWgBRIbmTlUAXTgqoXNzci
# tW2oynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NybC9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3JsMGEG
# CCsGAQUFBwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDovL3d3dy5taWNyb3NvZnQu
# Y29tL3BraW9wcy9jZXJ0cy9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3J0
# MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQELBQADggIBAJL5t6pVjIRlQ8j4dAFJ
# ZnMke3rRHeQDOPFxswM47HRvgQa2E1jea2aYiMk1WmdqWnYw1bal4IzRlSVf4czf
# zx2vjOIOiaGllW2ByHkfKApngOzJmAQ8F15xSHPRvNMmvpC3PFLvKMf3y5SyPJxh
# 922TTq0q5epJv1SgZDWlUlHL/Ex1nX8kzBRhHvc6D6F5la+oAO4A3o/ZC05OOgm4
# EJxZP9MqUi5iid2dw4Jg/HvtDpCcLj1GLIhCDaebKegajCJlMhhxnDXrGFLJfX8j
# 7k7LUvrZDsQniJZ3D66K+3SZTLhvwK7dMGVFuUUJUfDifrlCTjKG9mxsPDllfyck
# 4zGnRZv8Jw9RgE1zAghnU14L0vVUNOzi/4bE7wIsiRyIcCcVoXRneBA3n/frLXvd
# jDsbb2lpGu78+s1zbO5N0bhHWq4j5WMutrspBxEhqG2PSBjC5Ypi+jhtfu3+x76N
# mBvsyKuxx9+Hm/ALnlzKxr4KyMR3/z4IRMzA1QyppNk65Ui+jB14g+w4vole33M1
# pVqVckrmSebUkmjnCshCiH12IFgHZF7gRwE4YZrJ7QjxZeoZqHaKsQLRMp653beB
# fHfeva9zJPhBSdVcCW7x9q0c2HVPLJHX9YCUU714I+qtLpDGrdbZxD9mikPqL/To
# /1lDZ0ch8FtePhME7houuoPcMIIHejCCBWKgAwIBAgIKYQ6Q0gAAAAAAAzANBgkq
# hkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
# EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
# IDIwMTEwHhcNMTEwNzA4MjA1OTA5WhcNMjYwNzA4MjEwOTA5WjB+MQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQg
# Q29kZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
# CgKCAgEAq/D6chAcLq3YbqqCEE00uvK2WCGfQhsqa+laUKq4BjgaBEm6f8MMHt03
# a8YS2AvwOMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc6Whe0t+bU7IKLMOv2akr
# rnoJr9eWWcpgGgXpZnboMlImEi/nqwhQz7NEt13YxC4Ddato88tt8zpcoRb0Rrrg
# OGSsbmQ1eKagYw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+lD3v++MrWhAfTVYoonpy
# 4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+EGvKhL1nkkDstrjNYxbc+/jLTswM9
# sbKvkjh+0p2ALPVOVpEhNSXDOW5kf1O6nA+tGSOEy/S6A4aN91/w0FK/jJSHvMAh
# dCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rhKEmdX4jiJV3TIUs+UsS1Vz8k
# A/DRelsv1SPjcF0PUUZ3s/gA4bysAoJf28AVs70b1FVL5zmhD+kjSbwYuER8ReTB
# w3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zdsGbiwZeBe+3W7UvnSSmn
# Eyimp31ngOaKYnhfsi+E11ecXL93KCjx7W3DKI8sj0A3T8HhhUSJxAlMxdSlQy90
# lfdu+HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS4NaIjAsCAwEAAaOCAe0w
# ggHpMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBRIbmTlUAXTgqoXNzcitW2o
# ynUClTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYD
# VR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBRyLToCMZBDuRQFTuHqp8cx0SOJNDBa
# BgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
# bC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3JsMF4GCCsG
# AQUFBwEBBFIwUDBOBggrBgEFBQcwAoZCaHR0cDovL3d3dy5taWNyb3NvZnQuY29t
# L3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3J0MIGfBgNV
# HSAEgZcwgZQwgZEGCSsGAQQBgjcuAzCBgzA/BggrBgEFBQcCARYzaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1hcnljcHMuaHRtMEAGCCsG
# AQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAHAAbwBsAGkAYwB5AF8AcwB0AGEAdABl
# AG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn8oalmOBUeRou09h0ZyKb
# C5YR4WOSmUKWfdJ5DJDBZV8uLD74w3LRbYP+vj/oCso7v0epo/Np22O/IjWll11l
# hJB9i0ZQVdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0bpdS1HXeUOeLpZMlEPXh6
# I/MTfaaQdION9MsmAkYqwooQu6SpBQyb7Wj6aC6VoCo/KmtYSWMfCWluWpiW5IP0
# wI/zRive/DvQvTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvyCInWH8MyGOLwxS3OW560
# STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn+N4sOiBpmLJZiWhub6e3dMNABQam
# ASooPoI/E01mC8CzTfXhj38cbxV9Rad25UAqZaPDXVJihsMdYzaXht/a8/jyFqGa
# J+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw3MYbBL7fQccOKO7eZS/sl/ah
# XJbYANahRr1Z85elCUtIEJmAH9AAKcWxm6U/RXceNcbSoqKfenoi+kiVH6v7RyOA
# 9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sLgOppO6/8MO0ETI7f33Vt
# Y5E90Z1WTk+/gFcioXgRMiF670EKsT/7qMykXcGhiJtXcVZOSEXAQsmbdlsKgEhr
# /Xmfwb1tbWrJUnMTDXpQzTGCGXMwghlvAgEBMIGVMH4xCzAJBgNVBAYTAlVTMRMw
# EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVN
# aWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNp
# Z25pbmcgUENBIDIwMTECEzMAAALLt3U5+wJxQjYAAAAAAsswDQYJYIZIAWUDBAIB
# BQCgga4wGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEINtB1m9PHGgEpV2EU8rkJQfh
# GL66oYCwLnP9gtUKVx9hMEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8A
# cwBvAGYAdKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEB
# BQAEggEAkKytk5qEMEHTYkFgtrPHM5YWY62Fv3pUIpETMEDqkwwevYU007KZ950E
# 9MjZV0B7PqpEPcWw/76QDFBaXsfvj8O47+Q3ldZ9p9Lx0rom02RrWyru1t/WKCeQ
# mgJXYrXl+QLne9XEUJqFQEOQy/+QEZuXfGUc6of81DnEC2GMAkYql3XDCPIej87n
# BKq3qkDxuLas9LbJeQzjF8LzaH1j4+vTPO+YgyIL1sBe5N98mr2sZHRC9SBHRLXc
# aaLcvYrBRG9GxasmpSMng7CvWXUh06jVusygOEkR5/CUkzni/5lyEp8qP+WwYIB4
# 4j/pFxz1o474wRHaQlxq5B+vmm79WaGCFv0wghb5BgorBgEEAYI3AwMBMYIW6TCC
# FuUGCSqGSIb3DQEHAqCCFtYwghbSAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFRBgsq
# hkiG9w0BCRABBKCCAUAEggE8MIIBOAIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFl
# AwQCAQUABCBvbIItk+wOn3cpBH99qws5nb1UJujPSMH4z9fPcgXwxQIGZBNFrGzO
# GBMyMDIzMDMyNTA2Mzk0MS4yODRaMASAAgH0oIHQpIHNMIHKMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1l
# cmljYSBPcGVyYXRpb25zMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpFNUE2LUUy
# N0MtNTkyRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZaCC
# EVQwggcMMIIE9KADAgECAhMzAAABvvQgou6W1iDWAAEAAAG+MA0GCSqGSIb3DQEB
# CwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
# EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNV
# BAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMB4XDTIyMTEwNDE5MDEy
# MloXDTI0MDIwMjE5MDEyMlowgcoxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJTAjBgNVBAsTHE1pY3Jvc29mdCBBbWVyaWNhIE9wZXJhdGlvbnMx
# JjAkBgNVBAsTHVRoYWxlcyBUU1MgRVNOOkU1QTYtRTI3Qy01OTJFMSUwIwYDVQQD
# ExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNlMIICIjANBgkqhkiG9w0BAQEF
# AAOCAg8AMIICCgKCAgEApV/y2z7Da7nMu0tykLY8olh7Z03EqFNz3iFlMp9gOfVm
# ZABmheCc87RuPdQ2P+OHUJiqCQAWNuNSoI/Q1ixEw9AA657ldD8Z3/EktpmxKHHa
# vOhwQSFPcpTGFVXIxKCwoO824giyHPG84dfhdi6WU7f7D+85LaPB0dOsPHKKMGlC
# 9p66Lv9yQzvAhZGFmFhlusCy/egrz6JX/OHOT9qCwughrL0IPf47ULe1pQSEEihy
# 438JwS+rZU4AVyvQczlMC26XsTuDPgEQKlzx9ru7EvNV99l/KCU9bbFf5SnkN1mo
# UgKUq09NWlKxzLGEvhke2QNMopn86Jh1fl/PVevN/xrZSpV23rM4lB7lh7XSsCPe
# FslTYojKN2ioOC6p3By7kEmvZCh6rAsPKsARJISdzKQCMq+mqDuiP6mr/LvuWKin
# P+2ZGmK/C1/skvlTjtIehu50yoXNDlh1CN9B3QLglQY+UCBEqJog/BxAn3pWdR01
# o/66XIacgXI/d0wG2/x0OtbjEGAkacfQlmw0bDc02dhQFki/1Q9Vbwh4kC7VgAiJ
# A8bC5zEIYWHNU7C+He69B4/2dZpRjgd5pEpHbF9OYiAf7s5MnYEnHN/5o/bGO0aj
# Ab7VI4f9av62sC6xvhKTB5R4lhxEMWF0z4v7BQ5CHyMNkL+oTnzJLqnLVdXnuM0C
# AwEAAaOCATYwggEyMB0GA1UdDgQWBBTrKiAWoYRBoPGtbwvbhhX6a2+iqjAfBgNV
# HSMEGDAWgBSfpxVdAF5iXYP05dJlpxtTNRnpcjBfBgNVHR8EWDBWMFSgUqBQhk5o
# dHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9NaWNyb3NvZnQlMjBU
# aW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcmwwbAYIKwYBBQUHAQEEYDBeMFwG
# CCsGAQUFBzAChlBodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRz
# L01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0ElMjAyMDEwKDEpLmNydDAMBgNV
# HRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IC
# AQDHlfu9c0ImhdBis1yj56bBvOSyGpC/rSSty+1F49Tf6fmFEeqxhwTTHVHOeIRN
# d8gcDLSz0d79mXCqq8ynq6gJgy2u4LyyAr2LmwzFVuuxoGVR8YuUnRtvsDH5J+un
# ye/nMkwHiC+G82h3uQ8fcGj+2H0nKPmUpUtfQruMUXvzLjV5NyRjDiCL5c/f5ecm
# z01dnpnCvE6kIz/FTpkvOeVJk22I2akFZhPz24D6OT6KkTtwBRpSEHDYqCQ4cZ+7
# SXx7jzzd7b+0p9vDboqCy7SwWgKpGQG+wVbKrTm4hKkZDzcdAEgYqehXz78G00mY
# ILiDTyUikwQpoZ7am9pA6BdTPY+o1v6CRzcneIOnJYanHWz0R+KER/ZRFtLCyBMv
# LzSHEn0sR0+0kLklncKjGdA1YA42zOb611UeIGytZ9VhNwn4ws5GJ6n6PJmMPO+y
# PEkOy2f8OBiuhaqlipiWhzGtt5UsC0geG0sW9qwa4QAW1sQWIrhSl24MOOVwNl/A
# m9/ZqvLRWr1x4nupeR8G7+DNyn4MTg28yFZRU1ktSvyBMUSvN2K99BO6p1gSx/wv
# SsR45dG33PDG5fKqHOgDxctjBU5bX49eJqjNL7S/UndLF7S0OWL9mdk/jPVHP2I6
# XtN0K4VjdRwvIgr3jNib3GZyGJnORp/ZMbY2Dv1mKcx7dTCCB3EwggVZoAMCAQIC
# EzMAAAAVxedrngKbSZkAAAAAABUwDQYJKoZIhvcNAQELBQAwgYgxCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jvc29mdCBS
# b290IENlcnRpZmljYXRlIEF1dGhvcml0eSAyMDEwMB4XDTIxMDkzMDE4MjIyNVoX
# DTMwMDkzMDE4MzIyNVowfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
# b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwggIi
# MA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDk4aZM57RyIQt5osvXJHm9DtWC
# 0/3unAcH0qlsTnXIyjVX9gF/bErg4r25PhdgM/9cT8dm95VTcVrifkpa/rg2Z4VG
# Iwy1jRPPdzLAEBjoYH1qUoNEt6aORmsHFPPFdvWGUNzBRMhxXFExN6AKOG6N7dcP
# 2CZTfDlhAnrEqv1yaa8dq6z2Nr41JmTamDu6GnszrYBbfowQHJ1S/rboYiXcag/P
# XfT+jlPP1uyFVk3v3byNpOORj7I5LFGc6XBpDco2LXCOMcg1KL3jtIckw+DJj361
# VI/c+gVVmG1oO5pGve2krnopN6zL64NF50ZuyjLVwIYwXE8s4mKyzbnijYjklqwB
# Sru+cakXW2dg3viSkR4dPf0gz3N9QZpGdc3EXzTdEonW/aUgfX782Z5F37ZyL9t9
# X4C626p+Nuw2TPYrbqgSUei/BQOj0XOmTTd0lBw0gg/wEPK3Rxjtp+iZfD9M269e
# wvPV2HM9Q07BMzlMjgK8QmguEOqEUUbi0b1qGFphAXPKZ6Je1yh2AuIzGHLXpyDw
# wvoSCtdjbwzJNmSLW6CmgyFdXzB0kZSU2LlQ+QuJYfM2BjUYhEfb3BvR/bLUHMVr
# 9lxSUV0S2yW6r1AFemzFER1y7435UsSFF5PAPBXbGjfHCBUYP3irRbb1Hode2o+e
# FnJpxq57t7c+auIurQIDAQABo4IB3TCCAdkwEgYJKwYBBAGCNxUBBAUCAwEAATAj
# BgkrBgEEAYI3FQIEFgQUKqdS/mTEmr6CkTxGNSnPEP8vBO4wHQYDVR0OBBYEFJ+n
# FV0AXmJdg/Tl0mWnG1M1GelyMFwGA1UdIARVMFMwUQYMKwYBBAGCN0yDfQEBMEEw
# PwYIKwYBBQUHAgEWM2h0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvRG9j
# cy9SZXBvc2l0b3J5Lmh0bTATBgNVHSUEDDAKBggrBgEFBQcDCDAZBgkrBgEEAYI3
# FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAf
# BgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvXzpoYxDBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNS
# b29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0Nl
# ckF1dF8yMDEwLTA2LTIzLmNydDANBgkqhkiG9w0BAQsFAAOCAgEAnVV9/Cqt4Swf
# ZwExJFvhnnJL/Klv6lwUtj5OR2R4sQaTlz0xM7U518JxNj/aZGx80HU5bbsPMeTC
# j/ts0aGUGCLu6WZnOlNN3Zi6th542DYunKmCVgADsAW+iehp4LoJ7nvfam++Kctu
# 2D9IdQHZGN5tggz1bSNU5HhTdSRXud2f8449xvNo32X2pFaq95W2KFUn0CS9QKC/
# GbYSEhFdPSfgQJY4rPf5KYnDvBewVIVCs/wMnosZiefwC2qBwoEZQhlSdYo2wh3D
# YXMuLGt7bj8sCXgU6ZGyqVvfSaN0DLzskYDSPeZKPmY7T7uG+jIa2Zb0j/aRAfbO
# xnT99kxybxCrdTDFNLB62FD+CljdQDzHVG2dY3RILLFORy3BFARxv2T5JL5zbcqO
# Cb2zAVdJVGTZc9d/HltEAY5aGZFrDZ+kKNxnGSgkujhLmm77IVRrakURR6nxt67I
# 6IleT53S0Ex2tVdUCbFpAUR+fKFhbHP+CrvsQWY9af3LwUFJfn6Tvsv4O+S3Fb+0
# zj6lMVGEvL8CwYKiexcdFYmNcP7ntdAoGokLjzbaukz5m/8K6TT4JDVnK+ANuOaM
# mdbhIurwJ0I9JZTmdHRbatGePu1+oDEzfbzL6Xu/OHBE0ZDxyKs6ijoIYn/ZcGNT
# TY3ugm2lBRDBcQZqELQdVTNYs6FwZvKhggLLMIICNAIBATCB+KGB0KSBzTCByjEL
# MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1v
# bmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjElMCMGA1UECxMcTWlj
# cm9zb2Z0IEFtZXJpY2EgT3BlcmF0aW9uczEmMCQGA1UECxMdVGhhbGVzIFRTUyBF
# U046RTVBNi1FMjdDLTU5MkUxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1w
# IFNlcnZpY2WiIwoBATAHBgUrDgMCGgMVAGitWlL3vPu8ENOAe+i2+4wfTMB7oIGD
# MIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
# BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQG
# A1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwDQYJKoZIhvcNAQEF
# BQACBQDnyPiaMCIYDzIwMjMwMzI1MTIzNTM4WhgPMjAyMzAzMjYxMjM1MzhaMHQw
# OgYKKwYBBAGEWQoEATEsMCowCgIFAOfI+JoCAQAwBwIBAAICIAMwBwIBAAICEbEw
# CgIFAOfKShoCAQAwNgYKKwYBBAGEWQoEAjEoMCYwDAYKKwYBBAGEWQoDAqAKMAgC
# AQACAwehIKEKMAgCAQACAwGGoDANBgkqhkiG9w0BAQUFAAOBgQB8CHhabqoJ4lan
# ysrbhhFFxyt7ZWKqSnVp1rsub4Hb5Rl5X9ftiZN4Ud5hShylsNyT1NP4awDYBisL
# QgbmY+QsP3ON3RH+HzMXhM7NGbTKNXmyA5M/f75ySw1Cg7AA3vhZrZh6ABAVuT5v
# aAQEH2ND76v1Etew5+hE/+MeFFBp8jGCBA0wggQJAgEBMIGTMHwxCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBU
# aW1lLVN0YW1wIFBDQSAyMDEwAhMzAAABvvQgou6W1iDWAAEAAAG+MA0GCWCGSAFl
# AwQCAQUAoIIBSjAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwLwYJKoZIhvcN
# AQkEMSIEICgxnSI+XCZOqEzT/EV8d0wAef1phtxi1F9WzKiMOhpbMIH6BgsqhkiG
# 9w0BCRACLzGB6jCB5zCB5DCBvQQglO6Kr632/Oy9ZbXPrhEPidNAg/Fef7K3SZg+
# DxjhDC8wgZgwgYCkfjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAA
# Ab70IKLultYg1gABAAABvjAiBCB9kkrrFNEKWZXFMcL1Nv8gxUlJDbrJeuwnxYPe
# VDpWqTANBgkqhkiG9w0BAQsFAASCAgBtU2gtf7XE7eIpxPvjQJzhckuqDhDWPUZK
# m4Q0UuP3+6NNM62k3wc9ERZJ3DYqTTG//h9HbSEilZOB2F27uSPRhETqWNCVwvSC
# TUUO/2OkVEXNZ9jCVKvPz0GeNtDvlD1aJsV8NYp+dzmp+HZPY6/l2lBwjv/jMMU/
# YD82nAfAJD/S68vrTrLVtvb5QTAvcABMCRBtSNj9rkimh/gqX2zwbfC38ZiPQ5UW
# pL5qBNbeVJ8Q9zGmthqu/YhfbMBFlu7GAgPOoGLRtv6EHLfTJqLwIZv+0WDJiPZb
# V0tsA+DdFC5/z9Yddi4Sx37kESAu8KTQU7klHN4V/U5CiAWmHL+sfXpCWEtWnYL6
# 2ssbXEbqg+ukKsX3Tv11K21p7y0+zZwQylEQK2lWcYEQADadr6p1Al1hpHi449xf
# EH+IkYg5f8eVAf0fpP3Nb6Qqp7ENxNTwFq4Zv1ollP+6AYlhaJX4xx+2CBL1nfkl
# c7CwnEIb+hkwLagh58O4IyY5j77z/31Rqkr0d7MTgp3yffIMFCKg8J6bq5splWCx
# rxo5zuOpReY6S5axTc/vgZkkshOSgCC1dsaJ7NPtxVCu8odFghrKrZYMspKvAruj
# y/clRBzGDvvyu0k8XECzWTXhuDc4wkh1GO59QgE+8MpOsHv/KkFrs3rysdhVZIxB
# rmDKKi4YUw==
# SIG # End signature block
