PARAM (
    [Parameter(Mandatory=$true)]
    $log,
    [Parameter(Mandatory=$true)]
    [ValidateScript({Test-Path $_})]
    $webroot,
    [Parameter(Mandatory=$true)]
    [ValidateScript({Test-Path $_})]
    $packageDirectory
)

$ErrorActionPreference = 'Stop'

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking
Import-Module "$PSScriptRoot\AosEnvironmentUtilities.psm1" -Force -DisableNameChecking

$WebConfigPath = "$webroot\web.config"
$DbSyncBlastFolder = "$PSScriptRoot\DbSyncBlast"
$DbSyncVersionTestFile = "Microsoft.Dynamics.AX.Data.Management.Sync.dll"

Initialize-Log $log

function Replace-Files([string] $sourceFolder, [string[]] $sourceFiles, [string[]] $destinationFolders, [string] $backupFolderName, [hashtable] $backedUpFiles)
{
    $useBackup = (-not [String]::IsNullOrWhiteSpace($backupFolderName))

    foreach ($directory in $destinationFolders)
    {
        if ($useBackup)
        {
            $backupFolder = Join-Path $directory $backupFolderName

            if (-not (Test-Path $backupFolder))
            {
                Write-Log "Creating folder '$backupFolder'"
                New-Item -ItemType Directory -Path $backupFolder
            }
        }

        foreach ($file in $sourceFiles)
        {
            $sourcePath = Join-Path $sourceFolder $file
            $targetPath = Join-Path $directory $file

            if ($useBackup)
            {
                if (Test-Path $targetPath)
                {
                    $backupPath = Join-Path $backupFolder "$file.bak"

                    Write-Log "Copying file from: '$sourcePath' to '$targetPath', backing up target to '$backupPath'"

                    Copy-Item -Path $targetPath -Destination $backupPath -Force
                    $backedUpFiles[$targetPath] = $backupPath
                }
                else
                {
                    Write-Log "Copying file from: '$sourcePath' to '$targetPath'. Target file does not exist, skipping backup."

                    # Empty value means the file is new - delete/rename when restoring backup
                    $backedUpFiles[$targetPath] = ""
                }
            }
            else
            {
                Write-Log "Copying file from: '$sourcePath' to '$targetPath'. No backup folder specified, skipping backup."
            }

            Copy-Item -Path $sourcePath -Destination $targetPath -Force
                                
            Write-Log "Copy file succeeded."
        }
    }
}

function Restore-Backup([hashtable] $backedUpFiles)
{
    foreach ($file in $backedUpFiles.GetEnumerator())
    {
        try
        {
            if (-not [String]::IsNullOrWhiteSpace($file.Value))
            {
                Write-Log "Restoring backup from '$($file.Value)' to '$($file.Key)'"

                Copy-Item -Path $file.Value -Destination $file.Key -Force
                
                Write-Log "Restored successfully."
            }
            else
            {
                $renameTo = "$($file.Key).dbsyncblast_failed"

                Write-Log "Restoring backup for '$($file.Key)', which is newly created - renaming to '$renameTo' instead of deleting."

                Rename-Item -Path $file.Key -NewName $renameTo

                Write-Log "Renamed successfully."
            }
        }
        catch
        {
            Write-Exception $_
            Write-Log "Failed to restore backup for '$($file.Key)'"
        }
    }
}

function Test-RequiredPaths
{
    foreach ($path in $args)
    {
        if (-not (Test-Path $path))
        {
            Write-Log "Path '$path' not found, skipping DbSyncBlast."
            return $false
        }
    }

    return $true
}

function Get-FileVersion([string] $file)
{
    $fileInfo = Get-ChildItem -File $file

    if ($fileInfo -eq $null -or $fileInfo.VersionInfo -eq $null)
    {
        Write-Log "Could not get version info for '$fileInfo', skipping DbSyncBlast."
        return $null
    }

    return $fileInfo.VersionInfo
}

filter VersionsToDictionary
{
    begin { $dict = @{} }
    process { $dict[[int]($_.Split('.')[-1])] = $_ } # e.g. 7.0.30100.4970 --> $dict[4970] = 7.0.30100.4970
    end { return $dict }
}

$DbSyncBlastMinPlatformVersion = 5030 #PU20
$script:PlatformBuildVersion = 0
function Get-DbSyncApplicableVersionPath
{
    if (-not (Test-RequiredPaths $DbSyncBlastFolder))
    {
        return $null
    }
    
    # Default to no replace
    $dbSyncApplicableVersionPath = $null
            
    $platformVersionInfo = Get-ProductPlatformBuildFullVersion

    if ($platformVersionInfo -ne $null)
    {
        Write-Log "Current platform build: $platformVersionInfo"

        $dbSyncVersions = Get-ChildItem -Path $DbSyncBlastFolder -Directory -Name | VersionsToDictionary
        Write-Log "Found DbSyncBlast builds for platform builds >= $($dbSyncVersions.Keys -join ', ')."
        $invalidDbSyncBlastVersions = $dbSyncVersions.GetEnumerator() | ?{$_.Key -lt $DbSyncBlastMinPlatformVersion}
        if ($invalidDbSyncBlastVersions.Count -gt 0)
        {
            Write-Log "Skipping invalid DbSyncBlast builds: $($invalidDbSyncBlastVersions.Values -join ', '). Minimum supported platform build is $DbSyncBlastMinPlatformVersion."

            $dbSyncVersions = $dbSyncVersions.GetEnumerator() | ?{$_.Key -ge $DbSyncBlastMinPlatformVersion}
        }
        
        $script:PlatformBuildVersion = $platformVersionInfo.Build
        $useDbSyncPlatformVersion = [int](($dbSyncVersions.Keys | ?{$_ -le $script:PlatformBuildVersion} | measure -Max).Maximum)
        if ($useDbSyncPlatformVersion -ne 0)
        {
            Write-Log "Using DbSyncBlast build $($dbSyncVersions[$useDbSyncPlatformVersion])"
            $dbSyncApplicableVersionPath = Join-Path $DbSyncBlastFolder $dbSyncVersions[$useDbSyncPlatformVersion]
        }
        else
        {
            Write-Log "No applicable version found, skipping DbSyncBlast."
        }
    }
    
    return $dbSyncApplicableVersionPath
}

$EnsureMajorVersion = 7
$EnsureMinorVersion = 0
function Assert-IsNewerVersion([string] $newFile, [string] $oldFile)
{
    if (-not (Test-RequiredPaths $newFile $oldFile))
    {
        return $false
    }

    $newFileVersionInfo = Get-FileVersion -File $newFile
    $oldFileVersionInfo = Get-FileVersion -File $oldFile

    if ($newFileVersionInfo -eq $null -or $oldFileVersionInfo -eq $null)
    {
        return $false
    }

    Write-Log "Version of '$oldFile': $($oldFileVersionInfo.FileVersion)"
    Write-Log "Version of '$newFile': $($newFileVersionInfo.FileVersion)"

    if ($newFileVersionInfo.FileMajorPart -ne $EnsureMajorVersion -or $oldFileVersionInfo.FileMajorPart -ne $EnsureMajorVersion `
    -or $newFileVersionInfo.FileMinorPart -ne $EnsureMinorVersion -or $oldFileVersionInfo.FileMinorPart -ne $EnsureMinorVersion)
    {
        # If e.g. major version is changed from 7 to 10, we need to decide what to do. Skipping for now.
        Write-Log "Unexpected major or minor version, expected $EnsureMajorVersion.$EnsureMinorVersion.x.y. Skipping DbSyncBlast."
        return $false
    }

    $versionDiff = $newFileVersionInfo.FileBuildPart - $oldFileVersionInfo.FileBuildPart

    if ($versionDiff -lt 0)
    {
        Write-Log "Selected DbSyncBlast build version is lower than what is installed, skipping DbSyncBlast."
        return $false
    }

    if ($versionDiff -eq 0)
    {
        Write-Log "Selected DbSyncBlast build already installed, skipping DbSyncBlast."
        return $false
    }

    return $true
}

function Update-WebConfig
{
    # Update the Monitoring.ETWManifests key, adding Microsoft.Dynamics.ApplicationPlatform.DatabaseSynchronize.man if it does not exist in the list
    $key = "Monitoring.ETWManifests"
    $manFileName = "Microsoft.Dynamics.ApplicationPlatform.DatabaseSynchronize.man"

    # Check for PU20-21
    if ($script:PlatformBuildVersion -ge 5030 -and $script:PlatformBuildVersion -le 5073) {

        [System.Xml.XmlDocument] $xmlDoc = new-object System.Xml.XmlDocument
        $xmlDoc.Load($WebConfigPath)
        $node = $xmlDoc.SelectSingleNode("//add[@key='$key']")
        $oldValueSplit = $node.Value -split ";"

        if ($oldValueSplit -notcontains $manFileName)
        {
            Write-Log "Updating $key, adding $manFileName"
            $node.Value += ";$manFileName"
            $xmlDoc.Save($WebConfigPath)
        }
        else
        {
            Write-Log "Not updating $key, $manFileName already exists."
        }
    }
    else
    {
        Write-Log "Not updating $key, platform is not PU20 or PU21."
    }
}

try
{
    Write-Log "DbSyncBlast assembly replacement check: Start"

    $error.Clear()

    if (-not (Get-IsFlightEnabled "DbSyncBlastDisabled"))
    {
        $dbSyncApplicableVersionPath = Get-DbSyncApplicableVersionPath

        if ($dbSyncApplicableVersionPath -ne $null)
        {
            if (Assert-IsNewerVersion "$dbSyncApplicableVersionPath\$DbSyncVersionTestFile" "$webroot\bin\$DbSyncVersionTestFile")
            {
                # The list of assemblies to be replaced.
                # Microsoft.Dynamics.ApplicationPlatform.DatabaseSynchronize.Instrumentation.dll should be included in both lists.
                $sourceFiles = Get-ChildItem -Path $dbSyncApplicableVersionPath -File -Name
                $sourceFilesNoMan = $sourceFiles | ?{-not $_.EndsWith(".man")}
                $sourceFilesMan   = $sourceFiles | ?{     $_.EndsWith(".man") -or $_.EndsWith(".Instrumentation.dll")}

                Write-Log "Non-instrumentation files to blast: $sourceFilesNoMan"
                Write-Log "Instrumentation files to blast: $sourceFilesMan"

                # Build a destination directory list
                $ETWManifestPath = "$(Split-Path -parent $PSScriptRoot)\ETWManifest"
                if (-not (Test-Path $ETWManifestPath))
                {
                    Write-Log "Creating folder '$ETWManifestPath'"
                    New-Item -ItemType Directory -Path $ETWManifestPath
                }
                $copyNoManToDirectories = @(
                    "$webroot\bin",
                    "$packageDirectory\ApplicationPlatform\bin",
                    "$packageDirectory\bin")
                $copyManToDirectories = @(
                    "$webroot\Monitoring",
                    "$webroot\bin\Monitoring",
                    "$packageDirectory\Plugins\Monitoring",
                    "$ETWManifestPath\")

                Write-Log "Blasting non-instrumentation files to these locations: $copyNoManToDirectories"
                Write-Log "Blasting instrumentation files to these locations: $copyManToDirectories"

                # Backup to Original the first time, then to Last on subsequent runs
                $originalBackupFolder = "DbSyncOriginalBackup"
                $lastBackupFolder = "DbSyncLastBackup"
                Write-Log "Default backup folder is $originalBackupFolder"
                $backupFolder = $originalBackupFolder
                foreach ($directory in $copyNoManToDirectories)
                {
                    $originalBackupPath = Join-Path $directory $originalBackupFolder
                    if (Test-Path $originalBackupPath)
                    {
                        Write-Log "Found '$originalBackupPath', changing backup folder to $lastBackupFolder"
                        $backupFolder = $lastBackupFolder
                        break
                    }
                }
                
                $backedUpFiles = @{}
                try
                {
                    Replace-Files $dbSyncApplicableVersionPath $sourceFilesNoMan $copyNoManToDirectories $backupFolder $backedUpFiles
                    Replace-Files $dbSyncApplicableVersionPath $sourceFilesMan   $copyManToDirectories
                }
                catch
                {
                    # We do not want to leave mixed versions in case blast fails, so we do our best to return to the original state.
                    # However, this should not happen as we normally blast to a staging folder. Instead, the backed up files should
                    # be used if DbSync itself fails because of a regression.
                    Write-Exception $_
                    Restore-Backup $backedUpFiles
                    throw
                }

                Update-WebConfig

                Write-Log "DbSyncBlast assembly replacement succeeded."
            }
        }
    }
    else 
    {
       Write-Log "Skipping DbSyncBlast because of DbSyncBlastDisabled flight."
    } 
}
catch
{
    # Write the exception and that this process failed.  
    # Do not throw as this should not cause the deployment or update to fail.
    Write-Exception $_
    Write-Log "DbSyncBlast assembly replacement failed, see $log for details."
}
finally
{
    Write-Log "DbSyncBlast assembly replacement check: Stop"
}

# SIG # Begin signature block
# MIInygYJKoZIhvcNAQcCoIInuzCCJ7cCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCA9lsBXtv4PSpQ8
# U3CeeKm0RJfsYRAgHAy25FBy3TGUAqCCDYEwggX/MIID56ADAgECAhMzAAACzI61
# lqa90clOAAAAAALMMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjIwNTEyMjA0NjAxWhcNMjMwNTExMjA0NjAxWjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQCiTbHs68bADvNud97NzcdP0zh0mRr4VpDv68KobjQFybVAuVgiINf9aG2zQtWK
# No6+2X2Ix65KGcBXuZyEi0oBUAAGnIe5O5q/Y0Ij0WwDyMWaVad2Te4r1Eic3HWH
# UfiiNjF0ETHKg3qa7DCyUqwsR9q5SaXuHlYCwM+m59Nl3jKnYnKLLfzhl13wImV9
# DF8N76ANkRyK6BYoc9I6hHF2MCTQYWbQ4fXgzKhgzj4zeabWgfu+ZJCiFLkogvc0
# RVb0x3DtyxMbl/3e45Eu+sn/x6EVwbJZVvtQYcmdGF1yAYht+JnNmWwAxL8MgHMz
# xEcoY1Q1JtstiY3+u3ulGMvhAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUiLhHjTKWzIqVIp+sM2rOHH11rfQw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDcwNTI5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAeA8D
# sOAHS53MTIHYu8bbXrO6yQtRD6JfyMWeXaLu3Nc8PDnFc1efYq/F3MGx/aiwNbcs
# J2MU7BKNWTP5JQVBA2GNIeR3mScXqnOsv1XqXPvZeISDVWLaBQzceItdIwgo6B13
# vxlkkSYMvB0Dr3Yw7/W9U4Wk5K/RDOnIGvmKqKi3AwyxlV1mpefy729FKaWT7edB
# d3I4+hldMY8sdfDPjWRtJzjMjXZs41OUOwtHccPazjjC7KndzvZHx/0VWL8n0NT/
# 404vftnXKifMZkS4p2sB3oK+6kCcsyWsgS/3eYGw1Fe4MOnin1RhgrW1rHPODJTG
# AUOmW4wc3Q6KKr2zve7sMDZe9tfylonPwhk971rX8qGw6LkrGFv31IJeJSe/aUbG
# dUDPkbrABbVvPElgoj5eP3REqx5jdfkQw7tOdWkhn0jDUh2uQen9Atj3RkJyHuR0
# GUsJVMWFJdkIO/gFwzoOGlHNsmxvpANV86/1qgb1oZXdrURpzJp53MsDaBY/pxOc
# J0Cvg6uWs3kQWgKk5aBzvsX95BzdItHTpVMtVPW4q41XEvbFmUP1n6oL5rdNdrTM
# j/HXMRk1KCksax1Vxo3qv+13cCsZAaQNaIAvt5LvkshZkDZIP//0Hnq7NnWeYR3z
# 4oFiw9N2n3bb9baQWuWPswG0Dq9YT9kb+Cs4qIIwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIZnzCCGZsCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAsyOtZamvdHJTgAAAAACzDAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQglXrxCjhO
# nST/MMcB5SiCUk3JDiMxufgocBDpDFJUjl0wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQBWgIMz23MKWfa3k8/2p3AIUfISS5x77sYH3Sq7v0fb
# HwEjCjd8wUK1jPDaoidCs2Gt2rT/USDQ/HVgJx1iNsQ9rdsao5wvRCtzM/Bf6meI
# eM6UdL/Lopyai6LAd5u2dZVQW3SMofg0tju+R8Cf07WtXKz7V7IDUktPqY9dj7G2
# dq/kUelc99eZMS+24nPxvHbbNyW3JUXkzu4tc4n9LcqHujzlDxywfgv+RiHHkIGK
# n522ICy2Gwvu3lnb0voygyXfLfADOxIxPittrwvEiCCzqiNxVQHzcyx4I3uTXaKd
# eNjyf587961Wi9drtFrn3EK86WD3+5nVTKrpikXvSTz5oYIXKTCCFyUGCisGAQQB
# gjcDAwExghcVMIIXEQYJKoZIhvcNAQcCoIIXAjCCFv4CAQMxDzANBglghkgBZQME
# AgEFADCCAVkGCyqGSIb3DQEJEAEEoIIBSASCAUQwggFAAgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIEO/kgqCeYu0ikDLcwiGQUeViG9mkgQ+NSEhqPMg
# cyQZAgZjovMWCV8YEzIwMjMwMTA2MjE0NTE0LjM3OVowBIACAfSggdikgdUwgdIx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1p
# Y3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlvbnMgTGltaXRlZDEmMCQGA1UECxMdVGhh
# bGVzIFRTUyBFU046RkM0MS00QkQ0LUQyMjAxJTAjBgNVBAMTHE1pY3Jvc29mdCBU
# aW1lLVN0YW1wIFNlcnZpY2WgghF4MIIHJzCCBQ+gAwIBAgITMwAAAbn2AA1lVE+8
# AwABAAABuTANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Eg
# MjAxMDAeFw0yMjA5MjAyMDIyMTdaFw0yMzEyMTQyMDIyMTdaMIHSMQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQg
# SXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxlcyBUU1Mg
# RVNOOkZDNDEtNEJENC1EMjIwMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFt
# cCBTZXJ2aWNlMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA40k+yWH1
# FsfJAQJtQgg3EwXm5CTI3TtUhKEhNe5sulacA2AEIu8JwmXuj/Ycc5GexFyZIg0n
# +pyUCYsis6OdietuhwCeLGIwRcL5rWxnzirFha0RVjtVjDQsJzNj7zpT/yyGDGqx
# p7MqlauI85ylXVKHxKw7F/fTI7uO+V38gEDdPqUczalP8dGNaT+v27LHRDhq3HSa
# QtVhL3Lnn+hOUosTTSHv3ZL6Zpp0B3LdWBPB6LCgQ5cPvznC/eH5/Af/BNC0L2WE
# DGEw7in44/3zzxbGRuXoGpFZe53nhFPOqnZWv7J6fVDUDq6bIwHterSychgbkHUB
# xzhSAmU9D9mIySqDFA0UJZC/PQb2guBI8PwrLQCRfbY9wM5ug+41PhFx5Y9fRRVl
# Sxf0hSCztAXjUeJBLAR444cbKt9B2ZKyUBOtuYf/XwzlCuxMzkkg2Ny30bjbGo3x
# UX1nxY6IYyM1u+WlwSabKxiXlDKGsQOgWdBNTtsWsPclfR8h+7WxstZ4GpfBunhn
# zIAJO2mErZVvM6+Li9zREKZE3O9hBDY+Nns1pNcTga7e+CAAn6u3NRMB8mi285Kp
# wyA3AtlrVj4RP+VvRXKOtjAW4e2DRBbJCM/nfnQtOm/TzqnJVSHgDfD86zmFMYVm
# AV7lsLIyeljT0zTI90dpD/nqhhSxIhzIrJUCAwEAAaOCAUkwggFFMB0GA1UdDgQW
# BBS3sDhx21hDmgmMTVmqtKienjVEUjAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJl
# pxtTNRnpcjBfBgNVHR8EWDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NybC9NaWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAx
# MCgxKS5jcmwwbAYIKwYBBQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3Rh
# bXAlMjBQQ0ElMjAyMDEwKDEpLmNydDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQM
# MAoGCCsGAQUFBwMIMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQsFAAOCAgEA
# zdxns0VQdEywsrOOXusk8iS/ugn6z2SS63SFmJ/1ZK3rRLNgZQunXOZ0+pz7Dx4d
# OSGpfQYoKnZNOpLMFcGHAc6bz6nqFTE2UN7AYxlSiz3nZpNduUBPc4oGd9UEtDJR
# q+tKO4kZkBbfRw1jeuNUNSUYP5XKBAfJJoNq+IlBsrr/p9C9RQWioiTeV0Z+OcC2
# d5uxWWqHpZZqZVzkBl2lZHWNLM3+jEpipzUEbhLHGU+1x+sB0HP9xThvFVeoAB/T
# Y1mxy8k2lGc4At/mRWjYe6klcKyT1PM/k81baxNLdObCEhCY/GvQTRSo6iNSsElQ
# 6FshMDFydJr8gyW4vUddG0tBkj7GzZ5G2485SwpRbvX/Vh6qxgIscu+7zZx4NVBC
# 8/sYcQSSnaQSOKh9uNgSsGjaIIRrHF5fhn0e8CADgyxCRufp7gQVB/Xew/4qfdeA
# wi8luosl4VxCNr5JR45e7lx+TF7QbNM2iN3IjDNoeWE5+VVFk2vF57cH7JnB3ckc
# Mi+/vW5Ij9IjPO31xTYbIdBWrEFKtG0pbpbxXDvOlW+hWwi/eWPGD7s2IZKVdfWz
# vNsE0MxSP06fM6Ucr/eas5TxgS5F/pHBqRblQJ4ZqbLkyIq7Zi7IqIYEK/g4aE+y
# 017sAuQQ6HwFfXa3ie25i76DD0vrII9jSNZhpC3MA/0wggdxMIIFWaADAgECAhMz
# AAAAFcXna54Cm0mZAAAAAAAVMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9v
# dCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0yMTA5MzAxODIyMjVaFw0z
# MDA5MzAxODMyMjVaMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIICIjAN
# BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA5OGmTOe0ciELeaLL1yR5vQ7VgtP9
# 7pwHB9KpbE51yMo1V/YBf2xK4OK9uT4XYDP/XE/HZveVU3Fa4n5KWv64NmeFRiMM
# tY0Tz3cywBAY6GB9alKDRLemjkZrBxTzxXb1hlDcwUTIcVxRMTegCjhuje3XD9gm
# U3w5YQJ6xKr9cmmvHaus9ja+NSZk2pg7uhp7M62AW36MEBydUv626GIl3GoPz130
# /o5Tz9bshVZN7928jaTjkY+yOSxRnOlwaQ3KNi1wjjHINSi947SHJMPgyY9+tVSP
# 3PoFVZhtaDuaRr3tpK56KTesy+uDRedGbsoy1cCGMFxPLOJiss254o2I5JasAUq7
# vnGpF1tnYN74kpEeHT39IM9zfUGaRnXNxF803RKJ1v2lIH1+/NmeRd+2ci/bfV+A
# utuqfjbsNkz2K26oElHovwUDo9Fzpk03dJQcNIIP8BDyt0cY7afomXw/TNuvXsLz
# 1dhzPUNOwTM5TI4CvEJoLhDqhFFG4tG9ahhaYQFzymeiXtcodgLiMxhy16cg8ML6
# EgrXY28MyTZki1ugpoMhXV8wdJGUlNi5UPkLiWHzNgY1GIRH29wb0f2y1BzFa/Zc
# UlFdEtsluq9QBXpsxREdcu+N+VLEhReTwDwV2xo3xwgVGD94q0W29R6HXtqPnhZy
# acaue7e3PmriLq0CAwEAAaOCAd0wggHZMBIGCSsGAQQBgjcVAQQFAgMBAAEwIwYJ
# KwYBBAGCNxUCBBYEFCqnUv5kxJq+gpE8RjUpzxD/LwTuMB0GA1UdDgQWBBSfpxVd
# AF5iXYP05dJlpxtTNRnpcjBcBgNVHSAEVTBTMFEGDCsGAQQBgjdMg30BATBBMD8G
# CCsGAQUFBwIBFjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3Mv
# UmVwb3NpdG9yeS5odG0wEwYDVR0lBAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQC
# BAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYD
# VR0jBBgwFoAU1fZWy4/oolxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZF
# aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9v
# Q2VyQXV0XzIwMTAtMDYtMjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcw
# AoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJB
# dXRfMjAxMC0wNi0yMy5jcnQwDQYJKoZIhvcNAQELBQADggIBAJ1VffwqreEsH2cB
# MSRb4Z5yS/ypb+pcFLY+TkdkeLEGk5c9MTO1OdfCcTY/2mRsfNB1OW27DzHkwo/7
# bNGhlBgi7ulmZzpTTd2YurYeeNg2LpypglYAA7AFvonoaeC6Ce5732pvvinLbtg/
# SHUB2RjebYIM9W0jVOR4U3UkV7ndn/OOPcbzaN9l9qRWqveVtihVJ9AkvUCgvxm2
# EhIRXT0n4ECWOKz3+SmJw7wXsFSFQrP8DJ6LGYnn8AtqgcKBGUIZUnWKNsIdw2Fz
# Lixre24/LAl4FOmRsqlb30mjdAy87JGA0j3mSj5mO0+7hvoyGtmW9I/2kQH2zsZ0
# /fZMcm8Qq3UwxTSwethQ/gpY3UA8x1RtnWN0SCyxTkctwRQEcb9k+SS+c23Kjgm9
# swFXSVRk2XPXfx5bRAGOWhmRaw2fpCjcZxkoJLo4S5pu+yFUa2pFEUep8beuyOiJ
# Xk+d0tBMdrVXVAmxaQFEfnyhYWxz/gq77EFmPWn9y8FBSX5+k77L+DvktxW/tM4+
# pTFRhLy/AsGConsXHRWJjXD+57XQKBqJC4822rpM+Zv/Cuk0+CQ1ZyvgDbjmjJnW
# 4SLq8CdCPSWU5nR0W2rRnj7tfqAxM328y+l7vzhwRNGQ8cirOoo6CGJ/2XBjU02N
# 7oJtpQUQwXEGahC0HVUzWLOhcGbyoYIC1DCCAj0CAQEwggEAoYHYpIHVMIHSMQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNy
# b3NvZnQgSXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxl
# cyBUU1MgRVNOOkZDNDEtNEJENC1EMjIwMSUwIwYDVQQDExxNaWNyb3NvZnQgVGlt
# ZS1TdGFtcCBTZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQDHYh4YeGTnwxCTPNJaScZw
# uN+BOqCBgzCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMA0GCSqG
# SIb3DQEBBQUAAgUA52KJRzAiGA8yMDIzMDEwNjE5NDkyN1oYDzIwMjMwMTA3MTk0
# OTI3WjB0MDoGCisGAQQBhFkKBAExLDAqMAoCBQDnYolHAgEAMAcCAQACAgEEMAcC
# AQACAhFOMAoCBQDnY9rHAgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkK
# AwKgCjAIAgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZIhvcNAQEFBQADgYEAE5PE
# EH1bb0s1sl6NArO8OgFd9hR84X00EkFSNw+6HuTM8uPYxoJDUqU/yH1uHcof7hxa
# RsoHKbvzEI7lx7MnIEgd9+PX2jzvLftp6wzCxhqzSPYDHA/Y00RAU/MFrp1YJI3Z
# AVd7qiJo1qh3gZZ49UzJ/449nXkMm9KSwvOactkxggQNMIIECQIBATCBkzB8MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNy
# b3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAbn2AA1lVE+8AwABAAABuTAN
# BglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMC8G
# CSqGSIb3DQEJBDEiBCAj0DJwftVF2VSFlRT7BUi6yGZ/M+2PZq2DOucYPczbjjCB
# +gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EIGTrRs7xbzm5MB8lUQ7e9fZotpAV
# yBwal3Cw6iL5+g/0MIGYMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIw
# MTACEzMAAAG59gANZVRPvAMAAQAAAbkwIgQgmf46C1eV9GX+tWUz3KI2nzewnoUR
# PvdzGo3aitzf4kUwDQYJKoZIhvcNAQELBQAEggIA1u8wP45oD5pwkeqSU69ml4fd
# diqHXtdSjWrsH/P5OAUThgX/c8zOJIH3+42q+FObPq+js5E3PMyzqOrfi8bVVRBL
# na51GPa0dXAm68qrr7qJ72C2+Qneyjj5jNPm/kMYbXFKUWn633flxwWqnv3qyDbS
# mn+Vz9A39pkT8heQPVHDT/dUU/N+if72tTusiIdeWPoR8M0znJTUBt/4LzoFEVO6
# lbjJPXCRnvZoRmx6eh3xlf6VaFHqkUeGlzfnvoV37WYh3eRHOgjGh50PWbtg0Srb
# xK+WY7Nff+HcoxprMbNFV5ZGRA4/eS6kh8WNV7xIx6ej+AlammTQf1rrZD/wgK+X
# 0Kd6zEuGGmwMyrzJY4IKAIiOv5eS/TXAndL9UfnlQTZzG+xMCocXivAhN+VO/uOV
# 0/iOBmI6BDGzp30S86gLR1AODIVhlPJoU7w8xhSU2y/f6sdzE8gF6hpThGv/bMeW
# K9dM1/+XtdwyDl26+YRd2b1TEC4ddsgeA+9aAjltbHWWRnfInssbygNpzKZUa8H0
# G3bhu9xB7f7bw61lqwGr+MD5S7aQV1IR3kOkRogRMYpxA46tJY3CsI0/HCO5QW0p
# QMyB38ASZgQySFn73fV2TBvPvCs3eM/oo8sWmvCKwnpLUwu6l5D58m/Y01eSg6nL
# PjuJQDG6d77fuNp3pnU=
# SIG # End signature block
