﻿####################################################################################
####################################################################################
## ValidateAOS.ps1
## Validate AOS deployment
## Date: 09/29/2015
##
####################################################################################
####################################################################################
[CmdletBinding()]
Param(
   [Parameter(Mandatory = $true)]
   [string]$InputXml,
   [Parameter(Mandatory = $false)]
   [string]$CredentialsXml,
   [Parameter(Mandatory = $true)]
   [string]$Log
)

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking
Initialize-Log $Log

####################################################################################
## Helper Functions
####################################################################################

## Append XML Rows to Template
function Append-RowToXML
{
    [CmdletBinding()]
    Param(
       [Parameter(Mandatory = $false)]
       [string]$TestName,
       [Parameter(Mandatory = $false)]
       [string]$TestType,
       [Parameter(Mandatory = $false)]
       [string]$TestResult,
       [Parameter(Mandatory = $false)]
       [string]$RawResult,
       [Parameter(Mandatory = $false)]
       [string]$TimeStamp,
       [Parameter(Mandatory = $true)]
       [xml]$xmlTemplate
    )

    Write-Log "Getting existing rows from XML Template"
    $rows = $xmlTemplate.SelectSingleNode('CollectionResult/TabularResults/TabularData/Rows')
    Write-Log "Creating new row"
    $row = $xmlTemplate.CreateElement('ArrayOfStrings')
    $column = $xmlTemplate.CreateElement('string')#TestName
    $column.InnerText = $TestName
    $row.AppendChild($column)
    Write-Log "Adding column value: $TestName"
    $column = $xmlTemplate.CreateElement('string')#TestType
    $column.InnerText = $TestType
    $row.AppendChild($column)
    Write-Log "Adding column value: $TestType"
    $column = $xmlTemplate.CreateElement('string')#TestResult
    $column.InnerText = $TestResult
    $row.AppendChild($column)
    Write-Log "Adding column value: $TestResult"
    $column = $xmlTemplate.CreateElement('string')#RawResult
    $column.InnerText = $RawResult
    $row.AppendChild($column)
    
    $column = $xmlTemplate.CreateElement('string')#TimeStamp
    $column.InnerText = $TimeStamp
    $row.AppendChild($column)
    $rows.AppendChild($row)
    Write-Log "Adding column value: $TimeStamp"
    $xmlTemplate.CollectionResult.TabularResults.TabularData.AppendChild($rows)
    $xmlTemplate.Save($xmlTemplate)
    Write-Log "Saved rows to XML Template"
}

####################################################################################
## Validation Functions
####################################################################################

## Validate endpoint sending HTTP request to configured endpoint 
function Validate-Endpoint{ 
     [CmdletBinding()] 
    Param( 
     [Parameter(Mandatory = $true)] 
     [string]$EndPointUrl 
     ) 
 
     [bool]$result = $false 
     [string]$rawResult 
     [string]$timestamp 
 
     try{ 
         Write-Log "Connecting to '$EndPointUrl'"
         $CurrentTime = (Get-Date).ToUniversalTime() 
         $webRequest = Invoke-WebRequest -Uri $EndPointUrl -TimeoutSec (5 * 60) -UseBasicParsing 
         if($webRequest.StatusCode -eq 200){ 
             $result = $true 
             $UrlTime = [DateTime]::Parse($webRequest.Headers.Date).ToUniversalTime() 
             $rawResult = ('HttpResult: ' + $webRequest.StatusCode.ToString() + '; PingTime(ms): ' + ($CurrentTime - $UrlTime).TotalMilliseconds).ToString() 
             $timestamp = (Get-Date).ToString() 
             Write-Log "Web request returned - $rawResult"
         } 
     } 
     catch{ 
         $rawResult = $_.Exception 
         $timestamp = (Get-Date).ToString() 
         Write-Log "ERROR: $($_.Exception) CALLSTACK: $_"
     } 
 
     if($result){ 
         $returnProperties = @{ 
             Result=1; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     else{ 
         $returnProperties = @{ 
             Result=0; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     $resultObject = New-Object PsObject -Property $returnProperties 
     return $resultObject 
} 
 
## Validate install Files from a manifest file 
function Validate-Install{ 
     [CmdletBinding()] 
    Param( 
     [Parameter(Mandatory = $true)] 
     [string]$InstallPath, 
     [Parameter(Mandatory = $false)] 
     [string]$ManifestPath 
     ) 
 
     [bool]$result = $false 
     [string]$rawResult 
     [string]$timestamp 
 
     try{ 
         Write-Log "Validating Install at '$InstallPath'"
         if(Test-Path -Path $InstallPath) 
         { 
             Write-Log "Comparing '$InstallPath' to manifest"
             [System.Array]$installedfiles = @() 
             $installedFilesRaw = Get-ChildItem -Path $InstallPath -Recurse | Where {$_.PSIsContainer -eq $false} | Select-Object -Property Name 
            foreach($file in $installedFilesRaw){ 
                $installedfiles += $file.Name 
             } 
 
             if(Test-Path -Path $ManifestPath){ 
                 $manifestFiles = Get-Content -Path $ManifestPath 
                 $fileCompare = Compare-Object -ReferenceObject $manifestFiles -DifferenceObject $installedFiles -Property Name -PassThru 
                 $timestamp = (Get-Date).ToString() 
                 if($fileCompare -eq $null) 
                 { 
                     $rawResult = "Installed file ARE a match to the manifest" 
                     Write-Log "$rawResult"
                     $result = $true 
                 } 
                 else 
                 { 
                     $rawResult = ("{0} Files are missing." -f $fileCompare.Count ) 
                     Write-Log "$rawResult"
                 } 
             } 
             else{ 
                Throw "$ManifestPath does not exist." 
             } 
         } 
         else{ 
            Throw "$InstallPath does not exist." 
         } 
     } 
     catch{ 
         $rawResult = $_.Exception 
         $timestamp = (Get-Date).ToString() 
         Write-Log "ERROR: $($_.Exception) CALLSTACK: $_"
     } 
 
     if($result){ 
         $returnProperties = @{ 
             Result=1; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     else{ 
         $returnProperties = @{ 
             Result=0; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
    } 
     $resultObject = New-Object PsObject -Property $returnProperties 
     return $resultObject 
} 
 
## Validate service is running 
function Validate-Service{ 
    [CmdletBinding()] 
    Param( 
     [Parameter(Mandatory = $true)] 
     [string]$ServiceName, 
     [Parameter(Mandatory = $true)] 
     [ValidateSet("Running","Stopped","Paused")] 
     [string]$CurrentState 
     ) 
 
     [bool]$result = $false 
     [string]$rawResult 
     [string]$timestamp 
 
     try{ 
         Write-Log "Validating Service: '$ServiceName' is $CurrentState"
         $thisService = Get-Service -Name $ServiceName 
         $timestamp = (Get-Date).ToString() 
         $rawResult = ("ServiceName: {0}; DisplayName: {1}; Status: {2}" -f $thisService.Name, $thisService.DisplayName, $thisService.Status) 
         if($thisService.Status.ToString() -eq $CurrentState) 
         { 
            $result = $true 
         } 
         Write-Log "Service: $ServiceName is $($thisService.Status)"
 
     } 
     catch{ 
         $rawResult = $_.Exception 
         $timestamp = (Get-Date).ToString() 
         Write-Log "ERROR: $($_.Exception) CALLSTACK: $_"
     } 
 
     if($result){ 
         $returnProperties = @{ 
             Result=1; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     else{ 
         $returnProperties = @{ 
             Result=0; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     $resultObject = New-Object PsObject -Property $returnProperties 
     return $resultObject 
} 
 
## Validate appPool is started 
function Validate-AppPool{ 
    [CmdletBinding()] 
    Param( 
     [Parameter(Mandatory = $true)] 
     [string]$AppPoolName, 
     [Parameter(Mandatory = $true)] 
     [ValidateSet("Started","Stopped")] 
     [string]$CurrentState 
     ) 
 
     [bool]$result = $false 
     [string]$rawResult 
     [string]$timestamp 
 
     try{ 
         Write-Log "Validating AppPool: '$AppPoolName' is $CurrentState"
         Get-WebAppPoolState 
         $appPoolStatus = Get-WebAppPoolState -Name $AppPoolName 
         $timestamp = (Get-Date).ToString() 
         $rawResult = ("AppPoolName: {0}; Status: {1}" -f $AppPoolName, $appPoolStatus) 
         if($appPoolStatus.Value -eq $CurrentState) 
         { 
            $result = $true 
         } 
         Write-Log "AppPool: $AppPoolName is $($appPoolStatus.Value)"
     } 
     catch{ 
         $rawResult = $_.Exception 
         $timestamp = (Get-Date).ToString() 
         Write-Log "ERROR: $($_.Exception) CALLSTACK: $_"
     } 
 
     if($result){ 
         $returnProperties = @{ 
             Result=1; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     else{ 
         $returnProperties = @{ 
             Result=0; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     $resultObject = New-Object PsObject -Property $returnProperties 
     return $resultObject 
} 




####################################################################################
## Parameter Setting
####################################################################################
Write-Log "Setting DVT execution parameters"

if(Test-Path -Path $InputXML)
{
    Write-Log "Parsing the xml for parameters/settings"
    [xml]$DVTParams = Get-Content -Path $InputXML
    [string]$ServiceName = $DVTParams.DVTParameters.ServiceName    
    [string]$AosWebrootPath = $DVTParams.DVTParameters.AosWebRootPath    
    [string]$XmlOutputPath = $DVTParams.DVTParameters.OutputPath    
	[string]$endPoint = $DVTParams.DVTParameters.EndPoint #Infrastructure.HostUrl
    [string]$installPath = $DVTParams.DVTParameters.InstallPath
	[string]$manifestFile = $DVTParams.DVTParameters.ManifestPath
	[string]$ServiceState = $DVTParams.DVTParameters.ServiceState
	[string]$AppPoolName = $DVTParams.DVTParameters.AppPoolName
	[string]$AppPoolState = $DVTParams.DVTParameters.AppPoolState
    [string]$BatchService = $DVTParams.DVTParameters.BatchService
}
else
{
    throw "Unable to parse settings from service model. Xml doesnt exist at: $InputXML"
}

if(-not ([string]::IsNullOrEmpty($CredentialsXml)))
{
    Write-Log "Parsing the CredentialsXml"
    if(Test-Path -Path $CredentialsXml)
    {
        Write-Log "Parsing the xml for local credentials"
        $localCredentials = Import-Clixml -Path $CredentialsXml
        [string]$UserName = $localCredentials.GetNetworkCredential().UserName
        [string]$UserPassword = $localCredentials.GetNetworkCredential().Password
    }
    else
    {
        throw "Unable to parse credentials from service model. Xml doesnt exist at: $CredentialsXML"
    }
}

Write-Log "Setting diagnostics-related parameters"
[string]$CollectorName = "$($ServiceName).DVT"
[string]$CollectorType = 'PowerShellCollector'
[string]$TargetName = (hostname)

if(-not (Test-Path -Path $XmlOutputPath))
{
    Write-Log "Creating diagnostics result directory at $XmlOutputPath"
    New-Item -Path $XmlOutputPath -Type Container | Out-Null
}

[string]$XMLFilePath = Join-Path -Path $XmlOutputPath -ChildPath "$([System.DateTime]::Now.ToFileTimeUtc())_$($ServiceName)DVTResults.xml"

####################################################################################
## Diagnostics Collector XML Template
####################################################################################
[xml]$xmlTemplate = @"
<?xml version="1.0"?>
<CollectionResult xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema">
  <CollectorName>$CollectorName</CollectorName>
  <CollectorType>$CollectorType</CollectorType>
  <ErrorMessages />
  <TabularResults>
    <TabularData>
      <TargetName>$TargetName</TargetName>
      <Columns>
        <string>TestName</string>
        <string>TestType</string>
        <string>PassResult</string>
        <string>RawResult</string>
        <string>TimeStamp</string>
      </Columns>
      <Rows>
      </Rows>
    </TabularData>
  </TabularResults>
</CollectionResult>
"@

####################################################################################
## Main Execution
####################################################################################

Write-Log "Running validations for $ServiceName"
try
{
     #End point validation 
     Write-Log "Validate-Endpoint -EndPointUrl" 
     $endpointResult = Validate-Endpoint -EndPointUrl $endPoint
 
     Append-RowToXML -TestName 'AOS.Validate-Endpoint' -TestType 'DVT' -TestResult $endpointResult.Result -RawResult $endpointResult.RawResults -TimeStamp $endpointResult.TimeStamp -xmlTemplate $xmlTemplate | Out-Null 

    $ValidateBatch = (![System.String]::IsNullOrWhiteSpace($DVTParams.DVTParameters.ValidateBatch) -and [System.Convert]::ToBoolean($DVTParams.DVTParameters.ValidateBatch))
    if ($ValidateBatch)
    {
        #AXBatch Service 
        Write-Log "Validate-Service -ServiceName $BatchService -CurrentState $ServiceState" 
        $serviceResult = Validate-Service -ServiceName $BatchService -CurrentState $ServiceState 

        Append-RowToXML -TestName 'AOS.Validate-Service' -TestType 'DVT' -TestResult $serviceResult.Result -RawResult $serviceResult.RawResults -TimeStamp $serviceResult.TimeStamp -xmlTemplate $xmlTemplate | Out-Null 
    } 

     #IIS AppPool Validation 
     Write-Log "Validate-AppPool -AppPoolName $AppPoolName -CurrentState $AppPoolState"
     $apppoolResult = Validate-AppPool -AppPoolName $AppPoolName -CurrentState $AppPoolState 

     Append-RowToXML -TestName 'AOS.Validate-AppPool' -TestType 'DVT' -TestResult $apppoolResult.Result -RawResult $apppoolResult.RawResults -TimeStamp $apppoolResult.TimeStamp -xmlTemplate $xmlTemplate | Out-Null 

    #Writing XML results
    Write-Log "Writing DVT results to $XMLFilePath"
    $xmlTemplate.InnerXml | Out-File -FilePath $XMLFilePath -Force -Encoding utf8

    [bool]$dvtResult = $endpointResult.Result -and $apppoolResult.Result
    if ($ValidateBatch)
    {
        $dvtResult = $dvtResult -and $serviceResult.Result
    }

}
catch
{
    Write-Exception $_
}

if($dvtResult)
{
    $exitProperties = @{'ExitCode'= 0}
    $exitObject = New-Object PsObject -Property $exitProperties
    Write-Log "DVT Script Completed, ExitCode: $($exitObject.ExitCode)"
    return $exitObject
}
else
{
    $exitProperties = @{'ExitCode'= 1; 'Message'="DVT Validation failed, see log: '$Log' for further details, and '$XMLFilePath' for test results"}
    $exitObject = New-Object PsObject -Property $exitProperties
    Write-Log "DVT Script Completed, ExitCode: $($exitObject.ExitCode)"
    throw $exitObject
}


# SIG # Begin signature block
# MIInkwYJKoZIhvcNAQcCoIInhDCCJ4ACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBPDETZSfEdM3Dd
# jYdA0NRlWVmqeL50qgjbytga87dgeaCCDXYwggX0MIID3KADAgECAhMzAAACy7d1
# OfsCcUI2AAAAAALLMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjIwNTEyMjA0NTU5WhcNMjMwNTExMjA0NTU5WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQC3sN0WcdGpGXPZIb5iNfFB0xZ8rnJvYnxD6Uf2BHXglpbTEfoe+mO//oLWkRxA
# wppditsSVOD0oglKbtnh9Wp2DARLcxbGaW4YanOWSB1LyLRpHnnQ5POlh2U5trg4
# 3gQjvlNZlQB3lL+zrPtbNvMA7E0Wkmo+Z6YFnsf7aek+KGzaGboAeFO4uKZjQXY5
# RmMzE70Bwaz7hvA05jDURdRKH0i/1yK96TDuP7JyRFLOvA3UXNWz00R9w7ppMDcN
# lXtrmbPigv3xE9FfpfmJRtiOZQKd73K72Wujmj6/Su3+DBTpOq7NgdntW2lJfX3X
# a6oe4F9Pk9xRhkwHsk7Ju9E/AgMBAAGjggFzMIIBbzAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUrg/nt/gj+BBLd1jZWYhok7v5/w4w
# RQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEW
# MBQGA1UEBRMNMjMwMDEyKzQ3MDUyODAfBgNVHSMEGDAWgBRIbmTlUAXTgqoXNzci
# tW2oynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NybC9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3JsMGEG
# CCsGAQUFBwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDovL3d3dy5taWNyb3NvZnQu
# Y29tL3BraW9wcy9jZXJ0cy9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3J0
# MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQELBQADggIBAJL5t6pVjIRlQ8j4dAFJ
# ZnMke3rRHeQDOPFxswM47HRvgQa2E1jea2aYiMk1WmdqWnYw1bal4IzRlSVf4czf
# zx2vjOIOiaGllW2ByHkfKApngOzJmAQ8F15xSHPRvNMmvpC3PFLvKMf3y5SyPJxh
# 922TTq0q5epJv1SgZDWlUlHL/Ex1nX8kzBRhHvc6D6F5la+oAO4A3o/ZC05OOgm4
# EJxZP9MqUi5iid2dw4Jg/HvtDpCcLj1GLIhCDaebKegajCJlMhhxnDXrGFLJfX8j
# 7k7LUvrZDsQniJZ3D66K+3SZTLhvwK7dMGVFuUUJUfDifrlCTjKG9mxsPDllfyck
# 4zGnRZv8Jw9RgE1zAghnU14L0vVUNOzi/4bE7wIsiRyIcCcVoXRneBA3n/frLXvd
# jDsbb2lpGu78+s1zbO5N0bhHWq4j5WMutrspBxEhqG2PSBjC5Ypi+jhtfu3+x76N
# mBvsyKuxx9+Hm/ALnlzKxr4KyMR3/z4IRMzA1QyppNk65Ui+jB14g+w4vole33M1
# pVqVckrmSebUkmjnCshCiH12IFgHZF7gRwE4YZrJ7QjxZeoZqHaKsQLRMp653beB
# fHfeva9zJPhBSdVcCW7x9q0c2HVPLJHX9YCUU714I+qtLpDGrdbZxD9mikPqL/To
# /1lDZ0ch8FtePhME7houuoPcMIIHejCCBWKgAwIBAgIKYQ6Q0gAAAAAAAzANBgkq
# hkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
# EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
# IDIwMTEwHhcNMTEwNzA4MjA1OTA5WhcNMjYwNzA4MjEwOTA5WjB+MQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQg
# Q29kZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
# CgKCAgEAq/D6chAcLq3YbqqCEE00uvK2WCGfQhsqa+laUKq4BjgaBEm6f8MMHt03
# a8YS2AvwOMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc6Whe0t+bU7IKLMOv2akr
# rnoJr9eWWcpgGgXpZnboMlImEi/nqwhQz7NEt13YxC4Ddato88tt8zpcoRb0Rrrg
# OGSsbmQ1eKagYw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+lD3v++MrWhAfTVYoonpy
# 4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+EGvKhL1nkkDstrjNYxbc+/jLTswM9
# sbKvkjh+0p2ALPVOVpEhNSXDOW5kf1O6nA+tGSOEy/S6A4aN91/w0FK/jJSHvMAh
# dCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rhKEmdX4jiJV3TIUs+UsS1Vz8k
# A/DRelsv1SPjcF0PUUZ3s/gA4bysAoJf28AVs70b1FVL5zmhD+kjSbwYuER8ReTB
# w3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zdsGbiwZeBe+3W7UvnSSmn
# Eyimp31ngOaKYnhfsi+E11ecXL93KCjx7W3DKI8sj0A3T8HhhUSJxAlMxdSlQy90
# lfdu+HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS4NaIjAsCAwEAAaOCAe0w
# ggHpMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBRIbmTlUAXTgqoXNzcitW2o
# ynUClTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYD
# VR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBRyLToCMZBDuRQFTuHqp8cx0SOJNDBa
# BgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
# bC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3JsMF4GCCsG
# AQUFBwEBBFIwUDBOBggrBgEFBQcwAoZCaHR0cDovL3d3dy5taWNyb3NvZnQuY29t
# L3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3J0MIGfBgNV
# HSAEgZcwgZQwgZEGCSsGAQQBgjcuAzCBgzA/BggrBgEFBQcCARYzaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1hcnljcHMuaHRtMEAGCCsG
# AQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAHAAbwBsAGkAYwB5AF8AcwB0AGEAdABl
# AG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn8oalmOBUeRou09h0ZyKb
# C5YR4WOSmUKWfdJ5DJDBZV8uLD74w3LRbYP+vj/oCso7v0epo/Np22O/IjWll11l
# hJB9i0ZQVdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0bpdS1HXeUOeLpZMlEPXh6
# I/MTfaaQdION9MsmAkYqwooQu6SpBQyb7Wj6aC6VoCo/KmtYSWMfCWluWpiW5IP0
# wI/zRive/DvQvTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvyCInWH8MyGOLwxS3OW560
# STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn+N4sOiBpmLJZiWhub6e3dMNABQam
# ASooPoI/E01mC8CzTfXhj38cbxV9Rad25UAqZaPDXVJihsMdYzaXht/a8/jyFqGa
# J+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw3MYbBL7fQccOKO7eZS/sl/ah
# XJbYANahRr1Z85elCUtIEJmAH9AAKcWxm6U/RXceNcbSoqKfenoi+kiVH6v7RyOA
# 9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sLgOppO6/8MO0ETI7f33Vt
# Y5E90Z1WTk+/gFcioXgRMiF670EKsT/7qMykXcGhiJtXcVZOSEXAQsmbdlsKgEhr
# /Xmfwb1tbWrJUnMTDXpQzTGCGXMwghlvAgEBMIGVMH4xCzAJBgNVBAYTAlVTMRMw
# EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVN
# aWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNp
# Z25pbmcgUENBIDIwMTECEzMAAALLt3U5+wJxQjYAAAAAAsswDQYJYIZIAWUDBAIB
# BQCgga4wGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIFblnddwkWx06Q6BUqL75BZn
# V2OJrQnJQ/zIfszQnRDJMEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8A
# cwBvAGYAdKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEB
# BQAEggEAqi5BaKJZqUGyMpYx2Jm2F++y+jnDZ5H5bEXOetctrvBX/5mX4GSYXTZc
# SNMCqq1SW3fIEu7Td+aI/AKyv2FwahamHE9UWJkB7fb7pAR19yZxQukPok6dTEwo
# kV2tUMaU87EKE68qOnd2nwklW4XmkWskB2z7tMHVgYOL1b//GLU1SYhTCGcrzCcP
# 15URKQFQJOjhJgjEk4kLRGj5h6WpcfKi/6h2/UDF0oY9PJKtWzlvbzNj/kJE+wBA
# h+1Gc7Lzi0czahCkgRxw0rcxDa6Yyca6qUmx2fXt31IwvGTgXtLiLxEIJvcTyG/z
# 44dgrE4dLFZqTjE5hzEPjxS3OvtqJKGCFv0wghb5BgorBgEEAYI3AwMBMYIW6TCC
# FuUGCSqGSIb3DQEHAqCCFtYwghbSAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFRBgsq
# hkiG9w0BCRABBKCCAUAEggE8MIIBOAIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFl
# AwQCAQUABCADB4tyCpdM5mcP0BPrNcIV6rCgLpWMbuwasu+MnMB81gIGZBMfCeIS
# GBMyMDIzMDMyNTA2NDAxOC40OTNaMASAAgH0oIHQpIHNMIHKMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1l
# cmljYSBPcGVyYXRpb25zMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjo3QkYxLUUz
# RUEtQjgwODElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZaCC
# EVQwggcMMIIE9KADAgECAhMzAAAByPmw7mft6mtGAAEAAAHIMA0GCSqGSIb3DQEB
# CwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
# EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNV
# BAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMB4XDTIyMTEwNDE5MDEz
# N1oXDTI0MDIwMjE5MDEzN1owgcoxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJTAjBgNVBAsTHE1pY3Jvc29mdCBBbWVyaWNhIE9wZXJhdGlvbnMx
# JjAkBgNVBAsTHVRoYWxlcyBUU1MgRVNOOjdCRjEtRTNFQS1CODA4MSUwIwYDVQQD
# ExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNlMIICIjANBgkqhkiG9w0BAQEF
# AAOCAg8AMIICCgKCAgEAucudfihPgyRWwnnIuJCqc3TCtFk0XOimFcKjU9bS6WFn
# g2l+FrIid0mPZ7KWs6Ewj21X+ZkGkM6x+ozHlmNtnHSQ48pjIFdlKXIoh7fSo41A
# 4n0tQIlwhs8uIYIocp72xwDBHKSZxGaEa/0707iyOw+aXZXNcTxgNiREASb9thlL
# ZM75mfJIgBVvUmdLZc+XOUYwz/8ul7IEztPNH4cn8Cn0tJhIFfp2netr8GYNoiyI
# qxueG7+sSt2xXl7/igc5cHPZnWhfl9PaB4+SutrA8zAhzVHTnj4RffxA4R3k4BRb
# PdGowQfOf95ZeYxLTHf5awB0nqZxOY+yuGWhf6hp5RGRouc9beVZv98M1erYa55S
# 1ahZgGDQJycVtEy82RlmKfTYY2uNmlPLWtnD7sDlpVkhYQGKuTWnuwQKq9ZTSE+0
# V2cH8JaWBYJQMIuWWM83vLPo3IT/S/5jT2oZOS9nsJgwwCwRUtYtwtq8/PJtvt1V
# 6VoG4Wd2/MAifgEJOkHF7ARPqI9Xv28+riqJZ5mjLGz84dP2ryoe0lxYSz3PT5Er
# KoS0+zJpYNAcxbv2UXiTk3Wj/mZ3tulz6z4XnSl5gy0PLer+EVjz4G96GcZgK2d9
# G+uYylHWwBneIv9YFQj6yMdW/4sEpkEbrpiJNemcxUCmBipZ7Sc35rv4utkJ4/UC
# AwEAAaOCATYwggEyMB0GA1UdDgQWBBS1XC9JgbrSwLDTiJJT4iK7NUvk9TAfBgNV
# HSMEGDAWgBSfpxVdAF5iXYP05dJlpxtTNRnpcjBfBgNVHR8EWDBWMFSgUqBQhk5o
# dHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9NaWNyb3NvZnQlMjBU
# aW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcmwwbAYIKwYBBQUHAQEEYDBeMFwG
# CCsGAQUFBzAChlBodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRz
# L01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0ElMjAyMDEwKDEpLmNydDAMBgNV
# HRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IC
# AQDD1nJSyEPDqSgnfkFifIbteJb7NkZCbRj5yBGiT1f9fTGvUb5CW7k3eSp3uxUq
# om9LWykcNfQa/Yfw0libEim9YRjUNcL42oIFqtp/7rl9gg61oiB8PB+6vLEmjXkY
# xUUR8WjKKC5Q5dx96B21faSco2MOmvjYxGUR7An+4529lQPPLqbEKRjcNQb+p+mk
# QH2XeMbsh5EQCkTuYAimFTgnui2ZPFLEuBpxBK5z2HnKneHUJ9i4pcKWdCqF1AOV
# N8gXIH0R0FflMcCg5TW8v90Vwx/mP3aE2Ige1uE8M9YNBn5776PxmA16Z+c2s+hY
# I+9sJZhhRA8aSYacrlLz7aU/56OvEYRERQZttuAFkrV+M/J+tCeGNv0Gd75Y4lKL
# Mp5/0xoOviPBdB2rD5C/U+B8qt1bBqQLVZ1wHRy0/6HhJxbOi2IgGJaOCYLGX2zz
# 0VAT6mZ2BTWrJmcK6SDv7rX7psgC+Cf1t0R1aWCkCHJtpYuyKjf7UodRazevOf6V
# 01XkrARHKrI7bQoHFL+sun2liJCBjN51mDWoEgUCEvwB3l+RFYAL0aIisc5cTaGX
# /T8F+iAbz+j2GGVum85gEQS9uLzSedoYPyEXxTblwewGdAxqIZaKozRBow49OnL+
# 5CgooVMf3ZSqpxc2QC0E03l6c/vChkYyqMXq7Lwd4PnHqjCCB3EwggVZoAMCAQIC
# EzMAAAAVxedrngKbSZkAAAAAABUwDQYJKoZIhvcNAQELBQAwgYgxCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jvc29mdCBS
# b290IENlcnRpZmljYXRlIEF1dGhvcml0eSAyMDEwMB4XDTIxMDkzMDE4MjIyNVoX
# DTMwMDkzMDE4MzIyNVowfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
# b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwggIi
# MA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDk4aZM57RyIQt5osvXJHm9DtWC
# 0/3unAcH0qlsTnXIyjVX9gF/bErg4r25PhdgM/9cT8dm95VTcVrifkpa/rg2Z4VG
# Iwy1jRPPdzLAEBjoYH1qUoNEt6aORmsHFPPFdvWGUNzBRMhxXFExN6AKOG6N7dcP
# 2CZTfDlhAnrEqv1yaa8dq6z2Nr41JmTamDu6GnszrYBbfowQHJ1S/rboYiXcag/P
# XfT+jlPP1uyFVk3v3byNpOORj7I5LFGc6XBpDco2LXCOMcg1KL3jtIckw+DJj361
# VI/c+gVVmG1oO5pGve2krnopN6zL64NF50ZuyjLVwIYwXE8s4mKyzbnijYjklqwB
# Sru+cakXW2dg3viSkR4dPf0gz3N9QZpGdc3EXzTdEonW/aUgfX782Z5F37ZyL9t9
# X4C626p+Nuw2TPYrbqgSUei/BQOj0XOmTTd0lBw0gg/wEPK3Rxjtp+iZfD9M269e
# wvPV2HM9Q07BMzlMjgK8QmguEOqEUUbi0b1qGFphAXPKZ6Je1yh2AuIzGHLXpyDw
# wvoSCtdjbwzJNmSLW6CmgyFdXzB0kZSU2LlQ+QuJYfM2BjUYhEfb3BvR/bLUHMVr
# 9lxSUV0S2yW6r1AFemzFER1y7435UsSFF5PAPBXbGjfHCBUYP3irRbb1Hode2o+e
# FnJpxq57t7c+auIurQIDAQABo4IB3TCCAdkwEgYJKwYBBAGCNxUBBAUCAwEAATAj
# BgkrBgEEAYI3FQIEFgQUKqdS/mTEmr6CkTxGNSnPEP8vBO4wHQYDVR0OBBYEFJ+n
# FV0AXmJdg/Tl0mWnG1M1GelyMFwGA1UdIARVMFMwUQYMKwYBBAGCN0yDfQEBMEEw
# PwYIKwYBBQUHAgEWM2h0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvRG9j
# cy9SZXBvc2l0b3J5Lmh0bTATBgNVHSUEDDAKBggrBgEFBQcDCDAZBgkrBgEEAYI3
# FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAf
# BgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvXzpoYxDBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNS
# b29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0Nl
# ckF1dF8yMDEwLTA2LTIzLmNydDANBgkqhkiG9w0BAQsFAAOCAgEAnVV9/Cqt4Swf
# ZwExJFvhnnJL/Klv6lwUtj5OR2R4sQaTlz0xM7U518JxNj/aZGx80HU5bbsPMeTC
# j/ts0aGUGCLu6WZnOlNN3Zi6th542DYunKmCVgADsAW+iehp4LoJ7nvfam++Kctu
# 2D9IdQHZGN5tggz1bSNU5HhTdSRXud2f8449xvNo32X2pFaq95W2KFUn0CS9QKC/
# GbYSEhFdPSfgQJY4rPf5KYnDvBewVIVCs/wMnosZiefwC2qBwoEZQhlSdYo2wh3D
# YXMuLGt7bj8sCXgU6ZGyqVvfSaN0DLzskYDSPeZKPmY7T7uG+jIa2Zb0j/aRAfbO
# xnT99kxybxCrdTDFNLB62FD+CljdQDzHVG2dY3RILLFORy3BFARxv2T5JL5zbcqO
# Cb2zAVdJVGTZc9d/HltEAY5aGZFrDZ+kKNxnGSgkujhLmm77IVRrakURR6nxt67I
# 6IleT53S0Ex2tVdUCbFpAUR+fKFhbHP+CrvsQWY9af3LwUFJfn6Tvsv4O+S3Fb+0
# zj6lMVGEvL8CwYKiexcdFYmNcP7ntdAoGokLjzbaukz5m/8K6TT4JDVnK+ANuOaM
# mdbhIurwJ0I9JZTmdHRbatGePu1+oDEzfbzL6Xu/OHBE0ZDxyKs6ijoIYn/ZcGNT
# TY3ugm2lBRDBcQZqELQdVTNYs6FwZvKhggLLMIICNAIBATCB+KGB0KSBzTCByjEL
# MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1v
# bmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjElMCMGA1UECxMcTWlj
# cm9zb2Z0IEFtZXJpY2EgT3BlcmF0aW9uczEmMCQGA1UECxMdVGhhbGVzIFRTUyBF
# U046N0JGMS1FM0VBLUI4MDgxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1w
# IFNlcnZpY2WiIwoBATAHBgUrDgMCGgMVAN/OE1C7xjU0ClIDXQBiucAY7suyoIGD
# MIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
# BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQG
# A1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwDQYJKoZIhvcNAQEF
# BQACBQDnyNGUMCIYDzIwMjMwMzI1MDk0OTA4WhgPMjAyMzAzMjYwOTQ5MDhaMHQw
# OgYKKwYBBAGEWQoEATEsMCowCgIFAOfI0ZQCAQAwBwIBAAICJHEwBwIBAAICEeUw
# CgIFAOfKIxQCAQAwNgYKKwYBBAGEWQoEAjEoMCYwDAYKKwYBBAGEWQoDAqAKMAgC
# AQACAwehIKEKMAgCAQACAwGGoDANBgkqhkiG9w0BAQUFAAOBgQCEjp6pKUDfm+kS
# ldbw/eDMdBZiXhGCm5LAFMlCGIXyysJym+RZAfCftly7EgNlosq0xb6rIN5UDxeH
# o61mISZwbqtjkjnejqQqAPxaVzIx6GhXEXgUZqorI9QTw1+Zh9iQASoOw4ZGVdGp
# L9z8tVmIzGhZLjfLCROg19X+FDNDlzGCBA0wggQJAgEBMIGTMHwxCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBU
# aW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAByPmw7mft6mtGAAEAAAHIMA0GCWCGSAFl
# AwQCAQUAoIIBSjAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwLwYJKoZIhvcN
# AQkEMSIEIEXdRP5M2YQ/PnQphK4UuVYbPS+3Fb/psKMcCV7i8BzmMIH6BgsqhkiG
# 9w0BCRACLzGB6jCB5zCB5DCBvQQgYgCYz80/baMvxw6jcqSvL0FW4TdvA09nxHfs
# PhuEA2YwgZgwgYCkfjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAA
# Acj5sO5n7eprRgABAAAByDAiBCD/9qBoXBXgwdixePXUXCzJJpn+i+VI4eRQ02W0
# Lx53GDANBgkqhkiG9w0BAQsFAASCAgB8xcddy4m9qfX5q6Kza2PULwBkjay7HgIh
# vtq8LK+3SKQw0onOCzS1Fku6iXHAQPrbSETLN7NDJh4roN9s4OnhYBJ/kE6JcEYA
# KXyFz8lCIoQey7l5ALIBt4rQSJDk0q0cWyEjnMlxGnG1/HoVqDv1T5HXDqwbHVwH
# Xtm0eSSXAdCUKE8pDZloGxhpauBRZYjApqvzkML5GxwKXVDn2Fd3+NLbrq9YoYtG
# JTspqVG60Yqk9+OwaMEjrGGPlAB8w7eculYI4wA9yWgES9TRnOjLXE62FEYKS+/q
# g+0CgEt5jObHpmiwvVMxh19jPUfeHMjdn1dCS6u8gTbnFLYCInYeFWMfDe+xV1cC
# lWvpRS99K1PDj2q4FNucL7nP3IIIriUo2Wh5A3JrpYHV2vWDpzXlXVfnzXT+d8Kj
# PQrlLUuwhZSDb2Ps1w+PlHE7YN+9q3Ea1PvtcPDv0khVRKN31cyFuxt0cMup4HKC
# s1Wr64DFZUYIRNHaWpwpFfZYjN6ipZCAEYdSuacEDnr+CO40rDLFiOLrKmsoLdXf
# WEb4rLOcj1kfRDoCxTatZp0mYJ6ZY7Z/OSwUXA5k5JHdy2zU9fIGd/BJXfEICvS8
# zirhYISmafEbwrV0B/Z3iuVwJXmAoBUJItfwBUp/WBRpExKd0HtnSg6HqNU2o24B
# to9f7nBBaA==
# SIG # End signature block
