#######################################################
###              Restore Admin Account             ####
#######################################################

PARAM (
    [Parameter(Mandatory=$true)]
    $log,

    [Parameter(Mandatory=$false)]
    $UserInfoTableName = "USERINFO"
)

# Import modules and initialize the log
Import-Module WebAdministration
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking
Initialize-Log $log

function Add-AdminAccount()
{
    $expectedAdminSID = Get-Sid -Email $AdminPrincipalName -Domain $AdminIdentityProvider
    Write-LogWithoutEmail "Inserting a new Admin account record with SID='$($expectedAdminSID)', NetworkDomain='$($AdminIdentityProvider)', and NetworkAlias='$($AdminPrincipalName)'..."
    $target_sqlParams.Query = @"
    INSERT INTO $($UserInfoTableName) (ID, NAME, ENABLE, STATUSLINEINFO, TOOLBARINFO, DEBUGINFO, AUTOINFO, AUTOUPDATE, GARBAGECOLLECTLIMIT, HISTORYLIMIT, MESSAGELIMIT, GENERALINFO, SHOWSTATUSLINE, SHOWTOOLBAR, SHOWAOTLAYER, CONFIRMDELETE, REPORTFONTSIZE, FORMFONTSIZE, PROPERTYFONTSIZE, COMPANY, COMPILERWARNINGLEVEL, SID, NETWORKDOMAIN, NETWORKALIAS, LANGUAGE, HELPLANGUAGE, PREFERREDTIMEZONE, NOTIFYTIMEZONEMISMATCH, ACCOUNTTYPE, DEFAULTPARTITION, INTERACTIVELOGON)  
    VALUES ('Admin', 'Admin', 1, -538365, -1, 12, -1, 6, 20, 5, 1000, -8209, 1, 1, 4, -1, 9, 9, 9, 'DAT', 3, '$($expectedAdminSID)', '$($AdminIdentityProvider)', '$($AdminPrincipalName)', 'en-US', 'en-us', 58, 1, 2, 1, 1)
"@
    Invoke-SqlQueryWithLog $target_sqlParams | out-null
    Write-LogWithoutEmail "Added a new Admin record to $($UserInfoTableName) table."
}

function Ensure-AdminAccountIsEnabled($AdminAccount)
{
    if ($AdminAccount.ENABLE -ne 1)
    {
        Write-LogWithoutEmail "Admin account is disabled. Enabling Admin account..."
        $target_sqlParams.Query = @"
            UPDATE $($UserInfoTableName) set ENABLE = 1 WHERE ID = 'Admin'
"@
        Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
        Write-LogWithoutEmail "Finished enabling Admin account."
    }
    else
    {
        Write-LogWithoutEmail "Validated that Admin account is enabled."
    }
}

function Ensure-AdminAccountHasCorrectNetworkDomain($AdminAccount)
{
    if ($AdminAccount.NETWORKDOMAIN -ne $AdminIdentityProvider)
    {
        Write-LogWithoutEmail "The Admin account's NetworkDomain '$($AdminAccount.NETWORKDOMAIN)' in $($UserInfoTableName) does not match default provisioning admin identity provider '$($AdminIdentityProvider)'."
        $target_sqlParams.Query = @"
            UPDATE $($UserInfoTableName) set NETWORKDOMAIN = '$AdminIdentityProvider' WHERE ID = 'Admin'
"@
        Write-LogWithoutEmail "Updated the Admin account's NetworkDomain to '$($AdminIdentityProvider)' in $($UserInfoTableName)."
        Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
    }
    else
    {
        Write-LogWithoutEmail "Admin network domain is correct."
    }
}

function Ensure-AdminAccountHasCorrectSID($AdminAccount)
{
    $expectedAdminSID = Get-Sid -Email $AdminPrincipalName -Domain $AdminIdentityProvider
    if ([string]::IsNullOrEmpty($expectedAdminSID))
    {
        Write-LogWithoutEmail "Expected admin SID is null or empty. Ignoring database check for admin SID."
    }
    elseif ($AdminAccount.SID -ne $expectedAdminSID)
    {
        Write-LogWithoutEmail "SID in database '$($AdminAccount.SID)' does not match generated SID '$($expectedAdminSID)'. Updating SID in database..."
        $target_sqlParams.Query = @"
            UPDATE $($UserInfoTableName) set SID = '$($expectedAdminSID)' WHERE ID = 'Admin'
"@
        Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
        Write-LogWithoutEmail "Finished updating the SID."
    }
    else
    {
        Write-LogWithoutEmail "SID in database matches the generated SID."
    }
}

function Ensure-NoDuplicateAdminAccounts()
{
    $adminUserCount = Get-UserCount -Id "Admin"
    # Validate that ADMIN account is unique.
    if ($adminUserCount -gt 1)
    {
        Write-LogWithoutEmail "$($adminUserCount) admin records found. Removing duplicates."
        $target_sqlParams.Query = @"
        DECLARE @expectedCount int = 0,
        @adminPrincipalName varchar(100) = '$($AdminPrincipalName)'
            SELECT @expectedCount = COUNT(*) FROM $($UserInfoTableName) WHERE ID = 'Admin' and NETWORKALIAS = @adminPrincipalName;
    
                    IF @expectedCount >= 1 
                    BEGIN
                        DELETE $($UserInfoTableName) WHERE ID = 'Admin' and NETWORKALIAS != @adminPrincipalName;
                        IF @expectedCount > 1 
                        BEGIN
                            DELETE TOP(@expectedCount - 1) $($UserInfoTableName) WHERE ID = 'Admin'
                        END;
                    END;

                    ELSE
                    BEGIN
                        DELETE TOP($($adminUserCount) - 1) $($UserInfoTableName) WHERE ID = 'Admin'
                    END;
"@
        Invoke-SqlQueryWithLog $target_sqlParams | out-null
        Write-LogWithoutEmail "Removed the duplicate Admin records."
    }
    else
    {
        Write-LogWithoutEmail "No duplicate Admin records found in $($UserInfoTableName) table."
    }
}

function Ensure-NoDuplicateNetworkAliasAccounts($NetworkAlias)
{
    # Check if db has duplicate networkalias with admin account
    $adminUserCount = Get-UserCount -NetworkAlias $NetworkAlias

    # Validate that ADMIN networkalias account is not duplicated.
    if ($adminUserCount -gt 1)
    {
        $target_sqlParams.Query = @"
        DELETE $($UserInfoTableName) WHERE NETWORKALIAS = '$($NetworkAlias)' AND ID != 'Admin'
"@
        Invoke-SqlQueryWithLog $target_sqlParams | out-null
        Write-LogWithoutEmail "Removed the non Admin records where NetworkAlias was set to '$($NetworkAlias)'."
    }
}

function Get-AdminAccount()
{
    $target_sqlParams.Query = @"
    SELECT	ENABLE,
            NETWORKALIAS,
            IDENTITYPROVIDER,
            NETWORKDOMAIN,
            SID 
    FROM $($UserInfoTableName) WHERE ID = 'Admin'
"@
    return Invoke-SqlQueryWithLog $target_sqlParams
}

function Get-Sid($Email, $Domain)
{
    $sid = $null
    $dllPath = Join-Path $webrootBinPath "Microsoft.Dynamics.AX.Security.SidGenerator.dll"
    if (Test-Path $dllPath)
    {
        Write-LogWithoutEmail "Loading Microsoft.Dynamics.AX.Security.SidGenerator.dll from path '$($dllPath)' to generate SID..."
        $assembly = [Reflection.Assembly]::LoadFile($dllPath)

        Write-LogWithoutEmail "Loading SidGenerator type from assembly..."
        $sidGeneratorType = $assembly.GetType("Microsoft.Dynamics.Ax.Security.SidGenerator")

        Write-LogWithoutEmail "Attempting to find new SidGenerator method 'GenerateSIDByDefaultAlgo' (available in PU37+)..."
        $newGenerateSidMethod = $sidGeneratorType.GetMethod("GenerateSIDByDefaultAlgo")
        if ($null -ne $newGenerateSidMethod)
        {
            Write-LogWithoutEmail "New SidGenerator::GenerateSIDByDefaultAlgo(...) method found - using it."
            Write-LogWithoutEmail "Generating SID using email '$($Email)' and domain '$($Domain)'..."
            $sid = [Microsoft.Dynamics.Ax.Security.SidGenerator]::GenerateSIDByDefaultAlgo($Email, $Domain)
        }
        else
        {
            Write-LogWithoutEmail "Could not find method SidGenerator::GenerateSIDByDefaultAlgo(...) - will use old SidGenerator::Generate(email, domain) method instead."
            Write-LogWithoutEmail "Generating SID using email '$($Email)' and domain '$($Domain)'..."
            $sid = [Microsoft.Dynamics.Ax.Security.SidGenerator]::Generate($Email, $Domain)
        }

        Write-LogWithoutEmail "Generated SID: '$($sid)'"
    }
    else
    {
        Write-LogWithoutEmail "Unable to find SIDGenerator assembly '$($dllPath)'. Cannot generate SID."
    }

    return $sid
}

function Get-UserCount(
    [Parameter(Mandatory=$true, ParameterSetName="Id")]
    [String]
    $Id,

    [Parameter(Mandatory=$true, ParameterSetName="NetworkAlias")]
    [String]
    $NetworkAlias
)
{
    if (-not [string]::IsNullOrEmpty($Id))
    {
        $whereClause = "ID = '$($Id)'"
    }
    else
    {
        $whereClause = "NETWORKALIAS = '$($NetworkAlias)'"
    }

    $target_sqlParams.Query = @"
        SELECT COUNT(*) AS UserCount FROM $($UserInfoTableName) WHERE $($whereClause)
"@
    $userCountResults = Invoke-SqlQueryWithLog $target_sqlParams

    [int]$count = $userCountResults.UserCount
    Write-LogWithoutEmail "Number of users with $($whereClause): $($count)"
    return $count
}

function Initialize-Settings()
{
    $webroot = Get-AosWebSitePhysicalPath
    Write-LogWithoutEmail "The Service drive path of the webroot on the aos machine is: $($webroot)"

    $script:webrootBinPath = Join-Path $webroot 'bin'
    $environmentDllPath = Join-Path $webrootBinPath 'Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
    Add-Type -Path $environmentDllPath

    # Load application configuration
    $script:config = [Microsoft.Dynamics.ApplicationPlatform.Environment.EnvironmentFactory]::GetApplicationEnvironment()

    $script:target_sqlParams = @{
        'Database'       = $($config.DataAccess.Database)
        'UserName'       = $($config.DataAccess.SqlUser)
        'Password'       = $($config.DataAccess.SqlPwd)
        'ServerInstance' = $($config.DataAccess.DbServer)
        'Query'          = ''
        'QueryTimeout'   = 0 }


    $script:TenantId = $($config.Aad.AADTenantId)
    $script:AdminPrincipalName = $($config.Provisioning.AdminPrincipalName)
    $script:AdminIdentityProvider = $($config.Provisioning.AdminIdentityProvider)
    if (-not $AdminIdentityProvider) { $script:AdminIdentityProvider = 'https://sts.windows.net/' }

    if ($AdminPrincipalName -eq $null)
    {
        throw "Admin principal name in web.config is null so stopping restore of Admin account. Please check Admin account in DB manually and update the web.config for Admin principal name."
    }
}

function Invoke-SqlQueryWithLog($params)
{
    Write-LogWithoutEmail "Invoking SQL query:
$($params.Query)"
    try
    {
        return Invoke-SqlCmd @params -ErrorAction Stop
    }
    catch
    {
        Write-LogWithoutEmail "An exception occurred: $($_.Exception)"
    }
}

function Remove-DuplicateNetworkAliasAccounts($UserCount, $NetworkAlias)
{
    # Validate that Admin networkalias account is not duplicated.
    if ($UserCount -gt 1)
    {
        Write-LogWithoutEmail "Found $($UserCount) accounts with NetworkAlias '$($AdminPrincipalName)'. Removing duplicates (leaving only one record)..."

        # Remove duplicated records
        $target_sqlParams.Query = @"
            DELETE TOP($($UserCount-1)) FROM $($UserInfoTableName) WHERE NETWORKALIAS = '$AdminPrincipalName' 
"@
        Invoke-SqlQueryWithLog $target_sqlParams | out-null
        Write-LogWithoutEmail "Removed duplicated records for NetworkAlias '$($AdminPrincipalName)'."
    }
}

function Rename-AdminAccount()
{
    $PreviousAdminId = "OldAdmin{0:yyyyMMddHHmm}" -f [DateTime]::UtcNow

    # Update id of current admin account to different id
    $target_sqlParams.Query = @"
    UPDATE $($UserInfoTableName) set ID = '$PreviousAdminId' WHERE ID = 'Admin'
"@
    Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
    Write-LogWithoutEmail "Updated id of existing Admin account. New id is '$($PreviousAdminId)'."
}

function Set-AccountAsAdmin($NetworkAlias)
{
    Write-LogWithoutEmail "Updating ID (to 'Admin') and enabling account with NetworkAlias '$($NetworkAlias)'..."

    # Enable and update id of the user with networkalias of admin principal name to 'Admin' 
    $target_sqlParams.Query = @"
        UPDATE $($UserInfoTableName) set Enable = 1, ID = 'Admin' WHERE NETWORKALIAS = '$NetworkAlias'
"@
    Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
    Write-LogWithoutEmail "Updated id of account with NetworkAlias '$($NetworkAlias)'."
}

function Test-DoesNetworkAliasMatchTenantId($NetworkAlias)
{
    if (-not [string]::IsNullOrEmpty($NetworkAlias))
    {
        $NetworkAliasDomainSuffix = $NetworkAlias.SubString($NetworkAlias.IndexOf('@') + 1)

        # Validate that admin is the same provisioning admin and has the same domain as the tenant
        if ($NetworkAliasDomainSuffix -ne $TenantId)
        {
            Write-LogWithoutEmail "Admin in $($UserInfoTableName) has domain '$($NetworkAliasDomainSuffix)' in the NetworkAlias field, which doesn't match the 'Aad.AADTenantId' value in web.config: '$($TenantId)'."
        }
    }
}

function Write-LogWithoutEmail($LogMessage)
{
    # Regular expression from http://www.regular-expressions.info/email.html
    # Except we're not allowing ' to be the first character in the email address
    # since that messes with logging where we have a string like '<email address>' (single quotes included)
    # $1 in <email>@$1 is for the <Domain> capture group
    # meaning an email username@domain.com will be replaced with <email>@domain.com
    $LogMessageWithoutEmail = $LogMessage -replace "[a-z0-9!#\$%&*+/=?^_`{|}~-]+(?:\.[a-z0-9!#\$%&'*+/=?^_`{|}~-]+)*@(?<Domain>(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9]))?", '<email>@$1'
    Write-Log $LogMessageWithoutEmail
}

try
{
    Write-LogWithoutEmail "Restore Admin Account: Start"
    $error.Clear()

    Initialize-Settings

    Ensure-NoDuplicateAdminAccounts

    $adminUserCount = Get-UserCount -Id "Admin"
    if ($adminUserCount -eq 0)
    {
        Write-LogWithoutEmail "$($UserInfoTableName) has no Admin account. Need to create Admin account."
        Add-AdminAccount
    }
    else
    {
        Write-LogWithoutEmail "Validated that Admin account exists."
        
        $adminAccount = Get-AdminAccount
        $adminAccountNetworkAlias = $adminAccount.NETWORKALIAS
        Test-DoesNetworkAliasMatchTenantId -NetworkAlias $adminAccountNetworkAlias

        if ($adminAccountNetworkAlias -eq $AdminPrincipalName)
        {
            Write-LogWithoutEmail "Validated that admin network alias '$($adminAccountNetworkAlias)' matches the admin principal name '$($AdminPrincipalName)' in the config."
            Ensure-NoDuplicateNetworkAliasAccounts -NetworkAlias $AdminPrincipalName
        }
        else
        {
            Write-LogWithoutEmail "Admin network alias '$($adminAccountNetworkAlias)' in $($UserInfoTableName) is not the same as default provisioning admin principal name '$($AdminPrincipalName)'. Will update id of existing Admin account."
            Rename-AdminAccount

            $adminNetworkAliasUserCount = Get-UserCount -NetworkAlias $AdminPrincipalName

            # Validate if there is non admin account in USERINFO has same NETWORKALIAS with default provisioning admin principal name.
            if ($adminNetworkAliasUserCount -ge 1)
            {
                Remove-DuplicateNetworkAliasAccounts -UserCount $adminNetworkAliasUserCount -NetworkAlias $AdminPrincipalName
                Set-AccountAsAdmin -NetworkAlias $AdminPrincipalName
            }
            else
            {
                Write-LogWithoutEmail "No account found with NetworkAlias '$($AdminPrincipalName)'. Will create new Admin account record."
                Add-AdminAccount
            }
        }
    }

    $adminAccount = Get-AdminAccount
    Ensure-AdminAccountIsEnabled -AdminAccount $adminAccount
    Ensure-AdminAccountHasCorrectSID -AdminAccount $adminAccount
    Ensure-AdminAccountHasCorrectNetworkDomain -AdminAccount $adminAccount
}
Catch
{
    # Write the exception and that this process failed.  
    # Do not throw as this should not cause the deployment or update to fail.
    Write-Exception $_
    Write-LogWithoutEmail "Restore Admin Account failed, see $log for details."
}
Finally
{
    Write-LogWithoutEmail "Restore Admin Account: Stop"
}

# SIG # Begin signature block
# MIInygYJKoZIhvcNAQcCoIInuzCCJ7cCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBixKloG51Xld+P
# iHfYWRDWZcIge70qnkaDkV/xoMTj2KCCDYEwggX/MIID56ADAgECAhMzAAACzI61
# lqa90clOAAAAAALMMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjIwNTEyMjA0NjAxWhcNMjMwNTExMjA0NjAxWjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQCiTbHs68bADvNud97NzcdP0zh0mRr4VpDv68KobjQFybVAuVgiINf9aG2zQtWK
# No6+2X2Ix65KGcBXuZyEi0oBUAAGnIe5O5q/Y0Ij0WwDyMWaVad2Te4r1Eic3HWH
# UfiiNjF0ETHKg3qa7DCyUqwsR9q5SaXuHlYCwM+m59Nl3jKnYnKLLfzhl13wImV9
# DF8N76ANkRyK6BYoc9I6hHF2MCTQYWbQ4fXgzKhgzj4zeabWgfu+ZJCiFLkogvc0
# RVb0x3DtyxMbl/3e45Eu+sn/x6EVwbJZVvtQYcmdGF1yAYht+JnNmWwAxL8MgHMz
# xEcoY1Q1JtstiY3+u3ulGMvhAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUiLhHjTKWzIqVIp+sM2rOHH11rfQw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDcwNTI5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAeA8D
# sOAHS53MTIHYu8bbXrO6yQtRD6JfyMWeXaLu3Nc8PDnFc1efYq/F3MGx/aiwNbcs
# J2MU7BKNWTP5JQVBA2GNIeR3mScXqnOsv1XqXPvZeISDVWLaBQzceItdIwgo6B13
# vxlkkSYMvB0Dr3Yw7/W9U4Wk5K/RDOnIGvmKqKi3AwyxlV1mpefy729FKaWT7edB
# d3I4+hldMY8sdfDPjWRtJzjMjXZs41OUOwtHccPazjjC7KndzvZHx/0VWL8n0NT/
# 404vftnXKifMZkS4p2sB3oK+6kCcsyWsgS/3eYGw1Fe4MOnin1RhgrW1rHPODJTG
# AUOmW4wc3Q6KKr2zve7sMDZe9tfylonPwhk971rX8qGw6LkrGFv31IJeJSe/aUbG
# dUDPkbrABbVvPElgoj5eP3REqx5jdfkQw7tOdWkhn0jDUh2uQen9Atj3RkJyHuR0
# GUsJVMWFJdkIO/gFwzoOGlHNsmxvpANV86/1qgb1oZXdrURpzJp53MsDaBY/pxOc
# J0Cvg6uWs3kQWgKk5aBzvsX95BzdItHTpVMtVPW4q41XEvbFmUP1n6oL5rdNdrTM
# j/HXMRk1KCksax1Vxo3qv+13cCsZAaQNaIAvt5LvkshZkDZIP//0Hnq7NnWeYR3z
# 4oFiw9N2n3bb9baQWuWPswG0Dq9YT9kb+Cs4qIIwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIZnzCCGZsCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAsyOtZamvdHJTgAAAAACzDAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgJjvBas/0
# P1Jmru8VRxEBFLN6+mGZl1/3GCRpMdk4vv4wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQBo2qUOXDUEVbgz7IAQKHoSKXCiutH9gESBaibLdsEP
# 3mbPdFUgy2OJtSAw9HNbPLoa5Jj6Ravfl2UZLqkTgGM/WdvNLxM+T7vhTGLwTzHv
# MXXJaWmbdmO6zJ3bz96qNlTikzSBgyEQ7uKEkx04c5gnG1us1KEdjWjtHhkYGTNA
# PrvPRcA5rwWqgOXsT4vd5FPdXNYhi9V53McSgPqXXxz4iuHjGeJ/Wtp72fhkJDtd
# 5Z2EwDuBgtDRgD0BT0b3glRoEY0/2K6p+gUKPe/oVktJ842/lOwJ5cxFjStf0ILK
# yZleAftJ7bMVsrU0UtVzsE4prULWWawnUWN22cltnZvxoYIXKTCCFyUGCisGAQQB
# gjcDAwExghcVMIIXEQYJKoZIhvcNAQcCoIIXAjCCFv4CAQMxDzANBglghkgBZQME
# AgEFADCCAVkGCyqGSIb3DQEJEAEEoIIBSASCAUQwggFAAgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEICZwd5fyo8VpToakPtzQd37WZ9P3wIC+rXIu1/g0
# LwxlAgZjovLgAAQYEzIwMjMwMTA2MjE0NTE2LjAzNFowBIACAfSggdikgdUwgdIx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1p
# Y3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlvbnMgTGltaXRlZDEmMCQGA1UECxMdVGhh
# bGVzIFRTUyBFU046MkFENC00QjkyLUZBMDExJTAjBgNVBAMTHE1pY3Jvc29mdCBU
# aW1lLVN0YW1wIFNlcnZpY2WgghF4MIIHJzCCBQ+gAwIBAgITMwAAAbHKkEPuC/AD
# qwABAAABsTANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Eg
# MjAxMDAeFw0yMjA5MjAyMDIxNTlaFw0yMzEyMTQyMDIxNTlaMIHSMQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQg
# SXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxlcyBUU1Mg
# RVNOOjJBRDQtNEI5Mi1GQTAxMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFt
# cCBTZXJ2aWNlMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAhqKrPtXs
# G8fsg4w8R4MzZTAKkzwvEBQ94ntS+72rRGIMF0GCyEL9IOt7f9gkGoamfbtrtdY4
# y+KIFR8w19/nU3EoWhJfrYamrfpgtFmTaE3XCKCsI7rnrPmlVOMmndDyN1gAlfeu
# 4l5rdxx9ODECBPdS/+w/jDT7JkBhrYllqVXcwGAgWLdXAoUDgKVByv5XhKkbOrPx
# 9qppuZjKm4nflmfwb/bTWkA3aMMQ67tBoMLSsbIN3BJNWZdwczjoQVXo3YXr2fB+
# PYNmHviCcDUMHs0Vxmf7i/WSpBafsDMEn6WY7G8qtRGVX+7X0zDVg/7NVDLMqfn/
# iv++5hJGP+2Fmv4WZkBS1MBpwvOi4EQ25pIG45jWTffR4ynyed1I1SxSOP+efuBx
# 0WrN1A250lv5fGZHCL0vCMDT/w+U6wpNnxfDoQRY9Ut82iNK5alkxNozPP/DNI+n
# knTaSliaR2XnSXDIZEs7lfuJYg0qahfJJ1CZF2IYxOS9FK1crEigSb8QnEJoj6Th
# Lf4FYpYLTsRXlPdQbvBsVvgt++BttooznwfK0DKMOc718SLS+unwkVO0aF23CEQS
# Stoy0ZW34K+cbRmUfia+k9E+4luoTnT17oKqYfDNO5Rk8UwVa8mfh8+/R3fZaz2O
# /ZhiYT/RZHV9Quz5PHGlaCfXPQ8A6zFJlE8CAwEAAaOCAUkwggFFMB0GA1UdDgQW
# BBT0m2eR7w2thIr18WehUTSmvQ45kzAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJl
# pxtTNRnpcjBfBgNVHR8EWDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NybC9NaWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAx
# MCgxKS5jcmwwbAYIKwYBBQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3Rh
# bXAlMjBQQ0ElMjAyMDEwKDEpLmNydDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQM
# MAoGCCsGAQUFBwMIMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQsFAAOCAgEA
# 2Oc3kmql5VKEitAhoBCc1U6/VwMSYKQPqhC59f00Y5fbwnD+B2Qa0wnJqADSVVu6
# bBCVrks+EGbkuMhRb/lpiHNKVnuXF4PKTDnvCnYCqgwAmbttdxe0m38fJpGU3fmE
# CEFX4OYacEhFwTkLZtIUVjdqwPnQpRII+YqX/Q0Vp096g2puPllSdrxUB8xIOx3F
# 7LGOzyv/1WmrLyWAhUGpGte0W3qfX4YWkn7YCM+yl887tj5j+jO/l1MRi6bl4MsN
# 0PW2FCYeRbyzQEENsg5Pd351Z08ROR/nR8z+cAuQwR29ijaDKIms5IbRr1nZL/qZ
# skFSuCuSA+nYeMuTJxHg2HCXrt6ECFbEkYoPaBGTzxPYopcuJEcChhNlWkduCRgu
# ykEsmz0LvtmS7Fe68g4Zoh3sQkIE5VEwnKC3HwVemhK7eNYR1q7RYExfGFUDMQdO
# 7tQpbcPD4oaBbtFGWGu3nz1IryWs9K88zo8+eoQV/o9SxNU7Rs6TMqcLdM6C6Lgm
# GVaWKKC0S2DVKU8zFx0y5z25h1ZJ7X/Zhaav1mtXVG6+lJIq8ktJgOU5/pomumdf
# tgosxGjIp3NORy9fDUll+KQl4YmN9GzZxPYkhuI0QYriLmytBtUK+AK91hURVldV
# bUjP8sksr1dsiQwyOYQIkSxrTuhp0pw7h5329jphgEYwggdxMIIFWaADAgECAhMz
# AAAAFcXna54Cm0mZAAAAAAAVMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9v
# dCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0yMTA5MzAxODIyMjVaFw0z
# MDA5MzAxODMyMjVaMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIICIjAN
# BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA5OGmTOe0ciELeaLL1yR5vQ7VgtP9
# 7pwHB9KpbE51yMo1V/YBf2xK4OK9uT4XYDP/XE/HZveVU3Fa4n5KWv64NmeFRiMM
# tY0Tz3cywBAY6GB9alKDRLemjkZrBxTzxXb1hlDcwUTIcVxRMTegCjhuje3XD9gm
# U3w5YQJ6xKr9cmmvHaus9ja+NSZk2pg7uhp7M62AW36MEBydUv626GIl3GoPz130
# /o5Tz9bshVZN7928jaTjkY+yOSxRnOlwaQ3KNi1wjjHINSi947SHJMPgyY9+tVSP
# 3PoFVZhtaDuaRr3tpK56KTesy+uDRedGbsoy1cCGMFxPLOJiss254o2I5JasAUq7
# vnGpF1tnYN74kpEeHT39IM9zfUGaRnXNxF803RKJ1v2lIH1+/NmeRd+2ci/bfV+A
# utuqfjbsNkz2K26oElHovwUDo9Fzpk03dJQcNIIP8BDyt0cY7afomXw/TNuvXsLz
# 1dhzPUNOwTM5TI4CvEJoLhDqhFFG4tG9ahhaYQFzymeiXtcodgLiMxhy16cg8ML6
# EgrXY28MyTZki1ugpoMhXV8wdJGUlNi5UPkLiWHzNgY1GIRH29wb0f2y1BzFa/Zc
# UlFdEtsluq9QBXpsxREdcu+N+VLEhReTwDwV2xo3xwgVGD94q0W29R6HXtqPnhZy
# acaue7e3PmriLq0CAwEAAaOCAd0wggHZMBIGCSsGAQQBgjcVAQQFAgMBAAEwIwYJ
# KwYBBAGCNxUCBBYEFCqnUv5kxJq+gpE8RjUpzxD/LwTuMB0GA1UdDgQWBBSfpxVd
# AF5iXYP05dJlpxtTNRnpcjBcBgNVHSAEVTBTMFEGDCsGAQQBgjdMg30BATBBMD8G
# CCsGAQUFBwIBFjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3Mv
# UmVwb3NpdG9yeS5odG0wEwYDVR0lBAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQC
# BAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYD
# VR0jBBgwFoAU1fZWy4/oolxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZF
# aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9v
# Q2VyQXV0XzIwMTAtMDYtMjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcw
# AoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJB
# dXRfMjAxMC0wNi0yMy5jcnQwDQYJKoZIhvcNAQELBQADggIBAJ1VffwqreEsH2cB
# MSRb4Z5yS/ypb+pcFLY+TkdkeLEGk5c9MTO1OdfCcTY/2mRsfNB1OW27DzHkwo/7
# bNGhlBgi7ulmZzpTTd2YurYeeNg2LpypglYAA7AFvonoaeC6Ce5732pvvinLbtg/
# SHUB2RjebYIM9W0jVOR4U3UkV7ndn/OOPcbzaN9l9qRWqveVtihVJ9AkvUCgvxm2
# EhIRXT0n4ECWOKz3+SmJw7wXsFSFQrP8DJ6LGYnn8AtqgcKBGUIZUnWKNsIdw2Fz
# Lixre24/LAl4FOmRsqlb30mjdAy87JGA0j3mSj5mO0+7hvoyGtmW9I/2kQH2zsZ0
# /fZMcm8Qq3UwxTSwethQ/gpY3UA8x1RtnWN0SCyxTkctwRQEcb9k+SS+c23Kjgm9
# swFXSVRk2XPXfx5bRAGOWhmRaw2fpCjcZxkoJLo4S5pu+yFUa2pFEUep8beuyOiJ
# Xk+d0tBMdrVXVAmxaQFEfnyhYWxz/gq77EFmPWn9y8FBSX5+k77L+DvktxW/tM4+
# pTFRhLy/AsGConsXHRWJjXD+57XQKBqJC4822rpM+Zv/Cuk0+CQ1ZyvgDbjmjJnW
# 4SLq8CdCPSWU5nR0W2rRnj7tfqAxM328y+l7vzhwRNGQ8cirOoo6CGJ/2XBjU02N
# 7oJtpQUQwXEGahC0HVUzWLOhcGbyoYIC1DCCAj0CAQEwggEAoYHYpIHVMIHSMQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNy
# b3NvZnQgSXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxl
# cyBUU1MgRVNOOjJBRDQtNEI5Mi1GQTAxMSUwIwYDVQQDExxNaWNyb3NvZnQgVGlt
# ZS1TdGFtcCBTZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQDtZLG+pANsDu/LLr1OfTA/
# kEbHK6CBgzCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMA0GCSqG
# SIb3DQEBBQUAAgUA52KJEzAiGA8yMDIzMDEwNjE5NDgzNVoYDzIwMjMwMTA3MTk0
# ODM1WjB0MDoGCisGAQQBhFkKBAExLDAqMAoCBQDnYokTAgEAMAcCAQACAhcDMAcC
# AQACAhFdMAoCBQDnY9qTAgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkK
# AwKgCjAIAgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZIhvcNAQEFBQADgYEAP/yr
# aCX/nbJyn6MIe3nph2E2NT9EPNQimMvZNtcSU6f0Hv9jogaTo3Q75RqFh+LehDyq
# /KMae0KCgK1v+iHGXmxJDBBNLyg6mYrbtE5SUv2JxV1JIwcRlooA4xK1oN01BaVV
# zsbrwo/bsECq1tWstp3jwWf/qx6xpdCytniXnV4xggQNMIIECQIBATCBkzB8MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNy
# b3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAbHKkEPuC/ADqwABAAABsTAN
# BglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMC8G
# CSqGSIb3DQEJBDEiBCAW9dJ/Pyl4jTdk915xCP9fNj+v7s/fPI6N6RWzHsOXnjCB
# +gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EIIPtDYsUW9+p4OjL2Cm7fm3p1h6u
# sM7RwxOU4iibNM9sMIGYMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIw
# MTACEzMAAAGxypBD7gvwA6sAAQAAAbEwIgQg89YlwcloSiUvTvJ5wLMxL43BmI/H
# 4WPUPnxoqPYXF7QwDQYJKoZIhvcNAQELBQAEggIAOFn+CpTeuQl33k8heE7iHN4/
# Uo0GsHif9Wlgj2WgDkQmN8+qIBV1Kf/EKvBz+sIf8qyQqbjhUCbih54QjWZuL4s4
# lI8Q+t0aYcdIfTETpzjF2lW3wAuw7AfZOgsVmpHkqLdPtdlMfhL1hZrapYZ6HzG2
# oZ4kXDOlHeVPOzBueOOH1vyoR2JAgHOEUY7C2yxs8yU3uSU5iNaA/3snPdpjOVI7
# S/cyEXgWX9mfiybSWe3dok1g51KOJmji8eqCw5LY4osxIDrB4XahAPK6SHvPtrJR
# adYvFYMSUSg7TdAqI7jcJGh2diPaoKbjLpy0tQlyltohDRPx8EH1sUHbiR2XENdH
# 1bGiyK9/Gfuk5GuoTnTOpLZvttCikQPCMmqxqa/JzRq1H5BhP39yAQ+R5Hx60JHH
# bhb69ZA83qQy9pHrMf68RC8QKGhLnTVs1gfMocZ8ryRmsyFx0EGbncHA/i0MMfrS
# U66qisdfNuxTAR7blfGKdkKKrzDeBch3WmHHdxj5dDrjIB2GOpUWOWqwld7Sx6nE
# JDd8yrAmDgCpPPwBpngRVtsF48Nulf3hw+EOvzozOtaRJfKzxLUczWSv4XCJ/MU5
# aTak4lRlfWsiH/QLlmFFrX6VfLw+XQtRj6Y2MhmFDKp1SPhaONCLL+krDNFLBkEn
# xpbIl5+lS8gDu81wK0Y=
# SIG # End signature block
