﻿<#
.SYNOPSIS
    
Provides operations and functions for the retrieval of KeyVault Keys and Passwords as
well as some other utilities for manipulating the data that is returned.

.DESCRIPTION

Provides a common module for Unified Operations Setup to retrieve and manipulate Azure Secret/Keys.

.PARAMETER VaultName

The Name of the KeyVault you want to retrieve the key from. Azure specifies that a 
KeyVault name must match this regex "^[a-zA-Z0-9-]{3,24}$".

.PARAMETER ApplicationID

The ApplicationID of the ServicePrincipal we are going to use to authenticate with.

.PARAMETER TenantID

The TenantID guid that Application is under.

.PARAMETER CertificateThumprint

The Thumprint of the certificate that is associated with the Application.

.EXAMPLE

C:\PS> Import-Module ".\KeyVault.psm1" -ArgumentList ("<KeyVaultName>", "<ApplicationID>", "<TenantID>", "<Thumbprint>")

#>
[CmdletBinding()]
param
(
    [Parameter(Mandatory=$False, Position = 0)]
    [System.String]
    $VaultName,

    [Parameter(Mandatory=$False, Position = 1)]
    [System.String]
    $ApplicationID,

    [Parameter(Mandatory=$False, Position = 2)]
    [System.String]
    $CertificateThumprint
)

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking

if (![System.String]::IsNullOrWhiteSpace($VaultName))
{
    $Script:VaultName = $VaultName
}
else
{
    $Script:VaultName = "default"
}

if (![System.String]::IsNullOrWhiteSpace($ApplicationID) -and 
    ![System.String]::IsNullOrWhiteSpace($CertificateThumprint))
{
    $Script:ApplicationID = $ApplicationID
    $Script:CertificateThumprint = $CertificateThumprint 
    Import-Module "$PSScriptRoot\DynamicsVault.psm1" -Force -DisableNameChecking
}

function Get-KeyVaultKey
{
    <#
    .SYNOPSIS
    
    Retrieves a specified Key from a KeyVault.

    .DESCRIPTION

    Retrieves a specified Key from KeyVault using the Azure PowerShell Modules and wraps
    the function up to provide an easier call. 

    .PARAMETER Name

    The Name of the Key in the selected vault we want to retrieve.

    .PARAMETER Version

    The Version of the Key we want, this is an optional field. The default Version
    is the latest one in KeyVault.

    .PARAMETER VaultName

    The Name of the KeyVault you want to retrieve the key from. This parameter is optional,
    it will default to the $Script:VaultName parameter. Azure specifies that a KeyVault name
    must match this regex "^[a-zA-Z0-9-]{3,24}$".

    .EXAMPLE

    Retrieves the latest version of the Key.
    
    C:\PS> Get-KeyVaultKey "KeyName"

    .EXAMPLE
    
    Retrieves the specific version of the Key.

    C:\PS> Get-KeyVaultKey "KeyName" "KeyVersion"

    .EXAMPLE

    Retrieves the latest version of the Key from a different KeyVault than what was passed
    from the script parameters.

    C:\PS> Get-KeyVaultKey -Name "KeyName" -VaultName "VaultName"
    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateLength(1,127)]
        [System.String]
        $Name,

        [Parameter(ParameterSetName = "Version", Mandatory=$False, Position = 1)]
        [System.String]
        $Version,

        [Parameter(Mandatory=$False, Position = 2)]
        [ValidatePattern("^[a-zA-Z0-9-]{3,24}$")]
        [System.String]
        $VaultName = $Script:VaultName
    )


    switch ($PsCmdlet.ParameterSetName)
    {
        "Version" 
        { 
            $keyInformation = Get-AzureKeyVaultKey -VaultName $VaultName -Name $Name -Version $Version
        }
        default
        {
            $keyInformation = Get-AzureKeyVaultKey -VaultName $VaultName -Name $Name
        }
    }
    return $keyInformation
}

function Get-KeyVaultSecret
{
    <#
    .SYNOPSIS
    
    Retrieves a specified secret from a KeyVault.

    .DESCRIPTION

    Retrieves a specified secret from KeyVault using the Azure PowerShell Modules and wraps
    the function up to provide an easier call. 

    .PARAMETER VaultUri

    The Name of the secret in the selected vault we want to retrieve. The assumption is the
    URI is formatted as such "Vault://SecretName/SecretVersion", if it does not match this
    pattern we assume the secret is the VaultUri parameter.

    .PARAMETER VaultName

    The Name of the KeyVault you want to retrieve the secret from. This parameter is optional,
    it will default to the $Script:VaultName parameter. Azure specifies that a KeyVault name
    must match this regex "^[a-zA-Z0-9-]{3,24}$".

    .EXAMPLE

    Retrieves the latest version of the secret.
    
    C:\PS> Get-KeyVaultSecret "VaultUri"

    .EXAMPLE

    Retrieves the latest version of the secret from a different KeyVault than what was passed
    from the script parameters.

    C:\PS> Get-KeyVaultSecret -VaultUri "Vault://SecretName/SecretVersion" -VaultName "VaultName"
    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [System.String]
        $VaultUri,

        [Parameter(Mandatory=$False, Position = 1)]
        [ValidatePattern("^[a-zA-Z0-9-]{3,24}$")]
        [System.String]
        $VaultName = $Script:VaultName
    )

    if (!(Test-ValidKeyVaultUri -VaultUri $VaultUri))
    {
        return $VaultUri
    }
    else
    {
        [System.Uri]$Uri = [System.Uri]::new($VaultUri)

        $secretName = $Uri.Host

        if ($Uri.Segments.Count -ge 2)
        {
            $secret = Get-VaultCredential -ClientId $Script:ApplicationID -CertificateThumbprint $Script:CertificateThumprint -VaultName $VaultName -SecretName $secretName -Version $Uri.Segments[1]
        }
        else
        {
            $secret = Get-VaultCredential -ClientId $Script:ApplicationID -CertificateThumbprint $Script:CertificateThumprint -VaultName $VaultName -SecretName $secretName
        }

        return $secret
    }
}

function Get-CertificatefromBase64String
{
    <#
    .SYNOPSIS

    Takes in a Base64 String representation of a Private Certificate (PFX) and the
    password associated with it and loads the certificate into a X509Certificate2 object.

    .DESCRIPTION

    Upon receiving the certificate information from KeyVault this allows us to convert
    the Base64 string and the password and load a certificate into a X509Certificate2 object.

    .PARAMETER Base64String

    The Base64 string representation of the PFX file.

    .PARAMETER CertificatePassword

    The password of the PFX Certificate.

    .EXAMPLE

    C:\PS> Create-CertificatefromBase64String "<Base64String>" "<Password>"

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $Base64String,

        [Parameter(Mandatory=$False, Position = 1)]
        [System.Security.SecureString]
        $CertificatePassword
    )

    $certificateBytes = [System.Convert]::FromBase64String($Base64String)

    if ($CertificatePassword -ne $null)
    { 
        return [System.Security.Cryptography.X509Certificates.X509Certificate2]::new($CertificateBytes, $CertificatePassword)       
    }
    else
    {
        return [System.Security.Cryptography.X509Certificates.X509Certificate2]::new($CertificateBytes)
    }
}

function Get-CertificateFromJsonString
{
    <#
    .SYNOPSIS

    Takes in a JSON String representation of a Certificate and attributes 
    and loads the certificate in memory.

    .DESCRIPTION

    This functions makes the assumption that the JSON file is formatted like this.

    {
        "data": "<Base64String>",
        "dataType" :"pfx",
        "password": "<CertificatePassword>"
    }

    This is following the same format that the ARM Templates expect Certificates for 
    VM Deployments.

    .PARAMETER JsonString

    The Json String string representation of the Certificate file and its attributes.

    .PARAMETER CertificateType

    This is a reference object so that the caller of the function can determine the
    type of the certificate for example (Public or Private key).

    .EXAMPLE

    C:\PS> [System.String]$certificateType = [System.String]::Empty
    C:\PS> $jsonObject = @"
    C:\PS> {
    C:\PS> "data": "<Base64EncodedCert>",
    C:\PS> "dataType" :"pfx",
    C:\PS> "password": "<CertificatePassword>"
    C:\PS> }
    C:\PS> Get-CertificateFromJsonString $jsonObject ([ref]$certificateType)

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $JsonString,

        [Parameter(Mandatory=$True, Position = 1)]
        [Ref]
        $CertificateType
    )

    $jsonObject = $JsonString | ConvertFrom-Json

    $CertificateType.Value = $jsonObject.dataType
    [System.Security.SecureString]$certificatePassword = $null

    if (![System.String]::IsNullOrWhiteSpace($jsonObject.password))
    {
        $certificatePassword = ConvertTo-SecureString $jsonObject.password -AsPlainText -Force
    }

    $certificate = Get-CertificatefromBase64String -Base64String $jsonObject.data -CertificatePassword $certificatePassword

    return $certificate
}

function Get-CertificateFromVault
{
    <#
    .SYNOPSIS

    Takes in two Vault Uri's. One is for the Certificate file (base64 encoded) and the
    second the password for the Certificate (Optional).

    .DESCRIPTION

    Once the information for the certificate has been retrieved we take the information and 
    create a x509Certificate2 object to be consumed.

    .PARAMETER CertUri

    The Uri of the certificate file we want.

    .PARAMETER CertPasswordUri

    The Uri of the certificate password that corresponds to the Certificate

    .PARAMETER VaultName

    The name of the Vault to retrieve the secret information.

    .EXAMPLE

    # Uses default VaultName
    C:\PS> Get-CertificateFromVault "vault://certSecretName" "vault://certSecretPasswordName"

    # overrides default VaultName
    C:\PS> Get-CertificateFromVault "vault://certSecretName" "vault://certSecretPasswordName" "VaultName"

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $CertUri,

        [Parameter(Mandatory=$false, Position = 1)]
        [System.String]
        $CertPasswordUri,

        [Parameter(Mandatory=$False, Position = 2)]
        [System.String]
        $VaultName = $Script:VaultName
    )

    $base64Cert = Get-KeyVaultSecret -VaultUri $CertUri -VaultName $VaultName

    if (![System.String]::IsNullOrWhiteSpace($CertPasswordUri))
    {
        $certificatePassword = ConvertTo-SecureString -String $(Get-KeyVaultSecret -VaultUri $CertPasswordUri -VaultName $VaultName) -AsPlainText -Force
    }

    return Get-CertificatefromBase64String -Base64String $base64Cert -CertificatePassword $certificatePassword
}

function Get-Base64UTF8String
{
    <#
    .SYNOPSIS

    Takes in a UTF8 Encoded Base64 String and converts it to native string.

    .DESCRIPTION

    This is a helper function to unwrap the Private Key that is stored for the
    ARM Template deployment.

    Azure requires this JSON Structure to be wrapped like this 

    $JsonBlob = @{
        "data": "<Base64String>",
        "dataType" :"pfx",
        "password": "<CertificatePassword>"
    }@

    $ContentBytes = [System.Text.Encoding]::UTF8.GetBytes($JSONBlob)
    $Content = [System.Convert]::ToBase64String($ContentBytes)

    This will unwrap the JSON File.

    .PARAMETER Base64String

    The UTF8 Base64String Object to unwrap.

    .EXAMPLE

    C:\PS> Read-Base64UTF8String <UTF8 Encoded Base64String>

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $Base64String
    )

    $ReceivedBytes = [System.Convert]::FromBase64String($Base64String)

    return [System.Text.Encoding]::UTF8.GetString($ReceivedBytes)
}

function Create-FilefromBase64String
{
    <#
    .SYNOPSIS
    
    Takes in a Base64 String representation of a File and
    Creates a file to the specified path.
    
    .DESCRIPTION

    Takes in a Base64 String representation of a File and
    Creates a file to the specified path.

    .PARAMETER Base64String

    The Base64 string representation of the PFX file. 

    .PARAMETER DestinationPath

    The Path of the file we want to save too. This includes the path and 
    file name i.e. (C:\Temp\Cert.pfx)

    .EXAMPLE

    C:\PS> Create-FilefromBase64String "<Base64String>" "C:\Temp\Cert.pfx"

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $Base64String,

        [Parameter(Mandatory=$True, Position = 1)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $DestinationPath
    )
    
    $pfxBytes = [System.Convert]::FromBase64String($Base64String)
    
    [System.IO.File]::WriteAllBytes($DestinationPath, $pfxBytes)
}

function Test-ValidKeyVaultUri
{
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [System.String]
        $VaultUri
    )

    if (![System.Uri]::IsWellFormedUriString($VaultUri, [System.UriKind]::Absolute))
    {
        return $false
    }

    [System.Uri]$Uri = [System.Uri]::new($VaultUri)

    if ($Uri.Scheme -ne "vault")
    {
        return $false
    }

    return $true
}
# SIG # Begin signature block
# MIInpQYJKoZIhvcNAQcCoIInljCCJ5ICAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCYIEPfj2Yy48kJ
# gWEKNntz2Qhws7AgUPUErSkZHt+v0qCCDYUwggYDMIID66ADAgECAhMzAAACzfNk
# v/jUTF1RAAAAAALNMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjIwNTEyMjA0NjAyWhcNMjMwNTExMjA0NjAyWjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDrIzsY62MmKrzergm7Ucnu+DuSHdgzRZVCIGi9CalFrhwtiK+3FIDzlOYbs/zz
# HwuLC3hir55wVgHoaC4liQwQ60wVyR17EZPa4BQ28C5ARlxqftdp3H8RrXWbVyvQ
# aUnBQVZM73XDyGV1oUPZGHGWtgdqtBUd60VjnFPICSf8pnFiit6hvSxH5IVWI0iO
# nfqdXYoPWUtVUMmVqW1yBX0NtbQlSHIU6hlPvo9/uqKvkjFUFA2LbC9AWQbJmH+1
# uM0l4nDSKfCqccvdI5l3zjEk9yUSUmh1IQhDFn+5SL2JmnCF0jZEZ4f5HE7ykDP+
# oiA3Q+fhKCseg+0aEHi+DRPZAgMBAAGjggGCMIIBfjAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQU0WymH4CP7s1+yQktEwbcLQuR9Zww
# VAYDVR0RBE0wS6RJMEcxLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9wZXJh
# dGlvbnMgTGltaXRlZDEWMBQGA1UEBRMNMjMwMDEyKzQ3MDUzMDAfBgNVHSMEGDAW
# gBRIbmTlUAXTgqoXNzcitW2oynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8v
# d3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9NaWNDb2RTaWdQQ0EyMDExXzIw
# MTEtMDctMDguY3JsMGEGCCsGAQUFBwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDov
# L3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNDb2RTaWdQQ0EyMDEx
# XzIwMTEtMDctMDguY3J0MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQELBQADggIB
# AE7LSuuNObCBWYuttxJAgilXJ92GpyV/fTiyXHZ/9LbzXs/MfKnPwRydlmA2ak0r
# GWLDFh89zAWHFI8t9JLwpd/VRoVE3+WyzTIskdbBnHbf1yjo/+0tpHlnroFJdcDS
# MIsH+T7z3ClY+6WnjSTetpg1Y/pLOLXZpZjYeXQiFwo9G5lzUcSd8YVQNPQAGICl
# 2JRSaCNlzAdIFCF5PNKoXbJtEqDcPZ8oDrM9KdO7TqUE5VqeBe6DggY1sZYnQD+/
# LWlz5D0wCriNgGQ/TWWexMwwnEqlIwfkIcNFxo0QND/6Ya9DTAUykk2SKGSPt0kL
# tHxNEn2GJvcNtfohVY/b0tuyF05eXE3cdtYZbeGoU1xQixPZAlTdtLmeFNly82uB
# VbybAZ4Ut18F//UrugVQ9UUdK1uYmc+2SdRQQCccKwXGOuYgZ1ULW2u5PyfWxzo4
# BR++53OB/tZXQpz4OkgBZeqs9YaYLFfKRlQHVtmQghFHzB5v/WFonxDVlvPxy2go
# a0u9Z+ZlIpvooZRvm6OtXxdAjMBcWBAsnBRr/Oj5s356EDdf2l/sLwLFYE61t+ME
# iNYdy0pXL6gN3DxTVf2qjJxXFkFfjjTisndudHsguEMk8mEtnvwo9fOSKT6oRHhM
# 9sZ4HTg/TTMjUljmN3mBYWAWI5ExdC1inuog0xrKmOWVMIIHejCCBWKgAwIBAgIK
# YQ6Q0gAAAAAAAzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNV
# BAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
# c29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlm
# aWNhdGUgQXV0aG9yaXR5IDIwMTEwHhcNMTEwNzA4MjA1OTA5WhcNMjYwNzA4MjEw
# OTA5WjB+MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
# BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYD
# VQQDEx9NaWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG
# 9w0BAQEFAAOCAg8AMIICCgKCAgEAq/D6chAcLq3YbqqCEE00uvK2WCGfQhsqa+la
# UKq4BjgaBEm6f8MMHt03a8YS2AvwOMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc
# 6Whe0t+bU7IKLMOv2akrrnoJr9eWWcpgGgXpZnboMlImEi/nqwhQz7NEt13YxC4D
# dato88tt8zpcoRb0RrrgOGSsbmQ1eKagYw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+
# lD3v++MrWhAfTVYoonpy4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+EGvKhL1nk
# kDstrjNYxbc+/jLTswM9sbKvkjh+0p2ALPVOVpEhNSXDOW5kf1O6nA+tGSOEy/S6
# A4aN91/w0FK/jJSHvMAhdCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rhKEmd
# X4jiJV3TIUs+UsS1Vz8kA/DRelsv1SPjcF0PUUZ3s/gA4bysAoJf28AVs70b1FVL
# 5zmhD+kjSbwYuER8ReTBw3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zd
# sGbiwZeBe+3W7UvnSSmnEyimp31ngOaKYnhfsi+E11ecXL93KCjx7W3DKI8sj0A3
# T8HhhUSJxAlMxdSlQy90lfdu+HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS
# 4NaIjAsCAwEAAaOCAe0wggHpMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBRI
# bmTlUAXTgqoXNzcitW2oynUClTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTAL
# BgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBRyLToCMZBD
# uRQFTuHqp8cx0SOJNDBaBgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFf
# MDNfMjIuY3JsMF4GCCsGAQUFBwEBBFIwUDBOBggrBgEFBQcwAoZCaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFf
# MDNfMjIuY3J0MIGfBgNVHSAEgZcwgZQwgZEGCSsGAQQBgjcuAzCBgzA/BggrBgEF
# BQcCARYzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1h
# cnljcHMuaHRtMEAGCCsGAQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAHAAbwBsAGkA
# YwB5AF8AcwB0AGEAdABlAG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn
# 8oalmOBUeRou09h0ZyKbC5YR4WOSmUKWfdJ5DJDBZV8uLD74w3LRbYP+vj/oCso7
# v0epo/Np22O/IjWll11lhJB9i0ZQVdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0b
# pdS1HXeUOeLpZMlEPXh6I/MTfaaQdION9MsmAkYqwooQu6SpBQyb7Wj6aC6VoCo/
# KmtYSWMfCWluWpiW5IP0wI/zRive/DvQvTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvy
# CInWH8MyGOLwxS3OW560STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn+N4sOiBp
# mLJZiWhub6e3dMNABQamASooPoI/E01mC8CzTfXhj38cbxV9Rad25UAqZaPDXVJi
# hsMdYzaXht/a8/jyFqGaJ+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw3MYb
# BL7fQccOKO7eZS/sl/ahXJbYANahRr1Z85elCUtIEJmAH9AAKcWxm6U/RXceNcbS
# oqKfenoi+kiVH6v7RyOA9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sL
# gOppO6/8MO0ETI7f33VtY5E90Z1WTk+/gFcioXgRMiF670EKsT/7qMykXcGhiJtX
# cVZOSEXAQsmbdlsKgEhr/Xmfwb1tbWrJUnMTDXpQzTGCGXYwghlyAgEBMIGVMH4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01p
# Y3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTECEzMAAALN82S/+NRMXVEAAAAA
# As0wDQYJYIZIAWUDBAIBBQCgga4wGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQw
# HAYKKwYBBAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEILxv
# lgb5ANeIjvZsgvMjlnNEfIzVfFWaawyJ/VKEMb0gMEIGCisGAQQBgjcCAQwxNDAy
# oBSAEgBNAGkAYwByAG8AcwBvAGYAdKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20wDQYJKoZIhvcNAQEBBQAEggEADkNyial4zV2DL8W/NcNiA8zH9rLPGeR+hLJs
# rl7tCzpq+xUP4nj84q1IL6tct2gxPo7skB1wOpXMSsaZQyZMb/p174C7HfTxQkeg
# ZBh2bWuVWkxUfe34wDVew4jjuQTS/86BG4vb9k3VhH8e3Lx1OcyK82y5PVkij5rn
# D+PlZtBOb4FU3GAZfotWlsUN0yZOvBHfk5RW42EzQLDX0XGlaLRJJFNswRuXMEwv
# 8lv24K7AIDbKY2E6DCOivGP0YnuWbXqMQy85Z8+SuM7pqgZAzZiHtxjktibHlfxE
# S4LBn7RIfmCeF+yqe+2kVHn+k6DrgF2SVbnyj+waGIXokU0biqGCFwAwghb8Bgor
# BgEEAYI3AwMBMYIW7DCCFugGCSqGSIb3DQEHAqCCFtkwghbVAgEDMQ8wDQYJYIZI
# AWUDBAIBBQAwggFRBgsqhkiG9w0BCRABBKCCAUAEggE8MIIBOAIBAQYKKwYBBAGE
# WQoDATAxMA0GCWCGSAFlAwQCAQUABCCM9V9um0YZo+iThayfCe6dVRxKsl/p9AKQ
# pDgthItWeAIGZBMMvs9NGBMyMDIzMDMyNTA2MzkzNy40NjlaMASAAgH0oIHQpIHN
# MIHKMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQL
# ExxNaWNyb3NvZnQgQW1lcmljYSBPcGVyYXRpb25zMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjozQkJELUUzMzgtRTlBMTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCEVcwggcMMIIE9KADAgECAhMzAAABxjDNLtbTocD0AAEA
# AAHGMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTIyMTEwNDE5MDEzNFoXDTI0MDIwMjE5MDEzNFowgcoxCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJTAjBgNVBAsTHE1pY3Jvc29mdCBBbWVy
# aWNhIE9wZXJhdGlvbnMxJjAkBgNVBAsTHVRoYWxlcyBUU1MgRVNOOjNCQkQtRTMz
# OC1FOUExMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNlMIIC
# IjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA770iOr6v4Hk1m3SZj+1BR/iu
# ofv+f6eVb7Hc21YxzAzro4G6kKXF47YAsEgrWWT1ogvp0IroFm8CdRZTf/DlQ0fb
# NNO9pCA01KJ03zH82Clmk9ze9r1jPJ1ZJaMnsZmAy7VpY9mNqX9dhPvnW1/Zxbbi
# Hv7qwwgw9U2ST5mfcpPutsI/Qr/gLC6aTI3UCYziVPZ/Qfag8NQhKkpHZO3Kr5r8
# 3cy7jz4OWPy5M2WitWv5bJJ5rBTW518QPEzFwzq8e8P722CWKZJFjN8etBgsK05g
# HeHaN9kmlpYJJL84v9JiaX7NFJkORhApEFZiUIaZoLxJt4pcBDzf+WD9UAjRKCrA
# seJ/ckzQvOn95X4Ot4asnOuNhcCdcQWcrZoykFmEaoYkrsD7n/4nFFHwJDKUaBYZ
# ZLwPj7ux48S1Ye+cMccMxdRSjuoG4rqJqpEd6gzfz239v36L+LtOlQhfL5cnclhN
# SWdmKw1THyekH96RNtOpIE7c+9Tnsv1aE9hphejLOJxfsXTkyw0FIdilc0CP7zzU
# sqaCGF2mFXRwL4jfX1RyV7QGKEeOpvGZqQKLHVZbLD32ztW8Lfv99xQ24d/TIxO9
# LReuHsnwotn+8BsCrzu+/24vOcA9Xcpd7kIeW4zmUxhFsv4KCfiqMptZhhMAA0Sq
# Duj23cj10smXROLUnhUCAwEAAaOCATYwggEyMB0GA1UdDgQWBBRxX/lHiShECp1n
# 2lMa6G1uLvNglDAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJlpxtTNRnpcjBfBgNV
# HR8EWDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2Ny
# bC9NaWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcmwwbAYI
# KwYBBQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0ElMjAy
# MDEwKDEpLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
# CSqGSIb3DQEBCwUAA4ICAQALzF724jXugDU486PMBYR7Cc6aHr2nJDnGsnkqwZYm
# jRi28qv6S1Ly772zwm5DI189zgAQ99xhEjW6uYkrX5UrtVu7JUQP6bBBBJ98c28F
# AIPIK3fkQNCv2rxRjfQMJSdcwsJeTK7Ld09OuA5hY4PWCBgJpfY71LcaXz3FR8AN
# PFX6zcKYbgYOZregtpDub34N0QhR7wc/FcmV+g4I3IdTAoMD2/WI5ZsfKTzBUn/U
# 3ApUhNwbOl5YSC+f9S1LStbZLwPzMS+fLPXJUSe7SSvspfSsr/VEe0oQhmaR+5vc
# q+7MLw861WBVhYpJ7TB5YBS5ORO9XdIbcpbBFwcHPmb8iZqSIqW9JpgG76+5NQUL
# PVzZ75z5W2R5ZiyQktiHpMwjX2OO29Z8+nTw2tOsVCcwzH9LoELedv3PjcpbwOyL
# jtm1T4XHYd3qbd9DXoBjNYkSjdi37pNp58u+rITltLKOjjQCJwj1FpnuBY825B5C
# 0uC/NYESEKsTicEjhS/4ujBXLcNGDhVBl2vHE6qY/YW4ky1vcypvUrsG81gpv2+8
# /ihOwg4wTLO7XqikeIiU3ZWAUAoOpTl14tedQqxbHTDveJYR3OU0yKB2xwf87EWC
# Ab0CJimhDmyQaKEvSV0fLW9iVyI0wYcG4V2aVN6TrZ4mr+ffaqDQD9F+HpPhP0pl
# AzCCB3EwggVZoAMCAQICEzMAAAAVxedrngKbSZkAAAAAABUwDQYJKoZIhvcNAQEL
# BQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
# EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNV
# BAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eSAyMDEwMB4X
# DTIxMDkzMDE4MjIyNVoXDTMwMDkzMDE4MzIyNVowfDELMAkGA1UEBhMCVVMxEzAR
# BgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
# Y3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3Rh
# bXAgUENBIDIwMTAwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDk4aZM
# 57RyIQt5osvXJHm9DtWC0/3unAcH0qlsTnXIyjVX9gF/bErg4r25PhdgM/9cT8dm
# 95VTcVrifkpa/rg2Z4VGIwy1jRPPdzLAEBjoYH1qUoNEt6aORmsHFPPFdvWGUNzB
# RMhxXFExN6AKOG6N7dcP2CZTfDlhAnrEqv1yaa8dq6z2Nr41JmTamDu6GnszrYBb
# fowQHJ1S/rboYiXcag/PXfT+jlPP1uyFVk3v3byNpOORj7I5LFGc6XBpDco2LXCO
# Mcg1KL3jtIckw+DJj361VI/c+gVVmG1oO5pGve2krnopN6zL64NF50ZuyjLVwIYw
# XE8s4mKyzbnijYjklqwBSru+cakXW2dg3viSkR4dPf0gz3N9QZpGdc3EXzTdEonW
# /aUgfX782Z5F37ZyL9t9X4C626p+Nuw2TPYrbqgSUei/BQOj0XOmTTd0lBw0gg/w
# EPK3Rxjtp+iZfD9M269ewvPV2HM9Q07BMzlMjgK8QmguEOqEUUbi0b1qGFphAXPK
# Z6Je1yh2AuIzGHLXpyDwwvoSCtdjbwzJNmSLW6CmgyFdXzB0kZSU2LlQ+QuJYfM2
# BjUYhEfb3BvR/bLUHMVr9lxSUV0S2yW6r1AFemzFER1y7435UsSFF5PAPBXbGjfH
# CBUYP3irRbb1Hode2o+eFnJpxq57t7c+auIurQIDAQABo4IB3TCCAdkwEgYJKwYB
# BAGCNxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQUKqdS/mTEmr6CkTxGNSnPEP8v
# BO4wHQYDVR0OBBYEFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMFwGA1UdIARVMFMwUQYM
# KwYBBAGCN0yDfQEBMEEwPwYIKwYBBQUHAgEWM2h0dHA6Ly93d3cubWljcm9zb2Z0
# LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5Lmh0bTATBgNVHSUEDDAKBggrBgEF
# BQcDCDAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYD
# VR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvXzpoYxDBW
# BgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
# bC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYIKwYBBQUH
# AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtp
# L2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDANBgkqhkiG9w0BAQsF
# AAOCAgEAnVV9/Cqt4SwfZwExJFvhnnJL/Klv6lwUtj5OR2R4sQaTlz0xM7U518Jx
# Nj/aZGx80HU5bbsPMeTCj/ts0aGUGCLu6WZnOlNN3Zi6th542DYunKmCVgADsAW+
# iehp4LoJ7nvfam++Kctu2D9IdQHZGN5tggz1bSNU5HhTdSRXud2f8449xvNo32X2
# pFaq95W2KFUn0CS9QKC/GbYSEhFdPSfgQJY4rPf5KYnDvBewVIVCs/wMnosZiefw
# C2qBwoEZQhlSdYo2wh3DYXMuLGt7bj8sCXgU6ZGyqVvfSaN0DLzskYDSPeZKPmY7
# T7uG+jIa2Zb0j/aRAfbOxnT99kxybxCrdTDFNLB62FD+CljdQDzHVG2dY3RILLFO
# Ry3BFARxv2T5JL5zbcqOCb2zAVdJVGTZc9d/HltEAY5aGZFrDZ+kKNxnGSgkujhL
# mm77IVRrakURR6nxt67I6IleT53S0Ex2tVdUCbFpAUR+fKFhbHP+CrvsQWY9af3L
# wUFJfn6Tvsv4O+S3Fb+0zj6lMVGEvL8CwYKiexcdFYmNcP7ntdAoGokLjzbaukz5
# m/8K6TT4JDVnK+ANuOaMmdbhIurwJ0I9JZTmdHRbatGePu1+oDEzfbzL6Xu/OHBE
# 0ZDxyKs6ijoIYn/ZcGNTTY3ugm2lBRDBcQZqELQdVTNYs6FwZvKhggLOMIICNwIB
# ATCB+KGB0KSBzTCByjELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
# EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjElMCMGA1UECxMcTWljcm9zb2Z0IEFtZXJpY2EgT3BlcmF0aW9uczEmMCQGA1UE
# CxMdVGhhbGVzIFRTUyBFU046M0JCRC1FMzM4LUU5QTExJTAjBgNVBAMTHE1pY3Jv
# c29mdCBUaW1lLVN0YW1wIFNlcnZpY2WiIwoBATAHBgUrDgMCGgMVAC01yuYmIVvs
# okSacJmWe8Mu2QFtoIGDMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIw
# MTAwDQYJKoZIhvcNAQEFBQACBQDnyL5kMCIYDzIwMjMwMzI1MDgyNzE2WhgPMjAy
# MzAzMjYwODI3MTZaMHcwPQYKKwYBBAGEWQoEATEvMC0wCgIFAOfIvmQCAQAwCgIB
# AAICBV8CAf8wBwIBAAICEiswCgIFAOfKD+QCAQAwNgYKKwYBBAGEWQoEAjEoMCYw
# DAYKKwYBBAGEWQoDAqAKMAgCAQACAwehIKEKMAgCAQACAwGGoDANBgkqhkiG9w0B
# AQUFAAOBgQCVokgXZfEY89vN00VZn83Avpf3Vba0wWvG0f1bF4RTcjMZqlK6v6ju
# r1WdhJuYKxMxVZqSXymhyiP7nsMME+ESOi+Iki10g5lp+4UZflJXemw+2bvEJH1G
# LusuRgpvGqSOvUUilnsIwm60590wIFhayvQkAcVgo4B3CL12okHgHTGCBA0wggQJ
# AgEBMIGTMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
# VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAk
# BgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAABxjDNLtbT
# ocD0AAEAAAHGMA0GCWCGSAFlAwQCAQUAoIIBSjAaBgkqhkiG9w0BCQMxDQYLKoZI
# hvcNAQkQAQQwLwYJKoZIhvcNAQkEMSIEIMlLAUmLUcRXXDQbL0pnx4cr40v9tPpL
# /7i3QqCG20VMMIH6BgsqhkiG9w0BCRACLzGB6jCB5zCB5DCBvQQgVjETXFXJvZgp
# xiZqk/BwDWA5Pxw6hehULpkFr9rjry0wgZgwgYCkfjB8MQswCQYDVQQGEwJVUzET
# MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
# TWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1T
# dGFtcCBQQ0EgMjAxMAITMwAAAcYwzS7W06HA9AABAAABxjAiBCCkDwRTH/0SiSpF
# 0hP2L/DpoK+xHosxUSupSToH8JsFEzANBgkqhkiG9w0BAQsFAASCAgAYC87N0Sov
# /Hay02Yph4EyBWh4HwAwDL3vJhG/lS6REsi0NJ/74paFm87VOsIZKu1kD75e6IR7
# nMG0DRt1gvZWn7E6Z2ju6Ux6SezSpuEjE75Ild6gjG2Xi2mVUj3i5McbtMfpUjNH
# B5zfuCXmwYSmx/fW9MP1UJZSkWCWAdMjFeCycOU+Xo6GPw/j5zib19aDaxgwhQfh
# sXR530GVsccUuui2728Z5itlChZZnuFDSllQhNkRM0653JrMQXjxTsXGGjTdGGaD
# x5BWGtg3lL7RFEsmchQKAuf2ZzDgoU4R3V7cdS/E1VJ5fiDNLPcmy/57lOTKpSK+
# iuVVBlLEkjSE7tfjsHXGXy5eC26n425M9TFXggXqIX0ttzwfBhNg6a6MqoYQB7Ni
# DwPkp/oiwCBCYMHO5Wl5BYKdcS6rvnVRRegm3mh54A0xp26Zfdat+d9o807J5msw
# iJWYUG5snmW9HUst9PVXyBv0XG6Gvo5aqvS2MQnFh9PMzjp2BQa3Lwk3/gi4ZqJJ
# R++PVGvR/BOQ1jPhZCgc/XEcALEs6xEtekhMox+/pBNlmQZjZQd3Diy1mJJ9+V12
# XHSgveFLuUDS6r2iqA2jW2de3Oy4fSoYVsEk8jrCFtCo0PI/ZCIDpdZxdQsS4nv4
# UI5NGIprPKavUiNdniwCcPh4hUQEsi4Vng==
# SIG # End signature block
