﻿param
(
    [parameter(Position=0,Mandatory=$false)][boolean] $useServiceFabric=$false 
)

Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1"  -ArgumentList $useServiceFabric -DisableNameChecking

$script:EnvDllsLoaded = $false
function Get-ApplicationEnvironment
{
    $ErrorActionPreference = 'Stop'

    if (-not $script:EnvDllsLoaded)
    {        
        $webroot = Get-AosWebSitePhysicalPath
        $InstalledPlatformBuildVersion = Get-ProductPlatformBuildFullVersion
        $isKeyVault = Get-IsMSIKeyVault

        if (($isKeyVault -eq $true) -and ($InstalledPlatformBuildVersion -ne $null) -and ($InstalledPlatformBuildVersion.Build -ge 5644))
        {        
            # Get-PackageRoot comes from CommonRollbackUtilities.psm1. it resolves the root of the deployable package
            $packageRoot = Get-PackageRoot
            $packageTemp = ""

            if(![string]::IsNullOrEmpty($packageRoot) -and (Test-Path $packageRoot))
            {
                $packageTemp = Join-Path $packageRoot "RunbookWorkingFolder\KeyVaultAssemblyTemp"
            }

            if ([string]::IsNullOrEmpty($packageTemp))
            {   
                $suffix = $(Get-Date).ToString("yyyyMMdd_HHmmss")
                $dirName = "AssemblyTemp_$suffix"
                $packageTemp = Join-Path $env:Temp "dirName"
            }

            if(!(Test-Path $packageTemp -PathType Container))
            {
                New-Item $packageTemp -ItemType Container | Out-Null
            }
        
            #need to load the Microsoft.Dynamics.ApplicationPlatform.Environment.dll and all the dll it referenced
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Configuration.Base.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.Instrumentation.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Framework.EncryptionEngine.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.BusinessPlatform.SharedTypes.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.KeyVaultHelper.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Carbon.Flighting.Runtime.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Carbon.CertificateCommon.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Commerce.Flighting.Common.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Flighting.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.XppServices.Instrumentation.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.CE.VaultSDK.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.CE.VaultSDK.ADALv2.AuthHelper.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.KeyVault.Core.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.KeyVault.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.Common.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.Common.NetFramework.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Hyak.Common.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Authentication.Instrumentation.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.Extensions.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.Extensions.Desktop.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.IdentityModel.Clients.ActiveDirectory.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Practices.EnterpriseLibrary.TransientFaultHandling.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Newtonsoft.Json.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.SharedUtility.dll'
            CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp            

            try
            {
                $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
                $fileTempPath =  Join-Path $packageTemp $(Get-Item $dllPath).Name
                if (Test-Path $fileTempPath)
                {
                    # Load file from temp folder
                    [System.Reflection.Assembly]::LoadFrom($fileTempPath)
                }
            }
            catch
            {    
                #write-log "Failed to load from temp folder with Exception: [$($_.Exception)]"
                throw $_
            }
        }              
        else
        {
            #need to load the Microsoft.Dynamics.ApplicationPlatform.Environment.dll and all the dll it referenced
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
            Load-DllinMemory -dllPath $dllPath
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Configuration.Base.dll'
            Load-DllinMemory -dllPath $dllPath
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.Instrumentation.dll'
            if (Test-Path $dllPath)
            {
                Load-DllinMemory -dllPath $dllPath
            }
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Framework.EncryptionEngine.dll'
            Load-DllinMemory -dllPath $dllPath
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.BusinessPlatform.SharedTypes.dll'
            Load-DllinMemory -dllPath $dllPath  
        }

        $script:EnvDllsLoaded = $true
    }

    $config = [Microsoft.Dynamics.ApplicationPlatform.Environment.EnvironmentFactory]::GetApplicationEnvironment()
    
    return $config
}

function Load-DllinMemory([string] $dllPath)
{
    #try catch as not all dll exist in RTM version, some dependency/dll are introduced at update 1 or later
    #powershell cannot unload dll once it's loaded, the trick is to create an in-memory copy of the dll than load it
    #after the loading of in-memory dll, the physical dll stay unlocked

    try
    {
        if (Test-Path $dllPath)
        {
            $bytes = [System.IO.File]::ReadAllBytes($dllPath)
            [System.Reflection.Assembly]::Load($bytes) | Out-Null
        }
    }
    catch
    {}
}

function CopyDllToTempFolder([string] $dllPath, [string] $packageTemp)
{
    #try catch as not all dll exist in RTM version, some dependency/dll are introduced at update 1 or later    
    $fileTempPath =  Join-Path $packageTemp $(Get-Item $dllPath).Name
    try
    {
        if ((Test-Path $dllPath) -And !$(Test-Path $fileTempPath))
        {
            $Buffer = New-Object 'byte[]' 1024
            # Open file in read mode
            $SourceFile = [System.IO.File]::Open($dllPath, [System.IO.FileMode]::Open, [System.IO.FileAccess]::Read, [System.IO.FileShare]::Read)
            # Create the new file
            $DestinationFile = [System.IO.File]::Open($fileTempPath, [System.IO.FileMode]::CreateNew)
            try{
                # Copy the contents of the source file to the destination
                while(($readLength = $SourceFile.Read($Buffer, 0, $Buffer.Length)) -gt 0)
                {
                    $DestinationFile.Write($Buffer, 0, $readLength)
                }
            }
            catch{
                throw $_
            }
            finally{
                $SourceFile.Close()
                $DestinationFile.Close()
            }
        }
    }
    catch
    {
        Write-Log "Failed to copy file to temp folder with Exception: [$($_.Exception)]"
        throw $_
    }    
}

function Get-IsMSIKeyVault
{
    $webroot = Get-AosWebSitePhysicalPath
    $webconfig=join-path $webroot "web.config"
    $isMSIKeyVault=$false
    
    [System.Xml.XmlDocument] $xd=new-object System.Xml.XmlDocument
    $xd.Load($webconfig)
    $ns=New-Object System.Xml.XmlNamespaceManager($xd.NameTable)
    $ns.AddNamespace("ns",$xd.DocumentElement.NamespaceURI)

    $key="CertificateHandler.IsMSIKeyVault"

    $isKeyVaultValue = $xd.SelectSingleNode("//ns:add[@key='$key']",$ns)

    if($isKeyVaultValue -ne $null)
    {        
        $isMSIKeyVault = $isKeyVaultValue.GetAttribute("value")
    }

    return $isMSIKeyVault 
}

function GenerateMetadataModuleInstallationInfo
{
    try
    {
        $ErrorActionPreference = 'Stop'

        write-output "Creating Metadata Module Installation Info."
        
        $packagePath = Get-AOSPackageDirectory
        $CommonBin = $(Get-CommonBinDir)

        $dllPath = Join-Path $CommonBin 'bin\Microsoft.Dynamics.AX.AXInstallationInfo.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $CommonBin 'bin\Microsoft.Dynamics.AX.Metadata.Storage.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $CommonBin 'bin\Microsoft.Dynamics.AX.Metadata.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $CommonBin 'bin\Microsoft.Dynamics.AX.Metadata.Core.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $CommonBin 'bin\Microsoft.Dynamics.ApplicationPlatform.XppServices.Instrumentation.dll'
        Load-DllinMemory -dllPath $dllPath

        [Microsoft.Dynamics.AX.AXInstallationInfo.AXInstallationInfo]::ScanMetadataModelInRuntimePackage($packagePath)
    }
    catch
    {}
}

function Get-IsAppFallOrLater([Parameter(Mandatory=$false)] $webroot)
{
    $version = Get-ApplicationReleaseFromAOS -webroot:$webroot

    return $($version -ne "RTW")
}

function Get-ApplicationReleaseFromPackage
{
    $packageRoot = split-Path -parent "$(split-Path -parent $PSScriptRoot)" 
    $PackageInstallationInfo = "$packageRoot\HotfixInstallationInfo.xml"
    $XPath = '/HotfixInstallationInfo/ServiceModelList/ComponentModel/Release[../ServiceModelGroup = "Application"]'
    $ApplicationRelease =  Select-Xml -Path $PackageInstallationInfo -XPath $Xpath 
    $MRXPath = '/HotfixInstallationInfo/ServiceModelList/ComponentModel[Name = "MRApplicationService"]'
    $MRApplicationServiceModel =  Select-Xml -Path $PackageInstallationInfo -XPath $MRXpath 

    if($ApplicationRelease.Count -ge 1)
    {
        return $ApplicationRelease[0].Node.InnerText
    }
    elseif ($MRApplicationServiceModel -eq $null)
    {
        return ""
    }
    else
    {
        return "RTW"
    }
}

function Get-IsModulePartOfPlatformAsBinary ([string] $packageNugetFile)
{
    [Void][Reflection.Assembly]::LoadWithPartialName('System.IO.Compression.FileSystem') 

    try
    {
        $PackFiles = [IO.Compression.ZipFile]::OpenRead($packageNugetFile)
        $PackageSpec =  $PackFiles.Entries | where {($_.Name -like '*.nuspec')}

        if(!($PackageSpec))
        {
            Throw "Unable to get package information"
        }

        [System.Xml.XmlDocument] $xmlDoc = New-Object System.Xml.XmlDocument
        $XmlDoc.Load($PackageSpec.Open())

        $PackFiles.Dispose()

        $Description = $xmlDoc.GetElementsByTagName('description').Item(0).InnerText

        if($Description.Contains("[Platform Package]"))
        {
            return $true
        }
        else
        {
            return $false
        }        
    }
    catch
    {
        return $false
    }
}

function Get-IsModulePartOfApplicationAsBinary ([string] $PackageNugetFilePath)
{
    [Void][Reflection.Assembly]::LoadWithPartialName('System.IO.Compression.FileSystem') 

    try
    {
        $PackFiles = [IO.Compression.ZipFile]::OpenRead($PackageNugetFilePath)
        $PackageSpec =  $PackFiles.Entries | Where-Object {($_.Name -like '*.nuspec')}

        if(!($PackageSpec))
        {
            Throw "Unable to get package information"
        }

        [System.Xml.XmlDocument] $xmlDoc=new-object System.Xml.XmlDocument
        $XmlDoc.Load($PackageSpec.Open())

        $PackFiles.Dispose()

        $Description = $xmlDoc.GetElementsByTagName('description').Item(0).InnerText

        if($Description.Contains("[Application Package]"))
        {
            return $true
        }
        else
        {
            return $false
        }
    }
    catch
    {
        return $false
    }
}

function Get-IsPlatformUpdate3OrLater([Parameter(Mandatory=$false)] $webroot)
{
    if (!$webroot) {
    $webroot = Get-AosWebSitePhysicalPath
    }

    #must use job or process to load the production information provider dll or it'll lock it self
    #in memory copy is not usable as this dll have some special hard coded reference dll which won't resolve when loaded in memory.
    $job = Start-Job -ScriptBlock  {
        param($webrootBlock)
        $VersionDLLPath = Join-Path $webrootBlock 'bin\Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.dll'
        Add-Type -Path $VersionDLLPath
        $provider = [Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.ProductInfoProvider]::get_Provider();
        $version = $provider.get_PlatformVersion();
        $version
    } -ArgumentList $webroot
    Wait-Job -Job $job | Out-Null
    $version = Receive-Job -Job $job

    return $(($version -ne '7.0') -and ($version -ne '7.0.0.0') -and (![string]::IsNullOrEmpty($version)))    
}

function Get-IsPackageContainUpgradeBinaryForUpdate3
{
    $sourcePath = [IO.Path]::Combine($(split-Path -parent $PSScriptRoot), "Packages")
    #using a framework package from platform which customer cannot generate to identify if it's from platform update 3
    $files=get-childitem -Path:$sourcePath dynamicsax-framework-bin.*.nupkg
    foreach ($packageFile in $files) 
    {
        if(Get-IsModulePartOfPlatformAsBinary -packageNugetFile $packageFile.FullName)
        {
            return $true
        }
    }

    return $false
}

function Get-ProductPlatformBuildVersion([Parameter(Mandatory=$false)] $webroot)
{
    if (!$webroot) {
       $webroot = Get-AosWebSitePhysicalPath
    }

    #must use job or process to load the production information provider dll or it'll lock it self
    #in memory copy is not usable as this dll have some special hard coded reference dll which won't resolve when loaded in memory.
    $job = Start-Job -ScriptBlock  {
        param($webrootBlock)
        $VersionDLLPath = Join-Path $webrootBlock 'bin\Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.dll'
        Add-Type -Path $VersionDLLPath
        $provider = [Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.ProductInfoProvider]::get_Provider();
        $version = $provider.get_PlatformBuildVersion();
        $version
    } -ArgumentList $webroot
    Wait-Job -Job $job | Out-Null
    $version = Receive-Job -Job $job
   
    $build = [System.Version]::new()
    $releaseBuild
    if([System.Version]::TryParse($Version, [ref]$build))
    {
        $releaseBuild = $build.Build
    }
    else
    {
        #default to 0 from 7.0.0.0 
        $releaseBuild = 0
    }   

    return  $releaseBuild
}

function Get-ProductPlatformBuildFullVersion
{
    $webroot = Get-AosWebSitePhysicalPath

    #must use job or process to load the production information provider dll or it'll lock it self
    #in memory copy is not usable as this dll have some special hard coded reference dll which won't resolve when loaded in memory.
    $job = Start-Job -ScriptBlock  {
        param($webrootBlock)
        $VersionDLLPath = Join-Path $webrootBlock 'bin\Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.dll'
        Add-Type -Path $VersionDLLPath
        $provider = [Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.ProductInfoProvider]::get_Provider();
        $version = $provider.get_PlatformBuildVersion();
        $version
    } -ArgumentList $webroot
    Wait-Job -Job $job | Out-Null
    $version = Receive-Job -Job $job
   
    $build = [System.Version]::new()
    
    if([System.Version]::TryParse($Version, [ref]$build))
    {
        return $build
    }
    
    return $null
}

function Get-ProductApplicationVersion ([Parameter(Mandatory=$false)] $webroot)
{
    if($webroot -eq $null)
    {
        $webroot = Get-AosWebSitePhysicalPath
    }

    #must use job or process to load the production information provider dll or it'll lock it self
    #in memory copy is not usable as this dll have some special hard coded reference dll which won't resolve when loaded in memory.
    $job = Start-Job -ScriptBlock  {
        param($webrootBlock)
        $VersionDLLPath = Join-Path $webrootBlock 'bin\Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.dll'
        Add-Type -Path $VersionDLLPath
        $provider = [Microsoft.Dynamics.BusinessPlatform.ProductInformation.Provider.ProductInfoProvider]::get_Provider();
        $version = $provider.get_ApplicationBuildVersion();
        $version
    } -ArgumentList $webroot
    Wait-Job -Job $job | Out-Null
    $version = Receive-Job -Job $job    
    
    return  $version
}

function Get-DataAccessSqlPwd
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.SqlPwd
}

function Get-DataAccessDatabase
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.Database
}

function Get-DataAccessSqlUsr
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.SqlUser
}

function Get-DataAccessDbServer
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.DbServer
}


function Get-AOSPackageDirectory
{
    $Config = Get-ApplicationEnvironment
    return $Config.Aos.PackageDirectory
}

function Get-CommonBinDir
{
    $Config = Get-ApplicationEnvironment
    return $Config.Common.BinDir
}

function Get-BiReportingPersistentVirtualMachineIPAddressSSRS
{
    $Config = Get-ApplicationEnvironment
    return $Config.BiReporting.PersistentVirtualMachineIPAddressSSRS
}

function Get-BiReportingReportingServers
{
    $Config = Get-ApplicationEnvironment
    $reportingServers = $Config.BiReporting.ReportingServers
    if ([System.String]::IsNullOrWhiteSpace($reportingServers))
    {
        $reportingServers = $Config.BiReporting.PersistentVirtualMachineIPAddressSSRS
    }

    return $reportingServers
}

function Get-UseReportingCluster
{
    $webRoot = Get-AosWebSitePhysicalPath
    $webConfigPath = join-path $webRoot "web.config"
    
    if(Test-Path $webConfigPath){
        [System.Xml.XmlDocument] $webConfig = new-object System.Xml.XmlDocument
        $webConfig.Load($webConfigPath)

        $ns = New-Object System.Xml.XmlNamespaceManager($webConfig.NameTable)
        $ns.AddNamespace("ns",$webConfig.DocumentElement.NamespaceURI)
        $key = "UseReportingCluster"
        $userClusterKey = $webConfig.SelectSingleNode("//ns:add[@key='$key']",$ns)

        if($null -ne $userClusterKey){
            if($userClusterKey.Value -eq $true){
                return $true
            }
        }
    }

    return $false
}

function Get-InfrastructureClickonceAppsDirectory
{
    $Config = Get-ApplicationEnvironment
    return $Config.Infrastructure.ClickonceAppsDirectory
}

function Get-DevToolsInstalled
{
    $webroot = Get-AosWebSitePhysicalPath
    $webconfig=join-path $webroot "web.config"
    $DevInstall=$false
    
    [System.Xml.XmlDocument] $xd=new-object System.Xml.XmlDocument
    $xd.Load($webconfig)
    $ns=New-Object System.Xml.XmlNamespaceManager($xd.NameTable)
    $ns.AddNamespace("ns",$xd.DocumentElement.NamespaceURI)

    $key="Infrastructure.VSToolsCount"

    $VScount = $xd.SelectSingleNode("//ns:add[@key='$key']",$ns)

    if($VScount -ne $null){
        
        if($VScount.GetAttribute("value") -gt 0)
        {
            $DevInstall=$true
        }
    }
    return $DevInstall 
}

function Get-ProductVersionMajorMinor
{
    [string]  $productVersionMajorMinorString = '7.0'
    return $productVersionMajorMinorString
}

function Get-IsRetailProductSku
{
    $productVersion = Get-ProductVersionMajorMinor    
    $retailHQConfigurationLocationRegistryPath = "HKLM:\SOFTWARE\Microsoft\Dynamics\$productVersion\Setup\Metadata"
    if(Test-Path $retailHQConfigurationLocationRegistryPath)
    {
        $retailProductSku = Get-ItemProperty -Path $retailHQConfigurationLocationRegistryPath -Name 'ProductSku' -ErrorAction SilentlyContinue
        
        if($retailProductSku -and $retailProductSku.ProductSku -and ($retailProductSku.ProductSku -eq 'Dynamics365ForRetail'))
        {
            return $true
        }
    }

    return $false
}

function Get-IsOverlayeringDisabled
{
    $productVersion = Get-ProductVersionMajorMinor    
    $retailHQConfigurationLocationRegistryPath = "HKLM:\SOFTWARE\Microsoft\Dynamics\$productVersion\Setup\Metadata"
    if(Test-Path $retailHQConfigurationLocationRegistryPath)
    {
        $isOverlayeringDisabled = Get-ItemProperty -Path $retailHQConfigurationLocationRegistryPath -Name 'DisableOverlayering' -ErrorAction SilentlyContinue
        
        if($isOverlayeringDisabled -and $isOverlayeringDisabled.DisableOverlayering -and ($isOverlayeringDisabled.DisableOverlayering -like 'true'))
        {
            return $true
        }
    }

    return $false
}

function Get-IsAppSealed ([Parameter(Mandatory=$false)] $webroot)
{
    if($webroot -eq $null)
    {
        $webroot = Get-AosWebSitePhysicalPath
    }

    $Version = (Get-ProductApplicationVersion -webroot:$webroot)
    $build = [System.Version]::new()
    $minSealedVersion = new-object System.Version "8.1"

    if([System.Version]::TryParse($Version, [ref]$build))
    {
        # Any build major/minor 8.1+ is considered sealed.
        if($build.CompareTo($minSealedVersion) -ge 0)
        {
           return $true
        }

        # Sealing logic for 8.0 and earlier
        if($build.Major -ge "8")
        {
            ##Bug 240512: Receipt designer bits are not updated with the uptake of latest binary hotfix
            ##The appseal have a different meaning during package deployment than development
            ##if the appSeal == true we expect full binary hotfix, other wise we expect seperate x++ hotfix and special handle the remaining mofule (ie, clickonce... package)
            ##the app8.0, even though it's sealed from dev point of view, it's still releasing seperate x++ hotfix and the special handling code path for none sealed need to be used during package deployment
            if($build.Minor -ge "1") 
            {
                return $true
            }
        }
        elseif($build.Minor -ge "2")
        {
            if($build.Build.ToString().StartsWith("2"))
            {
                return $true
            }
        }
    }
    
    return $false
    
}

function Get-DependencyAXModelList([string] $sourcePath, [string] $metaPackageName)
{
    $microsoftPackages = @()
    [Void][Reflection.Assembly]::LoadWithPartialName('System.IO.Compression.FileSystem')
    $zipFile = Get-Item $sourcePath\$metaPackageName*.nupkg
    if($zipFile -ne $null)
    {
        $PackFiles = [IO.Compression.ZipFile]::OpenRead($zipFile).Entries
        $PackageSpec =  $PackFiles | where {($_.Name -like "*.nuspec")}

        if(!($PackageSpec))
        {
            Throw "Unable to get package information for $metaPackageName"
        }

        [System.Xml.XmlDocument] $xmlDoc=new-object System.Xml.XmlDocument
        $XmlDoc.Load($PackageSpec.Open())

        $Dependencies = $xmlDoc.GetElementsByTagName('dependency').id
        
        foreach($d in $Dependencies)
        {
            $microsoftPackages += $d
        }
    }
    
    return $microsoftPackages
}

function Get-ALMPackageCopyPath
{
    [string]$ALMBackupPath = ""

    # Is the ALM Service registered (environment variable "DynamicsSDK" would be set)
    if ($env:DynamicsSDK)
    {
        $RegPath = "HKLM:\SOFTWARE\Microsoft\Dynamics\AX\7.0\SDK"

        [string]$ALMBackupPath = $null
        
        # Do not fail when registry key is not found
        try
        {
            # Get the Dynamics SDK (ALM Service) registry key (throws if not found).
            $RegKey = Get-ItemProperty -Path $RegPath
            if ($RegKey -ne $null)
            {
                # Check if backup path set in registry
                $ALMBackupPath = $RegKey.BackupPath
            }
        }
        catch
        {
        }
        
        # If path not found in registry, check default paths
        if (-not $ALMBackupPath)
        {
            if (Test-Path "I:\DynamicsBackup")
            {
                $ALMBackupPath = "I:\DynamicsBackup"
            }
            elseif (Test-Path "$($env:SystemDrive)\DynamicsBackup")
            {
                $ALMBackupPath = "$($env:SystemDrive)\DynamicsBackup"
            }
        }
    }

    return $ALMBackupPath
}

function Get-EnvironmentId
{
    $Config = Get-ApplicationEnvironment
    return $Config.LCS.LCSEnvironmentId
}

function Get-DataAccessFlightingCachePath
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.FlightingCachePath
}

function Get-DataAccessFlightingEnvironment
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.FlightingEnvironment
}

function Get-DataAccessFlightingCertificateThumbprint
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.FlightingCertificateThumbprint
}

function Get-DataAccessFlightingServiceCatalogID
{
    $Config = Get-ApplicationEnvironment
    return $Config.DataAccess.FlightingServiceCatalogID
}

function Get-TenantDomainGUID
{
    $Config = Get-ApplicationEnvironment
    return $Config.Aad.TenantDomainGUID
}

function Get-AADTenantId
{
    $Config = Get-ApplicationEnvironment
    return $Config.Aad.AADTenantId
}

function Get-S2SCertificate
{
    try
    {
        $config = Get-ApplicationEnvironment
        $certHandlerSetttings = $config.CertificateHandler.GetCertificateHandlerSettings()

        $certificateHandler = [Microsoft.Dynamics.AX.Configuration.CertificateHandler.CertificateHandlerFactory]::CreateCertificateHandler($certHandlerSetttings)
    
        $s2sCertificate = $certificateHandler.GetSingleCertificateForId($config.Infrastructure.S2SCertThumbprint)

        return $s2sCertificate
    }
    catch
    {
        return $null
    }
}

function Load-FlightingDlls
{
    $webroot = Get-AosWebSitePhysicalPath

    Load-DllinMemory "$webroot\bin\Microsoft.Carbon.CertificateCommon.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Carbon.Flighting.Runtime.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Commerce.Flighting.Common.dll"

    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.BusinessPlatform.SharedTypes.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.ApplicationPlatform.XppServices.Instrumentation.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.ApplicationPlatform.Services.Instrumentation.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.AX.Data.Sql.Shared.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.ApplicationPlatform.Flighting.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.ApplicationPlatform.AOSFlighting.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Dynamics.ApplicationPlatform.PerformanceCounters.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Practices.EnterpriseLibrary.TransientFaultHandling.dll"
    Load-DllinMemory "$webroot\bin\Microsoft.Practices.EnterpriseLibrary.TransientFaultHandling.Data.dll"
    Load-DllinMemory "$webroot\bin\Newtonsoft.Json.dll"

    # Contained in XppServices.Instrumentation in earlier releases
    $dbSyncInstrumentationPath = "$webroot\bin\Microsoft.Dynamics.ApplicationPlatform.DatabaseSynchronize.Instrumentation.dll"
    if (Test-Path $dbSyncInstrumentationPath)
    {
        Load-DllinMemory $dbSyncInstrumentationPath
    }
}

$script:FlightingProvider = $null
function Get-IsFlightEnabled([string] $flightName)
{
    $ErrorActionPreference = 'Stop'

    if ([String]::IsNullOrWhiteSpace($flightName))
    {
        return $false
    }

    if ($script:FlightingProvider -eq $null)
    {
        $sqlPwd    = Get-DataAccessSqlPwd
        $sqlUser   = Get-DataAccessSqlUsr
        $sqlServer = Get-DataAccessDbServer
        $sqlDB     = Get-DataAccessDatabase
        $connectionString = "Data Source=$sqlServer; " +
            "Integrated Security=False; " +
            "User Id=$sqlUser; " +
            "Password=`"$sqlPwd`"; " +
            "Initial Catalog=$sqlDB"

        $flightingCacheFolder = Get-DataAccessFlightingCachePath
        if ([String]::IsNullOrWhiteSpace($flightingCacheFolder))
        {
            $flightingCacheFolder = Get-AosWebSitePhysicalPath # webroot
        }

        Load-FlightingDlls

        $flightingConfiguration = New-Object -TypeName Microsoft.Dynamics.ApplicationPlatform.AOSFlighting.StaticFlightingServiceConfiguration -ArgumentList `
            (Get-DataAccessFlightingEnvironment), `
            (Get-DataAccessFlightingCertificateThumbprint), `
            (Get-DataAccessFlightingServiceCatalogID), `
            $flightingCacheFolder, `
            $null, # object extraInformation `
            $connectionString, `
            (60 * 60) # int refreshDataIntervalInSec

        $script:FlightingProvider = [Microsoft.Dynamics.ApplicationPlatform.AOSFlighting.AOSFlightingSystem]::InitializeFlightService($flightingConfiguration)
        $script:FlightingProvider.AddInformationToCommonContext("TenantID", (Get-TenantDomainGUID))
        $script:FlightingProvider.AddInformationToCommonContext("EnvironmentId", (Get-EnvironmentId))

        Write-Log "Flighting provider is a $($script:FlightingProvider.GetType().Name)."
    }

    return $script:FlightingProvider.IsFeatureEnabled($flightName)
}

function IsLocalServer([string] $serverName)
{
    if ($serverName -eq $null) { return $false }
    
    if ("." -eq $serverName) { return $true }
    if ("(local)" -eq $serverName) { return $true }
    if ("localhost" -eq $serverName) { return $true }
    if ("127.0.0.1" -eq $serverName) { return $true }
    if ([System.Environment]::MachineName -eq $serverName.Split('.')[0]) { return $true }

    return $false
}

function Invoke-XppStaticMethod()
{
    Param(
        [Parameter(Mandatory=$true)]
        [ValidatePattern('^[a-zA-Z0-9_]+$')]
        [String]
        $ClassName,

        [Parameter(Mandatory=$true)]
        [ValidatePattern('^[a-zA-Z0-9_]+$')]
        [String]
        $MethodName,

        [Parameter(Mandatory=$true)]
        [ValidateNotNullOrEmpty()]
        [String]
        $RedirectStandardError
    )

    $commandParameter = "-bindir `"$(Get-CommonBinDir)`""
    $commandParameter += " -metadatadir `"$(Get-AOSPackageDirectory)`""
    $commandParameter += " -sqluser `"$(Get-DataAccessSqlUsr)`""
    $commandParameter += " -sqlserver `"$(Get-DataAccessDbServer)`""
    $commandParameter += " -sqldatabase `"$(Get-DataAccessDatabase)`""
    $commandParameter += " -sqlpwd `"$(Get-DataAccessSqlPwd)`""
    $commandParameter += " -setupmode runstaticxppmethod"
    $commandParameter += " -classname $ClassName"
    $commandParameter += " -methodname $MethodName"

    $webroot = Get-AosWebSitePhysicalPath
    $startProcessArgs = @{
        FilePath = "$webroot\bin\Microsoft.Dynamics.AX.Deployment.Setup.exe"
        ArgumentList = $commandParameter
        PassThru = $true
        Wait = $true
        RedirectStandardError = $RedirectStandardError
    }

    $process = Start-Process @startProcessArgs
    
    if ($process.ExitCode -ne 0)
    {
        $errorContent = Get-Content "$RedirectStandardError"
        if ($errorContent -eq $null)
        {
            $errorContent = "Process terminated with exit code $($process.ExitCode) (error stream is empty)."
        }
        throw $errorContent
    }
}

if(!$useServiceFabric)
{
    if (Test-Path "$($PSScriptRoot)\NonAdminDevToolsInterject.ps1")
    {
        & "$($PSScriptRoot)\NonAdminDevToolsInterject.ps1"
    }
}

function Get-HotfixInstallationInfo
{
    $packageRoot = split-Path -parent "$(split-Path -parent $PSScriptRoot)"

    # Using job to load the AXInstallationInfo dll to avoid file locks
    $job = Start-Job -ScriptBlock  {
        param($packageRootBlock)
        $axInstallationInfoDLLPath = Join-Path $packageRootBlock "Microsoft.Dynamics.AX.AXInstallationInfo.dll"
        $hotfixInstallationInfoFilePath = Join-Path $packageRootBlock "HotfixInstallationInfo.xml"
        try
        {
            if (Test-Path $axInstallationInfoDLLPath)
            {
                # Load AXInstallationInfo.dll
                [void][System.Reflection.Assembly]::LoadFrom($axInstallationInfoDLLPath)
                $hotfixInstallationInfo = New-Object Microsoft.Dynamics.AX.AXInstallationInfo.HotfixInstallationInfo

                if (Test-Path $hotfixInstallationInfoFilePath)
                {
                    # Initialize HotfixInstallationInfo from HotfixInstallationInfo.xml
                    $hotfixInstallationInfo.InitFromFile($hotfixInstallationInfoFilePath)
                    # Get HotfixInstallationInfo
                    $hotfixInstallationInfo
                }
            }
        }
        catch
        {
            Write-Warning "Failed to get HotfixInstallationInfo instance from deployable package"
            return $null
        }
    } -ArgumentList $packageRoot
    Wait-Job -Job $job | Out-Null
    $hotfixInstallationInfo  = Receive-Job -Job $job

    return $hotfixInstallationInfo
}

function Get-IsBinaryPackageTypeFromPackage
{
    $hotfixInstallationInfo = Get-HotfixInstallationInfo
    # Get Package Type from HotfixInstallationInfo for the package being deployed
    $packageType = $hotfixInstallationInfo.Type
    if(![string]::IsNullOrEmpty($packageType))
    {
        return ($packageType -eq 'PlatAppBinaryHotfixPackage' -or $packageType -eq 'MergedPackage')
    }

    Write-Warning "Failed to get package type from deployable package"
    return $false
}

function Get-MetadataModuleListFromPackage
{
    $hotfixInstallationInfo = Get-HotfixInstallationInfo
    # Get Metadata Module List from HotfixInstallationInfo of the package being deployed
    $metadataModuleList = $hotfixInstallationInfo.MetadataModuleList
    if(![string]::IsNullOrEmpty($metadataModuleList))
    {
        return $metadataModuleList
    }

    Write-Warning "Failed to get metadata module list from HotfixInstallationInfo.xml."
    return ""
}

function Get-PackageNamesFromDLL([Parameter(Mandatory=$false)] $webroot, [Parameter(Mandatory=$true)][string] $productInfoDLL)
{
    if (!$webroot) {
    $webroot = Get-AosWebSitePhysicalPath
    }

    # Using job to load the Product Information dll to avoid file locks
    $job = Start-Job -ScriptBlock  {
        param($webrootBlock, $productInfoDLLBlock)
        $metadataCoreDLLPath = Join-Path $webrootBlock "bin\Microsoft.Dynamics.AX.Metadata.Core.dll"
        $productInfoDLLPath = Join-Path $webrootBlock $productInfoDLLBlock
        try
        {
            if (Test-Path $metadataCoreDLLPath)
            {
                # Load Metadata.Core.dll
                [void][System.Reflection.Assembly]::LoadFrom($metadataCoreDLLPath)
                if (Test-Path $productInfoDLLPath)
                {
                    # Get package names from Product Information dll
                    $packageNames = [Microsoft.Dynamics.AX.Metadata.Core.CoreHelper]::GetPackagesNames($productInfoDLLPath, $null)
                    $packageNames
                }
            }
        }
        catch
        {
            Write-Warning "Failed to get package names from product information dll."
            return $null
        }
    } -ArgumentList $webroot,$productInfoDLL
    Wait-Job -Job $job | Out-Null
    $packageNames  = Receive-Job -Job $job
    return $packageNames
}

Export-ModuleMember -Function Get-DataAccessSqlPwd
Export-ModuleMember -Function Get-DataAccessDatabase
Export-ModuleMember -Function Get-DataAccessSqlUsr
Export-ModuleMember -Function Get-DataAccessDbServer
Export-ModuleMember -Function Get-AOSPackageDirectory
Export-ModuleMember -Function Get-CommonBinDir
Export-ModuleMember -Function Get-BiReportingPersistentVirtualMachineIPAddressSSRS
Export-ModuleMember -Function Get-BiReportingReportingServers
Export-ModuleMember -Function Get-UseReportingCluster
Export-ModuleMember -Function Get-InfrastructureClickonceAppsDirectory
Export-ModuleMember -Function Get-DevToolsInstalled
Export-ModuleMember -Function Get-IsModulePartOfPlatformAsBinary
Export-ModuleMember -Function Get-IsAppFallOrLater
Export-ModuleMember -Function Get-IsPlatformUpdate3OrLater
Export-ModuleMember -Function Get-IsPackageContainUpgradeBinaryForUpdate3
Export-ModuleMember -Function Get-ProductPlatformBuildVersion
Export-ModuleMember -Function Get-ProductApplicationVersion
Export-ModuleMember -Function Get-ProductPlatformBuildFullVersion
Export-ModuleMember -Function Get-ApplicationReleaseFromPackage
Export-ModuleMember -Function Get-ProductVersionMajorMinor
Export-ModuleMember -Function Get-IsRetailProductSku
Export-ModuleMember -Function Get-IsOverlayeringDisabled
Export-ModuleMember -Function Get-DependencyAXModelList
Export-ModuleMember -Function Get-IsAppSealed
Export-ModuleMember -Function Get-IsModulePartOfApplicationAsBinary
Export-ModuleMember -Function Get-ALMPackageCopyPath
Export-ModuleMember -Function Get-IsFlightEnabled
Export-ModuleMember -Function GenerateMetadataModuleInstallationInfo
Export-ModuleMember -Function Get-EnvironmentId
Export-ModuleMember -Function Get-AADTenantId
Export-ModuleMember -Function Get-TenantDomainGUID
Export-ModuleMember -Function Get-S2SCertificate
Export-ModuleMember -Function IsLocalServer
Export-ModuleMember -Function Invoke-XppStaticMethod
Export-ModuleMember -Function Get-IsBinaryPackageTypeFromPackage
Export-ModuleMember -Function Get-MetadataModuleListFromPackage
Export-ModuleMember -Function Get-PackageNamesFromDLL

# SIG # Begin signature block
# MIInygYJKoZIhvcNAQcCoIInuzCCJ7cCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCl/ccGLqnEnnJv
# 1HCBSSXueHFuMCYkbfotpy0/Dx7f8KCCDYEwggX/MIID56ADAgECAhMzAAACzI61
# lqa90clOAAAAAALMMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjIwNTEyMjA0NjAxWhcNMjMwNTExMjA0NjAxWjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQCiTbHs68bADvNud97NzcdP0zh0mRr4VpDv68KobjQFybVAuVgiINf9aG2zQtWK
# No6+2X2Ix65KGcBXuZyEi0oBUAAGnIe5O5q/Y0Ij0WwDyMWaVad2Te4r1Eic3HWH
# UfiiNjF0ETHKg3qa7DCyUqwsR9q5SaXuHlYCwM+m59Nl3jKnYnKLLfzhl13wImV9
# DF8N76ANkRyK6BYoc9I6hHF2MCTQYWbQ4fXgzKhgzj4zeabWgfu+ZJCiFLkogvc0
# RVb0x3DtyxMbl/3e45Eu+sn/x6EVwbJZVvtQYcmdGF1yAYht+JnNmWwAxL8MgHMz
# xEcoY1Q1JtstiY3+u3ulGMvhAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUiLhHjTKWzIqVIp+sM2rOHH11rfQw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDcwNTI5MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAeA8D
# sOAHS53MTIHYu8bbXrO6yQtRD6JfyMWeXaLu3Nc8PDnFc1efYq/F3MGx/aiwNbcs
# J2MU7BKNWTP5JQVBA2GNIeR3mScXqnOsv1XqXPvZeISDVWLaBQzceItdIwgo6B13
# vxlkkSYMvB0Dr3Yw7/W9U4Wk5K/RDOnIGvmKqKi3AwyxlV1mpefy729FKaWT7edB
# d3I4+hldMY8sdfDPjWRtJzjMjXZs41OUOwtHccPazjjC7KndzvZHx/0VWL8n0NT/
# 404vftnXKifMZkS4p2sB3oK+6kCcsyWsgS/3eYGw1Fe4MOnin1RhgrW1rHPODJTG
# AUOmW4wc3Q6KKr2zve7sMDZe9tfylonPwhk971rX8qGw6LkrGFv31IJeJSe/aUbG
# dUDPkbrABbVvPElgoj5eP3REqx5jdfkQw7tOdWkhn0jDUh2uQen9Atj3RkJyHuR0
# GUsJVMWFJdkIO/gFwzoOGlHNsmxvpANV86/1qgb1oZXdrURpzJp53MsDaBY/pxOc
# J0Cvg6uWs3kQWgKk5aBzvsX95BzdItHTpVMtVPW4q41XEvbFmUP1n6oL5rdNdrTM
# j/HXMRk1KCksax1Vxo3qv+13cCsZAaQNaIAvt5LvkshZkDZIP//0Hnq7NnWeYR3z
# 4oFiw9N2n3bb9baQWuWPswG0Dq9YT9kb+Cs4qIIwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIZnzCCGZsCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAsyOtZamvdHJTgAAAAACzDAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgCYg9BufB
# gGGHNQISK57qCPJTpXid15Fn+5byUuWmj44wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQBUVYUP5snD8guYFjtauI0fBaWAkT/gNzthjS2vY3kR
# suIugWAmNBaAPMwErPlMZW3qbOs1fUhj0f0Jvt5Tx6citr4eYE3fMcZNML5JQn77
# hYO+v8YROVMGnSvZkkAY2mXKQsm+XRIS1+zvohzlKtKPzKg7sBCX767sf0vGn7Br
# DUFCdf1WAZ40QSdjq0VmgmSKiSQnHegWOB7Pf1L28OXr+1HnkWAVb8img2znChJ1
# mR3vrntQhUTewcit4yGRLU/uUfnK/GlgfmOWclPIpo2fGg2hTMUGapP0vn/PNFR1
# IHq8IlQL1UZbKg4u+3ahUASd8pOiiAtKRDZHAyfKDl/xoYIXKTCCFyUGCisGAQQB
# gjcDAwExghcVMIIXEQYJKoZIhvcNAQcCoIIXAjCCFv4CAQMxDzANBglghkgBZQME
# AgEFADCCAVkGCyqGSIb3DQEJEAEEoIIBSASCAUQwggFAAgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIP4zP/IONTaosBYMsCI5IBAf2S5UV7Wjo639qxmV
# TTGmAgZjovLqrVAYEzIwMjMwMTA2MjE0NTE1LjIwMlowBIACAfSggdikgdUwgdIx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1p
# Y3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlvbnMgTGltaXRlZDEmMCQGA1UECxMdVGhh
# bGVzIFRTUyBFU046M0JENC00QjgwLTY5QzMxJTAjBgNVBAMTHE1pY3Jvc29mdCBU
# aW1lLVN0YW1wIFNlcnZpY2WgghF4MIIHJzCCBQ+gAwIBAgITMwAAAbT7gAhEBdIt
# +gABAAABtDANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Eg
# MjAxMDAeFw0yMjA5MjAyMDIyMDlaFw0yMzEyMTQyMDIyMDlaMIHSMQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQg
# SXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxlcyBUU1Mg
# RVNOOjNCRDQtNEI4MC02OUMzMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFt
# cCBTZXJ2aWNlMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAtEemnmUH
# MkIfvOiu27K86ZbwWhksGwV72Dl1uGdqr2pKm+mfzoT+Yngkq9aLEf+XDtADyA+2
# KIZU0iO8WG79eJjzz29flZpBKbKg8xl2P3O9drleuQw3TnNfNN4+QIgjMXpE3txP
# F7M7IRLKZMiOt3FfkFWVmiXJAA7E3OIwJgphg09th3Tvzp8MT8+HOtG3bdrRd/y2
# u8VrQsQTLZiVwTZ6qDYKNT8PQZl7xFrSSO3QzXa91LipZnYOl3siGJDCee1Ba7X1
# i13dQFHxKl5Ff4JzDduOBZ85e2VrpyFy1a3ayGUzBrIw59jhMbjIw9YVcQt9kUWn
# tyCmNk15WybCS+hXpEDDLVj1X5W9snmoW1qu03+unprQjWQaVuO7BfcvQdNVdyKS
# qAeKy1eT2Hcc5n1aAVeXFm6sbVJmZzPQEQR3Jr7W8YcTjkqC5hT2qrYuIcYGOf3P
# j4OqdXm1Qqhuwtskxviv7yy3Z+PxJpxKx+2e6zGRaoQmIlLfg/a42XNVHTf6Wzr5
# k7Q1w7v0uA/sFsgyKmI7HzKHX08xDDSmJooXA5btD6B0lx/Lqs6Qb4KthnA7N2IE
# dJ5sjMIhyHZwBr7fzDskU/+Sgp2UnfqrN1Vda/gb+pmlbJwi8MphvElYzjT7PZK2
# Dm4eorcjx7T2QVe3EIelLuGbxzybblZoRTkCAwEAAaOCAUkwggFFMB0GA1UdDgQW
# BBTLRIXl8ZS4Opy7Eii3Tt44zDLZfjAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJl
# pxtTNRnpcjBfBgNVHR8EWDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NybC9NaWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAx
# MCgxKS5jcmwwbAYIKwYBBQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3Rh
# bXAlMjBQQ0ElMjAyMDEwKDEpLmNydDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQM
# MAoGCCsGAQUFBwMIMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQsFAAOCAgEA
# EtEPBYwpt4JioSq0joGzwqYX6SoNH7YbqpgArdlnrdt6u3ukKREluKEVqS2XajXx
# x0UkXGc4Xi9dp2bSxpuyQnTkq+IQwkg7p1dKrwAa2vdmaNzz3mrSaeUEu40yCThH
# wquQkweoG4eqRRZe19OtVSmDDNC3ZQ6Ig0qz79vivXgy5dFWk4npxA5LxSGR4wBa
# XaIuVhoEa06vd/9/2YsQ99bCiR7SxJRt1XrQ5kJGHUi0Fhgz158qvXgfmq7qNqfq
# fTSmsQRrtbe4Zv/X+qPo/l6ae+SrLkcjRfr0ONV0vFVuNKx6Cb90D5LgNpc9x8V/
# qIHEr+JXbWXW6mARVVqNQCmXlVHjTBjhcXwSmadR1OotcN/sKp2EOM9JPYr86O9Y
# /JAZC9zug9qljKTroZTfYA7LIdcmPr69u1FSD/6ivL6HRHZd/k2EL7FtZwzNcRRd
# FF/VgpkOxHIfqvjXambwoMoT+vtGTtqgoruhhSk0bM1F/pBpi/nPZtVNLGTNaK8W
# t6kscbC9G6f09gz/wBBJOBmvTLPOOT/3taCGSoJoDABWnK+De5pie4KX8BxxKQbJ
# vxz7vRsVJ5R6mGx+Bvav5AjsxvZZw6eQmkI0vPRckxL9TCVCfWS0uyIKmyo6Tdos
# nbBO/osre7r0jS9AH8spEqVlhFcpQNfOg/CvdS2xNVMwggdxMIIFWaADAgECAhMz
# AAAAFcXna54Cm0mZAAAAAAAVMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9v
# dCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0yMTA5MzAxODIyMjVaFw0z
# MDA5MzAxODMyMjVaMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIICIjAN
# BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA5OGmTOe0ciELeaLL1yR5vQ7VgtP9
# 7pwHB9KpbE51yMo1V/YBf2xK4OK9uT4XYDP/XE/HZveVU3Fa4n5KWv64NmeFRiMM
# tY0Tz3cywBAY6GB9alKDRLemjkZrBxTzxXb1hlDcwUTIcVxRMTegCjhuje3XD9gm
# U3w5YQJ6xKr9cmmvHaus9ja+NSZk2pg7uhp7M62AW36MEBydUv626GIl3GoPz130
# /o5Tz9bshVZN7928jaTjkY+yOSxRnOlwaQ3KNi1wjjHINSi947SHJMPgyY9+tVSP
# 3PoFVZhtaDuaRr3tpK56KTesy+uDRedGbsoy1cCGMFxPLOJiss254o2I5JasAUq7
# vnGpF1tnYN74kpEeHT39IM9zfUGaRnXNxF803RKJ1v2lIH1+/NmeRd+2ci/bfV+A
# utuqfjbsNkz2K26oElHovwUDo9Fzpk03dJQcNIIP8BDyt0cY7afomXw/TNuvXsLz
# 1dhzPUNOwTM5TI4CvEJoLhDqhFFG4tG9ahhaYQFzymeiXtcodgLiMxhy16cg8ML6
# EgrXY28MyTZki1ugpoMhXV8wdJGUlNi5UPkLiWHzNgY1GIRH29wb0f2y1BzFa/Zc
# UlFdEtsluq9QBXpsxREdcu+N+VLEhReTwDwV2xo3xwgVGD94q0W29R6HXtqPnhZy
# acaue7e3PmriLq0CAwEAAaOCAd0wggHZMBIGCSsGAQQBgjcVAQQFAgMBAAEwIwYJ
# KwYBBAGCNxUCBBYEFCqnUv5kxJq+gpE8RjUpzxD/LwTuMB0GA1UdDgQWBBSfpxVd
# AF5iXYP05dJlpxtTNRnpcjBcBgNVHSAEVTBTMFEGDCsGAQQBgjdMg30BATBBMD8G
# CCsGAQUFBwIBFjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3Mv
# UmVwb3NpdG9yeS5odG0wEwYDVR0lBAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQC
# BAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYD
# VR0jBBgwFoAU1fZWy4/oolxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZF
# aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9v
# Q2VyQXV0XzIwMTAtMDYtMjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcw
# AoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJB
# dXRfMjAxMC0wNi0yMy5jcnQwDQYJKoZIhvcNAQELBQADggIBAJ1VffwqreEsH2cB
# MSRb4Z5yS/ypb+pcFLY+TkdkeLEGk5c9MTO1OdfCcTY/2mRsfNB1OW27DzHkwo/7
# bNGhlBgi7ulmZzpTTd2YurYeeNg2LpypglYAA7AFvonoaeC6Ce5732pvvinLbtg/
# SHUB2RjebYIM9W0jVOR4U3UkV7ndn/OOPcbzaN9l9qRWqveVtihVJ9AkvUCgvxm2
# EhIRXT0n4ECWOKz3+SmJw7wXsFSFQrP8DJ6LGYnn8AtqgcKBGUIZUnWKNsIdw2Fz
# Lixre24/LAl4FOmRsqlb30mjdAy87JGA0j3mSj5mO0+7hvoyGtmW9I/2kQH2zsZ0
# /fZMcm8Qq3UwxTSwethQ/gpY3UA8x1RtnWN0SCyxTkctwRQEcb9k+SS+c23Kjgm9
# swFXSVRk2XPXfx5bRAGOWhmRaw2fpCjcZxkoJLo4S5pu+yFUa2pFEUep8beuyOiJ
# Xk+d0tBMdrVXVAmxaQFEfnyhYWxz/gq77EFmPWn9y8FBSX5+k77L+DvktxW/tM4+
# pTFRhLy/AsGConsXHRWJjXD+57XQKBqJC4822rpM+Zv/Cuk0+CQ1ZyvgDbjmjJnW
# 4SLq8CdCPSWU5nR0W2rRnj7tfqAxM328y+l7vzhwRNGQ8cirOoo6CGJ/2XBjU02N
# 7oJtpQUQwXEGahC0HVUzWLOhcGbyoYIC1DCCAj0CAQEwggEAoYHYpIHVMIHSMQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNy
# b3NvZnQgSXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxl
# cyBUU1MgRVNOOjNCRDQtNEI4MC02OUMzMSUwIwYDVQQDExxNaWNyb3NvZnQgVGlt
# ZS1TdGFtcCBTZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQBlnNiQ85uX9nN4KRJt/gHk
# Jx4JCKCBgzCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMA0GCSqG
# SIb3DQEBBQUAAgUA52KJGzAiGA8yMDIzMDEwNjE5NDg0M1oYDzIwMjMwMTA3MTk0
# ODQzWjB0MDoGCisGAQQBhFkKBAExLDAqMAoCBQDnYokbAgEAMAcCAQACAgy1MAcC
# AQACAhEnMAoCBQDnY9qbAgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkK
# AwKgCjAIAgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZIhvcNAQEFBQADgYEAHUpd
# b95ne0GFt6y1/HCcJ+MeAKaCcGLla70FSDDtMHFdhIFRe0gNZ2+puOoR2KjL3UN/
# 3aQEPNkpX0YJQzdZn8yXJmSHfeGzUaZqFPSuy04gQQCX7scnMrrMUn1MBtEKZMKi
# aLpiYXncuAKxMaU0rVzlI2zzivyi/KjxPGdABWAxggQNMIIECQIBATCBkzB8MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNy
# b3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAbT7gAhEBdIt+gABAAABtDAN
# BglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMC8G
# CSqGSIb3DQEJBDEiBCAMw1/BNKPkr2tFwCDaPFSKERjCkoW5TGxyPs+ukC+91zCB
# +gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EINPI93vmozBwBlFxvfr/rElreFPR
# 4ux7vXKx2ni3AfcGMIGYMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIw
# MTACEzMAAAG0+4AIRAXSLfoAAQAAAbQwIgQgVzouNU9EvwqpWfnCWT5ztgfJn1Fy
# Zc/1ArY8qXIx43gwDQYJKoZIhvcNAQELBQAEggIAZle0tq9Oi9p0izk/ML+nzGrX
# H8mcWSQgXyvJqdaJLUf4KX2pZizdFtrMNmZgr7nc++/CTqkUnHO+L6eYgvxvYNqX
# VDXTEFhYT4bUubi9fjmS671pRSQrDfLA713/zf8MxYVDrfrDrIVexQxZ1zQk/1A3
# qYU0BiH6QAPY/27s/V5qxogul/pNqSNyMqBajT2Mmf2850cBg41VCiCVIdqyFBUZ
# Wp+W4Z9mFO/0EGD0SaLHoHiEtsxmUwoWf1oJBNfOct7TpNTleYIb81ZzAvM5Em3a
# ZfsCHFSyUjV0vaUOE0RMyLWcL3gPoM/Uf96LOpg3LLABVyhyVu9bhFwBN9DAJVeg
# ADWYcPk+f6g+M4TOeLtr7+Z9HWXuZQHw7Mtysh/JGL1CQQiAEAfJZXo6eDnyYdi7
# Hs7axyAJOByGyNN5q+/A/XofHYNG3arOQpgelf6t1GfxVBRQlru5aLZfzMHjqfkr
# JViUBHZtdHNvyaiTLza52G7olscklcSu2C1HgU3AUYopC3RLALQTFSuwFy2ODR/O
# G/LIe5/m6yRtAnfguXgdqx5Ci6k71lUEBoeaiAGVJWVgC0aeykV89Jru8uPN/9jw
# EQfsNFJMieBAJ2LXiREldERLKfSDNRtcVHTLkZRpD/EkihlO1KlaKMB3UmqeL5n4
# ejfQ5KhA5KXP6tRR/6M=
# SIG # End signature block
