﻿[CmdletBinding()]
param
(
    [Parameter(Mandatory=$false)]
    [string]$LogDir,

    [parameter(Mandatory=$false)]
    [boolean] $useServiceFabric=$false,

    [Parameter(Mandatory = $false)]
    [string]$ActivityId,

    [Parameter(Mandatory=$false)]
	[hashtable]$AxDRConnectionString,

    [Parameter(Mandatory=$false)]
	[string[]]$EnabledServicingFeatures,

    [Parameter(Mandatory = $false)]
    [string]$RunbookId,
    
    [Parameter(Mandatory=$false)]
    [hashtable]$Credentials
)

# Check whether PITR required after running full sync.
function IsPITRRequiredDuringRollback([string]$ActivityId)
{
  if(!$ActivityId)
  {
      return $true
  }

  $axDBPwd = Get-DataAccessSqlPwd
  $axDBUser = Get-DataAccessSqlUsr
  $axDBDatabaseName = Get-DataAccessDatabase
  $axDBServer = Get-DataAccessDbServer

    $target_sqlParams = @{
		'Database' = $axDBDatabaseName
		'UserName' = $axDBUser
		'Password' = $axDBPwd
		'ServerInstance' = $axDBServer
		'EncryptConnection' = !(IsLocalServer($axDBServer))
		'QueryTimeout' = 0
	    }

    $target_sqlParams.Query = "Select ISNULL(MAX(CAST(PITRRequiredForRollback as INT)), 1) as PITRRequiredForRollback, ISNULL(MAX(CAST(DBSyncRequiredForRollback as INT)), 0) as DBSyncRequiredForRollback from DBSyncExecStats where SyncStep = 'FullAll' and RelatedActivityId = '$ActivityId'"

    try
    {
        $result = Invoke-SqlCmd @target_sqlParams
        if ($result -ne $null)
        {
            $successMessage = "[$(Get-Date)]:Get PITR required status succeeded. PITRRequiredForRollback: $($result.PITRRequiredForRollback), DBSyncRequiredForRollback: $($result.DBSyncRequiredForRollback), activity id: $ActivityId."
            Write-Host $successMessage
            EventWrite-RunbookScriptTrace -Message $successMessage -Component "servicing" -RunbookId $RunbookId| Out-Null

            return ($result.PITRRequiredForRollback -or $result.DBSyncRequiredForRollback)
        }
    }
    catch
    {
        $warningMessage = "[$(Get-Date)]:Get PITR required status failed, activity id: $ActivityId, Exception: [$($_.Exception)]"
        Write-Host $warningMessage
        EventWrite-RunbookScriptTrace -Message $warningMessage -Component "servicing" -RunbookId $RunbookId| Out-Null
    }

    return $true
}

Import-Module "$PSScriptRoot\AosEnvironmentUtilities.psm1" -Force -DisableNameChecking
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -ArgumentList $useServiceFabric -DisableNameChecking
Import-Module "$PSScriptRoot\ExecuteSqltoAXDB.psm1" -Force -DisableNameChecking

$ErrorActionPreference = "stop"

$metadatadir = $(Get-AOSPackageDirectory)

$deltaSyncFolder = Join-Path "$(split-Path -parent $metadatadir)"  "DeltaSync"
$deltaSyncOutputSecurityMD = Join-Path $deltaSyncFolder  "DeltaSyncSecurityMD.xml"
$deltaSyncOutputTableMD = Join-Path $deltaSyncFolder  "DeltaSyncTableMD.xml"
$deltaSyncOutputEdtMD = Join-Path $deltaSyncFolder  "DeltaSyncEdtMD.xml"
$comparerDllPath = "$(split-Path -parent $PSScriptRoot)" | Split-Path -Parent

[System.Reflection.Assembly]::LoadFile("$comparerDllPath\Microsoft.Dynamics.AXCreateDeployablePackageBase.dll") >$null

$SecurityMdFileIdentical = [Microsoft.Dynamics.AXCreateDeployablePackageBase.DirectoryProcessing]::CustomDirectoryCompareIsIdentical($deltaSyncFolder,$metadatadir, "*AXSecurity*.md", $deltaSyncOutputSecurityMD  )
$TableMdFileIdentical = [Microsoft.Dynamics.AXCreateDeployablePackageBase.DirectoryProcessing]::CustomDirectoryCompareIsIdentical($deltaSyncFolder,$metadatadir, "*AXTable*.md", $deltaSyncOutputTableMD  )
$EdtMdFileIdentical = [Microsoft.Dynamics.AXCreateDeployablePackageBase.DirectoryProcessing]::CustomDirectoryCompareIsIdentical($deltaSyncFolder,$metadatadir, "*AXEdt*.md", $deltaSyncOutputEdtMD  )

$noSchemaImpactFlag = $(Join-Path $deltaSyncFolder  "noSchemaImpactMDChangeFound.xml")

if(($SecurityMdFileIdentical -eq $true) -and
($TableMdFileIdentical -eq $true) -and
($EdtMdFileIdentical -eq $true))
{
    Copy-Item $deltaSyncOutputTableMD -Destination $noSchemaImpactFlag
    Write-Output "No Schema Change detected, performing delta DbSync operation."
}
elseif(Test-Path $noSchemaImpactFlag)
{
    Remove-Item $noSchemaImpactFlag
}

if(!$LogDir)
{
    $LogDir = $PSScriptRoot
}

# Inject & Turned On flighings before DB Sync PreCheck
$axDBPwd = Get-DataAccessSqlPwd
$axDBUser = Get-DataAccessSqlUsr
$axDBDatabaseName = Get-DataAccessDatabase
$axDBServer = Get-DataAccessDbServer
$flightings='DMFEnableSqlRowVersionChangeTrackingIndexing','DMFEnableCreateRecIdIndexForDataSynchronization','DMFDisableRecurringIntegrationImportRetryV2','DisableNullFieldValueSupportedForFnOTables'

$target_sqlParams_insertFlightings = @{
    'Database' = $axDBDatabaseName
    'UserName' = $axDBUser
    'Password' = $axDBPwd
    'ServerInstance' = $axDBServer
    'EncryptConnection' = !(IsLocalServer($axDBServer))
    'QueryTimeout' = 0
}

foreach($Name in $flightings)
{
    $target_sqlParams_insertFlightings.Query = "SELECT FLIGHTNAME FROM SYSFLIGHTING WHERE FLIGHTNAME='$Name'"
    $res=Invoke-SqlCmd @target_sqlParams_insertFlightings
    if($res -eq $null)
    {
        $target_sqlParams_insertFlightings.Query = "INSERT INTO SYSFLIGHTING (FLIGHTNAME, ENABLED) VALUES('$Name', 1)"
        $result = Invoke-SqlCmd @target_sqlParams_insertFlightings            
    }
    else
    {
         Write-Host "$Name Exists in SYSFLIGHTING Table"
    }
}

# Run SQL Script to disable SysDeletedObjectSysRowVersion2 config based on status of SysDeletedObjectSysRowVersion while upgrading to PU63+
$TargetPUVersion = (Get-ProductPlatformBuildVersion)
if($TargetPUVersion -gt 7120)
{ 
    $configchecksql = Get-Content "$PSScriptRoot\Disable_Sysrowversion2Config_While_upgrade.sql" -Raw        
    Invoke-SQL -sqlCommand:$configchecksql
}

$deploySetupPs1 = "$PSScriptRoot\TriggerDeploymentSetupEngine.ps1"

# Run DB Sync PreCheck
$precheckScriptPath = "$PSScriptRoot\DBSyncPrecheck.ps1"
$precheckScriptPath = Resolve-Path -Path $precheckScriptPath
& $precheckScriptPath -LogDir $LogDir -ActivityId $ActivityId -RunbookId $RunbookId -EnabledServicingFeatures $EnabledServicingFeatures -Credentials $Credentials

# Run service sync (Full sync)
$deploymentSetupParameter = "-setupmode servicesync -syncmode fullall"
if($ActivityId)
{
    $deploymentSetupParameter += " -activityid $ActivityId"
}

[hashtable]$returnResult = @{}
$currentDateTime = [DateTime]::UtcNow.ToString("yyyyMMddHHmmss")
$logfile = Join-Path $LogDir "dbsync_$currentDateTime.log"
$errorfile = Join-Path $LogDir "dbsync.error_$currentDateTime.log"

try
{
    # Reset PITR required flag to true before invoking DB Sync
    Write-IsPITRRequiredDuringRollback -PitrRequired 'true'

    invoke-Expression "$deploySetupPs1 -deploymentSetupParameter: `"$deploymentSetupParameter`" -logfile:`"$logfile`" -errorfile:`"$errorfile`""
    Write-Output "DbSync completed."

    #Check whether PITR required on full sync success.    
    $pitrRequired = IsPITRRequiredDuringRollback -ActivityId $ActivityId

    # TODO: Uncomment below line after addressing Bug 533892
    # Write-IsPITRRequiredDuringRollback -PitrRequired $pitrRequired
}
catch
{
  #Check whether PITR required on full sync failure
  $pitrRequired = IsPITRRequiredDuringRollback -ActivityId $ActivityId  

  # TODO: Uncomment below line after addressing Bug 533892
  # Write-IsPITRRequiredDuringRollback -PitrRequired $pitrRequired

  throw $($_.Exception)
}

#creating AX ping service user
$webroot = Get-AosWebSitePhysicalPath
[boolean] $isPU3OrLater = Get-IsPlatformUpdate3OrLater -webroot:$webroot
if ($isPU3OrLater -eq $false)
{
    $deploymentSetupParameter = "-setupmode runstaticxppmethod -classname AxPingHelper -methodname createServiceUser"
    $logfile = Join-Path $LogDir "createAXPingServiceUser.log"
    $errorfile = Join-Path $LogDir "createAXPingServiceUser.error.log"

    try
    {
        invoke-Expression "$deploySetupPs1 -deploymentSetupParameter: `"$deploymentSetupParameter`" -logfile:`"$logfile`" -errorfile:`"$errorfile`""
    }
    catch
    {
        $createAXPingServiceUserError = Get-Content $errorfile
        if ($createAXPingServiceUserError -ne $null) 
        {
            if($createAXPingServiceUserError -like "*Couldn't find id for class AxPingHelper*")
            {
                Write-Output "AXPulse Module not found, skipping create AX Ping service user operation"
                return
            }
            throw $createAXPingServiceUserError
        }
    }
}

# SIG # Begin signature block
# MIIr5wYJKoZIhvcNAQcCoIIr2DCCK9QCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCCyKsqDOcVjdnq
# zPWakNAq/z3jUprRqO2Ofwvebj3Y6qCCEW4wggh+MIIHZqADAgECAhM2AAACAO38
# jbec3qFIAAIAAAIAMA0GCSqGSIb3DQEBCwUAMEExEzARBgoJkiaJk/IsZAEZFgNH
# QkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxFTATBgNVBAMTDEFNRSBDUyBDQSAwMTAe
# Fw0yNDExMDgxMjQzMjhaFw0yNTExMDgxMjQzMjhaMCQxIjAgBgNVBAMTGU1pY3Jv
# c29mdCBBenVyZSBDb2RlIFNpZ24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK
# AoIBAQC5L/UPrOpwYjxcoZC0TqqvMF1WUELvwXN+k27SrA5rohJknn7Cgbxg4hGT
# XKqpcdbtsVTN3ZY896SJ20uQ+INL5OVLzpW408nCNTPYg2LtGJbqHUjpNm0hLCJ+
# gO5Jn2T8DDzIJoUijGXj1m+hRLKb2nOIicCED2GuYBmuWXnaY7INmVEaU3peryty
# ZjDuxdyGDuiPURz8lW1SUiDzoszNp1oswVr+WjDvLDUx4HlxPsG8zUjIst0NnJ6o
# z4tNFKaUBDCetcMjQxpCETn29a1CuRddxZLjPHZHfcotr5sh1S6bNQdzVaMNsxV8
# L3wjHb7XJ6ZVm662mHEiPgpyNcLhAgMBAAGjggWKMIIFhjApBgkrBgEEAYI3FQoE
# HDAaMAwGCisGAQQBgjdbAQEwCgYIKwYBBQUHAwMwPQYJKwYBBAGCNxUHBDAwLgYm
# KwYBBAGCNxUIhpDjDYTVtHiE8Ys+hZvdFs6dEoFgg93NZoaUjDICAWQCAQ4wggJ2
# BggrBgEFBQcBAQSCAmgwggJkMGIGCCsGAQUFBzAChlZodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpaW5mcmEvQ2VydHMvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDEu
# YW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUy
# MDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDIuYW1lLmdibC9haWEv
# QlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBS
# BggrBgEFBQcwAoZGaHR0cDovL2NybDMuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAx
# LkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZG
# aHR0cDovL2NybDQuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDCBrQYIKwYBBQUHMAKGgaBsZGFwOi8vL0NO
# PUFNRSUyMENTJTIwQ0ElMjAwMSxDTj1BSUEsQ049UHVibGljJTIwS2V5JTIwU2Vy
# dmljZXMsQ049U2VydmljZXMsQ049Q29uZmlndXJhdGlvbixEQz1BTUUsREM9R0JM
# P2NBQ2VydGlmaWNhdGU/YmFzZT9vYmplY3RDbGFzcz1jZXJ0aWZpY2F0aW9uQXV0
# aG9yaXR5MB0GA1UdDgQWBBST/HE52ZUlmsYqZcZBdrXZ5u4ZnzAOBgNVHQ8BAf8E
# BAMCB4AwRQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEWMBQGA1UEBRMNMjM2MTY3KzUwMzE1NTCCAeYGA1UdHwSCAd0wggHZMIIB
# 1aCCAdGgggHNhj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpaW5mcmEvQ1JM
# L0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwxLmFtZS5nYmwv
# Y3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwyLmFtZS5n
# YmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwzLmFt
# ZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmw0
# LmFtZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGgb1sZGFwOi8v
# L0NOPUFNRSUyMENTJTIwQ0ElMjAwMSgyKSxDTj1CWTJQS0lDU0NBMDEsQ049Q0RQ
# LENOPVB1YmxpYyUyMEtleSUyMFNlcnZpY2VzLENOPVNlcnZpY2VzLENOPUNvbmZp
# Z3VyYXRpb24sREM9QU1FLERDPUdCTD9jZXJ0aWZpY2F0ZVJldm9jYXRpb25MaXN0
# P2Jhc2U/b2JqZWN0Q2xhc3M9Y1JMRGlzdHJpYnV0aW9uUG9pbnQwHwYDVR0jBBgw
# FoAUllGE4Gtve/7YBqvD8oXmKa5q+dQwHwYDVR0lBBgwFgYKKwYBBAGCN1sBAQYI
# KwYBBQUHAwMwDQYJKoZIhvcNAQELBQADggEBAEDd8Wf5RkHsB64vgn2slxDtHzSo
# It9xN/Dm3RdFjNZ0diTUPMgSPYQlSk8nIAfudnB9FLavGlvZLlyUpfrPSuikepj3
# i3pqNEFn6fNdNFv/wHMxv7hQTIDCmuoR1v1rX+w3oeleBPMnN3QmH4ff1NsynyV4
# dZdYgN9Cw9sC/S3pWZpJrbOs7YOM3vqyU6DciHhC4D9i2zByHCF2pu9nYfiQf5A2
# iUZenRvyo1E5rC+UP2VZXa4k7g66W20+zAajIKKIqEmRtWahekMkCcOIHFBY4RDA
# ybgPRSGur4VDAiZPjTXS90wQXrX9CwU20cfiCC6e76F4H95KtQjKYpzuNVAwggjo
# MIIG0KADAgECAhMfAAAAUeqP9pxzDKg7AAAAAABRMA0GCSqGSIb3DQEBCwUAMDwx
# EzARBgoJkiaJk/IsZAEZFgNHQkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxEDAOBgNV
# BAMTB2FtZXJvb3QwHhcNMjEwNTIxMTg0NDE0WhcNMjYwNTIxMTg1NDE0WjBBMRMw
# EQYKCZImiZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQD
# EwxBTUUgQ1MgQ0EgMDEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDJ
# mlIJfQGejVbXKpcyFPoFSUllalrinfEV6JMc7i+bZDoL9rNHnHDGfJgeuRIYO1LY
# /1f4oMTrhXbSaYRCS5vGc8145WcTZG908bGDCWr4GFLc411WxA+Pv2rteAcz0eHM
# H36qTQ8L0o3XOb2n+x7KJFLokXV1s6pF/WlSXsUBXGaCIIWBXyEchv+sM9eKDsUO
# LdLTITHYJQNWkiryMSEbxqdQUTVZjEz6eLRLkofDAo8pXirIYOgM770CYOiZrcKH
# K7lYOVblx22pdNawY8Te6a2dfoCaWV1QUuazg5VHiC4p/6fksgEILptOKhx9c+ia
# piNhMrHsAYx9pUtppeaFAgMBAAGjggTcMIIE2DASBgkrBgEEAYI3FQEEBQIDAgAC
# MCMGCSsGAQQBgjcVAgQWBBQSaCRCIUfL1Gu+Mc8gpMALI38/RzAdBgNVHQ4EFgQU
# llGE4Gtve/7YBqvD8oXmKa5q+dQwggEEBgNVHSUEgfwwgfkGBysGAQUCAwUGCCsG
# AQUFBwMBBggrBgEFBQcDAgYKKwYBBAGCNxQCAQYJKwYBBAGCNxUGBgorBgEEAYI3
# CgMMBgkrBgEEAYI3FQYGCCsGAQUFBwMJBggrBgEFBQgCAgYKKwYBBAGCN0ABAQYL
# KwYBBAGCNwoDBAEGCisGAQQBgjcKAwQGCSsGAQQBgjcVBQYKKwYBBAGCNxQCAgYK
# KwYBBAGCNxQCAwYIKwYBBQUHAwMGCisGAQQBgjdbAQEGCisGAQQBgjdbAgEGCisG
# AQQBgjdbAwEGCisGAQQBgjdbBQEGCisGAQQBgjdbBAEGCisGAQQBgjdbBAIwGQYJ
# KwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMBIGA1UdEwEB/wQI
# MAYBAf8CAQAwHwYDVR0jBBgwFoAUKV5RXmSuNLnrrJwNp4x1AdEJCygwggFoBgNV
# HR8EggFfMIIBWzCCAVegggFToIIBT4YxaHR0cDovL2NybC5taWNyb3NvZnQuY29t
# L3BraWluZnJhL2NybC9hbWVyb290LmNybIYjaHR0cDovL2NybDIuYW1lLmdibC9j
# cmwvYW1lcm9vdC5jcmyGI2h0dHA6Ly9jcmwzLmFtZS5nYmwvY3JsL2FtZXJvb3Qu
# Y3JshiNodHRwOi8vY3JsMS5hbWUuZ2JsL2NybC9hbWVyb290LmNybIaBqmxkYXA6
# Ly8vQ049YW1lcm9vdCxDTj1BTUVSb290LENOPUNEUCxDTj1QdWJsaWMlMjBLZXkl
# MjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9uLERDPUFNRSxE
# Qz1HQkw/Y2VydGlmaWNhdGVSZXZvY2F0aW9uTGlzdD9iYXNlP29iamVjdENsYXNz
# PWNSTERpc3RyaWJ1dGlvblBvaW50MIIBqwYIKwYBBQUHAQEEggGdMIIBmTBHBggr
# BgEFBQcwAoY7aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraWluZnJhL2NlcnRz
# L0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6Ly9jcmwyLmFt
# ZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6
# Ly9jcmwzLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUH
# MAKGK2h0dHA6Ly9jcmwxLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQw
# gaIGCCsGAQUFBzAChoGVbGRhcDovLy9DTj1hbWVyb290LENOPUFJQSxDTj1QdWJs
# aWMlMjBLZXklMjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9u
# LERDPUFNRSxEQz1HQkw/Y0FDZXJ0aWZpY2F0ZT9iYXNlP29iamVjdENsYXNzPWNl
# cnRpZmljYXRpb25BdXRob3JpdHkwDQYJKoZIhvcNAQELBQADggIBAFAQI7dPD+jf
# XtGt3vJp2pyzA/HUu8hjKaRpM3opya5G3ocprRd7vdTHb8BDfRN+AD0YEmeDB5HK
# QoG6xHPI5TXuIi5sm/LeADbV3C2q0HQOygS/VT+m1W7a/752hMIn+L4ZuyxVeSBp
# fwf7oQ4YSZPh6+ngZvBHgfBaVz4O9/wcfw91QDZnTgK9zAh9yRKKls2bziPEnxeO
# ZMVNaxyV0v152PY2xjqIafIkUjK6vY9LtVFjJXenVUAmn3WCPWNFC1YTIIHw/mD2
# cTfPy7QA1pT+GPARAKt0bKtq9aCd/Ym0b5tPbpgCiRtzyb7fbNS1dE740re0COE6
# 7YV2wbeo2sXixzvLftH8L7s9xv9wV+G22qyKt6lmKLjFK1yMw4Ni5fMabcgmzRvS
# jAcbqgp3tk4a8emaaH0rz8MuuIP+yrxtREPXSqL/C5bzMzsikuDW9xH10graZzSm
# PjilzpRfRdu20/9UQmC7eVPZ4j1WNa1oqPHfzET3ChIzJ6Q9G3NPCB+7KwX0OQmK
# yv7IDimj8U/GlsHD1z+EF/fYMf8YXG15LamaOAohsw/ywO6SYSreVW+5Y0mzJutn
# BC9Cm9ozj1+/4kqksrlhZgR/CSxhFH3BTweH8gP2FEISRtShDZbuYymynY1un+Ry
# fiK9+iVTLdD1h/SxyxDpZMtimb4CgJQlMYIZzzCCGcsCAQEwWDBBMRMwEQYKCZIm
# iZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQDEwxBTUUg
# Q1MgQ0EgMDECEzYAAAIA7fyNt5zeoUgAAgAAAgAwDQYJYIZIAWUDBAIBBQCgga4w
# GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisG
# AQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIHbFQvWZDA/PnB9uXmq0pTEoWwWxp6LO
# aBtzSbkdppSvMEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8AcwBvAGYA
# dKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEA
# i7JlsxOYRi7LE4T/DZaJlixNZwwEEu4yfbO2GF2V6/3xV7b4AWkohNt4GyLU1nic
# wSxban+Wh+mJdYD5gYSuiSBrH+w7gE43/ij1E41t0sPWkuHOXIrhOPm4W6nnha5Z
# W/iZDJP7arg3s08imTwGlk/eEbeYHdCVB/9IJJwt4Rc5EOuoSw8PRGieRl6aWzun
# MQmo6yDmBWWR2/LZEMw6BLd7McvpWngqMqe3M0qR5IM/Oif56xwL0ONZs60q1jpw
# kaw/S90DWD+SszXWYofigoixwUlK3ca2EoT5Qnw0REpUWqNxVOq4hsZW47xM8NG5
# SWapzfIyqPa+MeB5vOadoaGCF5cwgheTBgorBgEEAYI3AwMBMYIXgzCCF38GCSqG
# SIb3DQEHAqCCF3AwghdsAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFSBgsqhkiG9w0B
# CRABBKCCAUEEggE9MIIBOQIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFlAwQCAQUA
# BCB8vg8O87wwIK0dfIK8E4xfIys2a92dfv1qMJslZ+UgDwIGZ4kDlfzHGBMyMDI1
# MDEyMjIwNDE0OS45MTJaMASAAgH0oIHRpIHOMIHLMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1lcmljYSBP
# cGVyYXRpb25zMScwJQYDVQQLEx5uU2hpZWxkIFRTUyBFU046OTYwMC0wNUUwLUQ5
# NDcxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2WgghHtMIIH
# IDCCBQigAwIBAgITMwAAAe+JP1ahWMyo2gABAAAB7zANBgkqhkiG9w0BAQsFADB8
# MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
# bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1N
# aWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDAeFw0yMzEyMDYxODQ1NDhaFw0y
# NTAzMDUxODQ1NDhaMIHLMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1lcmljYSBPcGVyYXRpb25zMScwJQYD
# VQQLEx5uU2hpZWxkIFRTUyBFU046OTYwMC0wNUUwLUQ5NDcxJTAjBgNVBAMTHE1p
# Y3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2UwggIiMA0GCSqGSIb3DQEBAQUAA4IC
# DwAwggIKAoICAQCjC1jinwzgHwhOakZqy17oE4BIBKsm5kX4DUmCBWI0lFVpEiK5
# mZ2Kh59soL4ns52phFMQYGG5kypCipungwP9Nob4VGVE6aoMo5hZ9NytXR5ZRgb9
# Z8NR6EmLKICRhD4sojPMg/RnGRTcdf7/TYvyM10jLjmLyKEegMHfvIwPmM+AP7hz
# QLfExDdqCJ2u64Gd5XlnrFOku5U9jLOKk1y70c+Twt04/RLqruv1fGP8LmYmtHvr
# B4TcBsADXSmcFjh0VgQkX4zXFwqnIG8rgY+zDqJYQNZP8O1Yo4kSckHT43XC0oM4
# 0ye2+9l/rTYiDFM3nlZe2jhtOkGCO6GqiTp50xI9ITpJXi0vEek8AejT4PKMEO2b
# PxU63p63uZbjdN5L+lgIcCNMCNI0SIopS4gaVR4Sy/IoDv1vDWpe+I28/Ky8jWTe
# ed0O3HxPJMZqX4QB3I6DnwZrHiKn6oE38tgBTCCAKvEoYOTg7r2lF0Iubt/3+VPv
# KtTCUbZPFOG8jZt9q6AFodlvQntiolYIYtqSrLyXAQIlXGhZ4gNcv4dv1YAilnbW
# A9CsnYh+OKEFr/4w4M69lI+yaoZ3L/t/UfXpT/+yc7hS/FolcmrGFJTBYlS4nE1c
# uKblwZ/UOG26SLhDONWXGZDKMJKN53oOLSSk4ldR0HlsbT4heLlWlOElJQIDAQAB
# o4IBSTCCAUUwHQYDVR0OBBYEFO1MWqKFwrCbtrw9P8A63bAVSJzLMB8GA1UdIwQY
# MBaAFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMF8GA1UdHwRYMFYwVKBSoFCGTmh0dHA6
# Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY3Jvc29mdCUyMFRpbWUt
# U3RhbXAlMjBQQ0ElMjAyMDEwKDEpLmNybDBsBggrBgEFBQcBAQRgMF4wXAYIKwYB
# BQUHMAKGUGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY2VydHMvTWlj
# cm9zb2Z0JTIwVGltZS1TdGFtcCUyMFBDQSUyMDIwMTAoMSkuY3J0MAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwDgYDVR0PAQH/BAQDAgeAMA0G
# CSqGSIb3DQEBCwUAA4ICAQAYGZa3aCDudbk9EVdkP8xcQGZuIAIPRx9K1CA7uRzB
# t80fC0aWkuYYhQMvHHJRHUobSM4Uw3zN7fHEN8hhaBDb9NRaGnFWdtHxmJ9eMz6J
# pn6KiIyi9U5Og7QCTZMl17n2w4eddq5vtk4rRWOVvpiDBGJARKiXWB9u2ix0WH2E
# MFGHqjIhjWUXhPgR4C6NKFNXHvWvXecJ2WXrJnvvQGXAfNJGETJZGpR41nUN3ijf
# iCSjFDxamGPsy5iYu904Hv9uuSXYd5m0Jxf2WNJSXkPGlNhrO27pPxgT111myAR6
# 1S3S2hc572zN9yoJEObE98Vy5KEM3ZX53cLefN81F1C9p/cAKkE6u9V6ryyl/qSg
# xu1UqeOZCtG/iaHSKMoxM7Mq4SMFsPT/8ieOdwClYpcw0CjZe5KBx2xLa4B1neFi
# b8J8/gSosjMdF3nHiyHx1YedZDtxSSgegeJsi0fbUgdzsVMJYvqVw52WqQNu0GRC
# 79ZuVreUVKdCJmUMBHBpTp6VFopL0Jf4Srgg+zRD9iwbc9uZrn+89odpInbznYrn
# PKHiO26qe1ekNwl/d7ro2ItP/lghz0DoD7kEGeikKJWHdto7eVJoJhkrUcanTuUH
# 08g+NYwG6S+PjBSB/NyNF6bHa/xR+ceAYhcjx0iBiv90Mn0JiGfnA2/hLj5evhTc
# AjCCB3EwggVZoAMCAQICEzMAAAAVxedrngKbSZkAAAAAABUwDQYJKoZIhvcNAQEL
# BQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
# EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNV
# BAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eSAyMDEwMB4X
# DTIxMDkzMDE4MjIyNVoXDTMwMDkzMDE4MzIyNVowfDELMAkGA1UEBhMCVVMxEzAR
# BgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
# Y3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3Rh
# bXAgUENBIDIwMTAwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDk4aZM
# 57RyIQt5osvXJHm9DtWC0/3unAcH0qlsTnXIyjVX9gF/bErg4r25PhdgM/9cT8dm
# 95VTcVrifkpa/rg2Z4VGIwy1jRPPdzLAEBjoYH1qUoNEt6aORmsHFPPFdvWGUNzB
# RMhxXFExN6AKOG6N7dcP2CZTfDlhAnrEqv1yaa8dq6z2Nr41JmTamDu6GnszrYBb
# fowQHJ1S/rboYiXcag/PXfT+jlPP1uyFVk3v3byNpOORj7I5LFGc6XBpDco2LXCO
# Mcg1KL3jtIckw+DJj361VI/c+gVVmG1oO5pGve2krnopN6zL64NF50ZuyjLVwIYw
# XE8s4mKyzbnijYjklqwBSru+cakXW2dg3viSkR4dPf0gz3N9QZpGdc3EXzTdEonW
# /aUgfX782Z5F37ZyL9t9X4C626p+Nuw2TPYrbqgSUei/BQOj0XOmTTd0lBw0gg/w
# EPK3Rxjtp+iZfD9M269ewvPV2HM9Q07BMzlMjgK8QmguEOqEUUbi0b1qGFphAXPK
# Z6Je1yh2AuIzGHLXpyDwwvoSCtdjbwzJNmSLW6CmgyFdXzB0kZSU2LlQ+QuJYfM2
# BjUYhEfb3BvR/bLUHMVr9lxSUV0S2yW6r1AFemzFER1y7435UsSFF5PAPBXbGjfH
# CBUYP3irRbb1Hode2o+eFnJpxq57t7c+auIurQIDAQABo4IB3TCCAdkwEgYJKwYB
# BAGCNxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQUKqdS/mTEmr6CkTxGNSnPEP8v
# BO4wHQYDVR0OBBYEFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMFwGA1UdIARVMFMwUQYM
# KwYBBAGCN0yDfQEBMEEwPwYIKwYBBQUHAgEWM2h0dHA6Ly93d3cubWljcm9zb2Z0
# LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5Lmh0bTATBgNVHSUEDDAKBggrBgEF
# BQcDCDAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYD
# VR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvXzpoYxDBW
# BgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
# bC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYIKwYBBQUH
# AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtp
# L2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDANBgkqhkiG9w0BAQsF
# AAOCAgEAnVV9/Cqt4SwfZwExJFvhnnJL/Klv6lwUtj5OR2R4sQaTlz0xM7U518Jx
# Nj/aZGx80HU5bbsPMeTCj/ts0aGUGCLu6WZnOlNN3Zi6th542DYunKmCVgADsAW+
# iehp4LoJ7nvfam++Kctu2D9IdQHZGN5tggz1bSNU5HhTdSRXud2f8449xvNo32X2
# pFaq95W2KFUn0CS9QKC/GbYSEhFdPSfgQJY4rPf5KYnDvBewVIVCs/wMnosZiefw
# C2qBwoEZQhlSdYo2wh3DYXMuLGt7bj8sCXgU6ZGyqVvfSaN0DLzskYDSPeZKPmY7
# T7uG+jIa2Zb0j/aRAfbOxnT99kxybxCrdTDFNLB62FD+CljdQDzHVG2dY3RILLFO
# Ry3BFARxv2T5JL5zbcqOCb2zAVdJVGTZc9d/HltEAY5aGZFrDZ+kKNxnGSgkujhL
# mm77IVRrakURR6nxt67I6IleT53S0Ex2tVdUCbFpAUR+fKFhbHP+CrvsQWY9af3L
# wUFJfn6Tvsv4O+S3Fb+0zj6lMVGEvL8CwYKiexcdFYmNcP7ntdAoGokLjzbaukz5
# m/8K6TT4JDVnK+ANuOaMmdbhIurwJ0I9JZTmdHRbatGePu1+oDEzfbzL6Xu/OHBE
# 0ZDxyKs6ijoIYn/ZcGNTTY3ugm2lBRDBcQZqELQdVTNYs6FwZvKhggNQMIICOAIB
# ATCB+aGB0aSBzjCByzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
# EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjElMCMGA1UECxMcTWljcm9zb2Z0IEFtZXJpY2EgT3BlcmF0aW9uczEnMCUGA1UE
# CxMeblNoaWVsZCBUU1MgRVNOOjk2MDAtMDVFMC1EOTQ3MSUwIwYDVQQDExxNaWNy
# b3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQBLcI81gxbe
# a1Ex2mFbXx7ck+0g/6CBgzCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
# YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
# Q29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAy
# MDEwMA0GCSqGSIb3DQEBCwUAAgUA6ztqiDAiGA8yMDI1MDEyMjEzMDA1NloYDzIw
# MjUwMTIzMTMwMDU2WjB3MD0GCisGAQQBhFkKBAExLzAtMAoCBQDrO2qIAgEAMAoC
# AQACAithAgH/MAcCAQACAhNTMAoCBQDrPLwIAgEAMDYGCisGAQQBhFkKBAIxKDAm
# MAwGCisGAQQBhFkKAwKgCjAIAgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZIhvcN
# AQELBQADggEBAFPPAGcRiW87HphVusi75RZPNviW/L5xtna2l39IdPR8kMWfQIPo
# QY28nBsVwGZv+d+rCgsNfi1biN1QKq7NH/4jE5y0iY6zbZb/abVscE9Kckid8CNF
# 7De2ERAF6kUdCgmhYqIf3TJI82jixiuyf0KhOiLFjFuBf73CUiu+ZNeTyqUh/gg8
# 8YX9YODJ4TGnfqBwVZmaF98zc3ldyWsx+HmslYYfqwR3ekVNQ3O6+B2dVZUUwI40
# 8mUf99R4d94xtAmBeHIyCk9vU1ahiLVGRFVL5HxvOAbUyMAwTaHkr5Z5sfnzgL8z
# UNIyFXyZ2iRZ8Cu7pbVJxrEdm8rJHWCWEkQxggQNMIIECQIBATCBkzB8MQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
# MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3Nv
# ZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAe+JP1ahWMyo2gABAAAB7zANBglg
# hkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMC8GCSqG
# SIb3DQEJBDEiBCCL/oDCNPYsreZYniIqeA+Lsg9GLIvLHctvgJZctJJqYTCB+gYL
# KoZIhvcNAQkQAi8xgeowgecwgeQwgb0EIPBhKEW4Fo3wUz09NQx2a0DbcdsX8jov
# M5LizHmnyX+jMIGYMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
# bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
# b3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAC
# EzMAAAHviT9WoVjMqNoAAQAAAe8wIgQg0gAB/1IPbTlD8xFiaEAhnfSMtKIw2BX6
# PDXpJZ5MZfowDQYJKoZIhvcNAQELBQAEggIAI7xNd/fjsoqoljUK3jPeIzuF/br4
# PyIlT68WPNldQwOB5jYSsUK3BIsSAKv9epHs+ddBkHY5of0PcrDzQeXip7g2v1/p
# wsVcq5RxscUBT36Y4ZXBxbJE0u1hJ851GoMP4CNxjkQNBTqZAU339/Un1Jg6dj9+
# jIgX6ml1ouMTcPHvYym26K54BcgCIWRBli0ELni+iYhjLZbbYNaLZ7THEFAYBNCe
# UOFViFPkAJAAkLui+lJlME32HKh9AsLHNZDTlXUEmD7mAJrjnBcCVdnYEOew/bf3
# 6ZzZGJmg7E0PyEBqzP9xehS6aOb+paQln01u1a190wXV87jXzMS2gx5gKVuIKM0x
# da5J8SNeo37JEmz+xdwX2BcI9czPdqoa+rcTbBqn7D41QsoTuqJSUdqM7TxTnLC8
# h4XSABruwhHbquGK6XADIfBfoURTPwseFaa913nNyZIyIMXyE30skUp544oGvfPu
# ByGtreCSj4AdLKskS7brDAFEvhcpp1toElZZ9ZCKsIpQeDU3My+eyyHEBpOELgmr
# /vCNK7f4HqjAOcwgu+w7RdE/rXK1UoGAA+mjH+tUzsDx2mGPXoI3/bs8qxU+cjPk
# 3wuGszFts0Fsh14joGgL453oA9Yeplf7rx64OGEyoQMNxHWrxh0Oz3jYD6GOWkmD
# r0B3caaRyvwDPZc=
# SIG # End signature block
