﻿<#
.SYNOPSIS
    Stops the AOS service.

.DESCRIPTION

    Copyright © 2019 Microsoft. All rights reserved.
#>
[CmdletBinding()]
param
(
    [Parameter(Mandatory = $false, HelpMessage = "The path to the directory in which to write log files.")]
    [string]$LogDir
)

$ErrorActionPreference = "Stop"
Import-Module WebAdministration
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\CommonFaultHandling.psm1" -Force

function checkAosKernelFileLocked
{
    Param
    ( 
    [parameter(mandatory=$true,position=0)]
    [string]
    $AosKernelPathParam,
    [parameter(mandatory=$true, position=1)]
    [Int]
    $MaxWaitSecParam 
    )

    $AosKernelLockedFlag = $true

    $StopWatch = [System.Diagnostics.StopWatch]::StartNew()

    while ($AosKernelLockedFlag -and $StopWatch.Elapsed.TotalSeconds -lt $MaxWaitSecParam)
    {
        try
        {
            [IO.File]::OpenWrite($AosKernelPathParam).close()
            $AosKernelLockedFlag = $false
        }
        catch
        {
            Write-ServicingLog "Failed to open $($AosKernelPathParam) for write: $($_.Exception.Message)" -Vrb
            Start-Sleep -Seconds 1
        }
    }

    $StopWatch.Stop()
    return $AosKernelLockedFlag
}

# Initialize exit code.
[int]$ExitCode = 0

# Initialize the log file to use with Write-ServicingLog.
Set-ServicingLog -LogDir $LogDir

try
{
    Write-ServicingLog "Checking Visual C++ runtime version"

    $vcRuntimeDLL = Get-Item "$env:windir\system32\vcruntime140.dll" -ErrorAction SilentlyContinue
    if ($vcRuntimeDLL)
    {
        $vcRuntimeVersion = [System.Version]::new($vcRuntimeDLL.VersionInfo.ProductVersion)
        Write-ServicingLog "Visual C++ version detected: $vcRuntimeVersion "
        if ($vcRuntimeVersion -lt [System.Version]::new("14.36.32532.0"))
        {
            $ExitCode = 1436
            throw "Please upgrade to the latest Visual C++ redistributable package to continue with the installation. Fore more details, visit aka.ms/PreReqChanges"
        }
    }
    else
    {
        $ExitCode = 1436
        throw "Please install Visual C++ redistributable package to continue with the installation. Fore more details, visit aka.ms/PreReqChanges"
    }

    Write-ServicingLog "Stopping AOS..."

    # For non-admin developer machine scenarios, import module to replace functions that
    # will not work within running as administrator.
    if (Test-Path "$($PSScriptRoot)\NonAdminDevToolsInterject.ps1")
    {
        & "$($PSScriptRoot)\NonAdminDevToolsInterject.ps1"
    }

    # Sometimes Get-Process will throw an exception even when called with a
    # SilentlyContinue error action.
    try
    {
        $devenv = Get-Process -Name "devenv" -ErrorAction SilentlyContinue
    }
    catch
    {
        Write-ServicingLog "Warning: Unable to get devenv processes: $($_)" -Vrb
    }

    # If any devenv processes are found, throw an exception.
    if ($devenv)
    {
        throw "Please close all instance of Visual Studio to continue with the installation."
    }

    # Determine if running in admin or non-admin mode.
    $IsAdmin = Test-IsRunningAsAdministrator

    # Get IIS service and start it if it is not already running.
    $IisService = Get-Service -Name "W3SVC"
    if ($IisService -and $IisService.Status -ine "Running")
    {
        Write-ServicingLog "IIS service is not running, starting IIS Service..."
        Start-Service -Name "W3SVC"
        Write-ServicingLog "IIS service started."
    }

    # Get the AOS web site and app pool and make sure they are both started.
    $websiteName = Get-AosWebsiteName
    $appPoolName = Get-AosAppPoolName
    if ((![string]::IsNullOrWhitespace($websiteName)) -and (![string]::IsNullOrWhitespace($appPoolName)))
    {
        Write-ServicingLog "Stopping IIS AOS web site."
        Stop-Website -Name $websiteName

        # Check if in stopped state before stopping to avoid error.
        $AosWebAppPoolState = Get-WebAppPoolState -Name $appPoolName
        if ($AosWebAppPoolState.Value -ine "Stopped")
        {
            Write-ServicingLog "Stopping IIS AOS application pool."
            Stop-WebAppPool -Name $appPoolName
        }

        # Set properties on web site and app pool, so they do not restart during reboots / IISRESETs.
        # Note: Stopping / starting via IIS mananger GUI will set these same properties.
        # Note: This is not available for non-admin users.
        if ($IsAdmin)
        {
            Write-ServicingLog "Disabling IIS auto start properties for the AOS website and application pool." -Vrb
            Set-ItemProperty -Path "IIS:\Sites\$webSiteName" -Name serverAutoStart -Value $false
            Set-ItemProperty -Path "IIS:\AppPools\$appPoolName" -Name autoStart -Value $false
        }

        $productConfigurationPoolName = Get-ProductConfigurationAppPoolName
        if (![string]::IsNullOrWhitespace($productConfigurationPoolName))
        {
            # Check if in stopped state before stopping to avoid error.
            $ProductConfigurationAppPoolState = Get-WebAppPoolState $productConfigurationPoolName
            if ($ProductConfigurationAppPoolState.Value -ine "Stopped")
            {
                Write-ServicingLog "Stopping IIS product configuration application pool."
                Stop-WebAppPool -Name $productConfigurationPoolName
            }

            # Set property on app pool, so it does not restart during reboots / IISRESETs.
            # Note: Stopping / starting via IIS mananger GUI will set the same property.
            # Note: This is not available for non-admin users.
            if ($IsAdmin)
            {
                Write-ServicingLog "Disabling IIS auto start property for the product configuration application pool." -Vrb
                Set-ItemProperty -Path "IIS:\AppPools\$productConfigurationPoolName" -Name autoStart -Value $false
            }
        }
    }

    # Try to get a list of all xppc* processes and attempt to stop them.
    try
    {
        $XppcProcesses = @(Get-Process -Name "xppc*" -ErrorAction SilentlyContinue)
        if ($XppcProcesses.Count -gt 0)
        {
            Write-ServicingLog "Stopping $($XppcProcesses.Count) 'xppc*' processes..." -Vrb
            foreach ($XppcProcess in $XppcProcesses)
            {
                Write-ServicingLog "Stopping $($XppcProcess.Name) process ID $($XppcProcess.Id)..." -Vrb
                try
                {
                    $XppcProcess | Stop-Process -Force
                }
                catch
                {
                    Write-ServicingLog "Warning: Failed to stop $($XppcProcess.Name) process ID $($XppcProcess.Id): $($_)" -Vrb
                }
            }
        }
    }
    catch
    {
        Write-ServicingLog "Warning: Unable to get xppc processes to stop: $($_)" -Vrb
    }

    # Stop the batch service and set the startup type to disabled, so it does not get started on reboot.
    Write-ServicingLog "Stopping and disabling the batch service."
    
    # Logger to pass to the retry function
    $logger = {param($message) Write-ServicingLog $message}

    # Block to kill batch in the last iteration of the retry function
    $killBatchProcess = {
        #Kill the batch process if still running at the end of the retries.
        $batchSvcName = "DynamicsAxBatch"
        $batchProcess = Get-WmiObject Win32_Service | where {$_.name -eq $batchSvcName}
        if($batchProcess)
        {
          $batchProcessInstance = $batchProcess | Format-List | Out-String
          Write-Output "Found Batch service: $batchProcessInstance"

          $batchProcessId = $batchProcess.ProcessId

          if($batchProcessId -gt 0)
          {
            Write-Output "Stopping Batch process ID: $batchProcessId"
            Stop-Process -Id $batchProcessId -Force
            Start-Sleep 5
          }

          #Final attempt to stop/disable batch
          Write-Output "Final attempt to stop and disable batch after killing the batch process"
          Stop-ServiceAndDisableStartup -ServiceName $batchSvcName
        }
        else
        {
          throw "Unable to find batch service"
        }
      }

    # Attempt to stop batch for up to 7 minutes with incrimental backoff of 4n seconds between attempts
    $output=""
    Invoke-RetryWithBackoff -Command { Stop-ServiceAndDisableStartup -ServiceName "DynamicsAxBatch" } `
                            -Logger $logger `
                            -TimeoutSeconds 420 `
                            -RetryIntervalSec 4 `
                            -TimeoutAction $killBatchProcess `
                            -CommandOutput ([ref]$output) `
                            -TreatTimeoutActionAsSuccess

    if($output -ne "")
    {
       Write-ServicingLog $output
    }

    # Get the AOS web root and find the path to the AOS kernel DLL and check that there are no file
    # locks on it. Throw an exception if there is a lock and it does not go away within the timeout.
    $webroot = Get-AosWebSitePhysicalPath
    $AosKernelPath = Join-Path -Path $webroot -ChildPath "bin\AOSKernel.dll"
    if (Test-Path -Path $AosKernelPath -ErrorAction SilentlyContinue)
    {
        Write-ServicingLog "Validating that $($AosKernelPath) is not locked..." -Vrb

        $AosKernelLocked = $true
        $MaxWaitSec = 300

        $AosKernelLocked = checkAosKernelFileLocked -AosKernelPathParam $AosKernelPath -MaxWaitSecParam $MaxWaitSec

        if ($AosKernelLocked)
        {            
            $w3wpName = "w3wp"
            $axhostName = "axhost"
            $batchname = "Batch"
            $w3wpOraxHostTerminated  = $false

            # Get the locking processes
            Get-Process | foreach {
                $processVar = $_;
                $_.Modules | foreach {
                    if($_.FileName -eq $AosKernelPath)
                    {                        
                        # Terminate w3wp or axhost or batch process
                        if ($processVar.Name -eq $w3wpName -or $processVar.Name -eq $axhostName -or $processVar.Name -eq $batchname)
                        {
                            Write-ServicingLog "Terminate the locking process: $($processVar.Name) PID:$($processVar.id) Locked file:$($AosKernelPath), and try to continue." -Vrb
                            Stop-Process -Id $processVar.Id -Force                            
                            $w3wpOraxHostTerminated = $true
                        }
                    }
                }
            }

            if ($w3wpOraxHostTerminated)
            {
                # Check again for file locking.
                Write-ServicingLog "Revalidating that $($AosKernelPath) is not locked..." -Vrb
                $AosKernelLocked = checkAosKernelFileLocked -AosKernelPathParam $AosKernelPath -MaxWaitSecParam $MaxWaitSec
            }

            # still locked
            if ($AosKernelLocked)
            {
                $LockingProcessFound = $false
                # Get the locking processes
                Get-Process | foreach {
                    $processVar = $_;
                    $_.Modules | foreach {
                        if($_.FileName -eq $AosKernelPath)
                        {
                            $LockingProcessFound = $true
                            Write-ServicingLog "Locking process: $($processVar.Name) PID:$($processVar.id) Locked file:$($AosKernelPath)" -Vrb
                        }
                    }
                }

                $ErrorMessage = "File locks found on $($AosKernelPath) for more than $($MaxWaitSec) seconds."
                if ($LockingProcessFound)
                {
                    $ErrorMessage += " Check previous logs for identified locking processes."
                }
                else
                {
                    $ErrorMessage += " No locking process was able to be automatically identified. Please manually check which processes are keeping a lock on the file."
                }

                throw $ErrorMessage
            }
            else
            {
                Write-ServicingLog "No file locks found on $($AosKernelPath) after terminating processes." -Vrb
            }
        }
        else
        {
            Write-ServicingLog "No file locks found on $($AosKernelPath)." -Vrb
        }
    }
    else
    {
        Write-ServicingLog "Warning: No AOS kernel DLL found at: $($AosKernelPath)." -Vrb
    }
}
catch
{
    # Ensure non-zero exit code if an exception is caught and no exit code set.
    if ($ExitCode -eq 0)
    {
        $ExitCode = 1024
    }

    $ErrorMessage = "Error during AOS stop: $($_)"

    Write-ServicingLog $ErrorMessage
    Write-ServicingLog $($_) -Vrb
    Write-ServicingLog "AOS stop script failed with exit code: $($ExitCode)."

    # Use throw to indicate error to AXUpdateInstaller.
    # In case of exceptions, the output is not captured, so only the error message and
    # log file contents will be available for diagnostics.
    throw "$($ErrorMessage) [Log: $(Get-ServicingLog)]"
}

Write-ServicingLog "AOS stop script completed with exit code: $($ExitCode)."
exit $ExitCode
# SIG # Begin signature block
# MIIr5wYJKoZIhvcNAQcCoIIr2DCCK9QCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAul/o7llI5U0Vg
# xmjEnK7FbpDCYGTMMofAhvS4Phff9aCCEW4wggh+MIIHZqADAgECAhM2AAACAO38
# jbec3qFIAAIAAAIAMA0GCSqGSIb3DQEBCwUAMEExEzARBgoJkiaJk/IsZAEZFgNH
# QkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxFTATBgNVBAMTDEFNRSBDUyBDQSAwMTAe
# Fw0yNDExMDgxMjQzMjhaFw0yNTExMDgxMjQzMjhaMCQxIjAgBgNVBAMTGU1pY3Jv
# c29mdCBBenVyZSBDb2RlIFNpZ24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK
# AoIBAQC5L/UPrOpwYjxcoZC0TqqvMF1WUELvwXN+k27SrA5rohJknn7Cgbxg4hGT
# XKqpcdbtsVTN3ZY896SJ20uQ+INL5OVLzpW408nCNTPYg2LtGJbqHUjpNm0hLCJ+
# gO5Jn2T8DDzIJoUijGXj1m+hRLKb2nOIicCED2GuYBmuWXnaY7INmVEaU3peryty
# ZjDuxdyGDuiPURz8lW1SUiDzoszNp1oswVr+WjDvLDUx4HlxPsG8zUjIst0NnJ6o
# z4tNFKaUBDCetcMjQxpCETn29a1CuRddxZLjPHZHfcotr5sh1S6bNQdzVaMNsxV8
# L3wjHb7XJ6ZVm662mHEiPgpyNcLhAgMBAAGjggWKMIIFhjApBgkrBgEEAYI3FQoE
# HDAaMAwGCisGAQQBgjdbAQEwCgYIKwYBBQUHAwMwPQYJKwYBBAGCNxUHBDAwLgYm
# KwYBBAGCNxUIhpDjDYTVtHiE8Ys+hZvdFs6dEoFgg93NZoaUjDICAWQCAQ4wggJ2
# BggrBgEFBQcBAQSCAmgwggJkMGIGCCsGAQUFBzAChlZodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpaW5mcmEvQ2VydHMvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDEu
# YW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUy
# MDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDIuYW1lLmdibC9haWEv
# QlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBS
# BggrBgEFBQcwAoZGaHR0cDovL2NybDMuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAx
# LkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZG
# aHR0cDovL2NybDQuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDCBrQYIKwYBBQUHMAKGgaBsZGFwOi8vL0NO
# PUFNRSUyMENTJTIwQ0ElMjAwMSxDTj1BSUEsQ049UHVibGljJTIwS2V5JTIwU2Vy
# dmljZXMsQ049U2VydmljZXMsQ049Q29uZmlndXJhdGlvbixEQz1BTUUsREM9R0JM
# P2NBQ2VydGlmaWNhdGU/YmFzZT9vYmplY3RDbGFzcz1jZXJ0aWZpY2F0aW9uQXV0
# aG9yaXR5MB0GA1UdDgQWBBST/HE52ZUlmsYqZcZBdrXZ5u4ZnzAOBgNVHQ8BAf8E
# BAMCB4AwRQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEWMBQGA1UEBRMNMjM2MTY3KzUwMzE1NTCCAeYGA1UdHwSCAd0wggHZMIIB
# 1aCCAdGgggHNhj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpaW5mcmEvQ1JM
# L0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwxLmFtZS5nYmwv
# Y3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwyLmFtZS5n
# YmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwzLmFt
# ZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmw0
# LmFtZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGgb1sZGFwOi8v
# L0NOPUFNRSUyMENTJTIwQ0ElMjAwMSgyKSxDTj1CWTJQS0lDU0NBMDEsQ049Q0RQ
# LENOPVB1YmxpYyUyMEtleSUyMFNlcnZpY2VzLENOPVNlcnZpY2VzLENOPUNvbmZp
# Z3VyYXRpb24sREM9QU1FLERDPUdCTD9jZXJ0aWZpY2F0ZVJldm9jYXRpb25MaXN0
# P2Jhc2U/b2JqZWN0Q2xhc3M9Y1JMRGlzdHJpYnV0aW9uUG9pbnQwHwYDVR0jBBgw
# FoAUllGE4Gtve/7YBqvD8oXmKa5q+dQwHwYDVR0lBBgwFgYKKwYBBAGCN1sBAQYI
# KwYBBQUHAwMwDQYJKoZIhvcNAQELBQADggEBAEDd8Wf5RkHsB64vgn2slxDtHzSo
# It9xN/Dm3RdFjNZ0diTUPMgSPYQlSk8nIAfudnB9FLavGlvZLlyUpfrPSuikepj3
# i3pqNEFn6fNdNFv/wHMxv7hQTIDCmuoR1v1rX+w3oeleBPMnN3QmH4ff1NsynyV4
# dZdYgN9Cw9sC/S3pWZpJrbOs7YOM3vqyU6DciHhC4D9i2zByHCF2pu9nYfiQf5A2
# iUZenRvyo1E5rC+UP2VZXa4k7g66W20+zAajIKKIqEmRtWahekMkCcOIHFBY4RDA
# ybgPRSGur4VDAiZPjTXS90wQXrX9CwU20cfiCC6e76F4H95KtQjKYpzuNVAwggjo
# MIIG0KADAgECAhMfAAAAUeqP9pxzDKg7AAAAAABRMA0GCSqGSIb3DQEBCwUAMDwx
# EzARBgoJkiaJk/IsZAEZFgNHQkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxEDAOBgNV
# BAMTB2FtZXJvb3QwHhcNMjEwNTIxMTg0NDE0WhcNMjYwNTIxMTg1NDE0WjBBMRMw
# EQYKCZImiZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQD
# EwxBTUUgQ1MgQ0EgMDEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDJ
# mlIJfQGejVbXKpcyFPoFSUllalrinfEV6JMc7i+bZDoL9rNHnHDGfJgeuRIYO1LY
# /1f4oMTrhXbSaYRCS5vGc8145WcTZG908bGDCWr4GFLc411WxA+Pv2rteAcz0eHM
# H36qTQ8L0o3XOb2n+x7KJFLokXV1s6pF/WlSXsUBXGaCIIWBXyEchv+sM9eKDsUO
# LdLTITHYJQNWkiryMSEbxqdQUTVZjEz6eLRLkofDAo8pXirIYOgM770CYOiZrcKH
# K7lYOVblx22pdNawY8Te6a2dfoCaWV1QUuazg5VHiC4p/6fksgEILptOKhx9c+ia
# piNhMrHsAYx9pUtppeaFAgMBAAGjggTcMIIE2DASBgkrBgEEAYI3FQEEBQIDAgAC
# MCMGCSsGAQQBgjcVAgQWBBQSaCRCIUfL1Gu+Mc8gpMALI38/RzAdBgNVHQ4EFgQU
# llGE4Gtve/7YBqvD8oXmKa5q+dQwggEEBgNVHSUEgfwwgfkGBysGAQUCAwUGCCsG
# AQUFBwMBBggrBgEFBQcDAgYKKwYBBAGCNxQCAQYJKwYBBAGCNxUGBgorBgEEAYI3
# CgMMBgkrBgEEAYI3FQYGCCsGAQUFBwMJBggrBgEFBQgCAgYKKwYBBAGCN0ABAQYL
# KwYBBAGCNwoDBAEGCisGAQQBgjcKAwQGCSsGAQQBgjcVBQYKKwYBBAGCNxQCAgYK
# KwYBBAGCNxQCAwYIKwYBBQUHAwMGCisGAQQBgjdbAQEGCisGAQQBgjdbAgEGCisG
# AQQBgjdbAwEGCisGAQQBgjdbBQEGCisGAQQBgjdbBAEGCisGAQQBgjdbBAIwGQYJ
# KwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMBIGA1UdEwEB/wQI
# MAYBAf8CAQAwHwYDVR0jBBgwFoAUKV5RXmSuNLnrrJwNp4x1AdEJCygwggFoBgNV
# HR8EggFfMIIBWzCCAVegggFToIIBT4YxaHR0cDovL2NybC5taWNyb3NvZnQuY29t
# L3BraWluZnJhL2NybC9hbWVyb290LmNybIYjaHR0cDovL2NybDIuYW1lLmdibC9j
# cmwvYW1lcm9vdC5jcmyGI2h0dHA6Ly9jcmwzLmFtZS5nYmwvY3JsL2FtZXJvb3Qu
# Y3JshiNodHRwOi8vY3JsMS5hbWUuZ2JsL2NybC9hbWVyb290LmNybIaBqmxkYXA6
# Ly8vQ049YW1lcm9vdCxDTj1BTUVSb290LENOPUNEUCxDTj1QdWJsaWMlMjBLZXkl
# MjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9uLERDPUFNRSxE
# Qz1HQkw/Y2VydGlmaWNhdGVSZXZvY2F0aW9uTGlzdD9iYXNlP29iamVjdENsYXNz
# PWNSTERpc3RyaWJ1dGlvblBvaW50MIIBqwYIKwYBBQUHAQEEggGdMIIBmTBHBggr
# BgEFBQcwAoY7aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraWluZnJhL2NlcnRz
# L0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6Ly9jcmwyLmFt
# ZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6
# Ly9jcmwzLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUH
# MAKGK2h0dHA6Ly9jcmwxLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQw
# gaIGCCsGAQUFBzAChoGVbGRhcDovLy9DTj1hbWVyb290LENOPUFJQSxDTj1QdWJs
# aWMlMjBLZXklMjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9u
# LERDPUFNRSxEQz1HQkw/Y0FDZXJ0aWZpY2F0ZT9iYXNlP29iamVjdENsYXNzPWNl
# cnRpZmljYXRpb25BdXRob3JpdHkwDQYJKoZIhvcNAQELBQADggIBAFAQI7dPD+jf
# XtGt3vJp2pyzA/HUu8hjKaRpM3opya5G3ocprRd7vdTHb8BDfRN+AD0YEmeDB5HK
# QoG6xHPI5TXuIi5sm/LeADbV3C2q0HQOygS/VT+m1W7a/752hMIn+L4ZuyxVeSBp
# fwf7oQ4YSZPh6+ngZvBHgfBaVz4O9/wcfw91QDZnTgK9zAh9yRKKls2bziPEnxeO
# ZMVNaxyV0v152PY2xjqIafIkUjK6vY9LtVFjJXenVUAmn3WCPWNFC1YTIIHw/mD2
# cTfPy7QA1pT+GPARAKt0bKtq9aCd/Ym0b5tPbpgCiRtzyb7fbNS1dE740re0COE6
# 7YV2wbeo2sXixzvLftH8L7s9xv9wV+G22qyKt6lmKLjFK1yMw4Ni5fMabcgmzRvS
# jAcbqgp3tk4a8emaaH0rz8MuuIP+yrxtREPXSqL/C5bzMzsikuDW9xH10graZzSm
# PjilzpRfRdu20/9UQmC7eVPZ4j1WNa1oqPHfzET3ChIzJ6Q9G3NPCB+7KwX0OQmK
# yv7IDimj8U/GlsHD1z+EF/fYMf8YXG15LamaOAohsw/ywO6SYSreVW+5Y0mzJutn
# BC9Cm9ozj1+/4kqksrlhZgR/CSxhFH3BTweH8gP2FEISRtShDZbuYymynY1un+Ry
# fiK9+iVTLdD1h/SxyxDpZMtimb4CgJQlMYIZzzCCGcsCAQEwWDBBMRMwEQYKCZIm
# iZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQDEwxBTUUg
# Q1MgQ0EgMDECEzYAAAIA7fyNt5zeoUgAAgAAAgAwDQYJYIZIAWUDBAIBBQCgga4w
# GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisG
# AQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIJLUuU7GzhXklcopCWpFqvVl0a3AJDUf
# 2DbtF7Kv6nXcMEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8AcwBvAGYA
# dKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEA
# kSjvO6jL9f2cTlo8jesL1vnGSfD908CfoiscXFIdvQiYEpb/nW9JPa9RPcBTovIo
# RETCrDHrTQLtH/Me+T08ouCXazQ7DNmG7NkuQpnQYgsZXtBQt72PmrFf6BPjAdr5
# xrLFwtLwNvz8yd935/CRfZxeYYknvGeK2y7lEDRaYl4IYSLTuL8RtYKLI7Ee/8ME
# dVdKTyLZRahDShA+ampbzTNlqoyE/5tUFfCN0kiXiSe5f5V+iyn/58U+/sdDsaZR
# cdBqDParTz6/9yebEOf65lc69Wl25N1HSzXzJ3K4ecXU2ev8hXnWsT+gQYKBijN4
# qM6POHcqmfB12KZDGNiIB6GCF5cwgheTBgorBgEEAYI3AwMBMYIXgzCCF38GCSqG
# SIb3DQEHAqCCF3AwghdsAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFSBgsqhkiG9w0B
# CRABBKCCAUEEggE9MIIBOQIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFlAwQCAQUA
# BCCZ0kqlLtNlZlgwFvD7orikpqKIQuQZGRg3vH4RVDK40gIGZ1rp9Lt3GBMyMDI1
# MDEyMjIwNDI0NS4yNTVaMASAAgH0oIHRpIHOMIHLMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1lcmljYSBP
# cGVyYXRpb25zMScwJQYDVQQLEx5uU2hpZWxkIFRTUyBFU046REMwMC0wNUUwLUQ5
# NDcxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2WgghHtMIIH
# IDCCBQigAwIBAgITMwAAAehQsIDPK3KZTQABAAAB6DANBgkqhkiG9w0BAQsFADB8
# MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
# bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1N
# aWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDAeFw0yMzEyMDYxODQ1MjJaFw0y
# NTAzMDUxODQ1MjJaMIHLMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1lcmljYSBPcGVyYXRpb25zMScwJQYD
# VQQLEx5uU2hpZWxkIFRTUyBFU046REMwMC0wNUUwLUQ5NDcxJTAjBgNVBAMTHE1p
# Y3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2UwggIiMA0GCSqGSIb3DQEBAQUAA4IC
# DwAwggIKAoICAQDhQXdE0WzXG7wzeC9SGdH6eVwdGlF6YgpU7weOFBkpW9yuEmJS
# DE1ADBx/0DTuRBaplSD8CR1QqyQmxRDD/CdvDyeZFAcZ6l2+nlMssmZyC8TPt1GT
# WAUt3GXUU6g0F0tIrFNLgofCjOvm3G0j482VutKS4wZT6bNVnBVsChr2AjmVbGDN
# /6Qs/EqakL5cwpGel1te7UO13dUwaPjOy0Wi1qYNmR8i7T1luj2JdFdfZhMPyqyq
# /NDnZuONSbj8FM5xKBoar12ragC8/1CXaL1OMXBwGaRoJTYtksi9njuq4wDkcAwi
# tCZ5BtQ2NqPZ0lLiQB7O10Bm9zpHWn9x1/HmdAn4koMWKUDwH5sd/zDu4vi887FW
# xm54kkWNvk8FeQ7ZZ0Q5gqGKW4g6revV2IdAxBobWdorqwvzqL70WdsgDU/P5c0L
# 8vYIskUJZedCGHM2hHIsNRyw9EFoSolDM+yCedkz69787s8nIp55icLfDoKw5hak
# 5G6MWF6d71tcNzV9+v9RQKMa6Uwfyquredd5sqXWCXv++hek4A15WybIc6ufT0il
# azKYZvDvoaswgjP0SeLW7mvmcw0FELzF1/uWaXElLHOXIlieKF2i/YzQ6U50K9db
# hnMaDcJSsG0hXLRTy/LQbsOD0hw7FuK0nmzotSx/5fo9g7fCzoFjk3tDEwIDAQAB
# o4IBSTCCAUUwHQYDVR0OBBYEFPo5W8o980kMfRVQba6T34HwelLaMB8GA1UdIwQY
# MBaAFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMF8GA1UdHwRYMFYwVKBSoFCGTmh0dHA6
# Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY3Jvc29mdCUyMFRpbWUt
# U3RhbXAlMjBQQ0ElMjAyMDEwKDEpLmNybDBsBggrBgEFBQcBAQRgMF4wXAYIKwYB
# BQUHMAKGUGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY2VydHMvTWlj
# cm9zb2Z0JTIwVGltZS1TdGFtcCUyMFBDQSUyMDIwMTAoMSkuY3J0MAwGA1UdEwEB
# /wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwDgYDVR0PAQH/BAQDAgeAMA0G
# CSqGSIb3DQEBCwUAA4ICAQCWfcJm2rwXtPi74km6PKAkni9+BWotq+QtDGgeT5F3
# ro7PsIUNKRkUytuGqI8thL3Jcrb03x6DOppYJEA+pb6o2qPjFddO1TLqvSXrYm+O
# gCLL+7+3FmRmfkRu8rHvprab0O19wDbukgO8I5Oi1RegMJl8t5k/UtE0Wb3zAlOH
# nCjLGSzP/Do3ptwhXokk02IvD7SZEBbPboGbtw4LCHsT2pFakpGOBh+ISUMXBf83
# 5CuVNfddwxmyGvNSzyEyEk5h1Vh7tpwP7z7rJ+HsiP4sdqBjj6Avopuf4rxUAfrE
# bV6aj8twFs7WVHNiIgrHNna/55kyrAG9Yt19CPvkUwxYK0uZvPl2WC39nfc0jOTj
# ivC7s/IUozE4tfy3JNkyQ1cNtvZftiX3j5Dt+eLOeuGDjvhJvYMIEkpkV68XLNH7
# +ZBfYa+PmfRYaoFFHCJKEoRSZ3PbDJPBiEhZ9yuxMddoMMQ19Tkyftot6Ez0XhSm
# wjYBq39DvBFWhlyDGBhrU3GteDWiVd9YGSB2WnxuFMy5fbAK6o8PWz8QRMiptXHK
# 3HDBr2wWWEcrrgcTuHZIJTqepNoYlx9VRFvj/vCXaAFcmkW1nk7VE+owaXr5RJjr
# yDq9ubkyDq1mdrF/geaRALXcNZbfNXIkhXzXA6a8CiamcQW/DgmLJpiVQNriZYCH
# IDCCB3EwggVZoAMCAQICEzMAAAAVxedrngKbSZkAAAAAABUwDQYJKoZIhvcNAQEL
# BQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
# EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNV
# BAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eSAyMDEwMB4X
# DTIxMDkzMDE4MjIyNVoXDTMwMDkzMDE4MzIyNVowfDELMAkGA1UEBhMCVVMxEzAR
# BgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
# Y3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3Rh
# bXAgUENBIDIwMTAwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDk4aZM
# 57RyIQt5osvXJHm9DtWC0/3unAcH0qlsTnXIyjVX9gF/bErg4r25PhdgM/9cT8dm
# 95VTcVrifkpa/rg2Z4VGIwy1jRPPdzLAEBjoYH1qUoNEt6aORmsHFPPFdvWGUNzB
# RMhxXFExN6AKOG6N7dcP2CZTfDlhAnrEqv1yaa8dq6z2Nr41JmTamDu6GnszrYBb
# fowQHJ1S/rboYiXcag/PXfT+jlPP1uyFVk3v3byNpOORj7I5LFGc6XBpDco2LXCO
# Mcg1KL3jtIckw+DJj361VI/c+gVVmG1oO5pGve2krnopN6zL64NF50ZuyjLVwIYw
# XE8s4mKyzbnijYjklqwBSru+cakXW2dg3viSkR4dPf0gz3N9QZpGdc3EXzTdEonW
# /aUgfX782Z5F37ZyL9t9X4C626p+Nuw2TPYrbqgSUei/BQOj0XOmTTd0lBw0gg/w
# EPK3Rxjtp+iZfD9M269ewvPV2HM9Q07BMzlMjgK8QmguEOqEUUbi0b1qGFphAXPK
# Z6Je1yh2AuIzGHLXpyDwwvoSCtdjbwzJNmSLW6CmgyFdXzB0kZSU2LlQ+QuJYfM2
# BjUYhEfb3BvR/bLUHMVr9lxSUV0S2yW6r1AFemzFER1y7435UsSFF5PAPBXbGjfH
# CBUYP3irRbb1Hode2o+eFnJpxq57t7c+auIurQIDAQABo4IB3TCCAdkwEgYJKwYB
# BAGCNxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQUKqdS/mTEmr6CkTxGNSnPEP8v
# BO4wHQYDVR0OBBYEFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMFwGA1UdIARVMFMwUQYM
# KwYBBAGCN0yDfQEBMEEwPwYIKwYBBQUHAgEWM2h0dHA6Ly93d3cubWljcm9zb2Z0
# LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5Lmh0bTATBgNVHSUEDDAKBggrBgEF
# BQcDCDAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYD
# VR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvXzpoYxDBW
# BgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
# bC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYIKwYBBQUH
# AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtp
# L2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDANBgkqhkiG9w0BAQsF
# AAOCAgEAnVV9/Cqt4SwfZwExJFvhnnJL/Klv6lwUtj5OR2R4sQaTlz0xM7U518Jx
# Nj/aZGx80HU5bbsPMeTCj/ts0aGUGCLu6WZnOlNN3Zi6th542DYunKmCVgADsAW+
# iehp4LoJ7nvfam++Kctu2D9IdQHZGN5tggz1bSNU5HhTdSRXud2f8449xvNo32X2
# pFaq95W2KFUn0CS9QKC/GbYSEhFdPSfgQJY4rPf5KYnDvBewVIVCs/wMnosZiefw
# C2qBwoEZQhlSdYo2wh3DYXMuLGt7bj8sCXgU6ZGyqVvfSaN0DLzskYDSPeZKPmY7
# T7uG+jIa2Zb0j/aRAfbOxnT99kxybxCrdTDFNLB62FD+CljdQDzHVG2dY3RILLFO
# Ry3BFARxv2T5JL5zbcqOCb2zAVdJVGTZc9d/HltEAY5aGZFrDZ+kKNxnGSgkujhL
# mm77IVRrakURR6nxt67I6IleT53S0Ex2tVdUCbFpAUR+fKFhbHP+CrvsQWY9af3L
# wUFJfn6Tvsv4O+S3Fb+0zj6lMVGEvL8CwYKiexcdFYmNcP7ntdAoGokLjzbaukz5
# m/8K6TT4JDVnK+ANuOaMmdbhIurwJ0I9JZTmdHRbatGePu1+oDEzfbzL6Xu/OHBE
# 0ZDxyKs6ijoIYn/ZcGNTTY3ugm2lBRDBcQZqELQdVTNYs6FwZvKhggNQMIICOAIB
# ATCB+aGB0aSBzjCByzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
# EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjElMCMGA1UECxMcTWljcm9zb2Z0IEFtZXJpY2EgT3BlcmF0aW9uczEnMCUGA1UE
# CxMeblNoaWVsZCBUU1MgRVNOOkRDMDAtMDVFMC1EOTQ3MSUwIwYDVQQDExxNaWNy
# b3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQCMJG4vg0ju
# MOVn2BuKACUvP80FuqCBgzCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
# YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
# Q29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAy
# MDEwMA0GCSqGSIb3DQEBCwUAAgUA6ztzJTAiGA8yMDI1MDEyMjEzMzc0MVoYDzIw
# MjUwMTIzMTMzNzQxWjB3MD0GCisGAQQBhFkKBAExLzAtMAoCBQDrO3MlAgEAMAoC
# AQACAgDqAgH/MAcCAQACAhIfMAoCBQDrPMSlAgEAMDYGCisGAQQBhFkKBAIxKDAm
# MAwGCisGAQQBhFkKAwKgCjAIAgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZIhvcN
# AQELBQADggEBANirlkutR5gmfiaUHaxowgnANjgnDuDOHODkBrBQWP+nyHDr6X+0
# TTxtTeEoL/fObkd1TFMSTAs355oeTWUURB8aLp/o241nJpZD7uW+/V9lwGx5iiiL
# xiUN2KNy6m8OmSEZ1FqR8O08YBfK5yi8dQ3pPDeQ3YLPpjblCFyFCXECTpBJX6nF
# yZN4QtVv94m1FjbAYRFp4wcFD2/hNgi0O/rmIpqQeiTqA2wlHNaHxETdnzg56+UN
# X1oJ/yxoimkkdPOM6lOrq48S2xdlpg23/D7aAIDKSiJWKnI1yBxoaDCU0Aqs9KkD
# VLVfBfKZhdYb++Po+J3nD6lpoE1NRVGh4d0xggQNMIIECQIBATCBkzB8MQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
# MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3Nv
# ZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAehQsIDPK3KZTQABAAAB6DANBglg
# hkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMC8GCSqG
# SIb3DQEJBDEiBCA1ElqUI8M2YNEGIqH9qShqIUchVWHvOrEGXs27x2EnsjCB+gYL
# KoZIhvcNAQkQAi8xgeowgecwgeQwgb0EICrS2sTVAoQggkHR59pNqige0xfJT2J3
# U8W1Sc8H+OsdMIGYMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
# bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
# b3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAC
# EzMAAAHoULCAzytymU0AAQAAAegwIgQgoilORAaco430xd4l1BCq1YO3SnYM876g
# BZvscTxCmVgwDQYJKoZIhvcNAQELBQAEggIAGmXyMXrlBsAt57TqfbsS5plw2Obo
# hcD/FDnCKisufCdotC3YMcIjsY02LtDHT97rZ+WjFWUv/XiNZYnEn5ekf7YAerGt
# AQTJEaEKVWJL2MG9yYnjUraSLWLRnLeCU89GqVnp/sDGuiJqpzCmjnioK2IkMG5Y
# qkAKtGyszRCBXWzvklzEBEKWB9YEbdeemdA63a5yN9ynco7Pk2pQTECgZiEEjwdt
# SbE99YrGcVdsw3IlFWPg3SX/m/OXTSwgNL7H55otpDYQuatnEdzFOP6f8KT6qoK0
# s0fKsUfWnCjhUOfBDD/BhD/4qGWHFuoLt4wkrVt+ktr+6fE7agn/PBkQpOe4qpVB
# 1liZNzhHccR/9fG82KKz7nJl65RXSZYCnl9KXMiekWdxFWPsO/wN30yW4s1/BECU
# 9rlAgDHosvMTCO6TZPtdR/Mn+zNLw+h3QI8ryXiEX8MtPvSTngP8gSumJ1DdPRf7
# RZ7o0PqKL8Czi/9zPHqNJ8wSU0gaspdYz2GTqhOxtoRCiN+Qvf3sfIQe9UzWB9je
# 2y6tMFBGVSKlFE44dAsoQBioLCjvvCMjRuQBJ9haLUcLw0cOoRGJ6jLeP79E7IgX
# XAIArDiwufLspk5h3Om4Cwewpw9VU8seBZ2doDXJ4ZaMTul3ke6Lpw+47zZTr3zI
# QiPWdPXj47R50/o=
# SIG # End signature block
