﻿param
(
    [Parameter(Mandatory=$true)]
    $config,
    [Parameter(Mandatory=$true)]
    $log
)

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking

function RemoveHttpModule($webConfig, $location, $httpModuleName)
{
    Write-Log "Adding remove $httpModuleName HTTP module for $location location"
    $webHttpRemoveModules = $webConfig.SelectNodes("configuration/location[@path='$location']/system.webServer/modules")
    $httpRemoveModule = $webHttpRemoveModules.remove | where { $_.name -eq $httpModuleName }
    if ($httpRemoveModule -eq $null)
    {
        $newAdd = $webConfig.CreateElement("remove")
        $newAdd.SetAttribute("name", "$httpModuleName")
        $webHttpRemoveModules.AppendChild($newAdd) | Out-Null    
    }
}

function ConfigureHttp-WebConfig($settings)
{
    Write-Log "Configuring web.config for HTTP"
    $WebRootPath = $settings.'Infrastructure.WebRoot'

    Write-Log "Setting webHttpBinding security to None"
    [xml]$webConfig = Get-Content "$WebRootPath\web.config"
    $webBindingSecurity = $webConfig.SelectNodes("configuration/location/system.serviceModel/bindings/webHttpBinding/binding/security")

    foreach ($bindingSecurity in $webBindingSecurity)
    {
        $bindingSecurity.mode = "None"
    }

    Write-Log "Adding SecureCookieHttpModule HTTP module"
    $SecureCookieHttpModuleName = "SecureCookieHttpModule"
    $webHttpModules = $webConfig.SelectNodes("configuration/system.webServer/modules")
    $secureCookieModule = $webHttpModules.add | where { $_.name -eq $SecureCookieHttpModuleName }
    if ($secureCookieModule -eq $null)
    {
        $newAdd = $webConfig.CreateElement("add")
        $newAdd.SetAttribute("name", "$SecureCookieHttpModuleName")
        $newAdd.SetAttribute("type", "Microsoft.Dynamics.AX.HttpModule.SecureCookieHttpModule, Microsoft.Dynamics.AX.HttpModule.SecureCookieHttpModule, Version=7.0.0.0, Culture=neutral, PublicKeyToken=31bf3856ad364e35")
        $newAdd.SetAttribute("preCondition", "managedHandler")
        $webHttpModules.AppendChild($newAdd) | Out-Null    
    }

    $webConfig.Save("$WebRootPath\web.config")  
}

function ConfigureHttp-WifServicesConfig($settings)
{
    Write-Log "Configuring wif.services.config for HTTP"
    $WebRootPath = $settings.'Infrastructure.WebRoot'

    Write-Log "Setting requireSsl = false in wif.services.config"
    [xml]$wifServices = Get-Content "$WebRootPath\wif.services.config"
    $cookieHandler = $wifServices.SelectSingleNode("system.identityModel.services/federationConfiguration/cookieHandler")

    $cookieHandler.requireSsl = "false"

    $wifServices.Save("$WebRootPath\wif.services.config")   
}

function ConfigureHttp-IIS($settings)
{
    Write-Log "Adding website HTTP binding" 

    Import-Module WebAdministration

    $AosWebSiteName = $settings.'Infrastructure.ApplicationName'
    $AosWebSite = Get-Website -Name "$AosWebSiteName"
    $existingBinding = Get-WebBinding -Name "$AosWebSiteName" -Protocol "https" -Port 443
    if($existingBinding -eq $null)
    {    
        Write-Log "Https binding is not configured, configure http binding from IIS management console."
    }
    
    $hostheader = $existingBinding.bindingInformation.Split(":") | select -Last 1

    New-WebBinding -Name "$AosWebSiteName" -Force -HostHeader "$hostheader" -IPAddress "*" -port 80 -Protocol "http"
}

Initialize-Log $log
Write-Log "Decoding settings"
$settings = Decode-Settings $config

# in powershell this is case-insensitive
if ($settings.'Infrastructure.HttpProtocol' -eq "Http")
{
    Write-Log "Updating AOS web.config for HTTP"
    ConfigureHttp-WebConfig $settings
    Write-Log "AOS web.config update complete"

    Write-Log "Updating AOS wif.services.config for HTTP"
    ConfigureHttp-WifServicesConfig $settings
    Write-Log "AOS wif.services.config update complete"

<#
    Write-Log "Updating AOS IIS website configuration"
    ConfigureHttp-IIS $settings
    Write-Log "AOS IIS website configuration update complete"
#>

    Restart-IIS

    Write-Log "AOS Http configuration complete, exiting"
}
else
{
    Restart-IIS
    Write-Log "Protocol is set to $($settings.'Infrastructure.HttpProtocol'), exiting."
}
# SIG # Begin signature block
# MIIoRQYJKoZIhvcNAQcCoIIoNjCCKDICAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDjQNFLWqHSacp3
# LtQS1xEVu3IJaldRjfFeqLhyz8ouNqCCDXYwggX0MIID3KADAgECAhMzAAAEhV6Z
# 7A5ZL83XAAAAAASFMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjUwNjE5MTgyMTM3WhcNMjYwNjE3MTgyMTM3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDASkh1cpvuUqfbqxele7LCSHEamVNBfFE4uY1FkGsAdUF/vnjpE1dnAD9vMOqy
# 5ZO49ILhP4jiP/P2Pn9ao+5TDtKmcQ+pZdzbG7t43yRXJC3nXvTGQroodPi9USQi
# 9rI+0gwuXRKBII7L+k3kMkKLmFrsWUjzgXVCLYa6ZH7BCALAcJWZTwWPoiT4HpqQ
# hJcYLB7pfetAVCeBEVZD8itKQ6QA5/LQR+9X6dlSj4Vxta4JnpxvgSrkjXCz+tlJ
# 67ABZ551lw23RWU1uyfgCfEFhBfiyPR2WSjskPl9ap6qrf8fNQ1sGYun2p4JdXxe
# UAKf1hVa/3TQXjvPTiRXCnJPAgMBAAGjggFzMIIBbzAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUuCZyGiCuLYE0aU7j5TFqY05kko0w
# RQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEW
# MBQGA1UEBRMNMjMwMDEyKzUwNTM1OTAfBgNVHSMEGDAWgBRIbmTlUAXTgqoXNzci
# tW2oynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NybC9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3JsMGEG
# CCsGAQUFBwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDovL3d3dy5taWNyb3NvZnQu
# Y29tL3BraW9wcy9jZXJ0cy9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3J0
# MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQELBQADggIBACjmqAp2Ci4sTHZci+qk
# tEAKsFk5HNVGKyWR2rFGXsd7cggZ04H5U4SV0fAL6fOE9dLvt4I7HBHLhpGdE5Uj
# Ly4NxLTG2bDAkeAVmxmd2uKWVGKym1aarDxXfv3GCN4mRX+Pn4c+py3S/6Kkt5eS
# DAIIsrzKw3Kh2SW1hCwXX/k1v4b+NH1Fjl+i/xPJspXCFuZB4aC5FLT5fgbRKqns
# WeAdn8DsrYQhT3QXLt6Nv3/dMzv7G/Cdpbdcoul8FYl+t3dmXM+SIClC3l2ae0wO
# lNrQ42yQEycuPU5OoqLT85jsZ7+4CaScfFINlO7l7Y7r/xauqHbSPQ1r3oIC+e71
# 5s2G3ClZa3y99aYx2lnXYe1srcrIx8NAXTViiypXVn9ZGmEkfNcfDiqGQwkml5z9
# nm3pWiBZ69adaBBbAFEjyJG4y0a76bel/4sDCVvaZzLM3TFbxVO9BQrjZRtbJZbk
# C3XArpLqZSfx53SuYdddxPX8pvcqFuEu8wcUeD05t9xNbJ4TtdAECJlEi0vvBxlm
# M5tzFXy2qZeqPMXHSQYqPgZ9jvScZ6NwznFD0+33kbzyhOSz/WuGbAu4cHZG8gKn
# lQVT4uA2Diex9DMs2WHiokNknYlLoUeWXW1QrJLpqO82TLyKTbBM/oZHAdIc0kzo
# STro9b3+vjn2809D0+SOOCVZMIIHejCCBWKgAwIBAgIKYQ6Q0gAAAAAAAzANBgkq
# hkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
# EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
# IDIwMTEwHhcNMTEwNzA4MjA1OTA5WhcNMjYwNzA4MjEwOTA5WjB+MQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQg
# Q29kZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
# CgKCAgEAq/D6chAcLq3YbqqCEE00uvK2WCGfQhsqa+laUKq4BjgaBEm6f8MMHt03
# a8YS2AvwOMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc6Whe0t+bU7IKLMOv2akr
# rnoJr9eWWcpgGgXpZnboMlImEi/nqwhQz7NEt13YxC4Ddato88tt8zpcoRb0Rrrg
# OGSsbmQ1eKagYw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+lD3v++MrWhAfTVYoonpy
# 4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+EGvKhL1nkkDstrjNYxbc+/jLTswM9
# sbKvkjh+0p2ALPVOVpEhNSXDOW5kf1O6nA+tGSOEy/S6A4aN91/w0FK/jJSHvMAh
# dCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rhKEmdX4jiJV3TIUs+UsS1Vz8k
# A/DRelsv1SPjcF0PUUZ3s/gA4bysAoJf28AVs70b1FVL5zmhD+kjSbwYuER8ReTB
# w3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zdsGbiwZeBe+3W7UvnSSmn
# Eyimp31ngOaKYnhfsi+E11ecXL93KCjx7W3DKI8sj0A3T8HhhUSJxAlMxdSlQy90
# lfdu+HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS4NaIjAsCAwEAAaOCAe0w
# ggHpMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBRIbmTlUAXTgqoXNzcitW2o
# ynUClTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYD
# VR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBRyLToCMZBDuRQFTuHqp8cx0SOJNDBa
# BgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
# bC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3JsMF4GCCsG
# AQUFBwEBBFIwUDBOBggrBgEFBQcwAoZCaHR0cDovL3d3dy5taWNyb3NvZnQuY29t
# L3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3J0MIGfBgNV
# HSAEgZcwgZQwgZEGCSsGAQQBgjcuAzCBgzA/BggrBgEFBQcCARYzaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1hcnljcHMuaHRtMEAGCCsG
# AQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAHAAbwBsAGkAYwB5AF8AcwB0AGEAdABl
# AG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn8oalmOBUeRou09h0ZyKb
# C5YR4WOSmUKWfdJ5DJDBZV8uLD74w3LRbYP+vj/oCso7v0epo/Np22O/IjWll11l
# hJB9i0ZQVdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0bpdS1HXeUOeLpZMlEPXh6
# I/MTfaaQdION9MsmAkYqwooQu6SpBQyb7Wj6aC6VoCo/KmtYSWMfCWluWpiW5IP0
# wI/zRive/DvQvTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvyCInWH8MyGOLwxS3OW560
# STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn+N4sOiBpmLJZiWhub6e3dMNABQam
# ASooPoI/E01mC8CzTfXhj38cbxV9Rad25UAqZaPDXVJihsMdYzaXht/a8/jyFqGa
# J+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw3MYbBL7fQccOKO7eZS/sl/ah
# XJbYANahRr1Z85elCUtIEJmAH9AAKcWxm6U/RXceNcbSoqKfenoi+kiVH6v7RyOA
# 9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sLgOppO6/8MO0ETI7f33Vt
# Y5E90Z1WTk+/gFcioXgRMiF670EKsT/7qMykXcGhiJtXcVZOSEXAQsmbdlsKgEhr
# /Xmfwb1tbWrJUnMTDXpQzTGCGiUwghohAgEBMIGVMH4xCzAJBgNVBAYTAlVTMRMw
# EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVN
# aWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNp
# Z25pbmcgUENBIDIwMTECEzMAAASFXpnsDlkvzdcAAAAABIUwDQYJYIZIAWUDBAIB
# BQCgga4wGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIKGmLh8l1A0faMCDmySNoqus
# XYnPjT+vZhLcLNj4zhGIMEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8A
# cwBvAGYAdKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEB
# BQAEggEAVLRzDgJODGL2kk6cYp+5OFG3RARsiHQhOdoJ9XtugxkaERsmwYI5uNXK
# ZYQP39xn/fAnZqXM2+Q0gNteAvdzj+Grq8LLbMCjkEo++sgUDVzXfXCZxuphaHAG
# F3WaRBmre9VveQqPmEQD4E3Rk95eVt0+TzaJ4Ugpk/9qRwEGuHNeWkXyp3dpfxlo
# X3VyALCguN7iZyP2UlB68K1br+LaPhmkKn8jD02kR3JUwsAd4azLK01nm6vhYJFw
# GrkooMIfzwYMVjfsOIj5QDAy+arHtplFR2TysWeJF8kQh80DvIyVCEPSqaW6wwUl
# MIhLiZF6+1BZyhl54Ecfj6h9nNI3tKGCF68wgherBgorBgEEAYI3AwMBMYIXmzCC
# F5cGCSqGSIb3DQEHAqCCF4gwgheEAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFZBgsq
# hkiG9w0BCRABBKCCAUgEggFEMIIBQAIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFl
# AwQCAQUABCDSoaO5wtduH/aV6Ck4BzZ/NW+4uCGLBVY7V+yK1OfVlgIGaKSibepm
# GBIyMDI1MDkwNDA3MjYzNi4zNlowBIACAfSggdmkgdYwgdMxCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVs
# YW5kIE9wZXJhdGlvbnMgTGltaXRlZDEnMCUGA1UECxMeblNoaWVsZCBUU1MgRVNO
# OjZGMUEtMDVFMC1EOTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBT
# ZXJ2aWNloIIR/jCCBygwggUQoAMCAQICEzMAAAH8GKCvzGlahzoAAQAAAfwwDQYJ
# KoZIhvcNAQELBQAwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
# EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwHhcNMjQw
# NzI1MTgzMTE0WhcNMjUxMDIyMTgzMTE0WjCB0zELMAkGA1UEBhMCVVMxEzARBgNV
# BAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
# c29mdCBDb3Jwb3JhdGlvbjEtMCsGA1UECxMkTWljcm9zb2Z0IElyZWxhbmQgT3Bl
# cmF0aW9ucyBMaW1pdGVkMScwJQYDVQQLEx5uU2hpZWxkIFRTUyBFU046NkYxQS0w
# NUUwLUQ5NDcxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2Uw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQCnUMAovGltBw9Vg8eUkfJb
# tbmUFmMlZYOMx+XlbKG4DKU/8sObaFvjKZSZLKZXQQVFByYzwHKnCOFY/wXvHiI7
# zx370Zn0xCY2HeNIOiGFdeB/ys40QDJ8I9WYnMzvK2y6tn3Ghxu4Qvv4/sSQjCg6
# 4wWVMXYb/O6rHUT7AA2yWk+o+lvw8LnoFDGAkIjUs4XmWTcB48nvw4vdrD812WS9
# 1WXdWAFI9fg1rC3dSBSh+4+f9yn2/AooHC4hZAQVnzfsZdchOyF3Yx+zqhh/FcS2
# aPZIEYSHFWzvERG5YZvRwrpp/LudoqRtkja/VSqzG5m33iuf97LbKe+6eXHRHr9v
# qc2QLWs5MB9aWmwCP9CXPIzq5hNLFhkLZhbMtttGXSVG1LP7hN2lRT+gBlIH5j6z
# XZGqaDOLtFXE1wjaWb/+wISboDrecIkKBi0z4st72lOyGX9Z/w4649BGID6y1OyD
# z0E4b21uVrPaR946Rh/mF0fymEBu464NB+vqzlhrpb69nPoTRmx6fOLQ60x/lEJr
# aEANhWBTEP6CsLwSm19Z5UBaBgJpAWF4039kY1AySTvxXrfKg8F98kQC74HnGVM9
# iiKNR2j01ei8magZePpHxOCyj5A9oAYcrEJsdrVAv0BIwXc6zWOuiAIcaLKR+nV0
# oaeYDnAlPo3HsF52VIOwCQIDAQABo4IBSTCCAUUwHQYDVR0OBBYEFJjxpSuxRrOH
# EHIfv12GJAIv/tB2MB8GA1UdIwQYMBaAFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMF8G
# A1UdHwRYMFYwVKBSoFCGTmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMv
# Y3JsL01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0ElMjAyMDEwKDEpLmNybDBs
# BggrBgEFBQcBAQRgMF4wXAYIKwYBBQUHMAKGUGh0dHA6Ly93d3cubWljcm9zb2Z0
# LmNvbS9wa2lvcHMvY2VydHMvTWljcm9zb2Z0JTIwVGltZS1TdGFtcCUyMFBDQSUy
# MDIwMTAoMSkuY3J0MAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUH
# AwgwDgYDVR0PAQH/BAQDAgeAMA0GCSqGSIb3DQEBCwUAA4ICAQBPuyz9kBnRtNoW
# yuyIlh3/8tDoiYpv0llIqJHD3GzyHCTlj/vOKCf2Aa3B5EKELaUnwZVEeCsWQHEA
# KO64+mtLP9RhfufnbYkpy52da4h4TWnfQFmuo9M6exwP3wNmLpY1MDKadpLuK3mo
# CA0iet4AltNTZwCiwmh5cAevnRiBIaWYkz5w3qZgsAvpNMUy0Vgmukr1MzlHKHl5
# mabpRMmcaDSO3WkX/a7w9J3Pd0PLsMXsoMlp3aofFk5G8zeT1Xxifckjlgs+4uyj
# Ynmzd+lfIJWBH+GrzqDnON31tFHLKILyIsib833mrodZWwJ7JJ62up+wPJhZK3Av
# 3qHLsMjIsvmKHxgUx3QB2a9NGjKWYAO4rATZNAJO8+eOcuTvKklbb23XtjJrhX4m
# XPniwGc9TuQh5hmy9RP5gqDKQ/VAH6n65R1Efp7v1VqLP6J7Basglni1eQMyYvbW
# eHSP7nwTV5wBgO9PoHjIUh6ifED/oaX1ezsktyI8IgLqEZ2WKIQGnJh5QXSiFkWf
# s0pC7zQhnSN3nXVYDZurrH1pSr/MXJ/wSDD526dSPUq02hamrtW4mpqlyRiya+aZ
# gdlcKMrUS6ztXUj5peOsFi6rIz1Cl4VlThTTVgbXm5bBQJqPS5mVqH9EZJgx3zjL
# 0MKo6l94oTo8syTuEueG9w5CufE/qTCCB3EwggVZoAMCAQICEzMAAAAVxedrngKb
# SZkAAAAAABUwDQYJKoZIhvcNAQELBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQI
# EwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3Nv
# ZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmlj
# YXRlIEF1dGhvcml0eSAyMDEwMB4XDTIxMDkzMDE4MjIyNVoXDTMwMDkzMDE4MzIy
# NVowfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcT
# B1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UE
# AxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQDk4aZM57RyIQt5osvXJHm9DtWC0/3unAcH0qlsTnXI
# yjVX9gF/bErg4r25PhdgM/9cT8dm95VTcVrifkpa/rg2Z4VGIwy1jRPPdzLAEBjo
# YH1qUoNEt6aORmsHFPPFdvWGUNzBRMhxXFExN6AKOG6N7dcP2CZTfDlhAnrEqv1y
# aa8dq6z2Nr41JmTamDu6GnszrYBbfowQHJ1S/rboYiXcag/PXfT+jlPP1uyFVk3v
# 3byNpOORj7I5LFGc6XBpDco2LXCOMcg1KL3jtIckw+DJj361VI/c+gVVmG1oO5pG
# ve2krnopN6zL64NF50ZuyjLVwIYwXE8s4mKyzbnijYjklqwBSru+cakXW2dg3viS
# kR4dPf0gz3N9QZpGdc3EXzTdEonW/aUgfX782Z5F37ZyL9t9X4C626p+Nuw2TPYr
# bqgSUei/BQOj0XOmTTd0lBw0gg/wEPK3Rxjtp+iZfD9M269ewvPV2HM9Q07BMzlM
# jgK8QmguEOqEUUbi0b1qGFphAXPKZ6Je1yh2AuIzGHLXpyDwwvoSCtdjbwzJNmSL
# W6CmgyFdXzB0kZSU2LlQ+QuJYfM2BjUYhEfb3BvR/bLUHMVr9lxSUV0S2yW6r1AF
# emzFER1y7435UsSFF5PAPBXbGjfHCBUYP3irRbb1Hode2o+eFnJpxq57t7c+auIu
# rQIDAQABo4IB3TCCAdkwEgYJKwYBBAGCNxUBBAUCAwEAATAjBgkrBgEEAYI3FQIE
# FgQUKqdS/mTEmr6CkTxGNSnPEP8vBO4wHQYDVR0OBBYEFJ+nFV0AXmJdg/Tl0mWn
# G1M1GelyMFwGA1UdIARVMFMwUQYMKwYBBAGCN0yDfQEBMEEwPwYIKwYBBQUHAgEW
# M2h0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5
# Lmh0bTATBgNVHSUEDDAKBggrBgEFBQcDCDAZBgkrBgEEAYI3FAIEDB4KAFMAdQBi
# AEMAQTALBgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV
# 9lbLj+iiXGJo0T2UkFvXzpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3Js
# Lm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAx
# MC0wNi0yMy5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8v
# d3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2
# LTIzLmNydDANBgkqhkiG9w0BAQsFAAOCAgEAnVV9/Cqt4SwfZwExJFvhnnJL/Klv
# 6lwUtj5OR2R4sQaTlz0xM7U518JxNj/aZGx80HU5bbsPMeTCj/ts0aGUGCLu6WZn
# OlNN3Zi6th542DYunKmCVgADsAW+iehp4LoJ7nvfam++Kctu2D9IdQHZGN5tggz1
# bSNU5HhTdSRXud2f8449xvNo32X2pFaq95W2KFUn0CS9QKC/GbYSEhFdPSfgQJY4
# rPf5KYnDvBewVIVCs/wMnosZiefwC2qBwoEZQhlSdYo2wh3DYXMuLGt7bj8sCXgU
# 6ZGyqVvfSaN0DLzskYDSPeZKPmY7T7uG+jIa2Zb0j/aRAfbOxnT99kxybxCrdTDF
# NLB62FD+CljdQDzHVG2dY3RILLFORy3BFARxv2T5JL5zbcqOCb2zAVdJVGTZc9d/
# HltEAY5aGZFrDZ+kKNxnGSgkujhLmm77IVRrakURR6nxt67I6IleT53S0Ex2tVdU
# CbFpAUR+fKFhbHP+CrvsQWY9af3LwUFJfn6Tvsv4O+S3Fb+0zj6lMVGEvL8CwYKi
# excdFYmNcP7ntdAoGokLjzbaukz5m/8K6TT4JDVnK+ANuOaMmdbhIurwJ0I9JZTm
# dHRbatGePu1+oDEzfbzL6Xu/OHBE0ZDxyKs6ijoIYn/ZcGNTTY3ugm2lBRDBcQZq
# ELQdVTNYs6FwZvKhggNZMIICQQIBATCCAQGhgdmkgdYwgdMxCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVs
# YW5kIE9wZXJhdGlvbnMgTGltaXRlZDEnMCUGA1UECxMeblNoaWVsZCBUU1MgRVNO
# OjZGMUEtMDVFMC1EOTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBT
# ZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQBOQSklc5qojbB+oGzDg0tXCpiqqqCBgzCB
# gKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
# EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNV
# BAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMA0GCSqGSIb3DQEBCwUA
# AgUA7GOOuTAiGA8yMDI1MDkwNDA0MDYxN1oYDzIwMjUwOTA1MDQwNjE3WjB3MD0G
# CisGAQQBhFkKBAExLzAtMAoCBQDsY465AgEAMAoCAQACAhswAgH/MAcCAQACAhJD
# MAoCBQDsZOA5AgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkKAwKgCjAI
# AgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZIhvcNAQELBQADggEBABzvgHDhRjv0
# B/lgSD8zAk16pZqbxtGlr6y72VKdfdn4/saFDlrvmMEbZxj05vtHJecU7+WoOFiW
# Wh1jZKEB1cOpIhq0XcbuUwkWWIuoGo2SmnO67/+v7JMGxlLXAx06mm5yWbXF3TMY
# bDa9DmxjOs4KLsnCKCP3+jkwA9FW1YyNqumMLbydpmtL3usIsr1E5rz4QRvgxLA1
# m09LydMRqUSVkZ82Qe8jJZCym8EyeEW7lfl6ET2xcxBIRl9VIcFAzZTp48AG4LWi
# 6bl3p1b1QCFXDzf5NVyn0eNJPqnJu7KqACtcDcQmk7ZwLZ3mqGbf3TEtA39j6ljB
# cb/6Bt8aRNgxggQNMIIECQIBATCBkzB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Eg
# MjAxMAITMwAAAfwYoK/MaVqHOgABAAAB/DANBglghkgBZQMEAgEFAKCCAUowGgYJ
# KoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMC8GCSqGSIb3DQEJBDEiBCBiT3aN8mgf
# tClVaixlxa6rPozXMO6V6vKcOjuDjbQJ4jCB+gYLKoZIhvcNAQkQAi8xgeowgecw
# geQwgb0EIJVCr5C77+H8E5U/jDB5TBse4JSGH5PuGrd3kwJo0S1iMIGYMIGApH4w
# fDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
# ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMd
# TWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTACEzMAAAH8GKCvzGlahzoAAQAA
# AfwwIgQg0edNczsNyFSeR/VJ00Dm9ek8jX5tW9OxAd15CzP9sakwDQYJKoZIhvcN
# AQELBQAEggIALkUGr7sJSv6uVFzpI66rKQWtFNl4yF1V6R/hggC33s+dI2o1dTGX
# P4gQLS/NCDDiXVWJ98xgDyyokKz1lEIS3URDptcODzHlrhgqGWNuT0FJz9JEDMV9
# M6RwOIYvAed5NRYRx8uhvAdosGKO11H6PoiGi5OPAvVUJPZX3WSWXj07s/Ss/DJw
# l0c/y8TfeRqSgAhCtagiYAIRDP4wSrkNCZwEXe153oWfBq8/fhW7FPUclAaHyd8z
# EinG9BlxN4VUAWuGD8jXt5zkZk5c+A4Qo8HTjLhF+hlvv/gTzdubzvD9Pp2MXTjT
# X4vqc3v06PR/cZqXh3naZBebu0ZmvEXFFYLARo3m42yKJYV50iaKTTcUTeuXl3c5
# 9eKyROvrjjoEWTvTBgEGdDYx3SyxCYVbEj8/khCV7XMJn8qnakXrkTNEqQvLabTu
# ah+C9ifod92cqxBQvYXHwObaEyf1YVA1ZMBG9ZyqCQ9RTnlL3eyXSditGDdDPMOg
# /oBio2zOHHcUQKxyQK/sc/bz/Mvl3MA+YHqLbHJGcKJkOStFS/zOgEcQSy+feDql
# 9q1itfuDQ6TAqoq1dBtmzU13OptQa9lPDkqoltFJZxXkUSCRImRgfv6DBeBws4dU
# WN5IB01mfxUqhentHZAByuI0FYbQPxOfO4Ic7JYC1xYLpVDPoE/pu1E=
# SIG # End signature block
