﻿$global:logfile=""
$NEWSETTING = "###CTP8###"
$BatchService = "DynamicsAxBatch"
$PrintService = "DynamicsAXPrintService"
$DataSigningCertThumbprintLegacyKey="DataAccess.DataSigningCertificateThumbprintLegacy"
$DataEncryptionCertThumbprintLegacyKey="DataAccess.DataEncryptionCertificateThumbprintLegacy"
$DataSigningCertThumbprintKey="DataAccess.DataSigningCertificateThumbprint"
$DataEncryptionCertThumbprintKey="DataAccess.DataEncryptionCertificateThumbprint"
$configEncryptor="Microsoft.Dynamics.AX.Framework.ConfigEncryptor.exe"
$InternalServiceCertThumbprints="Infrastructure.InternalServiceCertificateThumbprints"
$DataAccessAxAdminSqlPwd="DataAccess.AxAdminSqlPwd"
$DataAccessAxAdminSqlUser="DataAccess.AxAdminSqlUser"

$ErrorActionPreference="Stop"

function Initialize-Log([string]$log)
{
    if(Test-Path -Path $log)
    {
        Write-Output "Removing the existing log file '$log'."
        Remove-Item -Path $log -Force|Out-Null
    }

    Write-Output "Creating the log file '$log'."
    New-Item -Path $log -ItemType File -Force|out-null
    $global:logfile=$log
}

function Write-Log([string]$message)
{
    $datetime=Get-Date -Format "MM-dd-yyyy:HH:mm:ss"
    Add-Content -Path $global:logfile -Value "$datetime`: $message"|out-null
    Write-Output "$datetime`: $message"
}

function Log-Error([string]$error,[switch]$throw)
{
    Write-Error $error
    if($throw)
    {
        throw $error
    }
}

function Create-Backup([string]$webroot,[string]$backupdir)
{
    $orig_webconfig= Join-Path -Path $webroot -ChildPath "web.config"
    $orig_wifconfig= Join-Path -Path $webroot -ChildPath "wif.config"
    $orig_wifservicesconfig=Join-Path -Path $webroot -ChildPath "wif.services.config"

    $backup_webconfig= Join-Path -Path $backupdir -ChildPath "web.config.backup"
    $backup_wifconfig= Join-Path -Path $backupdir -ChildPath "wif.config.backup"
    $backup_wifservicesconfig=Join-Path -Path $backupdir -ChildPath "wif.services.config.backup"
    
    Copy-Item -Path $orig_webconfig -Destination $backup_webconfig -Force|out-null
    Write-Log "Copied '$orig_webconfig' to '$backup_webconfig."

    Copy-item -Path $orig_wifconfig -Destination $backup_wifconfig -Force|out-null
    Write-Log "Copied '$orig_wifconfig' to '$backup_wifconfig'."

    Copy-item -Path $orig_wifservicesconfig -Destination $backup_wifservicesconfig -Force|out-null
    Write-Log "Copied '$orig_wifservicesconfig' to '$backup_wifservicesconfig'."
}

function Upgrade-Web-Config([string]$webroot,[string]$ctp8configdir)
{
    Decrypt-Config -webroot:$webroot
    Upgrade-App-Settings -webroot:$webroot -parentdir:$ctp8configdir
    Upgrade-Bindings -webroot:$webroot -parentdir:$ctp8configdir
    Upgrade-Service-Behaviors -webroot:$webroot -parentdir:$ctp8configdir
    Update-Machine-Config-Keys -webroot:$webroot -parenrdir:$ctp8configdir

    $ctp7webconfig=Join-Path -Path $webroot -ChildPath "web.config"
    $ctp7webconfigbackup=Join-Path -Path $webroot -ChildPath "web.config.ctp7"
    $ctp8webconfig=Join-Path -Path $webroot -ChildPath "web.config.ctp8"
    $webconfig=Join-Path -Path $webroot -ChildPath "web.config"
    
    Rename-File -from:$ctp8webconfig -to:$webconfig
    Encrypt-Config -webroot:$webroot
}

function Upgrade-Wif-Config([string]$webroot,[string]$ctp8configdir)
{
    $ctp7wifconfig=Join-Path -Path $webroot -ChildPath "wif.config"
    $ctp8_wifconfig_template=Join-Path -Path $ctp8configdir -ChildPath "wif.config"
    $ctp8wifconfig=Join-Path -Path $webroot -ChildPath "wif.config.ctp8"
    Copy-Item -Path $ctp8_wifconfig_template -Destination $ctp8wifconfig -Force |out-null
    Write-Log "Copied the CTP8 wif.config template from '$ctp8_wifconfig_template' to '$ctp8wifconfig'."

    [System.Xml.XmlDocument] $ctp7xd = new-object System.Xml.XmlDocument
    $ctp7xd.Load($ctp7wifconfig)

    [System.Xml.XmlDocument] $ctp8xd = new-object System.Xml.XmlDocument
    $ctp8xd.Load($ctp8wifconfig)

    #region audienceUris
    $ctp7audienceUri=$ctp7xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/audienceUris/add")
    if($ctp7audienceUri -ne $null)
    {
        $ctp7audienceUriValue=$ctp7audienceUri.Attributes.GetNamedItem("value").Value
        $ctp8audienceUri=$ctp8xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/audienceUris/add")
        $ctp8audienceUri.Attributes.GetNamedItem("value").Value=$ctp7audienceUriValue
        Write-Log "setting the audienceUri value to '$ctp7audienceUriValue'."
    }

    $ctp8xd.Save($ctp8wifconfig)
    #endregion

    #region authority
    $ctp8IssuerNameRegistry=$ctp8xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry")
    $ctp7AuthorityNameNodes=$ctp7xd.SelectNodes("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority/@name")
    $ctp8AuthorityNameNodes=$ctp8xd.SelectNodes("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority/@name")

    $ctp7AuthorityNames=@()
    $ctp8AuthorityNames=@()

    foreach($authority in $ctp7AuthorityNameNodes)
    {
        $ctp7AuthorityNames += $authority.'#text'
    }

    foreach($authority in $ctp8AuthorityNameNodes)
    {
        $ctp8AuthorityNames += $authority.'#text'
    }

    $ctp7UniqueAuthorities=$ctp7AuthorityNames|?{-not ($ctp8AuthorityNames -contains $_)}
    $intersection=$ctp7AuthorityNames|?{($ctp8AuthorityNames -contains $_)}
    
    # process the intersections
    foreach($name in $intersection)
    {
        $ctp7authority=$ctp7xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority[@name='$name']")
        $ctp8authoritykeys=$ctp8xd.SelectNodes("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority[@name='$name']/keys")

        # remove the CTP8 authority keys
        $ctp8authoritykeys.RemoveAll()|out-null

        # get the thumbprint value(s) from the ctp7 wif.config and update it in the ctp8 wif.config
        $ctp7authoritykeys=$ctp7xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority[@name='$name']/keys/add")

        # add the authority keys from the CTP7 wif.config
        foreach($authoritykey in $ctp7authoritykeys)
        {
            $thumbprint=$authoritykey.Attributes.GetNamedItem("thumbprint").Value
            $newauthority = $ctp8xd.CreateElement("add")
            $newauthority.SetAttribute("thumbprint",$thumbprint)|out-null
            $ctp8authoritykeys.AppendChild($newauthority)|out-null
            Write-Log "Added an authority key to authority '$name' with thumbprint '$thumbprint'."
        }
    }

    # add the ctp7 only authorities to ctp8
    foreach($name in $ctp7UniqueAuthorities)
    {
        $ctp7Authority=$ctp7xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority[@name='$name']")
        $newauthority=$ctp8xd.CreateElement("authority")
        $newauthority.SetAttribute("name",$ctp7Authority.Attributes.GetNamedItem("name").Value)|out-null
        $newkeys=$ctp8xd.CreateElement("keys")
        $newValidIssuers=$ctp8xd.CreateElement("validIssuers")

        # add thumbprints
        foreach($thumbprint in $ctp7Authority.keys.add)
        {
            $newthumbprint = $ctp8xd.CreateElement("add")
            $newthumbprint.SetAttribute("thumbprint",$thumbprint.Attributes.GetNamedItem("thumbprint").value)|out-null
            $newkeys.AppendChild($newthumbprint)|out-null
        }

        # add valid issuers
        foreach($validIssuer in $ctp7Authority.validIssuers.add)
        {
            $newValidIssuer = $ctp8xd.CreateElement("add")
            $newValidIssuer.SetAttribute("name",$validIssuer.Attributes.GetNamedItem("name").value)|out-null
            $newValidIssuers.AppendChild($newValidIssuer)|out-null
        }

        $newauthority.AppendChild($newkeys)|out-null
        $newauthority.AppendChild($newValidIssuers)|out-null
        $ctp8IssuerNameRegistry.AppendChild($newAuthority)|out-null
    }

    $ctp8xd.Save($ctp8wifconfig)
    
    # set the thumbprints for the AxTokenIssuer authority
    $wc = Join-Path -Path $webroot -ChildPath "web.config"
    [System.Xml.XmlDocument] $wcxd = new-object System.Xml.XmlDocument
    $wcxd.Load($wc)

    $ctp8xd.Load($ctp8wifconfig)
    $trustedThumbprintsNode = $wcxd.SelectSingleNode("/configuration/appSettings/add[@key='Infrastructure.TrustedCertificates']")
    if($trustedThumbprintsNode -ne $null)
    {
        $trustedThumbprints =  $trustedThumbprintsNode.Attributes.GetNamedItem("value").Value
        [string[]]$thumbprints = $trustedThumbprints.Split(";")
        $axTokenIssuerKeys = $ctp8xd.SelectSingleNode("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority[@name='AxTokenIssuer']/keys")
        $axTokenIssuers = $ctp8xd.SelectNodes("/system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry/authority[@name='AxTokenIssuer']/keys/add")

        if($axTokenIssuers -ne $null)
        {
            $axTokenIssuers.RemoveAll()|out-null
        }

        foreach($thumbprint in $thumbprints)
        {
            $newthumbprint = $ctp8xd.CreateElement("add")
            $newthumbprint.SetAttribute("thumbprint",$thumbprint)|out-null
            $axTokenIssuerKeys.AppendChild($newthumbprint)|out-null
        }

        $ctp8xd.Save($ctp8wifconfig)
    }
    #endregion

     Write-Log "Saved the CTP8 wif.config with updated values from CTP7 wif.config."

     # replace the wif.config file
     $wifconfig=Join-Path -Path $webroot -ChildPath "wif.config"
     Rename-File -from:$ctp8wifconfig -to:$wifconfig
}

function Upgrade-Wif-Services-Config([string]$webroot,[string]$ctp8configdir)
{
    $ctp7wifservicesconfig=Join-Path -Path $webroot -ChildPath "wif.services.config"
    $ctp8_wifservicesconfig_template=Join-Path -Path $ctp8configdir -ChildPath "wif.services.config"
    $ctp8wifservicesconfig=Join-Path -Path $webroot -ChildPath "wif.services.config.ctp8"
    Copy-Item -Path $ctp8_wifservicesconfig_template -Destination $ctp8wifservicesconfig -Force |out-null
    Write-Log "Copied the CTP8 wif.services.config template from '$ctp8_wifservicesconfig_template' to '$ctp8wifservicesconfig'."

    [System.Xml.XmlDocument] $ctp7xd = new-object System.Xml.XmlDocument
    $ctp7xd.Load($ctp7wifservicesconfig)

    [System.Xml.XmlDocument] $ctp8xd = new-object System.Xml.XmlDocument
    $ctp8xd.Load($ctp8wifservicesconfig)

    #region wsFederation
    $ctp8wsfederation=$ctp8xd.SelectSingleNode("/system.identityModel.services/federationConfiguration/wsFederation")
    $ctp7wsfederation=$ctp7xd.SelectSingleNode("/system.identityModel.services/federationConfiguration/wsFederation")

    # replace reply url
    $ctp7reply=$ctp7wsfederation.Attributes.GetNamedItem("reply").Value
    $ctp8wsfederation.Attributes.GetNamedItem("reply").Value=$ctp7reply
    Write-Log "Setting wsFederation reply='$ctp7reply'."

    # replace realm
    $ctp7realm=$ctp7wsfederation.Attributes.GetNamedItem("realm").Value
    $ctp8wsfederation.Attributes.GetNamedItem("realm").Value=$ctp7realm
    Write-Log "Setting wsFederation relam='$ctp7realm'."

    # replace issuer
    $ctp7issuer=$ctp7wsfederation.Attributes.GetNamedItem("issuer").Value
    $ctp8wsfederation.Attributes.GetNamedItem("issuer").Value=$ctp7issuer
    Write-Log "Setting wsFederation issuer='$ctp7issuer'."

    #endregion

    #region cookieHandler
    $ctp8cookieHandler=$ctp8xd.SelectSingleNode("/system.identityModel.services/federationConfiguration/cookieHandler")
    $ctp7cookieHandler=$ctp7xd.SelectSingleNode("/system.identityModel.services/federationConfiguration/cookieHandler")

    #replace path
    $ctp7path=$ctp7cookieHandler.Attributes.GetNamedItem("path").Value
    $ctp8cookieHandler.Attributes.GetNamedItem("path").Value=$ctp7path
    Write-Log "Setting cookie handler path='$path'."

    # replace domain
    $ctp7domain=$ctp7cookieHandler.Attributes.GetNamedItem("domain").Value
    $ctp8cookieHandler.Attributes.GetNamedItem("domain").Value=$ctp7domain
    Write-Log "Setting cookie handler domain='$ctp7domain'."
    #endregion

    #region certificateReference
    $ctp8certReference=$ctp8xd.SelectSingleNode("/system.identityModel.services/federationConfiguration/serviceCertificate/certificateReference")
    $ctp7certReference=$ctp7xd.SelectSingleNode("/system.identityModel.services/federationConfiguration/serviceCertificate/certificateReference")

    # replace findValue
    $ctp7findValue=$ctp7certReference.Attributes.GetNamedItem("findValue").Value
    $ctp8certReference.Attributes.GetNamedItem("findValue").Value=$ctp7findValue
    Write-Log "Setting certificateReference findValue='$ctp7findValue'."
    #endregion
    
    $ctp8xd.Save($ctp8wifservicesconfig)
    Write-Log "Saved the CTP8 wif.services.config with updated values from CTP7 wif.services.config."
    
    # replace the wif.services.config file
    $wifservicesconfig=Join-Path -Path $webroot -ChildPath "wif.services.config"
    Rename-File -from:$ctp8wifservicesconfig -to:$wifservicesconfig
}

function Upgrade-App-Settings([string]$webroot,[string]$parentdir)
{
    $ctp7webconfig=Join-Path -Path $webroot -ChildPath "web.config"
    $ctp8_webconfig_template=Join-Path -Path $ctp8configdir -ChildPath "web.config"
    $ctp8webconfig=Join-Path -Path $webroot -ChildPath "web.config.ctp8"
    Copy-Item -Path $ctp8_webconfig_template -Destination $ctp8webconfig -Force |out-null
    Write-Log "Copied the CTP8 web.config template from '$ctp8_webconfig_template' to '$ctp8webconfig'."

    # add/update appsettings
    [System.Xml.XmlDocument] $ctp7xd = new-object System.Xml.XmlDocument
    $ctp7xd.Load($ctp7webconfig)

    [System.Xml.XmlDocument] $ctp8xd = new-object System.Xml.XmlDocument
    $ctp8xd.Load($ctp8webconfig)

    $ctp8appsettings=$ctp8xd.SelectNodes("/configuration/appSettings/add")

    foreach($setting in $ctp8appsettings)
    {
        $ctp8keyname=$setting.Attributes.GetNamedItem("key").Value

        # special handling for the 'DataAccess.DataEncryptionCertificateThumbprintLegacy' and 'DataAccess.DataSigningCertificateThumbprintLegacy' keys
        if($ctp8keyname -eq $DataSigningCertThumbprintLegacyKey)
        {
            # get the legacy signing cert thumbprint from the CTP7 web.config
            $signingcertthumbprintnode=$ctp7xd.SelectSingleNode("/configuration/appSettings/add[@key='$DataSigningCertThumbprintKey']")
            $keyvalue=$signingcertthumbprintnode.Attributes.GetNamedItem("value").Value
        }
        elseif($ctp8keyname -eq $DataEncryptionCertThumbprintLegacyKey)
        {
            # get the legacy encryption cert thumbprint from the CTP7 web.config
            $encryptioncertthumbprintnode=$ctp7xd.SelectSingleNode("/configuration/appSettings/add[@key='$DataEncryptionCertThumbprintKey']")
            $keyvalue=$encryptioncertthumbprintnode.Attributes.GetNamedItem("value").Value
        }
        elseif($ctp8keyname -eq $DataEncryptionCertThumbprintKey -or
                $ctp8keyname -eq $DataSigningCertThumbprintKey
               )
        {
            # if the key is either a encryption or signing cert thumbprint key, continue
            continue
        }
        elseif($ctp8keyname -eq $InternalServiceCertThumbprints)
        {
            # combine the existing thumbrints from the CTP7 web.config and the thumbprint in the ctp8 web.config
            # the ctp8 cert thumbprint should be the third value after the values are combined
            $internalservicecertthumbprintsnode=$ctp7xd.SelectSingleNode("/configuration/appSettings/add[@key='$InternalServiceCertThumbprints']")
            $ctp7value=$internalservicecertthumbprintsnode.Attributes.GetNamedItem("value").Value
            $ctp8value=$setting.Attributes.GetNamedItem("value").Value
            $keyvalue="$ctp7value`;$ctp8value"
        }
        elseif($ctp8keyname -eq $DataAccessAxAdminSqlUser)
        {
            $keyvalue=$ctp7xd.SelectSingleNode("/configuration/appSettings/add[@key='DataAccess.SqlUser']").Attributes.GetNamedItem("value").Value
        }
        elseif($ctp8keyname -eq $DataAccessAxAdminSqlPwd)
        {
            $keyvalue=$ctp7xd.SelectSingleNode("/configuration/appSettings/add[@key='DataAccess.SqlPwd']").Attributes.GetNamedItem("value").Value
        }
        else
        {
            # if the key exists in the ctp7 web.config use its value
            $ctp7keyname=$ctp7xd.SelectSingleNode("/configuration/appSettings/add[@key='$ctp8keyname']")
            if($ctp7keyname -eq $null)
            {
                # we found a new CTP8 appsetting. Update the key value to indicate it
                # $keyvalue=$NEWSETTING
            }
            else
            {
                # use the value from the ctp7 appsetting in the ctp8 appsetting
                $keyvalue=$ctp7keyname.Attributes.GetNamedItem("value").Value
            }
        }

        $setting.Attributes.GetNamedItem("value").Value=$keyvalue
        Write-Log "Setting '$ctp8keyname=$keyvalue'."
    }
    
    $ctp8xd.Save($ctp8webconfig)
    Write-Log "Saved the CTP8 web.config with the updated appsettings."
}

function Upgrade-Service-Behaviors([string]$webroot,[string]$parentdir)
{
    $ctp7webconfig=Join-Path -Path $webroot -ChildPath "web.config"
    $ctp8webconfig=Join-Path -Path $webroot -ChildPath "web.config.ctp8"

    [System.Xml.XmlDocument] $ctp7xd = new-object System.Xml.XmlDocument
    $ctp7xd.Load($ctp7webconfig)

    [System.Xml.XmlDocument] $ctp8xd = new-object System.Xml.XmlDocument
    $ctp8xd.Load($ctp8webconfig)

    # retrieve the service certificate thumbprint
    $servicecertthumbprint=Get-ServiceBehavior-Thumbprint -xmldoc:$ctp7xd

    if([system.string]::IsNullOrEmpty($servicecertthumbprint))
    {
        Log-Error "Unable to find the certificate thumbprint that is required to configure the service behaviors." -throw
    }

    # set the service certificate thumbprint to the ctp7 value
    $ctp8servicecertificates=$ctp8xd.SelectNodes("/configuration/location/system.serviceModel/behaviors/serviceBehaviors/behavior/serviceCredentials/serviceCertificate")
    foreach($servicecertificate in $ctp8servicecertificates)
    {
        $servicecertificate.Attributes.GetNamedItem("findValue").Value=$servicecertthumbprint
    }

    Write-Log "Setting the service behavior certificate thumbprint to '$servicecertthumbprint'."
    $ctp8xd.Save($ctp8webconfig)
}

function Upgrade-Bindings([string]$webroot,[string]$parentdir)
{
    # open both files
    $ctp7webconfig=Join-Path -Path $webroot -ChildPath "web.config"
    $ctp8webconfig=Join-Path -Path $webroot -ChildPath "web.config.ctp8"

    [System.Xml.XmlDocument] $ctp7xd = new-object System.Xml.XmlDocument
    $ctp7xd.Load($ctp7webconfig)

    [System.Xml.XmlDocument] $ctp8xd = new-object System.Xml.XmlDocument
    $ctp8xd.Load($ctp8webconfig)

    #region customBinding
    $ctp8customBindings = $ctp8xd.SelectNodes("/configuration/location/system.serviceModel/bindings/customBinding")
    foreach($binding in $ctp8customBindings)
    {
        $name=$binding.name
        $ctp7CustomBinding = $ctp7xd.SelectNodes("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']")
        if($ctp7CustomBinding -ne $null)
        {
            $ctp7CustomBinding.sendTimeout = $binding.sendTimeout
            $ctp7CustomBinding.receiveTimeout = $binding.receiveTimeout
        


        # update security
        $ctp8Security = $ctp8xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']/security")
        $ctp7Security = $ctp7xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']/security")
     
        $ctp8Security.authenticationMode = $ctp7Security.authenticationMode
      
     

        # update issuedTokenParameters
        $ctp8IssuedTokenParameters = $ctp8xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']/security/issuedTokenParameters")
        $ctp7IssuedTokenParameters = $ctp7xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']/security/issuedTokenParameters")
        $ctp8IssuedTokenParameters.keyType = $ctp7IssuedTokenParameters.keyType
        $ctp8IssuedTokenParameters.tokenType = $ctp7IssuedTokenParameters.tokenType

        # update textMessageEncoding
        $ctp8textMessageEncodingReaderQuotas = $ctp8xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']/textMessageEncoding/readerQuotas")
        $ctp7textMessageEncodingReaderQuotas = $ctp7xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/customBinding/binding[@name='$name']/textMessageEncoding/readerQuotas")
        $ctp8textMessageEncodingReaderQuotas.maxDepth = $ctp7textMessageEncodingReaderQuotas.maxDepth 
        $ctp8textMessageEncodingReaderQuotas.maxStringContentLength = $ctp7textMessageEncodingReaderQuotas.maxStringContentLength
        $ctp8textMessageEncodingReaderQuotas.maxArrayLength = $ctp7textMessageEncodingReaderQuotas.maxArrayLength
        $ctp8textMessageEncodingReaderQuotas.maxBytesPerRead = $ctp7textMessageEncodingReaderQuotas.maxBytesPerRead
        $ctp8textMessageEncodingReaderQuotas.maxNameTableCharCount = $ctp7textMessageEncodingReaderQuotas.maxNameTableCharCount
        }
    }
    #endregion

    #region webHttpBinding
    $ctp8webHttpBinding = $ctp8xd.SelectNodes("/configuration/location/system.serviceModel/bindings/webHttpBinding")
    foreach($binding in $ctp8webHttpBinding)
    {
        $name=$binding.name
        $ctp7webHttpBinding = $ctp7xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/webHttpBinding/binding[@name='$name']")
        if($ctp7webHttpBinding -ne $null)
        {
            $binding.allowCookies = $ctp7webHttpBinding.allowCookies
            $binding.maxReceivedMessageSize = $ctp7webHttpBinding.maxReceivedMessageSize
            if($ctp7webHttpBinding.Attributes["contentTypeMapper"] -ne $null)
            {
                $binding.contentTypeMapper = $ctp7webHttpBinding.contentTypeMapper
            }
        

        # update security
        $ctp8webHttpBindingSecurity = $ctp8xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/webHttpBinding/binding[@name='$name']/security")
        $ctp7webHttpBindingSecurity = $ctp7xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/webHttpBinding/binding[@name='$name']/security")
        $ctp8webHttpBindingSecurity.mode = $ctp7webHttpBindingSecurity.mode

        # update readerQuotas
        $ctp8webHttpBindingReaderQuotas = $ctp8xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/webHttpBinding/binding[@name='$name']/readerQuotas")
        $ctp7webHttpBindingReaderQuotas = $ctp7xd.SelectSingleNode("/configuration/location/system.serviceModel/bindings/webHttpBinding/binding[@name='$name']/readerQuotas")
        $ctp8webHttpBindingReaderQuotas.maxStringContentLength = $ctp7webHttpBindingReaderQuotas.maxStringContentLength
        }
    }
    #endregion

    $ctp8xd.Save($ctp8webconfig)
    Write-Log "Upgraded the bindings."
}

function Update-Machine-Config-Keys([string]$webroot,[string]$parenrdir)
{
    $ctp7webconfig=Join-Path -Path $webroot -ChildPath "web.config"
    $ctp8webconfig=Join-Path -Path $webroot -ChildPath "web.config.ctp8"

    # load the web.config files
    [System.Xml.XmlDocument] $ctp7xd = new-object System.Xml.XmlDocument
    $ctp7xd.Load($ctp7webconfig)

    [System.Xml.XmlDocument] $ctp8xd = new-object System.Xml.XmlDocument
    $ctp8xd.Load($ctp8webconfig)

    # fetch the ctp7 machineKey values
    $ctp7machineKey = $ctp7xd.SelectSingleNode("/configuration/location/system.web/machineKey")
    $decryption=$ctp7machineKey.Attributes.GetNamedItem("decryption").Value
    $decryptionKey = $ctp7machineKey.Attributes.GetNamedItem("decryptionKey").Value
    $validationKey = $ctp7machineKey.Attributes.GetNamedItem("validationKey").Value
    $validation = $ctp7machineKey.Attributes.GetNamedItem("validation").Value

    # update the ctp8 machineKey values
    $ctp8machineKey=$ctp8xd.SelectSingleNode("/configuration/location/system.web/machineKey")
    $ctp8machineKey.Attributes.GetNamedItem("decryption").Value=$decryption
    $ctp8machineKey.Attributes.GetNamedItem("decryptionKey").Value=$decryptionKey
    $ctp8machineKey.Attributes.GetNamedItem("validation").Value=$validation
    $ctp8machineKey.Attributes.GetNamedItem("validationKey").Value=$validationKey

    # log
    Write-Log "Setting the machine decryption value to '$decryption'."
    Write-Log "Setting the machine decryption key value to '$decryptionKey'."
    Write-Log "Setting the machine validation value to '$validation'."
    Write-Log "Setting the machine validation key value to '$validationKey'."

    # save the ctp8 web.config with the updated values
    $ctp8xd.Save($ctp8webconfig)
}

function Get-ServiceBehavior-Thumbprint($xmldoc)
{
    # well-known service behavior names
    $behaviornames=@("AxILSessionServiceBehaviour","AxInteractiveILSessionServiceBehaviour","InteractionServiceBehavior","ODataQueryServiceBehavior")

    # try to get the service behavior thumbprint using one of the known behavior names
    foreach($name in $behaviornames)
    {
        $thumbprint=$xmldoc.SelectSingleNode("/configuration/location/system.serviceModel/behaviors/serviceBehaviors/behavior[@name='$name']/serviceCredentials/serviceCertificate/@findValue").value
        if(![system.string]::IsNullOrEmpty($thumbprint))
        {
            return $thumbprint
        }
    }

    return [string]::Empty
}

function Rename-File([string]$from,[string]$to)
{
    Move-Item -Path $from -Destination $to -Force|out-null
    Write-Log "Renamed file '$from' to '$to'."
}

function Validate
{
    Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
    
    $aosWebsiteName = Get-AosWebSiteName
    if(!(Is-Website-Stopped -name:$aosWebsiteName))
    {
        Log-Error "Cannot perform the upgrade as the AOS website '$aosWebsiteName' is started. Stop the website and retry." -throw
    }

    if(!(Is-Service-Stopped -name:$BatchService))
    {
        Log-Error "Cannot perform the upgrade as the NT service '$BatchService' is running. Stop the service and retry." -throw
    }

    if(!(Is-Service-Stopped -name:$PrintService))
    {
        Log-Error "Cannot perform the upgrade as the NT service '$PrintService' is running. Stop the service and retry." -throw
    }
}

function Is-Website-Stopped([string]$name)
{
    Import-Module WebAdministration
    Write-Log "Checking the state of the web site '$name'."|Out-Null
    $website=Get-Website -Name:$name -ErrorAction SilentlyContinue
    if($website -eq $null)
    {
        return $true
    }

    if($website.State -eq "Started")
    {
        return $false
    }

    return $true
}

function Is-Service-Stopped([string]$name)
{
    Write-Log "Checking the status of the service '$name'."|out-null
    $service=Get-Service -Name:$name -ErrorAction SilentlyContinue
    if($service -eq $null)
    {
        return $true
    }

    if($service.Status -eq "Running")
    {
        return $false
    }

    return $true
}

function Decrypt-Config([string]$webroot)
{
    $command = Join-Path -Path "$webroot\bin" -ChildPath $configEncryptor
    if(!(Test-Path -Path $command))
    {
        Log-Error "Cannot find the Microsoft.Dynamics.AX.Framework.ConfigEncryptor.exe at '$webroot\bin\'." -throw
    }

    $webconfig=Join-Path -Path $webroot -ChildPath "Web.config"
    $commandParameter = " -decrypt `"$webconfig`""
    $logdir=[System.IO.Path]::GetDirectoryName($global:logfile)
    $stdOut=Join-Path -Path $logdir -ChildPath "config_decrypt.log"
    $stdErr= Join-Path -Path $logdir -ChildPath "config_decrypt.error.log"
    Start-Process $command $commandParameter -PassThru -Wait -RedirectStandardOutput $stdOut -RedirectStandardError $stdErr

    $decryptError = Get-Content $stdErr
    if ($decryptError -ne $null) {
        Log-Error $decryptError -throw
    }

    Write-Log "Finished decrypting the CTP7 web.config."
}

function Encrypt-Config([string]$webroot)
{
    $command = Join-Path -Path "$PSScriptRoot\EncryptionTool" -ChildPath $configEncryptor
    if(!(Test-Path -Path $command))
    {
        Log-Error "Cannot find the CTP8 Microsoft.Dynamics.AX.Framework.ConfigEncryptor.exe at '$PSScriptRoot\EncryptionTool\'." -throw
    }

    $webconfig=Join-Path -Path $webroot -ChildPath "Web.config"
    $commandParameter = " -encrypt `"$webconfig`""
    $logdir=[System.IO.Path]::GetDirectoryName($global:logfile)
    $stdOut=Join-Path -Path $logdir -ChildPath "config_encrypt.log"
    $stdErr= Join-Path -Path $logdir -ChildPath "config_encrypt.error.log"
    Start-Process $command $commandParameter -PassThru -Wait -RedirectStandardOutput $stdOut -RedirectStandardError $stdErr

    $encryptError = Get-Content $stdErr
    if ($encryptError -ne $null) {
        Log-Error $encryptError -throw
    }

    Write-Log "Finished encrypting the CTP8 web.config."
}

Export-ModuleMember -Function Initialize-Log,Write-Log,Write-Error,Validate,Create-Backup,Upgrade-Web-Config,Upgrade-Wif-Config,Upgrade-Wif-Services-Config -Variable $logfile

# SIG # Begin signature block
# MIIr/wYJKoZIhvcNAQcCoIIr8DCCK+wCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAFtoHbPK5XPz6U
# yKSkeOH9OrU39SSz6puyJiuuoTo186CCEW4wggh+MIIHZqADAgECAhM2AAACAO38
# jbec3qFIAAIAAAIAMA0GCSqGSIb3DQEBCwUAMEExEzARBgoJkiaJk/IsZAEZFgNH
# QkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxFTATBgNVBAMTDEFNRSBDUyBDQSAwMTAe
# Fw0yNDExMDgxMjQzMjhaFw0yNTExMDgxMjQzMjhaMCQxIjAgBgNVBAMTGU1pY3Jv
# c29mdCBBenVyZSBDb2RlIFNpZ24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK
# AoIBAQC5L/UPrOpwYjxcoZC0TqqvMF1WUELvwXN+k27SrA5rohJknn7Cgbxg4hGT
# XKqpcdbtsVTN3ZY896SJ20uQ+INL5OVLzpW408nCNTPYg2LtGJbqHUjpNm0hLCJ+
# gO5Jn2T8DDzIJoUijGXj1m+hRLKb2nOIicCED2GuYBmuWXnaY7INmVEaU3peryty
# ZjDuxdyGDuiPURz8lW1SUiDzoszNp1oswVr+WjDvLDUx4HlxPsG8zUjIst0NnJ6o
# z4tNFKaUBDCetcMjQxpCETn29a1CuRddxZLjPHZHfcotr5sh1S6bNQdzVaMNsxV8
# L3wjHb7XJ6ZVm662mHEiPgpyNcLhAgMBAAGjggWKMIIFhjApBgkrBgEEAYI3FQoE
# HDAaMAwGCisGAQQBgjdbAQEwCgYIKwYBBQUHAwMwPQYJKwYBBAGCNxUHBDAwLgYm
# KwYBBAGCNxUIhpDjDYTVtHiE8Ys+hZvdFs6dEoFgg93NZoaUjDICAWQCAQ4wggJ2
# BggrBgEFBQcBAQSCAmgwggJkMGIGCCsGAQUFBzAChlZodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpaW5mcmEvQ2VydHMvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDEu
# YW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUy
# MDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDIuYW1lLmdibC9haWEv
# QlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBS
# BggrBgEFBQcwAoZGaHR0cDovL2NybDMuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAx
# LkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZG
# aHR0cDovL2NybDQuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDCBrQYIKwYBBQUHMAKGgaBsZGFwOi8vL0NO
# PUFNRSUyMENTJTIwQ0ElMjAwMSxDTj1BSUEsQ049UHVibGljJTIwS2V5JTIwU2Vy
# dmljZXMsQ049U2VydmljZXMsQ049Q29uZmlndXJhdGlvbixEQz1BTUUsREM9R0JM
# P2NBQ2VydGlmaWNhdGU/YmFzZT9vYmplY3RDbGFzcz1jZXJ0aWZpY2F0aW9uQXV0
# aG9yaXR5MB0GA1UdDgQWBBST/HE52ZUlmsYqZcZBdrXZ5u4ZnzAOBgNVHQ8BAf8E
# BAMCB4AwRQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEWMBQGA1UEBRMNMjM2MTY3KzUwMzE1NTCCAeYGA1UdHwSCAd0wggHZMIIB
# 1aCCAdGgggHNhj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpaW5mcmEvQ1JM
# L0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwxLmFtZS5nYmwv
# Y3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwyLmFtZS5n
# YmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwzLmFt
# ZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmw0
# LmFtZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGgb1sZGFwOi8v
# L0NOPUFNRSUyMENTJTIwQ0ElMjAwMSgyKSxDTj1CWTJQS0lDU0NBMDEsQ049Q0RQ
# LENOPVB1YmxpYyUyMEtleSUyMFNlcnZpY2VzLENOPVNlcnZpY2VzLENOPUNvbmZp
# Z3VyYXRpb24sREM9QU1FLERDPUdCTD9jZXJ0aWZpY2F0ZVJldm9jYXRpb25MaXN0
# P2Jhc2U/b2JqZWN0Q2xhc3M9Y1JMRGlzdHJpYnV0aW9uUG9pbnQwHwYDVR0jBBgw
# FoAUllGE4Gtve/7YBqvD8oXmKa5q+dQwHwYDVR0lBBgwFgYKKwYBBAGCN1sBAQYI
# KwYBBQUHAwMwDQYJKoZIhvcNAQELBQADggEBAEDd8Wf5RkHsB64vgn2slxDtHzSo
# It9xN/Dm3RdFjNZ0diTUPMgSPYQlSk8nIAfudnB9FLavGlvZLlyUpfrPSuikepj3
# i3pqNEFn6fNdNFv/wHMxv7hQTIDCmuoR1v1rX+w3oeleBPMnN3QmH4ff1NsynyV4
# dZdYgN9Cw9sC/S3pWZpJrbOs7YOM3vqyU6DciHhC4D9i2zByHCF2pu9nYfiQf5A2
# iUZenRvyo1E5rC+UP2VZXa4k7g66W20+zAajIKKIqEmRtWahekMkCcOIHFBY4RDA
# ybgPRSGur4VDAiZPjTXS90wQXrX9CwU20cfiCC6e76F4H95KtQjKYpzuNVAwggjo
# MIIG0KADAgECAhMfAAAAUeqP9pxzDKg7AAAAAABRMA0GCSqGSIb3DQEBCwUAMDwx
# EzARBgoJkiaJk/IsZAEZFgNHQkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxEDAOBgNV
# BAMTB2FtZXJvb3QwHhcNMjEwNTIxMTg0NDE0WhcNMjYwNTIxMTg1NDE0WjBBMRMw
# EQYKCZImiZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQD
# EwxBTUUgQ1MgQ0EgMDEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDJ
# mlIJfQGejVbXKpcyFPoFSUllalrinfEV6JMc7i+bZDoL9rNHnHDGfJgeuRIYO1LY
# /1f4oMTrhXbSaYRCS5vGc8145WcTZG908bGDCWr4GFLc411WxA+Pv2rteAcz0eHM
# H36qTQ8L0o3XOb2n+x7KJFLokXV1s6pF/WlSXsUBXGaCIIWBXyEchv+sM9eKDsUO
# LdLTITHYJQNWkiryMSEbxqdQUTVZjEz6eLRLkofDAo8pXirIYOgM770CYOiZrcKH
# K7lYOVblx22pdNawY8Te6a2dfoCaWV1QUuazg5VHiC4p/6fksgEILptOKhx9c+ia
# piNhMrHsAYx9pUtppeaFAgMBAAGjggTcMIIE2DASBgkrBgEEAYI3FQEEBQIDAgAC
# MCMGCSsGAQQBgjcVAgQWBBQSaCRCIUfL1Gu+Mc8gpMALI38/RzAdBgNVHQ4EFgQU
# llGE4Gtve/7YBqvD8oXmKa5q+dQwggEEBgNVHSUEgfwwgfkGBysGAQUCAwUGCCsG
# AQUFBwMBBggrBgEFBQcDAgYKKwYBBAGCNxQCAQYJKwYBBAGCNxUGBgorBgEEAYI3
# CgMMBgkrBgEEAYI3FQYGCCsGAQUFBwMJBggrBgEFBQgCAgYKKwYBBAGCN0ABAQYL
# KwYBBAGCNwoDBAEGCisGAQQBgjcKAwQGCSsGAQQBgjcVBQYKKwYBBAGCNxQCAgYK
# KwYBBAGCNxQCAwYIKwYBBQUHAwMGCisGAQQBgjdbAQEGCisGAQQBgjdbAgEGCisG
# AQQBgjdbAwEGCisGAQQBgjdbBQEGCisGAQQBgjdbBAEGCisGAQQBgjdbBAIwGQYJ
# KwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMBIGA1UdEwEB/wQI
# MAYBAf8CAQAwHwYDVR0jBBgwFoAUKV5RXmSuNLnrrJwNp4x1AdEJCygwggFoBgNV
# HR8EggFfMIIBWzCCAVegggFToIIBT4YxaHR0cDovL2NybC5taWNyb3NvZnQuY29t
# L3BraWluZnJhL2NybC9hbWVyb290LmNybIYjaHR0cDovL2NybDIuYW1lLmdibC9j
# cmwvYW1lcm9vdC5jcmyGI2h0dHA6Ly9jcmwzLmFtZS5nYmwvY3JsL2FtZXJvb3Qu
# Y3JshiNodHRwOi8vY3JsMS5hbWUuZ2JsL2NybC9hbWVyb290LmNybIaBqmxkYXA6
# Ly8vQ049YW1lcm9vdCxDTj1BTUVSb290LENOPUNEUCxDTj1QdWJsaWMlMjBLZXkl
# MjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9uLERDPUFNRSxE
# Qz1HQkw/Y2VydGlmaWNhdGVSZXZvY2F0aW9uTGlzdD9iYXNlP29iamVjdENsYXNz
# PWNSTERpc3RyaWJ1dGlvblBvaW50MIIBqwYIKwYBBQUHAQEEggGdMIIBmTBHBggr
# BgEFBQcwAoY7aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraWluZnJhL2NlcnRz
# L0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6Ly9jcmwyLmFt
# ZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6
# Ly9jcmwzLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUH
# MAKGK2h0dHA6Ly9jcmwxLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQw
# gaIGCCsGAQUFBzAChoGVbGRhcDovLy9DTj1hbWVyb290LENOPUFJQSxDTj1QdWJs
# aWMlMjBLZXklMjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9u
# LERDPUFNRSxEQz1HQkw/Y0FDZXJ0aWZpY2F0ZT9iYXNlP29iamVjdENsYXNzPWNl
# cnRpZmljYXRpb25BdXRob3JpdHkwDQYJKoZIhvcNAQELBQADggIBAFAQI7dPD+jf
# XtGt3vJp2pyzA/HUu8hjKaRpM3opya5G3ocprRd7vdTHb8BDfRN+AD0YEmeDB5HK
# QoG6xHPI5TXuIi5sm/LeADbV3C2q0HQOygS/VT+m1W7a/752hMIn+L4ZuyxVeSBp
# fwf7oQ4YSZPh6+ngZvBHgfBaVz4O9/wcfw91QDZnTgK9zAh9yRKKls2bziPEnxeO
# ZMVNaxyV0v152PY2xjqIafIkUjK6vY9LtVFjJXenVUAmn3WCPWNFC1YTIIHw/mD2
# cTfPy7QA1pT+GPARAKt0bKtq9aCd/Ym0b5tPbpgCiRtzyb7fbNS1dE740re0COE6
# 7YV2wbeo2sXixzvLftH8L7s9xv9wV+G22qyKt6lmKLjFK1yMw4Ni5fMabcgmzRvS
# jAcbqgp3tk4a8emaaH0rz8MuuIP+yrxtREPXSqL/C5bzMzsikuDW9xH10graZzSm
# PjilzpRfRdu20/9UQmC7eVPZ4j1WNa1oqPHfzET3ChIzJ6Q9G3NPCB+7KwX0OQmK
# yv7IDimj8U/GlsHD1z+EF/fYMf8YXG15LamaOAohsw/ywO6SYSreVW+5Y0mzJutn
# BC9Cm9ozj1+/4kqksrlhZgR/CSxhFH3BTweH8gP2FEISRtShDZbuYymynY1un+Ry
# fiK9+iVTLdD1h/SxyxDpZMtimb4CgJQlMYIZ5zCCGeMCAQEwWDBBMRMwEQYKCZIm
# iZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQDEwxBTUUg
# Q1MgQ0EgMDECEzYAAAIA7fyNt5zeoUgAAgAAAgAwDQYJYIZIAWUDBAIBBQCgga4w
# GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisG
# AQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIA01Ad+dm+rNhbCDvUolmvVIwRjHv+7l
# RSKMyvkM3lHzMEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8AcwBvAGYA
# dKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEA
# Ltr6dSQaPqd7AGbCA2TxBWeTlO1zxV/oxq97XRqvNmUefArXmH/eggzG85JwhC5x
# xqZeGpkX7Asb433me8KRM8tyDCYNUlyom0Lq2PykP7yniwGyfBRvnDYNkDZT/vkb
# gS0i9Wqq9rbpxn9Dgen+8gbmk1nhZXq+Rp+NnVsGwe/hwGM7qPzjBRriL2saKrfz
# xV5wVNT3abwa66JrfG4wPVJ5urVScHP5kHfXC7hUlNGlos+Q2mDuy0S+DcmSVgh4
# DJp9Ehz5cjWpwgbVyMOc+rnVrUAEb8buBJUsMDLEmtca5ZpXcgsmvnPU9WXjAL9q
# PTKkLckWP7IRnIjoU26DsqGCF68wgherBgorBgEEAYI3AwMBMYIXmzCCF5cGCSqG
# SIb3DQEHAqCCF4gwgheEAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFaBgsqhkiG9w0B
# CRABBKCCAUkEggFFMIIBQQIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFlAwQCAQUA
# BCC0U71ApUE7B6Wu6skPJCF0PQsL2hwPe1rOxAa3LmrQKgIGZ5DxR06sGBMyMDI1
# MDEyMjIwNDMwMy42NzdaMASAAgH0oIHZpIHWMIHTMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJlbGFuZCBP
# cGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjoyRDFB
# LTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZaCCEf0wggcoMIIFEKADAgECAhMzAAAB/XP5aFrNDGHtAAEAAAH9MA0GCSqGSIb3
# DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
# VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAk
# BgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMB4XDTI0MDcyNTE4
# MzExNloXDTI1MTAyMjE4MzExNlowgdMxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
# YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
# Q29ycG9yYXRpb24xLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlv
# bnMgTGltaXRlZDEnMCUGA1UECxMeblNoaWVsZCBUU1MgRVNOOjJEMUEtMDVFMC1E
# OTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNlMIICIjAN
# BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAoWWs+D+Ou4JjYnRHRedu0MTFYzNJ
# EVPnILzc02R3qbnujvhZgkhp+p/lymYLzkQyG2zpxYceTjIF7HiQWbt6FW3ARkBr
# thJUz05ZnKpcF31lpUEb8gUXiD2xIpo8YM+SD0S+hTP1TCA/we38yZ3BEtmZtcVn
# aLRp/Avsqg+5KI0Kw6TDJpKwTLl0VW0/23sKikeWDSnHQeTprO0zIm/btagSYm3V
# /8zXlfxy7s/EVFdSglHGsUq8EZupUO8XbHzz7tURyiD3kOxNnw5ox1eZX/c/XmW4
# H6b4yNmZF0wTZuw37yA1PJKOySSrXrWEh+H6++Wb6+1ltMCPoMJHUtPP3Cn0CNcN
# vrPyJtDacqjnITrLzrsHdOLqjsH229Zkvndk0IqxBDZgMoY+Ef7ffFRP2pPkrF1F
# 9IcBkYz8hL+QjX+u4y4Uqq4UtT7VRnsqvR/x/+QLE0pcSEh/XE1w1fcp6Jmq8RnH
# EXikycMLN/a/KYxpSP3FfFbLZuf+qIryFL0gEDytapGn1ONjVkiKpVP2uqVIYj4V
# iCjy5pLUceMeqiKgYqhpmUHCE2WssLLhdQBHdpl28+k+ZY6m4dPFnEoGcJHuMcIZ
# nw4cOwixojROr+Nq71cJj7Q4L0XwPvuTHQt0oH7RKMQgmsy7CVD7v55dOhdHXdYs
# yO69dAdK+nWlyYcCAwEAAaOCAUkwggFFMB0GA1UdDgQWBBTpDMXA4ZW8+yL2+3vA
# 6RmU7oEKpDAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJlpxtTNRnpcjBfBgNVHR8E
# WDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9N
# aWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcmwwbAYIKwYB
# BQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20v
# cGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0ElMjAyMDEw
# KDEpLmNydDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMA4G
# A1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQsFAAOCAgEAY9hYX+T5AmCrYGaH96Td
# R5T52/PNOG7ySYeopv4flnDWQLhBlravAg+pjlNv5XSXZrKGv8e4s5dJ5WdhfC9y
# wFQq4TmXnUevPXtlubZk+02BXK6/23hM0TSKs2KlhYiqzbRe8QbMfKXEDtvMoHSZ
# T7r+wI2IgjYQwka+3P9VXgERwu46/czz8IR/Zq+vO5523Jld6ssVuzs9uwIrJhfc
# YBj50mXWRBcMhzajLjWDgcih0DuykPcBpoTLlOL8LpXooqnr+QLYE4BpUep3JySM
# YfPz2hfOL3g02WEfsOxp8ANbcdiqM31dm3vSheEkmjHA2zuM+Tgn4j5n+Any7IOD
# YQkIrNVhLdML09eu1dIPhp24lFtnWTYNaFTOfMqFa3Ab8KDKicmp0AthRNZVg0BP
# AL58+B0UcoBGKzS9jscwOTu1JmNlisOKkVUVkSJ5Fo/ctfDSPdCTVaIXXF7l40k1
# cM/X2O0JdAS97T78lYjtw/PybuzX5shxBh/RqTPvCyAhIxBVKfN/hfs4CIoFaqWJ
# 0r/8SB1CGsyyIcPfEgMo8ceq1w5Zo0JfnyFi6Guo+z3LPFl/exQaRubErsAUTfyB
# Y5/5liyvjAgyDYnEB8vHO7c7Fg2tGd5hGgYs+AOoWx24+XcyxpUkAajDhky9Dl+8
# JZTjts6BcT9sYTmOodk/SgIwggdxMIIFWaADAgECAhMzAAAAFcXna54Cm0mZAAAA
# AAAVMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2Fz
# aGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
# cnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0ZSBB
# dXRob3JpdHkgMjAxMDAeFw0yMTA5MzAxODIyMjVaFw0zMDA5MzAxODMyMjVaMHwx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1p
# Y3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIICIjANBgkqhkiG9w0BAQEFAAOC
# Ag8AMIICCgKCAgEA5OGmTOe0ciELeaLL1yR5vQ7VgtP97pwHB9KpbE51yMo1V/YB
# f2xK4OK9uT4XYDP/XE/HZveVU3Fa4n5KWv64NmeFRiMMtY0Tz3cywBAY6GB9alKD
# RLemjkZrBxTzxXb1hlDcwUTIcVxRMTegCjhuje3XD9gmU3w5YQJ6xKr9cmmvHaus
# 9ja+NSZk2pg7uhp7M62AW36MEBydUv626GIl3GoPz130/o5Tz9bshVZN7928jaTj
# kY+yOSxRnOlwaQ3KNi1wjjHINSi947SHJMPgyY9+tVSP3PoFVZhtaDuaRr3tpK56
# KTesy+uDRedGbsoy1cCGMFxPLOJiss254o2I5JasAUq7vnGpF1tnYN74kpEeHT39
# IM9zfUGaRnXNxF803RKJ1v2lIH1+/NmeRd+2ci/bfV+AutuqfjbsNkz2K26oElHo
# vwUDo9Fzpk03dJQcNIIP8BDyt0cY7afomXw/TNuvXsLz1dhzPUNOwTM5TI4CvEJo
# LhDqhFFG4tG9ahhaYQFzymeiXtcodgLiMxhy16cg8ML6EgrXY28MyTZki1ugpoMh
# XV8wdJGUlNi5UPkLiWHzNgY1GIRH29wb0f2y1BzFa/ZcUlFdEtsluq9QBXpsxREd
# cu+N+VLEhReTwDwV2xo3xwgVGD94q0W29R6HXtqPnhZyacaue7e3PmriLq0CAwEA
# AaOCAd0wggHZMBIGCSsGAQQBgjcVAQQFAgMBAAEwIwYJKwYBBAGCNxUCBBYEFCqn
# Uv5kxJq+gpE8RjUpzxD/LwTuMB0GA1UdDgQWBBSfpxVdAF5iXYP05dJlpxtTNRnp
# cjBcBgNVHSAEVTBTMFEGDCsGAQQBgjdMg30BATBBMD8GCCsGAQUFBwIBFjNodHRw
# Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9yeS5odG0w
# EwYDVR0lBAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
# CwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYDVR0jBBgwFoAU1fZWy4/o
# olxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDovL2NybC5taWNy
# b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYt
# MjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+aHR0cDovL3d3dy5t
# aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5j
# cnQwDQYJKoZIhvcNAQELBQADggIBAJ1VffwqreEsH2cBMSRb4Z5yS/ypb+pcFLY+
# TkdkeLEGk5c9MTO1OdfCcTY/2mRsfNB1OW27DzHkwo/7bNGhlBgi7ulmZzpTTd2Y
# urYeeNg2LpypglYAA7AFvonoaeC6Ce5732pvvinLbtg/SHUB2RjebYIM9W0jVOR4
# U3UkV7ndn/OOPcbzaN9l9qRWqveVtihVJ9AkvUCgvxm2EhIRXT0n4ECWOKz3+SmJ
# w7wXsFSFQrP8DJ6LGYnn8AtqgcKBGUIZUnWKNsIdw2FzLixre24/LAl4FOmRsqlb
# 30mjdAy87JGA0j3mSj5mO0+7hvoyGtmW9I/2kQH2zsZ0/fZMcm8Qq3UwxTSwethQ
# /gpY3UA8x1RtnWN0SCyxTkctwRQEcb9k+SS+c23Kjgm9swFXSVRk2XPXfx5bRAGO
# WhmRaw2fpCjcZxkoJLo4S5pu+yFUa2pFEUep8beuyOiJXk+d0tBMdrVXVAmxaQFE
# fnyhYWxz/gq77EFmPWn9y8FBSX5+k77L+DvktxW/tM4+pTFRhLy/AsGConsXHRWJ
# jXD+57XQKBqJC4822rpM+Zv/Cuk0+CQ1ZyvgDbjmjJnW4SLq8CdCPSWU5nR0W2rR
# nj7tfqAxM328y+l7vzhwRNGQ8cirOoo6CGJ/2XBjU02N7oJtpQUQwXEGahC0HVUz
# WLOhcGbyoYIDWDCCAkACAQEwggEBoYHZpIHWMIHTMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJlbGFuZCBP
# cGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjoyRDFB
# LTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZaIjCgEBMAcGBSsOAwIaAxUAoj0WtVVQUNSKoqtrjinRAsBUdoOggYMwgYCkfjB8
# MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
# bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1N
# aWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQsFAAIFAOs7
# b8IwIhgPMjAyNTAxMjIxMzIzMTRaGA8yMDI1MDEyMzEzMjMxNFowdjA8BgorBgEE
# AYRZCgQBMS4wLDAKAgUA6ztvwgIBADAJAgEAAgFWAgH/MAcCAQACAhKaMAoCBQDr
# PMFCAgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkKAwKgCjAIAgEAAgMH
# oSChCjAIAgEAAgMBhqAwDQYJKoZIhvcNAQELBQADggEBAJ/oS06NyylWvHrbPwZ0
# FHUfqT3TYsqQmGc9xKKpBwolj4OnHaZrhFOB8yoHS1Cw5I7j429nCk8tFYEwFsr4
# 0YiF7t0Z7qrviCrQ+TJVg0J20qAyP0uh0NfJTLr6y+WtmjaWu4Oh3TJ15oe1+pPu
# ibOBY6OlQEcQVgdZcVPMbARYkClrQ4bCcDKZ4Ck+d+mgFh/y1a7tiA/T1N6keti8
# i9N0YcWHAhlebBLlFag2ypK4CdOuOsj/LPfF1ef1tIA8OTb5dcGWqW3vyTwUpXVr
# iGrtfmez1vWSipYbHFQu0eywMOgAd6S/RxHijchiG7qg1jRj70qqj9jNjXOC7AYn
# ChYxggQNMIIECQIBATCBkzB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
# Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
# cmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAIT
# MwAAAf1z+WhazQxh7QABAAAB/TANBglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcN
# AQkDMQ0GCyqGSIb3DQEJEAEEMC8GCSqGSIb3DQEJBDEiBCCBLNFKghR3LvcH4T+h
# VsRaHVbBGNWyxEvwiqhLPsYgJzCB+gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0E
# IIAoSA3JSjC8h/H94N9hK6keR4XWnoYXCMoGzyVEHg1HMIGYMIGApH4wfDELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9z
# b2Z0IFRpbWUtU3RhbXAgUENBIDIwMTACEzMAAAH9c/loWs0MYe0AAQAAAf0wIgQg
# +Nbq9pLRs+vJLOvfNZ/nHGjsejM7wCCTk70IDcmfQGAwDQYJKoZIhvcNAQELBQAE
# ggIAkrPAGmwNXxyMSowFX6XAAPWb77ow+xvto5BwTYDcporviIAwVX8IhOy8084w
# ppzlR2D9Nia/Eze4IdVrn7PL4Et8Y42t0f41ahIJj5RuYpS5bGRhtrqQlPF6aR3l
# GTDDo0ihiJw9DyF8B28zOsOWgYkqCJM/cnfXjrdbOTn3FB4NXWZi5tjsdzgWR9Zn
# LSiP0IaXVe1ynq6gsztkJH/RAH9QonvtF0oKbUTJMgUhQGqA1ndxVpgOEwQv1pB8
# Qm7V3IrfwpJuaEb+mMljrR7j1SUnZkdk8S7ekxcQFkCzTbiYqav7grHWnQOCrDDU
# KEuu47sPRSQuoB5GDCY6vwhIUl7LYPj6ZoG6Ue+W+jyhY2aXhbA0XQCAtoSzieD7
# o1Oxw+UBJFVknCTD939nAd+NYQ04IoiSWEk96kbtnpSjeyxpX91eOorOQWYdKDfQ
# f2aVL9ldVJODSAfnejAsi7mRJ5jhG1V6vnQaNNyVpD3cbD7/J+am+l1tv4gk6Z5r
# HKdCwiZQu6rIYZuuArYBjI3P9SNfsOlNc0bQJlWV+eLNtiMqF/qchAFo01eT5j1j
# NIXeN+vfGZz3cyqHkPUKLNJ0h7nrk25/T2dwd6IyQySb4q3AIma5RSM0HdJA/F0g
# 5OImtfV2PxmDCgI0nledSeZ4xm7OhnsMHwurSBFjeVvCI50=
# SIG # End signature block
