﻿param
(
    # web root directory for the AOS web application
    [Parameter(Mandatory=$true)]
    $webroot,
    # directory where the models are deployed
    [Parameter(Mandatory=$true)]
    $packagedir,
    # log file
    [Parameter(Mandatory=$true)]
    $log
)

$signature = @’
[DllImport("kernel32.dll")] 
public static extern bool CreateSymbolicLink(string symlinkFileName, string targetFileName, int flags);
‘@

$type = Add-Type -MemberDefinition $signature -Name Win32Utils -Namespace CreateSymbolicLink -PassThru

function InitializeLog
{
    if(Test-Path $Global:log){
        Write-Output "Deleting existing log file $Global:log."
        Remove-Item $Global:log -Force
    }

    New-Item -ItemType file $Global:log -Force
}

function Write-Log ([string]$message)
{
    $datetime=get-date -Format "MMddyyyyhhmmss"
    if($Global:log -ne $null){
        "$datetime`: $message" >> $Global:log
    }

    Write-Output "$datetime`: $message"
}

function CreateSymbolicLink([string]$webroot,[string]$packagedir)
{
    Write-Log "Starting the creation of symlinks."
    Write-Log "Dynamics 365 Unified Operations package directory is: $packagedir"
    Write-Log "AOS webroot is: $webroot"
    $appassembliesdir=Join-Path $webroot "bin\appassemblies"

    if(!(Test-Path $appassembliesdir)){
        Write-Log "Creating $appassembliesdir"
        New-Item -ItemType Directory -Path $appassembliesdir -Force
    }

    $packagesbindir=join-path $packagedir "bin"

    $directories=[System.IO.Directory]::EnumerateDirectories($packagedir,"*",[System.IO.SearchOption]::TopDirectoryOnly)
    foreach($moduledir in $directories)
    {
        if($moduledir -ne "$packagesbindir"){
            $bindir=Join-Path $moduledir "bin"
        }else{
            $bindir=$moduledir
        }

        if(Test-Path $bindir){
            Write-Log "Enumerating applicable files at '$bindir'."
            $files=New-Object 'System.Collections.Generic.List[System.String]'
            $files.AddRange([System.IO.Directory]::EnumerateFiles($bindir,"*.dll",[System.IO.SearchOption]::TopDirectoryOnly))
            $files.AddRange([System.IO.Directory]::EnumerateFiles($bindir,"*.netmodule",[System.IO.SearchOption]::TopDirectoryOnly))
            $files.AddRange([System.IO.Directory]::EnumerateFiles($bindir,"*.runtime",[System.IO.SearchOption]::TopDirectoryOnly))
            $files.AddRange([System.IO.Directory]::EnumerateFiles($bindir,"*.xml",[System.IO.SearchOption]::TopDirectoryOnly))

            foreach($filepath in $files){
               $file=[System.IO.Path]::GetFileName($filepath)
               $tmp1=Join-Path $webroot "bin\$file"
               $tmp2=Join-Path $appassembliesdir $file
               if(((Test-Path "$tmp1") -eq $false)){
                   if((Test-Path "$tmp2") -eq $true){
                        Write-Log "Removing existing symlink $tmp2"
                        Remove-Item $tmp2 -Force
                   }

                   Write-Log "Creating a symlink at: $tmp2 for source: $filepath"
                   if(-not $type::CreateSymbolicLink($tmp2,$filepath,0)){
                        throw "Symlink creation failed for file: $tmp2"
                   }
               }
            }
        }
    }

    Write-Log "Symlinks created."
}

function Run-Process($cmd, $arguments, [switch]$throwonerror){
    $process=New-Object System.Diagnostics.Process
    $process.StartInfo = New-Object System.Diagnostics.ProcessStartInfo
    $process.StartInfo.FileName=$cmd
    $process.StartInfo.Arguments=$arguments
    $process.StartInfo.UseShellExecute=$false
    $process.StartInfo.CreateNoWindow = $true
    $process.StartInfo.RedirectStandardError=$true
    $process.StartInfo.RedirectStandardOutput=$true
    Write-Log "Running: $cmd with arguments: $arguments"

    Write-Debug "Register ErrorDataReceived event on process"
    $action = { $errors += $Event.SourceEventArgs.Data }
    Register-ObjectEvent -InputObject $process -EventName ErrorDataReceived -Action $action | Out-Null
    $process.Start() | Out-Null
    $process.BeginErrorReadLine()  
    if ($process.StandardOutput -ne $null) { $ngenoutput = $process.StandardOutput.ReadToEnd() }
    $process.WaitForExit()
    Write-Log $ngenoutput
    Write-Log $errors
    if($throwonerror -and ($process.ExitCode -ne 0)){
        throw "$cmd failed."
    }
}

function UpdateProbingPath([string]$webroot){
    $webconfig=join-path $webroot "web.config"
    $appassembliesdir="bin\appassemblies"
    [System.Xml.XmlDocument] $xd=new-object System.Xml.XmlDocument
    $xd.Load($webconfig)
    $ns=New-Object System.Xml.XmlNamespaceManager($xd.NameTable)
    $ns.AddNamespace("ns",$xd.DocumentElement.NamespaceURI)

    # update probing path
    $tmp=join-path $webroot "bin\appassemblies"
    if(Test-Path $tmp){
        [string]$privatepath=$xd.Configuration.RunTime.AssemblyBinding.Probing.privatePath
        [string[]] $probingpaths=$privatepath.Split(";")
        if(!$probingpaths.Contains($appassembliesdir)){
            Write-Log "Adding $appassembliesdir to the private probing path in web.config"
            $privatepath += ";$appassembliesdir"
            $xd.Configuration.RunTime.AssemblyBinding.Probing.privatePath=$privatepath

            # save changes
            $xd.Save($webconfig)
        }
    }
}

function NgenFiles([string]$webroot,[switch]$updateprobingpath)
{
    Write-Log "Creating the native images for Dynamics assemblies."
    $appassembliesdir=Join-Path $webroot "bin\appassemblies"

    if(!(Test-Path "$appassembliesdir")){
        throw "bin\appassemblies (symlink to Dynamics assemblies) folder doesn't exist under the webroot."
    }
    
    $ngen=Join-Path ([System.Runtime.InteropServices.RuntimeEnvironment]::GetRuntimeDirectory()) "ngen.exe"
    Write-Log "Ngen.exe path: $ngen"

    # ngen.exe needs an exe and config to generate native images
    $ngenexe=join-path $webroot "__ngen_exe.exe"
    $ngenexeconfig=join-path $webroot "__ngen_exe.exe.config"

    # delete existing files
    if(Test-Path $ngenexe){
        Write-Log "Removing existing $ngenexe"
        Remove-Item $ngenexe -Force
    }

    if(Test-Path $ngenexeconfig){
        Write-Log "Removing existing $ngenexeconfig"
        Remove-Item $ngenexeconfig -Force
    }

    if($updateprobingpath){
        UpdateProbingPath -webroot:$webroot
    }

    $webconfig=Join-Path $webroot "web.config"
    Write-Log "Copying $webconfig to $ngenexe"
    Copy-Item $webconfig $ngenexe

    Write-Log "Copying $webconfig to $ngenexeconfig"
    Copy-Item $webconfig $ngenexeconfig

    $dlls=New-Object 'System.Collections.Generic.List[System.String]'
    $dlls.AddRange([System.IO.Directory]::EnumerateFiles($appassembliesdir,"Dynamics.Ax.*.dll",[System.IO.SearchOption]::TopDirectoryOnly))

    # uninstall
    Write-Log "Uninstalling existing Dynamics native assembly images."
    foreach($dllpath in $dlls){
        $argument="uninstall `"$dllpath`" /nologo"
        Run-Process -cmd:$ngen -arguments:$argument # ngen uninstall throws error if assembly isn't registered which is expected; don't fail on it
    }

<#
    #install
    Write-Log "Creating native images for Dynamics assemblies..."
    foreach($dllpath in $dlls){
        $argument="install `"$dllpath`" /ExeConfig:`"$ngenexe`" /queue:1 /nologo /verbose"
        Run-Process -cmd:$ngen -arguments:$argument -throwonerror
    }

    Write-Log "Executing queued Ngen.exe jobs."
    $argument="executeQueuedItems /nologo /verbose"
    Run-Process -cmd:$ngen -arguments:$argument
    
    Write-Log "Native images for the Dynamics assemblies are created."
#>
}

function UpdateWebconfig([string]$webroot,[string]$uselazytypeloader)
{
   	Write-Log "Updating the AOS web configuration."
    $webconfig=join-path $webroot "web.config"
    $appassembliesdir="bin\appassemblies"
    [System.Xml.XmlDocument] $xd=new-object System.Xml.XmlDocument
    $xd.Load($webconfig)
    $ns=New-Object System.Xml.XmlNamespaceManager($xd.NameTable)
    $ns.AddNamespace("ns",$xd.DocumentElement.NamespaceURI)

    # add UseLazyTypeLoader setting
    $key="UseLazyTypeLoader"
    $value=$uselazytypeloader
    $existingNode = $xd.SelectSingleNode("//ns:add[@key='$key']",$ns)

    if($existingNode -ne $null){
        $existingValue=$existingNode.GetAttribute("value")
        if($existingValue -ne $value){ # update if the existing value -ne new value
    	    Write-Log "Updating configuration '$key'='$value'"
		    $existingNode.SetAttribute("value",$value)
            $xd.Save($webconfig) # save changes
        }
	}
    else{
		$addElement=$xd.CreateElement("add")
		$addElement.SetAttribute("key",$key)
		$addElement.SetAttribute("value",$value)
        $appSettings=$xd.SelectSingleNode("//ns:appSettings",$ns)
		Write-Log "Adding configuration '$key'='$value' to web.config"
		$appSettings.AppendChild($addElement)
        $xd.Save($webconfig) # save changes
	}
}

function UpdateBatchConfig([string]$webroot,[string]$uselazytypeloader,[switch]$addprobingpath)
{
    Write-Log "Updating the Dynamics Batch configuration file."
    $batchconfig=join-path $webroot "bin\batch.exe.config"
    if(!(Test-Path $batchconfig)){
        Write-Log "Batch.exe.config is not found at $webroot\bin"
        return
    } 

    [System.Xml.XmlDocument] $xd=new-object System.Xml.XmlDocument
    $xd.Load($batchconfig)
    $ns=New-Object System.Xml.XmlNamespaceManager($xd.NameTable)
    $ns.AddNamespace("ns",$xd.DocumentElement.NamespaceURI)

    # add UseLazyTypeLoader setting
    $save=$false # flag to indicate if there are changes to save
    $key="UseLazyTypeLoader"
    $value=$uselazytypeloader
    $existingNode = $xd.SelectSingleNode("//ns:add[@key='$key']",$ns)

    if($existingNode -ne $null){
        $existingValue=$existingNode.GetAttribute("value")
        if($existingValue -ne $value){  # update if the existing value -ne new value
            Write-Log "Updating configuration '$key'='$value'"
            $existingNode.SetAttribute("value",$value)
            $save=$true
        }
    }
    else{
        # add UseLazyTypeLoader key
        $addElement=$xd.CreateElement("add")
        $addElement.SetAttribute("key",$key)
        $addElement.SetAttribute("value",$value)
        $appSettings=$xd.SelectSingleNode("//ns:appSettings",$ns)
        Write-Log "Adding configuration '$key'='$value' to Batch.exe.config"
        $appSettings.AppendChild($addElement)
        $save=$true
    }
    
    if($addprobingpath){
        # add appassemblies to the probing path
        $appassemblies="appassemblies"
        $probingelement=$xd.Configuration.RunTime.AssemblyBinding.Probing
        if($probingelement -ne $null){
            [string]$privatepath=$xd.Configuration.RunTime.AssemblyBinding.Probing.privatePath
            [string[]] $probingpaths=$privatepath.Split(";")
            if(-not $probingpaths.Contains($appassemblies)){
                Write-Log "Adding $appassemblies to the private probing path in Batch.exe.config"
                $privatepath += ";$appassemblies"
                $xd.Configuration.RunTime.AssemblyBinding.Probing.privatePath=$privatepath
                $save=$true
            }
        }
        else{
            # add probing element
            $assemblyBinding=$xd.Configuration.RunTime.AssemblyBinding
            $xmlns=$assemblyBinding.xmlns
            $probingElement=$xd.CreateElement("probing")
            $probingElement.SetAttribute("privatePath",$appassemblies)
            $probingElement.SetAttribute("xmlns",$xmlns)
            Write-Log "Adding private probing path to the batch.exe.config"
            $assemblyBinding.AppendChild($probingelement)
            $save=$true
        }
    }

    if($save -eq $true){
        Write-Log "Saving changes to the Dynamics Batch configuration file."
        $xd.Save($batchconfig)
    }
}

$Global:log=$log
InitializeLog
CreateSymbolicLink -webroot:$webroot -packagedir:$packagedir
NgenFiles -webroot:$webroot -updateprobingpath
UpdateWebconfig -webroot:$webroot -uselazytypeloader:"false"
UpdateBatchConfig -webroot:$webroot -uselazytypeloader:"false" -addprobingpath
# SIG # Begin signature block
# MIIoKgYJKoZIhvcNAQcCoIIoGzCCKBcCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCArBxJ/wB8b2s5N
# I4jXN16ByUYMnKtSQzopPF4CWt31cKCCDXYwggX0MIID3KADAgECAhMzAAAEhV6Z
# 7A5ZL83XAAAAAASFMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjUwNjE5MTgyMTM3WhcNMjYwNjE3MTgyMTM3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDASkh1cpvuUqfbqxele7LCSHEamVNBfFE4uY1FkGsAdUF/vnjpE1dnAD9vMOqy
# 5ZO49ILhP4jiP/P2Pn9ao+5TDtKmcQ+pZdzbG7t43yRXJC3nXvTGQroodPi9USQi
# 9rI+0gwuXRKBII7L+k3kMkKLmFrsWUjzgXVCLYa6ZH7BCALAcJWZTwWPoiT4HpqQ
# hJcYLB7pfetAVCeBEVZD8itKQ6QA5/LQR+9X6dlSj4Vxta4JnpxvgSrkjXCz+tlJ
# 67ABZ551lw23RWU1uyfgCfEFhBfiyPR2WSjskPl9ap6qrf8fNQ1sGYun2p4JdXxe
# UAKf1hVa/3TQXjvPTiRXCnJPAgMBAAGjggFzMIIBbzAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUuCZyGiCuLYE0aU7j5TFqY05kko0w
# RQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEW
# MBQGA1UEBRMNMjMwMDEyKzUwNTM1OTAfBgNVHSMEGDAWgBRIbmTlUAXTgqoXNzci
# tW2oynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NybC9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3JsMGEG
# CCsGAQUFBwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDovL3d3dy5taWNyb3NvZnQu
# Y29tL3BraW9wcy9jZXJ0cy9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3J0
# MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQELBQADggIBACjmqAp2Ci4sTHZci+qk
# tEAKsFk5HNVGKyWR2rFGXsd7cggZ04H5U4SV0fAL6fOE9dLvt4I7HBHLhpGdE5Uj
# Ly4NxLTG2bDAkeAVmxmd2uKWVGKym1aarDxXfv3GCN4mRX+Pn4c+py3S/6Kkt5eS
# DAIIsrzKw3Kh2SW1hCwXX/k1v4b+NH1Fjl+i/xPJspXCFuZB4aC5FLT5fgbRKqns
# WeAdn8DsrYQhT3QXLt6Nv3/dMzv7G/Cdpbdcoul8FYl+t3dmXM+SIClC3l2ae0wO
# lNrQ42yQEycuPU5OoqLT85jsZ7+4CaScfFINlO7l7Y7r/xauqHbSPQ1r3oIC+e71
# 5s2G3ClZa3y99aYx2lnXYe1srcrIx8NAXTViiypXVn9ZGmEkfNcfDiqGQwkml5z9
# nm3pWiBZ69adaBBbAFEjyJG4y0a76bel/4sDCVvaZzLM3TFbxVO9BQrjZRtbJZbk
# C3XArpLqZSfx53SuYdddxPX8pvcqFuEu8wcUeD05t9xNbJ4TtdAECJlEi0vvBxlm
# M5tzFXy2qZeqPMXHSQYqPgZ9jvScZ6NwznFD0+33kbzyhOSz/WuGbAu4cHZG8gKn
# lQVT4uA2Diex9DMs2WHiokNknYlLoUeWXW1QrJLpqO82TLyKTbBM/oZHAdIc0kzo
# STro9b3+vjn2809D0+SOOCVZMIIHejCCBWKgAwIBAgIKYQ6Q0gAAAAAAAzANBgkq
# hkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
# EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
# IDIwMTEwHhcNMTEwNzA4MjA1OTA5WhcNMjYwNzA4MjEwOTA5WjB+MQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQg
# Q29kZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
# CgKCAgEAq/D6chAcLq3YbqqCEE00uvK2WCGfQhsqa+laUKq4BjgaBEm6f8MMHt03
# a8YS2AvwOMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc6Whe0t+bU7IKLMOv2akr
# rnoJr9eWWcpgGgXpZnboMlImEi/nqwhQz7NEt13YxC4Ddato88tt8zpcoRb0Rrrg
# OGSsbmQ1eKagYw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+lD3v++MrWhAfTVYoonpy
# 4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+EGvKhL1nkkDstrjNYxbc+/jLTswM9
# sbKvkjh+0p2ALPVOVpEhNSXDOW5kf1O6nA+tGSOEy/S6A4aN91/w0FK/jJSHvMAh
# dCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rhKEmdX4jiJV3TIUs+UsS1Vz8k
# A/DRelsv1SPjcF0PUUZ3s/gA4bysAoJf28AVs70b1FVL5zmhD+kjSbwYuER8ReTB
# w3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zdsGbiwZeBe+3W7UvnSSmn
# Eyimp31ngOaKYnhfsi+E11ecXL93KCjx7W3DKI8sj0A3T8HhhUSJxAlMxdSlQy90
# lfdu+HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS4NaIjAsCAwEAAaOCAe0w
# ggHpMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBRIbmTlUAXTgqoXNzcitW2o
# ynUClTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYD
# VR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBRyLToCMZBDuRQFTuHqp8cx0SOJNDBa
# BgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
# bC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3JsMF4GCCsG
# AQUFBwEBBFIwUDBOBggrBgEFBQcwAoZCaHR0cDovL3d3dy5taWNyb3NvZnQuY29t
# L3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3J0MIGfBgNV
# HSAEgZcwgZQwgZEGCSsGAQQBgjcuAzCBgzA/BggrBgEFBQcCARYzaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1hcnljcHMuaHRtMEAGCCsG
# AQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAHAAbwBsAGkAYwB5AF8AcwB0AGEAdABl
# AG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn8oalmOBUeRou09h0ZyKb
# C5YR4WOSmUKWfdJ5DJDBZV8uLD74w3LRbYP+vj/oCso7v0epo/Np22O/IjWll11l
# hJB9i0ZQVdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0bpdS1HXeUOeLpZMlEPXh6
# I/MTfaaQdION9MsmAkYqwooQu6SpBQyb7Wj6aC6VoCo/KmtYSWMfCWluWpiW5IP0
# wI/zRive/DvQvTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvyCInWH8MyGOLwxS3OW560
# STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn+N4sOiBpmLJZiWhub6e3dMNABQam
# ASooPoI/E01mC8CzTfXhj38cbxV9Rad25UAqZaPDXVJihsMdYzaXht/a8/jyFqGa
# J+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw3MYbBL7fQccOKO7eZS/sl/ah
# XJbYANahRr1Z85elCUtIEJmAH9AAKcWxm6U/RXceNcbSoqKfenoi+kiVH6v7RyOA
# 9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sLgOppO6/8MO0ETI7f33Vt
# Y5E90Z1WTk+/gFcioXgRMiF670EKsT/7qMykXcGhiJtXcVZOSEXAQsmbdlsKgEhr
# /Xmfwb1tbWrJUnMTDXpQzTGCGgowghoGAgEBMIGVMH4xCzAJBgNVBAYTAlVTMRMw
# EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVN
# aWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNp
# Z25pbmcgUENBIDIwMTECEzMAAASFXpnsDlkvzdcAAAAABIUwDQYJYIZIAWUDBAIB
# BQCgga4wGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIL4fxgZ75N96fntIKZQxEruO
# w3REhE1eNc7iMTq3KO9uMEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8A
# cwBvAGYAdKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEB
# BQAEggEAg3bWsFiaXd94DyUr/yWyVjBsAdOZqmjBZtv+hr6eywHvrfHuerUSMnNH
# pzM11NHJ/Ra2HsLfa9Qdd2IoGTjBJlby/1BgOsSE1rQookq6fBUP27HF1CZ1A/YT
# kzmiK75hw5/yoW9arkLZjpxU6KWUx4fF2IijwRBDdQEEYWdi9uhXzkI5BvdrEmUd
# 3bdYFhYRfIvPf6UXAsK2zTGFtsqD0xw5fcf20vmViPtYSn5HiyyM0Kn98fNvN2/h
# aSRbJR+0UBTPmkvgaQvvbLp2qxrWh89Bs0mvzetSP2YJXCtxQMwoScMK5wDbhGUf
# WK2LzUpZKh0C47JpzpIXd0+gBv88F6GCF5QwgheQBgorBgEEAYI3AwMBMYIXgDCC
# F3wGCSqGSIb3DQEHAqCCF20wghdpAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFSBgsq
# hkiG9w0BCRABBKCCAUEEggE9MIIBOQIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFl
# AwQCAQUABCCbwErV5CMfMOXAJUpaOaAozZWuTFG+uBWLgkyZpAlHsAIGaKOig79P
# GBMyMDI1MDkwNDA3MzQwMS42NjJaMASAAgH0oIHRpIHOMIHLMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1l
# cmljYSBPcGVyYXRpb25zMScwJQYDVQQLEx5uU2hpZWxkIFRTUyBFU046MzcwMy0w
# NUUwLUQ5NDcxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2Wg
# ghHqMIIHIDCCBQigAwIBAgITMwAAAgpHshTZ7rKzDwABAAACCjANBgkqhkiG9w0B
# AQsFADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
# BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYD
# VQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDAeFw0yNTAxMzAxOTQy
# NTdaFw0yNjA0MjIxOTQyNTdaMIHLMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2Fz
# aGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
# cnBvcmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1lcmljYSBPcGVyYXRpb25z
# MScwJQYDVQQLEx5uU2hpZWxkIFRTUyBFU046MzcwMy0wNUUwLUQ5NDcxJTAjBgNV
# BAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2UwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCy7NzwEpb7BpwAk9LJ00Xq30TcTjcwNZ80TxAtAbhS
# aJ2kwnJA1Au/Do9/fEBjAHv6Mmtt3fmPDeIJnQ7VBeIq8RcfjcjrbPIg3wA5v5MQ
# flPNSBNOvcXRP+fZnAy0ELDzfnJHnCkZNsQUZ7GF7LxULTKOYY2YJw4TrmcHohkY
# 6DjCZyxhqmGQwwdbjoPWRbYu/ozFem/yfJPyjVBql1068bcVh58A8c5CD6TWN/L3
# u+Ny+7O8+Dver6qBT44Ey7pfPZMZ1Hi7yvCLv5LGzSB6o2OD5GIZy7z4kh8UYHdz
# jn9Wx+QZ2233SJQKtZhpI7uHf3oMTg0zanQfz7mgudefmGBrQEg1ox3n+3Tizh0D
# 9zVmNQP9sFjsPQtNGZ9ID9H8A+kFInx4mrSxA2SyGMOQcxlGM30ktIKM3iqCuFEU
# 9CHVMpN94/1fl4T6PonJ+/oWJqFlatYuMKv2Z8uiprnFcAxCpOsDIVBO9K1vHeAM
# iQQUlcE9CD536I1YLnmO2qHagPPmXhdOGrHUnCUtop21elukHh75q/5zH+OnNekp
# 5udpjQNZCviYAZdHsLnkU0NfUAr6r1UqDcSq1yf5RiwimB8SjsdmHll4gPjmqVi0
# /rmnM1oAEQm3PyWcTQQibYLiuKN7Y4io5bJTVwm+vRRbpJ5UL/D33C//7qnHbeoW
# BQIDAQABo4IBSTCCAUUwHQYDVR0OBBYEFAKvF0EEj4AyPfY8W/qrsAvftZwkMB8G
# A1UdIwQYMBaAFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMF8GA1UdHwRYMFYwVKBSoFCG
# Tmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY3Jvc29mdCUy
# MFRpbWUtU3RhbXAlMjBQQ0ElMjAyMDEwKDEpLmNybDBsBggrBgEFBQcBAQRgMF4w
# XAYIKwYBBQUHMAKGUGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY2Vy
# dHMvTWljcm9zb2Z0JTIwVGltZS1TdGFtcCUyMFBDQSUyMDIwMTAoMSkuY3J0MAwG
# A1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwDgYDVR0PAQH/BAQD
# AgeAMA0GCSqGSIb3DQEBCwUAA4ICAQCwk3PW0CyjOaqXCMOusTde7ep2CwP/xV1J
# 3o9KAiKSdq8a2UR5RCHYhnJseemweMUH2kNefpnAh2Bn8H2opDztDJkj8OYRd/KQ
# ysE12NwaY3KOwAW8Rg8OdXv5fUZIsOWgprkCQM0VoFHdXYExkJN3EzBbUCUw3yb4
# gAFPK56T+6cPpI8MJLJCQXHNMgti2QZhX9KkfRAffFYMFcpsbI+oziC5Brrk3361
# cJFHhgEJR0J42nqZTGSgUpDGHSZARGqNcAV5h+OQDLeF2p3URx/P6McUg1nJ2gMP
# YBsD+bwd9B0c/XIZ9Mt3ujlELPpkijjCdSZxhzu2M3SZWJr57uY+FC+LspvIOH1O
# pofanh3JGDosNcAEu9yUMWKsEBMngD6VWQSQYZ6X9F80zCoeZwTq0i9AujnYzzx5
# W2fEgZejRu6K1GCASmztNlYJlACjqafWRofTqkJhV/J2v97X3ruDvfpuOuQoUtVA
# wXrDsG2NOBuvVso5KdW54hBSsz/4+ORB4qLnq4/GNtajUHorKRKHGOgFo8DKaXG+
# UNANwhGNxHbILSa59PxExMgCjBRP3828yGKsquSEzzLNWnz5af9ZmeH4809fwItt
# I41JkuiY9X6hmMmLYv8OY34vvOK+zyxkS+9BULVAP6gt+yaHaBlrln8Gi4/dBr2y
# 6Srr/56g0DCCB3EwggVZoAMCAQICEzMAAAAVxedrngKbSZkAAAAAABUwDQYJKoZI
# hvcNAQELBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAw
# DgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
# MjAwBgNVBAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eSAy
# MDEwMB4XDTIxMDkzMDE4MjIyNVoXDTMwMDkzMDE4MzIyNVowfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTAwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoIC
# AQDk4aZM57RyIQt5osvXJHm9DtWC0/3unAcH0qlsTnXIyjVX9gF/bErg4r25Phdg
# M/9cT8dm95VTcVrifkpa/rg2Z4VGIwy1jRPPdzLAEBjoYH1qUoNEt6aORmsHFPPF
# dvWGUNzBRMhxXFExN6AKOG6N7dcP2CZTfDlhAnrEqv1yaa8dq6z2Nr41JmTamDu6
# GnszrYBbfowQHJ1S/rboYiXcag/PXfT+jlPP1uyFVk3v3byNpOORj7I5LFGc6XBp
# Dco2LXCOMcg1KL3jtIckw+DJj361VI/c+gVVmG1oO5pGve2krnopN6zL64NF50Zu
# yjLVwIYwXE8s4mKyzbnijYjklqwBSru+cakXW2dg3viSkR4dPf0gz3N9QZpGdc3E
# XzTdEonW/aUgfX782Z5F37ZyL9t9X4C626p+Nuw2TPYrbqgSUei/BQOj0XOmTTd0
# lBw0gg/wEPK3Rxjtp+iZfD9M269ewvPV2HM9Q07BMzlMjgK8QmguEOqEUUbi0b1q
# GFphAXPKZ6Je1yh2AuIzGHLXpyDwwvoSCtdjbwzJNmSLW6CmgyFdXzB0kZSU2LlQ
# +QuJYfM2BjUYhEfb3BvR/bLUHMVr9lxSUV0S2yW6r1AFemzFER1y7435UsSFF5PA
# PBXbGjfHCBUYP3irRbb1Hode2o+eFnJpxq57t7c+auIurQIDAQABo4IB3TCCAdkw
# EgYJKwYBBAGCNxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQUKqdS/mTEmr6CkTxG
# NSnPEP8vBO4wHQYDVR0OBBYEFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMFwGA1UdIARV
# MFMwUQYMKwYBBAGCN0yDfQEBMEEwPwYIKwYBBQUHAgEWM2h0dHA6Ly93d3cubWlj
# cm9zb2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5Lmh0bTATBgNVHSUEDDAK
# BggrBgEFBQcDCDAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDANBgkqhkiG
# 9w0BAQsFAAOCAgEAnVV9/Cqt4SwfZwExJFvhnnJL/Klv6lwUtj5OR2R4sQaTlz0x
# M7U518JxNj/aZGx80HU5bbsPMeTCj/ts0aGUGCLu6WZnOlNN3Zi6th542DYunKmC
# VgADsAW+iehp4LoJ7nvfam++Kctu2D9IdQHZGN5tggz1bSNU5HhTdSRXud2f8449
# xvNo32X2pFaq95W2KFUn0CS9QKC/GbYSEhFdPSfgQJY4rPf5KYnDvBewVIVCs/wM
# nosZiefwC2qBwoEZQhlSdYo2wh3DYXMuLGt7bj8sCXgU6ZGyqVvfSaN0DLzskYDS
# PeZKPmY7T7uG+jIa2Zb0j/aRAfbOxnT99kxybxCrdTDFNLB62FD+CljdQDzHVG2d
# Y3RILLFORy3BFARxv2T5JL5zbcqOCb2zAVdJVGTZc9d/HltEAY5aGZFrDZ+kKNxn
# GSgkujhLmm77IVRrakURR6nxt67I6IleT53S0Ex2tVdUCbFpAUR+fKFhbHP+Crvs
# QWY9af3LwUFJfn6Tvsv4O+S3Fb+0zj6lMVGEvL8CwYKiexcdFYmNcP7ntdAoGokL
# jzbaukz5m/8K6TT4JDVnK+ANuOaMmdbhIurwJ0I9JZTmdHRbatGePu1+oDEzfbzL
# 6Xu/OHBE0ZDxyKs6ijoIYn/ZcGNTTY3ugm2lBRDBcQZqELQdVTNYs6FwZvKhggNN
# MIICNQIBATCB+aGB0aSBzjCByzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
# bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
# b3JhdGlvbjElMCMGA1UECxMcTWljcm9zb2Z0IEFtZXJpY2EgT3BlcmF0aW9uczEn
# MCUGA1UECxMeblNoaWVsZCBUU1MgRVNOOjM3MDMtMDVFMC1EOTQ3MSUwIwYDVQQD
# ExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQDR
# AMVJlA6bKq93Vnu3UkJgm5HlYaCBgzCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1w
# IFBDQSAyMDEwMA0GCSqGSIb3DQEBCwUAAgUA7GM3YjAiGA8yMDI1MDkwMzIxNTMz
# OFoYDzIwMjUwOTA0MjE1MzM4WjB0MDoGCisGAQQBhFkKBAExLDAqMAoCBQDsYzdi
# AgEAMAcCAQACAiQXMAcCAQACAhJtMAoCBQDsZIjiAgEAMDYGCisGAQQBhFkKBAIx
# KDAmMAwGCisGAQQBhFkKAwKgCjAIAgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZI
# hvcNAQELBQADggEBAFBCPBg9rlvZDu9q6GmF3UXX0B7zFUCqgDQTiPS3bSOEQVkY
# EixpcZSATH30Kk/nOKZVbgj1N6XMDEsh6VXpZ26XL/FXpYoTNiQqp3nuXGEhyggG
# XQpyooj9fCmrxgzzvvr18m5VTvcpD4uiJVx1K5ufqEyCS6k2cqjG+p4uOTEcazPJ
# kEzvanNd5bvmibHfFEcW07UmCb1JsigJDnScl6UoagJWySuHhkWfdls5N5dNs7AN
# 9bWQGCdAsWb6F7R3Twxxl2ET1YyuRZOzLrW+FNK+YT6IEBpWrpRYJYEy118WBXn4
# tXfNgmfpsYxByLZFzyITceOKcZhd+4rRoe3+x0ExggQNMIIECQIBATCBkzB8MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNy
# b3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAgpHshTZ7rKzDwABAAACCjAN
# BglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMC8G
# CSqGSIb3DQEJBDEiBCCvMR0AA0PDgIZOUpvdEtMX/rzUEYJUHAnYt8dcfieSrDCB
# +gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EIE2ay/y0epK/X3Z03KTcloqE8u9I
# XRtdO7Mex0hw9+SaMIGYMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIw
# MTACEzMAAAIKR7IU2e6ysw8AAQAAAgowIgQgc8XQBAQS0TfiXZoNduTU68wmSVjN
# nY+CIE6fYz/YGuMwDQYJKoZIhvcNAQELBQAEggIAjH2NaT//jPh4lTqtFtUJAsZP
# K1ujGEcTF0S2iEQA92d1DKVD2SgKuo7o9Q2ug69k54SrFseZL9eT2xmbSK72hmRb
# pBC/WKR/gwlrUtHlSACHELzEldcQ2JA4HKGDehufLenoMLtFzxwYcmy8p2CFCM5V
# 4mDlaNV2HpG9QuaG9RX+AYVK4j8pgVdlNaslHybvlr71SQ6lVOzgyArApY0JOZW0
# sbMo2bdbM9yv87olyNPmHdZ/0jE6uYGciSC2Pt2X5P5q69gkUW6o4QUgvUSxcuMl
# SJG8q5mfaQZ/nU3Ey5fqNZ3idNf3roJP7Yrto13jCe8Plr/kyUga+tX9WVkiTlVZ
# sOoIw/Fe5ffpQI29HzlUUHZQGHVEXKdMZoGc6dDVV/FnJS+gtGaGDFmsoCxzSu1U
# jBLqieVcsQyKvNkQLpR2JuHMePFz8l1Tq+BDIW+/Zd5ECTbmpmgvtj4Pk7liYdwH
# rVnX6OAUHHUYPZFW7rCHA1MxwbCfnyR8h5pnZR4Vu9qXEAaDFdtfaQfwNqIOnioT
# XCJpf/ya+XzsVbqi6sREcPaU508zooYhMOBjT3ACBPnAA4c/4H5HdpH2PYfso0fD
# sSO2xgleXZJ4bd+1+pO/9ushgFTFHIIqj5Fk4ESy97coTjj3VW5AIInlQWYnbaym
# po5FKzlKbwNPl0Wjhrs=
# SIG # End signature block
