PARAM (
    [Parameter(Mandatory=$true)]
    $log,
    [Parameter(Mandatory=$true)]
    [ValidateScript({Test-Path $_})]
    $webroot,
    [Parameter(Mandatory=$true)]
    [ValidateScript({Test-Path $_})]
    $packageDirectory
)

$ErrorActionPreference = 'Stop'

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking
Import-Module "$PSScriptRoot\AosEnvironmentUtilities.psm1" -Force -DisableNameChecking

$WebConfigPath = "$webroot\web.config"
$DbSyncBlastFolder = "$PSScriptRoot\DbSyncBlast"
$DbSyncVersionTestFile = "Microsoft.Dynamics.AX.Data.Management.Sync.dll"

Initialize-Log $log

function Replace-Files([string] $sourceFolder, [string[]] $sourceFiles, [string[]] $destinationFolders, [string] $backupFolderName, [hashtable] $backedUpFiles)
{
    $useBackup = (-not [String]::IsNullOrWhiteSpace($backupFolderName))

    foreach ($directory in $destinationFolders)
    {
        if ($useBackup)
        {
            $backupFolder = Join-Path $directory $backupFolderName

            if (-not (Test-Path $backupFolder))
            {
                Write-Log "Creating folder '$backupFolder'"
                New-Item -ItemType Directory -Path $backupFolder
            }
        }

        foreach ($file in $sourceFiles)
        {
            $sourcePath = Join-Path $sourceFolder $file
            $targetPath = Join-Path $directory $file

            if ($useBackup)
            {
                if (Test-Path $targetPath)
                {
                    $backupPath = Join-Path $backupFolder "$file.bak"

                    Write-Log "Copying file from: '$sourcePath' to '$targetPath', backing up target to '$backupPath'"

                    Copy-Item -Path $targetPath -Destination $backupPath -Force
                    $backedUpFiles[$targetPath] = $backupPath
                }
                else
                {
                    Write-Log "Copying file from: '$sourcePath' to '$targetPath'. Target file does not exist, skipping backup."

                    # Empty value means the file is new - delete/rename when restoring backup
                    $backedUpFiles[$targetPath] = ""
                }
            }
            else
            {
                Write-Log "Copying file from: '$sourcePath' to '$targetPath'. No backup folder specified, skipping backup."
            }

            Copy-Item -Path $sourcePath -Destination $targetPath -Force
                                
            Write-Log "Copy file succeeded."
        }
    }
}

function Restore-Backup([hashtable] $backedUpFiles)
{
    foreach ($file in $backedUpFiles.GetEnumerator())
    {
        try
        {
            if (-not [String]::IsNullOrWhiteSpace($file.Value))
            {
                Write-Log "Restoring backup from '$($file.Value)' to '$($file.Key)'"

                Copy-Item -Path $file.Value -Destination $file.Key -Force
                
                Write-Log "Restored successfully."
            }
            else
            {
                $renameTo = "$($file.Key).dbsyncblast_failed"

                Write-Log "Restoring backup for '$($file.Key)', which is newly created - renaming to '$renameTo' instead of deleting."

                Rename-Item -Path $file.Key -NewName $renameTo

                Write-Log "Renamed successfully."
            }
        }
        catch
        {
            Write-Exception $_
            Write-Log "Failed to restore backup for '$($file.Key)'"
        }
    }
}

function Test-RequiredPaths
{
    foreach ($path in $args)
    {
        if (-not (Test-Path $path))
        {
            Write-Log "Path '$path' not found, skipping DbSyncBlast."
            return $false
        }
    }

    return $true
}

function Get-FileVersion([string] $file)
{
    $fileInfo = Get-ChildItem -File $file

    if ($fileInfo -eq $null -or $fileInfo.VersionInfo -eq $null)
    {
        Write-Log "Could not get version info for '$fileInfo', skipping DbSyncBlast."
        return $null
    }

    return $fileInfo.VersionInfo
}

filter VersionsToDictionary
{
    begin { $dict = @{} }
    process { $dict[[int]($_.Split('.')[-1])] = $_ } # e.g. 7.0.30100.4970 --> $dict[4970] = 7.0.30100.4970
    end { return $dict }
}

$DbSyncBlastMinPlatformVersion = 5030 #PU20
$script:PlatformBuildVersion = 0
function Get-DbSyncApplicableVersionPath
{
    if (-not (Test-RequiredPaths $DbSyncBlastFolder))
    {
        return $null
    }
    
    # Default to no replace
    $dbSyncApplicableVersionPath = $null
            
    $platformVersionInfo = Get-ProductPlatformBuildFullVersion

    if ($platformVersionInfo -ne $null)
    {
        Write-Log "Current platform build: $platformVersionInfo"

        $dbSyncVersions = Get-ChildItem -Path $DbSyncBlastFolder -Directory -Name | VersionsToDictionary
        Write-Log "Found DbSyncBlast builds for platform builds >= $($dbSyncVersions.Keys -join ', ')."
        $invalidDbSyncBlastVersions = $dbSyncVersions.GetEnumerator() | ?{$_.Key -lt $DbSyncBlastMinPlatformVersion}
        if ($invalidDbSyncBlastVersions.Count -gt 0)
        {
            Write-Log "Skipping invalid DbSyncBlast builds: $($invalidDbSyncBlastVersions.Values -join ', '). Minimum supported platform build is $DbSyncBlastMinPlatformVersion."

            $dbSyncVersions = $dbSyncVersions.GetEnumerator() | ?{$_.Key -ge $DbSyncBlastMinPlatformVersion}
        }
        
        $script:PlatformBuildVersion = $platformVersionInfo.Build
        $useDbSyncPlatformVersion = [int](($dbSyncVersions.Keys | ?{$_ -le $script:PlatformBuildVersion} | measure -Max).Maximum)
        if ($useDbSyncPlatformVersion -ne 0)
        {
            Write-Log "Using DbSyncBlast build $($dbSyncVersions[$useDbSyncPlatformVersion])"
            $dbSyncApplicableVersionPath = Join-Path $DbSyncBlastFolder $dbSyncVersions[$useDbSyncPlatformVersion]
        }
        else
        {
            Write-Log "No applicable version found, skipping DbSyncBlast."
        }
    }
    
    return $dbSyncApplicableVersionPath
}

$EnsureMajorVersion = 7
$EnsureMinorVersion = 0
function Assert-IsNewerVersion([string] $newFile, [string] $oldFile)
{
    if (-not (Test-RequiredPaths $newFile $oldFile))
    {
        return $false
    }

    $newFileVersionInfo = Get-FileVersion -File $newFile
    $oldFileVersionInfo = Get-FileVersion -File $oldFile

    if ($newFileVersionInfo -eq $null -or $oldFileVersionInfo -eq $null)
    {
        return $false
    }

    Write-Log "Version of '$oldFile': $($oldFileVersionInfo.FileVersion)"
    Write-Log "Version of '$newFile': $($newFileVersionInfo.FileVersion)"

    if ($newFileVersionInfo.FileMajorPart -ne $EnsureMajorVersion -or $oldFileVersionInfo.FileMajorPart -ne $EnsureMajorVersion `
    -or $newFileVersionInfo.FileMinorPart -ne $EnsureMinorVersion -or $oldFileVersionInfo.FileMinorPart -ne $EnsureMinorVersion)
    {
        # If e.g. major version is changed from 7 to 10, we need to decide what to do. Skipping for now.
        Write-Log "Unexpected major or minor version, expected $EnsureMajorVersion.$EnsureMinorVersion.x.y. Skipping DbSyncBlast."
        return $false
    }

    $versionDiff = $newFileVersionInfo.FileBuildPart - $oldFileVersionInfo.FileBuildPart

    if ($versionDiff -lt 0)
    {
        Write-Log "Selected DbSyncBlast build version is lower than what is installed, skipping DbSyncBlast."
        return $false
    }

    if ($versionDiff -eq 0)
    {
        Write-Log "Selected DbSyncBlast build already installed, skipping DbSyncBlast."
        return $false
    }

    return $true
}

function Update-WebConfig
{
    # Update the Monitoring.ETWManifests key, adding Microsoft.Dynamics.ApplicationPlatform.DatabaseSynchronize.man if it does not exist in the list
    $key = "Monitoring.ETWManifests"
    $manFileName = "Microsoft.Dynamics.ApplicationPlatform.DatabaseSynchronize.man"

    # Check for PU20-21
    if ($script:PlatformBuildVersion -ge 5030 -and $script:PlatformBuildVersion -le 5073) {

        [System.Xml.XmlDocument] $xmlDoc = new-object System.Xml.XmlDocument
        $xmlDoc.Load($WebConfigPath)
        $node = $xmlDoc.SelectSingleNode("//add[@key='$key']")
        $oldValueSplit = $node.Value -split ";"

        if ($oldValueSplit -notcontains $manFileName)
        {
            Write-Log "Updating $key, adding $manFileName"
            $node.Value += ";$manFileName"
            $xmlDoc.Save($WebConfigPath)
        }
        else
        {
            Write-Log "Not updating $key, $manFileName already exists."
        }
    }
    else
    {
        Write-Log "Not updating $key, platform is not PU20 or PU21."
    }
}

try
{
    Write-Log "DbSyncBlast assembly replacement check: Start"

    $error.Clear()

    if (-not (Get-IsFlightEnabled "DbSyncBlastDisabled"))
    {
        $dbSyncApplicableVersionPath = Get-DbSyncApplicableVersionPath

        if ($dbSyncApplicableVersionPath -ne $null)
        {
            if (Assert-IsNewerVersion "$dbSyncApplicableVersionPath\$DbSyncVersionTestFile" "$webroot\bin\$DbSyncVersionTestFile")
            {
                # The list of assemblies to be replaced.
                # Microsoft.Dynamics.ApplicationPlatform.DatabaseSynchronize.Instrumentation.dll should be included in both lists.
                $sourceFiles = Get-ChildItem -Path $dbSyncApplicableVersionPath -File -Name
                $sourceFilesNoMan = $sourceFiles | ?{-not $_.EndsWith(".man")}
                $sourceFilesMan   = $sourceFiles | ?{     $_.EndsWith(".man") -or $_.EndsWith(".Instrumentation.dll")}

                Write-Log "Non-instrumentation files to blast: $sourceFilesNoMan"
                Write-Log "Instrumentation files to blast: $sourceFilesMan"

                # Build a destination directory list
                $ETWManifestPath = "$(Split-Path -parent $PSScriptRoot)\ETWManifest"
                if (-not (Test-Path $ETWManifestPath))
                {
                    Write-Log "Creating folder '$ETWManifestPath'"
                    New-Item -ItemType Directory -Path $ETWManifestPath
                }
                $copyNoManToDirectories = @(
                    "$webroot\bin",
                    "$packageDirectory\ApplicationPlatform\bin",
                    "$packageDirectory\bin")
                $copyManToDirectories = @(
                    "$webroot\Monitoring",
                    "$webroot\bin\Monitoring",
                    "$packageDirectory\Plugins\Monitoring",
                    "$ETWManifestPath\")

                Write-Log "Blasting non-instrumentation files to these locations: $copyNoManToDirectories"
                Write-Log "Blasting instrumentation files to these locations: $copyManToDirectories"

                # Backup to Original the first time, then to Last on subsequent runs
                $originalBackupFolder = "DbSyncOriginalBackup"
                $lastBackupFolder = "DbSyncLastBackup"
                Write-Log "Default backup folder is $originalBackupFolder"
                $backupFolder = $originalBackupFolder
                foreach ($directory in $copyNoManToDirectories)
                {
                    $originalBackupPath = Join-Path $directory $originalBackupFolder
                    if (Test-Path $originalBackupPath)
                    {
                        Write-Log "Found '$originalBackupPath', changing backup folder to $lastBackupFolder"
                        $backupFolder = $lastBackupFolder
                        break
                    }
                }
                
                $backedUpFiles = @{}
                try
                {
                    Replace-Files $dbSyncApplicableVersionPath $sourceFilesNoMan $copyNoManToDirectories $backupFolder $backedUpFiles
                    Replace-Files $dbSyncApplicableVersionPath $sourceFilesMan   $copyManToDirectories
                }
                catch
                {
                    # We do not want to leave mixed versions in case blast fails, so we do our best to return to the original state.
                    # However, this should not happen as we normally blast to a staging folder. Instead, the backed up files should
                    # be used if DbSync itself fails because of a regression.
                    Write-Exception $_
                    Restore-Backup $backedUpFiles
                    throw
                }

                Update-WebConfig

                Write-Log "DbSyncBlast assembly replacement succeeded."
            }
        }
    }
    else 
    {
       Write-Log "Skipping DbSyncBlast because of DbSyncBlastDisabled flight."
    } 
}
catch
{
    # Write the exception and that this process failed.  
    # Do not throw as this should not cause the deployment or update to fail.
    Write-Exception $_
    Write-Log "DbSyncBlast assembly replacement failed, see $log for details."
}
finally
{
    Write-Log "DbSyncBlast assembly replacement check: Stop"
}

# SIG # Begin signature block
# MIIr/AYJKoZIhvcNAQcCoIIr7TCCK+kCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCA9lsBXtv4PSpQ8
# U3CeeKm0RJfsYRAgHAy25FBy3TGUAqCCEW4wggh+MIIHZqADAgECAhM2AAACAO38
# jbec3qFIAAIAAAIAMA0GCSqGSIb3DQEBCwUAMEExEzARBgoJkiaJk/IsZAEZFgNH
# QkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxFTATBgNVBAMTDEFNRSBDUyBDQSAwMTAe
# Fw0yNDExMDgxMjQzMjhaFw0yNTExMDgxMjQzMjhaMCQxIjAgBgNVBAMTGU1pY3Jv
# c29mdCBBenVyZSBDb2RlIFNpZ24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK
# AoIBAQC5L/UPrOpwYjxcoZC0TqqvMF1WUELvwXN+k27SrA5rohJknn7Cgbxg4hGT
# XKqpcdbtsVTN3ZY896SJ20uQ+INL5OVLzpW408nCNTPYg2LtGJbqHUjpNm0hLCJ+
# gO5Jn2T8DDzIJoUijGXj1m+hRLKb2nOIicCED2GuYBmuWXnaY7INmVEaU3peryty
# ZjDuxdyGDuiPURz8lW1SUiDzoszNp1oswVr+WjDvLDUx4HlxPsG8zUjIst0NnJ6o
# z4tNFKaUBDCetcMjQxpCETn29a1CuRddxZLjPHZHfcotr5sh1S6bNQdzVaMNsxV8
# L3wjHb7XJ6ZVm662mHEiPgpyNcLhAgMBAAGjggWKMIIFhjApBgkrBgEEAYI3FQoE
# HDAaMAwGCisGAQQBgjdbAQEwCgYIKwYBBQUHAwMwPQYJKwYBBAGCNxUHBDAwLgYm
# KwYBBAGCNxUIhpDjDYTVtHiE8Ys+hZvdFs6dEoFgg93NZoaUjDICAWQCAQ4wggJ2
# BggrBgEFBQcBAQSCAmgwggJkMGIGCCsGAQUFBzAChlZodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpaW5mcmEvQ2VydHMvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDEu
# YW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUy
# MDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDIuYW1lLmdibC9haWEv
# QlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBS
# BggrBgEFBQcwAoZGaHR0cDovL2NybDMuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAx
# LkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZG
# aHR0cDovL2NybDQuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDCBrQYIKwYBBQUHMAKGgaBsZGFwOi8vL0NO
# PUFNRSUyMENTJTIwQ0ElMjAwMSxDTj1BSUEsQ049UHVibGljJTIwS2V5JTIwU2Vy
# dmljZXMsQ049U2VydmljZXMsQ049Q29uZmlndXJhdGlvbixEQz1BTUUsREM9R0JM
# P2NBQ2VydGlmaWNhdGU/YmFzZT9vYmplY3RDbGFzcz1jZXJ0aWZpY2F0aW9uQXV0
# aG9yaXR5MB0GA1UdDgQWBBST/HE52ZUlmsYqZcZBdrXZ5u4ZnzAOBgNVHQ8BAf8E
# BAMCB4AwRQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEWMBQGA1UEBRMNMjM2MTY3KzUwMzE1NTCCAeYGA1UdHwSCAd0wggHZMIIB
# 1aCCAdGgggHNhj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpaW5mcmEvQ1JM
# L0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwxLmFtZS5nYmwv
# Y3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwyLmFtZS5n
# YmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwzLmFt
# ZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmw0
# LmFtZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGgb1sZGFwOi8v
# L0NOPUFNRSUyMENTJTIwQ0ElMjAwMSgyKSxDTj1CWTJQS0lDU0NBMDEsQ049Q0RQ
# LENOPVB1YmxpYyUyMEtleSUyMFNlcnZpY2VzLENOPVNlcnZpY2VzLENOPUNvbmZp
# Z3VyYXRpb24sREM9QU1FLERDPUdCTD9jZXJ0aWZpY2F0ZVJldm9jYXRpb25MaXN0
# P2Jhc2U/b2JqZWN0Q2xhc3M9Y1JMRGlzdHJpYnV0aW9uUG9pbnQwHwYDVR0jBBgw
# FoAUllGE4Gtve/7YBqvD8oXmKa5q+dQwHwYDVR0lBBgwFgYKKwYBBAGCN1sBAQYI
# KwYBBQUHAwMwDQYJKoZIhvcNAQELBQADggEBAEDd8Wf5RkHsB64vgn2slxDtHzSo
# It9xN/Dm3RdFjNZ0diTUPMgSPYQlSk8nIAfudnB9FLavGlvZLlyUpfrPSuikepj3
# i3pqNEFn6fNdNFv/wHMxv7hQTIDCmuoR1v1rX+w3oeleBPMnN3QmH4ff1NsynyV4
# dZdYgN9Cw9sC/S3pWZpJrbOs7YOM3vqyU6DciHhC4D9i2zByHCF2pu9nYfiQf5A2
# iUZenRvyo1E5rC+UP2VZXa4k7g66W20+zAajIKKIqEmRtWahekMkCcOIHFBY4RDA
# ybgPRSGur4VDAiZPjTXS90wQXrX9CwU20cfiCC6e76F4H95KtQjKYpzuNVAwggjo
# MIIG0KADAgECAhMfAAAAUeqP9pxzDKg7AAAAAABRMA0GCSqGSIb3DQEBCwUAMDwx
# EzARBgoJkiaJk/IsZAEZFgNHQkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxEDAOBgNV
# BAMTB2FtZXJvb3QwHhcNMjEwNTIxMTg0NDE0WhcNMjYwNTIxMTg1NDE0WjBBMRMw
# EQYKCZImiZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQD
# EwxBTUUgQ1MgQ0EgMDEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDJ
# mlIJfQGejVbXKpcyFPoFSUllalrinfEV6JMc7i+bZDoL9rNHnHDGfJgeuRIYO1LY
# /1f4oMTrhXbSaYRCS5vGc8145WcTZG908bGDCWr4GFLc411WxA+Pv2rteAcz0eHM
# H36qTQ8L0o3XOb2n+x7KJFLokXV1s6pF/WlSXsUBXGaCIIWBXyEchv+sM9eKDsUO
# LdLTITHYJQNWkiryMSEbxqdQUTVZjEz6eLRLkofDAo8pXirIYOgM770CYOiZrcKH
# K7lYOVblx22pdNawY8Te6a2dfoCaWV1QUuazg5VHiC4p/6fksgEILptOKhx9c+ia
# piNhMrHsAYx9pUtppeaFAgMBAAGjggTcMIIE2DASBgkrBgEEAYI3FQEEBQIDAgAC
# MCMGCSsGAQQBgjcVAgQWBBQSaCRCIUfL1Gu+Mc8gpMALI38/RzAdBgNVHQ4EFgQU
# llGE4Gtve/7YBqvD8oXmKa5q+dQwggEEBgNVHSUEgfwwgfkGBysGAQUCAwUGCCsG
# AQUFBwMBBggrBgEFBQcDAgYKKwYBBAGCNxQCAQYJKwYBBAGCNxUGBgorBgEEAYI3
# CgMMBgkrBgEEAYI3FQYGCCsGAQUFBwMJBggrBgEFBQgCAgYKKwYBBAGCN0ABAQYL
# KwYBBAGCNwoDBAEGCisGAQQBgjcKAwQGCSsGAQQBgjcVBQYKKwYBBAGCNxQCAgYK
# KwYBBAGCNxQCAwYIKwYBBQUHAwMGCisGAQQBgjdbAQEGCisGAQQBgjdbAgEGCisG
# AQQBgjdbAwEGCisGAQQBgjdbBQEGCisGAQQBgjdbBAEGCisGAQQBgjdbBAIwGQYJ
# KwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMBIGA1UdEwEB/wQI
# MAYBAf8CAQAwHwYDVR0jBBgwFoAUKV5RXmSuNLnrrJwNp4x1AdEJCygwggFoBgNV
# HR8EggFfMIIBWzCCAVegggFToIIBT4YxaHR0cDovL2NybC5taWNyb3NvZnQuY29t
# L3BraWluZnJhL2NybC9hbWVyb290LmNybIYjaHR0cDovL2NybDIuYW1lLmdibC9j
# cmwvYW1lcm9vdC5jcmyGI2h0dHA6Ly9jcmwzLmFtZS5nYmwvY3JsL2FtZXJvb3Qu
# Y3JshiNodHRwOi8vY3JsMS5hbWUuZ2JsL2NybC9hbWVyb290LmNybIaBqmxkYXA6
# Ly8vQ049YW1lcm9vdCxDTj1BTUVSb290LENOPUNEUCxDTj1QdWJsaWMlMjBLZXkl
# MjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9uLERDPUFNRSxE
# Qz1HQkw/Y2VydGlmaWNhdGVSZXZvY2F0aW9uTGlzdD9iYXNlP29iamVjdENsYXNz
# PWNSTERpc3RyaWJ1dGlvblBvaW50MIIBqwYIKwYBBQUHAQEEggGdMIIBmTBHBggr
# BgEFBQcwAoY7aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraWluZnJhL2NlcnRz
# L0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6Ly9jcmwyLmFt
# ZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6
# Ly9jcmwzLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUH
# MAKGK2h0dHA6Ly9jcmwxLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQw
# gaIGCCsGAQUFBzAChoGVbGRhcDovLy9DTj1hbWVyb290LENOPUFJQSxDTj1QdWJs
# aWMlMjBLZXklMjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9u
# LERDPUFNRSxEQz1HQkw/Y0FDZXJ0aWZpY2F0ZT9iYXNlP29iamVjdENsYXNzPWNl
# cnRpZmljYXRpb25BdXRob3JpdHkwDQYJKoZIhvcNAQELBQADggIBAFAQI7dPD+jf
# XtGt3vJp2pyzA/HUu8hjKaRpM3opya5G3ocprRd7vdTHb8BDfRN+AD0YEmeDB5HK
# QoG6xHPI5TXuIi5sm/LeADbV3C2q0HQOygS/VT+m1W7a/752hMIn+L4ZuyxVeSBp
# fwf7oQ4YSZPh6+ngZvBHgfBaVz4O9/wcfw91QDZnTgK9zAh9yRKKls2bziPEnxeO
# ZMVNaxyV0v152PY2xjqIafIkUjK6vY9LtVFjJXenVUAmn3WCPWNFC1YTIIHw/mD2
# cTfPy7QA1pT+GPARAKt0bKtq9aCd/Ym0b5tPbpgCiRtzyb7fbNS1dE740re0COE6
# 7YV2wbeo2sXixzvLftH8L7s9xv9wV+G22qyKt6lmKLjFK1yMw4Ni5fMabcgmzRvS
# jAcbqgp3tk4a8emaaH0rz8MuuIP+yrxtREPXSqL/C5bzMzsikuDW9xH10graZzSm
# PjilzpRfRdu20/9UQmC7eVPZ4j1WNa1oqPHfzET3ChIzJ6Q9G3NPCB+7KwX0OQmK
# yv7IDimj8U/GlsHD1z+EF/fYMf8YXG15LamaOAohsw/ywO6SYSreVW+5Y0mzJutn
# BC9Cm9ozj1+/4kqksrlhZgR/CSxhFH3BTweH8gP2FEISRtShDZbuYymynY1un+Ry
# fiK9+iVTLdD1h/SxyxDpZMtimb4CgJQlMYIZ5DCCGeACAQEwWDBBMRMwEQYKCZIm
# iZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQDEwxBTUUg
# Q1MgQ0EgMDECEzYAAAIA7fyNt5zeoUgAAgAAAgAwDQYJYIZIAWUDBAIBBQCgga4w
# GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisG
# AQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIJV68Qo4Tp0k/zDHAeUoglJNyQ4jMbn4
# KHAQ6QxSVI5dMEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8AcwBvAGYA
# dKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEA
# moApdL+ilsNcUiJFbdh7ldPpQJDcF3DvyRdDWJMPXkZ3VdMVPaeEGsgKs2TojS60
# //FiQQvkJIggyeyhVWFkQZ7BMk2RZoixc9fJD4qplJx0QwHFbXHS/I6C/CKk+Rgi
# On05RQ5HBvgpKkCNqcrT+dhDxgsn80TWYnPuoqidkh10nbkOwlFSV4v469QTbJbe
# 8KbZQoFiN2/5mAmKxyPUSh9dVbv7D35icyAHVeY1sr+fFP7KTurssqg0KpdG7ae7
# 8kGlfZM9KlAym4xfBGXXauzm51ZXzGPGdyaqImUZfDFCkp660QOHs+VFBoDTzkLL
# 4piNaOa8ijxu5W2Z1SdFs6GCF6wwgheoBgorBgEEAYI3AwMBMYIXmDCCF5QGCSqG
# SIb3DQEHAqCCF4UwgheBAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFaBgsqhkiG9w0B
# CRABBKCCAUkEggFFMIIBQQIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFlAwQCAQUA
# BCAnRKuHndif84DiAo0te7J2ZICz4j64PzwSTQ3sINELbwIGZ5EEr65eGBMyMDI1
# MDEyMjIwNDIxNC40MDZaMASAAgH0oIHZpIHWMIHTMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJlbGFuZCBP
# cGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjo0MDFB
# LTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZaCCEfowggcoMIIFEKADAgECAhMzAAAB/tCowns0IQsBAAEAAAH+MA0GCSqGSIb3
# DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
# VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAk
# BgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMB4XDTI0MDcyNTE4
# MzExOFoXDTI1MTAyMjE4MzExOFowgdMxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
# YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
# Q29ycG9yYXRpb24xLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlv
# bnMgTGltaXRlZDEnMCUGA1UECxMeblNoaWVsZCBUU1MgRVNOOjQwMUEtMDVFMC1E
# OTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNlMIICIjAN
# BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAvLwhFxWlqA43olsE4PCegZ4mSfsH
# 2YTSKEYv8Gn3362Bmaycdf5T3tQxpP3NWm62YHUieIQXw+0u4qlay4AN3IonI+47
# Npi9fo52xdAXMX0pGrc0eqW8RWN3bfzXPKv07O18i2HjDyLuywYyKA9FmWbePjah
# f9Mwd8QgygkPtwDrVQGLyOkyM3VTiHKqhGu9BCGVRdHW9lmPMrrUlPWiYV9LVCB5
# VYd+AEUtdfqAdqlzVxA53EgxSqhp6JbfEKnTdcfP6T8Mir0HrwTTtV2h2yDBtjXb
# QIaqycKOb633GfRkn216LODBg37P/xwhodXT81ZC2aHN7exEDmmbiWssjGvFJkli
# 2g6dt01eShOiGmhbonr0qXXcBeqNb6QoF8jX/uDVtY9pvL4j8aEWS49hKUH0mzsC
# ucIrwUS+x8MuT0uf7VXCFNFbiCUNRTofxJ3B454eGJhL0fwUTRbgyCbpLgKMKDiC
# Rub65DhaeDvUAAJT93KSCoeFCoklPavbgQyahGZDL/vWAVjX5b8Jzhly9gGCdK/q
# i6i+cxZ0S8x6B2yjPbZfdBVfH/NBp/1Ln7xbeOETAOn7OT9D3UGt0q+KiWgY42Hn
# Ljyhl1bAu5HfgryAO3DCaIdV2tjvkJay2qOnF7Dgj8a60KQT9QgfJfwXnr3ZKibY
# MjaUbCNIDnxz2ykCAwEAAaOCAUkwggFFMB0GA1UdDgQWBBRvznuJ9SU2g5l/5/b+
# 5CBibbHF3TAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJlpxtTNRnpcjBfBgNVHR8E
# WDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9N
# aWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcmwwbAYIKwYB
# BQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20v
# cGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0ElMjAyMDEw
# KDEpLmNydDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMA4G
# A1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQsFAAOCAgEAiT4NUvO2lw+0dDMtsBux
# mX2o3lVQqnQkuITAGIGCgI+sl7ZqZOTDd8LqxsH4GWCPTztc3tr8AgBvsYIzWjFw
# ioCjCQODq1oBMWNzEsKzckHxAzYo5Sze7OPkMA3DAxVq4SSR8y+TRC2GcOd0JReZ
# 1lPlhlPl9XI+z8OgtOPmQnLLiP9qzpTHwFze+sbqSn8cekduMZdLyHJk3Niw3Ang
# lU/WTzGsQAdch9SVV4LHifUnmwTf0i07iKtTlNkq3bx1iyWg7N7jGZABRWT2mX+Y
# AVHlK27t9n+WtYbn6cOJNX6LsH8xPVBRYAIRVkWsMyEAdoP9dqfaZzwXGmjuVQ93
# 1NhzHjjG+Efw118DXjk3Vq3qUI1re34zMMTRzZZEw82FupF3viXNR3DVOlS9JH4x
# 5emfINa1uuSac6F4CeJCD1GakfS7D5ayNsaZ2e+sBUh62KVTlhEsQRHZRwCTxbix
# 1Y4iJw+PDNLc0Hf19qX2XiX0u2SM9CWTTjsz9SvCjIKSxCZFCNv/zpKIlsHx7hQN
# QHSMbKh0/wwn86uiIALEjazUszE0+X6rcObDfU4h/O/0vmbF3BMR+45rAZMAETJs
# RDPxHJCo/5XGhWdg/LoJ5XWBrODL44YNrN7FRnHEAAr06sflqZ8eeV3FuDKdP5h1
# 9WUnGWwO1H/ZjUzOoVGiV3gwggdxMIIFWaADAgECAhMzAAAAFcXna54Cm0mZAAAA
# AAAVMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2Fz
# aGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
# cnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0ZSBB
# dXRob3JpdHkgMjAxMDAeFw0yMTA5MzAxODIyMjVaFw0zMDA5MzAxODMyMjVaMHwx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1p
# Y3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIICIjANBgkqhkiG9w0BAQEFAAOC
# Ag8AMIICCgKCAgEA5OGmTOe0ciELeaLL1yR5vQ7VgtP97pwHB9KpbE51yMo1V/YB
# f2xK4OK9uT4XYDP/XE/HZveVU3Fa4n5KWv64NmeFRiMMtY0Tz3cywBAY6GB9alKD
# RLemjkZrBxTzxXb1hlDcwUTIcVxRMTegCjhuje3XD9gmU3w5YQJ6xKr9cmmvHaus
# 9ja+NSZk2pg7uhp7M62AW36MEBydUv626GIl3GoPz130/o5Tz9bshVZN7928jaTj
# kY+yOSxRnOlwaQ3KNi1wjjHINSi947SHJMPgyY9+tVSP3PoFVZhtaDuaRr3tpK56
# KTesy+uDRedGbsoy1cCGMFxPLOJiss254o2I5JasAUq7vnGpF1tnYN74kpEeHT39
# IM9zfUGaRnXNxF803RKJ1v2lIH1+/NmeRd+2ci/bfV+AutuqfjbsNkz2K26oElHo
# vwUDo9Fzpk03dJQcNIIP8BDyt0cY7afomXw/TNuvXsLz1dhzPUNOwTM5TI4CvEJo
# LhDqhFFG4tG9ahhaYQFzymeiXtcodgLiMxhy16cg8ML6EgrXY28MyTZki1ugpoMh
# XV8wdJGUlNi5UPkLiWHzNgY1GIRH29wb0f2y1BzFa/ZcUlFdEtsluq9QBXpsxREd
# cu+N+VLEhReTwDwV2xo3xwgVGD94q0W29R6HXtqPnhZyacaue7e3PmriLq0CAwEA
# AaOCAd0wggHZMBIGCSsGAQQBgjcVAQQFAgMBAAEwIwYJKwYBBAGCNxUCBBYEFCqn
# Uv5kxJq+gpE8RjUpzxD/LwTuMB0GA1UdDgQWBBSfpxVdAF5iXYP05dJlpxtTNRnp
# cjBcBgNVHSAEVTBTMFEGDCsGAQQBgjdMg30BATBBMD8GCCsGAQUFBwIBFjNodHRw
# Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9yeS5odG0w
# EwYDVR0lBAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
# CwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYDVR0jBBgwFoAU1fZWy4/o
# olxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDovL2NybC5taWNy
# b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYt
# MjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+aHR0cDovL3d3dy5t
# aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5j
# cnQwDQYJKoZIhvcNAQELBQADggIBAJ1VffwqreEsH2cBMSRb4Z5yS/ypb+pcFLY+
# TkdkeLEGk5c9MTO1OdfCcTY/2mRsfNB1OW27DzHkwo/7bNGhlBgi7ulmZzpTTd2Y
# urYeeNg2LpypglYAA7AFvonoaeC6Ce5732pvvinLbtg/SHUB2RjebYIM9W0jVOR4
# U3UkV7ndn/OOPcbzaN9l9qRWqveVtihVJ9AkvUCgvxm2EhIRXT0n4ECWOKz3+SmJ
# w7wXsFSFQrP8DJ6LGYnn8AtqgcKBGUIZUnWKNsIdw2FzLixre24/LAl4FOmRsqlb
# 30mjdAy87JGA0j3mSj5mO0+7hvoyGtmW9I/2kQH2zsZ0/fZMcm8Qq3UwxTSwethQ
# /gpY3UA8x1RtnWN0SCyxTkctwRQEcb9k+SS+c23Kjgm9swFXSVRk2XPXfx5bRAGO
# WhmRaw2fpCjcZxkoJLo4S5pu+yFUa2pFEUep8beuyOiJXk+d0tBMdrVXVAmxaQFE
# fnyhYWxz/gq77EFmPWn9y8FBSX5+k77L+DvktxW/tM4+pTFRhLy/AsGConsXHRWJ
# jXD+57XQKBqJC4822rpM+Zv/Cuk0+CQ1ZyvgDbjmjJnW4SLq8CdCPSWU5nR0W2rR
# nj7tfqAxM328y+l7vzhwRNGQ8cirOoo6CGJ/2XBjU02N7oJtpQUQwXEGahC0HVUz
# WLOhcGbyoYIDVTCCAj0CAQEwggEBoYHZpIHWMIHTMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJlbGFuZCBP
# cGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjo0MDFB
# LTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZaIjCgEBMAcGBSsOAwIaAxUAhGNHD/a7Q0bQLWVG9JuGxgLRXseggYMwgYCkfjB8
# MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
# bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1N
# aWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQsFAAIFAOs7
# gywwIhgPMjAyNTAxMjIxNDQ2MDRaGA8yMDI1MDEyMzE0NDYwNFowczA5BgorBgEE
# AYRZCgQBMSswKTAKAgUA6zuDLAIBADAGAgEAAgE+MAcCAQACAhNUMAoCBQDrPNSs
# AgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkKAwKgCjAIAgEAAgMHoSCh
# CjAIAgEAAgMBhqAwDQYJKoZIhvcNAQELBQADggEBAFuhbZsBdf9JwoJf8AIjsosU
# onlg17t6o7J8NIl3NewR3kj6ND2qpteQT105BVEaM7rl1c6GaVNSQsBa6Kmb+0q/
# i6YDSI52xM1kDlE4RHj5G1p5bWGj6t6XtXtM8JCJoLewr34gULB9CF+IdXLyAuUd
# jr1wvJSkl868JgEowScQO6xkzbzpd7qIat8z44OKrNRR3Wfhpb1cidG7eRp+AvfN
# CZmp5hnWbe8hPZ3t3AOSp+HlTn0Idw2eJrOK3KzeWfK2GLnP6AwvFYk63bcUUoJs
# zu23FYMpgPF2YWyalB3if3k2nm3MhP4Qq79wZY0CamFKQAfGUUqF+llymiXLql0x
# ggQNMIIECQIBATCBkzB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAA
# Af7QqMJ7NCELAQABAAAB/jANBglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkD
# MQ0GCyqGSIb3DQEJEAEEMC8GCSqGSIb3DQEJBDEiBCAd+VvsihVjhCv/Wc9GIBQJ
# 29yHnizzk5yow53l88X05zCB+gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EIBGF
# zN38U8ifGNH3abaE9apz68Y4bX78jRa2QKy3KHR5MIGYMIGApH4wfDELMAkGA1UE
# BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAc
# BgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0
# IFRpbWUtU3RhbXAgUENBIDIwMTACEzMAAAH+0KjCezQhCwEAAQAAAf4wIgQgsa23
# SFrGD8GAu/YmJrKQbpMt/N9u+xDySqqNAr5X6SQwDQYJKoZIhvcNAQELBQAEggIA
# uSKJTDxcCXgofMJIk0TljG2HXOjr9q8Ft7ssIEnthIky77KtlvcQEAoPcdNqkKgp
# 7jqMIaFMmAd2w74EnwdZS/TteLjh/8gkrrsR/6iqRsOMBVsBA9yWt1AjxvK4PIpZ
# BqcFzgnULRgeVEfA+1IH5P2GcLbIq3kzxQyPlITHcSO/1AXa1fc7LFcUiDUntAqr
# PivJ/F+MXWvdu4vXafDrtP/csbhP7hTq31WGNM8y4CKAv/FQ7OJCD1Ga8E2wBgBl
# EU5BVXrxp8uuDEZq8vVHBOnMJZVLCLyQ4C5eiUj6m8Cwyv6WWMmBpgp8iBGSKN2N
# gbopLjOJeJzk+KHCuQ7xT9TyGvGC5D55uq+4gV+wZaZ/8zZh4eaI4U5Sx5xYDfA4
# HxDnz7p4cF380K7BacpMOOOnbHdu1MSIywuY4NYBDZYsa9+GoxhnmzQNG3bLJgRz
# hqktIjT8jdfSdEs9wEHuPRq/0SrZkT9EVtRTGnrpRc6NUStd5f+m6n3ozgLh8HZd
# vADXpSJxbhHVXpbcL8X9NEzPtYAQc8HLNG6phAqcXCDdX59IAnQ/vqPbuv58RxLN
# gh70HeO8RVxbpnyMqF3gZZ/DES4DT8rYb7iPEMKx/L+s5+ilvCRhYgX8YrXvfL3o
# DEh+VL8uak/1dBca052Cpayj8P/zcon81pIcbFInSiM=
# SIG # End signature block
