#######################################################
###              Restore Admin Account             ####
#######################################################

PARAM (
    [Parameter(Mandatory=$true)]
    $log,

    [Parameter(Mandatory=$false)]
    $UserInfoTableName = "USERINFO"
)

# Import modules and initialize the log
Import-Module WebAdministration
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking
Initialize-Log $log

function Add-AdminAccount()
{
    $expectedAdminSID = Get-Sid -Email $AdminPrincipalName -Domain $AdminIdentityProvider
    Write-LogWithoutEmail "Inserting a new Admin account record with SID='$($expectedAdminSID)', NetworkDomain='$($AdminIdentityProvider)', and NetworkAlias='$($AdminPrincipalName)'..."
    $target_sqlParams.Query = @"
    INSERT INTO $($UserInfoTableName) (ID, NAME, ENABLE, STATUSLINEINFO, TOOLBARINFO, DEBUGINFO, AUTOINFO, AUTOUPDATE, GARBAGECOLLECTLIMIT, HISTORYLIMIT, MESSAGELIMIT, GENERALINFO, SHOWSTATUSLINE, SHOWTOOLBAR, SHOWAOTLAYER, CONFIRMDELETE, REPORTFONTSIZE, FORMFONTSIZE, PROPERTYFONTSIZE, COMPANY, COMPILERWARNINGLEVEL, SID, NETWORKDOMAIN, NETWORKALIAS, LANGUAGE, HELPLANGUAGE, PREFERREDTIMEZONE, NOTIFYTIMEZONEMISMATCH, ACCOUNTTYPE, DEFAULTPARTITION, INTERACTIVELOGON)  
    VALUES ('Admin', 'Admin', 1, -538365, -1, 12, -1, 6, 20, 5, 1000, -8209, 1, 1, 4, -1, 9, 9, 9, 'DAT', 3, '$($expectedAdminSID)', '$($AdminIdentityProvider)', '$($AdminPrincipalName)', 'en-US', 'en-us', 58, 1, 2, 1, 1)
"@
    Invoke-SqlQueryWithLog $target_sqlParams | out-null
    Write-LogWithoutEmail "Added a new Admin record to $($UserInfoTableName) table."
}

function Ensure-AdminAccountIsEnabled($AdminAccount)
{
    if ($AdminAccount.ENABLE -ne 1)
    {
        Write-LogWithoutEmail "Admin account is disabled. Enabling Admin account..."
        $target_sqlParams.Query = @"
            UPDATE $($UserInfoTableName) set ENABLE = 1 WHERE ID = 'Admin'
"@
        Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
        Write-LogWithoutEmail "Finished enabling Admin account."
    }
    else
    {
        Write-LogWithoutEmail "Validated that Admin account is enabled."
    }
}

function Ensure-AdminAccountHasCorrectNetworkDomain($AdminAccount)
{
    if ($AdminAccount.NETWORKDOMAIN -ne $AdminIdentityProvider)
    {
        Write-LogWithoutEmail "The Admin account's NetworkDomain '$($AdminAccount.NETWORKDOMAIN)' in $($UserInfoTableName) does not match default provisioning admin identity provider '$($AdminIdentityProvider)'."
        $target_sqlParams.Query = @"
            UPDATE $($UserInfoTableName) set NETWORKDOMAIN = '$AdminIdentityProvider' WHERE ID = 'Admin'
"@
        Write-LogWithoutEmail "Updated the Admin account's NetworkDomain to '$($AdminIdentityProvider)' in $($UserInfoTableName)."
        Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
    }
    else
    {
        Write-LogWithoutEmail "Admin network domain is correct."
    }
}

function Ensure-AdminAccountHasCorrectSID($AdminAccount)
{
    $expectedAdminSID = Get-Sid -Email $AdminPrincipalName -Domain $AdminIdentityProvider
    if ([string]::IsNullOrEmpty($expectedAdminSID))
    {
        Write-LogWithoutEmail "Expected admin SID is null or empty. Ignoring database check for admin SID."
    }
    elseif ($AdminAccount.SID -ne $expectedAdminSID)
    {
        Write-LogWithoutEmail "SID in database '$($AdminAccount.SID)' does not match generated SID '$($expectedAdminSID)'. Updating SID in database..."
        $target_sqlParams.Query = @"
            UPDATE $($UserInfoTableName) set SID = '$($expectedAdminSID)' WHERE ID = 'Admin'
"@
        Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
        Write-LogWithoutEmail "Finished updating the SID."
    }
    else
    {
        Write-LogWithoutEmail "SID in database matches the generated SID."
    }
}

function Ensure-NoDuplicateAdminAccounts()
{
    $adminUserCount = Get-UserCount -Id "Admin"
    # Validate that ADMIN account is unique.
    if ($adminUserCount -gt 1)
    {
        Write-LogWithoutEmail "$($adminUserCount) admin records found. Removing duplicates."
        $target_sqlParams.Query = @"
        DECLARE @expectedCount int = 0,
        @adminPrincipalName varchar(100) = '$($AdminPrincipalName)'
            SELECT @expectedCount = COUNT(*) FROM $($UserInfoTableName) WHERE ID = 'Admin' and NETWORKALIAS = @adminPrincipalName;
    
                    IF @expectedCount >= 1 
                    BEGIN
                        DELETE $($UserInfoTableName) WHERE ID = 'Admin' and NETWORKALIAS != @adminPrincipalName;
                        IF @expectedCount > 1 
                        BEGIN
                            DELETE TOP(@expectedCount - 1) $($UserInfoTableName) WHERE ID = 'Admin'
                        END;
                    END;

                    ELSE
                    BEGIN
                        DELETE TOP($($adminUserCount) - 1) $($UserInfoTableName) WHERE ID = 'Admin'
                    END;
"@
        Invoke-SqlQueryWithLog $target_sqlParams | out-null
        Write-LogWithoutEmail "Removed the duplicate Admin records."
    }
    else
    {
        Write-LogWithoutEmail "No duplicate Admin records found in $($UserInfoTableName) table."
    }
}

function Ensure-NoDuplicateNetworkAliasAccounts($NetworkAlias)
{
    # Check if db has duplicate networkalias with admin account
    $adminUserCount = Get-UserCount -NetworkAlias $NetworkAlias

    # Validate that ADMIN networkalias account is not duplicated.
    if ($adminUserCount -gt 1)
    {
        $target_sqlParams.Query = @"
        DELETE $($UserInfoTableName) WHERE NETWORKALIAS = '$($NetworkAlias)' AND ID != 'Admin'
"@
        Invoke-SqlQueryWithLog $target_sqlParams | out-null
        Write-LogWithoutEmail "Removed the non Admin records where NetworkAlias was set to '$($NetworkAlias)'."
    }
}

function Get-AdminAccount()
{
    $target_sqlParams.Query = @"
    SELECT	ENABLE,
            NETWORKALIAS,
            IDENTITYPROVIDER,
            NETWORKDOMAIN,
            SID 
    FROM $($UserInfoTableName) WHERE ID = 'Admin'
"@
    return Invoke-SqlQueryWithLog $target_sqlParams
}

function Get-Sid($Email, $Domain)
{
    $sid = $null
    $dllPath = Join-Path $webrootBinPath "Microsoft.Dynamics.AX.Security.SidGenerator.dll"
    if (Test-Path $dllPath)
    {
        Write-LogWithoutEmail "Loading Microsoft.Dynamics.AX.Security.SidGenerator.dll from path '$($dllPath)' to generate SID..."
        $assembly = [Reflection.Assembly]::LoadFile($dllPath)

        Write-LogWithoutEmail "Loading SidGenerator type from assembly..."
        $sidGeneratorType = $assembly.GetType("Microsoft.Dynamics.Ax.Security.SidGenerator")

        Write-LogWithoutEmail "Attempting to find new SidGenerator method 'GenerateSIDByDefaultAlgo' (available in PU37+)..."
        $newGenerateSidMethod = $sidGeneratorType.GetMethod("GenerateSIDByDefaultAlgo")
        if ($null -ne $newGenerateSidMethod)
        {
            Write-LogWithoutEmail "New SidGenerator::GenerateSIDByDefaultAlgo(...) method found - using it."
            Write-LogWithoutEmail "Generating SID using email '$($Email)' and domain '$($Domain)'..."
            $sid = [Microsoft.Dynamics.Ax.Security.SidGenerator]::GenerateSIDByDefaultAlgo($Email, $Domain)
        }
        else
        {
            Write-LogWithoutEmail "Could not find method SidGenerator::GenerateSIDByDefaultAlgo(...) - will use old SidGenerator::Generate(email, domain) method instead."
            Write-LogWithoutEmail "Generating SID using email '$($Email)' and domain '$($Domain)'..."
            $sid = [Microsoft.Dynamics.Ax.Security.SidGenerator]::Generate($Email, $Domain)
        }

        Write-LogWithoutEmail "Generated SID: '$($sid)'"
    }
    else
    {
        Write-LogWithoutEmail "Unable to find SIDGenerator assembly '$($dllPath)'. Cannot generate SID."
    }

    return $sid
}

function Get-UserCount(
    [Parameter(Mandatory=$true, ParameterSetName="Id")]
    [String]
    $Id,

    [Parameter(Mandatory=$true, ParameterSetName="NetworkAlias")]
    [String]
    $NetworkAlias
)
{
    if (-not [string]::IsNullOrEmpty($Id))
    {
        $whereClause = "ID = '$($Id)'"
    }
    else
    {
        $whereClause = "NETWORKALIAS = '$($NetworkAlias)'"
    }

    $target_sqlParams.Query = @"
        SELECT COUNT(*) AS UserCount FROM $($UserInfoTableName) WHERE $($whereClause)
"@
    $userCountResults = Invoke-SqlQueryWithLog $target_sqlParams

    [int]$count = $userCountResults.UserCount
    Write-LogWithoutEmail "Number of users with $($whereClause): $($count)"
    return $count
}

function Initialize-Settings()
{
    $webroot = Get-AosWebSitePhysicalPath
    Write-LogWithoutEmail "The Service drive path of the webroot on the aos machine is: $($webroot)"

    $script:webrootBinPath = Join-Path $webroot 'bin'
    $environmentDllPath = Join-Path $webrootBinPath 'Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
    Add-Type -Path $environmentDllPath

    # Load application configuration
    $script:config = [Microsoft.Dynamics.ApplicationPlatform.Environment.EnvironmentFactory]::GetApplicationEnvironment()

    $script:target_sqlParams = @{
        'Database'       = $($config.DataAccess.Database)
        'UserName'       = $($config.DataAccess.SqlUser)
        'Password'       = $($config.DataAccess.SqlPwd)
        'ServerInstance' = $($config.DataAccess.DbServer)
        'Query'          = ''
        'QueryTimeout'   = 0 }


    $script:TenantId = $($config.Aad.AADTenantId)
    $script:AdminPrincipalName = $($config.Provisioning.AdminPrincipalName)
    $script:AdminIdentityProvider = $($config.Provisioning.AdminIdentityProvider)
    if (-not $AdminIdentityProvider) { $script:AdminIdentityProvider = 'https://sts.windows.net/' }

    if ($AdminPrincipalName -eq $null)
    {
        throw "Admin principal name in web.config is null so stopping restore of Admin account. Please check Admin account in DB manually and update the web.config for Admin principal name."
    }
}

function Invoke-SqlQueryWithLog($params)
{
    Write-LogWithoutEmail "Invoking SQL query:
$($params.Query)"
    try
    {
        return Invoke-SqlCmd @params -ErrorAction Stop
    }
    catch
    {
        Write-LogWithoutEmail "An exception occurred: $($_.Exception)"
    }
}

function Remove-DuplicateNetworkAliasAccounts($UserCount, $NetworkAlias)
{
    # Validate that Admin networkalias account is not duplicated.
    if ($UserCount -gt 1)
    {
        Write-LogWithoutEmail "Found $($UserCount) accounts with NetworkAlias '$($AdminPrincipalName)'. Removing duplicates (leaving only one record)..."

        # Remove duplicated records
        $target_sqlParams.Query = @"
            DELETE TOP($($UserCount-1)) FROM $($UserInfoTableName) WHERE NETWORKALIAS = '$AdminPrincipalName' 
"@
        Invoke-SqlQueryWithLog $target_sqlParams | out-null
        Write-LogWithoutEmail "Removed duplicated records for NetworkAlias '$($AdminPrincipalName)'."
    }
}

function Rename-AdminAccount()
{
    $PreviousAdminId = "OldAdmin{0:yyyyMMddHHmm}" -f [DateTime]::UtcNow

    # Update id of current admin account to different id
    $target_sqlParams.Query = @"
    UPDATE $($UserInfoTableName) set ID = '$PreviousAdminId' WHERE ID = 'Admin'
"@
    Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
    Write-LogWithoutEmail "Updated id of existing Admin account. New id is '$($PreviousAdminId)'."
}

function Set-AccountAsAdmin($NetworkAlias)
{
    Write-LogWithoutEmail "Updating ID (to 'Admin') and enabling account with NetworkAlias '$($NetworkAlias)'..."

    # Enable and update id of the user with networkalias of admin principal name to 'Admin' 
    $target_sqlParams.Query = @"
        UPDATE $($UserInfoTableName) set Enable = 1, ID = 'Admin' WHERE NETWORKALIAS = '$NetworkAlias'
"@
    Invoke-SqlQueryWithLog $target_sqlParams | Out-Null
    Write-LogWithoutEmail "Updated id of account with NetworkAlias '$($NetworkAlias)'."
}

function Test-DoesNetworkAliasMatchTenantId($NetworkAlias)
{
    if (-not [string]::IsNullOrEmpty($NetworkAlias))
    {
        $NetworkAliasDomainSuffix = $NetworkAlias.SubString($NetworkAlias.IndexOf('@') + 1)

        # Validate that admin is the same provisioning admin and has the same domain as the tenant
        if ($NetworkAliasDomainSuffix -ne $TenantId)
        {
            Write-LogWithoutEmail "Admin in $($UserInfoTableName) has domain '$($NetworkAliasDomainSuffix)' in the NetworkAlias field, which doesn't match the 'Aad.AADTenantId' value in web.config: '$($TenantId)'."
        }
    }
}

function Write-LogWithoutEmail($LogMessage)
{
    # Regular expression from http://www.regular-expressions.info/email.html
    # Except we're not allowing ' to be the first character in the email address
    # since that messes with logging where we have a string like '<email address>' (single quotes included)
    # $1 in <email>@$1 is for the <Domain> capture group
    # meaning an email username@domain.com will be replaced with <email>@domain.com
    $LogMessageWithoutEmail = $LogMessage -replace "[a-z0-9!#\$%&*+/=?^_`{|}~-]+(?:\.[a-z0-9!#\$%&'*+/=?^_`{|}~-]+)*@(?<Domain>(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9]))?", '<email>@$1'
    Write-Log $LogMessageWithoutEmail
}

try
{
    Write-LogWithoutEmail "Restore Admin Account: Start"
    $error.Clear()

    Initialize-Settings

    Ensure-NoDuplicateAdminAccounts

    $adminUserCount = Get-UserCount -Id "Admin"
    if ($adminUserCount -eq 0)
    {
        Write-LogWithoutEmail "$($UserInfoTableName) has no Admin account. Need to create Admin account."
        Add-AdminAccount
    }
    else
    {
        Write-LogWithoutEmail "Validated that Admin account exists."
        
        $adminAccount = Get-AdminAccount
        $adminAccountNetworkAlias = $adminAccount.NETWORKALIAS
        Test-DoesNetworkAliasMatchTenantId -NetworkAlias $adminAccountNetworkAlias

        if ($adminAccountNetworkAlias -eq $AdminPrincipalName)
        {
            Write-LogWithoutEmail "Validated that admin network alias '$($adminAccountNetworkAlias)' matches the admin principal name '$($AdminPrincipalName)' in the config."
            Ensure-NoDuplicateNetworkAliasAccounts -NetworkAlias $AdminPrincipalName
        }
        else
        {
            Write-LogWithoutEmail "Admin network alias '$($adminAccountNetworkAlias)' in $($UserInfoTableName) is not the same as default provisioning admin principal name '$($AdminPrincipalName)'. Will update id of existing Admin account."
            Rename-AdminAccount

            $adminNetworkAliasUserCount = Get-UserCount -NetworkAlias $AdminPrincipalName

            # Validate if there is non admin account in USERINFO has same NETWORKALIAS with default provisioning admin principal name.
            if ($adminNetworkAliasUserCount -ge 1)
            {
                Remove-DuplicateNetworkAliasAccounts -UserCount $adminNetworkAliasUserCount -NetworkAlias $AdminPrincipalName
                Set-AccountAsAdmin -NetworkAlias $AdminPrincipalName
            }
            else
            {
                Write-LogWithoutEmail "No account found with NetworkAlias '$($AdminPrincipalName)'. Will create new Admin account record."
                Add-AdminAccount
            }
        }
    }

    $adminAccount = Get-AdminAccount
    Ensure-AdminAccountIsEnabled -AdminAccount $adminAccount
    Ensure-AdminAccountHasCorrectSID -AdminAccount $adminAccount
    Ensure-AdminAccountHasCorrectNetworkDomain -AdminAccount $adminAccount
}
Catch
{
    # Write the exception and that this process failed.  
    # Do not throw as this should not cause the deployment or update to fail.
    Write-Exception $_
    Write-LogWithoutEmail "Restore Admin Account failed, see $log for details."
}
Finally
{
    Write-LogWithoutEmail "Restore Admin Account: Stop"
}

# SIG # Begin signature block
# MIIr/AYJKoZIhvcNAQcCoIIr7TCCK+kCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBixKloG51Xld+P
# iHfYWRDWZcIge70qnkaDkV/xoMTj2KCCEW4wggh+MIIHZqADAgECAhM2AAACAO38
# jbec3qFIAAIAAAIAMA0GCSqGSIb3DQEBCwUAMEExEzARBgoJkiaJk/IsZAEZFgNH
# QkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxFTATBgNVBAMTDEFNRSBDUyBDQSAwMTAe
# Fw0yNDExMDgxMjQzMjhaFw0yNTExMDgxMjQzMjhaMCQxIjAgBgNVBAMTGU1pY3Jv
# c29mdCBBenVyZSBDb2RlIFNpZ24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK
# AoIBAQC5L/UPrOpwYjxcoZC0TqqvMF1WUELvwXN+k27SrA5rohJknn7Cgbxg4hGT
# XKqpcdbtsVTN3ZY896SJ20uQ+INL5OVLzpW408nCNTPYg2LtGJbqHUjpNm0hLCJ+
# gO5Jn2T8DDzIJoUijGXj1m+hRLKb2nOIicCED2GuYBmuWXnaY7INmVEaU3peryty
# ZjDuxdyGDuiPURz8lW1SUiDzoszNp1oswVr+WjDvLDUx4HlxPsG8zUjIst0NnJ6o
# z4tNFKaUBDCetcMjQxpCETn29a1CuRddxZLjPHZHfcotr5sh1S6bNQdzVaMNsxV8
# L3wjHb7XJ6ZVm662mHEiPgpyNcLhAgMBAAGjggWKMIIFhjApBgkrBgEEAYI3FQoE
# HDAaMAwGCisGAQQBgjdbAQEwCgYIKwYBBQUHAwMwPQYJKwYBBAGCNxUHBDAwLgYm
# KwYBBAGCNxUIhpDjDYTVtHiE8Ys+hZvdFs6dEoFgg93NZoaUjDICAWQCAQ4wggJ2
# BggrBgEFBQcBAQSCAmgwggJkMGIGCCsGAQUFBzAChlZodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpaW5mcmEvQ2VydHMvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDEu
# YW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUy
# MDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDIuYW1lLmdibC9haWEv
# QlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBS
# BggrBgEFBQcwAoZGaHR0cDovL2NybDMuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAx
# LkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZG
# aHR0cDovL2NybDQuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDCBrQYIKwYBBQUHMAKGgaBsZGFwOi8vL0NO
# PUFNRSUyMENTJTIwQ0ElMjAwMSxDTj1BSUEsQ049UHVibGljJTIwS2V5JTIwU2Vy
# dmljZXMsQ049U2VydmljZXMsQ049Q29uZmlndXJhdGlvbixEQz1BTUUsREM9R0JM
# P2NBQ2VydGlmaWNhdGU/YmFzZT9vYmplY3RDbGFzcz1jZXJ0aWZpY2F0aW9uQXV0
# aG9yaXR5MB0GA1UdDgQWBBST/HE52ZUlmsYqZcZBdrXZ5u4ZnzAOBgNVHQ8BAf8E
# BAMCB4AwRQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEWMBQGA1UEBRMNMjM2MTY3KzUwMzE1NTCCAeYGA1UdHwSCAd0wggHZMIIB
# 1aCCAdGgggHNhj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpaW5mcmEvQ1JM
# L0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwxLmFtZS5nYmwv
# Y3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwyLmFtZS5n
# YmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwzLmFt
# ZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmw0
# LmFtZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGgb1sZGFwOi8v
# L0NOPUFNRSUyMENTJTIwQ0ElMjAwMSgyKSxDTj1CWTJQS0lDU0NBMDEsQ049Q0RQ
# LENOPVB1YmxpYyUyMEtleSUyMFNlcnZpY2VzLENOPVNlcnZpY2VzLENOPUNvbmZp
# Z3VyYXRpb24sREM9QU1FLERDPUdCTD9jZXJ0aWZpY2F0ZVJldm9jYXRpb25MaXN0
# P2Jhc2U/b2JqZWN0Q2xhc3M9Y1JMRGlzdHJpYnV0aW9uUG9pbnQwHwYDVR0jBBgw
# FoAUllGE4Gtve/7YBqvD8oXmKa5q+dQwHwYDVR0lBBgwFgYKKwYBBAGCN1sBAQYI
# KwYBBQUHAwMwDQYJKoZIhvcNAQELBQADggEBAEDd8Wf5RkHsB64vgn2slxDtHzSo
# It9xN/Dm3RdFjNZ0diTUPMgSPYQlSk8nIAfudnB9FLavGlvZLlyUpfrPSuikepj3
# i3pqNEFn6fNdNFv/wHMxv7hQTIDCmuoR1v1rX+w3oeleBPMnN3QmH4ff1NsynyV4
# dZdYgN9Cw9sC/S3pWZpJrbOs7YOM3vqyU6DciHhC4D9i2zByHCF2pu9nYfiQf5A2
# iUZenRvyo1E5rC+UP2VZXa4k7g66W20+zAajIKKIqEmRtWahekMkCcOIHFBY4RDA
# ybgPRSGur4VDAiZPjTXS90wQXrX9CwU20cfiCC6e76F4H95KtQjKYpzuNVAwggjo
# MIIG0KADAgECAhMfAAAAUeqP9pxzDKg7AAAAAABRMA0GCSqGSIb3DQEBCwUAMDwx
# EzARBgoJkiaJk/IsZAEZFgNHQkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxEDAOBgNV
# BAMTB2FtZXJvb3QwHhcNMjEwNTIxMTg0NDE0WhcNMjYwNTIxMTg1NDE0WjBBMRMw
# EQYKCZImiZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQD
# EwxBTUUgQ1MgQ0EgMDEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDJ
# mlIJfQGejVbXKpcyFPoFSUllalrinfEV6JMc7i+bZDoL9rNHnHDGfJgeuRIYO1LY
# /1f4oMTrhXbSaYRCS5vGc8145WcTZG908bGDCWr4GFLc411WxA+Pv2rteAcz0eHM
# H36qTQ8L0o3XOb2n+x7KJFLokXV1s6pF/WlSXsUBXGaCIIWBXyEchv+sM9eKDsUO
# LdLTITHYJQNWkiryMSEbxqdQUTVZjEz6eLRLkofDAo8pXirIYOgM770CYOiZrcKH
# K7lYOVblx22pdNawY8Te6a2dfoCaWV1QUuazg5VHiC4p/6fksgEILptOKhx9c+ia
# piNhMrHsAYx9pUtppeaFAgMBAAGjggTcMIIE2DASBgkrBgEEAYI3FQEEBQIDAgAC
# MCMGCSsGAQQBgjcVAgQWBBQSaCRCIUfL1Gu+Mc8gpMALI38/RzAdBgNVHQ4EFgQU
# llGE4Gtve/7YBqvD8oXmKa5q+dQwggEEBgNVHSUEgfwwgfkGBysGAQUCAwUGCCsG
# AQUFBwMBBggrBgEFBQcDAgYKKwYBBAGCNxQCAQYJKwYBBAGCNxUGBgorBgEEAYI3
# CgMMBgkrBgEEAYI3FQYGCCsGAQUFBwMJBggrBgEFBQgCAgYKKwYBBAGCN0ABAQYL
# KwYBBAGCNwoDBAEGCisGAQQBgjcKAwQGCSsGAQQBgjcVBQYKKwYBBAGCNxQCAgYK
# KwYBBAGCNxQCAwYIKwYBBQUHAwMGCisGAQQBgjdbAQEGCisGAQQBgjdbAgEGCisG
# AQQBgjdbAwEGCisGAQQBgjdbBQEGCisGAQQBgjdbBAEGCisGAQQBgjdbBAIwGQYJ
# KwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMBIGA1UdEwEB/wQI
# MAYBAf8CAQAwHwYDVR0jBBgwFoAUKV5RXmSuNLnrrJwNp4x1AdEJCygwggFoBgNV
# HR8EggFfMIIBWzCCAVegggFToIIBT4YxaHR0cDovL2NybC5taWNyb3NvZnQuY29t
# L3BraWluZnJhL2NybC9hbWVyb290LmNybIYjaHR0cDovL2NybDIuYW1lLmdibC9j
# cmwvYW1lcm9vdC5jcmyGI2h0dHA6Ly9jcmwzLmFtZS5nYmwvY3JsL2FtZXJvb3Qu
# Y3JshiNodHRwOi8vY3JsMS5hbWUuZ2JsL2NybC9hbWVyb290LmNybIaBqmxkYXA6
# Ly8vQ049YW1lcm9vdCxDTj1BTUVSb290LENOPUNEUCxDTj1QdWJsaWMlMjBLZXkl
# MjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9uLERDPUFNRSxE
# Qz1HQkw/Y2VydGlmaWNhdGVSZXZvY2F0aW9uTGlzdD9iYXNlP29iamVjdENsYXNz
# PWNSTERpc3RyaWJ1dGlvblBvaW50MIIBqwYIKwYBBQUHAQEEggGdMIIBmTBHBggr
# BgEFBQcwAoY7aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraWluZnJhL2NlcnRz
# L0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6Ly9jcmwyLmFt
# ZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6
# Ly9jcmwzLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUH
# MAKGK2h0dHA6Ly9jcmwxLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQw
# gaIGCCsGAQUFBzAChoGVbGRhcDovLy9DTj1hbWVyb290LENOPUFJQSxDTj1QdWJs
# aWMlMjBLZXklMjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9u
# LERDPUFNRSxEQz1HQkw/Y0FDZXJ0aWZpY2F0ZT9iYXNlP29iamVjdENsYXNzPWNl
# cnRpZmljYXRpb25BdXRob3JpdHkwDQYJKoZIhvcNAQELBQADggIBAFAQI7dPD+jf
# XtGt3vJp2pyzA/HUu8hjKaRpM3opya5G3ocprRd7vdTHb8BDfRN+AD0YEmeDB5HK
# QoG6xHPI5TXuIi5sm/LeADbV3C2q0HQOygS/VT+m1W7a/752hMIn+L4ZuyxVeSBp
# fwf7oQ4YSZPh6+ngZvBHgfBaVz4O9/wcfw91QDZnTgK9zAh9yRKKls2bziPEnxeO
# ZMVNaxyV0v152PY2xjqIafIkUjK6vY9LtVFjJXenVUAmn3WCPWNFC1YTIIHw/mD2
# cTfPy7QA1pT+GPARAKt0bKtq9aCd/Ym0b5tPbpgCiRtzyb7fbNS1dE740re0COE6
# 7YV2wbeo2sXixzvLftH8L7s9xv9wV+G22qyKt6lmKLjFK1yMw4Ni5fMabcgmzRvS
# jAcbqgp3tk4a8emaaH0rz8MuuIP+yrxtREPXSqL/C5bzMzsikuDW9xH10graZzSm
# PjilzpRfRdu20/9UQmC7eVPZ4j1WNa1oqPHfzET3ChIzJ6Q9G3NPCB+7KwX0OQmK
# yv7IDimj8U/GlsHD1z+EF/fYMf8YXG15LamaOAohsw/ywO6SYSreVW+5Y0mzJutn
# BC9Cm9ozj1+/4kqksrlhZgR/CSxhFH3BTweH8gP2FEISRtShDZbuYymynY1un+Ry
# fiK9+iVTLdD1h/SxyxDpZMtimb4CgJQlMYIZ5DCCGeACAQEwWDBBMRMwEQYKCZIm
# iZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQDEwxBTUUg
# Q1MgQ0EgMDECEzYAAAIA7fyNt5zeoUgAAgAAAgAwDQYJYIZIAWUDBAIBBQCgga4w
# GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisG
# AQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEICY7wWrP9D9SZq7vFUcRARSzevphmZdf
# 9xgkaTHZOL7+MEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8AcwBvAGYA
# dKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEA
# t9n1sIX2oPdoulH4Wp8IrKvsmIP7Ug3aQsxI3UgQTL3uG3FZ1lo5qFaoLovcn/nQ
# 4UzlvU2KOKH5HDXLMPfPdC0EbArcQ5xWsrgjlAulMfp4Cj0c+WFrrZIirzyxwRbC
# 9J3JrWYJ28SYcHn+wvN6FWauiv4kqt3baYv9YsUELYsTCupMGugUq8R/pVlXqpLG
# /BvZwf3Bazza0kLbQPq7lF5Oto7OkT6XeIaeyhkTorwG30L6giQotP9lE4beXGR8
# VYgMjQPq5SDhWhdVT8ulofnY4u91h4dQWzYkoCx9gmzzgbltNS4T7bDfXJWTDl+g
# VO4u1B8r6S1sL/dArvIp3qGCF6wwgheoBgorBgEEAYI3AwMBMYIXmDCCF5QGCSqG
# SIb3DQEHAqCCF4UwgheBAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFaBgsqhkiG9w0B
# CRABBKCCAUkEggFFMIIBQQIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFlAwQCAQUA
# BCAK7BwmIvXg2tEY+g2cR4NaMGRxB5l+l/UysK4n/qU8mgIGZ5EEr7QDGBMyMDI1
# MDEyMjIwNDMwOS4yNThaMASAAgH0oIHZpIHWMIHTMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJlbGFuZCBP
# cGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjo0MDFB
# LTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZaCCEfowggcoMIIFEKADAgECAhMzAAAB/tCowns0IQsBAAEAAAH+MA0GCSqGSIb3
# DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
# VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAk
# BgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMB4XDTI0MDcyNTE4
# MzExOFoXDTI1MTAyMjE4MzExOFowgdMxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
# YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
# Q29ycG9yYXRpb24xLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlv
# bnMgTGltaXRlZDEnMCUGA1UECxMeblNoaWVsZCBUU1MgRVNOOjQwMUEtMDVFMC1E
# OTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNlMIICIjAN
# BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAvLwhFxWlqA43olsE4PCegZ4mSfsH
# 2YTSKEYv8Gn3362Bmaycdf5T3tQxpP3NWm62YHUieIQXw+0u4qlay4AN3IonI+47
# Npi9fo52xdAXMX0pGrc0eqW8RWN3bfzXPKv07O18i2HjDyLuywYyKA9FmWbePjah
# f9Mwd8QgygkPtwDrVQGLyOkyM3VTiHKqhGu9BCGVRdHW9lmPMrrUlPWiYV9LVCB5
# VYd+AEUtdfqAdqlzVxA53EgxSqhp6JbfEKnTdcfP6T8Mir0HrwTTtV2h2yDBtjXb
# QIaqycKOb633GfRkn216LODBg37P/xwhodXT81ZC2aHN7exEDmmbiWssjGvFJkli
# 2g6dt01eShOiGmhbonr0qXXcBeqNb6QoF8jX/uDVtY9pvL4j8aEWS49hKUH0mzsC
# ucIrwUS+x8MuT0uf7VXCFNFbiCUNRTofxJ3B454eGJhL0fwUTRbgyCbpLgKMKDiC
# Rub65DhaeDvUAAJT93KSCoeFCoklPavbgQyahGZDL/vWAVjX5b8Jzhly9gGCdK/q
# i6i+cxZ0S8x6B2yjPbZfdBVfH/NBp/1Ln7xbeOETAOn7OT9D3UGt0q+KiWgY42Hn
# Ljyhl1bAu5HfgryAO3DCaIdV2tjvkJay2qOnF7Dgj8a60KQT9QgfJfwXnr3ZKibY
# MjaUbCNIDnxz2ykCAwEAAaOCAUkwggFFMB0GA1UdDgQWBBRvznuJ9SU2g5l/5/b+
# 5CBibbHF3TAfBgNVHSMEGDAWgBSfpxVdAF5iXYP05dJlpxtTNRnpcjBfBgNVHR8E
# WDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9N
# aWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcmwwbAYIKwYB
# BQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20v
# cGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0ElMjAyMDEw
# KDEpLmNydDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMA4G
# A1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQsFAAOCAgEAiT4NUvO2lw+0dDMtsBux
# mX2o3lVQqnQkuITAGIGCgI+sl7ZqZOTDd8LqxsH4GWCPTztc3tr8AgBvsYIzWjFw
# ioCjCQODq1oBMWNzEsKzckHxAzYo5Sze7OPkMA3DAxVq4SSR8y+TRC2GcOd0JReZ
# 1lPlhlPl9XI+z8OgtOPmQnLLiP9qzpTHwFze+sbqSn8cekduMZdLyHJk3Niw3Ang
# lU/WTzGsQAdch9SVV4LHifUnmwTf0i07iKtTlNkq3bx1iyWg7N7jGZABRWT2mX+Y
# AVHlK27t9n+WtYbn6cOJNX6LsH8xPVBRYAIRVkWsMyEAdoP9dqfaZzwXGmjuVQ93
# 1NhzHjjG+Efw118DXjk3Vq3qUI1re34zMMTRzZZEw82FupF3viXNR3DVOlS9JH4x
# 5emfINa1uuSac6F4CeJCD1GakfS7D5ayNsaZ2e+sBUh62KVTlhEsQRHZRwCTxbix
# 1Y4iJw+PDNLc0Hf19qX2XiX0u2SM9CWTTjsz9SvCjIKSxCZFCNv/zpKIlsHx7hQN
# QHSMbKh0/wwn86uiIALEjazUszE0+X6rcObDfU4h/O/0vmbF3BMR+45rAZMAETJs
# RDPxHJCo/5XGhWdg/LoJ5XWBrODL44YNrN7FRnHEAAr06sflqZ8eeV3FuDKdP5h1
# 9WUnGWwO1H/ZjUzOoVGiV3gwggdxMIIFWaADAgECAhMzAAAAFcXna54Cm0mZAAAA
# AAAVMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2Fz
# aGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
# cnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0ZSBB
# dXRob3JpdHkgMjAxMDAeFw0yMTA5MzAxODIyMjVaFw0zMDA5MzAxODMyMjVaMHwx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1p
# Y3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIICIjANBgkqhkiG9w0BAQEFAAOC
# Ag8AMIICCgKCAgEA5OGmTOe0ciELeaLL1yR5vQ7VgtP97pwHB9KpbE51yMo1V/YB
# f2xK4OK9uT4XYDP/XE/HZveVU3Fa4n5KWv64NmeFRiMMtY0Tz3cywBAY6GB9alKD
# RLemjkZrBxTzxXb1hlDcwUTIcVxRMTegCjhuje3XD9gmU3w5YQJ6xKr9cmmvHaus
# 9ja+NSZk2pg7uhp7M62AW36MEBydUv626GIl3GoPz130/o5Tz9bshVZN7928jaTj
# kY+yOSxRnOlwaQ3KNi1wjjHINSi947SHJMPgyY9+tVSP3PoFVZhtaDuaRr3tpK56
# KTesy+uDRedGbsoy1cCGMFxPLOJiss254o2I5JasAUq7vnGpF1tnYN74kpEeHT39
# IM9zfUGaRnXNxF803RKJ1v2lIH1+/NmeRd+2ci/bfV+AutuqfjbsNkz2K26oElHo
# vwUDo9Fzpk03dJQcNIIP8BDyt0cY7afomXw/TNuvXsLz1dhzPUNOwTM5TI4CvEJo
# LhDqhFFG4tG9ahhaYQFzymeiXtcodgLiMxhy16cg8ML6EgrXY28MyTZki1ugpoMh
# XV8wdJGUlNi5UPkLiWHzNgY1GIRH29wb0f2y1BzFa/ZcUlFdEtsluq9QBXpsxREd
# cu+N+VLEhReTwDwV2xo3xwgVGD94q0W29R6HXtqPnhZyacaue7e3PmriLq0CAwEA
# AaOCAd0wggHZMBIGCSsGAQQBgjcVAQQFAgMBAAEwIwYJKwYBBAGCNxUCBBYEFCqn
# Uv5kxJq+gpE8RjUpzxD/LwTuMB0GA1UdDgQWBBSfpxVdAF5iXYP05dJlpxtTNRnp
# cjBcBgNVHSAEVTBTMFEGDCsGAQQBgjdMg30BATBBMD8GCCsGAQUFBwIBFjNodHRw
# Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9yeS5odG0w
# EwYDVR0lBAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
# CwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYDVR0jBBgwFoAU1fZWy4/o
# olxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDovL2NybC5taWNy
# b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYt
# MjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+aHR0cDovL3d3dy5t
# aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5j
# cnQwDQYJKoZIhvcNAQELBQADggIBAJ1VffwqreEsH2cBMSRb4Z5yS/ypb+pcFLY+
# TkdkeLEGk5c9MTO1OdfCcTY/2mRsfNB1OW27DzHkwo/7bNGhlBgi7ulmZzpTTd2Y
# urYeeNg2LpypglYAA7AFvonoaeC6Ce5732pvvinLbtg/SHUB2RjebYIM9W0jVOR4
# U3UkV7ndn/OOPcbzaN9l9qRWqveVtihVJ9AkvUCgvxm2EhIRXT0n4ECWOKz3+SmJ
# w7wXsFSFQrP8DJ6LGYnn8AtqgcKBGUIZUnWKNsIdw2FzLixre24/LAl4FOmRsqlb
# 30mjdAy87JGA0j3mSj5mO0+7hvoyGtmW9I/2kQH2zsZ0/fZMcm8Qq3UwxTSwethQ
# /gpY3UA8x1RtnWN0SCyxTkctwRQEcb9k+SS+c23Kjgm9swFXSVRk2XPXfx5bRAGO
# WhmRaw2fpCjcZxkoJLo4S5pu+yFUa2pFEUep8beuyOiJXk+d0tBMdrVXVAmxaQFE
# fnyhYWxz/gq77EFmPWn9y8FBSX5+k77L+DvktxW/tM4+pTFRhLy/AsGConsXHRWJ
# jXD+57XQKBqJC4822rpM+Zv/Cuk0+CQ1ZyvgDbjmjJnW4SLq8CdCPSWU5nR0W2rR
# nj7tfqAxM328y+l7vzhwRNGQ8cirOoo6CGJ/2XBjU02N7oJtpQUQwXEGahC0HVUz
# WLOhcGbyoYIDVTCCAj0CAQEwggEBoYHZpIHWMIHTMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJlbGFuZCBP
# cGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjo0MDFB
# LTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZaIjCgEBMAcGBSsOAwIaAxUAhGNHD/a7Q0bQLWVG9JuGxgLRXseggYMwgYCkfjB8
# MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
# bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1N
# aWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQsFAAIFAOs7
# gywwIhgPMjAyNTAxMjIxNDQ2MDRaGA8yMDI1MDEyMzE0NDYwNFowczA5BgorBgEE
# AYRZCgQBMSswKTAKAgUA6zuDLAIBADAGAgEAAgE+MAcCAQACAhNUMAoCBQDrPNSs
# AgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkKAwKgCjAIAgEAAgMHoSCh
# CjAIAgEAAgMBhqAwDQYJKoZIhvcNAQELBQADggEBAFuhbZsBdf9JwoJf8AIjsosU
# onlg17t6o7J8NIl3NewR3kj6ND2qpteQT105BVEaM7rl1c6GaVNSQsBa6Kmb+0q/
# i6YDSI52xM1kDlE4RHj5G1p5bWGj6t6XtXtM8JCJoLewr34gULB9CF+IdXLyAuUd
# jr1wvJSkl868JgEowScQO6xkzbzpd7qIat8z44OKrNRR3Wfhpb1cidG7eRp+AvfN
# CZmp5hnWbe8hPZ3t3AOSp+HlTn0Idw2eJrOK3KzeWfK2GLnP6AwvFYk63bcUUoJs
# zu23FYMpgPF2YWyalB3if3k2nm3MhP4Qq79wZY0CamFKQAfGUUqF+llymiXLql0x
# ggQNMIIECQIBATCBkzB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAA
# Af7QqMJ7NCELAQABAAAB/jANBglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkD
# MQ0GCyqGSIb3DQEJEAEEMC8GCSqGSIb3DQEJBDEiBCBJFyoP/I7DFLl0upsyi713
# nE9SgKbCCQcx2lo2pDOzNzCB+gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EIBGF
# zN38U8ifGNH3abaE9apz68Y4bX78jRa2QKy3KHR5MIGYMIGApH4wfDELMAkGA1UE
# BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAc
# BgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0
# IFRpbWUtU3RhbXAgUENBIDIwMTACEzMAAAH+0KjCezQhCwEAAQAAAf4wIgQgsa23
# SFrGD8GAu/YmJrKQbpMt/N9u+xDySqqNAr5X6SQwDQYJKoZIhvcNAQELBQAEggIA
# RG398llaPIolTLgJWGLkKTkr5MSVIsqZuxgbIrA3OP8DupjGvcK2VsONDMVFbHXa
# ulNFQRH3KHjRI8OJwx87/8fw2XOzyDY3qRNqn0XPcAGYTyEtVOa9PM++pVVZn3KZ
# 73J+dPJZubu4WuzcjIKMJ2r3qN1+7gu7qlQP5wstd+41h9JgZbzkU5AsbF9TRk7p
# iNJHH3WqUizFJ/SQSHLpoJy6hcnjk6waSuRTFf6VMaRxSHWxdKbleO4d3GveV416
# RjZ8JvlQy44R9vn3WOQR1tmB4jJOb2Ug/8IiXj9TbuzHWEYsuJ4rLQwNs8zZgegq
# vVdwcmWiwOiM6hNrtZcgAdtJjZJ1F70clgEd1oENL0aQs2W9gn3MfTyUFqVLKqxN
# XtqX6rcK+wS8C3YzFBhSJZ/FX7E2Ipsaqbn691C9aUQ4BbRjJZS+JmY8dyMGYju6
# XLcM+z5kkxxo52aTeRQpHZI+7k1Smwrt04BmBdvd1ws8njGQyr25JiyArk7yJ2oi
# XfsNDxOrco1qxr+f6vLh9IE+hlUgPmxNNcQS/hjkPsuloyNzNAiys33eo61Jm0NF
# b04+4HfmfmS68COxRXykH2m4yjjGxGhNlxku/UZ3OEDb8AZ69Bo/5JyXwvAqA/ES
# 5Kq2Mt4jNVhQHA36CLuo8ix4ToIitSJPbccltOwQHvk=
# SIG # End signature block
