﻿function Add-Cert($pfx, $credentialXml)
{
	Write-Output "Start processing certificate '$pfx.Name'." 
    $certInstance = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2

	# Get the password for certificate.
	$pathNode = $credentialXml.Credentials.Certificates.Certificate | where { $_.Path -eq "Deployment\Onebox\"+$pfx.Name}
	if($pathNode.Count -gt 0)
	{
		$password = $pathNode[0].Password
	}
	else
	{
		$password = $pathNode.Password
	}
 
	if($password -ne $null -and [string]::IsNullOrEmpty($password) -eq $false )
	{
		$certInstance.import($pfx.FullName, $password, "Exportable,PersistKeySet,MachineKeySet"); 
		$certTP = $certInstance.Thumbprint;
        Add-CertToStore -certTP:$certTP -certInstance:$certInstance -pfx:$pfx
	}
}

function Add-CertToStore([string]$certTP,$certInstance,$pfx)
{
    $store = Get-Item Cert:\LocalMachine\My
    $store.Open("ReadWrite")

    $trustedStore = Get-Item Cert:\LocalMachine\Root
    $trustedStore.Open("ReadWrite")
    
    # add certificate to trusted root store if not present
	$trustedRootCert = Get-ChildItem "Cert:\LocalMachine\Root" | where {$_.Thumbprint -eq "$certTP"}
	if($trustedRootCert -eq $null -or [string]::IsNullOrEmpty($trustedRootCert) -eq $true)
	{
		Write-output "Adding certificate $pfx.Name to the machine trusted root store."
		$trustedStore.Add($certInstance)
	}

    # add certificate to local machine store if not present
	$localStoreCert = Get-ChildItem "Cert:\LocalMachine\My" |where {$_.Thumbprint -eq "$certTP"}
	if ($localStoreCert -eq $null -or $localStoreCert -eq "") 
	{
		Write-output "Adding certificate $pfx.Name to the local machine store."
		$store.Add($certInstance)

        # Grant permissin to Network Service to the private key.
		$sslCertPrivKey = $certInstance.PrivateKey.CspKeyContainerInfo.UniqueKeyContainerName
		$privateKeyCertFile = Get-Item -path "$ENV:ProgramData\Microsoft\Crypto\RSA\MachineKeys\*"  | where {$_.Name -eq $sslCertPrivKey} 
		$privateKeyAcl = (Get-Item -Path $privateKeyCertFile.FullName).GetAccessControl("Access") 
		$permission = "NT AUTHORITY\NETWORK SERVICE","Read","Allow" 
		$accessRule = new-object System.Security.AccessControl.FileSystemAccessRule $permission 
		$privateKeyAcl.AddAccessRule($accessRule) 
		Set-Acl $privateKeyCertFile.FullName $privateKeyAcl
		Write-output "Network Service account is granted access to the private key of certificate $pfx.Name"

		# Configure the SSL certificate "star1.cloud.onebox.dynamics.com.pfx" to the port 443
		try
		{
			if($pfx.Name -eq "star1.cloud.onebox.dynamics.com.pfx")
			{
				get-item cert:\LocalMachine\My\$certTP | new-item IIS:\SslBindings\0.0.0.0!443 -ErrorAction SilentlyContinue
			}
		}
		catch 
		{
			Write-output "SSL is configured for the AOS website."
			$_.Exception.Message    
		}
    }

    $store.Close()
    $trustedStore.Close()
}

function Remove-CertFromStore($certInstance)
{
    $store = Get-Item Cert:\LocalMachine\My
    $store.Open("ReadWrite")

    $trustedStore = Get-Item Cert:\LocalMachine\Root
    $trustedStore.Open("ReadWrite")

    # remove certificate from the trusted root store if present
	$trustedRootCert = Get-ChildItem "Cert:\LocalMachine\Root" | where {$_.Thumbprint -eq "$certTP"}
	if($trustedRootCert -ne $null -or [string]::IsNullOrEmpty($trustedRootCert) -eq $false)
	{
		Write-output "Removing certificate $pfx.Name from the machine trusted root store."
		$trustedStore.Remove($certInstance)
	}

    # remove certificate from the local machine store if present
	$localStoreCert = Get-ChildItem "Cert:\LocalMachine\My" |where {$_.Thumbprint -eq "$certTP"}
	if ($localStoreCert -ne $null -or [string]::IsNullOrEmpty($localStoreCert) -eq $false) 
	{
		Write-output "Removing certificate $pfx.Name from the local machine store."
		$store.Remove($certInstance)
    }

    $store.Close()
    $trustedStore.Close()
}

function Remove-Cert($pfx, $credentialXml)
{
    $certInstance = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2

    # Get the password for certificate.
    $pathNode = $credentialXml.Credentials.Certificates.Certificate | where { $_.Path -eq "Deployment\Onebox\"+$pfx.Name}
    
    if($pathNode.Count -gt 0)
    {
        $password = $pathNode[0].Password
    }
    else
    {
        $password = $pathNode.Password
    }
 
    if($password -ne $null -and [string]::IsNullOrEmpty($password) -eq $false )
    {
       $certInstance.import($pfx.FullName, $password, "Exportable,PersistKeySet,MachineKeySet"); 
       $certTP = $certInstance.Thumbprint;
       
       Remove-CertFromStore -certInstance:$certInstance
    }
}

function AddOrRemoveWebsiteCerts([string] $rootPath, [bool] $remove)
{
	# Add all certificates to store.
	$credentialXml = [xml](Get-Content "$rootPath\DObind\Deployment\Onebox\Credentials.xml")

	# Get all the available certificate files.
	$allPFX = Get-ChildItem "$rootPath\DObind\Deployment\Onebox\*.pfx"

	# Loop through all each certificate and install it if not installed.
	foreach($pfx in $allPFX)
	{
		if($remove)
		{
			Remove-Cert $pfx $credentialXml
		}
		else
		{
			Add-Cert $pfx $credentialXml
		}
	}
}

function Add-WebsiteCerts([string] $rootPath)
{
	AddOrRemoveWebsiteCerts $rootPath $false
}

function Remove-WebsiteCerts([string] $rootPath)
{
	AddOrRemoveWebsiteCerts $rootPath $true
}

function Remove-ExistingAppPools([string] $appPoolName)
{
	$machineName = $env:COMPUTERNAME;
    $serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)
    $existingAppPool = $serverManager.ApplicationPools | Where-Object{$_.Name -eq "$appPoolName"}
    if($existingAppPool -ne $null)
    {
        Write-Output "Removing app pool '$appPoolName'."
        $serverManager.ApplicationPools.Remove($existingAppPool)
		$serverManager.CommitChanges()
    }
}

function Remove-ExistingWebsites([string] $siteName, $port, $hostHeaderStrs)
{
	$machineName = $env:COMPUTERNAME;
	$serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)
    $existingSite = $serverManager.Sites | Where-Object{$_.Name -eq "$siteName"}
    if($existingSite -ne $null)
    {
        Write-Output "Removing website '$siteName'."
        $serverManager.Sites.Remove($existingSite)
		$serverManager.CommitChanges()
    }
 
    #remove anything already using that port
    [Microsoft.Web.Administration.Site]$site
    foreach($site in $serverManager.Sites) 
    {
        foreach($hostHeader in $hostHeaderStrs)
		{
            foreach($binding in $site.Bindings)
            {
                if($binding.EndPoint.Port -eq $port -and $binding.EndPoint.Address -eq $hostHeader)
                {
                    Write-Warning "Warning: Found an existing site '$($site.Name)' already using port $port; removing it."
				    $serverManager.Sites.Remove($site)
					$serverManager.CommitChanges()
				    Write-Output "Website $($site.Name) removed."
                }
            }
        }
    }
}

function New-AppPool([string] $appPoolName, [string] $user, [string] $password, [int] $pingTimeoutSeconds)
{
	$machineName = $env:COMPUTERNAME;
	$serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)

    Write-output "Creating an app pool named '$appPoolName' under v4.0 runtime, default (Integrated) pipeline."
    $pool = $serverManager.ApplicationPools.Add($appPoolName);
    $pool.managedRuntimeVersion = "v4.0"
    $pool.processModel.identityType = [Microsoft.Web.Administration.ProcessModelIdentityType]::NetworkService
	$pool.processModel.PingResponseTime = [System.TimeSpan]::FromSeconds($pingTimeoutSeconds)
	$pool.Failure.RapidFailProtection =$false
	
	if ($user -ne $null -AND $password -ne $null -AND $user -ne "" -AND $password -ne "") {
	    Write-Output "Setting AppPool to run as user '$user'."
		$pool.processmodel.identityType = [Microsoft.Web.Administration.ProcessModelIdentityType]::SpecificUser
		$pool.processmodel.username = $user
		$pool.processmodel.password = $password
	} 
	
    $serverManager.CommitChanges()

	for($tryCount= 30; $tryCount -gt 0; --$tryCount)
	{
		if ($serverManager.ApplicationPools[$appPoolName].State -eq [Microsoft.Web.Administration.ObjectState]::Started)
		{
			Write-Output "The AppPool was created and started successfully."
			return
		}

		Start-Sleep -Milliseconds 100
	}

	throw "App pool '$appPoolName' was created but did not start automatically."
}

function New-Website([string] $siteName, [string] $aosSitePath, [string] $port, $hostHeaderStrs,$cert, [string] $appPoolName)
{
	$machineName = $env:COMPUTERNAME
	$serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)
	$bindingArguments = "*:443:"+$hostHeaderStrs[0]
	Write-Output "Creating website. Site name: '$siteName'; Bindings: '$bindingArguments'; Website root: '$aosSitePath'."
	$website = $serverManager.Sites.Add($siteName, "https", $bindingArguments, $aosSitePath)
	for($i = 1; $i -lt $hostHeaderStrs.Length ; $i++)
	{
		$bindingArguments = "*:443:"+$hostHeaderStrs[$i]
		Write-Output "Adding binding '$bindingArguments' to the AOS website."
        $website.Bindings.Add($bindingArguments, $cert.GetCertHash(), "My")
	}

	Write-Output "Setting the website app pool to '$appPoolName'."
	# Set the app pool for the website
	$website.Applications[0].ApplicationPoolName = $appPoolName
    $serverManager.CommitChanges()

    
    Set-WebConfigurationProperty -pspath "MACHINE/WEBROOT/APPHOST/$siteName/Apps"  -filter "system.webServer/security/requestFiltering/fileExtensions/add[@fileExtension='.config']" -name "allowed" -value "True"


	for($tryCount= 100; $tryCount -gt 0; --$tryCount)
	{
		if ($serverManager.Sites[$siteName].State -eq [Microsoft.Web.Administration.ObjectState]::Started)
		{
			Write-Output "AOS website was created and started successfully."
			return
		}

		Start-Sleep -Milliseconds 100
	}
	
	throw "The AOS website '$siteName' was created but did not start after waiting for 10 seconds. Check the event log for further details."
}

function New-AosWebsite([string] $rootPath, [string] $siteName, [string] $aosSitePath, [string] $appPoolName, [string] $port, $hostHeaderStrs, [string] $user, [string] $userPassword, [string] $certTP, [int] $pingTimeoutSeconds = 600)
{
	# start IIS.
	Invoke-Command -script {iisreset.exe /start}

    # Cleanup old websites and app pools
	Remove-ExistingWebsites $siteName $appPoolName $port
	Remove-ExistingAppPools $appPoolName

	# Add certs
	Add-WebsiteCerts $rootPath

	# Create new app Pool
	New-AppPool $appPoolName $user $userPassword $pingTimeoutSeconds
	
	# Create the new site
	$cert = Get-ChildItem "Cert:\LocalMachine\My" |where {$_.Thumbprint -eq "$certTP"}
	New-Website $siteName $aosSitePath $port $hostHeaderStrs $cert $appPoolName
}

function Remove-Website([string] $rootPath, [string] $websiteName, [string] $appPoolName)
{
	# Cleanup certs
	Write-Output "Removing certificates."
	Remove-WebsiteCerts $rootPath

	# Cleanup old websites and app pools
	Write-Output "Remvoing website '$websiteName'."
	Remove-ExistingWebsites $websiteName $appPoolName 0

	Write-Output "Remvoing App pool '$appPoolName'."
	Remove-ExistingAppPools $appPoolName
}

function Delete-Service([string]$serviceName)
{
    try
    {
        $service = Get-WmiObject -Class Win32_Service -Filter "Name='$serviceName'" -ErrorAction SilentlyContinue
	    if($service -ne $null)
	    {
		    # stop and delete the batch Service
            Write-output "Stopping service '$serviceName'."
		    Stop-Service "$serviceName" -ErrorAction SilentlyContinue -Force
            $service = Get-Service $serviceName
            while($service -ne $null -and $service.Status -ne "Stopped")
            {
                $service = Get-Service $serviceName
                Write-output "Sleeping for 5 minutes while waiting for the service '$serviceName' to stop."
                Start-Sleep 5
            }

            Write-output "Deleting service '$serviceName'."
		    sc.exe delete "$serviceName" 
	    }
    }
    catch
    {
    }
}

function Terminate-process
{
    # kill process
	$processes = @("batch","batch.exe","mmc")
	foreach($process in $processes)
	{
		$handle = get-process "$process" -ErrorAction SilentlyContinue
		if($handle -ne $null)
		{
			try
			{
                # special case for Event viewer
                if($process -eq "mmc" -and $handle.MainWindowTitle -eq "Event Viewer")
                {
                    $procName = $handle.MainWindowTitle
                    Write-output "Terminating process '$procName'."
				    $handle.kill()
                }
                else
                {
                    Write-output "Terminating process '$handle'."
                    $handle.Kill()
                }
			}
			catch
			{
			}
		}
	}

    # stop and delete Dynamics Rainier services
    Delete-Service -serviceName:"DynamicsAxBatch"
}

function Register-RestartTask([string]$deploymentDir,[string]$logDir,[bool]$deployChocolateyPackage,[string]$packageDir)
{
    Write-Output "Registering a scheduled task to auto start Azure storage emulator upon machine restart."
    $localMachineName = $env:COMPUTERNAME
    $batchFilename = [IO.Path]::Combine("$deploymentDir","AOSDeploy.cmd")
    Write-Output "Creating scheduler task batch file '$batchFilename'."

    "$deploymentDir\DoBind\aosdeploy.exe /action:Restart /rd:`"$deploymentDir`" /ld:`"$logDir`" /td:`"$packageDir`" /chocopkg:$deployChocolateyPackage"|Set-Content $batchFilename

    $taskName = "DynamicsRainierRestartTask"
    & schtasks /create /ru SYSTEM /sc ONSTART /tn $taskName /tr "cmd /c $batchFilename" /F /RL HIGHEST
    if (!$?)
    {
        throw "There was an error creating the task $taskName."
    }
}

function Delete-RestartTask
{
    $taskName = "DynamicsRainierRestartTask"
    Write-Output "Deleting scheduled task '$taskName'."
    $SchTsk = Get-ScheduledTask | Select Name, State | ? {$_.Name -eq $taskName}
    if($SchTsk -ne $null)
    {
        & schtasks /delete /tn $taskName /F
    }
}

function Configure-ProductConfiguration([string]$sitename)
{
    Write-output "Starting the configuration of the production configuration application."
    $aossite=Get-Website -Name $sitename
    if($aossite -eq $null){
        $message="The website '$sitename' does not exist."
        Write-Output $message
        throw $message
    }

    $webroot=$aossite.physicalPath
    $productconfigurationdir=join-path $webRoot "productconfiguration"

    if(!(Test-Path "$productconfigurationdir")){
        Write-Output "Skipping product configuration as the product configuration directory does not exist under the aos web root $webroot."
        return
    }

    $pcapppool="productconfiguration"
    Write-output "Removing web app pool $pcapppool."
    Remove-WebAppPool -Name:$pcapppool -ErrorAction SilentlyContinue
    write-output "Creating web app pool $pcapppool."
    $newapppool=New-WebAppPool -Name:$pcapppool -ErrorAction SilentlyContinue
    write-output "Setting the identity for the web app pool '$pcapppool' to NetworkService."
    $newapppool.processModel.identityType="NetworkService"
    $newapppool|Set-Item
    write-output "Starting web app pool $pcapppool."
    Start-WebAppPool -Name:$pcapppool

    Write-Output "Deleting web application 'ProductConfiguration' under web site '$sitename'."
    Remove-WebApplication -Site:$aossite.Name -Name:"ProductConfiguration" -ErrorAction SilentlyContinue
    Write-Output "Creating new web application 'ProductConfiguration' under web site '$sitename' with application pool `"$pcapppool`"."
    New-WebApplication -Site:$aossite.Name -Name:"ProductConfiguration" -PhysicalPath:$productconfigurationdir -ApplicationPool:$pcapppool -Force 
    Write-Output "Product configuration application successfully configured."
}

Export-ModuleMember -Function New-AosWebsite,Remove-Website,Terminate-process,Register-RestartTask,Delete-RestartTask,Delete-Service,Configure-ProductConfiguration,Add-WebsiteCerts,Remove-WebsiteCerts
# SIG # Begin signature block
# MIIoVQYJKoZIhvcNAQcCoIIoRjCCKEICAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCD4vYul+/4MT2Y/
# shNR1m8EcMqg8PK/RZCg1SpB/rUJ2qCCDYUwggYDMIID66ADAgECAhMzAAAEhJji
# EuB4ozFdAAAAAASEMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjUwNjE5MTgyMTM1WhcNMjYwNjE3MTgyMTM1WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDtekqMKDnzfsyc1T1QpHfFtr+rkir8ldzLPKmMXbRDouVXAsvBfd6E82tPj4Yz
# aSluGDQoX3NpMKooKeVFjjNRq37yyT/h1QTLMB8dpmsZ/70UM+U/sYxvt1PWWxLj
# MNIXqzB8PjG6i7H2YFgk4YOhfGSekvnzW13dLAtfjD0wiwREPvCNlilRz7XoFde5
# KO01eFiWeteh48qUOqUaAkIznC4XB3sFd1LWUmupXHK05QfJSmnei9qZJBYTt8Zh
# ArGDh7nQn+Y1jOA3oBiCUJ4n1CMaWdDhrgdMuu026oWAbfC3prqkUn8LWp28H+2S
# LetNG5KQZZwvy3Zcn7+PQGl5AgMBAAGjggGCMIIBfjAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUBN/0b6Fh6nMdE4FAxYG9kWCpbYUw
# VAYDVR0RBE0wS6RJMEcxLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9wZXJh
# dGlvbnMgTGltaXRlZDEWMBQGA1UEBRMNMjMwMDEyKzUwNTM2MjAfBgNVHSMEGDAW
# gBRIbmTlUAXTgqoXNzcitW2oynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8v
# d3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9NaWNDb2RTaWdQQ0EyMDExXzIw
# MTEtMDctMDguY3JsMGEGCCsGAQUFBwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDov
# L3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNDb2RTaWdQQ0EyMDEx
# XzIwMTEtMDctMDguY3J0MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQELBQADggIB
# AGLQps1XU4RTcoDIDLP6QG3NnRE3p/WSMp61Cs8Z+JUv3xJWGtBzYmCINmHVFv6i
# 8pYF/e79FNK6P1oKjduxqHSicBdg8Mj0k8kDFA/0eU26bPBRQUIaiWrhsDOrXWdL
# m7Zmu516oQoUWcINs4jBfjDEVV4bmgQYfe+4/MUJwQJ9h6mfE+kcCP4HlP4ChIQB
# UHoSymakcTBvZw+Qst7sbdt5KnQKkSEN01CzPG1awClCI6zLKf/vKIwnqHw/+Wvc
# Ar7gwKlWNmLwTNi807r9rWsXQep1Q8YMkIuGmZ0a1qCd3GuOkSRznz2/0ojeZVYh
# ZyohCQi1Bs+xfRkv/fy0HfV3mNyO22dFUvHzBZgqE5FbGjmUnrSr1x8lCrK+s4A+
# bOGp2IejOphWoZEPGOco/HEznZ5Lk6w6W+E2Jy3PHoFE0Y8TtkSE4/80Y2lBJhLj
# 27d8ueJ8IdQhSpL/WzTjjnuYH7Dx5o9pWdIGSaFNYuSqOYxrVW7N4AEQVRDZeqDc
# fqPG3O6r5SNsxXbd71DCIQURtUKss53ON+vrlV0rjiKBIdwvMNLQ9zK0jy77owDy
# XXoYkQxakN2uFIBO1UNAvCYXjs4rw3SRmBX9qiZ5ENxcn/pLMkiyb68QdwHUXz+1
# fI6ea3/jjpNPz6Dlc/RMcXIWeMMkhup/XEbwu73U+uz/MIIHejCCBWKgAwIBAgIK
# YQ6Q0gAAAAAAAzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNV
# BAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
# c29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlm
# aWNhdGUgQXV0aG9yaXR5IDIwMTEwHhcNMTEwNzA4MjA1OTA5WhcNMjYwNzA4MjEw
# OTA5WjB+MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
# BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYD
# VQQDEx9NaWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG
# 9w0BAQEFAAOCAg8AMIICCgKCAgEAq/D6chAcLq3YbqqCEE00uvK2WCGfQhsqa+la
# UKq4BjgaBEm6f8MMHt03a8YS2AvwOMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc
# 6Whe0t+bU7IKLMOv2akrrnoJr9eWWcpgGgXpZnboMlImEi/nqwhQz7NEt13YxC4D
# dato88tt8zpcoRb0RrrgOGSsbmQ1eKagYw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+
# lD3v++MrWhAfTVYoonpy4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+EGvKhL1nk
# kDstrjNYxbc+/jLTswM9sbKvkjh+0p2ALPVOVpEhNSXDOW5kf1O6nA+tGSOEy/S6
# A4aN91/w0FK/jJSHvMAhdCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rhKEmd
# X4jiJV3TIUs+UsS1Vz8kA/DRelsv1SPjcF0PUUZ3s/gA4bysAoJf28AVs70b1FVL
# 5zmhD+kjSbwYuER8ReTBw3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zd
# sGbiwZeBe+3W7UvnSSmnEyimp31ngOaKYnhfsi+E11ecXL93KCjx7W3DKI8sj0A3
# T8HhhUSJxAlMxdSlQy90lfdu+HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS
# 4NaIjAsCAwEAAaOCAe0wggHpMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBRI
# bmTlUAXTgqoXNzcitW2oynUClTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTAL
# BgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBRyLToCMZBD
# uRQFTuHqp8cx0SOJNDBaBgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFf
# MDNfMjIuY3JsMF4GCCsGAQUFBwEBBFIwUDBOBggrBgEFBQcwAoZCaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFf
# MDNfMjIuY3J0MIGfBgNVHSAEgZcwgZQwgZEGCSsGAQQBgjcuAzCBgzA/BggrBgEF
# BQcCARYzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1h
# cnljcHMuaHRtMEAGCCsGAQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAHAAbwBsAGkA
# YwB5AF8AcwB0AGEAdABlAG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn
# 8oalmOBUeRou09h0ZyKbC5YR4WOSmUKWfdJ5DJDBZV8uLD74w3LRbYP+vj/oCso7
# v0epo/Np22O/IjWll11lhJB9i0ZQVdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0b
# pdS1HXeUOeLpZMlEPXh6I/MTfaaQdION9MsmAkYqwooQu6SpBQyb7Wj6aC6VoCo/
# KmtYSWMfCWluWpiW5IP0wI/zRive/DvQvTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvy
# CInWH8MyGOLwxS3OW560STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn+N4sOiBp
# mLJZiWhub6e3dMNABQamASooPoI/E01mC8CzTfXhj38cbxV9Rad25UAqZaPDXVJi
# hsMdYzaXht/a8/jyFqGaJ+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw3MYb
# BL7fQccOKO7eZS/sl/ahXJbYANahRr1Z85elCUtIEJmAH9AAKcWxm6U/RXceNcbS
# oqKfenoi+kiVH6v7RyOA9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sL
# gOppO6/8MO0ETI7f33VtY5E90Z1WTk+/gFcioXgRMiF670EKsT/7qMykXcGhiJtX
# cVZOSEXAQsmbdlsKgEhr/Xmfwb1tbWrJUnMTDXpQzTGCGiYwghoiAgEBMIGVMH4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01p
# Y3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTECEzMAAASEmOIS4HijMV0AAAAA
# BIQwDQYJYIZIAWUDBAIBBQCgga4wGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQw
# HAYKKwYBBAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEINtB
# 1m9PHGgEpV2EU8rkJQfhGL66oYCwLnP9gtUKVx9hMEIGCisGAQQBgjcCAQwxNDAy
# oBSAEgBNAGkAYwByAG8AcwBvAGYAdKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20wDQYJKoZIhvcNAQEBBQAEggEA6BPJRBclRG+7s52eRS51elbp6d+yLHbBEz8z
# UBkDjVsBEltZUBKUpEf9eMKL9C/TOnsCG2bnAk/ypVKUVSeTJkbkxPYOV7EpXWBp
# j9ciqCLIud5Ed/Hujhd2k4UKQt494MxD/uVgI0YhG+G36Xpi6JJ/LQIt6tZOqxTr
# NsOmVw0PeqsSl8sRtYzMBs0mtpCSCmFpLqfiQn3thhnWC2mgOD5P6lUux6nW1TH4
# gJIcL/meZltn23phdoKSQHlEDUV5qwQ05YOmkIEVVs71h3wmufiNH5WOrD3LYRuh
# MesFHCnj4cLtD7lOQtU7Mgh9nX5wqLnLONdwRsPdGbiOTKv+aKGCF7AwghesBgor
# BgEEAYI3AwMBMYIXnDCCF5gGCSqGSIb3DQEHAqCCF4kwgheFAgEDMQ8wDQYJYIZI
# AWUDBAIBBQAwggFaBgsqhkiG9w0BCRABBKCCAUkEggFFMIIBQQIBAQYKKwYBBAGE
# WQoDATAxMA0GCWCGSAFlAwQCAQUABCCOLC3YQygnq5zY2npbQWtBjIMTzi2Uikb+
# x6iW1/M/IgIGaKSAyVgnGBMyMDI1MDkwNDA3MjU0Ny4wNjlaMASAAgH0oIHZpIHW
# MIHTMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQL
# EyRNaWNyb3NvZnQgSXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsT
# Hm5TaGllbGQgVFNTIEVTTjo1OTFBLTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9z
# b2Z0IFRpbWUtU3RhbXAgU2VydmljZaCCEf4wggcoMIIFEKADAgECAhMzAAAB9BdG
# hcDLPznlAAEAAAH0MA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1w
# IFBDQSAyMDEwMB4XDTI0MDcyNTE4MzA1OVoXDTI1MTAyMjE4MzA1OVowgdMxCzAJ
# BgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1pY3Jv
# c29mdCBJcmVsYW5kIE9wZXJhdGlvbnMgTGltaXRlZDEnMCUGA1UECxMeblNoaWVs
# ZCBUU1MgRVNOOjU5MUEtMDVFMC1EOTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGlt
# ZS1TdGFtcCBTZXJ2aWNlMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA
# pwhOE6bQgC9qq4jJGX2A1yoObfk0qetQ8kkj+5m37WBxDlsZ5oJnjfzHspqPiOEV
# zZ2y2ygGgNZ3/xdZQN7f9A1Wp1Adh5qHXZZh3SBX8ABuc69Tb3cJ5KCZcXDsufwm
# XeCj81EzJEIZquVdV8STlQueB/b1MIYt5RKis3uwzdlfSl0ckHbGzoO91YTKg6IE
# xqKYojGreCopnIKxOvkr5VZsj2f95Bb1LGEvuhBIm/C7JysvJvBZWNtrspzyXVnu
# o+kDEyZwpkphsR8Zvdi+s/pQiofmdbW1UqzWlqXQVgoYXbaYkEyaSh/heBtwj1tu
# e+LcuOcHAPgbwZvQLksKaK46oktregOR4e0icsGiAWR9IL+ny4mlCUNA84F7GEEW
# OEvibig7wsrTa6ZbzuMsyTi2Az4qPV3QRkFgxSbp4R4OEKnin8Jz4XLI1wXhBhIp
# MGfA3BT850nqamzSiD5L5px+VtfCi0MJTS2LDF1PaVZwlyVZIVjVHK8oh2HYG9T2
# 6FjR9/I85i5ExxmhHpxM2Z+UhJeZA6Lz452m/+xrA4xrdYas5cm7FUhy24rPLVH+
# Fy+ZywHAp9c9oWTrtjfIKqLIvYtgJc41Q8WxbZPR7B1uft8BFsvz2dOSLkxPDLcX
# Wy16ANy73v0ipCxAwUEC9hssi0LdB8ThiNf/4A+RZ8sCAwEAAaOCAUkwggFFMB0G
# A1UdDgQWBBQrdGWhCtEsPid1LJzsTaLTKQbfmzAfBgNVHSMEGDAWgBSfpxVdAF5i
# XYP05dJlpxtTNRnpcjBfBgNVHR8EWDBWMFSgUqBQhk5odHRwOi8vd3d3Lm1pY3Jv
# c29mdC5jb20vcGtpb3BzL2NybC9NaWNyb3NvZnQlMjBUaW1lLVN0YW1wJTIwUENB
# JTIwMjAxMCgxKS5jcmwwbAYIKwYBBQUHAQEEYDBeMFwGCCsGAQUFBzAChlBodHRw
# Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFRp
# bWUtU3RhbXAlMjBQQ0ElMjAyMDEwKDEpLmNydDAMBgNVHRMBAf8EAjAAMBYGA1Ud
# JQEB/wQMMAoGCCsGAQUFBwMIMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQsF
# AAOCAgEA3cHSDxJKUDsgacIfRX60ugODShsBqwtEURUbUXeDmYYSa5oFj34RujW3
# gOeCt/ObDO45vfpnYG5OS5YowwsFw19giCI6JV+ccG/qqM08nxASbzwWtqtorzQi
# Jh9upsE4TVZeKYXmbyx7WN9tdbVIrCelVj7P6ifMHTSLt6BmyoS2xlC2cfgKPPA1
# 3vS3euqUl6zwe7GAhjfjNXjKlE4SNWJvdqgrv0GURKjqmamNvhmSJane6TYzpdDC
# egq8adlGH85I1EWKmfERb1lzKy5OMO2e9IkAlvydpUun0C3sNEtp0ehliT0Sraq8
# jcYVDH4A2C/MbLBIwikjwiFGQ4SlFLT2Tgb4GvvpcWVzBxwDo9IRBwpzngbyzbhh
# 95UVOrQL2rbWHrHDSE3dgdL2yuaHRgY7HYYLs5Lts30wU9Ouh8N54RUta6GFZFx5
# A4uITgyJcVdWVaN0qjs0eEjwEyNUv0cRLuHWJBejkMe3qRAhvCjnhro7DGRWaIld
# yfzZqln6FsnLQ3bl+ZvVJWTYJuL+IZLI2Si3IrIRfjccn29X2BX/vz2KcYubIjK6
# XfYvrZQN4XKbnvSqBNAwIPY2xJeB4o9PDEFI2rcPaLUyz5IV7JP3JRpgg3xsUqvF
# HlSG6uMIWjwH0GQIIwrC2zRy+lNZsOKnruyyHMQTP7jy5U92qEEwggdxMIIFWaAD
# AgECAhMzAAAAFcXna54Cm0mZAAAAAAAVMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
# MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3Nv
# ZnQgUm9vdCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0yMTA5MzAxODIy
# MjVaFw0zMDA5MzAxODMyMjVaMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA5OGmTOe0ciELeaLL1yR5
# vQ7VgtP97pwHB9KpbE51yMo1V/YBf2xK4OK9uT4XYDP/XE/HZveVU3Fa4n5KWv64
# NmeFRiMMtY0Tz3cywBAY6GB9alKDRLemjkZrBxTzxXb1hlDcwUTIcVxRMTegCjhu
# je3XD9gmU3w5YQJ6xKr9cmmvHaus9ja+NSZk2pg7uhp7M62AW36MEBydUv626GIl
# 3GoPz130/o5Tz9bshVZN7928jaTjkY+yOSxRnOlwaQ3KNi1wjjHINSi947SHJMPg
# yY9+tVSP3PoFVZhtaDuaRr3tpK56KTesy+uDRedGbsoy1cCGMFxPLOJiss254o2I
# 5JasAUq7vnGpF1tnYN74kpEeHT39IM9zfUGaRnXNxF803RKJ1v2lIH1+/NmeRd+2
# ci/bfV+AutuqfjbsNkz2K26oElHovwUDo9Fzpk03dJQcNIIP8BDyt0cY7afomXw/
# TNuvXsLz1dhzPUNOwTM5TI4CvEJoLhDqhFFG4tG9ahhaYQFzymeiXtcodgLiMxhy
# 16cg8ML6EgrXY28MyTZki1ugpoMhXV8wdJGUlNi5UPkLiWHzNgY1GIRH29wb0f2y
# 1BzFa/ZcUlFdEtsluq9QBXpsxREdcu+N+VLEhReTwDwV2xo3xwgVGD94q0W29R6H
# XtqPnhZyacaue7e3PmriLq0CAwEAAaOCAd0wggHZMBIGCSsGAQQBgjcVAQQFAgMB
# AAEwIwYJKwYBBAGCNxUCBBYEFCqnUv5kxJq+gpE8RjUpzxD/LwTuMB0GA1UdDgQW
# BBSfpxVdAF5iXYP05dJlpxtTNRnpcjBcBgNVHSAEVTBTMFEGDCsGAQQBgjdMg30B
# ATBBMD8GCCsGAQUFBwIBFjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3Bz
# L0RvY3MvUmVwb3NpdG9yeS5odG0wEwYDVR0lBAwwCgYIKwYBBQUHAwgwGQYJKwYB
# BAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMB
# Af8wHwYDVR0jBBgwFoAU1fZWy4/oolxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBL
# oEmgR4ZFaHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMv
# TWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggr
# BgEFBQcwAoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNS
# b29DZXJBdXRfMjAxMC0wNi0yMy5jcnQwDQYJKoZIhvcNAQELBQADggIBAJ1Vffwq
# reEsH2cBMSRb4Z5yS/ypb+pcFLY+TkdkeLEGk5c9MTO1OdfCcTY/2mRsfNB1OW27
# DzHkwo/7bNGhlBgi7ulmZzpTTd2YurYeeNg2LpypglYAA7AFvonoaeC6Ce5732pv
# vinLbtg/SHUB2RjebYIM9W0jVOR4U3UkV7ndn/OOPcbzaN9l9qRWqveVtihVJ9Ak
# vUCgvxm2EhIRXT0n4ECWOKz3+SmJw7wXsFSFQrP8DJ6LGYnn8AtqgcKBGUIZUnWK
# NsIdw2FzLixre24/LAl4FOmRsqlb30mjdAy87JGA0j3mSj5mO0+7hvoyGtmW9I/2
# kQH2zsZ0/fZMcm8Qq3UwxTSwethQ/gpY3UA8x1RtnWN0SCyxTkctwRQEcb9k+SS+
# c23Kjgm9swFXSVRk2XPXfx5bRAGOWhmRaw2fpCjcZxkoJLo4S5pu+yFUa2pFEUep
# 8beuyOiJXk+d0tBMdrVXVAmxaQFEfnyhYWxz/gq77EFmPWn9y8FBSX5+k77L+Dvk
# txW/tM4+pTFRhLy/AsGConsXHRWJjXD+57XQKBqJC4822rpM+Zv/Cuk0+CQ1Zyvg
# DbjmjJnW4SLq8CdCPSWU5nR0W2rRnj7tfqAxM328y+l7vzhwRNGQ8cirOoo6CGJ/
# 2XBjU02N7oJtpQUQwXEGahC0HVUzWLOhcGbyoYIDWTCCAkECAQEwggEBoYHZpIHW
# MIHTMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQL
# EyRNaWNyb3NvZnQgSXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsT
# Hm5TaGllbGQgVFNTIEVTTjo1OTFBLTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9z
# b2Z0IFRpbWUtU3RhbXAgU2VydmljZaIjCgEBMAcGBSsOAwIaAxUAv+LZ/Vg0s17X
# ek4iG9R9c/7+AI6ggYMwgYCkfjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2Fz
# aGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
# cnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAx
# MDANBgkqhkiG9w0BAQsFAAIFAOxjbREwIhgPMjAyNTA5MDQwMTQyNDFaGA8yMDI1
# MDkwNTAxNDI0MVowdzA9BgorBgEEAYRZCgQBMS8wLTAKAgUA7GNtEQIBADAKAgEA
# AgICOwIB/zAHAgEAAgITTTAKAgUA7GS+kQIBADA2BgorBgEEAYRZCgQCMSgwJjAM
# BgorBgEEAYRZCgMCoAowCAIBAAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEB
# CwUAA4IBAQBo5JSXgTaf3klzLey6ZTLSSE644mXIQMaFDt7/fWX65rhGAFlkGiXS
# iLGTEI4gnI5mV1mmkj7B3KyUa6z/3uSKwEutmdk6lwcOxdwlsnhK/kUQbIf5pLW2
# sKGflu5buE5NxOipKr6ZFx5G115fZuLm24c3Y8CiisOe4JQ3k1HxmJR6CifLiaUh
# tIJnMZVPzci7Q3YmZWm5S5C1h+ZsDVHB0qKx7EI+Bv6QpYyVg6QXMUltKJTvrrvo
# K5tIXQ5NajEca5wqxSUJUjwJgyqh3xbvF8v0gT0YbSbWlk6rXYzHOZA0OWIpqGR+
# oKUwF7PTd1/+RJZb+F8wPvICJ7nkBd1/MYIEDTCCBAkCAQEwgZMwfDELMAkGA1UE
# BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAc
# BgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0
# IFRpbWUtU3RhbXAgUENBIDIwMTACEzMAAAH0F0aFwMs/OeUAAQAAAfQwDQYJYIZI
# AWUDBAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG
# 9w0BCQQxIgQgKPt73wJfLlWCuPaVkYK50DOIuzGh8eG/ZDbNF/7wyxEwgfoGCyqG
# SIb3DQEJEAIvMYHqMIHnMIHkMIG9BCA/WMJ8biaT6njvkknB8Q7hSQIi8ys6vIBv
# Zg60RBjWazCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5n
# dG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9y
# YXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMz
# AAAB9BdGhcDLPznlAAEAAAH0MCIEIFV7hcIN6fKj8bXNN4TZDZCe9BwXQ6Dp6S4d
# MAkKWZrwMA0GCSqGSIb3DQEBCwUABIICABCfAODvIjqzjaJ/ktPCmB7OxM48ld3j
# P79ErQpttD2HSSZ9IEBQFU87obmH0m8hi9K7Ir06gnANHgzPr1EqN3f58rYhevM8
# 7vaiQXcyXaYHFf/3z+PHcfNWtXbLlrqtTJeIGVt2b6nFCZT1LVkXPXF481YwyGGe
# BAGCnxM2ZYrKe1RHDXEfWY0vuGnXpwb9zoPTa7EWVKUf+++qrifk7yx+73DJRPo0
# UWkB2b99R1PdyPZjHo5tpAATBKG0C/+rWR8oa0e6YTHE0oJNOXtxqyPpvuZlflrS
# 9t37Q8BU+bipr+ebPwWPNVcUA8uhvWU/CVq4erg0ypdqDbFgH6MY4ITMSifyPjsO
# UX81FsxWlNxT2m6Th8zEQa++5G8eKwRbuA1Pdv+Y0eCjT970xvgFPs5ahcVWjZ/T
# T14ybC6/DHaNCxjC9qHhN1rXX/hl1DXgcIO07YPP6YXtkPVjEhtxG3zmAPeJOk+g
# 8r59qm2+hD0UHmjfHSGd7XDHCmbKxVSd8HDkYysnLfox94TxKCpzg0xIND2WoYKq
# Yn6v3C72SluzRuu2859mbzGYnxykQimdhEuOPdIyILc7tcw6rk5wRQPgr67Kiwt6
# /40oJFvsXhnNaGzqTlLmmkhFlzsHEGJU4wEMJCoV43ll+cQf5KsV+KpuJAX8Wlzf
# legNyqfDUaXv
# SIG # End signature block
