﻿[CmdletBinding()]
param
(
    [Parameter(Mandatory=$true)]
    [string]$ServiceModelXml,

    [Parameter(Mandatory=$true)]
    [string]$log,

    [Parameter(Mandatory=$false)]
    [string]$config
)

Import-Module WebAdministration

#region configuration generators
function Write-Storage-Emulator-Configuration([string]$logDir)
{
    $registrykey="HKLM:\SOFTWARE\Wow6432Node\Microsoft\Windows Azure Storage Emulator\"
    $registrykeyvaluename="InstallPath"
    $emulatorInstallPath=Get-RegistryKeyValue -registryKey:$registrykey -registryKeyValueName:$registrykeyvaluename
    if([string]::IsNullOrEmpty($emulatorInstallPath)){
        Write-Log "Skipping Azure sdk configuration as the Azure emulator is not installed."
        break
    }

    $azurestorageemulator= join-path $emulatorInstallPath "azurestorageemulator.exe"

    $standardout=join-path $global:logdir "azurestorageemulator.stopprocess.output"
    $standarderr=join-path $global:logdir "azurestorageemulator.stopprocess.error"
    $script="`
    WindowsProcess StopEmulatorProcesses `
    { `
        Ensure=`"Absent`" `
        Arguments=`"start /inprocess`" `
        Path=`"$azurestorageemulator`" `
        DependsOn=@() `
    }"

    Write-Log "Creating the configuration to stop any storage emulator process(es)"
    Write-Content $script

    $script = @"
    DynamicsScheduledTask StartAzureStorageEmulator_ScheduledTask
    {
        TaskName = "DynamicsStartAzureStorageEmulator"
        Ensure = "Absent"
    }
"@
    Write-Content $script

}

function Write-Miscellaneous-Processes-Configuration([string]$webroot,[string]$logDir)
{
    $binarypathname=join-path $webroot "bin\Microsoft.Dynamics.AX.Deployment.Setup.exe"
    $standardoutdir=join-path $logDir "Microsoft.Dynamics.AX.Deployment.Setup.exe.output"
    $standarderrdir=join-path $logDir "Microsoft.Dynamics.AX.Deployment.Setup.exe.error"

    $script="`
    DynamicsProcess StopDeploymentSetupProcess `
    { `
        Ensure=`"Absent`" `
        Arguments=`"`" `
        File=`"$binarypathname`" `
        StandardErrorPath=`"$standardoutdir`" `
        StandardOutputPath=`"$standarderrdir`" `
        DependsOn=@(`"[WindowsProcess]StopEmulatorProcesses`") `
        MonitoringAssembly=`"$global:telemetrydll`" `
    } "

    Write-Log "Creating the configuration to stop the deployment setup process"
    Write-Content $script
}

function Write-BatchService-Configuration([string]$webroot,[string]$logDir)
{
    $webconfig= join-path $webroot "web.config"
    $binarypathname=join-path $webroot "bin\Batch.exe"
    $standardout=join-path $logDir "batch.exe.output"
    $standarderr=join-path $logDir "batch.exe.error"
    $deleteBatchServiceDependency=""
    if(Test-Path $binarypathname){
        $script="`
        WindowsProcess StopBatchProcess `
        { `
            Ensure=`"Absent`" `
            Arguments='-service $webconfig' `
            Path=`"$binarypathname`" `
        } "

        Write-Log "Creating the configuration to stop the Dynamics AX Batch process"
        Write-Content $script
        $deleteBatchServiceDependency="`"[WindowsProcess]StopBatchProcess`""
    }
    
    $servicepath=join-path $webroot "bin\Batch.exe -service $webconfig"
    $script = "`
    Service DeleteBatchService `
    { `
        Ensure=`"Absent`" `
        Name=`"DynamicsAxBatch`" `
        State=`"Stopped`" `
        DisplayName=`"Dynamics AX Batch Management`" `
        Path='$servicepath' `
        DependsOn=@($deleteBatchServiceDependency) `
    }"

    Write-Log "Creating the configuration to delete the Dynamics AX Batch service"
    Write-Content $script

    $script = @"
    DynamicsScheduledTask StartBatch_ScheduledTask
    {
        TaskName = "DynamicsStartBatch"
        Ensure = "Absent"
    }
"@
    Write-Content $script
}

function Write-WebSite-Configuration([string]$websitename,[string]$webroot,[string]$apppoolname)
{
    $source=$env:TEMP
    $script="`
    xWebSite DeleteAosWebSite `
    { `
        Ensure=`"Absent`" `
        Name=`"$websiteName`" `
        PhysicalPath=`"$webRoot`" `
        State=`"Stopped`" `
    }"

    Write-Log "Creating the configuration to delete the AOS web site"
    Write-Content $script

    if(![string]::IsNullOrEmpty($apppoolname)){
         # delete the aos apppool if it exists
        $script="`
        xWebAppPool DeleteAosAppPool `
        { `
            Ensure=`"Absent`" `
            State=`"Stopped`" `
            Name=`"$apppoolname`" `
            DependsOn=@(`"[xWebSite]DeleteAosWebSite`") `
        }"

        Write-Log "Creating the configuration to delete the AOS web app pool"
        Write-Content $script
    }

    $script= @"
    xWebAppPool DeleteProductConfigurationAppPool
    {
        Ensure="Absent"
        State="Stopped"
        Name="ProductConfiguration"
        DependsOn=@("[xWebSite]DeleteAosWebSite")
    }
"@
    Write-Log "Creating the configuration to delete the product configuration web app pool"
    Write-Content $script

    $script = "
    File DeleteWebSitePhysicalPath`
    { `
        Ensure=`"Absent`" `
        DestinationPath=`"$webRoot`" `
        Recurse=`$true `
        SourcePath=`"`" `
        Type=`"Directory`" `
        DependsOn=@(`"[xWebSite]DeleteAosWebSite`", `"[Service]DeleteBatchService`") `
        Force=`$true `
    }"

     Write-Log "Creating the configuration to delete the AOS web site files"
     Write-Content $script
    
}

function Write-Packages-Configuration([string]$packagedir)
{
    $script = "
    File DeletePackages`
    { `
        Ensure=`"Absent`" `
        DestinationPath=`"$packageDir`" `
        Recurse=`$true `
        SourcePath=`"`" `
        Type=`"Directory`" `
        Force=`$true `
        DependsOn=@(`"[xWebSite]DeleteAosWebSite`") `
    }"

    Write-Log "Creating the configuration to delete the Dynamics packages"
    Write-Content $script
}

function Write-Database-Configuration([string]$dbName,[string]$dbServer,[string]$dbUser,[string]$dbPwd,[string]$logDir)
{
    # for dropping the db, use the temp folder name as the package dir as it is not used
    $packageDir=$env:TEMP

    $script="`
    DynamicsDatabase DropDatabase `
    { `
        Ensure=`"Absent`" `
        PackageDirectory=`"$packageDir`" `
        DatabaseName=`"$dbName`" `
        DatabaseServer=`"$dbServer`" `
        UserName=`"$dbUser`" `
        Password=`"$dbPwd`" `
        DependsOn=@(`"[File]DeletePackages`") `
        MonitoringAssembly=`"$global:telemetrydll`" `
    }"

    Write-Log "Creating the configuration to drop the AX database"
    Write-Content $script
}

function Write-LCM-Configuration($thumbprint)
{
    Write-Verbose "Creating the LCM configuration to set the encryption certificate thumbprint..."
    $script="`
    LocalConfigurationManager 
    {
        CertificateId=`"$thumbprint`"
    }"
    
    write-content $script
}
#endregion

#region helper functions
function Initialize-Log([string]$logdir,[string]$logfile)
{
    if(-not (Test-Path $logdir)){
        New-Item -ItemType Directory -Path $logdir
    }
    
    if(Test-Path $logfile){
        Remove-Item $logfile -Force
    }

    New-item $logfile -ItemType File -Force
}

function Write-Log([string]$message)
{
    Add-Content $global:logfile $message
    Write-Host $message
}

function Write-Header
{
    $datetime=Get-Date
    $header="`
###
# ==++==
#
# Copyright (c) Microsoft Corporation. All rights reserved.
#
# Generated date: $datetime
#
# This file is auto-generated by a tool. Any changes made to this file will be lost.
#
# This file will be compiled to generate a Windows DSC configuration that will remove 
# the Dynamics AOS service on this machine.
###

#region service configuration
Configuration RemoveAosServiceConfiguration { `
`
Import-DscResource -Module xWebAdministration `
Import-DscResource -Module xDynamics `
Import-DscResource -Module PSDesiredStateConfiguration `

Node `"localhost`"{"

    Write-Content $header
}

function Write-Footer([string]$outputpath)
{
    $footer="}`
    } `

`$configData=@{
    AllNodes=@(
        @{NodeName=`"localhost`";CertificateFile=`"$global:encryptioncertpublickeyfile`";Thumbprint=`"$global:encryptioncertthumbprint`";PSDscAllowDomainUser=`$true; }
    )
}`

#endregion

# generate the MOF file `    
RemoveAosServiceConfiguration -OutputPath:$outputpath -ConfigurationData:`$configData"

    Write-Content $footer
}

function Write-Content([string]$content)
{
   Add-Content $global:dscconfigfile -Value $content
}

function Get-RegistryKeyValue([string]$registryKey,[string]$registryKeyValueName,[string]$defaultValue=[string]::Empty)
{
    $item=Get-ItemProperty "$registryKey" -ErrorAction SilentlyContinue
    $value=$item.$registryKeyValueName
    if([string]::IsNullOrEmpty($value)){
        return $defaultValue
    }

    return $value
}

function Copy-CustomDSCResources([string]$grandparentDir)
{
    if(Test-Path "$grandparentDir\Scripts\xDynamics"){
        Write-Log "Copying the custom DSC resources"
        Copy-Item -Path "$grandparentDir\Scripts\xDynamics" -Destination "$env:ProgramFiles\WindowsPowerShell\Modules" -Force -Recurse -Verbose
        Get-ChildItem -Path "$env:ProgramFiles\WindowsPowerShell\Modules" -Recurse | Unblock-File -Verbose  
    }else{
        Write-Log "No custom DSC resources to copy"
    }
}

function Save-EncryptionCertificate-PublicKey($certificate)
{
    Write-Log "Saving the encryption cert public key to file..."
    $global:encryptioncertpublickeyfile=join-path $global:logdir "EncryptCert.cer"
    Export-Certificate -cert:$certificate -FilePath $global:encryptioncertpublickeyfile -Force -Type CERT | Out-Null
}

function Get-EncryptionCertificate-Thumbprint
{
    $subject="MicrosoftDynamicsAXDSCEncryptionCert"

    #get or create a new self-signed encryption certificate to secure sensitive info in the MOF files
    $cert=Get-Make-Encryption-Cert -subject:$subject
    Save-EncryptionCertificate-PublicKey -certificate:$cert
    $cert.Thumbprint
}

function Get-Make-Encryption-Cert([string]$subject)
{
    Write-Log "Checking if a self-signed encryption cert with subject '$subject' exists..."
    $formattersubject="CN=$subject"
    $cert=Get-ChildItem Cert:\LocalMachine\My|where {$_.Subject -eq $formattersubject}
    if($cert -ne $null) # if cert existed make sure it is valid
    {
        if(!(Is-ValidCert -certificate:$cert))
        {
            Write-Log "Dynamics DSC self-signed encryption cert is expired. Generating a new self-signed encryption certificate..."
            Write-Log "Deleting the invalid self-signed encryption certificate with subject '$cert.subject'... "
            $thumbprint=$cert.Thumbprint
            Remove-Item -Path Cert:\LocalMachine\My\$thumbprint -Force -DeleteKey |out-null
            $cert=Make-Certificate -subject:$subject
        }
    }
    else
    {
        $cert=Make-Certificate -subject:$subject 
    }

    $cert
}

function Is-ValidCert($certificate)
{
    $subject=$certificate.Subject
    Write-Log "Checking if the certificate '$subject' is valid..."
    $thumbprint=$certificate.Thumbprint
    $cert=Get-ChildItem -Path Cert:\LocalMachine\My\$thumbprint
    if($cert -ne $null)
    {
        if($cert.NotAfter -lt (Get-Date)) #expired
        {
            return $false
        }
        else
        {
            return $true
        }
    }

    #if cert is not found, return false
    return $false
}

function Make-Certificate([string]$subject)
{
     Write-Log "Creating a new DSC self-signed encryption certificate with subject '$subject'..."
     return New-SelfSignedCertificate -DnsName $subject -CertStoreLocation cert:\LocalMachine\My
}

function Output-CurrentEnvironmentState([string]$websitePath, [string]$packagePath)
{
    $handleExePath=join-path $env:SystemDrive "SysInternals\Handle.exe"
    $openHandlesLog=join-path $global:logDir "aosservice-uninstallation-openhandles.log"
    if(-not (Test-Path $handleExePath))
    {
        return
    }
    
    #dump any handles to files in the website directory
    if(Test-Path $websitePath)
    {
        Write-Log "AOS WebRoot still exists at $websitePath. Dumping open handles to $openHandlesLog..."
        & "$handleExePath" -AcceptEula $websitePath | Out-File $openHandlesLog -Append
    }
    
    #dump any handles to files in the package directory
    if(Test-Path $packagePath)
    {
        Write-Log "AOS packages directory still exists at $packagePath. Dumping open handles to $openHandlesLog..."
        & "$handleExePath" -AcceptEula $packagePath | Out-File $openHandlesLog -Append
    }
}

#endregion

#region Main
$parentdir=Split-Path -parent $PSCommandPath
$grandparentdir=Split-Path -parent $parentdir
$global:logfile=$log
$global:logdir=[System.IO.Path]::GetDirectoryName($log)

Initialize-Log -logdir:$global:logdir -logfile:$log
Copy-CustomDSCResources -grandparentDir:$grandparentdir

$global:decodedservicemodelxml=[System.Text.Encoding]::UTF8.GetString([System.Convert]::FromBase64String($servicemodelxml))

$global:dscconfigfile=join-path $global:logdir "Uninstall.ps1"

if(Test-Path $global:dscconfigfile){
    Remove-Item $global:dscconfigfile -Force
}

$outputpath=join-path $global:logdir "Uninstall"
$etwpath=join-path $grandparentdir "ETWManifest"
$global:telemetrydll = join-path $etwpath "Microsoft.Dynamics.AX7Deployment.Instrumentation.dll"

if(-not (Test-Path $global:telemetrydll)){
    throw "The deployment telemetry assembly does not exist"
}

[System.Xml.XmlDocument] $xd = new-object System.Xml.XmlDocument
$xd.LoadXml($global:decodedservicemodelxml)
$ns=New-Object System.Xml.XmlNamespaceManager($xd.NameTable)
$ns.AddNamespace("ns",$xd.DocumentElement.NamespaceURI)

if($env:USERDOMAIN -eq $env:COMPUTERNAME){
    $global:domain="builtin"
}else{
    $global:domain=$env:USERDOMAIN
}

$global:username=$env:USERNAME

$global:encryptioncertthumbprint=Get-EncryptionCertificate-Thumbprint

[string]$websiteName=$xd.SelectSingleNode("//ns:ServiceModel",$ns).getAttribute("Name")
[string]$webRoot=$xd.SelectSingleNode("//ns:Setting[@Name='Infrastructure.WebRoot']",$ns).getAttribute("Value")
Write-Log "The web root is $webRoot"
[string]$packageDir=$xd.SelectSingleNode("//ns:WorkingFolder[@Name='PackagesLocalDirectory']",$ns).getAttribute("Directory")
Write-Log "The package directory is $packageDir"

$website=Get-Website $websiteName
if($website -ne $null)
{
    $apppool=$website.applicationPool
}

$uninstallAttempt=1
while($uninstallAttempt -le 3 -and !$success)
{
    if($uninstallAttempt -gt 1)
    {
        $retryAttempt=$uninstallAttempt-1
        Write-Log "Retry attempt $retryAttempt`: Retrying AOS service uninstallation..."
    }

    # create the configuration file
    Write-Header 
    Write-LCM-Configuration -thumbprint:$global:encryptioncertthumbprint
    Write-Storage-Emulator-Configuration -logDir:$log
    Write-Miscellaneous-Processes-Configuration -webroot:$webroot -logDir:$log
    Write-BatchService-Configuration -webroot:$webroot -logDir:$log
    Write-WebSite-Configuration -websitename:$websiteName -webroot:$webroot -apppoolname:$apppool
    Write-Packages-Configuration -packagedir:$packageDir

    Write-Footer -outputpath:$outputpath
    #endregion

    #region execute the configuration
    Write-Log "Executing the configuration.."
    & $global:dscconfigfile
    [bool]$success=$false
    $dscConfigApplied = $false
    try
    {
        $dscConfigApplied = $false
        Set-Location $outputpath
        Write-Log ("PSModulePath is currently: "+$env:PSModulePath)

        Write-Log "Setting up LCM to decrypt credentials..."
        Set-DscLocalConfigurationManager -path "$outputpath" -Verbose *>&1 | Tee-Object $log
        
        try
        {
            Write-Log("Dumping available DSC resources before applying the configuration...")
            $availableDSCResourceLog=join-path $global:logdir "aosservice-uninstallation-availabledscresources.log"
            Get-DSCResource -Name * | Format-List | Out-File -FilePath $availableDSCResourceLog
        }
        catch
        {
            Write-Log "Failed to get DSC resources, Error: $_"
        }
        
        Write-Log "Applying the configuration..."
        Start-DscConfiguration -wait -Verbose -path "$outputpath" -Force *>&1 | Tee-Object $log
        $dscConfigApplied = $true
    }
    catch
    {
        Write-Log "Uninstall attempt $uninstallAttempt`: Error: $_"
    }

    $configstatuslog=join-path $global:logdir "aosservice-uninstallation-status.log"
    $ConfigStatus = Get-DscConfigurationStatus
    $ConfigStatus | Format-List -Property * | Out-File -FilePath $configstatuslog -Force
    if($ConfigStatus.Status -ieq 'Success' -and $dscConfigApplied -eq $true)
    {
        $success=$true
    }
    else
    {
        Output-CurrentEnvironmentState $webRoot $packageDir
        Move-Item $global:dscconfigfile (join-path $global:logdir "Uninstall_Attempt_$uninstallAttempt.ps1")
        $uninstallAttempt++
    }
}

if($success)
{
    Write-Log "Configuration applied."
    return 0
}
else
{
    throw "AOS uninstallation did not complete after 3 retries, Message: $($ConfigJob.StatusMessage), see log for details."    
}

#endregion

# SIG # Begin signature block
# MIIoLQYJKoZIhvcNAQcCoIIoHjCCKBoCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCKnacf4dkJNSkT
# IWgihuC4CLuaeOl0h48NdxQmBhF2r6CCDXYwggX0MIID3KADAgECAhMzAAAEhV6Z
# 7A5ZL83XAAAAAASFMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjUwNjE5MTgyMTM3WhcNMjYwNjE3MTgyMTM3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDASkh1cpvuUqfbqxele7LCSHEamVNBfFE4uY1FkGsAdUF/vnjpE1dnAD9vMOqy
# 5ZO49ILhP4jiP/P2Pn9ao+5TDtKmcQ+pZdzbG7t43yRXJC3nXvTGQroodPi9USQi
# 9rI+0gwuXRKBII7L+k3kMkKLmFrsWUjzgXVCLYa6ZH7BCALAcJWZTwWPoiT4HpqQ
# hJcYLB7pfetAVCeBEVZD8itKQ6QA5/LQR+9X6dlSj4Vxta4JnpxvgSrkjXCz+tlJ
# 67ABZ551lw23RWU1uyfgCfEFhBfiyPR2WSjskPl9ap6qrf8fNQ1sGYun2p4JdXxe
# UAKf1hVa/3TQXjvPTiRXCnJPAgMBAAGjggFzMIIBbzAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUuCZyGiCuLYE0aU7j5TFqY05kko0w
# RQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEW
# MBQGA1UEBRMNMjMwMDEyKzUwNTM1OTAfBgNVHSMEGDAWgBRIbmTlUAXTgqoXNzci
# tW2oynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpb3BzL2NybC9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3JsMGEG
# CCsGAQUFBwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDovL3d3dy5taWNyb3NvZnQu
# Y29tL3BraW9wcy9jZXJ0cy9NaWNDb2RTaWdQQ0EyMDExXzIwMTEtMDctMDguY3J0
# MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQELBQADggIBACjmqAp2Ci4sTHZci+qk
# tEAKsFk5HNVGKyWR2rFGXsd7cggZ04H5U4SV0fAL6fOE9dLvt4I7HBHLhpGdE5Uj
# Ly4NxLTG2bDAkeAVmxmd2uKWVGKym1aarDxXfv3GCN4mRX+Pn4c+py3S/6Kkt5eS
# DAIIsrzKw3Kh2SW1hCwXX/k1v4b+NH1Fjl+i/xPJspXCFuZB4aC5FLT5fgbRKqns
# WeAdn8DsrYQhT3QXLt6Nv3/dMzv7G/Cdpbdcoul8FYl+t3dmXM+SIClC3l2ae0wO
# lNrQ42yQEycuPU5OoqLT85jsZ7+4CaScfFINlO7l7Y7r/xauqHbSPQ1r3oIC+e71
# 5s2G3ClZa3y99aYx2lnXYe1srcrIx8NAXTViiypXVn9ZGmEkfNcfDiqGQwkml5z9
# nm3pWiBZ69adaBBbAFEjyJG4y0a76bel/4sDCVvaZzLM3TFbxVO9BQrjZRtbJZbk
# C3XArpLqZSfx53SuYdddxPX8pvcqFuEu8wcUeD05t9xNbJ4TtdAECJlEi0vvBxlm
# M5tzFXy2qZeqPMXHSQYqPgZ9jvScZ6NwznFD0+33kbzyhOSz/WuGbAu4cHZG8gKn
# lQVT4uA2Diex9DMs2WHiokNknYlLoUeWXW1QrJLpqO82TLyKTbBM/oZHAdIc0kzo
# STro9b3+vjn2809D0+SOOCVZMIIHejCCBWKgAwIBAgIKYQ6Q0gAAAAAAAzANBgkq
# hkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
# EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
# IDIwMTEwHhcNMTEwNzA4MjA1OTA5WhcNMjYwNzA4MjEwOTA5WjB+MQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQg
# Q29kZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
# CgKCAgEAq/D6chAcLq3YbqqCEE00uvK2WCGfQhsqa+laUKq4BjgaBEm6f8MMHt03
# a8YS2AvwOMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc6Whe0t+bU7IKLMOv2akr
# rnoJr9eWWcpgGgXpZnboMlImEi/nqwhQz7NEt13YxC4Ddato88tt8zpcoRb0Rrrg
# OGSsbmQ1eKagYw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+lD3v++MrWhAfTVYoonpy
# 4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+EGvKhL1nkkDstrjNYxbc+/jLTswM9
# sbKvkjh+0p2ALPVOVpEhNSXDOW5kf1O6nA+tGSOEy/S6A4aN91/w0FK/jJSHvMAh
# dCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rhKEmdX4jiJV3TIUs+UsS1Vz8k
# A/DRelsv1SPjcF0PUUZ3s/gA4bysAoJf28AVs70b1FVL5zmhD+kjSbwYuER8ReTB
# w3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zdsGbiwZeBe+3W7UvnSSmn
# Eyimp31ngOaKYnhfsi+E11ecXL93KCjx7W3DKI8sj0A3T8HhhUSJxAlMxdSlQy90
# lfdu+HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS4NaIjAsCAwEAAaOCAe0w
# ggHpMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBRIbmTlUAXTgqoXNzcitW2o
# ynUClTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYD
# VR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBRyLToCMZBDuRQFTuHqp8cx0SOJNDBa
# BgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
# bC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3JsMF4GCCsG
# AQUFBwEBBFIwUDBOBggrBgEFBQcwAoZCaHR0cDovL3d3dy5taWNyb3NvZnQuY29t
# L3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNfMjIuY3J0MIGfBgNV
# HSAEgZcwgZQwgZEGCSsGAQQBgjcuAzCBgzA/BggrBgEFBQcCARYzaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1hcnljcHMuaHRtMEAGCCsG
# AQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAHAAbwBsAGkAYwB5AF8AcwB0AGEAdABl
# AG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn8oalmOBUeRou09h0ZyKb
# C5YR4WOSmUKWfdJ5DJDBZV8uLD74w3LRbYP+vj/oCso7v0epo/Np22O/IjWll11l
# hJB9i0ZQVdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0bpdS1HXeUOeLpZMlEPXh6
# I/MTfaaQdION9MsmAkYqwooQu6SpBQyb7Wj6aC6VoCo/KmtYSWMfCWluWpiW5IP0
# wI/zRive/DvQvTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvyCInWH8MyGOLwxS3OW560
# STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn+N4sOiBpmLJZiWhub6e3dMNABQam
# ASooPoI/E01mC8CzTfXhj38cbxV9Rad25UAqZaPDXVJihsMdYzaXht/a8/jyFqGa
# J+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw3MYbBL7fQccOKO7eZS/sl/ah
# XJbYANahRr1Z85elCUtIEJmAH9AAKcWxm6U/RXceNcbSoqKfenoi+kiVH6v7RyOA
# 9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sLgOppO6/8MO0ETI7f33Vt
# Y5E90Z1WTk+/gFcioXgRMiF670EKsT/7qMykXcGhiJtXcVZOSEXAQsmbdlsKgEhr
# /Xmfwb1tbWrJUnMTDXpQzTGCGg0wghoJAgEBMIGVMH4xCzAJBgNVBAYTAlVTMRMw
# EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVN
# aWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNp
# Z25pbmcgUENBIDIwMTECEzMAAASFXpnsDlkvzdcAAAAABIUwDQYJYIZIAWUDBAIB
# BQCgga4wGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIB72C3Uv0px2JPPjIKxbk72k
# eme4jKxDG2gDog35LyG4MEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8A
# cwBvAGYAdKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEB
# BQAEggEAdU+TLajWXb2vZkve5B4NmXsnjqSn2YYqwnU7DtLG8ae9UpzEXYX3YO6B
# yU/wcERuPPoHyoJX3pBMIR2OJhXI3Kgbt5gjiaULm03SZN/1VjGRF4HgRAecsxGE
# 2vv07Qln5p/tgc566q+6+W7US7d3PHpZZ7Cbt7s0bCZryx403mxOr1OvylpyuRYO
# HQmAkGsPjggjjA0RBY2NexTXg6kWRMGdSiAOIafEqIyyMJVHfw+gb+n5py7S+492
# 3S1M2ToNFkFe1OUZkPE6+4GoMqUvBSaDYTEESXnJXFMTJRQpRot2SGd9g3/mxjoD
# e57izTnEEpQXGQne2SHEUYcYUVzT8KGCF5cwgheTBgorBgEEAYI3AwMBMYIXgzCC
# F38GCSqGSIb3DQEHAqCCF3AwghdsAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFSBgsq
# hkiG9w0BCRABBKCCAUEEggE9MIIBOQIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFl
# AwQCAQUABCCKQ9LTgnme7tYQaqi0m9PzXipPRV3Y5UBF2BtdtJiX+gIGaKOaVy8A
# GBMyMDI1MDkwNDA3MjYyOS4zOTJaMASAAgH0oIHRpIHOMIHLMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1l
# cmljYSBPcGVyYXRpb25zMScwJQYDVQQLEx5uU2hpZWxkIFRTUyBFU046OEQwMC0w
# NUUwLUQ5NDcxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2Wg
# ghHtMIIHIDCCBQigAwIBAgITMwAAAg0Nd757No9/4wABAAACDTANBgkqhkiG9w0B
# AQsFADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
# BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYD
# VQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDAeFw0yNTAxMzAxOTQz
# MDFaFw0yNjA0MjIxOTQzMDFaMIHLMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2Fz
# aGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
# cnBvcmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1lcmljYSBPcGVyYXRpb25z
# MScwJQYDVQQLEx5uU2hpZWxkIFRTUyBFU046OEQwMC0wNUUwLUQ5NDcxJTAjBgNV
# BAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2UwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCxf6gH3fH3wgmFN5TV8zRF/N0TJguWUYCrQZPUmwA+
# QhSySNp7kiGmFZd4b5zsAfN0Wh+PzIJvYsVMgVCaZcbVr/DJBfexwnQfc+fgIjOi
# AzYSjg7EtSOdWoLk81b/mGiGIBC++fLcSAzbZO3KtW4PRKOSsdD/5eRdtNca/Ed4
# EAcUT32zAGS9Sq//4kDT92KEzRNXJj8z3NDL4oGGzCQMvA83tQG5mrnepxF0OsNf
# KKHYHMqjyOEP5pTgKfT5XMfz0sEG6ARAjlXJ79SG/joeuHh8TqC+cJMry9wB7ZLr
# dMAFy8rHN3W1+kkpw47Ko+9ize2ble+P5jMaqufK033Bu+2FXVSKphil2j0qBUWp
# n5vBtf2W+gsVqydA+eseBHfYxcDZ4+5oRoyDAg0tW9f79vOAv91P4bTzG+BZPBbD
# MzSDwmj8ASKDlVwruTeF1em7NWiedWAB+29gFH/c/NN1uTQLvwGDIOw1DcLnCD0V
# XNL7mOvifYvNWugTAHcMFLVlA1jeOH35E/IW9qcKKqrah7LyJax/6M5UHswQugGg
# LriiMNEvz3IqW+AiIJ097iYzMGzsDqbLSUztIjDEt9xfIHHUs/p3j9Bkr2bPP1v4
# z8vp/45Ck3mfFbW2F0EtjOCnGPMrJNjjGhEG9zAK1105Bg2kJ7Rn8WTWO5IbD/rD
# tQIDAQABo4IBSTCCAUUwHQYDVR0OBBYEFBWXjpDmDgNrTsISj26SjU1/YMOAMB8G
# A1UdIwQYMBaAFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMF8GA1UdHwRYMFYwVKBSoFCG
# Tmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY3Jvc29mdCUy
# MFRpbWUtU3RhbXAlMjBQQ0ElMjAyMDEwKDEpLmNybDBsBggrBgEFBQcBAQRgMF4w
# XAYIKwYBBQUHMAKGUGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY2Vy
# dHMvTWljcm9zb2Z0JTIwVGltZS1TdGFtcCUyMFBDQSUyMDIwMTAoMSkuY3J0MAwG
# A1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwDgYDVR0PAQH/BAQD
# AgeAMA0GCSqGSIb3DQEBCwUAA4ICAQAcH8gT42wVQ8GQZ+MHIXNJ+d4dQn0+vPG/
# AdFvTxk/tvOOkA2i7hnoEOEFcSbzS5EdIVCMi5Y5EiWT8hEJkztdeq5hXtUWsPY+
# 2lYSU9HdhKDfRDfjwVZ9kfCthrLRC3jw9Fah5MAGI9MHSETo9r7+cux8AUqQ3hma
# M2jmTNWvrFOLbO01B1ciNGbvE2xK+rbzBBh/uWd3k30pau6Lp0azg7rDYGMGv8jW
# otILfpKMBeCQoufMsI/gzxo4+k9dqYBkAwMk7h1hf23ycb5WngSMQV/Uxl3Sxbg+
# 64KK6GCsTSG6z7zNTgbL69PMGfwV2cnawY95Iy2cgJ6cbMORRUcYzvsuUd8oEQ87
# cW4XqqBLrheewJjROT6YyWrQ2oQ+jzGK2WJoGNnfanmNfqQnVKpi320onag95LMF
# jj8BwrflYsO9kEOiy7I5UngPBmF+RHSCv2hFSr8nK7gtuiy9SUOKP6FbQOzyMRvJ
# 3UxsmrH38477XzETb/tZLAj10TdYFfkjkFeFjlb3iMTSs/VrJSF0r0vON/oxZqKC
# Y8WZez+uQP0Try0QQ9wRp5D2FYJ8E1uIX/LvwuFkBdWf7X7qlb+pzdvPpSAcaCgB
# IWTlMn2bWgkU5uPzxRPHh/0u+FI7/eRCZGbLM2qnn3yXQvO/h9wQm8pIABRAvoda
# iV0bVmHbETCCB3EwggVZoAMCAQICEzMAAAAVxedrngKbSZkAAAAAABUwDQYJKoZI
# hvcNAQELBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAw
# DgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
# MjAwBgNVBAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eSAy
# MDEwMB4XDTIxMDkzMDE4MjIyNVoXDTMwMDkzMDE4MzIyNVowfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTAwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoIC
# AQDk4aZM57RyIQt5osvXJHm9DtWC0/3unAcH0qlsTnXIyjVX9gF/bErg4r25Phdg
# M/9cT8dm95VTcVrifkpa/rg2Z4VGIwy1jRPPdzLAEBjoYH1qUoNEt6aORmsHFPPF
# dvWGUNzBRMhxXFExN6AKOG6N7dcP2CZTfDlhAnrEqv1yaa8dq6z2Nr41JmTamDu6
# GnszrYBbfowQHJ1S/rboYiXcag/PXfT+jlPP1uyFVk3v3byNpOORj7I5LFGc6XBp
# Dco2LXCOMcg1KL3jtIckw+DJj361VI/c+gVVmG1oO5pGve2krnopN6zL64NF50Zu
# yjLVwIYwXE8s4mKyzbnijYjklqwBSru+cakXW2dg3viSkR4dPf0gz3N9QZpGdc3E
# XzTdEonW/aUgfX782Z5F37ZyL9t9X4C626p+Nuw2TPYrbqgSUei/BQOj0XOmTTd0
# lBw0gg/wEPK3Rxjtp+iZfD9M269ewvPV2HM9Q07BMzlMjgK8QmguEOqEUUbi0b1q
# GFphAXPKZ6Je1yh2AuIzGHLXpyDwwvoSCtdjbwzJNmSLW6CmgyFdXzB0kZSU2LlQ
# +QuJYfM2BjUYhEfb3BvR/bLUHMVr9lxSUV0S2yW6r1AFemzFER1y7435UsSFF5PA
# PBXbGjfHCBUYP3irRbb1Hode2o+eFnJpxq57t7c+auIurQIDAQABo4IB3TCCAdkw
# EgYJKwYBBAGCNxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQUKqdS/mTEmr6CkTxG
# NSnPEP8vBO4wHQYDVR0OBBYEFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMFwGA1UdIARV
# MFMwUQYMKwYBBAGCN0yDfQEBMEEwPwYIKwYBBQUHAgEWM2h0dHA6Ly93d3cubWlj
# cm9zb2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5Lmh0bTATBgNVHSUEDDAK
# BggrBgEFBQcDCDAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDANBgkqhkiG
# 9w0BAQsFAAOCAgEAnVV9/Cqt4SwfZwExJFvhnnJL/Klv6lwUtj5OR2R4sQaTlz0x
# M7U518JxNj/aZGx80HU5bbsPMeTCj/ts0aGUGCLu6WZnOlNN3Zi6th542DYunKmC
# VgADsAW+iehp4LoJ7nvfam++Kctu2D9IdQHZGN5tggz1bSNU5HhTdSRXud2f8449
# xvNo32X2pFaq95W2KFUn0CS9QKC/GbYSEhFdPSfgQJY4rPf5KYnDvBewVIVCs/wM
# nosZiefwC2qBwoEZQhlSdYo2wh3DYXMuLGt7bj8sCXgU6ZGyqVvfSaN0DLzskYDS
# PeZKPmY7T7uG+jIa2Zb0j/aRAfbOxnT99kxybxCrdTDFNLB62FD+CljdQDzHVG2d
# Y3RILLFORy3BFARxv2T5JL5zbcqOCb2zAVdJVGTZc9d/HltEAY5aGZFrDZ+kKNxn
# GSgkujhLmm77IVRrakURR6nxt67I6IleT53S0Ex2tVdUCbFpAUR+fKFhbHP+Crvs
# QWY9af3LwUFJfn6Tvsv4O+S3Fb+0zj6lMVGEvL8CwYKiexcdFYmNcP7ntdAoGokL
# jzbaukz5m/8K6TT4JDVnK+ANuOaMmdbhIurwJ0I9JZTmdHRbatGePu1+oDEzfbzL
# 6Xu/OHBE0ZDxyKs6ijoIYn/ZcGNTTY3ugm2lBRDBcQZqELQdVTNYs6FwZvKhggNQ
# MIICOAIBATCB+aGB0aSBzjCByzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
# bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
# b3JhdGlvbjElMCMGA1UECxMcTWljcm9zb2Z0IEFtZXJpY2EgT3BlcmF0aW9uczEn
# MCUGA1UECxMeblNoaWVsZCBUU1MgRVNOOjhEMDAtMDVFMC1EOTQ3MSUwIwYDVQQD
# ExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQB7
# LCwoj6G3nQ7Oxhl/pfne4yATPaCBgzCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1w
# IFBDQSAyMDEwMA0GCSqGSIb3DQEBCwUAAgUA7GMvNzAiGA8yMDI1MDkwMzIxMTg0
# N1oYDzIwMjUwOTA0MjExODQ3WjB3MD0GCisGAQQBhFkKBAExLzAtMAoCBQDsYy83
# AgEAMAoCAQACAghmAgH/MAcCAQACAhLQMAoCBQDsZIC3AgEAMDYGCisGAQQBhFkK
# BAIxKDAmMAwGCisGAQQBhFkKAwKgCjAIAgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJ
# KoZIhvcNAQELBQADggEBAAtWFK3uGGMKA4Amw+/u2tEulHecS/hvd9RIQ2brlD51
# OEin9JkovD72iG8rve/3oXuJPWGdlj4yZhICb0zHUnXVy/mMHcUH0yBFvmTIV0oA
# u1oPXPV1NLVsxMk8w8CxA/fhzJpzLlcMTEDZCi9c+34jG+faG97g9bN6dMa4g3B4
# Z2rFtuFLqJgRSxBxI/IwYOmifl3hwQLCKp4Zw6f79IojfhygA2Bqp0g4yOWgt+Rs
# +ZVAxG2kKBoyRngSFe1XZyuSfTg+wnW/wEjR8BraaJLsC3gaQ+b19cnFP4IPMPHy
# RyVUS2Fuc0VSPxhw0aoHjgRLUO9y+aG8L4tSTRWkv2YxggQNMIIECQIBATCBkzB8
# MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
# bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1N
# aWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAg0Nd757No9/4wABAAAC
# DTANBglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEE
# MC8GCSqGSIb3DQEJBDEiBCBbfjsRlMe0gpIV5Q22vgbfvy+fRDm1l/axAlpgJb9j
# DzCB+gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EIGPqB5TsJGqI8OuknBKtSvb0
# Ffq6w5NSs5veTVwka/hAMIGYMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgT
# Cldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29m
# dCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENB
# IDIwMTACEzMAAAINDXe+ezaPf+MAAQAAAg0wIgQg3uz0PRSdsAXaucbJoviKH4X8
# XctFoLhquHxtAwJEVjAwDQYJKoZIhvcNAQELBQAEggIAsO8M3U0vOfvDebUuoEqT
# i46FnhldTCJ2U7R/d48NUiUSb1+zQFMpxAzlcobg/lUccFwbpbka9gE8KULlU0x3
# EYY+yemMKKpCDBzoA7MN6FuQLQFnhI0Y4T7uWmTtqCTFJpTS9IellJgZl0pWb4YN
# cD8ILKBoKf8udi6Nu3tB56xg/TRAVDWd+v4JW/ui7T1Qmv9dZPvP/w4ir8TP1h38
# ArZnMj3tL6tvAFUdIpquxwVUNz/pY966yJ0EHdX4y/Te9tIOqeyJeukdRT1XBsVM
# U7grXyeMFXoYCJyWTAdlXC6xY6up7D6GLZJ+3XbbQSg/Lu8j4jcLMvPVBXYnTreD
# UBH0RK1iodu7Ku4yMA7ZtkXkT/yvvkzUn0Fdk1e9cBliOvBs2lT1BF8GyKQ5+xiG
# dW5vmC+0YpkPgg59F+NoeFd353LFTlOGcrURl29BCpNvQYpKLc3JNK6AjBRuGL9A
# Egt8ff57TEg6LCTaqIWDefSxn4/3p6w/l7bLaAfpBAYhvWz929zXaNBrUwTp0mBS
# F0dHTsGskp3hG1+roe8Q/F1kR2xgiwV6qqCwQHtnG3HCFYLQl6uX2+WFcIAgLafp
# QMH23eAh27DAyEpUuSUtaCmz70s/EIxOoHwTI9eetAPfdfFnp7KJwuy8dRWlBwHH
# fUWnFryXpz0uxXikRvId1uM=
# SIG # End signature block
