﻿#***************************************************************************************************************
# This script modifies Global.asax.cs files
# It does the following:
#   *   Adds or removes TLS 1.2 enforcment in method Application_Start
#   *   A log with a transcript of all actions taken is generated
#***************************************************************************************************************

#************************************************ SCRIPT USAGE  ************************************************
# .\Update-GlobalAsaxCsToTls12.ps1
#   -log                        :   path and name of the log file
#   -packagesLocalDirectoryPath :   path to the packages local directory
#   -EnforceTls12               :   $true/$false, default($false), use $true to explicitly enforce TLS 1.2
#***************************************************************************************************************

PARAM (
    [Parameter(Mandatory = $false, HelpMessage = "The path to the directory in which to write log files.")]
    [string]$LogDir,
    [Parameter(Mandatory = $false, HelpMessage = "Indicate that this is being run on a staging environment.")]
    [switch]$useStaging = $false,
    [Parameter(Mandatory = $false, HelpMessage = "Indicates whether or not to enforce TLS 1.2")]
    [bool]$EnforceTls12 = $true
)

# Module import
Import-Module WebAdministration
Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\AosEnvironmentUtilities.psm1" -Force -DisableNameChecking

#
# Determines if the file contains TLS 1.2 enforcement
#
function IsTls12EnforcementInFile
{
    param(
        [Parameter()]
        [ValidateNotNullOrEmpty()]
        [string]$filePath
    )

    $pattern = "^(.*?(\bSecurityProtocolType\.Tls12\b)[^$]*)$"

    if (Test-Path $filePath)
    {
        # get the file contents
        $contents = Get-Content -Path $filePath

        # loop through the lines in the file
        foreach ($line in $contents)
        {
            if ($line -match $pattern)
            {
                Write-Log "Tls 1.2 is enforced in file: $($filePath)"
                
                return $true
            }
        }
    }

    Write-Log "Tls 1.2 is NOT enforced in file: $($filePath)"

    return $false
}

#
# Gets the paths to the Global.asax.cs files that are to be inspected
#
function GetFilePathsToGlobalAsaxCs
{
    $paths = $null

    Write-Log "GetFilePathsToGlobalAsaxCs : useStaging : $($useStaging)"

    if ($useStaging)
    {
        # Work from the Staging root
        $webroot = Join-Path -Path $AosServiceStagingPath -ChildPath "webroot"
    }
    else 
    {
        # Work from webroot
        $webroot = Get-AosWebSitePhysicalPath
    }

    if (![string]::IsNullOrEmpty($webroot))
    {
        $paths = "$($webroot)\App_Code\Global.asax.cs","$($webroot)\bin\App_Code\Global.asax.cs"
    }

    Write-Log "GetFilePathsToGlobalAsaxCs returning: $($serviceDrive)"

    return $paths
}


#
# Adds or removes TLS 1.2 enforcment from the Global.asax.cs files
#
function EnforceTls12InGlobalAsaxCsFiles
{
    param(
        [Parameter()]
        [bool]$enforce
    )

    # default to false
    $isUpdated = $false

    # get the file paths
    $paths = GetFilePathsToGlobalAsaxCs

    if ($null -eq $paths)
    {
        Write-Log "Unable to determine Global.asax.cs file paths"
        return $isUpdaed
    }

    # loop through the paths
    foreach ($path in $paths)
    {
        if (Test-Path -Path $path)
        {
            $result = EnforceTls12InGlobalAsaxCs -filePath $path -enforce $enforce
            $isUpdated = $isUpdated -or $result
        }
    }
    
    return $isUpdated
}

#
# Adds or removes TLS 1.2 enforcment from the Global.asax.cs file provided
#
function EnforceTls12InGlobalAsaxCs
{
    param(
        [Parameter()]
        [ValidateNotNullOrEmpty()]
        [string]$filePath,
        [Parameter()]
        [bool]$enforce
    )

    # assume no changes
    $isUpdated = $false

    if (!(Test-Path -Path $filePath))
    {
        Write-Log "EnforceTls12InGlobalAsaxCs - File not found: $($filePath)"
        return $isUpdated
    }

    # set to false to begin with
    $doUpdateFile = $false

    # is the TLS 1.2 setting already in the file?
    $isTls12Enforced = IsTls12EnforcementInFile -filePath $filePath

    if ($enforce)
    {
        if ($isTls12Enforced)
        {
            # Nothing to do, so return
            return $isUpdated
        }

        #
        # ADD TLS 1.2 enforcment to the file
        #

        # get the file contents
        $contents = Get-Content -Path $filePath

        # search for the Application_Start method
        $pattern = "^(.*?(\bApplication_Start\b)[^$]*)$"

        $found = $false
        $lineCount = 0

        [String[]] $updatedContents = @()
        foreach ($Line in $contents)
        {
            $updatedContents += $Line

            if ($true -eq $found)
            {
                $lineCount += 1
            
                if (1 -eq $lineCount)
                {
                    # the file will need to be updated
                    $doUpdateFile = $true

                    # Add line to enforce TLS 1.2
                    $updatedContents += '            System.Net.ServicePointManager.SecurityProtocol = System.Net.SecurityProtocolType.Tls12;'

                    # reset found
                    $found = $false
                    $line = 0

                    Write-Log "Added line: System.Net.ServicePointManager.SecurityProtocol = System.Net.SecurityProtocolType.Tls12;"
                }
            }

            if ($Line -match $pattern)
            {
                $found = $true

                Write-Log "Search pattern match found"
            }
        }
    }
    else # $enforce == $false
    {
        if (!$isTls12Enforced)
        {
            # Nothing to do, so return
            return $isUpdated
        }
        
        #
        # REMOVE TLS 1.2 enforcment from the file
        #

        # get the file contents
        $contents = Get-Content -Path $filePath

        # search for SecurityProtocolType.Tls12
        $pattern = "^(.*?(\bSecurityProtocolType\.Tls12\b)[^$]*)$"

        [String[]] $updatedContents = @()
        foreach ($Line in $contents)
        {
            if ($Line -match $pattern)
            {
                $doUpdateFile = $true

                # Log that the line is found and being removed
                Write-Log "Removing line: System.Net.ServicePointManager.SecurityProtocol = System.Net.SecurityProtocolType.Tls12;"
            }
            else 
            {
                # include the line
                $updatedContents += $Line
            }
        }
    }
    
    if ($doUpdateFile)
    {
        # construct the backup file name
        $backupFile = "$($filePath).$(get-date -f yyyyMMdd-HHmmss)"

        # create a backup
        Copy-Item $filePath $backupFile -Force

        Write-Log "Created backup file: $($backupFile)"

        Try
        {
            # update the file contents
            Set-Content $filePath $updatedContents

            # set return value
            $isUpdated = $true

            Write-Log "File updated: $($filePath)"
        }
        Catch
        {
            Write-Exception $_
            Write-Log "Unable to update file: $($filePath)" 
        }      
    }

    return $isUpdated
}

#***************************** END FUNCTIONS **********************************

#***************************** BEGIN EXECUTION ********************************

# Import the AosCommon module and initialize the log
Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking

if (!$LogDir)
{
    $LogDir = $PSScriptRoot
}

$dt = [System.DateTime]::Now.ToString("yyyyMMddHHmmss")
$logFile = Join-Path -Path $LogDir -ChildPath "Update-GlobalAsaxCsToTls12.$dt.log"

Initialize-Log $logFile

try
{
	$error.Clear()

    # Start writing to the logs
    Write-Log "========== Begin execution of Update-GlobalAsaxToTls12.ps1 =========="

    # Update Global.asax.cs files if necessary
    EnforceTls12InGlobalAsaxCsFiles -enforce $EnforceTls12

    Write-Log "=============== End of logging for a script execution ==============="
}
catch
{
	# write the exception and that this process failed. Do not throw as this should not 
	# cause the deployment or update to fail.
    Write-Exception $_       
    Write-Log "Checking '$($ModulesNames -join ', ')' assemblies versions failed, see $log for details."
}
# SIG # Begin signature block
# MIIr4AYJKoZIhvcNAQcCoIIr0TCCK80CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCSwLRzm+D/m3+A
# gB/M4+wkDUGJycnNpc4COhCZJ4ggkqCCEW4wggh+MIIHZqADAgECAhM2AAACAO38
# jbec3qFIAAIAAAIAMA0GCSqGSIb3DQEBCwUAMEExEzARBgoJkiaJk/IsZAEZFgNH
# QkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxFTATBgNVBAMTDEFNRSBDUyBDQSAwMTAe
# Fw0yNDExMDgxMjQzMjhaFw0yNTExMDgxMjQzMjhaMCQxIjAgBgNVBAMTGU1pY3Jv
# c29mdCBBenVyZSBDb2RlIFNpZ24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK
# AoIBAQC5L/UPrOpwYjxcoZC0TqqvMF1WUELvwXN+k27SrA5rohJknn7Cgbxg4hGT
# XKqpcdbtsVTN3ZY896SJ20uQ+INL5OVLzpW408nCNTPYg2LtGJbqHUjpNm0hLCJ+
# gO5Jn2T8DDzIJoUijGXj1m+hRLKb2nOIicCED2GuYBmuWXnaY7INmVEaU3peryty
# ZjDuxdyGDuiPURz8lW1SUiDzoszNp1oswVr+WjDvLDUx4HlxPsG8zUjIst0NnJ6o
# z4tNFKaUBDCetcMjQxpCETn29a1CuRddxZLjPHZHfcotr5sh1S6bNQdzVaMNsxV8
# L3wjHb7XJ6ZVm662mHEiPgpyNcLhAgMBAAGjggWKMIIFhjApBgkrBgEEAYI3FQoE
# HDAaMAwGCisGAQQBgjdbAQEwCgYIKwYBBQUHAwMwPQYJKwYBBAGCNxUHBDAwLgYm
# KwYBBAGCNxUIhpDjDYTVtHiE8Ys+hZvdFs6dEoFgg93NZoaUjDICAWQCAQ4wggJ2
# BggrBgEFBQcBAQSCAmgwggJkMGIGCCsGAQUFBzAChlZodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpaW5mcmEvQ2VydHMvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDEu
# YW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUy
# MDAxKDIpLmNydDBSBggrBgEFBQcwAoZGaHR0cDovL2NybDIuYW1lLmdibC9haWEv
# QlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBS
# BggrBgEFBQcwAoZGaHR0cDovL2NybDMuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAx
# LkFNRS5HQkxfQU1FJTIwQ1MlMjBDQSUyMDAxKDIpLmNydDBSBggrBgEFBQcwAoZG
# aHR0cDovL2NybDQuYW1lLmdibC9haWEvQlkyUEtJQ1NDQTAxLkFNRS5HQkxfQU1F
# JTIwQ1MlMjBDQSUyMDAxKDIpLmNydDCBrQYIKwYBBQUHMAKGgaBsZGFwOi8vL0NO
# PUFNRSUyMENTJTIwQ0ElMjAwMSxDTj1BSUEsQ049UHVibGljJTIwS2V5JTIwU2Vy
# dmljZXMsQ049U2VydmljZXMsQ049Q29uZmlndXJhdGlvbixEQz1BTUUsREM9R0JM
# P2NBQ2VydGlmaWNhdGU/YmFzZT9vYmplY3RDbGFzcz1jZXJ0aWZpY2F0aW9uQXV0
# aG9yaXR5MB0GA1UdDgQWBBST/HE52ZUlmsYqZcZBdrXZ5u4ZnzAOBgNVHQ8BAf8E
# BAMCB4AwRQYDVR0RBD4wPKQ6MDgxHjAcBgNVBAsTFU1pY3Jvc29mdCBDb3Jwb3Jh
# dGlvbjEWMBQGA1UEBRMNMjM2MTY3KzUwMzE1NTCCAeYGA1UdHwSCAd0wggHZMIIB
# 1aCCAdGgggHNhj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpaW5mcmEvQ1JM
# L0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwxLmFtZS5nYmwv
# Y3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwyLmFtZS5n
# YmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmwzLmFt
# ZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGMWh0dHA6Ly9jcmw0
# LmFtZS5nYmwvY3JsL0FNRSUyMENTJTIwQ0ElMjAwMSgyKS5jcmyGgb1sZGFwOi8v
# L0NOPUFNRSUyMENTJTIwQ0ElMjAwMSgyKSxDTj1CWTJQS0lDU0NBMDEsQ049Q0RQ
# LENOPVB1YmxpYyUyMEtleSUyMFNlcnZpY2VzLENOPVNlcnZpY2VzLENOPUNvbmZp
# Z3VyYXRpb24sREM9QU1FLERDPUdCTD9jZXJ0aWZpY2F0ZVJldm9jYXRpb25MaXN0
# P2Jhc2U/b2JqZWN0Q2xhc3M9Y1JMRGlzdHJpYnV0aW9uUG9pbnQwHwYDVR0jBBgw
# FoAUllGE4Gtve/7YBqvD8oXmKa5q+dQwHwYDVR0lBBgwFgYKKwYBBAGCN1sBAQYI
# KwYBBQUHAwMwDQYJKoZIhvcNAQELBQADggEBAEDd8Wf5RkHsB64vgn2slxDtHzSo
# It9xN/Dm3RdFjNZ0diTUPMgSPYQlSk8nIAfudnB9FLavGlvZLlyUpfrPSuikepj3
# i3pqNEFn6fNdNFv/wHMxv7hQTIDCmuoR1v1rX+w3oeleBPMnN3QmH4ff1NsynyV4
# dZdYgN9Cw9sC/S3pWZpJrbOs7YOM3vqyU6DciHhC4D9i2zByHCF2pu9nYfiQf5A2
# iUZenRvyo1E5rC+UP2VZXa4k7g66W20+zAajIKKIqEmRtWahekMkCcOIHFBY4RDA
# ybgPRSGur4VDAiZPjTXS90wQXrX9CwU20cfiCC6e76F4H95KtQjKYpzuNVAwggjo
# MIIG0KADAgECAhMfAAAAUeqP9pxzDKg7AAAAAABRMA0GCSqGSIb3DQEBCwUAMDwx
# EzARBgoJkiaJk/IsZAEZFgNHQkwxEzARBgoJkiaJk/IsZAEZFgNBTUUxEDAOBgNV
# BAMTB2FtZXJvb3QwHhcNMjEwNTIxMTg0NDE0WhcNMjYwNTIxMTg1NDE0WjBBMRMw
# EQYKCZImiZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQD
# EwxBTUUgQ1MgQ0EgMDEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDJ
# mlIJfQGejVbXKpcyFPoFSUllalrinfEV6JMc7i+bZDoL9rNHnHDGfJgeuRIYO1LY
# /1f4oMTrhXbSaYRCS5vGc8145WcTZG908bGDCWr4GFLc411WxA+Pv2rteAcz0eHM
# H36qTQ8L0o3XOb2n+x7KJFLokXV1s6pF/WlSXsUBXGaCIIWBXyEchv+sM9eKDsUO
# LdLTITHYJQNWkiryMSEbxqdQUTVZjEz6eLRLkofDAo8pXirIYOgM770CYOiZrcKH
# K7lYOVblx22pdNawY8Te6a2dfoCaWV1QUuazg5VHiC4p/6fksgEILptOKhx9c+ia
# piNhMrHsAYx9pUtppeaFAgMBAAGjggTcMIIE2DASBgkrBgEEAYI3FQEEBQIDAgAC
# MCMGCSsGAQQBgjcVAgQWBBQSaCRCIUfL1Gu+Mc8gpMALI38/RzAdBgNVHQ4EFgQU
# llGE4Gtve/7YBqvD8oXmKa5q+dQwggEEBgNVHSUEgfwwgfkGBysGAQUCAwUGCCsG
# AQUFBwMBBggrBgEFBQcDAgYKKwYBBAGCNxQCAQYJKwYBBAGCNxUGBgorBgEEAYI3
# CgMMBgkrBgEEAYI3FQYGCCsGAQUFBwMJBggrBgEFBQgCAgYKKwYBBAGCN0ABAQYL
# KwYBBAGCNwoDBAEGCisGAQQBgjcKAwQGCSsGAQQBgjcVBQYKKwYBBAGCNxQCAgYK
# KwYBBAGCNxQCAwYIKwYBBQUHAwMGCisGAQQBgjdbAQEGCisGAQQBgjdbAgEGCisG
# AQQBgjdbAwEGCisGAQQBgjdbBQEGCisGAQQBgjdbBAEGCisGAQQBgjdbBAIwGQYJ
# KwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMBIGA1UdEwEB/wQI
# MAYBAf8CAQAwHwYDVR0jBBgwFoAUKV5RXmSuNLnrrJwNp4x1AdEJCygwggFoBgNV
# HR8EggFfMIIBWzCCAVegggFToIIBT4YxaHR0cDovL2NybC5taWNyb3NvZnQuY29t
# L3BraWluZnJhL2NybC9hbWVyb290LmNybIYjaHR0cDovL2NybDIuYW1lLmdibC9j
# cmwvYW1lcm9vdC5jcmyGI2h0dHA6Ly9jcmwzLmFtZS5nYmwvY3JsL2FtZXJvb3Qu
# Y3JshiNodHRwOi8vY3JsMS5hbWUuZ2JsL2NybC9hbWVyb290LmNybIaBqmxkYXA6
# Ly8vQ049YW1lcm9vdCxDTj1BTUVSb290LENOPUNEUCxDTj1QdWJsaWMlMjBLZXkl
# MjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9uLERDPUFNRSxE
# Qz1HQkw/Y2VydGlmaWNhdGVSZXZvY2F0aW9uTGlzdD9iYXNlP29iamVjdENsYXNz
# PWNSTERpc3RyaWJ1dGlvblBvaW50MIIBqwYIKwYBBQUHAQEEggGdMIIBmTBHBggr
# BgEFBQcwAoY7aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraWluZnJhL2NlcnRz
# L0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6Ly9jcmwyLmFt
# ZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUHMAKGK2h0dHA6
# Ly9jcmwzLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQwNwYIKwYBBQUH
# MAKGK2h0dHA6Ly9jcmwxLmFtZS5nYmwvYWlhL0FNRVJvb3RfYW1lcm9vdC5jcnQw
# gaIGCCsGAQUFBzAChoGVbGRhcDovLy9DTj1hbWVyb290LENOPUFJQSxDTj1QdWJs
# aWMlMjBLZXklMjBTZXJ2aWNlcyxDTj1TZXJ2aWNlcyxDTj1Db25maWd1cmF0aW9u
# LERDPUFNRSxEQz1HQkw/Y0FDZXJ0aWZpY2F0ZT9iYXNlP29iamVjdENsYXNzPWNl
# cnRpZmljYXRpb25BdXRob3JpdHkwDQYJKoZIhvcNAQELBQADggIBAFAQI7dPD+jf
# XtGt3vJp2pyzA/HUu8hjKaRpM3opya5G3ocprRd7vdTHb8BDfRN+AD0YEmeDB5HK
# QoG6xHPI5TXuIi5sm/LeADbV3C2q0HQOygS/VT+m1W7a/752hMIn+L4ZuyxVeSBp
# fwf7oQ4YSZPh6+ngZvBHgfBaVz4O9/wcfw91QDZnTgK9zAh9yRKKls2bziPEnxeO
# ZMVNaxyV0v152PY2xjqIafIkUjK6vY9LtVFjJXenVUAmn3WCPWNFC1YTIIHw/mD2
# cTfPy7QA1pT+GPARAKt0bKtq9aCd/Ym0b5tPbpgCiRtzyb7fbNS1dE740re0COE6
# 7YV2wbeo2sXixzvLftH8L7s9xv9wV+G22qyKt6lmKLjFK1yMw4Ni5fMabcgmzRvS
# jAcbqgp3tk4a8emaaH0rz8MuuIP+yrxtREPXSqL/C5bzMzsikuDW9xH10graZzSm
# PjilzpRfRdu20/9UQmC7eVPZ4j1WNa1oqPHfzET3ChIzJ6Q9G3NPCB+7KwX0OQmK
# yv7IDimj8U/GlsHD1z+EF/fYMf8YXG15LamaOAohsw/ywO6SYSreVW+5Y0mzJutn
# BC9Cm9ozj1+/4kqksrlhZgR/CSxhFH3BTweH8gP2FEISRtShDZbuYymynY1un+Ry
# fiK9+iVTLdD1h/SxyxDpZMtimb4CgJQlMYIZyDCCGcQCAQEwWDBBMRMwEQYKCZIm
# iZPyLGQBGRYDR0JMMRMwEQYKCZImiZPyLGQBGRYDQU1FMRUwEwYDVQQDEwxBTUUg
# Q1MgQ0EgMDECEzYAAAIA7fyNt5zeoUgAAgAAAgAwDQYJYIZIAWUDBAIBBQCgga4w
# GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisG
# AQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIOJ1hFgV0fDXUKZE4SxUsbDsoL6oOKO+
# Usv56dnHEQW6MEIGCisGAQQBgjcCAQwxNDAyoBSAEgBNAGkAYwByAG8AcwBvAGYA
# dKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEA
# jx2f8gI00D+Xr84pz1W0nx6FATJcUd0obXrSsJp1/79OLrIWI9HFKL18XkcahBdG
# 6iKjFcA72Sykr0lu39omljWCpPDEnpcx0DmTIb9Nd1g2m8kPP/ze6RzN52NNPZui
# +NsVJKQY5cOz8WEvgiMUzL3genW0TzJznjCQRyHCoBsGMwxNUgEvk+/Kmy7eIbIf
# UVmKr6kEGt4lOK+33HMGUQMhif1SDCRypbOWqYOG5X76m3nQUaYgpniRZNgtlDOH
# a4xghnMg5bG77XpL89PoeRdAxzjaDRSI8FDVQGtTV3mHIPJM34Wt7S6hPFdEh6Dp
# x/GsGhnEEHwXvPc3QycOlaGCF5AwgheMBgorBgEEAYI3AwMBMYIXfDCCF3gGCSqG
# SIb3DQEHAqCCF2kwghdlAgEDMQ8wDQYJYIZIAWUDBAIBBQAwggFOBgsqhkiG9w0B
# CRABBKCCAT0EggE5MIIBNQIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFlAwQCAQUA
# BCCpylWJPEt/FPO0p30+IlfoIM0b6w1HdSsD4iXnF8SJXwIGZ4j8D6jUGA8yMDI1
# MDEyMjIwNDMwM1owBIACAfSggdGkgc4wgcsxCzAJBgNVBAYTAlVTMRMwEQYDVQQI
# EwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3Nv
# ZnQgQ29ycG9yYXRpb24xJTAjBgNVBAsTHE1pY3Jvc29mdCBBbWVyaWNhIE9wZXJh
# dGlvbnMxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjo5MjAwLTA1RTAtRDk0NzEl
# MCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZaCCEeowggcgMIIF
# CKADAgECAhMzAAAB5y6PL5MLTxvpAAEAAAHnMA0GCSqGSIb3DQEBCwUAMHwxCzAJ
# BgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jv
# c29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMB4XDTIzMTIwNjE4NDUxOVoXDTI1MDMw
# NTE4NDUxOVowgcsxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAw
# DgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
# JTAjBgNVBAsTHE1pY3Jvc29mdCBBbWVyaWNhIE9wZXJhdGlvbnMxJzAlBgNVBAsT
# Hm5TaGllbGQgVFNTIEVTTjo5MjAwLTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9z
# b2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBAMJXny/gi5Drn1c8zUO1pYy/38dFQLmR2IQXz1gE/r9GfuSOoyRnkRJ6
# Z/kSWLgIu1BVJ59GkXWPtLkssqKwxY4ZFotxpVsZN9yYjW8xEnW3MzAI0igKr+/L
# xYfxB1XUH8Bvmwr5D3Ii/MbDjtN9c8TxGWtq7Ar976dafAy3TrRqQRmIknPVWHUu
# FJgpqI/1nbcRmYYRMJaKCQpty4CeG+HfKsxrz24F9p4dBkQcZCp2yQzjwQFxZJZ2
# mJJIGIDHKEdSRuSeX08/O0H9JTHNFmNTNYeD1t/WapnRwiIBYLQSMrs42GVB8pJE
# dUsos0+mXf/5QvheNzRi92pzzyA4tSv/zhP3/Ermvza6W9GnYDz9qv1wbhbvrnS4
# poDFECaAviEqAhfn/RogCxvKok5ro4gZIX1r4N9eXUulA80pHv3axwXu2MPlarAi
# 6J9L1hSIcy9EuOMqTRJIJX+alcLQGg+STlqx/GuslsKwl48dI4RuWknNGbNo/o4x
# fBFytvtNcVA6xOQq6qRa+9gg+9XMLrxQz4yyQs+V3V6p044wrtJtt/a0ZJl/f6I7
# BZAxxZcH2DDmArcAhgrTxaQkm7LM+p+K2C5t1EKZiv0JWw065b7AcNgaFyIkMXYu
# SuOQVSNRxdIgl31/ayxiK1n0K6sZXvgFBx+vGO+TUvyO+03ua6UjAgMBAAGjggFJ
# MIIBRTAdBgNVHQ4EFgQUz/7gmICfNjh2kR/9mWuHUrvej1gwHwYDVR0jBBgwFoAU
# n6cVXQBeYl2D9OXSZacbUzUZ6XIwXwYDVR0fBFgwVjBUoFKgUIZOaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWljcm9zb2Z0JTIwVGltZS1TdGFt
# cCUyMFBDQSUyMDIwMTAoMSkuY3JsMGwGCCsGAQUFBwEBBGAwXjBcBggrBgEFBQcw
# AoZQaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNyb3Nv
# ZnQlMjBUaW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcnQwDAYDVR0TAQH/BAIw
# ADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAOBgNVHQ8BAf8EBAMCB4AwDQYJKoZI
# hvcNAQELBQADggIBAHSh8NuT6WVaLVwLqex+J7km2nT2jpvoBEKm+0M+rYoU/6GL
# 5Q00/ssZyIq5ySpcKYFMUiF8F4ZLG+TrJyiR1CvfzXmkQ5phZOce9DT7yErLzqvU
# Xit8G7igcHlxPLTxPiiGsb85gb8H+A2fPQ6Xq/u7+oSPPjzNdnpmXEobJnAqYplZ
# oF3YNgTDMql0uQHGzoDp6dZlHSNj6rkV1tXjmCEZMqBKvkQIA6csPieMnB+MirSZ
# FlbANlChe0lJpUdK7aUdAvdgcQWKS6dtRMl818EMsvsa/6xOZGINmTLk4DGgsbaB
# pN+6IVt+mZJ89yCXkI5TN8xCfOkp9fr4WQjRBA2+4+lawNTyxH66eLZWYOjuuaom
# uibiKGBU10tox81Sq8EvlmJIrXOZoQsEn1r5g6MTmmZJqtbmwZufuJWQXZb0lAg4
# fq0ZYsUlLkezfrNqGSgeHyIP3rct4aNmqQW6wppRbvbIyP/LFN4YQM6givfmTBfG
# vVS77OS6vbL4W41jShmOmnOn3kBbWV6E/TFo76gFXVd+9oK6v8Hk9UCnbHOuiwwR
# RwDCkmmKj5Vh8i58aPuZ5dwZBhYDxSavwroC6j4mWPwh4VLqVK8qGpCmZ0HMAwao
# 85Aq3U7DdlfF6Eru8CKKbdmIAuUzQrnjqTSxmvF1k+CmbPs7zD2Acu7JkBB7MIIH
# cTCCBVmgAwIBAgITMwAAABXF52ueAptJmQAAAAAAFTANBgkqhkiG9w0BAQsFADCB
# iDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
# ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMp
# TWljcm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMjEw
# OTMwMTgyMjI1WhcNMzAwOTMwMTgzMjI1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UE
# CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
# b2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQ
# Q0EgMjAxMDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAOThpkzntHIh
# C3miy9ckeb0O1YLT/e6cBwfSqWxOdcjKNVf2AX9sSuDivbk+F2Az/1xPx2b3lVNx
# WuJ+Slr+uDZnhUYjDLWNE893MsAQGOhgfWpSg0S3po5GawcU88V29YZQ3MFEyHFc
# UTE3oAo4bo3t1w/YJlN8OWECesSq/XJprx2rrPY2vjUmZNqYO7oaezOtgFt+jBAc
# nVL+tuhiJdxqD89d9P6OU8/W7IVWTe/dvI2k45GPsjksUZzpcGkNyjYtcI4xyDUo
# veO0hyTD4MmPfrVUj9z6BVWYbWg7mka97aSueik3rMvrg0XnRm7KMtXAhjBcTyzi
# YrLNueKNiOSWrAFKu75xqRdbZ2De+JKRHh09/SDPc31BmkZ1zcRfNN0Sidb9pSB9
# fvzZnkXftnIv231fgLrbqn427DZM9ituqBJR6L8FA6PRc6ZNN3SUHDSCD/AQ8rdH
# GO2n6Jl8P0zbr17C89XYcz1DTsEzOUyOArxCaC4Q6oRRRuLRvWoYWmEBc8pnol7X
# KHYC4jMYctenIPDC+hIK12NvDMk2ZItboKaDIV1fMHSRlJTYuVD5C4lh8zYGNRiE
# R9vcG9H9stQcxWv2XFJRXRLbJbqvUAV6bMURHXLvjflSxIUXk8A8FdsaN8cIFRg/
# eKtFtvUeh17aj54WcmnGrnu3tz5q4i6tAgMBAAGjggHdMIIB2TASBgkrBgEEAYI3
# FQEEBQIDAQABMCMGCSsGAQQBgjcVAgQWBBQqp1L+ZMSavoKRPEY1Kc8Q/y8E7jAd
# BgNVHQ4EFgQUn6cVXQBeYl2D9OXSZacbUzUZ6XIwXAYDVR0gBFUwUzBRBgwrBgEE
# AYI3TIN9AQEwQTA/BggrBgEFBQcCARYzaHR0cDovL3d3dy5taWNyb3NvZnQuY29t
# L3BraW9wcy9Eb2NzL1JlcG9zaXRvcnkuaHRtMBMGA1UdJQQMMAoGCCsGAQUFBwMI
# MBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1UdDwQEAwIBhjAPBgNVHRMB
# Af8EBTADAQH/MB8GA1UdIwQYMBaAFNX2VsuP6KJcYmjRPZSQW9fOmhjEMFYGA1Ud
# HwRPME0wS6BJoEeGRWh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3By
# b2R1Y3RzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNybDBaBggrBgEFBQcBAQRO
# MEwwSgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2Vy
# dHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3J0MA0GCSqGSIb3DQEBCwUAA4IC
# AQCdVX38Kq3hLB9nATEkW+Geckv8qW/qXBS2Pk5HZHixBpOXPTEztTnXwnE2P9pk
# bHzQdTltuw8x5MKP+2zRoZQYIu7pZmc6U03dmLq2HnjYNi6cqYJWAAOwBb6J6Gng
# ugnue99qb74py27YP0h1AdkY3m2CDPVtI1TkeFN1JFe53Z/zjj3G82jfZfakVqr3
# lbYoVSfQJL1AoL8ZthISEV09J+BAljis9/kpicO8F7BUhUKz/AyeixmJ5/ALaoHC
# gRlCGVJ1ijbCHcNhcy4sa3tuPywJeBTpkbKpW99Jo3QMvOyRgNI95ko+ZjtPu4b6
# MhrZlvSP9pEB9s7GdP32THJvEKt1MMU0sHrYUP4KWN1APMdUbZ1jdEgssU5HLcEU
# BHG/ZPkkvnNtyo4JvbMBV0lUZNlz138eW0QBjloZkWsNn6Qo3GcZKCS6OEuabvsh
# VGtqRRFHqfG3rsjoiV5PndLQTHa1V1QJsWkBRH58oWFsc/4Ku+xBZj1p/cvBQUl+
# fpO+y/g75LcVv7TOPqUxUYS8vwLBgqJ7Fx0ViY1w/ue10CgaiQuPNtq6TPmb/wrp
# NPgkNWcr4A245oyZ1uEi6vAnQj0llOZ0dFtq0Z4+7X6gMTN9vMvpe784cETRkPHI
# qzqKOghif9lwY1NNje6CbaUFEMFxBmoQtB1VM1izoXBm8qGCA00wggI1AgEBMIH5
# oYHRpIHOMIHLMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
# A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSUw
# IwYDVQQLExxNaWNyb3NvZnQgQW1lcmljYSBPcGVyYXRpb25zMScwJQYDVQQLEx5u
# U2hpZWxkIFRTUyBFU046OTIwMC0wNUUwLUQ5NDcxJTAjBgNVBAMTHE1pY3Jvc29m
# dCBUaW1lLVN0YW1wIFNlcnZpY2WiIwoBATAHBgUrDgMCGgMVALNyBOcZqxLB792u
# 75w97U0X+/BDoIGDMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
# bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
# b3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAw
# DQYJKoZIhvcNAQELBQACBQDrO2MBMCIYDzIwMjUwMTIyMTIyODQ5WhgPMjAyNTAx
# MjMxMjI4NDlaMHQwOgYKKwYBBAGEWQoEATEsMCowCgIFAOs7YwECAQAwBwIBAAIC
# HLYwBwIBAAICEoYwCgIFAOs8tIECAQAwNgYKKwYBBAGEWQoEAjEoMCYwDAYKKwYB
# BAGEWQoDAqAKMAgCAQACAwehIKEKMAgCAQACAwGGoDANBgkqhkiG9w0BAQsFAAOC
# AQEAOdacQKzrlEs+lYmcDAHoiG7R7Dxu6B3NU254RSvwn81nZZiIDYD6p/c0Bv0C
# 2jn6mBNOeda/mLhYrTq1vXYc1+9//VEMrgbzvs0h9yWyoLkbABQkgWLJ1KrDsQ7K
# TjjBon23NFaOxpVsGXN/rvzmoRkkajRTp8r7YZrxC/ryHg9RTkgfEICmOMnAwT5G
# 42FluB/J/dMI4ZXbbhtRBeqeBV3bUi2IA6+nTDQUm9CaTWNTnSzlUuPCWLL/DO/+
# yBLnDjjId41AMb4Wu0+TrbwTja1EdRo6mKQ9c34JSv8XEE5XfFRXWVYt5UFSvIqZ
# mwELzXL8Yg6wHCpCrCyaUZJk7jGCBA0wggQJAgEBMIGTMHwxCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1l
# LVN0YW1wIFBDQSAyMDEwAhMzAAAB5y6PL5MLTxvpAAEAAAHnMA0GCWCGSAFlAwQC
# AQUAoIIBSjAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwLwYJKoZIhvcNAQkE
# MSIEIM8bMKEE+0cDTzSAhlqxDmRF+qUOuXDax8XCX6gzwuUxMIH6BgsqhkiG9w0B
# CRACLzGB6jCB5zCB5DCBvQQg5TZdDXZqhv0N4MVcz1QUd4RfvgW/QAG9AwbuoLnW
# c60wgZgwgYCkfjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
# MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
# MSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAecu
# jy+TC08b6QABAAAB5zAiBCAs7gMGdwxOn6e6X5ACQD4NnFQap5UMSuTmkPBae+TJ
# 7TANBgkqhkiG9w0BAQsFAASCAgBs2wsUPQqLuxj2lVe/dyLUvkey0ep9EWG+85UZ
# I9MQ2ycM39JBhieRLoZ5ijAJj4KqLCo1YvpLywB+Hz8yJihq144GeIAqfapG8l1c
# NL4XjhLwdSqZcMQPzZfoXzCWXZMXKVlBk27onY1DBsx8N8wka3GfSnxnU/XylvUN
# UY2CLkmoxqT5zF+hhVv6MwEsle21+Ri91Wc2AV2/kXORxcYVAAci2wWKmt1+9rM8
# 1Wj/yReMfEPVktvngZWr+5hf945uW+s+E8hT6cjBr1O+EfZnWWpMejKlOQZYNZ5n
# V/6Pjz98b/gAtlR5DXKco9UCL2oMshzRKePlgCRj6Wa6eDLeOftx+LzCrmKzMIBL
# 61wFGI3DOSe8JNsnDsBQyz2kDF744WXRPIOrk9fBZTzZwS0OZlhvVKDZCOKCPBeU
# 6yWZacqgPlLReuSZZ1+VYMQF4gZnZBLKd78EptdXQj8g3qqAssoDmgyamNDcq110
# KEkXqmsQcoKIY/Wg9nt3DkcKWOFUKUhW+WUVp9GJQbkgIpNnNz7Av59LVc+hnzAA
# MLzmintyXoX5K3qejoFo8owOzA6l8C6pR+JWZ5QXeJ9k17ZKsGXlQCaqnr7ou4l8
# 8CUsUG6W6ul+3SBWV4tqOKyJOEHwZmDoATeAl/iFPtfZMcTmhTZqLDgTyXyVpFqN
# Q+zfaw==
# SIG # End signature block
