﻿####################################################################################
####################################################################################
## ValidateAOS.ps1
## Validate AOS deployment
## Date: 09/29/2015
##
####################################################################################
####################################################################################
[CmdletBinding()]
Param(
   [Parameter(Mandatory = $true)]
   [string]$InputXml,
   [Parameter(Mandatory = $false)]
   [string]$CredentialsXml,
   [Parameter(Mandatory = $true)]
   [string]$Log
)

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking
Initialize-Log $Log

####################################################################################
## Helper Functions
####################################################################################

## Append XML Rows to Template
function Append-RowToXML
{
    [CmdletBinding()]
    Param(
       [Parameter(Mandatory = $false)]
       [string]$TestName,
       [Parameter(Mandatory = $false)]
       [string]$TestType,
       [Parameter(Mandatory = $false)]
       [string]$TestResult,
       [Parameter(Mandatory = $false)]
       [string]$RawResult,
       [Parameter(Mandatory = $false)]
       [string]$TimeStamp,
       [Parameter(Mandatory = $true)]
       [xml]$xmlTemplate
    )

    Write-Log "Getting existing rows from XML Template"
    $rows = $xmlTemplate.SelectSingleNode('CollectionResult/TabularResults/TabularData/Rows')
    Write-Log "Creating new row"
    $row = $xmlTemplate.CreateElement('ArrayOfStrings')
    $column = $xmlTemplate.CreateElement('string')#TestName
    $column.InnerText = $TestName
    $row.AppendChild($column)
    Write-Log "Adding column value: $TestName"
    $column = $xmlTemplate.CreateElement('string')#TestType
    $column.InnerText = $TestType
    $row.AppendChild($column)
    Write-Log "Adding column value: $TestType"
    $column = $xmlTemplate.CreateElement('string')#TestResult
    $column.InnerText = $TestResult
    $row.AppendChild($column)
    Write-Log "Adding column value: $TestResult"
    $column = $xmlTemplate.CreateElement('string')#RawResult
    $column.InnerText = $RawResult
    $row.AppendChild($column)
    
    $column = $xmlTemplate.CreateElement('string')#TimeStamp
    $column.InnerText = $TimeStamp
    $row.AppendChild($column)
    $rows.AppendChild($row)
    Write-Log "Adding column value: $TimeStamp"
    $xmlTemplate.CollectionResult.TabularResults.TabularData.AppendChild($rows)
    $xmlTemplate.Save($xmlTemplate)
    Write-Log "Saved rows to XML Template"
}

####################################################################################
## Validation Functions
####################################################################################

## Validate endpoint sending HTTP request to configured endpoint 
function Validate-Endpoint{ 
     [CmdletBinding()] 
    Param( 
     [Parameter(Mandatory = $true)] 
     [string]$EndPointUrl 
     ) 
 
     [bool]$result = $false 
     [string]$rawResult 
     [string]$timestamp 
 
     try{ 
         Write-Log "Connecting to '$EndPointUrl'"
         $CurrentTime = (Get-Date).ToUniversalTime() 
         $webRequest = Invoke-WebRequest -Uri $EndPointUrl -TimeoutSec (5 * 60) -UseBasicParsing 
         if($webRequest.StatusCode -eq 200){ 
             $result = $true 
             $UrlTime = [DateTime]::Parse($webRequest.Headers.Date).ToUniversalTime() 
             $rawResult = ('HttpResult: ' + $webRequest.StatusCode.ToString() + '; PingTime(ms): ' + ($CurrentTime - $UrlTime).TotalMilliseconds).ToString() 
             $timestamp = (Get-Date).ToString() 
             Write-Log "Web request returned - $rawResult"
         } 
     } 
     catch{ 
         $rawResult = $_.Exception 
         $timestamp = (Get-Date).ToString() 
         Write-Log "ERROR: $($_.Exception) CALLSTACK: $_"
     } 
 
     if($result){ 
         $returnProperties = @{ 
             Result=1; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     else{ 
         $returnProperties = @{ 
             Result=0; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     $resultObject = New-Object PsObject -Property $returnProperties 
     return $resultObject 
} 
 
## Validate install Files from a manifest file 
function Validate-Install{ 
     [CmdletBinding()] 
    Param( 
     [Parameter(Mandatory = $true)] 
     [string]$InstallPath, 
     [Parameter(Mandatory = $false)] 
     [string]$ManifestPath 
     ) 
 
     [bool]$result = $false 
     [string]$rawResult 
     [string]$timestamp 
 
     try{ 
         Write-Log "Validating Install at '$InstallPath'"
         if(Test-Path -Path $InstallPath) 
         { 
             Write-Log "Comparing '$InstallPath' to manifest"
             [System.Array]$installedfiles = @() 
             $installedFilesRaw = Get-ChildItem -Path $InstallPath -Recurse | Where {$_.PSIsContainer -eq $false} | Select-Object -Property Name 
            foreach($file in $installedFilesRaw){ 
                $installedfiles += $file.Name 
             } 
 
             if(Test-Path -Path $ManifestPath){ 
                 $manifestFiles = Get-Content -Path $ManifestPath 
                 $fileCompare = Compare-Object -ReferenceObject $manifestFiles -DifferenceObject $installedFiles -Property Name -PassThru 
                 $timestamp = (Get-Date).ToString() 
                 if($fileCompare -eq $null) 
                 { 
                     $rawResult = "Installed file ARE a match to the manifest" 
                     Write-Log "$rawResult"
                     $result = $true 
                 } 
                 else 
                 { 
                     $rawResult = ("{0} Files are missing." -f $fileCompare.Count ) 
                     Write-Log "$rawResult"
                 } 
             } 
             else{ 
                Throw "$ManifestPath does not exist." 
             } 
         } 
         else{ 
            Throw "$InstallPath does not exist." 
         } 
     } 
     catch{ 
         $rawResult = $_.Exception 
         $timestamp = (Get-Date).ToString() 
         Write-Log "ERROR: $($_.Exception) CALLSTACK: $_"
     } 
 
     if($result){ 
         $returnProperties = @{ 
             Result=1; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     else{ 
         $returnProperties = @{ 
             Result=0; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
    } 
     $resultObject = New-Object PsObject -Property $returnProperties 
     return $resultObject 
} 
 
## Validate service is running 
function Validate-Service{ 
    [CmdletBinding()] 
    Param( 
     [Parameter(Mandatory = $true)] 
     [string]$ServiceName, 
     [Parameter(Mandatory = $true)] 
     [ValidateSet("Running","Stopped","Paused")] 
     [string]$CurrentState 
     ) 
 
     [bool]$result = $false 
     [string]$rawResult 
     [string]$timestamp 
 
     try{ 
         Write-Log "Validating Service: '$ServiceName' is $CurrentState"
         $thisService = Get-Service -Name $ServiceName 
         $timestamp = (Get-Date).ToString() 
         $rawResult = ("ServiceName: {0}; DisplayName: {1}; Status: {2}" -f $thisService.Name, $thisService.DisplayName, $thisService.Status) 
         if($thisService.Status.ToString() -eq $CurrentState) 
         { 
            $result = $true 
         } 
         Write-Log "Service: $ServiceName is $($thisService.Status)"
 
     } 
     catch{ 
         $rawResult = $_.Exception 
         $timestamp = (Get-Date).ToString() 
         Write-Log "ERROR: $($_.Exception) CALLSTACK: $_"
     } 
 
     if($result){ 
         $returnProperties = @{ 
             Result=1; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     else{ 
         $returnProperties = @{ 
             Result=0; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     $resultObject = New-Object PsObject -Property $returnProperties 
     return $resultObject 
} 
 
## Validate appPool is started 
function Validate-AppPool{ 
    [CmdletBinding()] 
    Param( 
     [Parameter(Mandatory = $true)] 
     [string]$AppPoolName, 
     [Parameter(Mandatory = $true)] 
     [ValidateSet("Started","Stopped")] 
     [string]$CurrentState 
     ) 
 
     [bool]$result = $false 
     [string]$rawResult 
     [string]$timestamp 
 
     try{ 
         Write-Log "Validating AppPool: '$AppPoolName' is $CurrentState"
         Get-WebAppPoolState 
         $appPoolStatus = Get-WebAppPoolState -Name $AppPoolName 
         $timestamp = (Get-Date).ToString() 
         $rawResult = ("AppPoolName: {0}; Status: {1}" -f $AppPoolName, $appPoolStatus) 
         if($appPoolStatus.Value -eq $CurrentState) 
         { 
            $result = $true 
         } 
         Write-Log "AppPool: $AppPoolName is $($appPoolStatus.Value)"
     } 
     catch{ 
         $rawResult = $_.Exception 
         $timestamp = (Get-Date).ToString() 
         Write-Log "ERROR: $($_.Exception) CALLSTACK: $_"
     } 
 
     if($result){ 
         $returnProperties = @{ 
             Result=1; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     else{ 
         $returnProperties = @{ 
             Result=0; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     $resultObject = New-Object PsObject -Property $returnProperties 
     return $resultObject 
} 




####################################################################################
## Parameter Setting
####################################################################################
Write-Log "Setting DVT execution parameters"

if(Test-Path -Path $InputXML)
{
    Write-Log "Parsing the xml for parameters/settings"
    [xml]$DVTParams = Get-Content -Path $InputXML
    [string]$ServiceName = $DVTParams.DVTParameters.ServiceName    
    [string]$AosWebrootPath = $DVTParams.DVTParameters.AosWebRootPath    
    [string]$XmlOutputPath = $DVTParams.DVTParameters.OutputPath    
	[string]$endPoint = $DVTParams.DVTParameters.EndPoint #Infrastructure.HostUrl
    [string]$installPath = $DVTParams.DVTParameters.InstallPath
	[string]$manifestFile = $DVTParams.DVTParameters.ManifestPath
	[string]$ServiceState = $DVTParams.DVTParameters.ServiceState
	[string]$AppPoolName = $DVTParams.DVTParameters.AppPoolName
	[string]$AppPoolState = $DVTParams.DVTParameters.AppPoolState
    [string]$BatchService = $DVTParams.DVTParameters.BatchService
}
else
{
    throw "Unable to parse settings from service model. Xml doesnt exist at: $InputXML"
}

if(-not ([string]::IsNullOrEmpty($CredentialsXml)))
{
    Write-Log "Parsing the CredentialsXml"
    if(Test-Path -Path $CredentialsXml)
    {
        Write-Log "Parsing the xml for local credentials"
        $localCredentials = Import-Clixml -Path $CredentialsXml
        [string]$UserName = $localCredentials.GetNetworkCredential().UserName
        [string]$UserPassword = $localCredentials.GetNetworkCredential().Password
    }
    else
    {
        throw "Unable to parse credentials from service model. Xml doesnt exist at: $CredentialsXML"
    }
}

Write-Log "Setting diagnostics-related parameters"
[string]$CollectorName = "$($ServiceName).DVT"
[string]$CollectorType = 'PowerShellCollector'
[string]$TargetName = (hostname)

if(-not (Test-Path -Path $XmlOutputPath))
{
    Write-Log "Creating diagnostics result directory at $XmlOutputPath"
    New-Item -Path $XmlOutputPath -Type Container | Out-Null
}

[string]$XMLFilePath = Join-Path -Path $XmlOutputPath -ChildPath "$([System.DateTime]::Now.ToFileTimeUtc())_$($ServiceName)DVTResults.xml"

####################################################################################
## Diagnostics Collector XML Template
####################################################################################
[xml]$xmlTemplate = @"
<?xml version="1.0"?>
<CollectionResult xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema">
  <CollectorName>$CollectorName</CollectorName>
  <CollectorType>$CollectorType</CollectorType>
  <ErrorMessages />
  <TabularResults>
    <TabularData>
      <TargetName>$TargetName</TargetName>
      <Columns>
        <string>TestName</string>
        <string>TestType</string>
        <string>PassResult</string>
        <string>RawResult</string>
        <string>TimeStamp</string>
      </Columns>
      <Rows>
      </Rows>
    </TabularData>
  </TabularResults>
</CollectionResult>
"@

####################################################################################
## Main Execution
####################################################################################

Write-Log "Running validations for $ServiceName"
try
{
     #End point validation 
     Write-Log "Validate-Endpoint -EndPointUrl" 
     $endpointResult = Validate-Endpoint -EndPointUrl $endPoint
 
     Append-RowToXML -TestName 'AOS.Validate-Endpoint' -TestType 'DVT' -TestResult $endpointResult.Result -RawResult $endpointResult.RawResults -TimeStamp $endpointResult.TimeStamp -xmlTemplate $xmlTemplate | Out-Null 

    $ValidateBatch = (![System.String]::IsNullOrWhiteSpace($DVTParams.DVTParameters.ValidateBatch) -and [System.Convert]::ToBoolean($DVTParams.DVTParameters.ValidateBatch))
    if ($ValidateBatch)
    {
        #AXBatch Service 
        Write-Log "Validate-Service -ServiceName $BatchService -CurrentState $ServiceState" 
        $serviceResult = Validate-Service -ServiceName $BatchService -CurrentState $ServiceState 

        Append-RowToXML -TestName 'AOS.Validate-Service' -TestType 'DVT' -TestResult $serviceResult.Result -RawResult $serviceResult.RawResults -TimeStamp $serviceResult.TimeStamp -xmlTemplate $xmlTemplate | Out-Null 
    } 

     #IIS AppPool Validation 
     Write-Log "Validate-AppPool -AppPoolName $AppPoolName -CurrentState $AppPoolState"
     $apppoolResult = Validate-AppPool -AppPoolName $AppPoolName -CurrentState $AppPoolState 

     Append-RowToXML -TestName 'AOS.Validate-AppPool' -TestType 'DVT' -TestResult $apppoolResult.Result -RawResult $apppoolResult.RawResults -TimeStamp $apppoolResult.TimeStamp -xmlTemplate $xmlTemplate | Out-Null 

    #Writing XML results
    Write-Log "Writing DVT results to $XMLFilePath"
    $xmlTemplate.InnerXml | Out-File -FilePath $XMLFilePath -Force -Encoding utf8

    [bool]$dvtResult = $endpointResult.Result -and $apppoolResult.Result
    if ($ValidateBatch)
    {
        $dvtResult = $dvtResult -and $serviceResult.Result
    }

}
catch
{
    Write-Exception $_
}

if($dvtResult)
{
    $exitProperties = @{'ExitCode'= 0}
    $exitObject = New-Object PsObject -Property $exitProperties
    Write-Log "DVT Script Completed, ExitCode: $($exitObject.ExitCode)"
    return $exitObject
}
else
{
    $exitProperties = @{'ExitCode'= 1; 'Message'="DVT Validation failed, see log: '$Log' for further details, and '$XMLFilePath' for test results"}
    $exitObject = New-Object PsObject -Property $exitProperties
    Write-Log "DVT Script Completed, ExitCode: $($exitObject.ExitCode)"
    throw $exitObject
}


# SIG # Begin signature block
# MIIoPAYJKoZIhvcNAQcCoIIoLTCCKCkCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBPDETZSfEdM3Dd
# jYdA0NRlWVmqeL50qgjbytga87dgeaCCDYUwggYDMIID66ADAgECAhMzAAAEhJji
# EuB4ozFdAAAAAASEMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjUwNjE5MTgyMTM1WhcNMjYwNjE3MTgyMTM1WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDtekqMKDnzfsyc1T1QpHfFtr+rkir8ldzLPKmMXbRDouVXAsvBfd6E82tPj4Yz
# aSluGDQoX3NpMKooKeVFjjNRq37yyT/h1QTLMB8dpmsZ/70UM+U/sYxvt1PWWxLj
# MNIXqzB8PjG6i7H2YFgk4YOhfGSekvnzW13dLAtfjD0wiwREPvCNlilRz7XoFde5
# KO01eFiWeteh48qUOqUaAkIznC4XB3sFd1LWUmupXHK05QfJSmnei9qZJBYTt8Zh
# ArGDh7nQn+Y1jOA3oBiCUJ4n1CMaWdDhrgdMuu026oWAbfC3prqkUn8LWp28H+2S
# LetNG5KQZZwvy3Zcn7+PQGl5AgMBAAGjggGCMIIBfjAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUBN/0b6Fh6nMdE4FAxYG9kWCpbYUw
# VAYDVR0RBE0wS6RJMEcxLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9wZXJh
# dGlvbnMgTGltaXRlZDEWMBQGA1UEBRMNMjMwMDEyKzUwNTM2MjAfBgNVHSMEGDAW
# gBRIbmTlUAXTgqoXNzcitW2oynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8v
# d3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9NaWNDb2RTaWdQQ0EyMDExXzIw
# MTEtMDctMDguY3JsMGEGCCsGAQUFBwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDov
# L3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNDb2RTaWdQQ0EyMDEx
# XzIwMTEtMDctMDguY3J0MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQELBQADggIB
# AGLQps1XU4RTcoDIDLP6QG3NnRE3p/WSMp61Cs8Z+JUv3xJWGtBzYmCINmHVFv6i
# 8pYF/e79FNK6P1oKjduxqHSicBdg8Mj0k8kDFA/0eU26bPBRQUIaiWrhsDOrXWdL
# m7Zmu516oQoUWcINs4jBfjDEVV4bmgQYfe+4/MUJwQJ9h6mfE+kcCP4HlP4ChIQB
# UHoSymakcTBvZw+Qst7sbdt5KnQKkSEN01CzPG1awClCI6zLKf/vKIwnqHw/+Wvc
# Ar7gwKlWNmLwTNi807r9rWsXQep1Q8YMkIuGmZ0a1qCd3GuOkSRznz2/0ojeZVYh
# ZyohCQi1Bs+xfRkv/fy0HfV3mNyO22dFUvHzBZgqE5FbGjmUnrSr1x8lCrK+s4A+
# bOGp2IejOphWoZEPGOco/HEznZ5Lk6w6W+E2Jy3PHoFE0Y8TtkSE4/80Y2lBJhLj
# 27d8ueJ8IdQhSpL/WzTjjnuYH7Dx5o9pWdIGSaFNYuSqOYxrVW7N4AEQVRDZeqDc
# fqPG3O6r5SNsxXbd71DCIQURtUKss53ON+vrlV0rjiKBIdwvMNLQ9zK0jy77owDy
# XXoYkQxakN2uFIBO1UNAvCYXjs4rw3SRmBX9qiZ5ENxcn/pLMkiyb68QdwHUXz+1
# fI6ea3/jjpNPz6Dlc/RMcXIWeMMkhup/XEbwu73U+uz/MIIHejCCBWKgAwIBAgIK
# YQ6Q0gAAAAAAAzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNV
# BAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
# c29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlm
# aWNhdGUgQXV0aG9yaXR5IDIwMTEwHhcNMTEwNzA4MjA1OTA5WhcNMjYwNzA4MjEw
# OTA5WjB+MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
# BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYD
# VQQDEx9NaWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG
# 9w0BAQEFAAOCAg8AMIICCgKCAgEAq/D6chAcLq3YbqqCEE00uvK2WCGfQhsqa+la
# UKq4BjgaBEm6f8MMHt03a8YS2AvwOMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc
# 6Whe0t+bU7IKLMOv2akrrnoJr9eWWcpgGgXpZnboMlImEi/nqwhQz7NEt13YxC4D
# dato88tt8zpcoRb0RrrgOGSsbmQ1eKagYw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+
# lD3v++MrWhAfTVYoonpy4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+EGvKhL1nk
# kDstrjNYxbc+/jLTswM9sbKvkjh+0p2ALPVOVpEhNSXDOW5kf1O6nA+tGSOEy/S6
# A4aN91/w0FK/jJSHvMAhdCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rhKEmd
# X4jiJV3TIUs+UsS1Vz8kA/DRelsv1SPjcF0PUUZ3s/gA4bysAoJf28AVs70b1FVL
# 5zmhD+kjSbwYuER8ReTBw3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zd
# sGbiwZeBe+3W7UvnSSmnEyimp31ngOaKYnhfsi+E11ecXL93KCjx7W3DKI8sj0A3
# T8HhhUSJxAlMxdSlQy90lfdu+HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS
# 4NaIjAsCAwEAAaOCAe0wggHpMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBRI
# bmTlUAXTgqoXNzcitW2oynUClTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTAL
# BgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBRyLToCMZBD
# uRQFTuHqp8cx0SOJNDBaBgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsLm1pY3Jv
# c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFf
# MDNfMjIuY3JsMF4GCCsGAQUFBwEBBFIwUDBOBggrBgEFBQcwAoZCaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFf
# MDNfMjIuY3J0MIGfBgNVHSAEgZcwgZQwgZEGCSsGAQQBgjcuAzCBgzA/BggrBgEF
# BQcCARYzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1h
# cnljcHMuaHRtMEAGCCsGAQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAHAAbwBsAGkA
# YwB5AF8AcwB0AGEAdABlAG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn
# 8oalmOBUeRou09h0ZyKbC5YR4WOSmUKWfdJ5DJDBZV8uLD74w3LRbYP+vj/oCso7
# v0epo/Np22O/IjWll11lhJB9i0ZQVdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0b
# pdS1HXeUOeLpZMlEPXh6I/MTfaaQdION9MsmAkYqwooQu6SpBQyb7Wj6aC6VoCo/
# KmtYSWMfCWluWpiW5IP0wI/zRive/DvQvTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvy
# CInWH8MyGOLwxS3OW560STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn+N4sOiBp
# mLJZiWhub6e3dMNABQamASooPoI/E01mC8CzTfXhj38cbxV9Rad25UAqZaPDXVJi
# hsMdYzaXht/a8/jyFqGaJ+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw3MYb
# BL7fQccOKO7eZS/sl/ahXJbYANahRr1Z85elCUtIEJmAH9AAKcWxm6U/RXceNcbS
# oqKfenoi+kiVH6v7RyOA9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sL
# gOppO6/8MO0ETI7f33VtY5E90Z1WTk+/gFcioXgRMiF670EKsT/7qMykXcGhiJtX
# cVZOSEXAQsmbdlsKgEhr/Xmfwb1tbWrJUnMTDXpQzTGCGg0wghoJAgEBMIGVMH4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01p
# Y3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTECEzMAAASEmOIS4HijMV0AAAAA
# BIQwDQYJYIZIAWUDBAIBBQCgga4wGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQw
# HAYKKwYBBAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIFbl
# nddwkWx06Q6BUqL75BZnV2OJrQnJQ/zIfszQnRDJMEIGCisGAQQBgjcCAQwxNDAy
# oBSAEgBNAGkAYwByAG8AcwBvAGYAdKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20wDQYJKoZIhvcNAQEBBQAEggEAE5zbRSayjEkciEp7qNKlUIMh4JIoVdFswADn
# 39+E64LDZh2sbLFCSStjgx18wEC9gUHypZFdpKKBGmUdQVunO8/5MZDIhBM7UX1q
# dYPvLOWKUUyriNpRsOr6mypx5N/JF9tHZZhdYT7G+yn8PhKciHXrbdKPf5/K0Byp
# E5ajg2042WNpfIMAHqggLxm6SuC7ejTX8zn35OMnUjnalzxT1x67u09hvUWHd4CN
# qtZ9nOjMhzX0C4O4aJI0e0FsfTch2rqIsBH5k/Fd0z/ks3Y2QdUVP15SdNuqumJe
# Toayr1ijY5Se2swff5sqaCqvnusXmGlmUDdjq5s1UmMFI14I8aGCF5cwgheTBgor
# BgEEAYI3AwMBMYIXgzCCF38GCSqGSIb3DQEHAqCCF3AwghdsAgEDMQ8wDQYJYIZI
# AWUDBAIBBQAwggFSBgsqhkiG9w0BCRABBKCCAUEEggE9MIIBOQIBAQYKKwYBBAGE
# WQoDATAxMA0GCWCGSAFlAwQCAQUABCC+WZuXgrWrA0RlPZh9uTipmZysY7uZbzIf
# XlWyOBhTcwIGaKOjxwa+GBMyMDI1MDkwNDA3MjYxOS41NTZaMASAAgH0oIHRpIHO
# MIHLMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQL
# ExxNaWNyb3NvZnQgQW1lcmljYSBPcGVyYXRpb25zMScwJQYDVQQLEx5uU2hpZWxk
# IFRTUyBFU046OTYwMC0wNUUwLUQ5NDcxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1l
# LVN0YW1wIFNlcnZpY2WgghHtMIIHIDCCBQigAwIBAgITMwAAAgTY4A4HlzJYmAAB
# AAACBDANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2Fz
# aGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
# cnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAx
# MDAeFw0yNTAxMzAxOTQyNDdaFw0yNjA0MjIxOTQyNDdaMIHLMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1l
# cmljYSBPcGVyYXRpb25zMScwJQYDVQQLEx5uU2hpZWxkIFRTUyBFU046OTYwMC0w
# NUUwLUQ5NDcxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2Uw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDw3Sbcee2d66vkWGTIXhfG
# qqgQGxQXTnq44XlUvNzFSt7ELtO4B939jwZFX7DrRt/4fpzGNkFdGpc7EL5S86qK
# Yv360eXjW+fIv1lAqDD31d/p8Ai9/AZz8M95zo0rDpK2csz9WAyR9FtUDx52VOs9
# qP3/pgpHvgUvD8s6/3KNITzms8QC1tJ3TMw1cRn9CZgVIYzw2iD/ZvOW0sbF/DRd
# gM8UdtxjFIKTXTaI/bJhsQge3TwayKQ2j85RafFFVCR5/ChapkrBQWGwNFaPzpmY
# N46mPiOvUxriISC9nQ/GrDXUJWzLDmchrmr2baABJevvw31UYlTlLZY6zUmjkgaR
# fpozd+Glq9TY2E3Dglr6PtTEKgPu2hM6v8NiU5nTvxhDnxdmcf8UN7goeVlELXbO
# m7j8yw1xM9IyyQuUMWkorBaN/5r9g4lvYkMohRXEYB0tMaOPt0FmZmQMLBFpNRVn
# XBTa4haXvn1adKrvTz8VlfnHxkH6riA/h2AlqYWhv0YULsEcHnaDWgqA29ry+jH0
# 97MpJ/FHGHxk+d9kH2L5aJPpAYuNmMNPB7FDTPWAx7Apjr/J5MhUx0i07gV2brAZ
# 9J9RHi+fMPbS+Qm4AonC5iOTj+dKCttVRs+jKKuO63CLwqlljvnUCmuSavOX54IX
# OtKcFZkfDdOZ7cE4DioP1QIDAQABo4IBSTCCAUUwHQYDVR0OBBYEFBp1dktAcGpW
# /Km6qm+vu4M1GaJfMB8GA1UdIwQYMBaAFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMF8G
# A1UdHwRYMFYwVKBSoFCGTmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMv
# Y3JsL01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0ElMjAyMDEwKDEpLmNybDBs
# BggrBgEFBQcBAQRgMF4wXAYIKwYBBQUHMAKGUGh0dHA6Ly93d3cubWljcm9zb2Z0
# LmNvbS9wa2lvcHMvY2VydHMvTWljcm9zb2Z0JTIwVGltZS1TdGFtcCUyMFBDQSUy
# MDIwMTAoMSkuY3J0MAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUH
# AwgwDgYDVR0PAQH/BAQDAgeAMA0GCSqGSIb3DQEBCwUAA4ICAQBecv6sRw2HTLMy
# UC1WJJ+FR+DgA9Jkv0lGsIt4y69CmOj8R63oFbhSmcdpakxqNbr8v9dyTb4RDyNq
# tohiiXbtrXmQK5X7y/Q++F0zMotTtTpTPvG3eltyV/LvO15mrLoNQ7W4VH58aLt0
# 30tORxs8VnAQQF5BmQQMOua+EQgH4f1F4uF6rl3EC17JBSJ0wjHSea/n0WYiHPR0
# qkz/NRAf8lSUUV0gbIMawGIjn7+RKyCr+8l1xdNkK/F0UYuX3hG0nE+9Wc0L4A/e
# nluUN7Pa9vOV6Vi3BOJST0RY/ax7iZ45leM8kqCw7BFPcTIkWzxpjr2nCtirnkw7
# OBQ6FNgwIuAvYNTU7r60W421YFOL5pTsMZcNDOOsA01xv7ymCF6zknMGpRHuw0Rb
# 2BAJC9quU7CXWbMbAJLdZ6XINKariSmCX3/MLdzcW5XOycK0QhoRNRf4WqXRshEB
# aY2ymJvHO48oSSY/kpuYvBS3ljAAuLN7Rp8jWS7t916paGeE7prmrP9FJsoy1LFK
# mFnW+vg43ANhByuAEXq9Cay5o7K2H5NFnR5wj/SLRKwK1iyUX926i1TEviEiAh/P
# VyJbAD4koipig28p/6HDuiYOZ0wUkm/a5W8orIjoOdU3XsJ4i08CfNp5I73CsvB5
# QPYMcLpF9NO/1LvoQAw3UPdL55M5HTCCB3EwggVZoAMCAQICEzMAAAAVxedrngKb
# SZkAAAAAABUwDQYJKoZIhvcNAQELBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQI
# EwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3Nv
# ZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmlj
# YXRlIEF1dGhvcml0eSAyMDEwMB4XDTIxMDkzMDE4MjIyNVoXDTMwMDkzMDE4MzIy
# NVowfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcT
# B1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UE
# AxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQDk4aZM57RyIQt5osvXJHm9DtWC0/3unAcH0qlsTnXI
# yjVX9gF/bErg4r25PhdgM/9cT8dm95VTcVrifkpa/rg2Z4VGIwy1jRPPdzLAEBjo
# YH1qUoNEt6aORmsHFPPFdvWGUNzBRMhxXFExN6AKOG6N7dcP2CZTfDlhAnrEqv1y
# aa8dq6z2Nr41JmTamDu6GnszrYBbfowQHJ1S/rboYiXcag/PXfT+jlPP1uyFVk3v
# 3byNpOORj7I5LFGc6XBpDco2LXCOMcg1KL3jtIckw+DJj361VI/c+gVVmG1oO5pG
# ve2krnopN6zL64NF50ZuyjLVwIYwXE8s4mKyzbnijYjklqwBSru+cakXW2dg3viS
# kR4dPf0gz3N9QZpGdc3EXzTdEonW/aUgfX782Z5F37ZyL9t9X4C626p+Nuw2TPYr
# bqgSUei/BQOj0XOmTTd0lBw0gg/wEPK3Rxjtp+iZfD9M269ewvPV2HM9Q07BMzlM
# jgK8QmguEOqEUUbi0b1qGFphAXPKZ6Je1yh2AuIzGHLXpyDwwvoSCtdjbwzJNmSL
# W6CmgyFdXzB0kZSU2LlQ+QuJYfM2BjUYhEfb3BvR/bLUHMVr9lxSUV0S2yW6r1AF
# emzFER1y7435UsSFF5PAPBXbGjfHCBUYP3irRbb1Hode2o+eFnJpxq57t7c+auIu
# rQIDAQABo4IB3TCCAdkwEgYJKwYBBAGCNxUBBAUCAwEAATAjBgkrBgEEAYI3FQIE
# FgQUKqdS/mTEmr6CkTxGNSnPEP8vBO4wHQYDVR0OBBYEFJ+nFV0AXmJdg/Tl0mWn
# G1M1GelyMFwGA1UdIARVMFMwUQYMKwYBBAGCN0yDfQEBMEEwPwYIKwYBBQUHAgEW
# M2h0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5
# Lmh0bTATBgNVHSUEDDAKBggrBgEFBQcDCDAZBgkrBgEEAYI3FAIEDB4KAFMAdQBi
# AEMAQTALBgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV
# 9lbLj+iiXGJo0T2UkFvXzpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3Js
# Lm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAx
# MC0wNi0yMy5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8v
# d3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2
# LTIzLmNydDANBgkqhkiG9w0BAQsFAAOCAgEAnVV9/Cqt4SwfZwExJFvhnnJL/Klv
# 6lwUtj5OR2R4sQaTlz0xM7U518JxNj/aZGx80HU5bbsPMeTCj/ts0aGUGCLu6WZn
# OlNN3Zi6th542DYunKmCVgADsAW+iehp4LoJ7nvfam++Kctu2D9IdQHZGN5tggz1
# bSNU5HhTdSRXud2f8449xvNo32X2pFaq95W2KFUn0CS9QKC/GbYSEhFdPSfgQJY4
# rPf5KYnDvBewVIVCs/wMnosZiefwC2qBwoEZQhlSdYo2wh3DYXMuLGt7bj8sCXgU
# 6ZGyqVvfSaN0DLzskYDSPeZKPmY7T7uG+jIa2Zb0j/aRAfbOxnT99kxybxCrdTDF
# NLB62FD+CljdQDzHVG2dY3RILLFORy3BFARxv2T5JL5zbcqOCb2zAVdJVGTZc9d/
# HltEAY5aGZFrDZ+kKNxnGSgkujhLmm77IVRrakURR6nxt67I6IleT53S0Ex2tVdU
# CbFpAUR+fKFhbHP+CrvsQWY9af3LwUFJfn6Tvsv4O+S3Fb+0zj6lMVGEvL8CwYKi
# excdFYmNcP7ntdAoGokLjzbaukz5m/8K6TT4JDVnK+ANuOaMmdbhIurwJ0I9JZTm
# dHRbatGePu1+oDEzfbzL6Xu/OHBE0ZDxyKs6ijoIYn/ZcGNTTY3ugm2lBRDBcQZq
# ELQdVTNYs6FwZvKhggNQMIICOAIBATCB+aGB0aSBzjCByzELMAkGA1UEBhMCVVMx
# EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoT
# FU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjElMCMGA1UECxMcTWljcm9zb2Z0IEFtZXJp
# Y2EgT3BlcmF0aW9uczEnMCUGA1UECxMeblNoaWVsZCBUU1MgRVNOOjk2MDAtMDVF
# MC1EOTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNloiMK
# AQEwBwYFKw4DAhoDFQC6PYHRw9+9SH+1pwy6qzVG3k9lbqCBgzCBgKR+MHwxCzAJ
# BgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jv
# c29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMA0GCSqGSIb3DQEBCwUAAgUA7GM4pzAi
# GA8yMDI1MDkwMzIxNTkwM1oYDzIwMjUwOTA0MjE1OTAzWjB3MD0GCisGAQQBhFkK
# BAExLzAtMAoCBQDsYzinAgEAMAoCAQACAiA3AgH/MAcCAQACAhMwMAoCBQDsZIon
# AgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkKAwKgCjAIAgEAAgMHoSCh
# CjAIAgEAAgMBhqAwDQYJKoZIhvcNAQELBQADggEBAAokOE1jwnNouU3utWEaPSZb
# eONfx7MrIUNXqWKWpI3fSk20beUYR5LPfe+KErMuLdP/Vg94XogHnm/GgXdVIjzw
# Q6u76RSchwAsb0a6hUB4Nq3z79AFMtTJj+KceobvbkXR50AAulisKklv31j3Z4hj
# cV/vZvWputPMfgODGkdlMRVCkybALWvj7ZWC17CrsD+rrjgdB+NtQskkqAlq4gwS
# ghOhLPsiJ7R3+NB0EWswUmrjayM7LXFOpRuhuWeT87sxTgB4cXvlt+G+hAt+1jaD
# ER6/5oToCuHm6qv0SZrBRfYoBswHMP2upwIOpA7khCZtmro2vh/bERxxwZBsrAUx
# ggQNMIIECQIBATCBkzB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAA
# AgTY4A4HlzJYmAABAAACBDANBglghkgBZQMEAgEFAKCCAUowGgYJKoZIhvcNAQkD
# MQ0GCyqGSIb3DQEJEAEEMC8GCSqGSIb3DQEJBDEiBCBa+T3j4VsFyiQIy7d6DTMM
# 7R8C0CqMFi3s8kUSqWNUIjCB+gYLKoZIhvcNAQkQAi8xgeowgecwgeQwgb0EIPnt
# eGX9Wwq8VdJM6mjfx1GEJsu7/6kU6l0SS5rcebn+MIGYMIGApH4wfDELMAkGA1UE
# BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAc
# BgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0
# IFRpbWUtU3RhbXAgUENBIDIwMTACEzMAAAIE2OAOB5cyWJgAAQAAAgQwIgQg247L
# pV5yG5Ove8/3tmAoGgRxL1uSR0JUROXKeBmhLjgwDQYJKoZIhvcNAQELBQAEggIA
# RLa/LZeEH3vnX5LVmEEO3SwmHrJybwS16hjbehN8xeipPH8Vg7XYDH7V49ryThHL
# Xck5sK1EXKZkD8qeFlQ/QoqiV+5qRXGx0y182u+2Y/U/966d3SBoiu+mkUalGyrc
# BN7eAqhGH1ub8ZCWBx1hwK4CuLX2go/1HQ/UD023aCXdRYI+1BrIepK7JU3dm6d3
# wFNyIAmcy+0dnOVdGyn7ZrLFXRRtHZai0Bs4ugDD211q6leEB6rKmYwfD7ZlXLRE
# UPdp3ZwIKTTRTEQVyz3xfy7rKCMNtGZKgegShenxn7pUAjAfWltZ2LJc7qdmhNmk
# JapPg2jL8WaW+AkP4BLcRRErLvW/LKIPFLXECpQKxrMQXhjYpzz4phQhuvCA3rcL
# DtccjR3e4L4+T/Za4gwTVfkn8vZAShxYF6cYtv1a+PqZTiZagPIxEybK5qAMD2AF
# 3p3bObI2XKulJIAKJkwoT+1jpE5JVg2Xq9BQt8aa3+ctnYJBl+ZrBYyffGE9NlM0
# YSRquZsvb77F3CvJtbpLFLibHp8a2AgJKjVC5TW5v3qSkFvOD4HUo55j2cG4es5Y
# TB86XNWjNO/CFX1xhjP/7j8KHqst3W4vBbzHcRIOZD+IqhH6lhT3uHbx4etk4Mwi
# d0tERUhhLLG2nvrLvkyCK8LKpxjiZlS+aNtCQAhxMOA=
# SIG # End signature block
