﻿<#
.SYNOPSIS
    
Provides operations and functions for the retrieval of KeyVault Keys and Passwords as
well as some other utilities for manipulating the data that is returned.

.DESCRIPTION

Provides a common module for Unified Operations Setup to retrieve and manipulate Azure Secret/Keys.

.PARAMETER VaultName

The Name of the KeyVault you want to retrieve the key from. Azure specifies that a 
KeyVault name must match this regex "^[a-zA-Z0-9-]{3,24}$".

.PARAMETER ApplicationID

The ApplicationID of the ServicePrincipal we are going to use to authenticate with.

.PARAMETER TenantID

The TenantID guid that Application is under.

.PARAMETER CertificateThumprint

The Thumprint of the certificate that is associated with the Application.

.EXAMPLE

C:\PS> Import-Module ".\KeyVault.psm1" -ArgumentList ("<KeyVaultName>", "<ApplicationID>", "<TenantID>", "<Thumbprint>")

#>
[CmdletBinding()]
param
(
    [Parameter(Mandatory=$False, Position = 0)]
    [System.String]
    $VaultName,

    [Parameter(Mandatory=$False, Position = 1)]
    [System.String]
    $ApplicationID,

    [Parameter(Mandatory=$False, Position = 2)]
    [System.String]
    $CertificateThumprint
)

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking

if (![System.String]::IsNullOrWhiteSpace($VaultName))
{
    $Script:VaultName = $VaultName
}
else
{
    $Script:VaultName = "default"
}

if (![System.String]::IsNullOrWhiteSpace($ApplicationID) -and 
    ![System.String]::IsNullOrWhiteSpace($CertificateThumprint))
{
    $Script:ApplicationID = $ApplicationID
    $Script:CertificateThumprint = $CertificateThumprint 
    Import-Module "$PSScriptRoot\DynamicsVault.psm1" -Force -DisableNameChecking
}

function Get-KeyVaultKey
{
    <#
    .SYNOPSIS
    
    Retrieves a specified Key from a KeyVault.

    .DESCRIPTION

    Retrieves a specified Key from KeyVault using the Azure PowerShell Modules and wraps
    the function up to provide an easier call. 

    .PARAMETER Name

    The Name of the Key in the selected vault we want to retrieve.

    .PARAMETER Version

    The Version of the Key we want, this is an optional field. The default Version
    is the latest one in KeyVault.

    .PARAMETER VaultName

    The Name of the KeyVault you want to retrieve the key from. This parameter is optional,
    it will default to the $Script:VaultName parameter. Azure specifies that a KeyVault name
    must match this regex "^[a-zA-Z0-9-]{3,24}$".

    .EXAMPLE

    Retrieves the latest version of the Key.
    
    C:\PS> Get-KeyVaultKey "KeyName"

    .EXAMPLE
    
    Retrieves the specific version of the Key.

    C:\PS> Get-KeyVaultKey "KeyName" "KeyVersion"

    .EXAMPLE

    Retrieves the latest version of the Key from a different KeyVault than what was passed
    from the script parameters.

    C:\PS> Get-KeyVaultKey -Name "KeyName" -VaultName "VaultName"
    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateLength(1,127)]
        [System.String]
        $Name,

        [Parameter(ParameterSetName = "Version", Mandatory=$False, Position = 1)]
        [System.String]
        $Version,

        [Parameter(Mandatory=$False, Position = 2)]
        [ValidatePattern("^[a-zA-Z0-9-]{3,24}$")]
        [System.String]
        $VaultName = $Script:VaultName
    )


    switch ($PsCmdlet.ParameterSetName)
    {
        "Version" 
        { 
            $keyInformation = Get-AzureKeyVaultKey -VaultName $VaultName -Name $Name -Version $Version
        }
        default
        {
            $keyInformation = Get-AzureKeyVaultKey -VaultName $VaultName -Name $Name
        }
    }
    return $keyInformation
}

function Get-KeyVaultSecret
{
    <#
    .SYNOPSIS
    
    Retrieves a specified secret from a KeyVault.

    .DESCRIPTION

    Retrieves a specified secret from KeyVault using the Azure PowerShell Modules and wraps
    the function up to provide an easier call. 

    .PARAMETER VaultUri

    The Name of the secret in the selected vault we want to retrieve. The assumption is the
    URI is formatted as such "Vault://SecretName/SecretVersion", if it does not match this
    pattern we assume the secret is the VaultUri parameter.

    .PARAMETER VaultName

    The Name of the KeyVault you want to retrieve the secret from. This parameter is optional,
    it will default to the $Script:VaultName parameter. Azure specifies that a KeyVault name
    must match this regex "^[a-zA-Z0-9-]{3,24}$".

    .EXAMPLE

    Retrieves the latest version of the secret.
    
    C:\PS> Get-KeyVaultSecret "VaultUri"

    .EXAMPLE

    Retrieves the latest version of the secret from a different KeyVault than what was passed
    from the script parameters.

    C:\PS> Get-KeyVaultSecret -VaultUri "Vault://SecretName/SecretVersion" -VaultName "VaultName"
    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [System.String]
        $VaultUri,

        [Parameter(Mandatory=$False, Position = 1)]
        [ValidatePattern("^[a-zA-Z0-9-]{3,24}$")]
        [System.String]
        $VaultName = $Script:VaultName
    )

    if (!(Test-ValidKeyVaultUri -VaultUri $VaultUri))
    {
        return $VaultUri
    }
    else
    {
        [System.Uri]$Uri = [System.Uri]::new($VaultUri)

        $secretName = $Uri.Host

        if ($Uri.Segments.Count -ge 2)
        {
            $secret = Get-VaultCredential -ClientId $Script:ApplicationID -CertificateThumbprint $Script:CertificateThumprint -VaultName $VaultName -SecretName $secretName -Version $Uri.Segments[1]
        }
        else
        {
            $secret = Get-VaultCredential -ClientId $Script:ApplicationID -CertificateThumbprint $Script:CertificateThumprint -VaultName $VaultName -SecretName $secretName
        }

        return $secret
    }
}

function Get-CertificatefromBase64String
{
    <#
    .SYNOPSIS

    Takes in a Base64 String representation of a Private Certificate (PFX) and the
    password associated with it and loads the certificate into a X509Certificate2 object.

    .DESCRIPTION

    Upon receiving the certificate information from KeyVault this allows us to convert
    the Base64 string and the password and load a certificate into a X509Certificate2 object.

    .PARAMETER Base64String

    The Base64 string representation of the PFX file.

    .PARAMETER CertificatePassword

    The password of the PFX Certificate.

    .EXAMPLE

    C:\PS> Create-CertificatefromBase64String "<Base64String>" "<Password>"

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $Base64String,

        [Parameter(Mandatory=$False, Position = 1)]
        [System.Security.SecureString]
        $CertificatePassword
    )

    $certificateBytes = [System.Convert]::FromBase64String($Base64String)

    if ($CertificatePassword -ne $null)
    { 
        return [System.Security.Cryptography.X509Certificates.X509Certificate2]::new($CertificateBytes, $CertificatePassword)       
    }
    else
    {
        return [System.Security.Cryptography.X509Certificates.X509Certificate2]::new($CertificateBytes)
    }
}

function Get-CertificateFromJsonString
{
    <#
    .SYNOPSIS

    Takes in a JSON String representation of a Certificate and attributes 
    and loads the certificate in memory.

    .DESCRIPTION

    This functions makes the assumption that the JSON file is formatted like this.

    {
        "data": "<Base64String>",
        "dataType" :"pfx",
        "password": "<CertificatePassword>"
    }

    This is following the same format that the ARM Templates expect Certificates for 
    VM Deployments.

    .PARAMETER JsonString

    The Json String string representation of the Certificate file and its attributes.

    .PARAMETER CertificateType

    This is a reference object so that the caller of the function can determine the
    type of the certificate for example (Public or Private key).

    .EXAMPLE

    C:\PS> [System.String]$certificateType = [System.String]::Empty
    C:\PS> $jsonObject = @"
    C:\PS> {
    C:\PS> "data": "<Base64EncodedCert>",
    C:\PS> "dataType" :"pfx",
    C:\PS> "password": "<CertificatePassword>"
    C:\PS> }
    C:\PS> Get-CertificateFromJsonString $jsonObject ([ref]$certificateType)

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $JsonString,

        [Parameter(Mandatory=$True, Position = 1)]
        [Ref]
        $CertificateType
    )

    $jsonObject = $JsonString | ConvertFrom-Json

    $CertificateType.Value = $jsonObject.dataType
    [System.Security.SecureString]$certificatePassword = $null

    if (![System.String]::IsNullOrWhiteSpace($jsonObject.password))
    {
        $certificatePassword = ConvertTo-SecureString $jsonObject.password -AsPlainText -Force
    }

    $certificate = Get-CertificatefromBase64String -Base64String $jsonObject.data -CertificatePassword $certificatePassword

    return $certificate
}

function Get-CertificateFromVault
{
    <#
    .SYNOPSIS

    Takes in two Vault Uri's. One is for the Certificate file (base64 encoded) and the
    second the password for the Certificate (Optional).

    .DESCRIPTION

    Once the information for the certificate has been retrieved we take the information and 
    create a x509Certificate2 object to be consumed.

    .PARAMETER CertUri

    The Uri of the certificate file we want.

    .PARAMETER CertPasswordUri

    The Uri of the certificate password that corresponds to the Certificate

    .PARAMETER VaultName

    The name of the Vault to retrieve the secret information.

    .EXAMPLE

    # Uses default VaultName
    C:\PS> Get-CertificateFromVault "vault://certSecretName" "vault://certSecretPasswordName"

    # overrides default VaultName
    C:\PS> Get-CertificateFromVault "vault://certSecretName" "vault://certSecretPasswordName" "VaultName"

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $CertUri,

        [Parameter(Mandatory=$false, Position = 1)]
        [System.String]
        $CertPasswordUri,

        [Parameter(Mandatory=$False, Position = 2)]
        [System.String]
        $VaultName = $Script:VaultName
    )

    $base64Cert = Get-KeyVaultSecret -VaultUri $CertUri -VaultName $VaultName

    if (![System.String]::IsNullOrWhiteSpace($CertPasswordUri))
    {
        $certificatePassword = ConvertTo-SecureString -String $(Get-KeyVaultSecret -VaultUri $CertPasswordUri -VaultName $VaultName) -AsPlainText -Force
    }

    return Get-CertificatefromBase64String -Base64String $base64Cert -CertificatePassword $certificatePassword
}

function Get-Base64UTF8String
{
    <#
    .SYNOPSIS

    Takes in a UTF8 Encoded Base64 String and converts it to native string.

    .DESCRIPTION

    This is a helper function to unwrap the Private Key that is stored for the
    ARM Template deployment.

    Azure requires this JSON Structure to be wrapped like this 

    $JsonBlob = @{
        "data": "<Base64String>",
        "dataType" :"pfx",
        "password": "<CertificatePassword>"
    }@

    $ContentBytes = [System.Text.Encoding]::UTF8.GetBytes($JSONBlob)
    $Content = [System.Convert]::ToBase64String($ContentBytes)

    This will unwrap the JSON File.

    .PARAMETER Base64String

    The UTF8 Base64String Object to unwrap.

    .EXAMPLE

    C:\PS> Read-Base64UTF8String <UTF8 Encoded Base64String>

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $Base64String
    )

    $ReceivedBytes = [System.Convert]::FromBase64String($Base64String)

    return [System.Text.Encoding]::UTF8.GetString($ReceivedBytes)
}

function Create-FilefromBase64String
{
    <#
    .SYNOPSIS
    
    Takes in a Base64 String representation of a File and
    Creates a file to the specified path.
    
    .DESCRIPTION

    Takes in a Base64 String representation of a File and
    Creates a file to the specified path.

    .PARAMETER Base64String

    The Base64 string representation of the PFX file. 

    .PARAMETER DestinationPath

    The Path of the file we want to save too. This includes the path and 
    file name i.e. (C:\Temp\Cert.pfx)

    .EXAMPLE

    C:\PS> Create-FilefromBase64String "<Base64String>" "C:\Temp\Cert.pfx"

    #>
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $Base64String,

        [Parameter(Mandatory=$True, Position = 1)]
        [ValidateNotNullOrEmpty()]
        [System.String]
        $DestinationPath
    )
    
    $pfxBytes = [System.Convert]::FromBase64String($Base64String)
    
    [System.IO.File]::WriteAllBytes($DestinationPath, $pfxBytes)
}

function Test-ValidKeyVaultUri
{
    param
    (
        [Parameter(Mandatory=$True, Position = 0)]
        [System.String]
        $VaultUri
    )

    if (![System.Uri]::IsWellFormedUriString($VaultUri, [System.UriKind]::Absolute))
    {
        return $false
    }

    [System.Uri]$Uri = [System.Uri]::new($VaultUri)

    if ($Uri.Scheme -ne "vault")
    {
        return $false
    }

    return $true
}
# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCYIEPfj2Yy48kJ
# gWEKNntz2Qhws7AgUPUErSkZHt+v0qCCDYEwggX/MIID56ADAgECAhMzAAABh3IX
# chVZQMcJAAAAAAGHMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAwMzA0MTgzOTQ3WhcNMjEwMzAzMTgzOTQ3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDOt8kLc7P3T7MKIhouYHewMFmnq8Ayu7FOhZCQabVwBp2VS4WyB2Qe4TQBT8aB
# znANDEPjHKNdPT8Xz5cNali6XHefS8i/WXtF0vSsP8NEv6mBHuA2p1fw2wB/F0dH
# sJ3GfZ5c0sPJjklsiYqPw59xJ54kM91IOgiO2OUzjNAljPibjCWfH7UzQ1TPHc4d
# weils8GEIrbBRb7IWwiObL12jWT4Yh71NQgvJ9Fn6+UhD9x2uk3dLj84vwt1NuFQ
# itKJxIV0fVsRNR3abQVOLqpDugbr0SzNL6o8xzOHL5OXiGGwg6ekiXA1/2XXY7yV
# Fc39tledDtZjSjNbex1zzwSXAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUhov4ZyO96axkJdMjpzu2zVXOJcsw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDU4Mzg1MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAixmy
# S6E6vprWD9KFNIB9G5zyMuIjZAOuUJ1EK/Vlg6Fb3ZHXjjUwATKIcXbFuFC6Wr4K
# NrU4DY/sBVqmab5AC/je3bpUpjtxpEyqUqtPc30wEg/rO9vmKmqKoLPT37svc2NV
# BmGNl+85qO4fV/w7Cx7J0Bbqk19KcRNdjt6eKoTnTPHBHlVHQIHZpMxacbFOAkJr
# qAVkYZdz7ikNXTxV+GRb36tC4ByMNxE2DF7vFdvaiZP0CVZ5ByJ2gAhXMdK9+usx
# zVk913qKde1OAuWdv+rndqkAIm8fUlRnr4saSCg7cIbUwCCf116wUJ7EuJDg0vHe
# yhnCeHnBbyH3RZkHEi2ofmfgnFISJZDdMAeVZGVOh20Jp50XBzqokpPzeZ6zc1/g
# yILNyiVgE+RPkjnUQshd1f1PMgn3tns2Cz7bJiVUaqEO3n9qRFgy5JuLae6UweGf
# AeOo3dgLZxikKzYs3hDMaEtJq8IP71cX7QXe6lnMmXU/Hdfz2p897Zd+kU+vZvKI
# 3cwLfuVQgK2RZ2z+Kc3K3dRPz2rXycK5XCuRZmvGab/WbrZiC7wJQapgBodltMI5
# GMdFrBg9IeF7/rP4EqVQXeKtevTlZXjpuNhhjuR+2DMt/dWufjXpiW91bo3aH6Ea
# jOALXmoxgltCp1K7hrS6gmsvj94cLRf50QQ4U8Qwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAYdyF3IVWUDHCQAAAAABhzAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgvG+WBvkA
# 14iO9myC8yOWc0R8jNV8VZprDIn9UoQxvSAwQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQCtL7jxkNbnbHf18C6mc6qTCgF0krxtuZML+e/LzInI
# QvZozpNIVpKgCFgdoqDtm36u0IDYjUvuhw6DLcqMvCuyeMpdecI9JbmJ5ndyH7+E
# YOSXw/CvUjguNoqn2Encb0NYe3g/KSXvtqo3ibq3pf1fJlzSBBLnKc6jO4YWRP27
# oagb4jnwY3DFWV8a38BCU+UIe7RXaatq7HQNCom1/kMW9Z/edmeo26LzbBzUFWho
# xryd+K3Bpe66ze4/dCfzowfYAV+ge18eO121ET2ddaQnvCzTS1G+SCFV7V4MMlOg
# V5wHQICXv7ImIoim/jvvWddxKmRgp6YAeuptyZ1Qc+s9oYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIPgeuZJVWPbNtXzOhB7BEZiRI5x9xgGQKgfCa7hU
# fG8GAgZfu+uDx+cYEzIwMjAxMjE3MDgwODI2LjkwN1owBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjozMkJELUUzRDUtM0IxRDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABLqjSGQeT9GvoAAAA
# AAEuMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTE5MTIxOTAxMTUwNVoXDTIxMDMxNzAxMTUwNVowgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjozMkJE
# LUUzRDUtM0IxRDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAK7TTKJRU196LFIjMQ9q
# /UjpPhz43m5RnHgHAVp2YGni74+ltsYoO1nZ58rTbJhCQ8GYHy8B4devgbqqYPQN
# U3i+drpEtEcNLbsMr4MEq3SM+vO3a6QMFd1lDRy7IQLPJNLKvcM69Nt7ku1YyM5N
# nPNDcRJsnUb/8Yx/zcW5cWjnoj8s9fQ93BPf/J74qM1ql2CdzQV74PBisMP/tppA
# nSuNwo8I7+uWr6vfpBynSWDvJeMDrcsa62Xsm7DbB1NnSsPGAGt3RzlBV9KViciz
# e4U3fo4chdoB2+QLu17PaEmj07qq700CG5XJkpEYOjedNFiByApF7YRvQrOZQ07Q
# YiMCAwEAAaOCARswggEXMB0GA1UdDgQWBBSGmokmTguJN7uqSTQ1UhLwt1RObDAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQCN4ARqpzCuutNqY2nWJDDXj35iaidl
# gtJ/bspYsAX8atJl19IfUKIzTuuSVU3caXZ6/YvMMYMcbsNa/4J28us23K6PWZAl
# jIj0G8QtwDMlQHjrKnrcr4FBAz6ZqvB6SrN3/Wbb0QSK/OlxsU0mfD7z87R2JM4g
# wKJvH6EILuAEtjwUGSB1NKm3Twrm51fCD0jxvWxzaUS2etvMPrh8DNrrHLJBR3UH
# vg/NXS2IzdQn20xjjsW0BUAiTf+NCRpxUvu/j80Nb1++vnejibfpQJ2IlXiJdIi+
# Hb+OL3XOr8MaDDSYOaRFAIfcoq3VPi4BkvSC8QGrvhjAZafkE7R6L5FJMIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjoz
# MkJELUUzRDUtM0IxRDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUA+1/CN6BILeU1yDGo+b6WkpLoQpuggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AOOFY/AwIhgPMjAyMDEyMTcwODU4MjRaGA8yMDIwMTIxODA4NTgyNFowdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA44Vj8AIBADAKAgEAAgImRQIB/zAHAgEAAgITjDAK
# AgUA44a1cAIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBABphlMcnYjhA0+uG
# PO7KmhT1uAi5KompJhV+/KKDWGCCzpfCfojjhAYDT64pclOFb/DePV0UJgH8qXsj
# Fti6NHTChtutoKedBaQI/O2kHV9Jt8Rw6seSi1bigFXOvFnw01rh2buY/zWmOwOV
# RYH4ALLMOXWYJN9qRQdXeo0LEfNTMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAEuqNIZB5P0a+gAAAAAAS4wDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQg3JixA+L5yDBu355VnJfNHuhgwYSS4wlRB8Ki6CFOEdUwgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCDa/s3O8YhWiqpVN0kTeK+x2m0RAh17JpR6DiFo
# TILJKTCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# LqjSGQeT9GvoAAAAAAEuMCIEICJXWP5hlwCqjWy3iLZHWkOK/zE+IdXHOVC8S86/
# 0+vXMA0GCSqGSIb3DQEBCwUABIIBAFCST16ZjgJ3s//h1aTWQLFhwiZ8wQMU8XYo
# K4k0sqJKINx3fxL3kjNyg079+d2ed9IFxYkRlzrPhk5XMFaVsTOjzyy+WZG8WMTI
# 1kxPW0y9ggQhUa4w6CjRXWRgYtyP6exIcv2/jZ7+8B16bNZok23DVxZ41xLviYL8
# KbDddyFyAWutpikfPMy+e9Mv+MM+w122Zxsl7Y1E6478u7y4oGz0QLJu3AIJsg00
# kUwDbx+zfy1759kyiIpFafMFRJHFuIcHfr5D3T2WxViQnE2W0tJlRk5s0zZ+KoRl
# 6CfNIR4UJboZmpOoJ30T4MftZ8DOVRKYELtysopWmEfs3BnDjtA=
# SIG # End signature block
