PARAM (
    [Parameter(Mandatory=$true)]
    $log,
    [Parameter(Mandatory=$true)]
    [ValidateScript({Test-Path $_})]
    $webroot,
    [Parameter(Mandatory=$true)]
    [ValidateScript({Test-Path $_})]
    $packageDirectory
)

$ErrorActionPreference = 'Stop'

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking
Import-Module "$PSScriptRoot\AosEnvironmentUtilities.psm1" -Force -DisableNameChecking

$WebConfigPath = "$webroot\web.config"
$DbSyncBlastFolder = "$PSScriptRoot\DbSyncBlast"
$DbSyncVersionTestFile = "Microsoft.Dynamics.AX.Data.Management.Sync.dll"

Initialize-Log $log

function Replace-Files([string] $sourceFolder, [string[]] $sourceFiles, [string[]] $destinationFolders, [string] $backupFolderName, [hashtable] $backedUpFiles)
{
    $useBackup = (-not [String]::IsNullOrWhiteSpace($backupFolderName))

    foreach ($directory in $destinationFolders)
    {
        if ($useBackup)
        {
            $backupFolder = Join-Path $directory $backupFolderName

            if (-not (Test-Path $backupFolder))
            {
                Write-Log "Creating folder '$backupFolder'"
                New-Item -ItemType Directory -Path $backupFolder
            }
        }

        foreach ($file in $sourceFiles)
        {
            $sourcePath = Join-Path $sourceFolder $file
            $targetPath = Join-Path $directory $file

            if ($useBackup)
            {
                if (Test-Path $targetPath)
                {
                    $backupPath = Join-Path $backupFolder "$file.bak"

                    Write-Log "Copying file from: '$sourcePath' to '$targetPath', backing up target to '$backupPath'"

                    Copy-Item -Path $targetPath -Destination $backupPath -Force
                    $backedUpFiles[$targetPath] = $backupPath
                }
                else
                {
                    Write-Log "Copying file from: '$sourcePath' to '$targetPath'. Target file does not exist, skipping backup."

                    # Empty value means the file is new - delete/rename when restoring backup
                    $backedUpFiles[$targetPath] = ""
                }
            }
            else
            {
                Write-Log "Copying file from: '$sourcePath' to '$targetPath'. No backup folder specified, skipping backup."
            }

            Copy-Item -Path $sourcePath -Destination $targetPath -Force
                                
            Write-Log "Copy file succeeded."
        }
    }
}

function Restore-Backup([hashtable] $backedUpFiles)
{
    foreach ($file in $backedUpFiles.GetEnumerator())
    {
        try
        {
            if (-not [String]::IsNullOrWhiteSpace($file.Value))
            {
                Write-Log "Restoring backup from '$($file.Value)' to '$($file.Key)'"

                Copy-Item -Path $file.Value -Destination $file.Key -Force
                
                Write-Log "Restored successfully."
            }
            else
            {
                $renameTo = "$($file.Key).dbsyncblast_failed"

                Write-Log "Restoring backup for '$($file.Key)', which is newly created - renaming to '$renameTo' instead of deleting."

                Rename-Item -Path $file.Key -NewName $renameTo

                Write-Log "Renamed successfully."
            }
        }
        catch
        {
            Write-Exception $_
            Write-Log "Failed to restore backup for '$($file.Key)'"
        }
    }
}

function Test-RequiredPaths
{
    foreach ($path in $args)
    {
        if (-not (Test-Path $path))
        {
            Write-Log "Path '$path' not found, skipping DbSyncBlast."
            return $false
        }
    }

    return $true
}

function Get-FileVersion([string] $file)
{
    $fileInfo = Get-ChildItem -File $file

    if ($fileInfo -eq $null -or $fileInfo.VersionInfo -eq $null)
    {
        Write-Log "Could not get version info for '$fileInfo', skipping DbSyncBlast."
        return $null
    }

    return $fileInfo.VersionInfo
}

filter VersionsToDictionary
{
    begin { $dict = @{} }
    process { $dict[[int]($_.Split('.')[-1])] = $_ } # e.g. 7.0.30100.4970 --> $dict[4970] = 7.0.30100.4970
    end { return $dict }
}

$DbSyncBlastMinPlatformVersion = 5030 #PU20
$script:PlatformBuildVersion = 0
function Get-DbSyncApplicableVersionPath
{
    if (-not (Test-RequiredPaths $DbSyncBlastFolder))
    {
        return $null
    }
    
    # Default to no replace
    $dbSyncApplicableVersionPath = $null
            
    $platformVersionInfo = Get-ProductPlatformBuildFullVersion

    if ($platformVersionInfo -ne $null)
    {
        Write-Log "Current platform build: $platformVersionInfo"

        $dbSyncVersions = Get-ChildItem -Path $DbSyncBlastFolder -Directory -Name | VersionsToDictionary
        Write-Log "Found DbSyncBlast builds for platform builds >= $($dbSyncVersions.Keys -join ', ')."
        $invalidDbSyncBlastVersions = $dbSyncVersions.GetEnumerator() | ?{$_.Key -lt $DbSyncBlastMinPlatformVersion}
        if ($invalidDbSyncBlastVersions.Count -gt 0)
        {
            Write-Log "Skipping invalid DbSyncBlast builds: $($invalidDbSyncBlastVersions.Values -join ', '). Minimum supported platform build is $DbSyncBlastMinPlatformVersion."

            $dbSyncVersions = $dbSyncVersions.GetEnumerator() | ?{$_.Key -ge $DbSyncBlastMinPlatformVersion}
        }
        
        $script:PlatformBuildVersion = $platformVersionInfo.Build
        $useDbSyncPlatformVersion = [int](($dbSyncVersions.Keys | ?{$_ -le $script:PlatformBuildVersion} | measure -Max).Maximum)
        if ($useDbSyncPlatformVersion -ne 0)
        {
            Write-Log "Using DbSyncBlast build $($dbSyncVersions[$useDbSyncPlatformVersion])"
            $dbSyncApplicableVersionPath = Join-Path $DbSyncBlastFolder $dbSyncVersions[$useDbSyncPlatformVersion]
        }
        else
        {
            Write-Log "No applicable version found, skipping DbSyncBlast."
        }
    }
    
    return $dbSyncApplicableVersionPath
}

$EnsureMajorVersion = 7
$EnsureMinorVersion = 0
function Assert-IsNewerVersion([string] $newFile, [string] $oldFile)
{
    if (-not (Test-RequiredPaths $newFile $oldFile))
    {
        return $false
    }

    $newFileVersionInfo = Get-FileVersion -File $newFile
    $oldFileVersionInfo = Get-FileVersion -File $oldFile

    if ($newFileVersionInfo -eq $null -or $oldFileVersionInfo -eq $null)
    {
        return $false
    }

    Write-Log "Version of '$oldFile': $($oldFileVersionInfo.FileVersion)"
    Write-Log "Version of '$newFile': $($newFileVersionInfo.FileVersion)"

    if ($newFileVersionInfo.FileMajorPart -ne $EnsureMajorVersion -or $oldFileVersionInfo.FileMajorPart -ne $EnsureMajorVersion `
    -or $newFileVersionInfo.FileMinorPart -ne $EnsureMinorVersion -or $oldFileVersionInfo.FileMinorPart -ne $EnsureMinorVersion)
    {
        # If e.g. major version is changed from 7 to 10, we need to decide what to do. Skipping for now.
        Write-Log "Unexpected major or minor version, expected $EnsureMajorVersion.$EnsureMinorVersion.x.y. Skipping DbSyncBlast."
        return $false
    }

    $versionDiff = $newFileVersionInfo.FileBuildPart - $oldFileVersionInfo.FileBuildPart

    if ($versionDiff -lt 0)
    {
        Write-Log "Selected DbSyncBlast build version is lower than what is installed, skipping DbSyncBlast."
        return $false
    }

    if ($versionDiff -eq 0)
    {
        Write-Log "Selected DbSyncBlast build already installed, skipping DbSyncBlast."
        return $false
    }

    return $true
}

function Update-WebConfig
{
    # Update the Monitoring.ETWManifests key, adding Microsoft.Dynamics.ApplicationPlatform.DatabaseSynchronize.man if it does not exist in the list
    $key = "Monitoring.ETWManifests"
    $manFileName = "Microsoft.Dynamics.ApplicationPlatform.DatabaseSynchronize.man"

    # Check for PU20-21
    if ($script:PlatformBuildVersion -ge 5030 -and $script:PlatformBuildVersion -le 5073) {

        [System.Xml.XmlDocument] $xmlDoc = new-object System.Xml.XmlDocument
        $xmlDoc.Load($WebConfigPath)
        $node = $xmlDoc.SelectSingleNode("//add[@key='$key']")
        $oldValueSplit = $node.Value -split ";"

        if ($oldValueSplit -notcontains $manFileName)
        {
            Write-Log "Updating $key, adding $manFileName"
            $node.Value += ";$manFileName"
            $xmlDoc.Save($WebConfigPath)
        }
        else
        {
            Write-Log "Not updating $key, $manFileName already exists."
        }
    }
    else
    {
        Write-Log "Not updating $key, platform is not PU20 or PU21."
    }
}

try
{
    Write-Log "DbSyncBlast assembly replacement check: Start"

    $error.Clear()

    if (-not (Get-IsFlightEnabled "DbSyncBlastDisabled"))
    {
        $dbSyncApplicableVersionPath = Get-DbSyncApplicableVersionPath

        if ($dbSyncApplicableVersionPath -ne $null)
        {
            if (Assert-IsNewerVersion "$dbSyncApplicableVersionPath\$DbSyncVersionTestFile" "$webroot\bin\$DbSyncVersionTestFile")
            {
                # The list of assemblies to be replaced.
                # Microsoft.Dynamics.ApplicationPlatform.DatabaseSynchronize.Instrumentation.dll should be included in both lists.
                $sourceFiles = Get-ChildItem -Path $dbSyncApplicableVersionPath -File -Name
                $sourceFilesNoMan = $sourceFiles | ?{-not $_.EndsWith(".man")}
                $sourceFilesMan   = $sourceFiles | ?{     $_.EndsWith(".man") -or $_.EndsWith(".Instrumentation.dll")}

                Write-Log "Non-instrumentation files to blast: $sourceFilesNoMan"
                Write-Log "Instrumentation files to blast: $sourceFilesMan"

                # Build a destination directory list
                $ETWManifestPath = "$(Split-Path -parent $PSScriptRoot)\ETWManifest"
                if (-not (Test-Path $ETWManifestPath))
                {
                    Write-Log "Creating folder '$ETWManifestPath'"
                    New-Item -ItemType Directory -Path $ETWManifestPath
                }
                $copyNoManToDirectories = @(
                    "$webroot\bin",
                    "$packageDirectory\ApplicationPlatform\bin",
                    "$packageDirectory\bin")
                $copyManToDirectories = @(
                    "$webroot\Monitoring",
                    "$webroot\bin\Monitoring",
                    "$packageDirectory\Plugins\Monitoring",
                    "$ETWManifestPath\")

                Write-Log "Blasting non-instrumentation files to these locations: $copyNoManToDirectories"
                Write-Log "Blasting instrumentation files to these locations: $copyManToDirectories"

                # Backup to Original the first time, then to Last on subsequent runs
                $originalBackupFolder = "DbSyncOriginalBackup"
                $lastBackupFolder = "DbSyncLastBackup"
                Write-Log "Default backup folder is $originalBackupFolder"
                $backupFolder = $originalBackupFolder
                foreach ($directory in $copyNoManToDirectories)
                {
                    $originalBackupPath = Join-Path $directory $originalBackupFolder
                    if (Test-Path $originalBackupPath)
                    {
                        Write-Log "Found '$originalBackupPath', changing backup folder to $lastBackupFolder"
                        $backupFolder = $lastBackupFolder
                        break
                    }
                }
                
                $backedUpFiles = @{}
                try
                {
                    Replace-Files $dbSyncApplicableVersionPath $sourceFilesNoMan $copyNoManToDirectories $backupFolder $backedUpFiles
                    Replace-Files $dbSyncApplicableVersionPath $sourceFilesMan   $copyManToDirectories
                }
                catch
                {
                    # We do not want to leave mixed versions in case blast fails, so we do our best to return to the original state.
                    # However, this should not happen as we normally blast to a staging folder. Instead, the backed up files should
                    # be used if DbSync itself fails because of a regression.
                    Write-Exception $_
                    Restore-Backup $backedUpFiles
                    throw
                }

                Update-WebConfig

                Write-Log "DbSyncBlast assembly replacement succeeded."
            }
        }
    }
    else 
    {
       Write-Log "Skipping DbSyncBlast because of DbSyncBlastDisabled flight."
    } 
}
catch
{
    # Write the exception and that this process failed.  
    # Do not throw as this should not cause the deployment or update to fail.
    Write-Exception $_
    Write-Log "DbSyncBlast assembly replacement failed, see $log for details."
}
finally
{
    Write-Log "DbSyncBlast assembly replacement check: Stop"
}

# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCA9lsBXtv4PSpQ8
# U3CeeKm0RJfsYRAgHAy25FBy3TGUAqCCDYEwggX/MIID56ADAgECAhMzAAABh3IX
# chVZQMcJAAAAAAGHMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAwMzA0MTgzOTQ3WhcNMjEwMzAzMTgzOTQ3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDOt8kLc7P3T7MKIhouYHewMFmnq8Ayu7FOhZCQabVwBp2VS4WyB2Qe4TQBT8aB
# znANDEPjHKNdPT8Xz5cNali6XHefS8i/WXtF0vSsP8NEv6mBHuA2p1fw2wB/F0dH
# sJ3GfZ5c0sPJjklsiYqPw59xJ54kM91IOgiO2OUzjNAljPibjCWfH7UzQ1TPHc4d
# weils8GEIrbBRb7IWwiObL12jWT4Yh71NQgvJ9Fn6+UhD9x2uk3dLj84vwt1NuFQ
# itKJxIV0fVsRNR3abQVOLqpDugbr0SzNL6o8xzOHL5OXiGGwg6ekiXA1/2XXY7yV
# Fc39tledDtZjSjNbex1zzwSXAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUhov4ZyO96axkJdMjpzu2zVXOJcsw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDU4Mzg1MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAixmy
# S6E6vprWD9KFNIB9G5zyMuIjZAOuUJ1EK/Vlg6Fb3ZHXjjUwATKIcXbFuFC6Wr4K
# NrU4DY/sBVqmab5AC/je3bpUpjtxpEyqUqtPc30wEg/rO9vmKmqKoLPT37svc2NV
# BmGNl+85qO4fV/w7Cx7J0Bbqk19KcRNdjt6eKoTnTPHBHlVHQIHZpMxacbFOAkJr
# qAVkYZdz7ikNXTxV+GRb36tC4ByMNxE2DF7vFdvaiZP0CVZ5ByJ2gAhXMdK9+usx
# zVk913qKde1OAuWdv+rndqkAIm8fUlRnr4saSCg7cIbUwCCf116wUJ7EuJDg0vHe
# yhnCeHnBbyH3RZkHEi2ofmfgnFISJZDdMAeVZGVOh20Jp50XBzqokpPzeZ6zc1/g
# yILNyiVgE+RPkjnUQshd1f1PMgn3tns2Cz7bJiVUaqEO3n9qRFgy5JuLae6UweGf
# AeOo3dgLZxikKzYs3hDMaEtJq8IP71cX7QXe6lnMmXU/Hdfz2p897Zd+kU+vZvKI
# 3cwLfuVQgK2RZ2z+Kc3K3dRPz2rXycK5XCuRZmvGab/WbrZiC7wJQapgBodltMI5
# GMdFrBg9IeF7/rP4EqVQXeKtevTlZXjpuNhhjuR+2DMt/dWufjXpiW91bo3aH6Ea
# jOALXmoxgltCp1K7hrS6gmsvj94cLRf50QQ4U8Qwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAYdyF3IVWUDHCQAAAAABhzAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQglXrxCjhO
# nST/MMcB5SiCUk3JDiMxufgocBDpDFJUjl0wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQA40uH0L9X81Znh5G+fzxwvQB3K4BozMefcqLoYO2Pw
# ghhfF/dLqUb3+vpbKsgIYj2+LbYzuJDJHqkDW7yOqxR9hLfNrq5iPZyE6fpqCXo0
# IiPvlK8FL5aKmlWxQ1kS4l/Xnn4ShmraAENM1lzefCzsZWEMVVMHFQvsx1cj+l2u
# yZ4zqvyGGmTlQ35JxZgrh0rQCtTdUCYO0rJgYJUoAK+hJ7waioSEM8W0iIpGkftV
# 60nvmucHOYYQgrrLwAX6XYFQEYk38AgW2qEFm0bgBAb7lknH1cR86V4x52preUNs
# jYfAGiIj2eze4CGArKBosW5JjvsXdZMs24cgtpN6blvGoYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIDOFE+wVydB5ykSFc+RdYn+WOOT1JqbCdktzLfQo
# ZtpqAgZftBALcZcYEzIwMjAxMTIwMTkxMzE4LjY1N1owBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjpEOURFLUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABLS5NQcpjZTOgAAAA
# AAEtMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTE5MTIxOTAxMTUwNFoXDTIxMDMxNzAxMTUwNFowgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpEOURF
# LUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKlhjfR1STqYRTS3s0i4
# jIcSMV+G4N0oYgwlQK+pl4DVMFmr1iTZHFLj3Tt7V6F+M/BXx0h9i0uu1yBnqCqN
# OkuJERTbVnM4u3JvRxzsQfCjBfqD/CNwoMNekoylIBzxP50Skjp1pPsnQBKHaCP8
# tguvYVzoTQ54q2VpYEP/+OYTQeEPqWFi8WggvsckuercUGkhYWM8DV/4JU7N/rbD
# rtamYbe8LtkViTQYbigUSCAor9DhtAZvq8A0A73XFH2df2wDlLtAnKCcsVvXSmZ3
# 5bAqneN4uEQVy8NQdReGI1tI6UxoC7XnjGvK4McDdKhavNJ7DAnSP5+G/DTkdWD+
# lN8CAwEAAaOCARswggEXMB0GA1UdDgQWBBTZbGR8QgEh+E4Oiv8vQ7408p2GzTAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQB9awNk906recBuoO7Ezq7B8UGu9EoF
# XiL8ac0bbsZDBY9z/3p8atVZRCxHN43a3WGbCMZoKYxSBH6UCkcDcwXIfNKEbVMz
# nF1mjpQEGbqhR+rPNqHXZotSV+vn85AxmefAM3bcLt+WNBpEuOZZ4kPZVcFtMo4Y
# yQjxoNRPiwmp+B0HkhQs/l/VIg0XJY6k5FRKE/JFEcVY4256NdqUZ+3jou3b4OAk
# tE2urr4V6VRw1fffOlxZb8MyvE5mqvTVJOStVxCuhuqg1rIe8la1gZ5iiuIyWeft
# ONfMw0nSZchGLigDeInw6XfwwgFnC5Ql8Pbf2jOxCUluAYbzykI+MnBiMIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpE
# OURFLUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUAn85fx36He7F0vgmyUlz2w82l0LGggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AONiguYwIhgPMjAyMDExMjAyMjAxMTBaGA8yMDIwMTEyMTIyMDExMFowdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA42KC5gIBADAKAgEAAgIOywIB/zAHAgEAAgIQxjAK
# AgUA42PUZgIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBALUS5g3+vNnizkfz
# 87KNBQ3cgKZSInKBhigEQ9NJjGCue+ff3fBmk+XP380Dte5uJHT5wsVKf84mwlAE
# cHlQuFYNXFv3KUKFa0dvYZzaXD5IYXfEikSEgJoUjmZ1789qYZbbJRjCf44ohJp8
# sUGPXOXNXlMnBHEK7EkeCpPeWZuHMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAEtLk1BymNlM6AAAAAAAS0wDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQghRP0dnXLpc3Qae6zk7pTtoTG6JJSe4pK8lw3JGI23vQwgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCCO8Vpycn0gB4/ilRAPPDbS+Cmbqj/uC011moc5
# oeGDwTCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# LS5NQcpjZTOgAAAAAAEtMCIEIIKTjd4M3WE/bYdlbVNax+eq8W7OipvX/UQPPa+N
# d0ZbMA0GCSqGSIb3DQEBCwUABIIBAGJZJ5jhlRqz/e6qe9lrIMSDQ1BsLQNR7ykJ
# qEKbn/EAFqCeYoFLpUsrhjGUwlD0PZfShd3L/mccdblWh3rjA5/oieywA0Vwzimt
# Q30kZDoOAXj+IF4fNf5NofowHOAZMwAncYqXR3LJkAadOmInzAl+jl6pzCkLfSt+
# MDGdMxYy0UkPwPQmykj/Y30gSgR3dx1fBrqFsKKO3b7Qdr7Tj9n4w6E8jF+n01g2
# pFi0dJbOphJQLYi7LIZDli4JIx5WTrhqAjyFGnIUUtwOR3gGoGoqG7dsXVcyo3xz
# z2VmI+cHcN/ZVV+00DfaBTp88kI4c5z97q+cHSIDU78MLC3HNAQ=
# SIG # End signature block
