﻿function Add-Cert($pfx, $credentialXml)
{
	Write-Output "Start processing certificate '$pfx.Name'." 
    $certInstance = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2

	# Get the password for certificate.
	$pathNode = $credentialXml.Credentials.Certificates.Certificate | where { $_.Path -eq "Deployment\Onebox\"+$pfx.Name}
	if($pathNode.Count -gt 0)
	{
		$password = $pathNode[0].Password
	}
	else
	{
		$password = $pathNode.Password
	}
 
	if($password -ne $null -and [string]::IsNullOrEmpty($password) -eq $false )
	{
		$certInstance.import($pfx.FullName, $password, "Exportable,PersistKeySet,MachineKeySet"); 
		$certTP = $certInstance.Thumbprint;
        Add-CertToStore -certTP:$certTP -certInstance:$certInstance -pfx:$pfx
	}
}

function Add-CertToStore([string]$certTP,$certInstance,$pfx)
{
    $store = Get-Item Cert:\LocalMachine\My
    $store.Open("ReadWrite")

    $trustedStore = Get-Item Cert:\LocalMachine\Root
    $trustedStore.Open("ReadWrite")
    
    # add certificate to trusted root store if not present
	$trustedRootCert = Get-ChildItem "Cert:\LocalMachine\Root" | where {$_.Thumbprint -eq "$certTP"}
	if($trustedRootCert -eq $null -or [string]::IsNullOrEmpty($trustedRootCert) -eq $true)
	{
		Write-output "Adding certificate $pfx.Name to the machine trusted root store."
		$trustedStore.Add($certInstance)
	}

    # add certificate to local machine store if not present
	$localStoreCert = Get-ChildItem "Cert:\LocalMachine\My" |where {$_.Thumbprint -eq "$certTP"}
	if ($localStoreCert -eq $null -or $localStoreCert -eq "") 
	{
		Write-output "Adding certificate $pfx.Name to the local machine store."
		$store.Add($certInstance)

        # Grant permissin to Network Service to the private key.
		$sslCertPrivKey = $certInstance.PrivateKey.CspKeyContainerInfo.UniqueKeyContainerName
		$privateKeyCertFile = Get-Item -path "$ENV:ProgramData\Microsoft\Crypto\RSA\MachineKeys\*"  | where {$_.Name -eq $sslCertPrivKey} 
		$privateKeyAcl = (Get-Item -Path $privateKeyCertFile.FullName).GetAccessControl("Access") 
		$permission = "NT AUTHORITY\NETWORK SERVICE","Read","Allow" 
		$accessRule = new-object System.Security.AccessControl.FileSystemAccessRule $permission 
		$privateKeyAcl.AddAccessRule($accessRule) 
		Set-Acl $privateKeyCertFile.FullName $privateKeyAcl
		Write-output "Network Service account is granted access to the private key of certificate $pfx.Name"

		# Configure the SSL certificate "star1.cloud.onebox.dynamics.com.pfx" to the port 443
		try
		{
			if($pfx.Name -eq "star1.cloud.onebox.dynamics.com.pfx")
			{
				get-item cert:\LocalMachine\My\$certTP | new-item IIS:\SslBindings\0.0.0.0!443 -ErrorAction SilentlyContinue
			}
		}
		catch 
		{
			Write-output "SSL is configured for the AOS website."
			$_.Exception.Message    
		}
    }

    $store.Close()
    $trustedStore.Close()
}

function Remove-CertFromStore($certInstance)
{
    $store = Get-Item Cert:\LocalMachine\My
    $store.Open("ReadWrite")

    $trustedStore = Get-Item Cert:\LocalMachine\Root
    $trustedStore.Open("ReadWrite")

    # remove certificate from the trusted root store if present
	$trustedRootCert = Get-ChildItem "Cert:\LocalMachine\Root" | where {$_.Thumbprint -eq "$certTP"}
	if($trustedRootCert -ne $null -or [string]::IsNullOrEmpty($trustedRootCert) -eq $false)
	{
		Write-output "Removing certificate $pfx.Name from the machine trusted root store."
		$trustedStore.Remove($certInstance)
	}

    # remove certificate from the local machine store if present
	$localStoreCert = Get-ChildItem "Cert:\LocalMachine\My" |where {$_.Thumbprint -eq "$certTP"}
	if ($localStoreCert -ne $null -or [string]::IsNullOrEmpty($localStoreCert) -eq $false) 
	{
		Write-output "Removing certificate $pfx.Name from the local machine store."
		$store.Remove($certInstance)
    }

    $store.Close()
    $trustedStore.Close()
}

function Remove-Cert($pfx, $credentialXml)
{
    $certInstance = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2

    # Get the password for certificate.
    $pathNode = $credentialXml.Credentials.Certificates.Certificate | where { $_.Path -eq "Deployment\Onebox\"+$pfx.Name}
    
    if($pathNode.Count -gt 0)
    {
        $password = $pathNode[0].Password
    }
    else
    {
        $password = $pathNode.Password
    }
 
    if($password -ne $null -and [string]::IsNullOrEmpty($password) -eq $false )
    {
       $certInstance.import($pfx.FullName, $password, "Exportable,PersistKeySet,MachineKeySet"); 
       $certTP = $certInstance.Thumbprint;
       
       Remove-CertFromStore -certInstance:$certInstance
    }
}

function AddOrRemoveWebsiteCerts([string] $rootPath, [bool] $remove)
{
	# Add all certificates to store.
	$credentialXml = [xml](Get-Content "$rootPath\DObind\Deployment\Onebox\Credentials.xml")

	# Get all the available certificate files.
	$allPFX = Get-ChildItem "$rootPath\DObind\Deployment\Onebox\*.pfx"

	# Loop through all each certificate and install it if not installed.
	foreach($pfx in $allPFX)
	{
		if($remove)
		{
			Remove-Cert $pfx $credentialXml
		}
		else
		{
			Add-Cert $pfx $credentialXml
		}
	}
}

function Add-WebsiteCerts([string] $rootPath)
{
	AddOrRemoveWebsiteCerts $rootPath $false
}

function Remove-WebsiteCerts([string] $rootPath)
{
	AddOrRemoveWebsiteCerts $rootPath $true
}

function Remove-ExistingAppPools([string] $appPoolName)
{
	$machineName = $env:COMPUTERNAME;
    $serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)
    $existingAppPool = $serverManager.ApplicationPools | Where-Object{$_.Name -eq "$appPoolName"}
    if($existingAppPool -ne $null)
    {
        Write-Output "Removing app pool '$appPoolName'."
        $serverManager.ApplicationPools.Remove($existingAppPool)
		$serverManager.CommitChanges()
    }
}

function Remove-ExistingWebsites([string] $siteName, $port, $hostHeaderStrs)
{
	$machineName = $env:COMPUTERNAME;
	$serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)
    $existingSite = $serverManager.Sites | Where-Object{$_.Name -eq "$siteName"}
    if($existingSite -ne $null)
    {
        Write-Output "Removing website '$siteName'."
        $serverManager.Sites.Remove($existingSite)
		$serverManager.CommitChanges()
    }
 
    #remove anything already using that port
    [Microsoft.Web.Administration.Site]$site
    foreach($site in $serverManager.Sites) 
    {
        foreach($hostHeader in $hostHeaderStrs)
		{
            foreach($binding in $site.Bindings)
            {
                if($binding.EndPoint.Port -eq $port -and $binding.EndPoint.Address -eq $hostHeader)
                {
                    Write-Warning "Warning: Found an existing site '$($site.Name)' already using port $port; removing it."
				    $serverManager.Sites.Remove($site)
					$serverManager.CommitChanges()
				    Write-Output "Website $($site.Name) removed."
                }
            }
        }
    }
}

function New-AppPool([string] $appPoolName, [string] $user, [string] $password, [int] $pingTimeoutSeconds)
{
	$machineName = $env:COMPUTERNAME;
	$serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)

    Write-output "Creating an app pool named '$appPoolName' under v4.0 runtime, default (Integrated) pipeline."
    $pool = $serverManager.ApplicationPools.Add($appPoolName);
    $pool.managedRuntimeVersion = "v4.0"
    $pool.processModel.identityType = [Microsoft.Web.Administration.ProcessModelIdentityType]::NetworkService
	$pool.processModel.PingResponseTime = [System.TimeSpan]::FromSeconds($pingTimeoutSeconds)
	$pool.Failure.RapidFailProtection =$false
	
	if ($user -ne $null -AND $password -ne $null -AND $user -ne "" -AND $password -ne "") {
	    Write-Output "Setting AppPool to run as user '$user'."
		$pool.processmodel.identityType = [Microsoft.Web.Administration.ProcessModelIdentityType]::SpecificUser
		$pool.processmodel.username = $user
		$pool.processmodel.password = $password
	} 
	
    $serverManager.CommitChanges()

	for($tryCount= 30; $tryCount -gt 0; --$tryCount)
	{
		if ($serverManager.ApplicationPools[$appPoolName].State -eq [Microsoft.Web.Administration.ObjectState]::Started)
		{
			Write-Output "The AppPool was created and started successfully."
			return
		}

		Start-Sleep -Milliseconds 100
	}

	throw "App pool '$appPoolName' was created but did not start automatically."
}

function New-Website([string] $siteName, [string] $aosSitePath, [string] $port, $hostHeaderStrs,$cert, [string] $appPoolName)
{
	$machineName = $env:COMPUTERNAME
	$serverManager = [Microsoft.Web.Administration.ServerManager]::OpenRemote($machineName)
	$bindingArguments = "*:443:"+$hostHeaderStrs[0]
	Write-Output "Creating website. Site name: '$siteName'; Bindings: '$bindingArguments'; Website root: '$aosSitePath'."
	$website = $serverManager.Sites.Add($siteName, "https", $bindingArguments, $aosSitePath)
	for($i = 1; $i -lt $hostHeaderStrs.Length ; $i++)
	{
		$bindingArguments = "*:443:"+$hostHeaderStrs[$i]
		Write-Output "Adding binding '$bindingArguments' to the AOS website."
        $website.Bindings.Add($bindingArguments, $cert.GetCertHash(), "My")
	}

	Write-Output "Setting the website app pool to '$appPoolName'."
	# Set the app pool for the website
	$website.Applications[0].ApplicationPoolName = $appPoolName
    $serverManager.CommitChanges()

    
    Set-WebConfigurationProperty -pspath "MACHINE/WEBROOT/APPHOST/$siteName/Apps"  -filter "system.webServer/security/requestFiltering/fileExtensions/add[@fileExtension='.config']" -name "allowed" -value "True"


	for($tryCount= 100; $tryCount -gt 0; --$tryCount)
	{
		if ($serverManager.Sites[$siteName].State -eq [Microsoft.Web.Administration.ObjectState]::Started)
		{
			Write-Output "AOS website was created and started successfully."
			return
		}

		Start-Sleep -Milliseconds 100
	}
	
	throw "The AOS website '$siteName' was created but did not start after waiting for 10 seconds. Check the event log for further details."
}

function New-AosWebsite([string] $rootPath, [string] $siteName, [string] $aosSitePath, [string] $appPoolName, [string] $port, $hostHeaderStrs, [string] $user, [string] $userPassword, [string] $certTP, [int] $pingTimeoutSeconds = 600)
{
	# start IIS.
	Invoke-Command -script {iisreset.exe /start}

    # Cleanup old websites and app pools
	Remove-ExistingWebsites $siteName $appPoolName $port
	Remove-ExistingAppPools $appPoolName

	# Add certs
	Add-WebsiteCerts $rootPath

	# Create new app Pool
	New-AppPool $appPoolName $user $userPassword $pingTimeoutSeconds
	
	# Create the new site
	$cert = Get-ChildItem "Cert:\LocalMachine\My" |where {$_.Thumbprint -eq "$certTP"}
	New-Website $siteName $aosSitePath $port $hostHeaderStrs $cert $appPoolName
}

function Remove-Website([string] $rootPath, [string] $websiteName, [string] $appPoolName)
{
	# Cleanup certs
	Write-Output "Removing certificates."
	Remove-WebsiteCerts $rootPath

	# Cleanup old websites and app pools
	Write-Output "Remvoing website '$websiteName'."
	Remove-ExistingWebsites $websiteName $appPoolName 0

	Write-Output "Remvoing App pool '$appPoolName'."
	Remove-ExistingAppPools $appPoolName
}

function Delete-Service([string]$serviceName)
{
    try
    {
        $service = Get-WmiObject -Class Win32_Service -Filter "Name='$serviceName'" -ErrorAction SilentlyContinue
	    if($service -ne $null)
	    {
		    # stop and delete the batch Service
            Write-output "Stopping service '$serviceName'."
		    Stop-Service "$serviceName" -ErrorAction SilentlyContinue -Force
            $service = Get-Service $serviceName
            while($service -ne $null -and $service.Status -ne "Stopped")
            {
                $service = Get-Service $serviceName
                Write-output "Sleeping for 5 minutes while waiting for the service '$serviceName' to stop."
                Start-Sleep 5
            }

            Write-output "Deleting service '$serviceName'."
		    sc.exe delete "$serviceName" 
	    }
    }
    catch
    {
    }
}

function Terminate-process
{
    # kill process
	$processes = @("batch","batch.exe","mmc")
	foreach($process in $processes)
	{
		$handle = get-process "$process" -ErrorAction SilentlyContinue
		if($handle -ne $null)
		{
			try
			{
                # special case for Event viewer
                if($process -eq "mmc" -and $handle.MainWindowTitle -eq "Event Viewer")
                {
                    $procName = $handle.MainWindowTitle
                    Write-output "Terminating process '$procName'."
				    $handle.kill()
                }
                else
                {
                    Write-output "Terminating process '$handle'."
                    $handle.Kill()
                }
			}
			catch
			{
			}
		}
	}

    # stop and delete Dynamics Rainier services
    Delete-Service -serviceName:"DynamicsAxBatch"
}

function Register-RestartTask([string]$deploymentDir,[string]$logDir,[bool]$deployChocolateyPackage,[string]$packageDir)
{
    Write-Output "Registering a scheduled task to auto start Azure storage emulator upon machine restart."
    $localMachineName = $env:COMPUTERNAME
    $batchFilename = [IO.Path]::Combine("$deploymentDir","AOSDeploy.cmd")
    Write-Output "Creating scheduler task batch file '$batchFilename'."

    "$deploymentDir\DoBind\aosdeploy.exe /action:Restart /rd:`"$deploymentDir`" /ld:`"$logDir`" /td:`"$packageDir`" /chocopkg:$deployChocolateyPackage"|Set-Content $batchFilename

    $taskName = "DynamicsRainierRestartTask"
    & schtasks /create /ru SYSTEM /sc ONSTART /tn $taskName /tr "cmd /c $batchFilename" /F /RL HIGHEST
    if (!$?)
    {
        throw "There was an error creating the task $taskName."
    }
}

function Delete-RestartTask
{
    $taskName = "DynamicsRainierRestartTask"
    Write-Output "Deleting scheduled task '$taskName'."
    $SchTsk = Get-ScheduledTask | Select Name, State | ? {$_.Name -eq $taskName}
    if($SchTsk -ne $null)
    {
        & schtasks /delete /tn $taskName /F
    }
}

function Configure-ProductConfiguration([string]$sitename)
{
    Write-output "Starting the configuration of the production configuration application."
    $aossite=Get-Website -Name $sitename
    if($aossite -eq $null){
        $message="The website '$sitename' does not exist."
        Write-Output $message
        throw $message
    }

    $webroot=$aossite.physicalPath
    $productconfigurationdir=join-path $webRoot "productconfiguration"

    if(!(Test-Path "$productconfigurationdir")){
        Write-Output "Skipping product configuration as the product configuration directory does not exist under the aos web root $webroot."
        return
    }

    $pcapppool="productconfiguration"
    Write-output "Removing web app pool $pcapppool."
    Remove-WebAppPool -Name:$pcapppool -ErrorAction SilentlyContinue
    write-output "Creating web app pool $pcapppool."
    $newapppool=New-WebAppPool -Name:$pcapppool -ErrorAction SilentlyContinue
    write-output "Setting the identity for the web app pool '$pcapppool' to NetworkService."
    $newapppool.processModel.identityType="NetworkService"
    $newapppool|Set-Item
    write-output "Starting web app pool $pcapppool."
    Start-WebAppPool -Name:$pcapppool

    Write-Output "Deleting web application 'ProductConfiguration' under web site '$sitename'."
    Remove-WebApplication -Site:$aossite.Name -Name:"ProductConfiguration" -ErrorAction SilentlyContinue
    Write-Output "Creating new web application 'ProductConfiguration' under web site '$sitename' with application pool `"$pcapppool`"."
    New-WebApplication -Site:$aossite.Name -Name:"ProductConfiguration" -PhysicalPath:$productconfigurationdir -ApplicationPool:$pcapppool -Force 
    Write-Output "Product configuration application successfully configured."
}

Export-ModuleMember -Function New-AosWebsite,Remove-Website,Terminate-process,Register-RestartTask,Delete-RestartTask,Delete-Service,Configure-ProductConfiguration,Add-WebsiteCerts,Remove-WebsiteCerts
# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCD4vYul+/4MT2Y/
# shNR1m8EcMqg8PK/RZCg1SpB/rUJ2qCCDYEwggX/MIID56ADAgECAhMzAAABh3IX
# chVZQMcJAAAAAAGHMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAwMzA0MTgzOTQ3WhcNMjEwMzAzMTgzOTQ3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDOt8kLc7P3T7MKIhouYHewMFmnq8Ayu7FOhZCQabVwBp2VS4WyB2Qe4TQBT8aB
# znANDEPjHKNdPT8Xz5cNali6XHefS8i/WXtF0vSsP8NEv6mBHuA2p1fw2wB/F0dH
# sJ3GfZ5c0sPJjklsiYqPw59xJ54kM91IOgiO2OUzjNAljPibjCWfH7UzQ1TPHc4d
# weils8GEIrbBRb7IWwiObL12jWT4Yh71NQgvJ9Fn6+UhD9x2uk3dLj84vwt1NuFQ
# itKJxIV0fVsRNR3abQVOLqpDugbr0SzNL6o8xzOHL5OXiGGwg6ekiXA1/2XXY7yV
# Fc39tledDtZjSjNbex1zzwSXAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUhov4ZyO96axkJdMjpzu2zVXOJcsw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDU4Mzg1MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAixmy
# S6E6vprWD9KFNIB9G5zyMuIjZAOuUJ1EK/Vlg6Fb3ZHXjjUwATKIcXbFuFC6Wr4K
# NrU4DY/sBVqmab5AC/je3bpUpjtxpEyqUqtPc30wEg/rO9vmKmqKoLPT37svc2NV
# BmGNl+85qO4fV/w7Cx7J0Bbqk19KcRNdjt6eKoTnTPHBHlVHQIHZpMxacbFOAkJr
# qAVkYZdz7ikNXTxV+GRb36tC4ByMNxE2DF7vFdvaiZP0CVZ5ByJ2gAhXMdK9+usx
# zVk913qKde1OAuWdv+rndqkAIm8fUlRnr4saSCg7cIbUwCCf116wUJ7EuJDg0vHe
# yhnCeHnBbyH3RZkHEi2ofmfgnFISJZDdMAeVZGVOh20Jp50XBzqokpPzeZ6zc1/g
# yILNyiVgE+RPkjnUQshd1f1PMgn3tns2Cz7bJiVUaqEO3n9qRFgy5JuLae6UweGf
# AeOo3dgLZxikKzYs3hDMaEtJq8IP71cX7QXe6lnMmXU/Hdfz2p897Zd+kU+vZvKI
# 3cwLfuVQgK2RZ2z+Kc3K3dRPz2rXycK5XCuRZmvGab/WbrZiC7wJQapgBodltMI5
# GMdFrBg9IeF7/rP4EqVQXeKtevTlZXjpuNhhjuR+2DMt/dWufjXpiW91bo3aH6Ea
# jOALXmoxgltCp1K7hrS6gmsvj94cLRf50QQ4U8Qwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAYdyF3IVWUDHCQAAAAABhzAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQg20HWb08c
# aASlXYRTyuQlB+EYvrqhgLAuc/2C1QpXH2EwQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQCamBAPVKphD4HqVv0DfAPIf2tg5ikRNejPbE+NClXw
# t1FhVZaBzgQRThm1g0l+IyNx4JkjdqmF4QUb4EvkLQuWlZMO0g8P1CUceis5IAvy
# lD0Hpd2bWSxL9cq5tDbKH3ZGgkHog6e2Gv1hF56ECyOpGiyoAwZx8fdzWdrONQUx
# t45wj2nLKVpqIurSJUeBqOd4G1hcvy65VTWoraxGbr/axMIRfL0jLO2s7jMUOjBv
# qKpaNVvX95a9C+HZCt5sEBMpFmKxDCqJEOacfFRKaq4/dK9kQdtu8EwgN8ih0Yim
# 0I/YEgg2x3uv/juHUYz56AuyUBOOQv1Ixc4d0opTHjn6oYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIGTAqbp47oJ+Qy8QZLS+MR+4UGzpRPmPe2PPrYjX
# maWyAgZfu+oE2hwYEzIwMjAxMjE3MDgwNjI0Ljc3NlowBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjpEOURFLUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABLS5NQcpjZTOgAAAA
# AAEtMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTE5MTIxOTAxMTUwNFoXDTIxMDMxNzAxMTUwNFowgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpEOURF
# LUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKlhjfR1STqYRTS3s0i4
# jIcSMV+G4N0oYgwlQK+pl4DVMFmr1iTZHFLj3Tt7V6F+M/BXx0h9i0uu1yBnqCqN
# OkuJERTbVnM4u3JvRxzsQfCjBfqD/CNwoMNekoylIBzxP50Skjp1pPsnQBKHaCP8
# tguvYVzoTQ54q2VpYEP/+OYTQeEPqWFi8WggvsckuercUGkhYWM8DV/4JU7N/rbD
# rtamYbe8LtkViTQYbigUSCAor9DhtAZvq8A0A73XFH2df2wDlLtAnKCcsVvXSmZ3
# 5bAqneN4uEQVy8NQdReGI1tI6UxoC7XnjGvK4McDdKhavNJ7DAnSP5+G/DTkdWD+
# lN8CAwEAAaOCARswggEXMB0GA1UdDgQWBBTZbGR8QgEh+E4Oiv8vQ7408p2GzTAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQB9awNk906recBuoO7Ezq7B8UGu9EoF
# XiL8ac0bbsZDBY9z/3p8atVZRCxHN43a3WGbCMZoKYxSBH6UCkcDcwXIfNKEbVMz
# nF1mjpQEGbqhR+rPNqHXZotSV+vn85AxmefAM3bcLt+WNBpEuOZZ4kPZVcFtMo4Y
# yQjxoNRPiwmp+B0HkhQs/l/VIg0XJY6k5FRKE/JFEcVY4256NdqUZ+3jou3b4OAk
# tE2urr4V6VRw1fffOlxZb8MyvE5mqvTVJOStVxCuhuqg1rIe8la1gZ5iiuIyWeft
# ONfMw0nSZchGLigDeInw6XfwwgFnC5Ql8Pbf2jOxCUluAYbzykI+MnBiMIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpE
# OURFLUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUAn85fx36He7F0vgmyUlz2w82l0LGggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AOOFYpswIhgPMjAyMDEyMTcwODUyNDNaGA8yMDIwMTIxODA4NTI0M1owdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA44VimwIBADAKAgEAAgIqQAIB/zAHAgEAAgIRMTAK
# AgUA44a0GwIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBAET/pP5efMv+eo+W
# x6yjNh9OMfPsQsvDsI/stlWtQ/EFyKQKPWpC9NheRMl1h+2vTaCs+mksEzHwbuvp
# 7X5QXlv1sfqC9d9+rAqINH0lCz3ZQRTXOKcZm7wxSc4A1ftVufzRRDVuAFUZ2QKJ
# krdF19cqrYqd6bcSjdBcSav+TUQhMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAEtLk1BymNlM6AAAAAAAS0wDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQgr8NsC8a4mxzDCn7cA30b5+pyWnfEq8I/QiR51oSD9z0wgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCCO8Vpycn0gB4/ilRAPPDbS+Cmbqj/uC011moc5
# oeGDwTCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# LS5NQcpjZTOgAAAAAAEtMCIEINL782oYNUKrzB3nFEy/e//8y5ov1NFAV+IMZ0jd
# ZaPlMA0GCSqGSIb3DQEBCwUABIIBAGhV20VuZvi1oXKs/vpcMsnFAdEfvQPZXK13
# D85mC5HYfxAPyFyiFC6GlEHX8g3UknLtVOgr8wVMcsZbfNWvn2h5SKdxy0i29UNz
# K2KTz7jaA2XaWtk5mUs327eYo5xlFjGaYu4Gns4CfoUvIeAubyfjLsHp+1YPykZ6
# LiCoJ2NQDJyejQdoRnXksrKxW/R2U9K/PCar602HFPhpmOwHDQrpNxKP9SuYNilA
# mJjVK/OOTzhSdCm2kintM3hOFZICLFANnLHsLUcIOnDVwnguyv9yZZD+eEJuCzfc
# eaeweQ1Bebq9AbpIo2XDMhrH3kcLQyfjvaOlGhs+O1O/8KZtOVk=
# SIG # End signature block
