﻿####################################################################################
####################################################################################
## ValidateAOS.ps1
## Validate AOS deployment
## Date: 09/29/2015
##
####################################################################################
####################################################################################
[CmdletBinding()]
Param(
   [Parameter(Mandatory = $true)]
   [string]$InputXml,
   [Parameter(Mandatory = $false)]
   [string]$CredentialsXml,
   [Parameter(Mandatory = $true)]
   [string]$Log
)

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking
Initialize-Log $Log

####################################################################################
## Helper Functions
####################################################################################

## Append XML Rows to Template
function Append-RowToXML
{
    [CmdletBinding()]
    Param(
       [Parameter(Mandatory = $false)]
       [string]$TestName,
       [Parameter(Mandatory = $false)]
       [string]$TestType,
       [Parameter(Mandatory = $false)]
       [string]$TestResult,
       [Parameter(Mandatory = $false)]
       [string]$RawResult,
       [Parameter(Mandatory = $false)]
       [string]$TimeStamp,
       [Parameter(Mandatory = $true)]
       [xml]$xmlTemplate
    )

    Write-Log "Getting existing rows from XML Template"
    $rows = $xmlTemplate.SelectSingleNode('CollectionResult/TabularResults/TabularData/Rows')
    Write-Log "Creating new row"
    $row = $xmlTemplate.CreateElement('ArrayOfStrings')
    $column = $xmlTemplate.CreateElement('string')#TestName
    $column.InnerText = $TestName
    $row.AppendChild($column)
    Write-Log "Adding column value: $TestName"
    $column = $xmlTemplate.CreateElement('string')#TestType
    $column.InnerText = $TestType
    $row.AppendChild($column)
    Write-Log "Adding column value: $TestType"
    $column = $xmlTemplate.CreateElement('string')#TestResult
    $column.InnerText = $TestResult
    $row.AppendChild($column)
    Write-Log "Adding column value: $TestResult"
    $column = $xmlTemplate.CreateElement('string')#RawResult
    $column.InnerText = $RawResult
    $row.AppendChild($column)
    
    $column = $xmlTemplate.CreateElement('string')#TimeStamp
    $column.InnerText = $TimeStamp
    $row.AppendChild($column)
    $rows.AppendChild($row)
    Write-Log "Adding column value: $TimeStamp"
    $xmlTemplate.CollectionResult.TabularResults.TabularData.AppendChild($rows)
    $xmlTemplate.Save($xmlTemplate)
    Write-Log "Saved rows to XML Template"
}

####################################################################################
## Validation Functions
####################################################################################

## Validate endpoint sending HTTP request to configured endpoint 
function Validate-Endpoint{ 
     [CmdletBinding()] 
    Param( 
     [Parameter(Mandatory = $true)] 
     [string]$EndPointUrl 
     ) 
 
     [bool]$result = $false 
     [string]$rawResult 
     [string]$timestamp 
 
     try{ 
         Write-Log "Connecting to '$EndPointUrl'"
         $CurrentTime = (Get-Date).ToUniversalTime() 
         $webRequest = Invoke-WebRequest -Uri $EndPointUrl -TimeoutSec (5 * 60) -UseBasicParsing 
         if($webRequest.StatusCode -eq 200){ 
             $result = $true 
             $UrlTime = [DateTime]::Parse($webRequest.Headers.Date).ToUniversalTime() 
             $rawResult = ('HttpResult: ' + $webRequest.StatusCode.ToString() + '; PingTime(ms): ' + ($CurrentTime - $UrlTime).TotalMilliseconds).ToString() 
             $timestamp = (Get-Date).ToString() 
             Write-Log "Web request returned - $rawResult"
         } 
     } 
     catch{ 
         $rawResult = $_.Exception 
         $timestamp = (Get-Date).ToString() 
         Write-Log "ERROR: $($_.Exception) CALLSTACK: $_"
     } 
 
     if($result){ 
         $returnProperties = @{ 
             Result=1; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     else{ 
         $returnProperties = @{ 
             Result=0; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     $resultObject = New-Object PsObject -Property $returnProperties 
     return $resultObject 
} 
 
## Validate install Files from a manifest file 
function Validate-Install{ 
     [CmdletBinding()] 
    Param( 
     [Parameter(Mandatory = $true)] 
     [string]$InstallPath, 
     [Parameter(Mandatory = $false)] 
     [string]$ManifestPath 
     ) 
 
     [bool]$result = $false 
     [string]$rawResult 
     [string]$timestamp 
 
     try{ 
         Write-Log "Validating Install at '$InstallPath'"
         if(Test-Path -Path $InstallPath) 
         { 
             Write-Log "Comparing '$InstallPath' to manifest"
             [System.Array]$installedfiles = @() 
             $installedFilesRaw = Get-ChildItem -Path $InstallPath -Recurse | Where {$_.PSIsContainer -eq $false} | Select-Object -Property Name 
            foreach($file in $installedFilesRaw){ 
                $installedfiles += $file.Name 
             } 
 
             if(Test-Path -Path $ManifestPath){ 
                 $manifestFiles = Get-Content -Path $ManifestPath 
                 $fileCompare = Compare-Object -ReferenceObject $manifestFiles -DifferenceObject $installedFiles -Property Name -PassThru 
                 $timestamp = (Get-Date).ToString() 
                 if($fileCompare -eq $null) 
                 { 
                     $rawResult = "Installed file ARE a match to the manifest" 
                     Write-Log "$rawResult"
                     $result = $true 
                 } 
                 else 
                 { 
                     $rawResult = ("{0} Files are missing." -f $fileCompare.Count ) 
                     Write-Log "$rawResult"
                 } 
             } 
             else{ 
                Throw "$ManifestPath does not exist." 
             } 
         } 
         else{ 
            Throw "$InstallPath does not exist." 
         } 
     } 
     catch{ 
         $rawResult = $_.Exception 
         $timestamp = (Get-Date).ToString() 
         Write-Log "ERROR: $($_.Exception) CALLSTACK: $_"
     } 
 
     if($result){ 
         $returnProperties = @{ 
             Result=1; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     else{ 
         $returnProperties = @{ 
             Result=0; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
    } 
     $resultObject = New-Object PsObject -Property $returnProperties 
     return $resultObject 
} 
 
## Validate service is running 
function Validate-Service{ 
    [CmdletBinding()] 
    Param( 
     [Parameter(Mandatory = $true)] 
     [string]$ServiceName, 
     [Parameter(Mandatory = $true)] 
     [ValidateSet("Running","Stopped","Paused")] 
     [string]$CurrentState 
     ) 
 
     [bool]$result = $false 
     [string]$rawResult 
     [string]$timestamp 
 
     try{ 
         Write-Log "Validating Service: '$ServiceName' is $CurrentState"
         $thisService = Get-Service -Name $ServiceName 
         $timestamp = (Get-Date).ToString() 
         $rawResult = ("ServiceName: {0}; DisplayName: {1}; Status: {2}" -f $thisService.Name, $thisService.DisplayName, $thisService.Status) 
         if($thisService.Status.ToString() -eq $CurrentState) 
         { 
            $result = $true 
         } 
         Write-Log "Service: $ServiceName is $($thisService.Status)"
 
     } 
     catch{ 
         $rawResult = $_.Exception 
         $timestamp = (Get-Date).ToString() 
         Write-Log "ERROR: $($_.Exception) CALLSTACK: $_"
     } 
 
     if($result){ 
         $returnProperties = @{ 
             Result=1; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     else{ 
         $returnProperties = @{ 
             Result=0; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     $resultObject = New-Object PsObject -Property $returnProperties 
     return $resultObject 
} 
 
## Validate appPool is started 
function Validate-AppPool{ 
    [CmdletBinding()] 
    Param( 
     [Parameter(Mandatory = $true)] 
     [string]$AppPoolName, 
     [Parameter(Mandatory = $true)] 
     [ValidateSet("Started","Stopped")] 
     [string]$CurrentState 
     ) 
 
     [bool]$result = $false 
     [string]$rawResult 
     [string]$timestamp 
 
     try{ 
         Write-Log "Validating AppPool: '$AppPoolName' is $CurrentState"
         Get-WebAppPoolState 
         $appPoolStatus = Get-WebAppPoolState -Name $AppPoolName 
         $timestamp = (Get-Date).ToString() 
         $rawResult = ("AppPoolName: {0}; Status: {1}" -f $AppPoolName, $appPoolStatus) 
         if($appPoolStatus.Value -eq $CurrentState) 
         { 
            $result = $true 
         } 
         Write-Log "AppPool: $AppPoolName is $($appPoolStatus.Value)"
     } 
     catch{ 
         $rawResult = $_.Exception 
         $timestamp = (Get-Date).ToString() 
         Write-Log "ERROR: $($_.Exception) CALLSTACK: $_"
     } 
 
     if($result){ 
         $returnProperties = @{ 
             Result=1; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     else{ 
         $returnProperties = @{ 
             Result=0; 
            RawResults=$rawResult; 
            TimeStamp=$timestamp 
         } 
     } 
     $resultObject = New-Object PsObject -Property $returnProperties 
     return $resultObject 
} 




####################################################################################
## Parameter Setting
####################################################################################
Write-Log "Setting DVT execution parameters"

if(Test-Path -Path $InputXML)
{
    Write-Log "Parsing the xml for parameters/settings"
    [xml]$DVTParams = Get-Content -Path $InputXML
    [string]$ServiceName = $DVTParams.DVTParameters.ServiceName    
    [string]$AosWebrootPath = $DVTParams.DVTParameters.AosWebRootPath    
    [string]$XmlOutputPath = $DVTParams.DVTParameters.OutputPath    
	[string]$endPoint = $DVTParams.DVTParameters.EndPoint #Infrastructure.HostUrl
    [string]$installPath = $DVTParams.DVTParameters.InstallPath
	[string]$manifestFile = $DVTParams.DVTParameters.ManifestPath
	[string]$ServiceState = $DVTParams.DVTParameters.ServiceState
	[string]$AppPoolName = $DVTParams.DVTParameters.AppPoolName
	[string]$AppPoolState = $DVTParams.DVTParameters.AppPoolState
    [string]$BatchService = $DVTParams.DVTParameters.BatchService
}
else
{
    throw "Unable to parse settings from service model. Xml doesnt exist at: $InputXML"
}

if(-not ([string]::IsNullOrEmpty($CredentialsXml)))
{
    Write-Log "Parsing the CredentialsXml"
    if(Test-Path -Path $CredentialsXml)
    {
        Write-Log "Parsing the xml for local credentials"
        $localCredentials = Import-Clixml -Path $CredentialsXml
        [string]$UserName = $localCredentials.GetNetworkCredential().UserName
        [string]$UserPassword = $localCredentials.GetNetworkCredential().Password
    }
    else
    {
        throw "Unable to parse credentials from service model. Xml doesnt exist at: $CredentialsXML"
    }
}

Write-Log "Setting diagnostics-related parameters"
[string]$CollectorName = "$($ServiceName).DVT"
[string]$CollectorType = 'PowerShellCollector'
[string]$TargetName = (hostname)

if(-not (Test-Path -Path $XmlOutputPath))
{
    Write-Log "Creating diagnostics result directory at $XmlOutputPath"
    New-Item -Path $XmlOutputPath -Type Container | Out-Null
}

[string]$XMLFilePath = Join-Path -Path $XmlOutputPath -ChildPath "$([System.DateTime]::Now.ToFileTimeUtc())_$($ServiceName)DVTResults.xml"

####################################################################################
## Diagnostics Collector XML Template
####################################################################################
[xml]$xmlTemplate = @"
<?xml version="1.0"?>
<CollectionResult xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema">
  <CollectorName>$CollectorName</CollectorName>
  <CollectorType>$CollectorType</CollectorType>
  <ErrorMessages />
  <TabularResults>
    <TabularData>
      <TargetName>$TargetName</TargetName>
      <Columns>
        <string>TestName</string>
        <string>TestType</string>
        <string>PassResult</string>
        <string>RawResult</string>
        <string>TimeStamp</string>
      </Columns>
      <Rows>
      </Rows>
    </TabularData>
  </TabularResults>
</CollectionResult>
"@

####################################################################################
## Main Execution
####################################################################################

Write-Log "Running validations for $ServiceName"
try
{
     #End point validation 
     Write-Log "Validate-Endpoint -EndPointUrl" 
     $endpointResult = Validate-Endpoint -EndPointUrl $endPoint
 
     Append-RowToXML -TestName 'AOS.Validate-Endpoint' -TestType 'DVT' -TestResult $endpointResult.Result -RawResult $endpointResult.RawResults -TimeStamp $endpointResult.TimeStamp -xmlTemplate $xmlTemplate | Out-Null 

    $ValidateBatch = (![System.String]::IsNullOrWhiteSpace($DVTParams.DVTParameters.ValidateBatch) -and [System.Convert]::ToBoolean($DVTParams.DVTParameters.ValidateBatch))
    if ($ValidateBatch)
    {
        #AXBatch Service 
        Write-Log "Validate-Service -ServiceName $BatchService -CurrentState $ServiceState" 
        $serviceResult = Validate-Service -ServiceName $BatchService -CurrentState $ServiceState 

        Append-RowToXML -TestName 'AOS.Validate-Service' -TestType 'DVT' -TestResult $serviceResult.Result -RawResult $serviceResult.RawResults -TimeStamp $serviceResult.TimeStamp -xmlTemplate $xmlTemplate | Out-Null 
    } 

     #IIS AppPool Validation 
     Write-Log "Validate-AppPool -AppPoolName $AppPoolName -CurrentState $AppPoolState"
     $apppoolResult = Validate-AppPool -AppPoolName $AppPoolName -CurrentState $AppPoolState 

     Append-RowToXML -TestName 'AOS.Validate-AppPool' -TestType 'DVT' -TestResult $apppoolResult.Result -RawResult $apppoolResult.RawResults -TimeStamp $apppoolResult.TimeStamp -xmlTemplate $xmlTemplate | Out-Null 

    #Writing XML results
    Write-Log "Writing DVT results to $XMLFilePath"
    $xmlTemplate.InnerXml | Out-File -FilePath $XMLFilePath -Force -Encoding utf8

    [bool]$dvtResult = $endpointResult.Result -and $apppoolResult.Result
    if ($ValidateBatch)
    {
        $dvtResult = $dvtResult -and $serviceResult.Result
    }

}
catch
{
    Write-Exception $_
}

if($dvtResult)
{
    $exitProperties = @{'ExitCode'= 0}
    $exitObject = New-Object PsObject -Property $exitProperties
    Write-Log "DVT Script Completed, ExitCode: $($exitObject.ExitCode)"
    return $exitObject
}
else
{
    $exitProperties = @{'ExitCode'= 1; 'Message'="DVT Validation failed, see log: '$Log' for further details, and '$XMLFilePath' for test results"}
    $exitObject = New-Object PsObject -Property $exitProperties
    Write-Log "DVT Script Completed, ExitCode: $($exitObject.ExitCode)"
    throw $exitObject
}


# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBPDETZSfEdM3Dd
# jYdA0NRlWVmqeL50qgjbytga87dgeaCCDYEwggX/MIID56ADAgECAhMzAAABh3IX
# chVZQMcJAAAAAAGHMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAwMzA0MTgzOTQ3WhcNMjEwMzAzMTgzOTQ3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDOt8kLc7P3T7MKIhouYHewMFmnq8Ayu7FOhZCQabVwBp2VS4WyB2Qe4TQBT8aB
# znANDEPjHKNdPT8Xz5cNali6XHefS8i/WXtF0vSsP8NEv6mBHuA2p1fw2wB/F0dH
# sJ3GfZ5c0sPJjklsiYqPw59xJ54kM91IOgiO2OUzjNAljPibjCWfH7UzQ1TPHc4d
# weils8GEIrbBRb7IWwiObL12jWT4Yh71NQgvJ9Fn6+UhD9x2uk3dLj84vwt1NuFQ
# itKJxIV0fVsRNR3abQVOLqpDugbr0SzNL6o8xzOHL5OXiGGwg6ekiXA1/2XXY7yV
# Fc39tledDtZjSjNbex1zzwSXAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUhov4ZyO96axkJdMjpzu2zVXOJcsw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDU4Mzg1MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAixmy
# S6E6vprWD9KFNIB9G5zyMuIjZAOuUJ1EK/Vlg6Fb3ZHXjjUwATKIcXbFuFC6Wr4K
# NrU4DY/sBVqmab5AC/je3bpUpjtxpEyqUqtPc30wEg/rO9vmKmqKoLPT37svc2NV
# BmGNl+85qO4fV/w7Cx7J0Bbqk19KcRNdjt6eKoTnTPHBHlVHQIHZpMxacbFOAkJr
# qAVkYZdz7ikNXTxV+GRb36tC4ByMNxE2DF7vFdvaiZP0CVZ5ByJ2gAhXMdK9+usx
# zVk913qKde1OAuWdv+rndqkAIm8fUlRnr4saSCg7cIbUwCCf116wUJ7EuJDg0vHe
# yhnCeHnBbyH3RZkHEi2ofmfgnFISJZDdMAeVZGVOh20Jp50XBzqokpPzeZ6zc1/g
# yILNyiVgE+RPkjnUQshd1f1PMgn3tns2Cz7bJiVUaqEO3n9qRFgy5JuLae6UweGf
# AeOo3dgLZxikKzYs3hDMaEtJq8IP71cX7QXe6lnMmXU/Hdfz2p897Zd+kU+vZvKI
# 3cwLfuVQgK2RZ2z+Kc3K3dRPz2rXycK5XCuRZmvGab/WbrZiC7wJQapgBodltMI5
# GMdFrBg9IeF7/rP4EqVQXeKtevTlZXjpuNhhjuR+2DMt/dWufjXpiW91bo3aH6Ea
# jOALXmoxgltCp1K7hrS6gmsvj94cLRf50QQ4U8Qwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAYdyF3IVWUDHCQAAAAABhzAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgVuWd13CR
# bHTpDoFSovvkFmdXY4mtCclD/Mh+zNCdEMkwQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQBVIRM3F/2qfqi8cQMJYgAIrog6gOI64FDdik0XgszR
# Q1QdN/ma+O5X8zSA8l9pI+v+pUug8plg7q/7Webn1xqvSp2byKFUeoGAnlyWGdGd
# B9Gjt57rropcaYEXr/WKlMPBsJ23bFwt9AIvqFLI49d2HJQDv5Ut9MgTE/v8JPhC
# MtEMuhQMeGlxIit19ZXFmqagWSyknBj50XT/ExD2UMu+8tul6NKxTEoFgoFrPHyV
# uZ/xconCtvPkWAOsxv5KepPd34LzsqStr2/JvvCGcbg4/2w9gj6YhVElp+mNerja
# RTuJAAaw90S4DSpS9yQUih0jjutJJ7PVyhmumaEmSP7toYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIMPl0zRtoQXhZd4DxMQX0W5bZIdigzIAoSlI8zge
# COwzAgZfu+oE2oEYEzIwMjAxMjE3MDgwNjI3LjcxM1owBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjpEOURFLUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABLS5NQcpjZTOgAAAA
# AAEtMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTE5MTIxOTAxMTUwNFoXDTIxMDMxNzAxMTUwNFowgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpEOURF
# LUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKlhjfR1STqYRTS3s0i4
# jIcSMV+G4N0oYgwlQK+pl4DVMFmr1iTZHFLj3Tt7V6F+M/BXx0h9i0uu1yBnqCqN
# OkuJERTbVnM4u3JvRxzsQfCjBfqD/CNwoMNekoylIBzxP50Skjp1pPsnQBKHaCP8
# tguvYVzoTQ54q2VpYEP/+OYTQeEPqWFi8WggvsckuercUGkhYWM8DV/4JU7N/rbD
# rtamYbe8LtkViTQYbigUSCAor9DhtAZvq8A0A73XFH2df2wDlLtAnKCcsVvXSmZ3
# 5bAqneN4uEQVy8NQdReGI1tI6UxoC7XnjGvK4McDdKhavNJ7DAnSP5+G/DTkdWD+
# lN8CAwEAAaOCARswggEXMB0GA1UdDgQWBBTZbGR8QgEh+E4Oiv8vQ7408p2GzTAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQB9awNk906recBuoO7Ezq7B8UGu9EoF
# XiL8ac0bbsZDBY9z/3p8atVZRCxHN43a3WGbCMZoKYxSBH6UCkcDcwXIfNKEbVMz
# nF1mjpQEGbqhR+rPNqHXZotSV+vn85AxmefAM3bcLt+WNBpEuOZZ4kPZVcFtMo4Y
# yQjxoNRPiwmp+B0HkhQs/l/VIg0XJY6k5FRKE/JFEcVY4256NdqUZ+3jou3b4OAk
# tE2urr4V6VRw1fffOlxZb8MyvE5mqvTVJOStVxCuhuqg1rIe8la1gZ5iiuIyWeft
# ONfMw0nSZchGLigDeInw6XfwwgFnC5Ql8Pbf2jOxCUluAYbzykI+MnBiMIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpE
# OURFLUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUAn85fx36He7F0vgmyUlz2w82l0LGggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AOOFYpswIhgPMjAyMDEyMTcwODUyNDNaGA8yMDIwMTIxODA4NTI0M1owdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA44VimwIBADAKAgEAAgIqQAIB/zAHAgEAAgIRMTAK
# AgUA44a0GwIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBAET/pP5efMv+eo+W
# x6yjNh9OMfPsQsvDsI/stlWtQ/EFyKQKPWpC9NheRMl1h+2vTaCs+mksEzHwbuvp
# 7X5QXlv1sfqC9d9+rAqINH0lCz3ZQRTXOKcZm7wxSc4A1ftVufzRRDVuAFUZ2QKJ
# krdF19cqrYqd6bcSjdBcSav+TUQhMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAEtLk1BymNlM6AAAAAAAS0wDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQglafcnM4xMgrlDWF7qJ4dBPgP1DmbLG5MsKrfhqKE+awwgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCCO8Vpycn0gB4/ilRAPPDbS+Cmbqj/uC011moc5
# oeGDwTCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# LS5NQcpjZTOgAAAAAAEtMCIEINL782oYNUKrzB3nFEy/e//8y5ov1NFAV+IMZ0jd
# ZaPlMA0GCSqGSIb3DQEBCwUABIIBABNJd3I5y/BnqDWY5Z4zqMUVzEP/AbEsLTbs
# eYommg1OjVohOZZ6e6FcNPyfMbiBwfU2ETRVEwTaXyc15bk+lTuQjqkjnc4Ti+qx
# gx2YJ+ZNz6L4qryNTlARrZ+t0cJBXF2D4Dgx4swqteZsvjm1lifvK0y46dfiMSYG
# 9MghIdbN0Srl1FRTtiHO3FhW5faUCijIvV6fhlzCXghX7PyVT6Vc8B84UWCIYet9
# 3jtUXlJbHopypTOkQiivoyjRdzwdAUnrnpIAIBRJpCy8kGr5A+q9Bjdd83r65j+2
# hQy8sy9Mu4eDDwQoGUWVZQ5FWgQIwnieFUbxMON2SSEAm3l0elo=
# SIG # End signature block
