﻿[CmdletBinding()]
param
(
	# Directory to place logs. Passed in when upgrade is triggered through LCS.
    [string]
	$LogDir
)

Import-Module "$PSScriptRoot\CommonRollbackUtilities.psm1" -DisableNameChecking
Import-Module "$PSScriptRoot\AOSEnvironmentUtilities.psm1" -DisableNameChecking

function Get-ApplicationEnvironment
{
    $ErrorActionPreference = 'Stop'
	       
    $webroot = Get-AosWebSitePhysicalPath
    $InstalledPlatformBuildVersion = Get-ProductPlatformBuildFullVersion
	$isKeyVault = Get-IsMSIKeyVault

    if (($isKeyVault -eq $true) -and ($InstalledPlatformBuildVersion -ne $null) -and ($InstalledPlatformBuildVersion.Build -ge 5644))
	{        
        # Get-PackageRoot comes from CommonRollbackUtilities.psm1. it resolves the root of the deployable package
        $packageRoot = Get-PackageRoot
        $packageTemp = ""
        if(![string]::IsNullOrEmpty($packageRoot) -and (Test-Path $packageRoot))
        {
            $packageTemp = Join-Path $packageRoot "RunbookWorkingFolder\KeyVaultAssemblyTemp"
        }

        if ([string]::IsNullOrEmpty($packageTemp))
        {   
            $suffix = $(Get-Date).ToString("yyyyMMdd_HHmmss")
            $dirName = "AssemblyTemp_$suffix"
            $packageTemp = Join-Path $env:Temp "dirName"
        }

        if(!(Test-Path $packageTemp -PathType Container))
        {
            New-Item $packageTemp -ItemType Container | Out-Null
        }
        
        #need to load the Microsoft.Dynamics.ApplicationPlatform.Environment.dll and all the dll it referenced
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Configuration.Base.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.Instrumentation.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Framework.EncryptionEngine.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.BusinessPlatform.SharedTypes.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.KeyVaultHelper.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Carbon.Flighting.Runtime.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Carbon.CertificateCommon.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Commerce.Flighting.Common.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Flighting.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.XppServices.Instrumentation.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.CE.VaultSDK.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.CE.VaultSDK.ADALv2.AuthHelper.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.KeyVault.Core.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.KeyVault.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.Common.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Azure.Common.NetFramework.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Hyak.Common.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Authentication.Instrumentation.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.Extensions.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Threading.Tasks.Extensions.Desktop.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.IdentityModel.Clients.ActiveDirectory.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Practices.EnterpriseLibrary.TransientFaultHandling.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Newtonsoft.Json.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.SharedUtility.dll'
        CopyDllToTempFolder -dllPath $dllPath -packageTemp $packageTemp    
                
        try
        {
            $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
            $fileTempPath =  Join-Path $packageTemp $(Get-Item $dllPath).Name
	        if (Test-Path $fileTempPath)
            {
                # Load file from temp folder
	            [System.Reflection.Assembly]::LoadFrom($fileTempPath)
            }
        }
        catch
        {    
            #write-log "Failed to load from temp folder with Exception: [$($_.Exception)]"
            throw $_
        }
	}              
    else
    {
        #need to load the Microsoft.Dynamics.ApplicationPlatform.Environment.dll and all the dll it referenced
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.ApplicationPlatform.Environment.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Configuration.Base.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Security.Instrumentation.dll'
        if (Test-Path $dllPath)
        {
            Load-DllinMemory -dllPath $dllPath
        }
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.AX.Framework.EncryptionEngine.dll'
        Load-DllinMemory -dllPath $dllPath
        $dllPath = Join-Path $webroot 'bin\Microsoft.Dynamics.BusinessPlatform.SharedTypes.dll'
        Load-DllinMemory -dllPath $dllPath  
	}

    $config = [Microsoft.Dynamics.ApplicationPlatform.Environment.EnvironmentFactory]::GetApplicationEnvironment()
    
    return $config
}

function Load-DllinMemory([string] $dllPath)
{
    #try catch as not all dll exist in RTM version, some dependency/dll are introduced at update 1 or later
    #powershell cannot unload dll once it's loaded, the trick is to create an in-memory copy of the dll than load it
    #after the loading of in-memory dll, the physical dll stay unlocked

    try
    {
	    if (Test-Path $dllPath)
        {
            $bytes = [System.IO.File]::ReadAllBytes($dllPath)
            [System.Reflection.Assembly]::Load($bytes) | Out-Null
        }
    }
    catch
    {}
}

function CopyDllToTempFolder([string] $dllPath, [string] $packageTemp)
{
    #try catch as not all dll exist in RTM version, some dependency/dll are introduced at update 1 or later    
    $fileTempPath =  Join-Path $packageTemp $(Get-Item $dllPath).Name
    try
    {
	    if ((Test-Path $dllPath) -And !$(Test-Path $fileTempPath))
        {
            $Buffer = New-Object 'byte[]' 1024
            # Open file in read mode
            $SourceFile = [System.IO.File]::Open($dllPath, [System.IO.FileMode]::Open, [System.IO.FileAccess]::Read, [System.IO.FileShare]::Read)
            # Create the new file
            $DestinationFile = [System.IO.File]::Open($fileTempPath, [System.IO.FileMode]::CreateNew)
            try{
                # Copy the contents of the source file to the destination
                while(($readLength = $SourceFile.Read($Buffer, 0, $Buffer.Length)) -gt 0)
                {
                    $DestinationFile.Write($Buffer, 0, $readLength)
                }
            }
            catch{
                throw $_
            }
            finally{
                $SourceFile.Close()
                $DestinationFile.Close()
            }
        }
    }
    catch
    {
        Write-Log "Failed to copy file to temp folder with Exception: [$($_.Exception)]"
        throw $_
    }    
}

function Get-IsMSIKeyVault
{
    $webroot = Get-AosWebSitePhysicalPath
    $webconfig=join-path $webroot "web.config"
    $isMSIKeyVault=$false
    
    [System.Xml.XmlDocument] $xd=new-object System.Xml.XmlDocument
    $xd.Load($webconfig)
    $ns=New-Object System.Xml.XmlNamespaceManager($xd.NameTable)
    $ns.AddNamespace("ns",$xd.DocumentElement.NamespaceURI)

    $key="CertificateHandler.IsMSIKeyVault"

    $isKeyVaultValue = $xd.SelectSingleNode("//ns:add[@key='$key']",$ns)

    if($isKeyVaultValue -ne $null)
    {        
        $isMSIKeyVault = $isKeyVaultValue.GetAttribute("value")
    }

    return $isMSIKeyVault 
}

Start-Sleep -s 20

Import-Module "$PSScriptRoot\AOSCommon.psm1" -DisableNameChecking

$datetime=get-date -Format "MMddyyyyhhmmss"

if(!$LogDir)
{
    $LogDir = $PSScriptRoot
}

$log=join-path $LogDir "AutoRunDVT_AOS_$datetime.log"
  
Initialize-Log $log

$TestList = Get-ChildItem "$PSScriptRoot\DVT" -Filter 'Validate*.ps1'

$config = Get-ApplicationEnvironment

$ServiceName = $config.Infrastructure.ApplicationName
$endpointValue = $config.Infrastructure.HostUrl
$batchService = 'DynamicsAxBatch'
$serviceState = 'Running'
$appPoolState = 'Started'
$installPath = $config.Infrastructure.AOSWebRoot
$manifestPath = Join-Path $installPath 'web.config'
$appPoolName = $config.Infrastructure.ApplicationName
#Need to update the environmentassembly to add this value.
[xml]$WebConfig = Get-Content "$(join-path $installPath "web.config")"
$IsPrivateAOSInstance = $($WebConfig.configuration.appSettings.add | where { $_.key -eq 'Infrastructure.IsPrivateAOSInstance' }).value
$ValidateBatch = ([System.String]::IsNullOrWhiteSpace($IsPrivateAOSInstance) -or ![System.Convert]::ToBoolean($IsPrivateAOSInstance))

if (!$env:SERVICEDRIVE)
{
	Write-Log '%SERVICEDRIVE% was not defined. Exiting.' -Warning
	exit
}

Write-Log "Running DVT Process for $ServiceName"

$DiagnosticsPath = Join-Path -Path $env:SERVICEDRIVE -ChildPath "DynamicsDiagnostics\$ServiceName"

foreach ($test in $testList)
{
    Write-Log "Processing $test"    
    
    $DVTLocalRoot = Join-Path -Path $DiagnosticsPath -ChildPath ([System.IO.Path]::GetFileNameWithoutExtension($test))

    $DVTLocalBin = Join-Path $DVTLocalRoot "input"
    if(-not (Test-Path -Path $DVTLocalBin))
    {
        Write-Log "Creating DVT local bin at: '$DVTLocalBin'"
        New-Item -Path $DVTLocalBin -Type Directory -Force | Out-Null
    }   
        
    $DVTScript = Join-Path -Path "$PSScriptRoot\DVT" -ChildPath $test
    Write-Log "Copy DVT Script '$DVTScript' to $DVTLocalBin"
    Copy-Item -Path $DVTScript -Destination $DVTLocalBin -Recurse -Force | Out-Null

    Write-Log "Copy AosCommon.psm1 to DVTLocalBin"
    $AosCommon = Join-Path -Path $PSScriptRoot -ChildPath "AosCommon.psm1"
    Copy-Item -Path $AosCommon -Destination $DVTLocalBin -Recurse -Force | Out-Null

        
    [string]$DVTOutputBin = Join-Path -Path $DVTLocalRoot -ChildPath "Output"

    ####################################################################################
    ## DVT input XML Template
    ####################################################################################
    [xml]$xmlTemplate = "<?xml version=`"1.0`"?>
    <DVTParameters xmlns:xsi=`"http://www.w3.org/2001/XMLSchema-instance`" xmlns:xsd=`"http://www.w3.org/2001/XMLSchema`">
        <AosWebRootPath>$installPath</AosWebRootPath>
        <ServiceName>$ServiceName</ServiceName>        
        <OutputPath>$DVTOutputBin</OutputPath>
        <EndPoint>$endpointValue</EndPoint>
        <InstallPath>$installPath</InstallPath>
        <ManifestPath>$manifestPath</ManifestPath>
        <BatchService>$batchService</BatchService>
        <ServiceState>$serviceState</ServiceState>
        <AppPoolName>$appPoolName</AppPoolName>
        <AppPoolState>$appPoolState</AppPoolState>
        <ValidateBatch>$ValidateBatch</ValidateBatch>
    </DVTParameters>"

    $XMLInputPath = Join-Path -Path $DVTLocalBin -ChildPath "$([System.IO.Path]::GetFileNameWithoutExtension($test)).xml"
    Write-Log "Executing DVT XML at: $XMLInputPath"
    $xmlTemplate.InnerXml | Out-File -FilePath $XMLInputPath -Force -Encoding utf8
    
    ####################################################################################
    ## Execute DVT Script
    ####################################################################################
    try
    {
        $DVTLocalScript = Join-Path -Path $DVTLocalBin -ChildPath $test
        if(Test-Path -Path $DVTLocalScript)
        {
            Write-Log "Executing DVT Script: $DVTLocalScript"
            $commandArgs = @{
                "InputXML" = $XMLInputPath;                
                "Log" = $Log
            }
    
            & $DVTLocalScript @commandArgs
            
        }
        else
        {
            throw "$DVTLocalScript was not found."
        }
    }
    catch
    {     
        Write-Exception $_       
        throw "DVT Script Failed, see $log for details."
    }
}    

# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCA86YOltOSTzBet
# okCiOQ7oUgszHKt0kwcQFZAZfzQiC6CCDYEwggX/MIID56ADAgECAhMzAAABh3IX
# chVZQMcJAAAAAAGHMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAwMzA0MTgzOTQ3WhcNMjEwMzAzMTgzOTQ3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDOt8kLc7P3T7MKIhouYHewMFmnq8Ayu7FOhZCQabVwBp2VS4WyB2Qe4TQBT8aB
# znANDEPjHKNdPT8Xz5cNali6XHefS8i/WXtF0vSsP8NEv6mBHuA2p1fw2wB/F0dH
# sJ3GfZ5c0sPJjklsiYqPw59xJ54kM91IOgiO2OUzjNAljPibjCWfH7UzQ1TPHc4d
# weils8GEIrbBRb7IWwiObL12jWT4Yh71NQgvJ9Fn6+UhD9x2uk3dLj84vwt1NuFQ
# itKJxIV0fVsRNR3abQVOLqpDugbr0SzNL6o8xzOHL5OXiGGwg6ekiXA1/2XXY7yV
# Fc39tledDtZjSjNbex1zzwSXAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUhov4ZyO96axkJdMjpzu2zVXOJcsw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDU4Mzg1MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAixmy
# S6E6vprWD9KFNIB9G5zyMuIjZAOuUJ1EK/Vlg6Fb3ZHXjjUwATKIcXbFuFC6Wr4K
# NrU4DY/sBVqmab5AC/je3bpUpjtxpEyqUqtPc30wEg/rO9vmKmqKoLPT37svc2NV
# BmGNl+85qO4fV/w7Cx7J0Bbqk19KcRNdjt6eKoTnTPHBHlVHQIHZpMxacbFOAkJr
# qAVkYZdz7ikNXTxV+GRb36tC4ByMNxE2DF7vFdvaiZP0CVZ5ByJ2gAhXMdK9+usx
# zVk913qKde1OAuWdv+rndqkAIm8fUlRnr4saSCg7cIbUwCCf116wUJ7EuJDg0vHe
# yhnCeHnBbyH3RZkHEi2ofmfgnFISJZDdMAeVZGVOh20Jp50XBzqokpPzeZ6zc1/g
# yILNyiVgE+RPkjnUQshd1f1PMgn3tns2Cz7bJiVUaqEO3n9qRFgy5JuLae6UweGf
# AeOo3dgLZxikKzYs3hDMaEtJq8IP71cX7QXe6lnMmXU/Hdfz2p897Zd+kU+vZvKI
# 3cwLfuVQgK2RZ2z+Kc3K3dRPz2rXycK5XCuRZmvGab/WbrZiC7wJQapgBodltMI5
# GMdFrBg9IeF7/rP4EqVQXeKtevTlZXjpuNhhjuR+2DMt/dWufjXpiW91bo3aH6Ea
# jOALXmoxgltCp1K7hrS6gmsvj94cLRf50QQ4U8Qwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAYdyF3IVWUDHCQAAAAABhzAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgTyzjztgi
# Vrky53xY9c0M+0qxUAh/DzUpgO8DId4sak0wQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQAG0YSoUh+YkUbntXfOvQjojmBRFXwwCa9m22CAuxGh
# /kSkJk7XOaqnjjnqm161yvo3fQ/jpwEDjRH6H2mfuJFUKMgD0dSqqiLLUNkDrtYV
# YN9YtFERDMu7GHGwzyx4QMlzvmqJuusFbRpPtJx9QOqSOm08qJrIcbY5xUksy792
# 7QlSCARV5KGv7r58MyzzJtedNQ2N5Fn7Q6I6Ua9JQtRECJ6Al66+5yZ6hCfoQJ1C
# WZZ4yF6GUuDp8O6yPS48+izDaUlb7KGpPXR3Tj0EuQ5DOCqn8fPL1LjsWHcS7YLr
# gedrfAX0atM+kNElac1lcPMZcdk0ljFKe76bpxRSuuuAoYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEII+HZLirVMQgLR/GAfL1TfFJcN/016TaUCvFer57
# cTfKAgZfs/cijEAYEzIwMjAxMTIwMTkxMzIyLjM1N1owBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjpDNEJELUUzN0YtNUZGQzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABIziw5K3YWpCdAAAA
# AAEjMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTE5MTIxOTAxMTQ1NloXDTIxMDMxNzAxMTQ1Nlowgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpDNEJE
# LUUzN0YtNUZGQzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJ280MmwZKXcAS7x2ZvA
# 4TOOCzw+63Xs9ULGWtdZDN3Vl+aGQEsMwErIkgzQi0fTO0sOD9N3hO1HaTWHoS80
# N/Qb6oLR2WCZkv/VM4WFnThOv3yA5zSt+vuKNwrjEHFC0jlMDCJqaU7St6WJbl/k
# AP5sgM0qtpEEQhxtVaf8IoV5lq8vgMJNr30O4rqLYEi/YZWQZYwQHAiVMunCYpJi
# ccnNONRRdg2D3Tyu22eEJwPQP6DkeEioMy9ehMmBrkjADVOgQV+T4mir+hAJeINy
# sps6wgRO5qjuV5+jvczNQa1Wm7jxsqBv04GClIp5NHvrXQmZ9mpZdj3rjxFuZbKj
# d3ECAwEAAaOCARswggEXMB0GA1UdDgQWBBSB1GwUgNONG/kRwrBo3hkV+AyeHjAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQBb5QG3qlfoW6r1V4lT+kO8LUzF32S3
# fUfgn/S1QQBPzORs/6ujj09ytHWNDfOewwkSya1f8S9e+BbnXknH5l3R6nS2BkRT
# ANtTmXxvMLTCyveYe/JQIfos+Z3iJ0b1qHDSnEj6Qmdf1MymrPAk5jxhxhiiXlwI
# LUjvH56y7rLHxK0wnsH12EO9MnkaSNXJNCmSmfgUEkDNzu53C39l6XNRAPauz2/W
# slIUZcX3NDCMgv5hZi2nhd99HxyaJJscn1f8hZXA++f1HNbq8bdkh3OYgRnNr7Qd
# nO+Guvtu3dyGqYdQMMGPnAt4L7Ew9ykjy0Uoz64/r0SbQIRYty5eM9M3MIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpD
# NEJELUUzN0YtNUZGQzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUAuhdmjeDinhfC7gw1KBCeM/v7V4GggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AONiaaUwIhgPMjAyMDExMjAyMDEzMjVaGA8yMDIwMTEyMTIwMTMyNVowdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA42JppQIBADAKAgEAAgIEpQIB/zAHAgEAAgIR+TAK
# AgUA42O7JQIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBAJASWoxkMewnB0mV
# FfDVo4fUWNUDwbUwp1n5aC3dR5ig56R/HAqd2U53xrP2airq3NhcInQk7K9lNvX+
# n+WgZvf4QIlALulvNFcWBWKSpD5FxH+cjPb6yLiDpsJ2umHsrPsMTRKOKRQLs9wU
# 6KDsTlNq8FFTHppQd7pB7dHGGHDBMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAEjOLDkrdhakJ0AAAAAASMwDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQgN4qWHglWqDXwWoSU1YZNuTIk0o+X/liDC3KUWueByKowgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCARmjODP/tEzMQNo6OsxKQADL8pwKJkM5YnXKrq
# +xWBszCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# Iziw5K3YWpCdAAAAAAEjMCIEIKeozB9rVUP2SeV4fBt2QQVX0Djb3fsaKWbEtpFJ
# RTdZMA0GCSqGSIb3DQEBCwUABIIBABa7hkP4ohDn0IP8CogR2A5RRbNWFsuiVLCg
# vp+PNPC1Rz0kRcQ1xFu53fm6pqXOFczS2lZmQLCjUsEebRuYUlceBIcspIwExwME
# 0fnNv3eIie4NS/vr+Myw//TZ5Tpr/zupTSLYVuuTIDKYKGkY+z0lHYt5MJVX9Ak+
# i3plM4YyUBgz8EuByQa/a0GjAe7f64Y8DDyRVAqQJ9KcGT11pQtPlYSHpDlGDLuh
# pIowHtPZ8SbTTPtgY6NY4IZqKhQT/bXRn7RqLf2drBQp18rJOkYnLQE5ouqmnjEt
# xgIy3NcKMSXseupQrfNaCakfTCGaGY+2IVtKrdBOtJ05NgsnDlU=
# SIG # End signature block
