﻿param
(
    [Parameter(Mandatory=$true)]
    $config,
    [Parameter(Mandatory=$true)]
    $log
)

Import-Module "$PSScriptRoot\AosCommon.psm1" -Force -DisableNameChecking

function Set-RegistryValues($settings)
{
    New-Item -Path HKLM:\SOFTWARE\Microsoft\Dynamics -Name Deployment -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name DeploymentDirectory -PropertyType String -Value $($settings.'Infrastructure.WebRoot') -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name BinDir -PropertyType String -Value $($settings.'Common.BinDir') -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name metadatadirectory -PropertyType String -Value $($settings.'Aos.MetadataDirectory') -Force -Confirm:$false
    New-ItemProperty -Path HKLM:\SOFTWARE\Microsoft\Dynamics\Deployment -Name frameworkdirectory -PropertyType String -Value $($settings.'Aos.PackageDirectory') -Force -Confirm:$false
}

function Configure-ClaimIssuers($webConfig, $settings)
{
    Write-Log "Start processing claim issuer restrictions."
    $ClaimIssuers = $settings.'Infrastructure.ClaimIssuers'
    if (![System.String]::IsNullOrWhitespace($ClaimIssuers))
    {
        $Issuers = $ClaimIssuers.Split(';')
        Write-Log "Processing claim issuer restrictions: $Issuers"

        #add missing elements to the web.config
        if ($Issuers.Count -gt 0)
        {
            $ClaimIssuerRestrictions = $webConfig.SelectNodes("configuration/claimIssuerRestrictions")
            if ($ClaimIssuerRestrictions -eq $null)
            {
                Write-Log "Web config does not contain element configuration/claimIssuerRestrictions. Creating."
                $ClaimIssuerRestrictions = $webConfig.CreateElement("claimIssuerRestrictions")
                $web.configuration.AppendChild($ClaimIssuerRestrictions)
            }

            $IssuerRestrictions = $ClaimIssuerRestrictions.issuerRestrictions
            if ($IssuerRestrictions -eq $null)
            {
                Write-Log "Web config does not contain element configuration/claimIssuerRestrictions/issuerRestrictions. Creating."
                $IssuerRestrictions = $webConfig.CreateElement("issuerRestrictions")
                $ClaimIssuerRestrictions.AppendChild($IssuerRestrictions)
            }
        }

        #add claim issuer restrictions
        foreach($Issuer in $Issuers)
        {
            $Tokens = $Issuer.Split(':')
            if ($Tokens.Count -lt 2 -or $Tokens.Count -gt 3)
            {
                throw "Claim issuer restriction is not in a valid format (expected format: issuerName:userId[,userId,userId,...[:defaultUserId]]): $Issuer"
            }

            $IssuerName = $Tokens[0]
            $AllowedUserIds = $Tokens[1]

            Write-Log "Processing issuer $IssuerName."

            $IssuerRestriction = $IssuerRestrictions.add | where { ($_.name -eq $IssuerName) }
            if ($IssuerRestriction -eq $null)
            {
                Write-Log "Creating new claim issuer restriction for issuer $IssuerName."

                $IssuerRestriction = $webConfig.CreateElement("add")
                $IssuerRestriction.SetAttribute("name", $IssuerName)
                $IssuerRestrictions.AppendChild($IssuerRestriction) | Out-Null
            }
            else
            {
                Write-Log "Claim issuer restriction already exists for issuer $IssuerName. Overriding with new restrictions."
            }

            $IssuerRestriction.SetAttribute("allowedUserIds", $AllowedUserIds)
            Write-Log "User ids $AllowedUserIds were set for $IssuerName."

            if ($Tokens.Count -eq 3)
            {
                $DefaultUserId = $Tokens[2]
                Write-Log "Setting default user id $DefaultUserId for $IssuerName."
                $IssuerRestriction.SetAttribute("defaultUserId", $DefaultUserId)
            }
        }
    }

    Write-Log "Finished processing claim issuer restrictions."
}

function Configure-MachineKeys($webConfig, $settings)
{
    Write-Log "Start processing machine keys."
    $DecryptionKey = $settings.'Infrastructure.MachineDecryptionKey'
    $ValidationKey = $settings.'Infrastructure.MachineValidationKey'

    if ($DecryptionKey -eq $null)
    {
        Write-Log "DecryptionKey not provided, skipping configuration."
        return
    }

    if ($ValidationKey -eq $null)
    {
        Write-Log "ValidationKey not provided, skipping configuration."
        return
    }

    $Web = $webConfig.SelectNodes("configuration/location[@path='.']/system.web")

    $MachineKey = $Web.machineKey

    if ($MachineKey -eq $null)
    {
        Write-Log "Creating machineKey element."

        $MachineKey = $webConfig.CreateElement("machineKey")
        $MachineKey.SetAttribute("decryption", "AES")
        $MachineKey.SetAttribute("decryptionKey", $DecryptionKey)
        $MachineKey.SetAttribute("validation", "HMACSHA256")
        $MachineKey.SetAttribute("validationKey", $ValidationKey)
        $Web.AppendChild($MachineKey) | Out-Null
    }
    else
    {
        Write-Log "Updating machineKey element."
        $MachineKey.decryption = "AES"
        $MachineKey.decryptionKey = $DecryptionKey
        $MachineKey.validation = "HMACSHA256"
        $MachineKey.validationKey = $ValidationKey
    }

    Write-Log "Finished processing machine keys."
}

function CreateFlightingCacheFolder($settings)
{
    Write-Log "Creating the Flighting Cache Folder for Application Platform Flighting"
    $webroot= $settings.'Infrastructure.WebRoot'
    $flightingcachefolder = $settings.'DataAccess.FlightingServiceCacheFolder'
    $flightingcachepath = Join-Path $webroot $flightingcachefolder

    if(!(Test-Path $flightingcachepath))
    {
        Write-Log "Creating $flightingcachepath."
        New-Item -ItemType Directory -Path $flightingcachepath -Force
    }

    #Flighting is enabled for SyncEngine.exe, it may be running by Admin,
    #so "Network Service" must be added at beginning in case W3Wp can not access it later.

    $User = "NETWORK SERVICE"
    $FullControlPermission = "FullControl"
    $Permission = (Get-Acl $flightingcachepath).Access | ?{$_.IdentityReference -match $User -and $_.FileSystemRights -match $FullControlPermission} | Select IdentityReference,FileSystemRights
    if ($Permission)
    {
        $Permission | % {Write-Log "User $($_.IdentityReference) has '$($_.FileSystemRights)' rights on folder $flightingcachepath"}
    }
    else
    {
        Write-Log "$User doesn't have full control on $flightingcachepath, adding $User"
        $Acl = Get-Acl $flightingcachepath

        $NetWorkService = New-Object System.Security.AccessControl.FileSystemAccessRule($User, $FullControlPermission, "ContainerInherit,ObjectInherit", "None", "Allow")

        $Acl.SetAccessRule($NetWorkService)
        Set-Acl $flightingcachepath $Acl

        Write-Log "added $User on $flightingcachepath"
    }
}

function Configure-FlightingCachePath($webConfig, $settings)
{
    Write-Log "Start processing Flighting Cache Path."

    #add missing elements to the web.config
    $add = $webConfig.configuration.appSettings.add | where key -eq 'DataAccess.FlightingCachePath' | select -First 1
    if ($add -eq $null)
    {
        $webRootDirectory = ($webConfig.configuration.appSettings.add | Where {$_.Key -eq 'Infrastructure.WebRoot'}).Value
        if ([System.String]::IsNullOrWhitespace($webRootDirectory))
        {
            Write-Log "Could not find web root!!!!"
        }
        else
        {
            $flightingcachefolder = $settings.'DataAccess.FlightingServiceCacheFolder'
            $flightingcachepath = Join-Path $webRootDirectory $flightingcachefolder
            $add = $webConfig.CreateElement("add")
            $add.SetAttribute("key", "DataAccess.FlightingCachePath")
            $add.SetAttribute("value", $flightingcachepath)
            $webConfig.configuration.appSettings.AppendChild($add) | Out-Null
        }
    }
}

function Update-WifServicesConfig($settings)
{
    $siteAdmin = $settings.'Provisioning.AdminPrincipalName'

    Write-Log "Reading wif.services.config at $($settings.'Infrastructure.WebRoot')"

    [xml]$wifServicesConfig = Get-Content "$($settings.'Infrastructure.WebRoot')\wif.services.config"
    [Uri]$uri = New-Object Uri $($settings.'Aad.AADLoginWsfedEndpointFormat' -f $($siteAdmin.Split('@')[1]))

    $wifServicesConfig.'system.identityModel.services'.federationConfiguration.wsFederation.issuer = "$($uri.AbsoluteUri)"

    $HostUri = New-Object Uri $settings.'Infrastructure.HostUrl'
    $wifServicesConfig.'system.identityModel.services'.federationConfiguration.cookieHandler.domain = $HostUri.DnsSafeHost

    $wifServicesConfig.Save("$($settings.'Infrastructure.WebRoot')\wif.services.config")
}

function Update-AADTenantInfo($settings, $web)
{
    Write-Log "Reading AAD metadata"

    $aadMetadataLocation = $($settings.'Aad.AADMetadataLocationFormat' -f $($settings.'Provisioning.AdminPrincipalName'.Split('@')[1]))
    Write-Log "AAD metadata location [$aadMetadataLocation]"

    $tempAadMetadataFile = [System.IO.Path]::GetTempFileName()
    Write-Log "Metadata will be stored in temporary file [$tempAadMetadataFile]"

    Write-Log "Invoking request"
    Invoke-WebRequest $aadMetadataLocation -UseBasicParsing -OutFile $tempAadMetadataFile

    Write-Log "Reading metadata from file at [$tempAadMetadataFile]"
    [xml]$aadMetadata = Get-Content $tempAadMetadataFile

    Write-Log "Removing temporary file at [$tempAadMetadataFile]"
    Remove-Item $tempAadMetadataFile -Force -Confirm:$false

    Write-Log "Extracting tenant guid from metadata"
    $url = New-Object System.Uri($aadMetadata.EntityDescriptor.entityId)

    $add = $web.configuration.appSettings.add | where key -eq 'Aad.TenantDomainGUID' | select -First 1
    $add.value = $url.AbsolutePath.Trim('/')

    Write-Log "Tenant guid updated"
}

function Update-AADTrustedIssuers($web)
{
    Write-Log "Start updating TrustedIssuers settings"

    $aadIssuerNameFormatNode = $web.configuration.appSettings.add | where key -eq 'Aad.AADIssuerNameFormat' | select -First 1
    $ppe = $aadIssuerNameFormatNode.value -like 'https://sts.windows-ppe.net/{0}/'

    $aadName = if($ppe) { 'PPE' } else { 'PROD' }
    Write-Log "TrustedIssuers settings will be applied for $aadName AAD"

    $trustedIssuersNode = $web.configuration.appSettings.add | where key -eq 'Aad.TrustedIssuers' | select -First 1
    Write-Log "Current TrustedIssuers settings = '$($trustedIssuersNode.value)'"

    $trustedIssuers = @($trustedIssuersNode.value -split ';' | ? { $_ })

    $requiredTrustedIssuers = 'https://sts.windows.net/72f988bf-86f1-41af-91ab-2d7cd011db47/', # MSFT tenant
                              'https://sts.windows.net/975f013f-7f24-47e8-a7d3-abc4752bf346/'  # PME tenant

    if($ppe) {
        # For dev/test/tie environments Retail test tenant should be a trusted issuer
        $requiredTrustedIssuers += 'https://sts.windows.net/34b54b07-9b04-4369-839c-380f28f3e0fe/' # Dynamics 365 - Retail - Test (dynamics365RetailTest.onmicrosoft.com)
    } else {
        $requiredTrustedIssuers += 'https://sts.windows.net/57206206-ec82-4579-9724-0a098ed1b99f/' # Dynamics 365 - Retail (dynamics365Retail.onmicrosoft.com)
    }

    foreach($issuer in $requiredTrustedIssuers) {
        if($issuer -notin $trustedIssuers) {
            $trustedIssuers += $issuer
        }
    }

    $updatedTrustedIssuers = $trustedIssuers -join ';'
    Write-Log "Storing updated TrustedIssuers = $updatedTrustedIssuers"

    if(-not $trustedIssuersNode) {
        $trustedIssuersNode = $web.CreateElement("add")
        $trustedIssuersNode.SetAttribute('key', 'Aad.TrustedIssuers')
        $trustedIssuersNode.SetAttribute('value', $updatedTrustedIssuers)
        $web.configuration.appSettings.AppendChild($trustedIssuersNode) | Out-Null
    }
    else {
        $trustedIssuersNode.value = $updatedTrustedIssuers
    }

    $trustedIssuersMetadataAddress = 'https://login.windows.net/common/FederationMetadata/2007-06/FederationMetadata.xml'
    Write-Log "Storing $trustedIssuersMetadataAddress address"

    $trustedIssuersMetadataAddressNode = $web.configuration.appSettings.add | where key -eq 'Aad.TrustedIssuersMetadataAddress' | select -First 1
    if(-not $trustedIssuersMetadataAddressNode) {
        $trustedIssuersMetadataAddressNode = $web.CreateElement("add")
        $trustedIssuersMetadataAddressNode.SetAttribute('key', 'Aad.TrustedIssuersMetadataAddress')
        $trustedIssuersMetadataAddressNode.SetAttribute('value', $trustedIssuersMetadataAddress)
        $web.configuration.appSettings.AppendChild($trustedIssuersMetadataAddressNode) | Out-Null
    }
    else {
        $trustedIssuersMetadataAddressNode.value = $trustedIssuersMetadataAddress
    }
}

function Update-DataAccessFlightingInfo($settings, $web)
{
    Write-Log "Reading DataAccess Flighting metadata"

    $flightingEnvironment = $settings.'DataAccess.FlightingEnvironment'
    Write-Log "flighting Environment [$flightingEnvironment]"

    $flightingCertificateThumbprint = $settings.'DataAccess.FlightingCertificateThumbprint'
    Write-Log "flighting Certificate Thumbprint [$flightingCertificateThumbprint]"

    $flightingServiceCatalogID = $settings.'DataAccess.FlightingServiceCatalogID'
    Write-Log "flighting ServiceCatalogID [$flightingServiceCatalogID]"

    $add = $web.configuration.appSettings.add | where key -eq 'DataAccess.FlightingEnvironment' | select -First 1
    $add.value = $flightingEnvironment

    $add = $web.configuration.appSettings.add | where key -eq 'DataAccess.FlightingCertificateThumbprint' | select -First 1
    $add.value = $flightingCertificateThumbprint

    $add = $web.configuration.appSettings.add | where key -eq 'DataAccess.FlightingServiceCatalogID' | select -First 1
    $add.value = $flightingServiceCatalogID

    Write-Log "DataAccess Flighting Infomation updated"
}

function Update-WifConfigWithPerfSdkCustomTokenHandler($settings, $wif)
{
    Write-Log "Checking if wif.config needs to be updated with PerfSDK custom token handler"

    $updateWifConfigForPerfSdkFlagString = $settings.'WIF.UpdateWifConfigForPerfSdk'
    $updateWifConfigForPerfSdkFlagBool = $null

    if(![string]::IsNullOrEmpty($updateWifConfigForPerfSdkFlagString))
    {
        if([bool]::TryParse($updateWifConfigForPerfSdkFlagString, [ref]$updateWifConfigForPerfSdkFlagBool))
        {
            if($updateWifConfigForPerfSdkFlagBool)
            {
                Write-Log "Adding custom token handler"

                $securityTokenHandlerConfiguration = $wif.SelectSingleNode("system.identityModel/identityConfiguration/securityTokenHandlers")

                $removeNode = $wif.CreateElement("remove")
                $removeNode.SetAttribute("type","System.IdentityModel.Tokens.Saml2SecurityTokenHandler, System.IdentityModel, Version=4.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089")

                $addNode = $wif.CreateElement("add")
                $addNode.SetAttribute("type","MS.Dynamics.TestTools.CloudCommonTestUtilities.Authentication.PerfSdkSaml2TokenHandler, MS.Dynamics.TestTools.CloudCommonTestUtilities")

                $securityTokenHandlerConfiguration.AppendChild($removeNode)
                $securityTokenHandlerConfiguration.AppendChild($addNode)
            }
            else
            {
                Write-Log "WIF.UpdateWifConfigForPerfSdk is false. Custom token handler is not added"
            }
        }
        else
        {
            Write-Log "WIF.UpdateWifConfigForPerfSdk cannot be parsed as a boolean value. Custom token handler is not added"
        }
    }
    else
    {
        Write-Log "WIF.UpdateWifConfigForPerfSdk is null or empty. Custom token handler is not added"
    }
}

function Update-WebConfig($settings)
{
    Write-Log "Reading web.config at $($settings.'Infrastructure.WebRoot')"

    [xml]$web = Get-Content "$($settings.'Infrastructure.WebRoot')\web.config"

    $uri = New-Object Uri $settings.'Infrastructure.HostUrl'

    $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.FullyQualifiedDomainName' | select -First 1
    $add.value = $uri.DnsSafeHost

    $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.HostedServiceName' | select -First 1
    $add.value = $uri.DnsSafeHost.Split('.')[0]

    $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.HostName' | select -First 1
    $add.value = $uri.DnsSafeHost

    $siteAdmin = $settings.'Provisioning.AdminPrincipalName'
    $add = $web.configuration.appSettings.add | where key -eq 'Aad.AADTenantId' | select -First 1
    $add.value = $siteAdmin.Split('@')[1]

    Update-AADTenantInfo $settings $web

    Update-DataAccessFlightingInfo $settings $web

    Update-AADTrustedIssuers $web

    $ClickOnceUnpackDir = $settings.'Infrastructure.ClickonceAppsDirectory'

    Write-Log "Click-once package dir $ClickOnceUnpackDir"

    if ((Test-Path $ClickOnceUnpackDir))
    {
        $ClickOnceTargetDir = Join-Path $settings.'Infrastructure.WebRoot' "apps"
        Write-Log "Click-once package dir $ClickOnceUnpackDir exists, copying click-one packages to webroot $ClickOnceTargetDir"

        if (!(Test-Path $ClickOnceTargetDir))
        {
            New-Item $ClickOnceTargetDir -ItemType Directory -Force
        }

        Copy-Item "$ClickOnceUnpackDir\*" $ClickOnceTargetDir -Recurse -Confirm:$false -Force

        $add = $web.configuration.appSettings.add | where key -eq 'Infrastructure.ClickonceAppsDirectory' | select -First 1
        $add.value = "$(split-path $ClickOnceTargetDir -Leaf)"

        Write-Log "Providing read access to the configuration files in APPS folder under web root: required for clickonce packages to work"
        Set-WebConfigurationProperty -pspath "MACHINE/WEBROOT/APPHOST/$($settings.'Infrastructure.ApplicationName')/Apps" -filter "system.webServer/security/requestFiltering/fileExtensions/add[@fileExtension='.config']" -name "allowed" -value "True"
    }

    Configure-ClaimIssuers $web $settings

    Configure-MachineKeys $web $settings

    CreateFlightingCacheFolder $settings
    Configure-FlightingCachePath $web $settings

    $web.Save("$($settings.'Infrastructure.WebRoot')\web.config")
}

function Update-WifConfig($settings)
{
    $siteAdmin = $settings.'Provisioning.AdminPrincipalName'

    Write-Log "Reading wif.config at $($settings.'Infrastructure.WebRoot')"

    $wifConfig = (Get-Content "$($settings.'Infrastructure.WebRoot')\wif.config").Replace('TENANT_ID_PLACEHOLDER',"$($siteAdmin.Split('@')[1])")
    $wifConfig = $wifConfig.Replace('https://sts.windows.net/',$settings.'Provisioning.AdminIdentityProvider')

    Add-Type -AssemblyName "System.Xml"

    Write-Log "Adding dynamic trusted certificates to wif.config"
    [xml]$wif = Get-Content "$($settings.'Infrastructure.WebRoot')\wif.config"

    $issuerNameRegistry = $wif.SelectSingleNode("system.identityModel/identityConfiguration/securityTokenHandlers/securityTokenHandlerConfiguration/issuerNameRegistry")
    $authorities = $issuerNameRegistry.SelectNodes("authority")
    $TrustedCerts = $($settings.'Infrastructure.TrustedCertificates').Split(';')

    $TrustedIssuer = $null
    $TokenIssuer = $authorities | where { $_.name -eq $($settings.'Aad.AcsTokenIssuer') }
    $TokenIssuerKeys = $null

    if ($TokenIssuer -ne $null)
    {
        $TokenIssuerKeys = $TokenIssuer.SelectSingleNode("keys")
    }

    foreach($TrustedCert in $TrustedCerts)
    {
        foreach ($auth in $authorities)
        {
            $TrustedIssuer = $auth.keys.add | where thumbprint -eq $TrustedCert

            if ($TrustedIssuer -ne $null)
            {
                break
            }
        }

        if ($TrustedIssuer -eq $null)
        {
            if ($TokenIssuer -eq $null)
            {
                $TokenIssuer = $wif.CreateElement("authority")
                $TokenIssuer.SetAttribute("name",$($settings.'Aad.AcsTokenIssuer'))

                $TokenIssuerKeys = $wif.CreateElement("keys")
                $TokenIssuer.AppendChild($TokenIssuerKeys)

                $ValidIssuers = $wif.CreateElement("validIssuers")

                $newAdd = $wif.CreateElement("add")
                $newAdd.SetAttribute("name", $($settings.'Aad.AcsTokenIssuer'))
                $ValidIssuers.AppendChild($newAdd)

                $TokenIssuer.AppendChild($ValidIssuers)

                $issuerNameRegistry.AppendChild($TokenIssuer) | Out-Null
            }

            $newAdd = $wif.CreateElement("add")
            $newAdd.SetAttribute("thumbprint", $TrustedCert)
            $TokenIssuerKeys.AppendChild($newAdd)
        }
    }

    Write-Log "Removing duplicate authorities in wif.config"
    #we only dedup based on the authority name since a static authority only has one issuer/thumbprint

    $authorities = $issuerNameRegistry.SelectNodes("authority")
    $uniqueAuthorities  = New-Object 'System.Collections.Generic.HashSet[System.Xml.XmlElement]'
    foreach($auth in $authorities)
    {
        $existingAuth = $uniqueAuthorities | Where-Object {$_.name -eq $auth.name}
        if ($existingAuth -eq $null)
        {
            $newAuth = $wif.CreateElement("authority")
            $newAuth.SetAttribute("name",$auth.name)
            $newAuth.InnerXml = $auth.InnerXml
            $uniqueAuthorities.Add($newAuth) | Out-Null
        }
    }

    $type = $issuerNameRegistry.type
    $issuerNameRegistry.RemoveAll()
    $issuerNameRegistry.SetAttribute("type", $type)

    foreach($auth in $uniqueAuthorities)
    {
        $issuerNameRegistry.AppendChild($auth) | Out-Null
    }

    Update-WifConfigWithPerfSdkCustomTokenHandler $settings $wif

    $wif.Save("$($settings.'Infrastructure.WebRoot')\wif.config")
}

Initialize-Log $log
Write-Log "Decoding settings"
$settings = Decode-Settings $config

Write-Log "Updating AOS web.config"
Update-WebConfig $settings
Write-Log "AOS web.config update complete"

Write-Log "Updating AOS wif.config"
Update-WifConfig $settings
Write-Log "AOS wif.config update complete"

Write-Log "Updating AOS web.services.config"
Update-WifServicesConfig $settings
Write-Log "AOS web.services.config update complete"

#Temporary removing this:
#if these are present DBSync assumes we are in local onebox mode and behaves differently
#Set-RegistryValues $settings

Write-Log "Restarting IIS"
iisreset

Write-Log "Finished, exiting"

# SIG # Begin signature block
# MIIjkgYJKoZIhvcNAQcCoIIjgzCCI38CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBIJatmlmZTMd2x
# Q7AXCOdB1XmDTyoqcdmjSiKF5bvBOKCCDYEwggX/MIID56ADAgECAhMzAAABh3IX
# chVZQMcJAAAAAAGHMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMjAwMzA0MTgzOTQ3WhcNMjEwMzAzMTgzOTQ3WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDOt8kLc7P3T7MKIhouYHewMFmnq8Ayu7FOhZCQabVwBp2VS4WyB2Qe4TQBT8aB
# znANDEPjHKNdPT8Xz5cNali6XHefS8i/WXtF0vSsP8NEv6mBHuA2p1fw2wB/F0dH
# sJ3GfZ5c0sPJjklsiYqPw59xJ54kM91IOgiO2OUzjNAljPibjCWfH7UzQ1TPHc4d
# weils8GEIrbBRb7IWwiObL12jWT4Yh71NQgvJ9Fn6+UhD9x2uk3dLj84vwt1NuFQ
# itKJxIV0fVsRNR3abQVOLqpDugbr0SzNL6o8xzOHL5OXiGGwg6ekiXA1/2XXY7yV
# Fc39tledDtZjSjNbex1zzwSXAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUhov4ZyO96axkJdMjpzu2zVXOJcsw
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDU4Mzg1MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAixmy
# S6E6vprWD9KFNIB9G5zyMuIjZAOuUJ1EK/Vlg6Fb3ZHXjjUwATKIcXbFuFC6Wr4K
# NrU4DY/sBVqmab5AC/je3bpUpjtxpEyqUqtPc30wEg/rO9vmKmqKoLPT37svc2NV
# BmGNl+85qO4fV/w7Cx7J0Bbqk19KcRNdjt6eKoTnTPHBHlVHQIHZpMxacbFOAkJr
# qAVkYZdz7ikNXTxV+GRb36tC4ByMNxE2DF7vFdvaiZP0CVZ5ByJ2gAhXMdK9+usx
# zVk913qKde1OAuWdv+rndqkAIm8fUlRnr4saSCg7cIbUwCCf116wUJ7EuJDg0vHe
# yhnCeHnBbyH3RZkHEi2ofmfgnFISJZDdMAeVZGVOh20Jp50XBzqokpPzeZ6zc1/g
# yILNyiVgE+RPkjnUQshd1f1PMgn3tns2Cz7bJiVUaqEO3n9qRFgy5JuLae6UweGf
# AeOo3dgLZxikKzYs3hDMaEtJq8IP71cX7QXe6lnMmXU/Hdfz2p897Zd+kU+vZvKI
# 3cwLfuVQgK2RZ2z+Kc3K3dRPz2rXycK5XCuRZmvGab/WbrZiC7wJQapgBodltMI5
# GMdFrBg9IeF7/rP4EqVQXeKtevTlZXjpuNhhjuR+2DMt/dWufjXpiW91bo3aH6Ea
# jOALXmoxgltCp1K7hrS6gmsvj94cLRf50QQ4U8Qwggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIVZzCCFWMCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAYdyF3IVWUDHCQAAAAABhzAN
# BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQg4+XnYZJs
# hApHfYlO7rsoHzK0hIEPLX0NfSajLObWECYwQgYKKwYBBAGCNwIBDDE0MDKgFIAS
# AE0AaQBjAHIAbwBzAG8AZgB0oRqAGGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTAN
# BgkqhkiG9w0BAQEFAASCAQCMtDCq3mY2tx+z09MweEbjOiGkGEmMj9W93WuL7dnX
# EevnROFbUOIY+v0PDrFWkCoNrHKLFDE5tpRy9rwZOrLeXEzEkNZNUQAl79jq5oUY
# shKRWgltiriDfzIVIIaEx+w52GYZ49F4Mf2WiCHrdB3Lfr24TnlqBnswBW0CeBI9
# JQrjh1CLySaAOKjII03j6k1X7sA48s0jOKYhUyNyHcfJMG1txkvbQDDDqRGvWfh2
# FMoUsL/QrnRgHJtrHbMndm+yP1SKJm5j+iv54WwW8u8uv8JU258uyoH7Ok5GYuJG
# SpoH/jaZRPB5mNM+3LOb/0cUbB1WtddkWtfu0JqteAVwoYIS8TCCEu0GCisGAQQB
# gjcDAwExghLdMIIS2QYJKoZIhvcNAQcCoIISyjCCEsYCAQMxDzANBglghkgBZQME
# AgEFADCCAVUGCyqGSIb3DQEJEAEEoIIBRASCAUAwggE8AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIDdlOaIV58evUhFehCWrs8Qm2wfQC+kijBBI2rUO
# Hq4PAgZfu+oE3VAYEzIwMjAxMjE3MDgwNzIxLjM1M1owBIACAfSggdSkgdEwgc4x
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
# b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1p
# Y3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMg
# VFNTIEVTTjpEOURFLUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
# U3RhbXAgU2VydmljZaCCDkQwggT1MIID3aADAgECAhMzAAABLS5NQcpjZTOgAAAA
# AAEtMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
# aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
# cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEw
# MB4XDTE5MTIxOTAxMTUwNFoXDTIxMDMxNzAxMTUwNFowgc4xCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVy
# YXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpEOURF
# LUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
# ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKlhjfR1STqYRTS3s0i4
# jIcSMV+G4N0oYgwlQK+pl4DVMFmr1iTZHFLj3Tt7V6F+M/BXx0h9i0uu1yBnqCqN
# OkuJERTbVnM4u3JvRxzsQfCjBfqD/CNwoMNekoylIBzxP50Skjp1pPsnQBKHaCP8
# tguvYVzoTQ54q2VpYEP/+OYTQeEPqWFi8WggvsckuercUGkhYWM8DV/4JU7N/rbD
# rtamYbe8LtkViTQYbigUSCAor9DhtAZvq8A0A73XFH2df2wDlLtAnKCcsVvXSmZ3
# 5bAqneN4uEQVy8NQdReGI1tI6UxoC7XnjGvK4McDdKhavNJ7DAnSP5+G/DTkdWD+
# lN8CAwEAAaOCARswggEXMB0GA1UdDgQWBBTZbGR8QgEh+E4Oiv8vQ7408p2GzTAf
# BgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
# hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
# aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
# BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0
# YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsG
# AQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQB9awNk906recBuoO7Ezq7B8UGu9EoF
# XiL8ac0bbsZDBY9z/3p8atVZRCxHN43a3WGbCMZoKYxSBH6UCkcDcwXIfNKEbVMz
# nF1mjpQEGbqhR+rPNqHXZotSV+vn85AxmefAM3bcLt+WNBpEuOZZ4kPZVcFtMo4Y
# yQjxoNRPiwmp+B0HkhQs/l/VIg0XJY6k5FRKE/JFEcVY4256NdqUZ+3jou3b4OAk
# tE2urr4V6VRw1fffOlxZb8MyvE5mqvTVJOStVxCuhuqg1rIe8la1gZ5iiuIyWeft
# ONfMw0nSZchGLigDeInw6XfwwgFnC5Ql8Pbf2jOxCUluAYbzykI+MnBiMIIGcTCC
# BFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
# b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1WhcN
# MjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0
# VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEw
# RA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQe
# dGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJUGKx
# Xf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw2k4G
# kbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0CAwEA
# AaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
# AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
# zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
# cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYI
# KwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
# b20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0g
# AQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93
# d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYIKwYB
# BQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0AGUA
# bQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9naOh
# IW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
# +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzymXlK
# kVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
# /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
# PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/
# fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCII
# YdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0
# cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7a
# KLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxEPJdQ
# cdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+
# NR4Iuto229Nfj950iEkSoYIC0jCCAjsCAQEwgfyhgdSkgdEwgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjpE
# OURFLUUzOUEtNDNGRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZaIjCgEBMAcGBSsOAwIaAxUAn85fx36He7F0vgmyUlz2w82l0LGggYMwgYCk
# fjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
# Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDANBgkqhkiG9w0BAQUFAAIF
# AOOFYpswIhgPMjAyMDEyMTcwODUyNDNaGA8yMDIwMTIxODA4NTI0M1owdzA9Bgor
# BgEEAYRZCgQBMS8wLTAKAgUA44VimwIBADAKAgEAAgIqQAIB/zAHAgEAAgIRMTAK
# AgUA44a0GwIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
# AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUAA4GBAET/pP5efMv+eo+W
# x6yjNh9OMfPsQsvDsI/stlWtQ/EFyKQKPWpC9NheRMl1h+2vTaCs+mksEzHwbuvp
# 7X5QXlv1sfqC9d9+rAqINH0lCz3ZQRTXOKcZm7wxSc4A1ftVufzRRDVuAFUZ2QKJ
# krdF19cqrYqd6bcSjdBcSav+TUQhMYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNV
# BAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgUENBIDIwMTACEzMAAAEtLk1BymNlM6AAAAAAAS0wDQYJYIZIAWUD
# BAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0B
# CQQxIgQgp/BBV8IQjOu83agF4p0XIKS/0iON/5R22/2MXpUkc60wgfoGCyqGSIb3
# DQEJEAIvMYHqMIHnMIHkMIG9BCCO8Vpycn0gB4/ilRAPPDbS+Cmbqj/uC011moc5
# oeGDwTCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB
# LS5NQcpjZTOgAAAAAAEtMCIEINL782oYNUKrzB3nFEy/e//8y5ov1NFAV+IMZ0jd
# ZaPlMA0GCSqGSIb3DQEBCwUABIIBADhC+2dIVO+zKU0j5Cwf8MaC7ffWkEYj6UtG
# W0flECGC9UEi9MEVDtEqjEwZi+HpEEKcLnfg57UAqidu9xg6bx4ESyFfPQC+CUAB
# wWX9I50H2eHfqQLanMXl9hK2TAFMCQ1JBUsLEeZmSOdosnOE4QlvWqw0/0DhUNTt
# ujXaf065SKX0DWfKNY3YzhgFAQXycR6sRtjIoMogsqqowXDrYGvwfQ46liwOiT+N
# fTK7vZx07HgHGaItKTd7KXIcl3PWdoT2hgdCcyciBUa0LDtZx/qDOV1+/tFhLwCo
# QKmj9v6CZKdNHJw0zxlC8Qy6OJ4vmJ6ARA36d/Rr/o0+Xe0r0CM=
# SIG # End signature block
